-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- Jamba - Jafula's Awesome Multi-Boxer's Assistant Addon.
-- JambaMerchant - Handles character's merchant interaction.
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
local MAJOR, MINOR = "JambaMerchant-1.0", 0.1
local JambaMerchant = LibStub:NewLibrary(MAJOR, MINOR)
if not JambaMerchant then 
	return 
end

-------------------------------------------------------------------------------------------------------------
-- Locale.
-------------------------------------------------------------------------------------------------------------
local L = LibStub( "AceLocale-3.0" ):GetLocale( "Jamba" )

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- INITIALIZE, OPTIONS AND CONFIGURATION
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: Initialize
-- Initializes JambaMerchant.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:Initialize( dbOptions )

	-- Initialise utility functions and constants.
	self.utils = LibStub( "JambaUtils-1.0" )
		
	-- Use the global JambaComms for communication.
	self.comms = JambaComms
	
	-- Remember the player name.
	self.playerName = UnitName( "player" )

	-- Sets the options database used to store user options.
	self.db = dbOptions

	-- Initialize the options configuration.
	LibStub( "AceConfig-3.0" ):RegisterOptionsTable( "Jamba-Merchant", self:GetOptionsConfiguration() )
	self.optionsFrame = LibStub( "AceConfigDialog-3.0" ):AddToBlizOptions( "Jamba-Merchant", "Jamba-Merchant" )

	-- Initialize static popups.
	self.InitializePopUps()	
	
	-- Temporary table for selected auto buy items that are marked for removal.
	self.autoBuyItemsMarkedForRemoval = {}
	
	-- Temporary amount to buy when adding auto buy items via the GUI.
	self.autoBuyAmountToBuy = 20

	-- Cap amount to buy for auto buy to 4800 (24 slot bag * 200 ammo per slot).
	self.maximumAutoBuyAmount = 4800

	-- Item link of item to add to auto buy list.
	self.autoBuyItemLink = nil

	-- The tag to add to the item to buy.
	self.autoBuyItemTag = L["all"]
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant:InitializePopUps
-- Initializes popups.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:InitializePopUps()

   -- Confirm removing characters from member list.
   StaticPopupDialogs["JAMBAMERCHANT_CONFIRM_REMOVE_AUTO_BUY_ITEMS"] = {
        text = L["Are you sure you wish to remove the selected items from the auto buy list?"],
        button1 = ACCEPT,
        button2 = CANCEL,
        timeout = 0,
		whileDead = 1,
		hideOnEscape = 1,
        OnAccept = function()
			JambaMerchant:RemoveSelectedAutoBuyItems()
		end,
    }
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: GetOptionsConfiguration
-- Returns the configuration options table for JambaMerchant.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:GetOptionsConfiguration()

	if not self.configuration then
	
		self.configuration = {
			name = "Jamba-Merchant",
			handler = JambaMerchant,
			type = "group",
			childGroups  = "tab",
			args = {			
				autoRepairGroup = {
					name = L["Auto Repair"],
					handler = JambaMerchant,
					type = "group",
					order = 0,	
					args = {			
						autoRepair = {
							type = "toggle",
							name = L["Auto Repair"],
							desc = L["Automatically repair all items."],
							get = "CanAutoRepair",
							set = "ToggleAutoRepair",
							width = "full",
							order = 1,
						},        
						autoRepairUseGuildFunds = {
							type = "toggle",
							name = L["Auto Repair With Guild Funds"],
							desc = L["Automatically repair all items using guild funds."],
							get = "CanAutoRepairWithGuildFunds",
							set = "ToggleAutoRepairWithGuildFunds",
							disabled = function() return not JambaMerchant:CanAutoRepair() end,
							width = "full",
							order = 2,
						}, 
					},
				},       
				autoSellGroup = {
					name = L["Auto Sell"],
					handler = JambaMerchant,
					type = "group",
					order = 1,	
					args = {					
						autoSell = {
							type = "toggle",
							name = L["Auto Sell Poor Quality Items"],
							desc = L["Automatically sell poor quality (grey) items."],
							get = "CanAutoSell",
							set = "ToggleAutoSell",
							width = "full",
							order = 3,
						},
					},
				},
				autoBuyGroup = {
					name = L["Auto Buy"],
					handler = JambaMerchant,
					type = "group",
					order = 2,	
					args = {					
						autoBuy = {
							type = "toggle",
							name = L["Auto Buy"],
							desc = L["Automatically re-stock items."],
							get = "CanAutoBuy",
							set = "ToggleAutoBuy",
							width = "half",
							order = 0,						
						}, 
						autoBuyOverflow = {
							type = "toggle",
							name = L["Auto Buy Overflow"],
							desc = L["Allow automatic buying to overflow on maximum amount when buying items in stacks."],
							get = "CanAutoBuyOverflow",
							set = "ToggleAutoBuyOverflow",
							disabled = function() return not JambaMerchant:CanAutoBuy() end,
							order = 1,												
						},  
						autoBuyItemAddGroupGui = {
							name = L["Add Items"],
							type = "group",
							inline = true,
							order = 2,
							disabled = function() return not JambaMerchant:CanAutoBuy() end,
							args = {
								autoBuyItemAmountGui = {
									type = "input",
									name = L["Amount"],
									desc = L["The amount of the item to buy."],
									pattern = "^(%d+)$",
									get = "GetAutoBuyItemAddAmount",
									set = "SetAutoBuyItemAddAmount",
									width = "half",
									order = 0,	
									cmdHidden = true,
								},					
								autoBuyItemGui = {
									type = "input",
									name = L["Item (drag an item to the edit box)"],
									desc = L["The item to buy (drag item to the edit box)."],
									get = "GetAutoBuyItemAdd",
									set = "SetAutoBuyItemAdd",
									order = 1,	
									cmdHidden = true,
								},						
								autoBuyItemTagGui = {
									type = "input",
									name = L["Item Tag"],
									desc = L["The tag for this item.  Only characters that have this tag will buy this item."],
									get = "GetAutoBuyItemAddTag",
									set = "SetAutoBuyItemAddTag",
									order = 2,	
									cmdHidden = true,
								},								
								autoBuyItemAddGui = {
									type = "execute",
									name = L["Add"],
									desc = L["Add an item to the auto buy list."],
									func = "AddAutoBuyItemGui",
									width = "half",
									order = 3,					
									cmdHidden = true,
								},
							},
						},								
						autoBuyItemListGroupGui = {
							type = "group",
							name = L["Items to Buy"],
							inline = true,
							order = 3,
							disabled = function() return not JambaMerchant:CanAutoBuy() end,
							args = {				
								autoBuyItems = {
									type = "multiselect",
									name = L["Items to Buy (check to mark for removal)"],
									desc = L["Items To Buy Automatically"],
									values = "GetAutoBuyItemsDisplay",
									get = "IsAutoBuyItemSelected",
									set = "ToggleAutoBuyItemSelection",
									width = "full",
									order = 0,					
									cmdHidden = true,
								},					
								autoBuyItemRemoveGui = {
									type = "execute",
									name = L["Remove Items"],
									desc = L["Remove selected items from the auto buy list."],
									func = "RemoveSelectedAutoBuyItemsGui",
									order = 1,					
									cmdHidden = true,
								},
							},
						},
						addItem = {
							type = "input",
							name = L["Add Item"], 
							desc = L["Add an item to the auto buy list."],
							get = false,
							set = "AddAutoBuyItem",
							usage = L["[Item Link] <number of items> <tag>"],
							guiHidden = true,
						},					
						removeItem = {
							type = "input",
							name = L["Remove Item"], 
							desc = L["Remove an item from the auto buy list."],
							get = false,
							set = "RemoveAutoBuyItem",
							usage = L["[Item Link]"],
							guiHidden = true,
						},					
					},
				},
				pushSettingsToOtherCharacters = {
					type = "execute",
					name = L["Push To Characters"],
					desc = L["Push these settings to all characters that are enabled (in Jamba) and online."],
					func = "PushSettingsToEnabledCharacters",
					order = 3,			
				},	
				doMerchantBusinessQuietly = {
					type = "toggle",
					name = L["Do Merchant Business Silently"],
					desc = L["Do not tell the master if the character has completed business with a merchant."],
					get = "CanDoMerchantBusinessQuietly",
					set = "ToggleDoMerchantBusinessQuietly",
					width = "full",
					order = 4,						
				}, 				
			},
		}
		
	end
	
	return self.configuration
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: GetOptionsFrame
-- Returns the blizzard interface options frame created for JambaMerchant.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:GetOptionsFrame()
	
	return self.optionsFrame
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant:RefreshOptions
-- Refresh the JambaMerchant options dialog.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:RefreshOptions()
	
	LibStub( "AceConfigRegistry-3.0" ):NotifyChange( "Jamba-Merchant" )
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant:PushSettingsToEnabledCharacters
-- Push the settings for JambaMerchant to other enabled and connected characters.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:PushSettingsToEnabledCharacters()

	-- Push the settings to all.
	self.comms:PushSettings( self.db, "Jamba-Merchant" )
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant:ReceiveSettings
-- Receive settings for JambaMerchant.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:ReceiveSettings( settings )

	-- Set the new settings.
    self.db.autoRepair = settings.autoRepair
    self.db.autoRepairUseGuildFunds = settings.autoRepairUseGuildFunds
    self.db.autoBuy = settings.autoBuy
    self.db.autoBuyOverflow = settings.autoBuyOverflow
    self.db.doMerchantBusinessQuietly = settings.doMerchantBusinessQuietly
    self.db.autoSell = settings.autoSell
    self.utils:ClearTableWithInnerTable( self.db.autoBuyItems )
	for itemLink, pushedItemTable in pairs( settings.autoBuyItems ) do
		self.db.autoBuyItems[itemLink] = {}
		newItemTable = self.db.autoBuyItems[itemLink]
		for itemTableKey, itemTableValue in pairs( pushedItemTable ) do
			newItemTable[itemTableKey] = itemTableValue
		end
	end
	
	-- Refresh the options.
	self:RefreshOptions()
	
end

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- CONFIGURATION CALLBACKS
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: CanAutoSell
-- Return the auto sell flag.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:CanAutoSell( info )

    return self.db.autoSell
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: ToggleAutoSell
-- Toggle the auto sell flag.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:ToggleAutoSell( info, value )

    self.db.autoSell = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: CanAutoRepair
-- Return the auto repair flag.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:CanAutoRepair( info )

    return self.db.autoRepair
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: ToggleAutoRepair
-- Toggle the auto repair flag.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:ToggleAutoRepair( info, value )

    self.db.autoRepair = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: CanAutoRepairWithGuildFunds
-- Return the auto repair with guild funds flag.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:CanAutoRepairWithGuildFunds( info )

    return self.db.autoRepairUseGuildFunds
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: ToggleAutoRepairWithGuildFunds
-- Toggle the auto repair with guild funds flag.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:ToggleAutoRepairWithGuildFunds( info, value )

    self.db.autoRepairUseGuildFunds = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: CanAutoBuy
-- Return the auto buy flag.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:CanAutoBuy( info )

    return self.db.autoBuy
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: ToggleAutoBuy
-- Toggle the auto buy flag.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:ToggleAutoBuy( info, value )

    self.db.autoBuy = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: CanAutoBuyOverflow
-- Return the auto buy overflow flag.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:CanAutoBuyOverflow( info )

    return self.db.autoBuyOverflow
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: ToggleAutoBuyOverflow
-- Toggle the auto buy overflow flag.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:ToggleAutoBuyOverflow( info, value )

    self.db.autoBuyOverflow = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: CanDoMerchantBusinessQuietly
-- Return the do merchant business quietly flag.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:CanDoMerchantBusinessQuietly( info )

    return self.db.doMerchantBusinessQuietly
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: ToggleDoMerchantBusinessQuietly
-- Toggle the do merchant business quietly flag.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:ToggleDoMerchantBusinessQuietly( info, value )

    self.db.doMerchantBusinessQuietly = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: AddAutoBuyItem
-- Add an item to auto buy.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:AddAutoBuyItem( info, value )

	-- Add an item to the list of items to auto buy.
	local itemLink, amountToBuy, itemTag = select( 3, value:find( "(.+)%s(%d+)%s(.+)" ) )
	
	-- If an item link and number of items to buy was provided...
	if itemLink and amountToBuy and itemTag then
		
		-- Add the item to the list.
		JambaMerchant:AddAutoBuyItemList( itemLink, tonumber(amountToBuy), itemTag )
		
	end

end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: AddAutoBuyItemList
-- Add an item to the auto buy list.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:AddAutoBuyItemList( itemLink, amountToBuy, itemTag )
		
	-- Get some more information about the item.
	itemName = GetItemInfo( itemLink )

	-- If the item could be found.
	if itemName then
		
		-- Cap amount to buy?
		if amountToBuy > self.maximumAutoBuyAmount then
			amountToBuy = self.maximumAutoBuyAmount
		end
			
		-- Add the item and amount to buy to the database. 
		amountToBuy = tonumber( amountToBuy )
		local itemTable = self.db.autoBuyItems[itemLink]
		if not itemTable then
			self.db.autoBuyItems[itemLink] = {}
			itemTable = self.db.autoBuyItems[itemLink]
		end
		itemTable["amount"] = amountToBuy
		itemTable["tag"] = itemTag
		itemTable["name"] = itemName
		
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: GetAutoBuyItemAddAmount
-- Get the amount to buy to use when adding an auto buy item via the Gui.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:GetAutoBuyItemAddAmount( info )
	
	return tostring( self.autoBuyAmountToBuy )
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: SetAutoBuyItemAddAmount
-- Set the amount to buy to use when adding an auto buy item via the Gui.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:SetAutoBuyItemAddAmount( info, value )

	local amount = tonumber( value )
	
	if amount > self.maximumAutoBuyAmount then
		amount = self.maximumAutoBuyAmount
	end		
	
	self.autoBuyAmountToBuy = amount
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: GetAutoBuyItemAdd
-- Get the item to buy to use when adding an auto buy item via the Gui.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:GetAutoBuyItemAdd( info )
	
	return self.autoBuyItemLink
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: SetAutoBuyItemAdd
-- Set the item to buy to use when adding an auto buy item via the Gui.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:SetAutoBuyItemAdd( info, value )
	
	self.autoBuyItemLink = value
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: GetAutoBuyItemAddTag
-- Get the tag of an item to buy to use when adding an auto buy item via the Gui.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:GetAutoBuyItemAddTag( info )
	
	return self.autoBuyItemTag
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: SetAutoBuyItemAddTag
-- Set the tag of an item to buy to use when adding an auto buy item via the Gui.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:SetAutoBuyItemAddTag( info, value )
	
	self.autoBuyItemTag = value
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: AddAutoBuyItemGui
-- Add an item to the item buy list using a GUI.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:AddAutoBuyItemGui()
	
	if self.autoBuyItemLink and self.autoBuyItemTag and (self.autoBuyAmountToBuy > 0) then
		JambaMerchant:AddAutoBuyItemList( self.autoBuyItemLink, self.autoBuyAmountToBuy, self.autoBuyItemTag )
		self.autoBuyItemLink = nil
		self:RefreshOptions()
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: GetAutoBuyItemsDisplay
-- Returns a list of item buy items in a displayable state.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:GetAutoBuyItemsDisplay()

	local itemDisplay = {}
	local message = "%s (%d) [%s]"
	
	for itemLink, itemTable in pairs( self.db.autoBuyItems ) do
		itemDisplay[itemLink] = message:format( itemTable["name"], itemTable["amount"], itemTable["tag"] )
	end
	
	return itemDisplay
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: IsAutoBuyItemSelected
-- Return whether or not an item in the auto buy list is selected.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:IsAutoBuyItemSelected( info, key )

	local itemLink = key
	local isSelected = false
	
	if not self.autoBuyItemsMarkedForRemoval[itemLink] then
		self.autoBuyItemsMarkedForRemoval[itemLink] = isSelected
	end
	
	isSelected = self.autoBuyItemsMarkedForRemoval[itemLink]
	
	return isSelected

end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: ToggleAutoBuyItemSelection
-- Toggle whether or not an item in the auto buy list is selected.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:ToggleAutoBuyItemSelection( info, key, state )

	local itemLink = key
	local isSelected = state
	
	self.autoBuyItemsMarkedForRemoval[itemLink] = isSelected
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: RemoveAutoBuyItem
-- Remove an item from the auto buy list.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:RemoveAutoBuyItem( info, value )
	
	local itemLink = value
	
	if self.db.autoBuyItems[itemLink] then
		self.utils:ClearTable( self.db.autoBuyItems[itemLink] )
		self.db.autoBuyItems[itemLink] = nil
	end

	self:RefreshOptions()
		
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: RemoveSelectedAutoBuyItemsGui
-- Remove selected items from the item buy list.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:RemoveSelectedAutoBuyItemsGui()

	StaticPopup_Show( "JAMBAMERCHANT_CONFIRM_REMOVE_AUTO_BUY_ITEMS" )
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: RemoveSelectedAutoBuyItems
-- Remove selected items from the item buy list.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:RemoveSelectedAutoBuyItems()

	for itemLink, remove in pairs( self.autoBuyItemsMarkedForRemoval ) do 
		if remove then
			JambaMerchant:RemoveAutoBuyItem( nil, itemLink )
			self.autoBuyItemsMarkedForRemoval[itemLink] = nil
		end
	end
	
	self:RefreshOptions()
	
end

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- MERCHANT PROCESSING
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: ProcessMerchantShow
-- Do auto sell, auto repair and auto buy of items.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:ProcessMerchantShow()

	-- Get the user options.
	local autoSell = self:CanAutoSell()
	local autoRepair = self:CanAutoRepair()
	local autoRepairUseGuildFunds = self:CanAutoRepairWithGuildFunds()
	local autoBuy = self:CanAutoBuy()
	local autoBuyItems = self.db.autoBuyItems
	local autoBuyOverflow = self:CanAutoBuyOverflow()
		
	-- Does the user want to auto sell?
	if autoSell then
		self:DoMerchantSell()
	end
	
	-- Does the user want to auto repair?
	if autoRepair then
		-- Can this merchant repair?
		if CanMerchantRepair() then 
			-- Merchant repair.
			self:DoMerchantRepair( autoRepairUseGuildFunds )
		end
	end
	
	-- Does the user want to auto buy?
	if autoBuy then		
		-- Merchant auto buy.
		self:DoMerchantAutoBuy( autoBuyItems, autoBuyOverflow )
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: DoMerchantSell
-- Actions auto selling of grey items.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:DoMerchantSell()

	-- Iterate each bag the player has.		
	for bag = self.utils.BAG_PLAYER_BACKPACK, self.utils.BAG_PLAYER_MAXIMUM do 
		-- Iterate each slot in the bag.
		for slot = 1, GetContainerNumSlots( bag ) do 
			-- Get the item link for the item in this slot.
			local itemLink = GetContainerItemLink( bag, slot )
			-- If there is an item...
			if itemLink then
				-- Get the item's quality.
				local itemQuality = select( 3, GetItemInfo( itemLink ) )			
				-- If the item is of a poor quality...
				if itemQuality == self.utils.ITEM_QUALITY_POOR then 
					-- Then use it (effectively selling it as the character is talking to a merchant).
					UseContainerItem( bag, slot ) 
					-- Tell the boss.
					if not self.db.doMerchantBusinessQuietly then
						self.comms:TellMaster( self.playerName, L["I have sold: "]..itemLink )	
					end
				end
			end
		end
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: DoMerchantRepair
-- autoRepairUseGuildFunds - true to attempt to use guild funds.
-- Actions auto repair.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:DoMerchantRepair( autoRepairUseGuildFunds )

	-- How much to repair?
	local totalAmountSpentOnRepair = 0
	local repairCost = GetRepairAllCost()
	-- At least some cost...
	if repairCost > 0 then
		-- If allowed to use guild funds, then attempt to repair using guild funds.
		if autoRepairUseGuildFunds then
			if IsInGuild() and CanWithdrawGuildBankMoney() then
				RepairAllItems( 1 )
				totalAmountSpentOnRepair = totalAmountSpentOnRepair + repairCost
			end
		end
		-- After guild funds used, still need to repair?
		repairCost = GetRepairAllCost()
		-- At least some cost...
		if repairCost > 0 then
			-- How much money available?
			local moneyAvailable = GetMoney()
			-- More or equal money than cost?
			if moneyAvailable >= repairCost then
				-- Yes, repair.
				RepairAllItems()
				totalAmountSpentOnRepair = totalAmountSpentOnRepair + repairCost
			else
				-- Nope, tell the boss.
				self.comms:TellMaster( self.playerName, L["I do not have enough money to repair all my items."] )			
			end
		end
	end
	
	if totalAmountSpentOnRepair > 0 then
		-- Tell the boss how much that cost.
		if not self.db.doMerchantBusinessQuietly then
			local costString = self.utils:FormatMoneyString( totalAmountSpentOnRepair )
			self.comms:TellMasterSpecial( self.playerName, L["Repairing cost me: "]..costString )			
		end
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: DoMerchantAutoBuy
-- autoBuyItems - items and amount to buy.
-- overBuy - Over buy to meet need?
-- Does a merchant to sell any items of interest?  Is so, buy items up to the amount requested.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:DoMerchantAutoBuy(
	autoBuyItems,
	overBuy
)

	-- Flags will be set if the character does not have enough bag space or money.
	local outOfBagSpace = false
	local outOfMoney = false
	local outOfOtherCurrency = false

	-- Iterate all the wanted items...
	for itemLink, itemInfoTable in pairs( autoBuyItems ) do
		local maxItemAmount = itemInfoTable["amount"]
		local itemTag = itemInfoTable["tag"]
		-- Does this character have the item tag?  No, don't buy.
		if self.comms:CharacterHasTag( self.playerName, itemTag ) then
			-- Does the merchant have the item in stock?
			itemIndexMerchant = JambaMerchant:DoesMerchantHaveItemInStock( itemLink )
			if itemIndexMerchant then
				-- Yes, item is in stock, how many does the character need?
				local amountNeeded = JambaMerchant:GetAmountNeededForItemTopUp( itemLink, maxItemAmount )
				-- Need more than 0 items, buy them.
				if amountNeeded > 0 then
					-- Attempt to buy the items.
					local noFreeBagSpace, notEnoughMoney, notEnoughOtherCurrency = 
						self:BuyItemFromMerchant( itemIndexMerchant, amountNeeded, overBuy )
					-- Set flags if problems occurred.
					if noFreeBagSpace then
						outOfBagSpace = true		
					end
					if notEnoughMoney then
						outOfMoney = true
					end
					if notEnoughOtherCurrency then 
						outOfOtherCurrency = true
					end
				end
			end
		end
	end

	-- If there was a problem, tell the master.
	if outOfBagSpace then
		self.comms:TellMaster( self.playerName, L["I do not have enough space in my bags to complete my purchases."] )			
	end
	if outOfMoney then
		self.comms:TellMaster( self.playerName, L["I do not have enough money to complete my purchases."] )
	end
	if outOfOtherCurrency then
		self.comms:TellMaster( self.playerName, L["I do not have enough other currency to complete my purchases."] )
	end	
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: DoesMerchantHaveItemInStock
-- itemLink - the link to the item that is requested from the merchant.
-- Checks to see if a merchant has an item in stock.  If so, returns the merchant slot number the item is 
-- located in (index), otherwise nil.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:DoesMerchantHaveItemInStock(
	itemLink
)

	-- The index of the item to be found in the merchants inventory; initially nil, not found.
	local indexOfItemToFind = nil 
	-- Get the name of the item to find from the item link.
	local itemNameToFind = GetItemInfo( itemLink )
	-- How many items does the merchant have?
	local numberMerchantItems = GetMerchantNumItems()
	
	-- Iterate all the merchants items...
	for merchantIndex = 1, numberMerchantItems do
		-- Is there an item link for this item.
		local merchantItemLink = GetMerchantItemLink( merchantIndex )
		if merchantItemLink then
			-- Yes, get the item name.
			local itemNameMerchant = GetItemInfo( merchantItemLink )
			if itemNameMerchant == itemNameToFind then
				indexOfItemToFind = merchantIndex
				break
			end
		end
	end
	
	-- Return the index into the merchants inventory of the item.
	return indexOfItemToFind
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: GetAmountNeededForItemTopUp
-- itemLink - the link to the item that is requested from the merchant.
-- maxItemAmount - the maximum amount of items the character has requested.
-- Returns the amount of items a character needs to buy in order to top up an item.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:GetAmountNeededForItemTopUp(
	itemLink,
	maxItemAmount
)

	-- The amount of the item needed to top up the item.
	local amountNeeded = 0
	-- How much of this item does the character have in it's bags?
	local amountInBags = GetItemCount( itemLink )
	-- Does the character need more?
	if amountInBags < maxItemAmount then
		-- Yes, how much more?
		amountNeeded = maxItemAmount - amountInBags
	end
	-- Return the amount needed.
	return amountNeeded
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMerchant: BuyItemFromMerchant
-- itemIndexMerchant - the index to the item in the merchants inventory.
-- amountToBuy - the amount to buy.
-- overBuy - over buy to meet need?
-- Returns noFreeBagSpace, notEnoughMoney, notEnoughOtherCurrency  - flags that will be set if the character 
-- does not have enough bag space or money.
-------------------------------------------------------------------------------------------------------------
function JambaMerchant:BuyItemFromMerchant(
	itemIndexMerchant,
	amountToBuy,
	overBuy
)

	-- Flags will be set if the character does not have enough bag space or money.
	local noFreeBagSpace = false
	local notEnoughMoney = false
	local notEnoughOtherCurrency = false
	
	-- Processing variables.
	local buyThisAmount = 0
	local amountLeftToBuy = amountToBuy
	local actualAmountToBuy = 0
	local costToBuy = 0
	local moneyAvailable = 0
	
	-- Get information about the item from the merchant.
	local name, texture, price, itemsPerStack, numberAvailable, isUsable, extendedCost = 
		GetMerchantItemInfo( itemIndexMerchant )	
	local maximumCanBuyAtATime = GetMerchantItemMaxStack( itemIndexMerchant )
		
	-- Loop buying stacks from the merchant until the required number has been purchased.
	repeat
	
		-- Still need to buy more than the maximum?
		if amountLeftToBuy >= (maximumCanBuyAtATime * itemsPerStack) then
			-- Yes, buy the maximum amount.
			buyThisAmount = (maximumCanBuyAtATime * itemsPerStack)
		else
			-- No, just buy the amount left.
			buyThisAmount = amountLeftToBuy
		end
		
		-- Attempt to buy this amount from the merchant; although actual amount bought may differ,
		-- depending on merchant stock and over buy flag.

		-- How many does the merchant have left?
		numberAvailable = select( 5, GetMerchantItemInfo( itemIndexMerchant ) )

		-- Calculate how many to buy depending on the stacksize and whether over buying is allowed.
		actualAmountToBuy = buyThisAmount / itemsPerStack
		if overBuy then
			actualAmountToBuy = ceil(actualAmountToBuy)
		else
			actualAmountToBuy = floor(actualAmountToBuy)
		end

		-- If requesting more than the number available, then just buy as much as possible.
		-- If numberAvailable is -1 then there is unlimited stock available.
		if numberAvailable ~= -1 then
			if actualAmountToBuy > numberAvailable then
				actualAmountToBuy = numberAvailable 
			end
		end

		-- Does the character have enough money?
		costToBuy = actualAmountToBuy * price
		moneyAvailable = GetMoney()
		if moneyAvailable < costToBuy then			
			notEnoughMoney = true
		end

		-- Is there enough free space for this item in the characters bags?				
		--TODO - need to find items family type and compare to each container.
		--freeSlots, itemFamily = GetContainerNumFreeSlots(bagIndex)
		
		-- Buy from the merchant, if there is a valid amount to buy and the character has enough money.
		if (actualAmountToBuy > 0) and (not notEnoughMoney) then
			BuyMerchantItem( itemIndexMerchant, actualAmountToBuy )
		end				
				
		-- How much left to buy?
		amountLeftToBuy = amountLeftToBuy - buyThisAmount
		
	until amountLeftToBuy == 0
	
	-- TODO
	-- Return the success flags.
	return noFreeBagSpace, notEnoughMoney, notEnoughOtherCurrency
	
end

