﻿-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- Jamba - Jafula's Awesome Multi-Boxer's Assistant Addon.
-- JambaMe - Handles a character own-ness :-).
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
local MAJOR, MINOR = "JambaMe-1.0", 0.1
local JambaMe = LibStub:NewLibrary(MAJOR, MINOR)
if not JambaMe then 
	return 
end

-------------------------------------------------------------------------------------------------------------
-- Locale.
-------------------------------------------------------------------------------------------------------------
local L = LibStub( "AceLocale-3.0" ):GetLocale( "Jamba" )

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- INITIALIZE, OPTIONS AND CONFIGURATION
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-------------------------------------------------------------------------------------------------------------
-- JambaMe:Initialize
-- Initializes JambaMe.
-------------------------------------------------------------------------------------------------------------
function JambaMe:Initialize( dbOptions )

	-- Initialise utility functions and constants.
	self.utils = LibStub( "JambaUtils-1.0" )
	
	-- Use the global JambaComms for communication.
	self.comms = JambaComms

	-- Remember the player name.
	self.playerName = UnitName( "player" )

	-- Sets the options database used to store user options.
	self.db = dbOptions

	-- Initialize the options configuration.
	LibStub( "AceConfig-3.0" ):RegisterOptionsTable( "Jamba-Me", self:GetOptionsConfiguration() )
	self.optionsFrame = LibStub( "AceConfigDialog-3.0" ):AddToBlizOptions( "Jamba-Me", "Jamba-Me" )
					
	-- Flag set when told the master about health falling below a certain percentage.
	self.toldMasterAboutHealth = false

	-- Flag set when told the master about mana falling below a certain percentage.
	self.toldMasterAboutMana = false

	-- Have been hit flag.
	self.haveBeenHit = false
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMe:GetOptionsConfiguration
-- Returns the configuration options table for JambaMe.
-------------------------------------------------------------------------------------------------------------
function JambaMe:GetOptionsConfiguration()

	if not self.configuration then
	
		self.configuration = {
			name = "Jamba-Me",
			handler = JambaMe,
			type = 'group',
			args = {
				autoAcceptResurrectRequests = {
					type = "toggle",
					name = L["Auto Accept Resurrect Request"],
					desc = L["Automatically accept a resurrection request."],
					get = "CanAutoAcceptResurrectRequests",
					set = "ToggleAutoAcceptResurrectRequests",
					width = "full",
					order = 0,
				}, 				
				warnHitFirstTimeCombat = {
					type = "toggle",
					name = L["Warn If Hit First Time In Combat (Slave)"],
					desc = L["Warn the master when I get hit for the first time in combat."],
					get = "CanWarnFirstTimeHitCombat",
					set = "ToggleWarnFirstTimeHitCombat",
					width = "full",
					order = 1,
				}, 
				hitFirstTimeMessage = {
					type = "input",
					name = L["Hit First Time Message"],
					desc = L["The message to send the master when hit for the first time on combat."],
					get = "GetHitFirstTimeMessage",
					set = "SetHitFirstTimeMessage",
					width = "full",
					order = 2,
				}, 				
				warnTargetNotMasterEnterCombat = {
					type = "toggle",
					name = L["Warn If Target Not Master On Combat (Slave)"],
					desc = L["Warn the master if my target is not the master when entering combat."],
					get = "CanWarnTargetNotMasterEnterCombat",
					set = "ToggleWarnTargetNotMasterEnterCombat",
					width = "full",
					order = 3,
				},			
				warnTargetNotMasterMessage = {
					type = "input",
					name = L["Warn Target Not Master Message"],
					desc = L["The message to send the master when entering combat and the target is not the master."],
					get = "GetWarnTargetNotMasterMessage",
					set = "SetWarnTargetNotMasterMessage",
					width = "full",
					order = 4,
				}, 									
				warnWhenHealthDropsBelowX = {
					type = "toggle",
					name = L["Warn If My Health Drops Below"],
					desc = L["Warn the master if my health drops below a set percentage."],
					get = "CanWarnHealthDropsBelowX",
					set = "ToggleWarnHealthDropsBelowX",
					width = "full",
					order = 5,
				},					
				warnWhenHealthDropsAmount = {
					type = "input",
					name = L["Health Amount"],
					desc = L["The percentage of health allowed before warning."],
					pattern = "^(%d+)$",
					get = "GetWarnWhenHealthDropsAmount",
					set = "SetWarnWhenHealthDropsAmount",
					width = "half",
					order = 6,
				},	
				warnHealthDropsMessage = {
					type = "input",
					name = L["Warn Health Drop Message"],
					desc = L["The message to send the master when my health drops below a set percentage."],
					get = "GetWarnHealthDropsMessage",
					set = "SetWarnHealthDropsMessage",
					width = "full",
					order = 7,
				}, 									
				warnWhenManaDropsBelowX = {
					type = "toggle",
					name = L["Warn If My Mana Drops Below"],
					desc = L["Warn the master if my mana drops below a set percentage."],
					get = "CanWarnManaDropsBelowX",
					set = "ToggleWarnManaDropsBelowX",
					width = "full",
					order = 8,
				},				
				warnWhenManaDropsAmount = {
					type = "input",
					name = L["Mana Amount"],
					desc = L["The percentage of mana allowed before warning."],
					pattern = "^(%d+)$",
					get = "GetWarnWhenManaDropsAmount",
					set = "SetWarnWhenManaDropsAmount",
					width = "half",
					order = 9,
				},										
				warnManaDropsMessage = {
					type = "input",
					name = L["Warn Mana Drop Message"],
					desc = L["The message to send the master when my mana drops below a set percentage."],
					get = "GetWarnManaDropsMessage",
					set = "SetWarnManaDropsMessage",
					width = "full",
					order = 10,
				}, 													
				pushSettingsToOtherCharacters = {
					type = "execute",
					name = L["Push To Characters"],
					desc = L["Push these settings to all characters that are enabled (in Jamba) and online."],
					func = "PushSettingsToEnabledCharacters",
					order = 11,			
				},					
			},
		}
		
	end
	
	return self.configuration
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMe:GetOptionsFrame
-- Returns the blizzard interface options frame created for JambaMe.
-------------------------------------------------------------------------------------------------------------
function JambaMe:GetOptionsFrame()
	
	return self.optionsFrame
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMe:RefreshOptions
-- Refresh the JambaMe options dialog.
-------------------------------------------------------------------------------------------------------------
function JambaMe:RefreshOptions()
	
	LibStub( "AceConfigRegistry-3.0" ):NotifyChange( "Jamba-Me" )
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMe:PushSettingsToEnabledCharacters
-- Push the settings for JambaMe to other enabled and connected characters.
-------------------------------------------------------------------------------------------------------------
function JambaMe:PushSettingsToEnabledCharacters()

	-- Push the settings to all.
	self.comms:PushSettings( self.db, "Jamba-Me" )
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMe:ReceiveSettings
-- Receive settings for JambaMe.
-------------------------------------------------------------------------------------------------------------
function JambaMe:ReceiveSettings( settings )

	-- Set the new settings.
	self.db.warnHitFirstTimeCombat = settings.warnHitFirstTimeCombat
	self.db.hitFirstTimeMessage = settings.hitFirstTimeMessage
	self.db.warnTargetNotMasterEnterCombat = settings.warnTargetNotMasterEnterCombat
	self.db.warnTargetNotMasterMessage = settings.warnTargetNotMasterMessage
	self.db.warnWhenHealthDropsBelowX = settings.warnWhenHealthDropsBelowX
	self.db.warnWhenHealthDropsAmount = settings.warnWhenHealthDropsAmount
	self.db.warnHealthDropsMessage = settings.warnHealthDropsMessage
	self.db.warnWhenManaDropsBelowX = settings.warnWhenManaDropsBelowX
	self.db.warnWhenManaDropsAmount = settings.warnWhenManaDropsAmount
	self.db.warnManaDropsMessage = settings.warnManaDropsMessage
	self.db.autoAcceptResurrectRequest = settings.autoAcceptResurrectRequest
	
	-- Refresh the options.
	self:RefreshOptions()
	
end

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- CONFIGURATION CALLBACKS
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-------------------------------------------------------------------------------------------------------------
-- JambaMe: CanAutoAcceptResurrectRequests
-- Return the auto accept resurrect requests flag.
-------------------------------------------------------------------------------------------------------------
function JambaMe:CanAutoAcceptResurrectRequests( info )

    return self.db.autoAcceptResurrectRequest
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaMe: ToggleAutoAcceptResurrectRequests
-- Toggle the auto accept resurrect requests flag.
-------------------------------------------------------------------------------------------------------------
function JambaMe:ToggleAutoAcceptResurrectRequests( info, value )

    self.db.autoAcceptResurrectRequest = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaMe: CanWarnFirstTimeHitCombat
-- Return the can warn first time hit combat flag.
-------------------------------------------------------------------------------------------------------------
function JambaMe:CanWarnFirstTimeHitCombat( info )

    return self.db.warnHitFirstTimeCombat
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaMe: ToggleWarnFirstTimeHitCombat
-- Toggle the can warn first time hit combat flag.
-------------------------------------------------------------------------------------------------------------
function JambaMe:ToggleWarnFirstTimeHitCombat( info, value )

    self.db.warnHitFirstTimeCombat = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaMe: GethitFirstTimeMessage
-- Get the hit first time message.
-------------------------------------------------------------------------------------------------------------
function JambaMe:GetHitFirstTimeMessage( info )
	
	return self.db.hitFirstTimeMessage
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMe: SetHitFirstTimeMessage
-- Set the hit first time message.
-------------------------------------------------------------------------------------------------------------
function JambaMe:SetHitFirstTimeMessage( info, value )
	
	self.db.hitFirstTimeMessage = value
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMe: CanWarnTargetNotMasterEnterCombat
-- Return the can warn target not master enter combat flag.
-------------------------------------------------------------------------------------------------------------
function JambaMe:CanWarnTargetNotMasterEnterCombat( info )

    return self.db.warnTargetNotMasterEnterCombat
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaMe: ToggleWarnTargetNotMasterEnterCombat
-- Toggle the can warn target not master enter combat  flag.
-------------------------------------------------------------------------------------------------------------
function JambaMe:ToggleWarnTargetNotMasterEnterCombat( info, value )

    self.db.warnTargetNotMasterEnterCombat = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaMe: GetWarnTargetNotMasterMessage
-- Get the warn target not master message.
-------------------------------------------------------------------------------------------------------------
function JambaMe:GetWarnTargetNotMasterMessage( info )
	
	return self.db.warnTargetNotMasterMessage
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMe: SetWarnTargetNotMasterMessage
-- Set the warn target not master message.
-------------------------------------------------------------------------------------------------------------
function JambaMe:SetWarnTargetNotMasterMessage( info, value )
	
	self.db.warnTargetNotMasterMessage = value
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMe: CanWarnHealthDropsBelowX
-- Return the can warn when health drops below X flag.
-------------------------------------------------------------------------------------------------------------
function JambaMe:CanWarnHealthDropsBelowX( info )

    return self.db.warnWhenHealthDropsBelowX
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaMe: ToggleWarnHealthDropsBelowX
-- Toggle the can warn when health drops below X flag.
-------------------------------------------------------------------------------------------------------------
function JambaMe:ToggleWarnHealthDropsBelowX( info, value )

    self.db.warnWhenHealthDropsBelowX = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaMe: GetWarnWhenHealthDropsAmount
-- Get the amount of health allowed before warning.
-------------------------------------------------------------------------------------------------------------
function JambaMe:GetWarnWhenHealthDropsAmount( info )
	
	return tostring( self.db.warnWhenHealthDropsAmount )
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMe: SetWarnWhenHealthDropsAmount
-- Set the amount of health allowed before warning.
-------------------------------------------------------------------------------------------------------------
function JambaMe:SetWarnWhenHealthDropsAmount( info, value )

	local amount = tonumber( value )

	amount = self.utils:FixValueToRange( amount, 0, 100 )
		
	self.db.warnWhenHealthDropsAmount = amount
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMe: GetWarnHealthDropsMessage
-- Get the warn health drops message.
-------------------------------------------------------------------------------------------------------------
function JambaMe:GetWarnHealthDropsMessage( info )
	
	return self.db.warnHealthDropsMessage
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMe: SetWarnHealthDropsMessage
-- Set the warn health drops message.
-------------------------------------------------------------------------------------------------------------
function JambaMe:SetWarnHealthDropsMessage( info, value )
	
	self.db.warnHealthDropsMessage = value
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMe: CanWarnManaDropsBelowX
-- Return the can warn when mana drops below X flag.
-------------------------------------------------------------------------------------------------------------
function JambaMe:CanWarnManaDropsBelowX( info )

    return self.db.warnWhenManaDropsBelowX
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaMe: ToggleWarnManaDropsBelowX
-- Toggle the can warn when mana drops below X flag.
-------------------------------------------------------------------------------------------------------------
function JambaMe:ToggleWarnManaDropsBelowX( info, value )

    self.db.warnWhenManaDropsBelowX = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaMe: GetWarnWhenManaDropsAmount
-- Get the amount of mana allowed before warning.
-------------------------------------------------------------------------------------------------------------
function JambaMe:GetWarnWhenManaDropsAmount( info )
	
	return tostring( self.db.warnWhenManaDropsAmount )
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMe: SetWarnWhenManaDropsAmount
-- Set the amount of mana allowed before warning.
-------------------------------------------------------------------------------------------------------------
function JambaMe:SetWarnWhenManaDropsAmount( info, value )

	local amount = tonumber( value )

	amount = self.utils:FixValueToRange( amount, 0, 100 )
		
	self.db.warnWhenManaDropsAmount = amount
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMe: GetWarnManaDropsMessage
-- Get the warn mana drops message.
-------------------------------------------------------------------------------------------------------------
function JambaMe:GetWarnManaDropsMessage( info )
	
	return self.db.warnManaDropsMessage
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMe: SetWarnManaDropsMessage
-- Set the warn mana drops message.
-------------------------------------------------------------------------------------------------------------
function JambaMe:SetWarnManaDropsMessage( info, value )
	
	self.db.warnManaDropsMessage = value
	
end

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- ME PROCESSING
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-------------------------------------------------------------------------------------------------------------
-- JambaMe:ProcessUnitHealth
-- Process the unit health event.
-------------------------------------------------------------------------------------------------------------
function JambaMe:ProcessUnitHealth()

	local unitAffected = arg1
	
	-- If warn if unit health is low on...
	if self.db.warnWhenHealthDropsBelowX then
		
		-- If this health message is about the player...
		if unitAffected == "player" then
		
			-- Get the players current health and tell the master if it is below the warning level.
			local currentHealth = (UnitHealth( "player" ) / UnitHealthMax( "player" ) * 100)
			if self.toldMasterAboutHealth then
				if currentHealth >= self.db.warnWhenHealthDropsAmount then
					self.toldMasterAboutHealth = false
				end
			else
				if currentHealth < self.db.warnWhenHealthDropsAmount then
					self.toldMasterAboutHealth = true
					self.comms:WarnMaster( self.db.warnHealthDropsMessage )		
                end
            end
            
        end
        
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMe:ProcessUnitMana
-- Process the unit mana event.
-------------------------------------------------------------------------------------------------------------
function JambaMe:ProcessUnitMana()

	local unitAffected = arg1
	
	-- If warn if unit mana is low on...
	if self.db.warnWhenManaDropsBelowX then
		
		-- If this mana message is about the player...
		if unitAffected == "player" then
		
			-- Get the players current mana and tell the master if it is below the warning level.
			local currentMana = (UnitMana( "player" ) / UnitManaMax( "player" ) * 100)
			if self.toldMasterAboutMana then
				if currentMana >= self.db.warnWhenManaDropsAmount then
					self.toldMasterAboutMana = false
				end
			else
				if currentMana < self.db.warnWhenManaDropsAmount then
					self.toldMasterAboutMana = true
					self.comms:WarnMaster( self.db.warnManaDropsMessage )		
                end
            end
            
        end
        
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMe:ProcessPlayerRegenEnabled
-- Process the player regen enabled event.
-------------------------------------------------------------------------------------------------------------
function JambaMe:ProcessPlayerRegenEnabled()

	self.haveBeenHit = false
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMe:ProcessPlayerRegenDisabled
-- Process the player regen disabled event.
-------------------------------------------------------------------------------------------------------------
function JambaMe:ProcessPlayerRegenDisabled()

	self.haveBeenHit = false
	
	-- Entering combat, warn the master if the master is not the characters target?
	if self.db.warnTargetNotMasterEnterCombat then
		if not self.comms:CharacterIsTheMaster( self.playerName ) then
			if UnitName( "target" ) ~= self.comms:GetMasterName() then
				self.comms:WarnMaster( self.db.warnTargetNotMasterMessage )
			end
		end
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMe:ProcessUnitCombat
-- Process the unit combat event.
-------------------------------------------------------------------------------------------------------------
function JambaMe:ProcessUnitCombat()

	local unitAffected = arg1
	local action = arg2
	
	-- Warn the master if hit for the first time in combat?
	if self.db.warnHitFirstTimeCombat then
		if not self.comms:CharacterIsTheMaster( self.playerName ) then
			--TODO: test
			-- Only warn if in combat.
			if InCombatLockdown() then
				-- Is the player affected and not healed and has not been hit before?
				if unitAffected == "player" and action ~= "HEAL" and not self.haveBeenHit then
					-- Yes, set the have been hit flag (it's happening now...)
					self.haveBeenHit = true
					self.comms:WarnMaster( self.db.hitFirstTimeMessage )
				end
			end
		end		
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaMe:ProcessResurrectRequest
-- Process the resurrect request event.
-------------------------------------------------------------------------------------------------------------
function JambaMe:ProcessResurrectRequest( requester )

	if self.db.autoAcceptResurrectRequest then
		AcceptResurrect()
		StaticPopup_Hide( "RESURRECT_NO_SICKNESS" )
	end
	
end
