-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- Jamba - Jafula's Awesome Multi-Boxer's Assistant Addon.
-- JambaGroup - Handles character's group interaction.
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
local MAJOR, MINOR = "JambaGroup-1.0", 0.1
local JambaGroup = LibStub:NewLibrary(MAJOR, MINOR)
if not JambaGroup then 
	return 
end

-------------------------------------------------------------------------------------------------------------
-- Locale.
-------------------------------------------------------------------------------------------------------------
local L = LibStub( "AceLocale-3.0" ):GetLocale( "Jamba" )

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- INITIALIZE, OPTIONS AND CONFIGURATION
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-------------------------------------------------------------------------------------------------------------
-- JambaGroup:Initialize
-- Initializes JambaGroup.
-------------------------------------------------------------------------------------------------------------
function JambaGroup:Initialize( dbOptions )

	-- Initialise utility functions and constants.
	self.utils = LibStub( "JambaUtils-1.0" )
	
	-- Use the global JambaComms for communication.
	self.comms = JambaComms

	-- Remember the player name.
	self.playerName = UnitName( "player" )

	-- Sets the options database used to store user options.
	self.db = dbOptions

	-- Initialize the options configuration.
	LibStub( "AceConfig-3.0" ):RegisterOptionsTable( "Jamba-Group", self:GetOptionsConfiguration() )
	self.optionsFrame = LibStub( "AceConfigDialog-3.0" ):AddToBlizOptions( "Jamba-Group", "Jamba-Group" )
				
end

-------------------------------------------------------------------------------------------------------------
-- JambaGroup:OnEnable
-- Called when the addon is enabled.
-------------------------------------------------------------------------------------------------------------
function JambaGroup:OnEnable()

	-- Update the list of friends from the server.
	ShowFriends()

end

-------------------------------------------------------------------------------------------------------------
-- JambaGroup:GetOptionsConfiguration
-- Returns the configuration options table for JambaGroup.
-------------------------------------------------------------------------------------------------------------
function JambaGroup:GetOptionsConfiguration()

	if not self.configuration then
	
		self.configuration = {
			name = "Jamba-Group",
			handler = JambaGroup,
			type = 'group',
			args = {
				autoAcceptMembersOnly = {
					type = "toggle",
					name = L["Auto Accept Members"],
					desc = L["Automatically accept group invites from enabled members."],
					get = "IsAutoAcceptMembersOnly",
					set = "ToggleAutoAcceptMembersOnly",
					width = "full",
					order = 1,
					cmdHidden = true,
				},
				autoAcceptFriendsOnly = {
					type = "toggle",
					name = L["Auto Accept Friends"],
					desc = L["Automatically accept group invites from friends."],
					get = "IsAutoAcceptFriendsOnly",
					set = "ToggleAutoAcceptFriendsOnly",
					width = "full",
					order = 2,
					cmdHidden = true,
				},										
				autoDecline = {
					type = "toggle",
					name = L["Auto Decline Stranger Invites"],
					desc = L["Automatically decline group invites from strangers."],
					get = "IsAutoDeclineStrangers",
					set = "ToggleAutoDeclineStrangers",
					width = "full",
					order = 3,
					cmdHidden = true,
				},    
				autoFFA = {
					type = "toggle",
					name = L["Auto Set Loot To Free For All"],
					desc = L["When the party leader is this character or the loot method changes, automatically set the party loot method to 'free for all'."],
					get = "CanAutoSetLootToFreeForAll",
					set = "ToggleAutoSetLootToFreeForAll",
					width = "full",
					order = 4,
					cmdHidden = true,
				}, 
				promoteLeader = {
					type = "toggle",
					name = L["Auto Promote Master To Leader"],
					desc = L["Promote the master to the party leader when the master changes."],
					get = "CanPromoteToLeaderOnMasterChange",
					set = "TogglePromoteToLeaderOnMasterChange",
					width = "full",
					order = 5,
					cmdHidden = true,
				}, 				
				autoPassLoot = {
					type = "toggle",
					name = L["Auto Pass On Loot Rolls"],
					desc = L["Automatically pass on loot rolls."],
					get = "CanAutoPassOnLootRolls",
					set = "ToggleAutoPassOnLootRolls",
					width = "full",
					order = 6,
					cmdHidden = true,
				}, 				
				setOptOutLoot = {
					type = "toggle",
					name = L["Opt Out Of Loot"],
					desc = L["When in a party, automatically set opt out of loot to on."],
					get = "CanAutoOptOutLoot",
					set = "ToggleAutoOptOutLoot",
					width = "full",
					order = 7,
					cmdHidden = true,
				}, 
				invite = {
					type = "execute",
					name = L["Invite Team"],
					desc = L["Invite all the enabled members in the team to a group.  You can also use /jamba-group invite."],
					func = "InviteTeamToGroup",
					order = 8,
				},	
				disband = {
					type = "execute",
					name = L["Disband Team"],
					desc = L["All the enabled members in the team leave their group.  You can also use /jamba-group disband."],
					func = "DisbandTeamFromGroup",
					order = 9,
				},					
				pushSettingsToOtherCharacters = {
					type = "execute",
					name = L["Push To Characters"],
					desc = L["Push these settings to all characters that are enabled (in Jamba) and online."],
					func = "PushSettingsToEnabledCharacters",
					order = 10,		
					cmdHidden = true,	
				},													
			},
		}
		
	end
	
	return self.configuration
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaGroup:GetOptionsFrame
-- Returns the blizzard interface options frame created for JambaGroup.
-------------------------------------------------------------------------------------------------------------
function JambaGroup:GetOptionsFrame()
	
	return self.optionsFrame
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaGroup:RefreshOptions
-- Refresh the JambaGroup options dialog.
-------------------------------------------------------------------------------------------------------------
function JambaGroup:RefreshOptions()
	
	LibStub( "AceConfigRegistry-3.0" ):NotifyChange( "Jamba-Group" )
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaGroup:PushSettingsToEnabledCharacters
-- Push the settings for JambaGroup to other enabled and connected characters.
-------------------------------------------------------------------------------------------------------------
function JambaGroup:PushSettingsToEnabledCharacters()

	-- Push the settings to all.
	self.comms:PushSettings( self.db, "Jamba-Group" )
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaGroup:ReceiveSettings
-- Receive settings for JambaGroup.
-------------------------------------------------------------------------------------------------------------
function JambaGroup:ReceiveSettings( settings )

	-- Set the new settings.
	self.db.autoAcceptFriendsOnly = settings.autoAcceptFriendsOnly
    self.db.autoAcceptMembersOnly = settings.autoAcceptMembersOnly
    self.db.autoDecline = settings.autoDecline
	self.db.autoSetLootToFreeForAll = settings.autoSetLootToFreeForAll
	self.db.autoPassOnLootRolls = settings.autoPassOnLootRolls
	self.db.setOptOutOfLootRolls = settings.setOptOutOfLootRolls 
	self.db.promoteToLeaderOnMasterChange = settings.promoteToLeaderOnMasterChange
	
	-- Refresh the options.
	self:RefreshOptions()
	
end

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- CONFIGURATION CALLBACKS
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-------------------------------------------------------------------------------------------------------------
-- JambaGroup: CanAutoSetLootToFreeForAll
-- Return the can auto set loot to free for all.
-------------------------------------------------------------------------------------------------------------
function JambaGroup:CanAutoSetLootToFreeForAll( info )

    return self.db.autoSetLootToFreeForAll
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaGroup: ToggleAutoSetLootToFreeForAll
-- Toggle the can auto set loot to free for all.
-------------------------------------------------------------------------------------------------------------
function JambaGroup:ToggleAutoSetLootToFreeForAll( info, value )

    self.db.autoSetLootToFreeForAll = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaGroup: CanPromoteToLeaderOnMasterChange
-- Return the promote to leader on master change flag.
-------------------------------------------------------------------------------------------------------------
function JambaGroup:CanPromoteToLeaderOnMasterChange( info )

    return self.db.promoteToLeaderOnMasterChange
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaGroup: TogglePromoteToLeaderOnMasterChange
-- Toggle the promote to leader on master change flag.
-------------------------------------------------------------------------------------------------------------
function JambaGroup:TogglePromoteToLeaderOnMasterChange( info, value )

    self.db.promoteToLeaderOnMasterChange = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaGroup: CanAutoPassOnLootRolls
-- Return the can auto pass on loot rolls flag.
-------------------------------------------------------------------------------------------------------------
function JambaGroup:CanAutoPassOnLootRolls( info )

    return self.db.autoPassOnLootRolls
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaGroup: ToggleAutoPassOnLootRolls
-- Toggle the can auto pass on loot rolls flag.
-------------------------------------------------------------------------------------------------------------
function JambaGroup:ToggleAutoPassOnLootRolls( info, value )

    self.db.autoPassOnLootRolls = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaGroup: CanAutoOptOutLoot
-- Return the can opt out of loot rolls flag.
-------------------------------------------------------------------------------------------------------------
function JambaGroup:CanAutoOptOutLoot( info )

    return self.db.setOptOutOfLootRolls
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaGroup: ToggleAutoOptOutLoot
-- Toggle the can opt out of loot rolls flag.
-------------------------------------------------------------------------------------------------------------
function JambaGroup:ToggleAutoOptOutLoot( info, value )

    self.db.setOptOutOfLootRolls = value
    
end


-------------------------------------------------------------------------------------------------------------
-- JambaGroup: IsAutoAcceptMembersOnly
-- Return the auto accept friends only flag.
-------------------------------------------------------------------------------------------------------------
function JambaGroup:IsAutoAcceptMembersOnly( info )

    return self.db.autoAcceptMembersOnly
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaGroup: ToggleAutoAcceptMembersOnly
-- Toggle the auto accept friends only flag.
-------------------------------------------------------------------------------------------------------------
function JambaGroup:ToggleAutoAcceptMembersOnly( info, value )

    self.db.autoAcceptMembersOnly = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaGroup: IsAutoAcceptFriendsOnly
-- Return the auto accept friends only flag.
-------------------------------------------------------------------------------------------------------------
function JambaGroup:IsAutoAcceptFriendsOnly( info )

    return self.db.autoAcceptFriendsOnly
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaGroup: ToggleAutoAcceptFriendsOnly
-- Toggle the auto accept friends only flag.
-------------------------------------------------------------------------------------------------------------
function JambaGroup:ToggleAutoAcceptFriendsOnly( info, value )

    self.db.autoAcceptFriendsOnly = value
    
end
-------------------------------------------------------------------------------------------------------------
-- JambaGroup: IsAutoDeclineStrangers
-- Return the auto decline strangers flag.
-------------------------------------------------------------------------------------------------------------
function JambaGroup:IsAutoDeclineStrangers( info )

    return self.db.autoDecline
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaGroup: ToggleAutoDeclineStrangers
-- Toggle the auto decline strangers flag.
-------------------------------------------------------------------------------------------------------------
function JambaGroup:ToggleAutoDeclineStrangers( info, value )

    self.db.autoDecline = value
    
end

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- GROUP PROCESSING
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-------------------------------------------------------------------------------------------------------------
-- JambaGroup: OnMasterChange
-- Called when the master changes.
---------------------------------------------------------------------------
function JambaGroup:OnMasterChange()

	if self.db.promoteToLeaderOnMasterChange == true then
		if GetNumPartyMembers() > 0 and IsPartyLeader() then
			PromoteToLeader( self.comms:GetMasterName() )
		end
	end
	
end
			
-------------------------------------------------------------------------------------------------------------
-- JambaGroup: ProcessGroupInvite
-- inviter - The name of the character that invited this character.
-- Process group invite requests.  
-------------------------------------------------------------------------------------------------------------
function JambaGroup:ProcessGroupInvite(	inviter )

	-- Get the user options.
	local autoAcceptMembersOnly = self:IsAutoAcceptMembersOnly()
	local autoAcceptFriendsOnly = self:IsAutoAcceptFriendsOnly()
	local autoDecline = self:IsAutoDeclineStrangers()
		
	-- Accept this invite, initially no.
	local acceptInvite = false

	-- How many party members does this character have already?  If 0 then not in a group.
	if GetNumPartyMembers() == 0 then	
	
		-- Accept an invite from members?
		if autoAcceptMembersOnly then
			-- Inviter found in members list, allow the invite to be accepted.
			if self.comms:CharacterIsAnEnabledMember( inviter ) then
				acceptInvite = true
			end
		end			
		
		-- Accept an invite from friends?
		if autoAcceptFriendsOnly then
			-- Iterate each friend; searching for the inviter in the friends list.
			for friendIndex = 1, GetNumFriends() do
				local friendName = GetFriendInfo(friendIndex)
				-- Inviter found in friends list, allow the invite to be accepted.
				if inviter == friendName then
					acceptInvite = true
					break
				end
			end	
		end					
			
	end	

	-- Hide the party invite popup?
	local hidePopup = false;
	
	-- Accept the group invite if allowed.
	if acceptInvite then
		AcceptGroup()
		hidePopup = true;
	else
		-- Otherwise decline the invite if permitted.
		if autoDecline then
			DeclineGroup()
			hidePopup = true;
		end
	end
			
	-- Hide the popup group invitation request if accepted or declined the invite.
	if hidePopup then
		StaticPopup_Hide( "PARTY_INVITE" )
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaGroup:ProcessPartyLeaderChanged
-- Handle the party leader changed event.
-------------------------------------------------------------------------------------------------------------
function JambaGroup:ProcessPartyLeaderChanged()

	-- Automatically set the loot to free for all?
	if self.db.autoSetLootToFreeForAll then
		JambaGroup:SetPartyLootToFreeForAll()
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaGroup:ProcessPartyMembersChanged
-- Handle the party members changed event.
-------------------------------------------------------------------------------------------------------------
function JambaGroup:ProcessPartyMembersChanged()

	-- Set opt out of loot rolls?
	if self.db.setOptOutOfLootRolls then
		-- Party members changed fires once per character per event; only process this on the first time 
		-- around when the first joins the party (or event fires).
		if GetNumPartyMembers() == 1 then
			JambaGroup:SetPartyOptOutOfLoot( true )
		end
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaGroup:SetPartyOptOutOfLoot
-- Set the opt out of loot setting.
-------------------------------------------------------------------------------------------------------------
function JambaGroup:SetPartyOptOutOfLoot( toggle )

	SetOptOutOfLoot( toggle )

end

-------------------------------------------------------------------------------------------------------------
-- JambaGroup:InviteTeamToGroup
-- Invites all enable members in the team to a group.
-------------------------------------------------------------------------------------------------------------
function JambaGroup:InviteTeamToGroup()

	-- Iterate each enabled member and invite them to a group.
	for characterName, enabled in pairs( self.comms:EnabledMemberList() ) do
		if enabled then
			-- As long as they are not the player doing the inviting.
			if characterName ~= self.playerName then
				InviteUnit( characterName )			
			end
		end
	end

	-- Automatically set the loot to free for all?
	self:ProcessPartyLeaderChanged()
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaGroup:DisbandTeamFromGroup
-- All the enabled members in the team leave their group.
-------------------------------------------------------------------------------------------------------------
function JambaGroup:DisbandTeamFromGroup()

	self.comms:CommandAll( self.utils.COMMAND_LEAVE_GROUP )
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaGroup:LeaveTheGroup
-- This character leaves the group.
-------------------------------------------------------------------------------------------------------------
function JambaGroup:LeaveTheGroup()

	-- Is the character in a party?
	if GetNumPartyMembers() > 0 then
		-- Yes, leave group.
		LeaveParty()
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaGroup:SetPartyLootToFreeForAll
-- Set party loot to free for all.
-------------------------------------------------------------------------------------------------------------
function JambaGroup:SetPartyLootToFreeForAll()
	
	-- Is this character in a party and the party leader?
	if GetNumPartyMembers() > 0 and IsPartyLeader() then
		local lootMethod, partyMaster, raidMaster = GetLootMethod()
		if lootMethod ~= self.utils.PARTY_LOOT_FREEFORALL then
			SetLootMethod( self.utils.PARTY_LOOT_FREEFORALL )
		end
	end
		
end

-------------------------------------------------------------------------------------------------------------
-- JambaGroup:ProcessStartLootRoll
-- Handle the start loot roll event.
-------------------------------------------------------------------------------------------------------------
function JambaGroup:ProcessStartLootRoll( rollIdentifer, rollTime )   
	
	if self.db.autoPassOnLootRolls then
		RollOnLoot( rollIdentifer, self.utils.LOOT_ROLL_PASS )
	end
	
end