﻿-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- Jamba - Jafula's Awesome Multi-Boxer's Assistant Addon.
-- JambaFollow - Handles character's follow options.
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
local MAJOR, MINOR = "JambaFollow-1.0", 0.1
local JambaFollow = LibStub:NewLibrary(MAJOR, MINOR)
if not JambaFollow then 
	return 
end
LibStub( "AceTimer-3.0" ):Embed( JambaFollow )

-------------------------------------------------------------------------------------------------------------
-- Locale.
-------------------------------------------------------------------------------------------------------------
local L = LibStub( "AceLocale-3.0" ):GetLocale( "Jamba" )

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- INITIALIZE, OPTIONS AND CONFIGURATION
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-------------------------------------------------------------------------------------------------------------
-- JambaFollow:Initialize
-- Initializes JambaFollow.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:Initialize( dbOptions )

	-- Initialise utility functions and constants.
	self.utils = LibStub( "JambaUtils-1.0" )
	
	-- Use the global JambaComms for communication.
	self.comms = JambaComms

	-- Remember the player name.
	self.playerName = UnitName( "player" )

	-- Sets the options database used to store user options.
	self.db = dbOptions

	-- Initialize the options configuration.
	LibStub( "AceConfig-3.0" ):RegisterOptionsTable( "Jamba-Follow", self:GetOptionsConfiguration() )
	self.optionsFrame = LibStub( "AceConfigDialog-3.0" ):AddToBlizOptions( "Jamba-Follow", "Jamba-Follow" )

	-- Current follow target.
	self.currentFollowTarget = self.comms:GetMasterName()
	
	-- Set to true if jamba initiated a follow.
	self.jambaSetFollowTarget = false
	
	-- Set to true if jamba is strobing follow.
	self.followingStrobing = false
	self.followStrobeTimer = nil
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow:GetOptionsConfiguration
-- Returns the configuration options table for JambaFollow.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:GetOptionsConfiguration()

	if not self.configuration then
	
		self.configuration = {
			name = "Jamba-Follow",
			handler = JambaFollow,
			type = 'group',
			args = {
				headerFollow = {
					type = "header",
					name = L["Follow Options"],
					order = 0,
				},							
				warnWhenFollowBreaks = {
					type = "toggle",
					name = L["Warn If I Stop Following"],
					desc = L["Warn the master if I stop following."],
					get = "CanWarnWhenFollowBreaks",
					set = "ToggleWarnWhenFollowBreaks",
					width = "full",
					order = 1,
					cmdHidden = true,
				},									
				followBrokenMessage = {
					type = "input",
					name = L["Follow Broken Message"],
					desc = L["The message to tell the master when follow has broken."],
					get = "GetFollowBrokenMessage",
					set = "SetFollowBrokenMessage",
					width = "full",
					order = 2,	
					cmdHidden = true,
				},													
				autoFollowAfterCombat = {
					type = "toggle",
					name = L["Auto Follow After Combat"],
					desc = L["Automatically follow after combat."],
					get = "CanAutoFollowAfterCombat",
					set = "ToggleAutoFollowAfterCombat",
					width = "full",
					order = 3,
					cmdHidden = true,
				},	
				headerFollowStrobing = {
					type = "header",
					name = L["Follow Strobing"],
					order = 4,
				},				
				strobeFrequencySeconds = {
					type = "input",
					name = L["Seconds Between Strobe Follow"],
					desc = L["The number of seconds delay before issuing a follow command."],
					get = "GetStrobeFrequencySeconds",
					set = "SetStrobeFrequencySeconds",
					--width = "full",
					order = 5,
					cmdHidden = true,
				},	
				headerPush = {
					type = "header",
					name = L["Push To Characters"],
					order = 6,
				},
				pushSettingsToOtherCharacters = {
					type = "execute",
					name = L["Push To Characters"],
					desc = L["Push these settings to all characters that are enabled (in Jamba) and online."],
					func = "PushSettingsToEnabledCharacters",
					order = 7,		
					cmdHidden = true,					
				},											
				master = {
					type = "input",
					name = L["Follow The Master"],
					desc = L["Follow the current master."],
					usage = "/jamba-follow master <tag>",
					get = false,
					set = "FollowMasterCommand",
					guiHidden = true,
				},					
				target = {
					type = "input",
					name = L["Follow A Target"],
					desc = L["Follow the target specified."],
					usage = "/jamba-follow target <target> <tag>",
					get = false,
					set = "FollowTargetCommand",
					guiHidden = true,
				},					
				afterCombat = {
					type = "input",
					name = L["Auto Follow After Combat"],
					desc = L["Automatically follow after combat."],
					usage = "/jamba-follow aftercombat <on|off> <tag>",
					get = "CanAutoFollowAfterCombat",
					set = "AutoFollowAfterCombatCommand",
					guiHidden = true,
				},															
				strobeOn = {
					type = "input",
					name = L["Begin Follow Strobing Target."],
					desc = L["Begin a sequence of follow commands that strobe every second (configurable) a specified target."],
					usage = "/jamba-follow strobeon <target> <tag>",
					get = false,
					set = "FollowStrobeOnCommand",
					guiHidden = true,
				},	
				strobeOnMe = {
					type = "input",
					name = L["Begin Follow Strobing Me."],
					desc = L["Begin a sequence of follow commands that strobe every second (configurable) this character."],
					usage = "/jamba-follow strobeonme <tag>",
					get = false,
					set = "FollowStrobeOnMeCommand",
					guiHidden = true,
				},								
				strobeOnLast = {
					type = "input",
					name = L["Begin Follow Strobing Last Target."],
					desc = L["Begin a sequence of follow commands that strobe every second (configurable) the last follow target character."],
					usage = "/jamba-follow strobeonlast <tag>",
					get = false,
					set = "FollowStrobeOnLastCommand",
					guiHidden = true,
				},				
				strobeOff = {
					type = "input",
					name = L["End Follow Strobing."],
					desc = L["End the strobing of follow commands."],
					usage = "/jamba-follow strobeoff <tag>",
					get = false,
					set = "FollowStrobeOffCommand",
					guiHidden = true,
				},								
			},
		}
			
	end
	
	return self.configuration
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow:GetOptionsFrame
-- Returns the blizzard interface options frame created for JambaFollow.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:GetOptionsFrame()
	
	return self.optionsFrame
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow:RefreshOptions
-- Refresh the JambaFollow options dialog.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:RefreshOptions()
	
	LibStub( "AceConfigRegistry-3.0" ):NotifyChange( "Jamba-Follow" )
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow:PushSettingsToEnabledCharacters
-- Push the settings for JambaFollow to other enabled and connected characters.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:PushSettingsToEnabledCharacters()

	-- Push the settings to all.
	self.comms:PushSettings( self.db, "Jamba-Follow" )
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow:ReceiveSettings
-- Receive settings for JambaFollow.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:ReceiveSettings( settings )

	-- Set the new settings.
	self.db.warnWhenFollowBreaks = settings.warnWhenFollowBreaks
	self.db.followBrokenMessage = settings.followBrokenMessage
	self.db.autoFollowAfterCombat = settings.autoFollowAfterCombat
	self.db.strobeFrequencySeconds = settings.strobeFrequencySeconds
	
	-- Refresh the options.
	self:RefreshOptions()
	
end

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- CONFIGURATION CALLBACKS
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-------------------------------------------------------------------------------------------------------------
-- JambaFollow: CanWarnWhenFollowBreaks
-- Return the can warn when follow breaks flag.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:CanWarnWhenFollowBreaks( info )

    return self.db.warnWhenFollowBreaks
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow: ToggleWarnWhenFollowBreaks
-- Toggle the can warn when follow breaks flag.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:ToggleWarnWhenFollowBreaks( info, value )

    self.db.warnWhenFollowBreaks = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow: GetFollowBrokenMessage
-- Get the follow broken message.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:GetFollowBrokenMessage( info )
	
	return self.db.followBrokenMessage
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow: SetFollowBrokenMessage
-- Set the follow broken message.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:SetFollowBrokenMessage( info, value )
	
	self.db.followBrokenMessage = value
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow: CanAutoFollowAfterCombat
-- Return the auto follow after combat flag.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:CanAutoFollowAfterCombat( info )

    return self.db.autoFollowAfterCombat
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow: ToggleAutoFollowAfterCombat
-- Toggle the auto follow after combat flag.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:ToggleAutoFollowAfterCombat( info, value )

    self.db.autoFollowAfterCombat = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow: GetStrobeFrequencySeconds
-- Get the strobe frequency seconds.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:GetStrobeFrequencySeconds( info )

    return self.db.strobeFrequencySeconds
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow: SetStrobeFrequencySeconds
-- Set the strobe frequency seconds.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:SetStrobeFrequencySeconds( info, value )

    self.db.strobeFrequencySeconds = value
    
end

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- FOLLOW PROCESSING
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-------------------------------------------------------------------------------------------------------------
-- JambaFollow:ProcessAutoFollowBegin
-- Process the auto follow begin event.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:ProcessAutoFollowBegin()

	-- If warn if auto follow breaks is on...
	if self.db.warnWhenFollowBreaks then
		-- Nothing to do.
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow:ProcessAutoFollowEnd
-- Process the auto follow end event.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:ProcessAutoFollowEnd()

	local followEndedBecause = arg1
	
	-- If warn if auto follow breaks is on...
	if self.db.warnWhenFollowBreaks then
		if not followEndedBecause then
			if self.jambaSetFollowTarget == false then
				self.comms:WarnMaster( self.db.followBrokenMessage )
			end
		end
	end
	
	self.jambaSetFollowTarget = false		
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow:ProcessPlayerRegenEnabled
-- Process the player regen enabled event.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:ProcessPlayerRegenEnabled()

	self.outOfCombat = true
	
	-- Is auto follow after combat on?
	if self.db.autoFollowAfterCombat then
		self:FollowTarget( self.currentFollowTarget )
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow:ProcessPlayerRegenDisabled
-- Process the player regen disabled event.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:ProcessPlayerRegenDisabled()

	self.outOfCombat = false

end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow:AutoFollowAfterCombatCommand
-- Auto follow after combat command handler.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:AutoFollowAfterCombatCommand( info, parameters )

	-- Get the on/off state and the tag of who to send to.
	local state, tag = strsplit( " ", parameters )
					
	if tag then
		self:AutoFollowAfterCombatSendCommand( state, tag )
	else
		self:DoToggleAutoFollowAfterCombat( state )
	end	
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow:AutoFollowAfterCombatSendCommand
-- Send the auto follow after combat command.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:AutoFollowAfterCombatSendCommand( state, tag )

	self.comms:CommandAll( self.utils.COMMAND_AUTO_FOLLOW_AFTER_COMBAT, state, tag )

end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow:AutoFollowAfterCombatReceiveCommand
-- Receive and handle the auto follow after combat command.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:AutoFollowAfterCombatReceiveCommand( state, tag )

	-- If this character responds to this tag...
	if self.comms:CharacterHasTag( self.playerName, tag ) then
		self:DoToggleAutoFollowAfterCombat( state )
	end

end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow:DoToggleAutoFollowAfterCombat
-- Do the auto follow after combat command.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:DoToggleAutoFollowAfterCombat( state )

	-- Translate the on/off state from string to boolean/nil.
	local setToOn = self.utils:GetOnOrOffFromCommand( state, L["on"], L["off"] )
	
	-- If nil, then assume false.
	if setToOn == nil then
		setToOn = fase
	end		
	
	-- Then set the flag appropriately.
	self:ToggleAutoFollowAfterCombat( nil, setToOn )
	
	-- Refresh the options.
	self:RefreshOptions()
		
end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow:FollowMasterCommand
-- Follow the master command handler.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:FollowMasterCommand( info, parameters )

	-- The only parameter for this command is tag.  If there is a tag, send the command to all
	-- the members, otherwise just this character.
	local tag = parameters
	
	-- Set the current follow target to the master.
	self.currentFollowTarget = self.comms:GetMasterName()
	
	if tag then
		self:FollowTargetSendCommand( self.currentFollowTarget, tag )
	else
		self:FollowTarget( self.currentFollowTarget )
	end	

end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow:FollowTargetCommand
-- Follow a target command handler.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:FollowTargetCommand( info, parameters )

	local target, tag = strsplit( " ", parameters )
		
	if tag then
		self:FollowTargetSendCommand( target, tag )
	else
		self:FollowTarget( target )
	end	

end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow:FollowTargetSendCommand
-- Send the follow target command.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:FollowTargetSendCommand( target, tag )

	self.comms:CommandAll( self.utils.COMMAND_FOLLOW_TARGET, target, tag )

end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow:FollowTargetReceiveCommand
-- Receive and handle the follow target command.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:FollowTargetReceiveCommand( target, tag )

	-- If this character responds to this tag...
	if self.comms:CharacterHasTag( self.playerName, tag ) then
		-- Then follow the target specified.
		self:FollowTarget( target )
	end

end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow:FollowStrobeOnMeCommand
-- Follow me strobing on command handler.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:FollowStrobeOnMeCommand( info, parameters )

	local tag = parameters
	
	if tag then
		self:FollowStrobeOnSendCommand( self.playerName, tag )
	else
		self:FollowStrobeOn( self.playerName )
	end	

end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow:FollowStrobeOnLastCommand
-- Follow a target strobing on last command handler.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:FollowStrobeOnLastCommand( info, parameters )

	local tag = parameters
	
	if tag then
		self:FollowStrobeOnSendCommand( self.currentFollowTarget, tag )
	else
		self:FollowStrobeOn( self.currentFollowTarget )
	end	

end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow:FollowStrobeOnCommand
-- Follow a target strobing on command handler.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:FollowStrobeOnCommand( info, parameters )

	local target, tag = strsplit( " ", parameters )
	
	if tag then
		self:FollowStrobeOnSendCommand( target, tag )
	else
		self:FollowStrobeOn( target )
	end	

end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow:FollowStrobeOnSendCommand
-- Send the follow strobe command.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:FollowStrobeOnSendCommand( target, tag )

	self.comms:CommandAll( self.utils.COMMAND_FOLLOW_STROBE_ON, target, tag )

end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow:FollowStrobeOnReceiveCommand
-- Receive and handle the follow strobe command.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:FollowStrobeOnReceiveCommand( target, tag )

	-- If this character responds to this tag...
	if self.comms:CharacterHasTag( self.playerName, tag ) then
		-- Then follow the target specified - strobing.
		self:FollowStrobeOn( target )
	end

end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow:FollowStrobeOffCommand
-- Follow a target strobe off command handler.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:FollowStrobeOffCommand( info, parameters )

	local tag = parameters
	
	if tag then
		self:FollowStrobeOffSendCommand( tag )
	else
		self:FollowStrobeOff()
	end	

end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow:FollowStrobeOffSendCommand
-- Send the follow strobe off command.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:FollowStrobeOffSendCommand( tag )

	self.comms:CommandAll( self.utils.COMMAND_FOLLOW_STROBE_OFF, tag )

end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow:FollowStrobeOffReceiveCommand
-- Receive and handle the follow strobe off command.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:FollowStrobeOffReceiveCommand( tag )

	-- If this character responds to this tag...
	if self.comms:CharacterHasTag( self.playerName, tag ) then
		-- Then follow the target specified - turn off strobing.
		self:FollowStrobeOff()
	end

end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow:FollowTarget
-- Follow the target specified.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:FollowTarget( target )

    -- Set the jamba set this flag toggle, so not to complain about follow broken after combat.
	self.jambaSetFollowTarget = true
	
	-- Follow unit only works when in a party or raid for resolving against player names.
	FollowUnit( target )
	
	-- Remember this unit as the current follow target.
	self.currentFollowTarget = target

end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow:FollowStrobeOn
-- Follow the target specified strobing the command.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:FollowStrobeOn( target )

	-- Do the initial follow.
    self:FollowTarget( target )

	-- If the timer is running, then 
	if self.followingStrobing == true then
		self:FollowStrobeOff( target )
	end
	
	-- Set up a timer to do another follow command.
	self.followingStrobing = true
	self.followStrobeTimer = self:ScheduleRepeatingTimer( "FollowTarget", tonumber( self.db.strobeFrequencySeconds ), target )

end

-------------------------------------------------------------------------------------------------------------
-- JambaFollow:FollowStrobeOff
-- Turn off following the target specified strobing the command.
-------------------------------------------------------------------------------------------------------------
function JambaFollow:FollowStrobeOff()

	-- Stop the timer from doing another follow command.
	if self.followingStrobing == true then
		self.followingStrobing = false
		self:CancelTimer( self.followStrobeTimer )
	end
	
end
