-------------------------------------------------------------------------------------------------------------
-- Jamba - Jafula's Awesome Multi-Boxer's Assistant Addon.
-- JambaCore - The glue that binds the Jamba goodness together.
-- Controls the user options database, events, slash commands and calls the other parts as needed.
-------------------------------------------------------------------------------------------------------------
Jamba = LibStub( "AceAddon-3.0" ):NewAddon( "Jamba", "AceConsole-3.0", "AceEvent-3.0", "AceComm-3.0", "AceHook-3.0" )

-------------------------------------------------------------------------------------------------------------
-- Locale.
-------------------------------------------------------------------------------------------------------------
local L = LibStub( "AceLocale-3.0" ):GetLocale( "Jamba" )

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- INITIALISE, ENABLE AND DISABLE
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-- Jamba key bindings.
BINDING_HEADER_JAMBA = L["Jamba"]
BINDING_NAME_JAMBASETFOCUS = L["Set Focus"]
BINDING_NAME_JAMBASETMASTERASTARGET = L["Set Master As Target"]

-------------------------------------------------------------------------------------------------------------
-- Jamba:InitializePreferences()
-- Initialises user preferences and associated defaults.
-------------------------------------------------------------------------------------------------------------
function Jamba:InitializePreferences()

    -- Preferences - the values to store and their defaults for the options database.
    local preferences = {
        char = {
            communication = {
                master = "",
                memberList = {},
                memberEnabled = {},
                memberTagList = {},
				displayWarningInDefaultChat = true,
				displayWarningInRaidWarning = true,	
				sendWarningToChannel = false,
				warningChannelName = "",
				warningChannelPassword = "",		
				sendWarningToGuildChat = false,	
				sendWarningToGuildOfficerChat = false,
				sendWarningToPartyChat = false,	
				sendWarningToMasterViaWhisper = false,
				displayMessageInDefaultChat = true,
				sendMessageToChannel = false,
				messageChannelName = "",
				messageChannelPassword = "",		
				sendMessageToGuildChat = false,	
				sendMessageToGuildOfficerChat = false,
				sendMessageToPartyChat = false,	
				sendMessageToMasterViaWhisper = false,		
				displaySpecialMessageInDefaultChat = true,		
				setFocusToMasterOnMasterChange = false,
                setTargetToMasterOnMasterChange = false,
                changeMasterOnFocusChange = false,
            },
            me = {
				warnHitFirstTimeCombat = true,
				hitFirstTimeMessage = L["I'm Attacked!"],
				warnTargetNotMasterEnterCombat = true,
				warnTargetNotMasterMessage = L["Not Targeting!"],
				warnWhenHealthDropsBelowX = true,
				warnWhenHealthDropsAmount = 50,
				warnHealthDropsMessage = L["Low Health!"],
				warnWhenManaDropsBelowX = true,
				warnWhenManaDropsAmount = 15,
				warnManaDropsMessage = L["Low Mana!"],
				autoAcceptResurrectRequest = true,
            },
            group = {
                autoAcceptMembersOnly = true,
                autoAcceptFriendsOnly = true,
                autoDecline = false,
                autoSetLootToFreeForAll = true,
				autoPassOnLootRolls = false,   
				setOptOutOfLootRolls = false,   
				promoteToLeaderOnMasterChange = false,          
            },              
            talk = {
                forwardWhispers = true,
                requestWhoInformation = true,
                forwardSystemMessages = true,
                systemMessagesToFilter = { [L["Level"]] = L["Level"] },
                chatSnippets = {},
            },
            merchant = {
                autoRepair = true,
                autoRepairUseGuildFunds = true,
                autoBuy = false,
                autoBuyOverflow = false,
                doMerchantBusinessQuietly = false,
                autoBuyItems = {},
                autoSell = false,
            },
            taxi = {
                takeMastersTaxi = true,
            },
            quest = {
                mirrorMasterQuestSelection = true,
                slaveAutoAcceptQuest = false,
                slaveAutoAcceptEscortQuest = true,
                mirrorMasterDeclineQuest = true,
                mirrorMasterAbandonQuest = true,
                mirrorMasterCompleteQuestNoReward = true,
                mirrorMasterCompleteQuestWithReward = false,
                slaveAutoCompleteQuest = false,
				mirrorMasterQuestWatch = true,
                questSilently = false,
                show = false,
                showJambaQuestLogWithWowLog = true,
            },
            justSayNo = {
                autoDenyDuels = true,
                autoDenyDuelsSilently = false,
                autoDenyGuildInvites = true,
                autoDenyGuildInvitesSilently = false,
            },
            follow = {
				warnWhenFollowBreaks = true, 
				followBrokenMessage = L["Follow Broken!"],
				autoFollowAfterCombat = false,  
				strobeFrequencySeconds = "1",
            },
        },
    }

    -- Create the options database supplying the options values along with defaults (preferences).
    self.db = LibStub( "AceDB-3.0" ):New( "JambaDB2", preferences )
    
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:OnInitialize()
-- Initial setup when Jamba is first loaded.  Builds options database, configuration database and registers
-- slash commands.
-------------------------------------------------------------------------------------------------------------
function Jamba:OnInitialize()

    -- Initialise utility functions and constants.
    self.utils = LibStub( "JambaUtils-1.0" )

    -- Initialises user preferences and associated defaults.
    self:InitializePreferences()
    
    -- Initialise JambaComms, set option store, option configuration and slash command.
    JambaComms = LibStub( "JambaComms-1.0" )
    JambaComms:Initialize( self.db.char.communication )
    self:RegisterChatCommand( "jamba", "JambaCommsChatCommand" )
    self:RegisterComm( self.utils.COMMUNICATION_PREFIX, "JambaCommsMessageReceived" )
    self:RegisterComm( self.utils.COMMAND_PREFIX, "JambaCommsCommandReceived" )
    self:RegisterComm( self.utils.WARNING_PREFIX, "JambaCommsWarningReceived" )
    
    -- Set communication items for JambaCore.
    self.comms = JambaComms
    self.playerName = UnitName( "player" )

    -- Initialise JambaGroup, set option store, option configuration and slash command.
    JambaGroup = LibStub( "JambaGroup-1.0", true )
    if JambaGroup then
        JambaGroup:Initialize( self.db.char.group )
        self:RegisterChatCommand( "jamba-group", "JambaGroupChatCommand" )
    end
    
    -- Initialise JambaMerchant, set option store, option configuration and slash command.
    JambaMerchant = LibStub( "JambaMerchant-1.0", true )
    if JambaMerchant then
        JambaMerchant:Initialize( self.db.char.merchant )
        self:RegisterChatCommand( "jamba-merchant", "JambaMerchantChatCommand" )
    end

    -- Initialise JambaTalk, set option store, option configuration and slash command.
    JambaTalk = LibStub( "JambaTalk-1.0", true )
    if JambaTalk then
        JambaTalk:Initialize( self.db.char.talk )
        self:RegisterChatCommand( "jamba-talk", "JambaTalkChatCommand" )
    end

    -- Initialise JambaTaxi, set option store, option configuration and slash command.
    JambaTaxi = LibStub( "JambaTaxi-1.0", true )
    if JambaTaxi then
        JambaTaxi:Initialize( self.db.char.taxi )
        self:RegisterChatCommand( "jamba-taxi", "JambaTaxiChatCommand" )
    end

    -- Initialise JambaQuest, set option store, option configuration and slash command.
    JambaQuest = LibStub( "JambaQuest-1.0", true )
    if JambaQuest then
        JambaQuest:Initialize( self.db.char.quest )
        self:RegisterChatCommand( "jamba-quest", "JambaQuestChatCommand" )
    end

    -- Initialise JambaNo, set option store, option configuration and slash command.
    JambaNo = LibStub( "JambaNo-1.0", true )
    if JambaNo then
        JambaNo:Initialize( self.db.char.justSayNo )
        self:RegisterChatCommand( "jamba-no", "JambaNoChatCommand" )
    end

    -- Initialise JambaMe, set option store, option configuration and slash command.
    JambaMe = LibStub( "JambaMe-1.0", true )
    if JambaMe then
        JambaMe:Initialize( self.db.char.me )
        self:RegisterChatCommand( "jamba-me", "JambaMeChatCommand" )
    end

    -- Initialise JambaFollow, set option store, option configuration and slash command.
    JambaFollow = LibStub( "JambaFollow-1.0", true )
    if JambaFollow then
        JambaFollow:Initialize( self.db.char.follow )
        self:RegisterChatCommand( "jamba-follow", "JambaFollowChatCommand" )
    end

end

-------------------------------------------------------------------------------------------------------------
-- Jamba:OnEnable()
-- Enable Jamba.  Registers for wow events.
-------------------------------------------------------------------------------------------------------------
function Jamba:OnEnable()

	-- Call on enable for modules that need it.
	if JambaComms then
		JambaComms:OnEnable()
	end
	if JambaGroup then
		JambaGroup:OnEnable()
	end	

	-- If JambaComms is loaded then...
    if JambaComms then
        -- Comms events.
        self:RegisterEvent( "PLAYER_FOCUS_CHANGED" )
    end
   				
    -- If JambaMerchant is loaded then...
    if JambaMerchant then
        -- Merchant events.
        self:RegisterEvent( "MERCHANT_SHOW" )
    end
    
    -- If JambaGroup is loaded then...
    if JambaGroup then
        -- Group events.
        self:RegisterEvent( "PARTY_INVITE_REQUEST" )
        self:RegisterEvent( "PARTY_LEADER_CHANGED" )
        self:RegisterEvent( "PARTY_MEMBERS_CHANGED" )
    end
    
    -- If JambaTalk is loaded then...
    if JambaTalk then
        -- Talk events.
        self:RegisterEvent( "CHAT_MSG_SYSTEM" )
        self:RegisterEvent( "CHAT_MSG_WHISPER" )
    end

    -- If JambaTaxi is loaded then...
    if JambaTaxi then
        -- Taxi hooks.
		self:SecureHook( "TakeTaxiNode", "JambaHookTaxiTakeTaxiNode" )
    end

    -- If JambaQuest is loaded then...
    if JambaQuest then
        -- Quest events.
        self:RegisterEvent( "QUEST_DETAIL" )
        self:RegisterEvent( "QUEST_ACCEPT_CONFIRM" )
        self:RegisterEvent( "QUEST_COMPLETE" )
        -- Quest post hooks.
        self:SecureHook( "SelectGossipOption", "JambaHookQuestSelectGossipOption" )
        self:SecureHook( "SelectGossipActiveQuest", "JambaHookQuestSelectGossipActiveQuest" )
        self:SecureHook( "SelectGossipAvailableQuest", "JambaHookQuestSelectGossipAvailableQuest" )
        self:SecureHook( "SelectActiveQuest", "JambaHookQuestSelectActiveQuest" )
        self:SecureHook( "SelectAvailableQuest", "JambaHookQuestSelectAvailableQuest" )
        self:SecureHook( "AcceptQuest", "JambaHookQuestAcceptQuest" )
        self:SecureHook( "CompleteQuest", "JambaHookQuestCompleteQuest" )
        self:SecureHook( "GetQuestReward", "JambaHookQuestGetQuestReward" )
        self:SecureHook( "DeclineQuest", "JambaHookQuestDeclineQuest" )
		self:SecureHook( "SelectQuestLogEntry", "JambaHookQuestSelectQuestLogEntry" )
		self:SecureHook( "ToggleQuestLog", "JambaHookQuestToggleQuestLog" )
    end

    -- If JambaNo is loaded then...
    if JambaNo then
        -- Just say no events.
        self:RegisterEvent( "DUEL_REQUESTED" )
        self:RegisterEvent( "GUILD_INVITE_REQUEST" )
    end

	-- If JambaMe is loaded then...
	if JambaMe then
		-- Warning events.
		self:RegisterEvent( "UNIT_HEALTH" )
		self:RegisterEvent( "UNIT_MANA" )
		self:RegisterEvent( "RESURRECT_REQUEST" )
		self:RegisterEvent( "START_LOOT_ROLL" )
		self:RegisterEvent( "UNIT_COMBAT" )
	end

	-- If JambaFollow is loaded then...
	if JambaFollow then
		-- Fellow events.
		self:RegisterEvent( "AUTOFOLLOW_BEGIN" )
		self:RegisterEvent( "AUTOFOLLOW_END" )
	end
	
	-- Unit combat and player regen (follow and me).
	if JambaFollow or JambaMe then
		self:RegisterEvent( "PLAYER_REGEN_DISABLED" )
		self:RegisterEvent( "PLAYER_REGEN_ENABLED" )
	end
				
	-- Initialise key bindings.
	self.keyBindingFrame = CreateFrame( "Frame", nil, UIParent )
	self:RegisterEvent( "UPDATE_BINDINGS" )	
	ClearOverrideBindings( self.keyBindingFrame )
	
	-- Key bindings for JambaComms.
	if JambaComms then
		JambaComms:UpdateKeyBindings( self.keyBindingFrame )
	end

end

-------------------------------------------------------------------------------------------------------------
-- Jamba:OnDisable()
-- Disable Jamba.  Does nothing.
-------------------------------------------------------------------------------------------------------------
function Jamba:OnDisable()

    -- Does nothing.
    
end

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- COMMUNICATION HANDLERS
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-------------------------------------------------------------------------------------------------------------
-- Jamba:JambaCommsMessageSend()
-- Handle sending messages on communication channels.
-------------------------------------------------------------------------------------------------------------
function Jamba:JambaCommsMessageSend( ... )

    self:SendCommMessage( ... )
        
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:JambaCommsMessageReceived()
-- Handle incoming messages on communication channels.
-------------------------------------------------------------------------------------------------------------
function Jamba:JambaCommsMessageReceived( ... )

    self.comms:MessageReceived( ... )
        
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:JambaCommsCommandReceived()
-- Handle incoming commands on communication channels.
-------------------------------------------------------------------------------------------------------------
function Jamba:JambaCommsCommandReceived( ... )

    self.comms:CommandReceived( ... )
        
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:JambaCommsWarningReceived()
-- Handle incoming warnings on communication channels.
-------------------------------------------------------------------------------------------------------------
function Jamba:JambaCommsWarningReceived( ... )

    self.comms:WarningReceived( ... )
        
end

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- CHAT COMMAND HANDLERS
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-------------------------------------------------------------------------------------------------------------
-- Jamba:JambaCommsChatCommand()
-- Handle the jamba chat command.
-------------------------------------------------------------------------------------------------------------
function Jamba:JambaCommsChatCommand( input )

    if not input or input:trim() == "" then
        InterfaceOptionsFrame_OpenToFrame( JambaComms:GetOptionsFrame() )
    else
        LibStub( "AceConfigCmd-3.0" ).HandleCommand( JambaComms, "jamba", "Jamba-Comms", input )
    end
    
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:JambaGroupChatCommand()
-- Handle the group chat command.
-------------------------------------------------------------------------------------------------------------
function Jamba:JambaGroupChatCommand( input )

    if not input or input:trim() == "" then
        InterfaceOptionsFrame_OpenToFrame( JambaGroup:GetOptionsFrame() )
    else
        LibStub( "AceConfigCmd-3.0" ).HandleCommand( JambaGroup, "jamba-group", "Jamba-Group", input )
    end
    
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:JambaMerchantChatCommand()
-- Handle the merchant chat command.
-------------------------------------------------------------------------------------------------------------
function Jamba:JambaMerchantChatCommand( input )

    if not input or input:trim() == "" then
        InterfaceOptionsFrame_OpenToFrame( JambaMerchant:GetOptionsFrame() )
    else
        LibStub( "AceConfigCmd-3.0" ).HandleCommand( JambaMerchant, "jamba-merchant", "Jamba-Merchant", input )
    end
    
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:JambaTalkChatCommand()
-- Handle the talk chat command.
-------------------------------------------------------------------------------------------------------------
function Jamba:JambaTalkChatCommand( input )

    if not input or input:trim() == "" then
        InterfaceOptionsFrame_OpenToFrame( JambaTalk:GetOptionsFrame() )
    else
        LibStub( "AceConfigCmd-3.0" ).HandleCommand( JambaTalk, "jamba-talk", "Jamba-Talk", input )
    end
    
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:JambaTaxiChatCommand()
-- Handle the taxi chat command.
-------------------------------------------------------------------------------------------------------------
function Jamba:JambaTaxiChatCommand( input )

    if not input or input:trim() == "" then
        InterfaceOptionsFrame_OpenToFrame( JambaTaxi:GetOptionsFrame() )
    else
        LibStub( "AceConfigCmd-3.0" ).HandleCommand( JambaTaxi, "jamba-taxi", "Jamba-Taxi", input )
    end
    
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:JambaQuestChatCommand()
-- Handle the quest chat command.
-------------------------------------------------------------------------------------------------------------
function Jamba:JambaQuestChatCommand( input )

    if not input or input:trim() == "" then
        InterfaceOptionsFrame_OpenToFrame( JambaQuest:GetOptionsFrame() )
    else
        LibStub( "AceConfigCmd-3.0" ).HandleCommand( JambaQuest, "jamba-quest", "Jamba-Quest", input )
    end
    
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:JambaNoChatCommand()
-- Handle the no chat command.
-------------------------------------------------------------------------------------------------------------
function Jamba:JambaNoChatCommand( input )

    if not input or input:trim() == "" then
        InterfaceOptionsFrame_OpenToFrame( JambaNo:GetOptionsFrame() )
    else
        LibStub( "AceConfigCmd-3.0" ).HandleCommand( JambaNo, "jamba-no", "Jamba-No", input )
    end
    
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:JambaFollowChatCommand()
-- Handle the follow chat command.
-------------------------------------------------------------------------------------------------------------
function Jamba:JambaFollowChatCommand( input )

    if not input or input:trim() == "" then
        InterfaceOptionsFrame_OpenToFrame( JambaFollow:GetOptionsFrame() )
    else
        LibStub( "AceConfigCmd-3.0" ).HandleCommand( JambaFollow, "jamba-follow", "Jamba-Follow", input )
    end
    
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:JambaMeChatCommand()
-- Handle the me chat command.
-------------------------------------------------------------------------------------------------------------
function Jamba:JambaMeChatCommand( input )

    if not input or input:trim() == "" then
        InterfaceOptionsFrame_OpenToFrame( JambaMe:GetOptionsFrame() )
    else
        LibStub( "AceConfigCmd-3.0" ).HandleCommand( JambaMe, "jamba-me", "Jamba-Me", input )
    end
    
end

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- EVENT HANDLERS
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-------------------------------------------------------------------------------------------------------------
-- Jamba:PLAYER_FOCUS_CHANGED()
-- Handle the player focus changed event.
-------------------------------------------------------------------------------------------------------------
function Jamba:PLAYER_FOCUS_CHANGED()

	-- If JambaComms is loaded then...
    if JambaComms then
        -- Process the player focus changed event.
        JambaComms:ProcessPlayerFocusChanged()      
    end

end

-------------------------------------------------------------------------------------------------------------
-- Jamba:UPDATE_BINDINGS()
-- Handle the key binding update event.
-------------------------------------------------------------------------------------------------------------
function Jamba:UPDATE_BINDINGS()

	-- Clear any temporary key bindings attached to the key binding frame.
	ClearOverrideBindings( self.keyBindingFrame )

	-- If JambaComms is loaded then...
    if JambaComms then
        -- Process the key binding update.
        JambaComms:UpdateKeyBindings( self.keyBindingFrame )       
    end

end

-------------------------------------------------------------------------------------------------------------
-- Jamba:PARTY_INVITE_REQUEST()
-- Handle the party invite request event.
-------------------------------------------------------------------------------------------------------------
function Jamba:PARTY_INVITE_REQUEST()

    -- If JambaGroup is loaded then...
    if JambaGroup then
        -- Process the group invite passing in the name of the character that issued the invite.
        JambaGroup:ProcessGroupInvite( arg1 )       
    end
        
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:PARTY_LEADER_CHANGED()
-- Handle the party leader changed event.
-------------------------------------------------------------------------------------------------------------
function Jamba:PARTY_LEADER_CHANGED()

    -- If JambaGroup is loaded then...
    if JambaGroup then
        -- Process the party leader changed event.
        JambaGroup:ProcessPartyLeaderChanged()       
    end
        
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:PARTY_MEMBERS_CHANGED()
-- Handle the party members changed event.
-------------------------------------------------------------------------------------------------------------
function Jamba:PARTY_MEMBERS_CHANGED()

    -- If JambaGroup is loaded then...
    if JambaGroup then
        -- Process the party members changed event.
        JambaGroup:ProcessPartyMembersChanged()       
    end
        
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:MERCHANT_SHOW()
-- Handle the merchant show event.
-------------------------------------------------------------------------------------------------------------
function Jamba:MERCHANT_SHOW()

    -- If JambaMerchant is loaded then...
    if JambaMerchant then
        -- Process the merchant show event.
        JambaMerchant:ProcessMerchantShow()     
    end
            
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:CHAT_MSG_WHISPER()
-- Handle the whisper message event.
-------------------------------------------------------------------------------------------------------------
function Jamba:CHAT_MSG_WHISPER()

    -- If JambaTalk is loaded then...
    if JambaTalk then
        -- Process the whisper received event.
        JambaTalk:ProcessWhisperReceived()      
    end
                
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:CHAT_MSG_SYSTEM()
-- Handle the system message event.
-------------------------------------------------------------------------------------------------------------
function Jamba:CHAT_MSG_SYSTEM()

    -- If JambaTalk is loaded then...
    if JambaTalk then
        -- Process the whisper received event.
        JambaTalk:ProcessSystemMessageReceived()        
    end
                
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:QUEST_DETAIL()
-- Handle the quest detail event (fires when quest detail frame is shown).
-------------------------------------------------------------------------------------------------------------
function Jamba:QUEST_DETAIL()

    -- If JambaQuest is loaded then...
    if JambaQuest then
        -- Process the quest detail event.
        JambaQuest:ProcessQuestDetail() 
    end
                
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:QUEST_ACCEPT_CONFIRM()
-- Handle the quest accept confirm event (fires for escort quests - do you want to join question).
-------------------------------------------------------------------------------------------------------------
function Jamba:QUEST_ACCEPT_CONFIRM()

    -- If JambaQuest is loaded then...
    if JambaQuest then
        -- Process the quest detail event.
        JambaQuest:ProcessQuestAcceptConfirm()  
    end
                
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:QUEST_COMPLETE()
-- Handle the quest complete event (fires for before the complete quest button is shown).
-------------------------------------------------------------------------------------------------------------
function Jamba:QUEST_COMPLETE()

    -- If JambaQuest is loaded then...
    if JambaQuest then
        -- Process the quest detail event.
        JambaQuest:ProcessQuestComplete()  
    end
                
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:DUEL_REQUESTED()
-- Handle the duel requested event.
-------------------------------------------------------------------------------------------------------------
function Jamba:DUEL_REQUESTED()

    -- If JambaNo is loaded then...
    if JambaNo then
        -- Process the duel requested detail event.
        JambaNo:ProcessDuelRequested()   
    end
                
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:GUILD_INVITE_REQUEST()
-- Handle the guild invite event.
-------------------------------------------------------------------------------------------------------------
function Jamba:GUILD_INVITE_REQUEST()

    -- If JambaNo is loaded then...
    if JambaNo then
        -- Process the guild invite request event.
        JambaNo:ProcessGuildInvite()     
    end
                
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:AUTOFOLLOW_BEGIN()
-- Handle the auto follow begin event.
-------------------------------------------------------------------------------------------------------------
function Jamba:AUTOFOLLOW_BEGIN()

    -- If JambaFollow is loaded then...
    if JambaFollow then
        -- Process the auto follow begin event.
        JambaFollow:ProcessAutoFollowBegin()   
    end
                
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:AUTOFOLLOW_END()
-- Handle the auto follow end event.
-------------------------------------------------------------------------------------------------------------
function Jamba:AUTOFOLLOW_END()

    -- If JambaFollow is loaded then...
    if JambaFollow then
        -- Process the auto follow end event.
        JambaFollow:ProcessAutoFollowEnd()   
    end
                
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:UNIT_HEALTH()
-- Handle the unit health event.
-------------------------------------------------------------------------------------------------------------
function Jamba:UNIT_HEALTH()

    -- If JambaMe is loaded then...
    if JambaMe then
        -- Process the unit health event.
        JambaMe:ProcessUnitHealth()   
    end
                
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:UNIT_MANA()
-- Handle the unit mana event.
-------------------------------------------------------------------------------------------------------------
function Jamba:UNIT_MANA()

    -- If JambaMe is loaded then...
    if JambaMe then
        -- Process the unit mana event.
        JambaMe:ProcessUnitMana()   
    end
                
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:PLAYER_REGEN_ENABLED()
-- Handle the player regen enabled event.
-------------------------------------------------------------------------------------------------------------
function Jamba:PLAYER_REGEN_ENABLED()

    -- If JambaMe is loaded then...
    if JambaMe then
        -- Process the player regen enabled event.
        JambaMe:ProcessPlayerRegenEnabled()   
    end

    -- If JambaFollow is loaded then...
    if JambaFollow then
        -- Process the player regen enabled event.
        JambaFollow:ProcessPlayerRegenEnabled()   
    end
                
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:PLAYER_REGEN_DISABLED()
-- Handle the player regen disabled event.
-------------------------------------------------------------------------------------------------------------
function Jamba:PLAYER_REGEN_DISABLED()

    -- If JambaMe is loaded then...
    if JambaMe then
        -- Process the player regen disabled event.
        JambaMe:ProcessPlayerRegenDisabled()   
    end

    -- If JambaFollow is loaded then...
    if JambaFollow then
        -- Process the player regen disabled event.
        JambaFollow:ProcessPlayerRegenDisabled()   
    end
                
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:UNIT_COMBAT()
-- Handle the unit combat event.
-------------------------------------------------------------------------------------------------------------
function Jamba:UNIT_COMBAT()

    -- If JambaMe is loaded then...
    if JambaMe then
        -- Process the unit combat event.
        JambaMe:ProcessUnitCombat()   
    end
                
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:RESURRECT_REQUEST()
-- Handle the resurrect request.
-------------------------------------------------------------------------------------------------------------
function Jamba:RESURRECT_REQUEST()

    -- If JambaMe is loaded then...
    if JambaMe then
        -- Process the resurrect request event.
        JambaMe:ProcessResurrectRequest( arg1 )   
    end
                
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:START_LOOT_ROLL()
-- Handle the start loot roll event.
-------------------------------------------------------------------------------------------------------------
function Jamba:START_LOOT_ROLL()

    -- If JambaGroup is loaded then...
    if JambaGroup then
        -- Process the start loot roll event.
        JambaGroup:ProcessStartLootRoll( arg1, arg2 )   
    end
                
end

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- (SECURE) FUNCTION HOOK HANDLERS
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-------------------------------------------------------------------------------------------------------------
-- Jamba:JambaHookTaxiTakeTaxiNode()
-- Handle the has taken flight event.
-------------------------------------------------------------------------------------------------------------
function Jamba:JambaHookTaxiTakeTaxiNode( ... )

    if JambaTaxi then
        JambaTaxi:HookTakeTaxiNode( ... )
    end
        
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:JambaHookQuestSelectGossipOption()
-- Handle a quest function (post hook).
-------------------------------------------------------------------------------------------------------------
function Jamba:JambaHookQuestSelectGossipOption( ... )

    if JambaQuest then
        JambaQuest:HookSelectGossipOption( ... )
    end
        
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:JambaHookQuestSelectGossipActiveQuest()
-- Handle a quest function (post hook).
-------------------------------------------------------------------------------------------------------------
function Jamba:JambaHookQuestSelectGossipActiveQuest( ... )

    if JambaQuest then
        JambaQuest:HookSelectGossipActiveQuest( ... )
    end
        
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:JambaHookQuestSelectGossipAvailableQuest()
-- Handle a quest function (post hook).
-------------------------------------------------------------------------------------------------------------
function Jamba:JambaHookQuestSelectGossipAvailableQuest( ... )

    if JambaQuest then
        JambaQuest:HookSelectGossipAvailableQuest( ... )
    end
        
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:JambaHookQuestSelectActiveQuest()
-- Handle a quest function (post hook).
-------------------------------------------------------------------------------------------------------------
function Jamba:JambaHookQuestSelectActiveQuest( ... )

    if JambaQuest then
        JambaQuest:HookSelectActiveQuest( ... )
    end
        
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:JambaHookQuestSelectAvailableQuest()
-- Handle a quest function (post hook).
-------------------------------------------------------------------------------------------------------------
function Jamba:JambaHookQuestSelectAvailableQuest( ... )

    if JambaQuest then
        JambaQuest:HookSelectAvailableQuest( ... )
    end
        
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:JambaHookuestAcceptQuest()
-- Handle a quest function (post hook).
-------------------------------------------------------------------------------------------------------------
function Jamba:JambaHookQuestAcceptQuest( ... )

    if JambaQuest then
        JambaQuest:HookAcceptQuest( ... )
    end
        
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:JambaHookQuestCompleteQuest()
-- Handle a quest function (post hook).
-------------------------------------------------------------------------------------------------------------
function Jamba:JambaHookQuestCompleteQuest( ... )

    if JambaQuest then
        JambaQuest:HookCompleteQuest( ... )
    end
        
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:JambaHookQuestGetQuestReward()
-- Handle a quest function (post hook).
-------------------------------------------------------------------------------------------------------------
function Jamba:JambaHookQuestGetQuestReward( ... )

    if JambaQuest then
        JambaQuest:HookGetQuestReward( ... )
    end
        
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:JambaHookQuestDeclineQuest()
-- Handle a quest function (post hook).
-------------------------------------------------------------------------------------------------------------
function Jamba:JambaHookQuestDeclineQuest( ... )

    if JambaQuest then
        JambaQuest:HookDeclineQuest( ... )
    end
        
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:JambaHookQuestSelectQuestLogEntry()
-- Handle a quest function (post hook).
-------------------------------------------------------------------------------------------------------------
function Jamba:JambaHookQuestSelectQuestLogEntry( ... )

    if JambaQuest then
        JambaQuest:HookSelectQuestLogEntry( ... )
    end
        
end

-------------------------------------------------------------------------------------------------------------
-- Jamba:JambaHookQuestToggleQuestLog()
-- Handle a quest function (post hook).
-------------------------------------------------------------------------------------------------------------
function Jamba:JambaHookQuestToggleQuestLog( ... )

    if JambaQuest then
        JambaQuest:HookToggleQuestLog( ... )
    end
        
end

