-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- Jamba - Jafula's Awesome Multi-Boxer's Assistant Addon.
-- JambaComms - Handles character's communication; manages who is the multiboxers 'posse' and who is the 
-- leader.
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
local MAJOR, MINOR = "JambaComms-1.0", 0.1
local JambaComms = LibStub:NewLibrary(MAJOR, MINOR)
if not JambaComms then 
	return 
end

-------------------------------------------------------------------------------------------------------------
-- Locale.
-------------------------------------------------------------------------------------------------------------
local L = LibStub( "AceLocale-3.0" ):GetLocale( "Jamba" )

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- INITIALIZE, OPTIONS AND CONFIGURATION
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-------------------------------------------------------------------------------------------------------------
-- JambaComms:Initialize
-- Initializes JambaComms.
-------------------------------------------------------------------------------------------------------------
function JambaComms:Initialize( dbOptions )

	-- Initialise utility functions and constants.
	self.utils = LibStub( "JambaUtils-1.0" )
	
	-- Use the global JambaComms for communication.
	self.comms = JambaComms

	-- Remember the player name.
	self.playerName = UnitName( "player" )

	-- Sets the options database used to store user options.
	self.db = dbOptions

	-- Initialize the options configuration.
	LibStub( "AceConfig-3.0" ):RegisterOptionsTable( "Jamba-Comms", self:GetOptionsConfiguration() )
	self.optionsFrame = LibStub( "AceConfigDialog-3.0" ):AddToBlizOptions( "Jamba-Comms", "Jamba" )

	-- A list of selected tags.
	self.selectedTags = {}
	
	-- The character to edit tags for.
	self.characterToEditTagsFor = self.playerName
	
	-- Last message sent to master saves.
	self.lastTellMasterCharacterName = ""
	self.lastTellMasterMessage = ""

	-- Set Focus keys.
	self.setFocusKey1 = nil
	self.setFocusKey2 = nil

	-- Set Master As Target keys.
	self.setMasterAsTargetKey1 = nil
	self.setMasterAsTargetKey2 = nil
	
	-- Make sure this character is a member, add and enable if not on the list.
	self.comms:ConfirmPlayerIsMember()
	
	-- Make sure there is a master, if none, set this character.
	self.comms:ConfirmThereIsAMaster()

	-- A secure action button so that focus can be set to the master.
	JambaSecureButtonSetFocus = CreateFrame( "CheckButton", "JambaSecureButtonSetFocus", nil, "SecureActionButtonTemplate" ); 
	JambaSecureButtonSetFocus:SetAttribute( "type", "macro" )
	JambaSecureButtonSetFocus:SetAttribute( "macrotext", "" )
	JambaSecureButtonSetFocus:Hide()

	-- A secure action button so that master can be set as target.
	JambaSecureButtonSetMasterAsTarget = CreateFrame( "CheckButton", "JambaSecureButtonSetMasterAsTarget", nil, "SecureActionButtonTemplate" ); 
	JambaSecureButtonSetMasterAsTarget:SetAttribute( "type", "macro" )
	JambaSecureButtonSetMasterAsTarget:SetAttribute( "macrotext", "" )
	JambaSecureButtonSetMasterAsTarget:Hide()
	
	-- Initialize static popups.
	self:InitializePopUps()	
				
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:OnEnable
-- Called when the addon is enabled.
-------------------------------------------------------------------------------------------------------------
function JambaComms:OnEnable()

	-- Join warning channel if requested.
	self:JoinWarningChannel()

	-- If set focus to master on master change is set; then attempt to do just that.
	if self.db.setFocusToMasterOnMasterChange == true then
		JambaComms:FocusUpdateMacroText( self.db.master )
	end

	-- If set target to master on master change is set; then attempt to do just that.
	if self.db.setTargetToMasterOnMasterChange == true then
		JambaComms:MasterAsTargetUpdateMacroText( self.db.master )
	end
				
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:InitializePopUps
-- Initializes popups.
-------------------------------------------------------------------------------------------------------------
function JambaComms:InitializePopUps()

   -- Ask the name of the character to add as a new member.
   StaticPopupDialogs["JAMBACOMMS_ASK_CHARACTER_NAME"] = {
        text = L["Enter name of character to add:"],
        button1 = ACCEPT,
        button2 = CANCEL,
        hasEditBox = 1,
        timeout = 0,
		whileDead = 1,
		hideOnEscape = 1,
		OnShow = function()
            local name = this:GetName()
            getglobal(name.."EditBox"):SetText("")
            getglobal(name.."Button1"):Disable()
            getglobal(name.."EditBox"):SetFocus()
        end,		
        OnAccept = function()
		    local characterName = getglobal(this:GetParent():GetName().."EditBox"):GetText()
			JambaComms:AddMember( nil, characterName )
		end,
        EditBoxOnTextChanged = function()
            local name = this:GetParent():GetName()
            local button = getglobal(name.."Button1")
            local characterName = getglobal(this:GetParent():GetName().."EditBox"):GetText()
            if not characterName or characterName:trim() == "" then
				button:Disable()
            else
                button:Enable()
            end
        end,		
		EditBoxOnEnterPressed = function()
            if getglobal(this:GetParent():GetName().."Button1"):IsEnabled() then
				local characterName = getglobal(this:GetParent():GetName().."EditBox"):GetText()
				JambaComms:AddMember( nil, characterName )
            end
            this:GetParent():Hide()
        end,		
    }

   -- Confirm removing characters from member list.
   StaticPopupDialogs["JAMBACOMMS_CONFIRM_REMOVE_CHARACTERS"] = {
        text = L["Are you sure you wish to remove the selected characters from the member list?"],
        button1 = ACCEPT,
        button2 = CANCEL,
        timeout = 0,
		whileDead = 1,
		hideOnEscape = 1,
        OnAccept = function()
			JambaComms:RemoveSelectedMembers()
		end,
    }

   -- Ask for a tag.
   StaticPopupDialogs["JAMBACOMMS_ASK_TAG"] = {
        text = L["Enter a tag:"],
        button1 = ACCEPT,
        button2 = CANCEL,
        hasEditBox = 1,
        timeout = 0,
		whileDead = 1,
		hideOnEscape = 1,
		OnShow = function()
            local name = this:GetName()
            getglobal(name.."EditBox"):SetText("")
            getglobal(name.."Button1"):Disable()
            getglobal(name.."EditBox"):SetFocus()
        end,		
        OnAccept = function()
		    local tag = getglobal(this:GetParent():GetName().."EditBox"):GetText()
			JambaComms:AddTag( nil, tag )
		end,
        EditBoxOnTextChanged = function()
            local name = this:GetParent():GetName()
            local button = getglobal(name.."Button1")
            local tag = getglobal(this:GetParent():GetName().."EditBox"):GetText()
            if not tag or tag:trim() == "" then
				button:Disable()
            else
                button:Enable()
            end
        end,		
		EditBoxOnEnterPressed = function()
            if getglobal(this:GetParent():GetName().."Button1"):IsEnabled() then
                local tag = getglobal(this:GetParent():GetName().."EditBox"):GetText()
                JambaComms:AddTag( nil, tag )
            end
            this:GetParent():Hide()
        end,		
    }

   -- Confirm removing tags.
   StaticPopupDialogs["JAMBACOMMS_CONFIRM_REMOVE_TAGS"] = {
        text = L["Are you sure you wish to remove the selected tags?"],
        button1 = ACCEPT,
        button2 = CANCEL,
        timeout = 0,
		whileDead = 1,
		hideOnEscape = 1,
        OnAccept = function()
			JambaComms:RemoveSelectedTags()
		end,
    }
        
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:GetOptionsConfiguration
-- Returns the configuration options table for JambaComms.
-------------------------------------------------------------------------------------------------------------
function JambaComms:GetOptionsConfiguration()

	if not self.configuration then

		self.configuration = {
			name = "Jamba",
			handler = JambaComms,
			type = 'group',
			childGroups  = "tab",
			args = {				
				information = {
					type = "description",
					name = L["Welcome to Jamba.  For information please visit: http://wow.jafula.com/addons/jamba"],
					image = [[Interface\Icons\INV_Misc_Toy_02]],
					order = 0,
					cmdHidden = true,
				},	
				teamConfiguration = {
					name = L["Team"],
					handler = JambaComms,
					type = "group",
					order = 0,
					args = {								
						setMaster = {
							type = "select",
							name = L["Select Master"],
							desc = L["Select the master character."],
							values = self.db.memberList,
							get = "GetMaster",
							set = "SetMaster",
							style = "dropdown",
							order = 1,
							cmdHidden = true,					
						},		
						memberList = {
							type = "multiselect",
							name = L["Team List"],
							desc = L["Characters in your 'group'; check to enable communication with them."],
							values = self.db.memberList,
							get = "IsCharacterEnabled",
							set = "ToggleCharacterEnabled",
							width = "full",
							order = 2,					
							cmdHidden = true,
						},
						addMemberGui = {
							type = "execute",
							name = L["Add"],
							desc = L["Add a new member to the member list."],
							func = "AddMemberGui",
							width = "half",
							order = 3,					
							cmdHidden = true,
						},					
						removeMemberGui = {
							type = "execute",
							name = L["Remove"],
							desc = L["Remove selected members from the member list."],
							func = "RemoveSelectedMembersGui",
							width = "half",
							order = 4,					
							cmdHidden = true,
						},								
						setFocusToMasterOnMasterChange = {
							type = "toggle",
							name = L["Set Focus To Master When Changing Master"],
							desc = L["When the master changes, make sure that Focus changes to the new master."],
							get = "CanFocusMasterOnMasterChange",
							set = "ToggleFocusMasterOnMasterChange",
							width = "full",
							order = 5,
						}, 											
						setTargetToMasterOnMasterChange = {
							type = "toggle",
							name = L["Set Target To Master When Changing Master"],
							desc = L["When the master changes, make sure that target changes to the new master."],
							get = "CanTargetMasterOnMasterChange",
							set = "ToggleTargetMasterOnMasterChange",
							width = "full",
							order = 6,
						}, 	
						changeMasterOnFocusChange = {
							type = "toggle",
							name = L["On Focus Change Set Focus To Master"],
							desc = L["When this characters Focus changes, set the new focus to be the master."],
							get = "CanChangeMasterOnFocusChange",
							set = "ToggleChangeMasterOnFocusChange",
							width = "full",
							order = 7,
						}, 																
						pushSettingsToOtherCharactersTeam = {
							type = "execute",
							name = L["Push To Characters"],
							desc = L["Push these settings to all characters that are enabled (in Jamba) and online."],
							func = "PushSettingsToEnabledCharactersTeam",
							order = 8,			
						},														
					},
				},
				tagConfiguration = {
					name = L["Tags"],
					handler = JambaComms,
					type = "group",
					order = 1,
					args = {	
						setSelectedMember = {
							type = "select",
							name = L["Edit Tags For"],
							desc = L["Select the character to edit tags for."],
							values = self.db.memberList,
							get = "GetMemberToEditTagsFor",
							set = "SetMemberToEditTagsFor",
							style = "dropdown",
							order = 0,					
						},						
						tagList = {
							type = "multiselect",
							name = L["Tags"],
							desc = L["Names that this character answers to when given a command."],
							values = "GetTagListForMemberToEditTagsFor",
							get = "IsTagSelected",
							set = "ToggleTagSelected",
							width = "full",
							order = 1,					
							cmdHidden = true,
						},
						addTagGui = {
							type = "execute",
							name = L["Add"],
							desc = L["Add a new tag to the tag list."],
							func = "AddTagGui",
							width = "half",
							order = 2,					
							cmdHidden = true,
						},					
						removeSelectedTagsGui = {
							type = "execute",
							name = L["Remove"],
							desc = L["Remove selected tags from the tag list."],
							func = "RemoveSelectedTagsGui",
							width = "half",
							order = 3,					
							cmdHidden = true,
						},	
						pushSettingsToOtherCharactersTags = {
							type = "execute",
							name = L["Push To Characters"],
							desc = L["Push these settings to all characters that are enabled (in Jamba) and online."],
							func = "PushSettingsToEnabledCharactersTags",
							order = 4,			
						},							
					},
				},
				chatConfiguration = {
					name = L["Chat"],
					handler = JambaComms,
					type = "group",
					order = 2,
					args = {							
						displayMessageInDefaultChat = {
							type = "toggle",
							name = L["Show Messages In Default Chat (On Master)"],
							desc = L["Display messages from slaves in default chat.  Must be checked on both master and slave."],
							get = "CanDisplayMessageInDefaultChat",
							set = "ToggleDisplayMessageInDefaultChat",
							width = "full",
							order = 0,
						},
						displaySpecialMessageInDefaultChat = {
							type = "toggle",
							name = L["Show System Messages In Default Chat (On Master)"],
							desc = L["Display system messages (clickable links) from slaves in default chat.  Must be checked on both master and slave."],
							get = "CanDisplaySpecialMessageInDefaultChat",
							set = "ToggleDisplaySpecialMessageInDefaultChat",
							width = "full",
							order = 1,
						},
						sendMessageToChannel = {
							type = "toggle",
							name = L["Send Messages To Channel (All)"],
							desc = L["Send all messages to a channel."],
							get = "CanSendMessageToChannel",
							set = "ToggleSendMessageToChannel",
							width = "full",
							order = 2,	
						},				
						messageChannelName = {
							type = "input",
							name = L["Message Channel Name"],
							desc = L["The name of the channel to join for messages."],
							get = "GetMessageChannelName",
							set = "SetMessageChannelName",
							disabled = function() return not self.db.sendMessageToChannel end,
							width = "full",
							order = 3,	
						},												
						messageChannelPassword = {
							type = "input",
							name = L["Message Channel Password"],
							desc = L["The password of the channel to join for messages."],
							get = "GetMessageChannelPassword",
							set = "SetMessageChannelPassword",
							disabled = function() return not self.db.sendMessageToChannel end,
							width = "full",
							order = 4,	
						},	
						sendMessageToGuildChat = {
							type = "toggle",
							name = L["Send Messages To Guild Chat (All)"],
							desc = L["Send all messages to guild chat."],
							get = "CanSendMessageToGuildChat",
							set = "ToggleSendMessageToGuildChat",
							width = "full",
							order = 5,	
						},							
						sendMessageToGuildOfficerChat = {
							type = "toggle",
							name = L["Send Messages To Guild Officer Chat (All)"],
							desc = L["Send all messages to guild officer chat."],
							get = "CanSendMessageToGuildOfficerChat",
							set = "ToggleSendMessageToGuildOfficerChat",
							width = "full",
							order = 6,	
						},						
						sendMessageToPartyChat = {
							type = "toggle",
							name = L["Send Messages To Party Chat (All)"],
							desc = L["Send all messages to party chat."],
							get = "CanSendMessageToPartyChat",
							set = "ToggleSendMessageToPartyChat",
							width = "full",
							order = 7,	
						},						
						sendMessageToMasterViaWhisper = {
							type = "toggle",
							name = L["Send Messages Via Whisper (To Master)"],
							desc = L["Send all messages to the master via whisper."],
							get = "CanSendMessageToMasterViaWhisper",
							set = "ToggleSendMessageToMasterViaWhisper",
							width = "full",
							order = 8,	
						},																		
						pushSettingsToOtherCharactersChat = {
							type = "execute",
							name = L["Push To Characters"],
							desc = L["Push these settings to all characters that are enabled (in Jamba) and online."],
							func = "PushSettingsToEnabledCharactersChat",
							order = 9,			
						},																							
					},
				},								
				warningConfiguration = {
					name = L["Warning"],
					handler = JambaComms,
					type = "group",
					order = 3,
					args = {				
						displayWarningInDefaultChat = {
							type = "toggle",
							name = L["Show Warnings In Default Chat (On Master)"],
							desc = L["Display warnings from slaves in default chat.  Must be checked on both master and slave."],
							get = "CanDisplayWarningInDefaultChat",
							set = "ToggleDisplayWarningInDefaultChat",
							width = "full",
							order = 0,
						},					
						displayWarningInRaidWarning = {
							type = "toggle",
							name = L["Show Warnings In Raid Warning (On Master)"],
							desc = L["Display warnings from slaves using a raid warning.  Must be checked on both master and slave."],
							get = "CanDisplayWarningInRaidWarning",
							set = "ToggleDisplayWarningInRaidWarning",
							width = "full",
							order = 1,
						},			
						sendWarningToChannel = {
							type = "toggle",
							name = L["Send Warnings To Channel (All)"],
							desc = L["Send all warnings to a channel."],
							get = "CanSendWarningToChannel",
							set = "ToggleSendWarningToChannel",
							width = "full",
							order = 2,	
						},				
						warningChannelName = {
							type = "input",
							name = L["Warning Channel Name"],
							desc = L["The name of the channel to join for warnings."],
							get = "GetWarningChannelName",
							set = "SetWarningChannelName",
							disabled = function() return not self.db.sendWarningToChannel end,
							width = "full",
							order = 3,	
						},												
						warningChannelPassword = {
							type = "input",
							name = L["Warning Channel Password"],
							desc = L["The password of the channel to join for warnings."],
							get = "GetWarningChannelPassword",
							set = "SetWarningChannelPassword",
							disabled = function() return not self.db.sendWarningToChannel end,
							width = "full",
							order = 4,	
						},	
						sendWarningToGuildChat = {
							type = "toggle",
							name = L["Send Warnings To Guild Chat (All)"],
							desc = L["Send all warnings to guild chat."],
							get = "CanSendWarningToGuildChat",
							set = "ToggleSendWarningToGuildChat",
							width = "full",
							order = 5,	
						},							
						sendWarningToGuildOfficerChat = {
							type = "toggle",
							name = L["Send Warnings To Guild Officer Chat (All)"],
							desc = L["Send all warnings to guild officer chat."],
							get = "CanSendWarningToGuildOfficerChat",
							set = "ToggleSendWarningToGuildOfficerChat",
							width = "full",
							order = 6,	
						},						
						sendWarningToPartyChat = {
							type = "toggle",
							name = L["Send Warnings To Party Chat (All)"],
							desc = L["Send all warnings to party chat."],
							get = "CanSendWarningToPartyChat",
							set = "ToggleSendWarningToPartyChat",
							width = "full",
							order = 7,	
						},						
						sendWarningToMasterViaWhisper = {
							type = "toggle",
							name = L["Send Warnings Via Whisper (To Master)"],
							desc = L["Send all warnings to the master via whisper."],
							get = "CanSendWarningToMasterViaWhisper",
							set = "ToggleSendWarningToMasterViaWhisper",
							width = "full",
							order = 8,	
						},		
						pushSettingsToOtherCharactersWarn = {
							type = "execute",
							name = L["Push To Characters"],
							desc = L["Push these settings to all characters that are enabled (in Jamba) and online."],
							func = "PushSettingsToEnabledCharactersWarning",
							order = 9,			
						},																	
					},
				},	
				addMember = {
					type = "input",
					name = L["Add"],
					desc = L["Add a new member to the member list."],
					usage = "/jamba addmember <name>",
					get = false,
					set = "AddMemberCommand",
					guiHidden = true,
				},					
				removeMember = {
					type = "input",
					name = L["Remove"],
					desc = L["Remove selected members from the member list."],
					usage = "/jamba removemember <name>",
					get = false,
					set = "RemoveMemberCommand",
					guiHidden = true,
				},						
				enableMember = {
					type = "input",
					name = L["Enable"],
					desc = L["Enable a member."],
					usage = "/jamba enablemember <name>",
					get = false,
					set = "EnableMemberCommand",
					guiHidden = true,
				},					
				disableMember = {
					type = "input",
					name = L["Disable"],
					desc = L["Disable a member."],
					usage = "/jamba disablemember <name>",
					get = false,
					set = "DisableMemberCommand",
					guiHidden = true,
				},											
				setFocusMaster = {
					type = "input",
					name = L["Set Focus To The Master"],
					desc = L["Update the set Focus command to set Focus to the master.  You need to press the key assigned to 'Set Focus' to actually set the Focus."],
					usage = "/jamba setfocusmaster <tag>",
					get = false,
					set = "FocusUpdateMacroTextSetMaster",
					guiHidden = true,
				},										
				setFocus = {
					type = "input",
					name = L["Set Focus To A Target"],
					desc = L["Update the set Focus command to set Focus to a target specified as the first parameter.  You need to press the key assigned to 'Set Focus' to actually set the Focus."],
					usage = "/jamba setfocus <target> <tag>",
					get = false,
					set = "FocusUpdateMacroTextSetTarget",
					guiHidden = true,
				},																										
				setMaster = {
					type = "input",
					name = L["Set Master"],
					desc = L["Set the target specified in the first parameter to be the master."],
					usage = "/jamba setmaster <target> <tag>",
					get = false,
					set = "SetMasterToTarget",
					guiHidden = true,
				},																						
				setMeAsMaster = {
					type = "input",
					name = L["Set Me As Master"],
					desc = L["Set this character as the master."],
					usage = "/jamba setmeasmaster <tag>",
					get = false,
					set = "SetMeAsMaster",
					guiHidden = true,
				},					
				setTargetToMaster = {
					type = "input",
					name = L["Set Master As Target"],
					desc = L["Update the set master as target command to set the current master as the target.  You need to press the key assigned to 'Set Master As Target' to actually set the master as the target."],
					usage = "/jamba settargettomaster <tag>",
					get = false,
					set = "SetMasterToTarget",
					guiHidden = true,
				},		
				addTag = {
					type = "input",
					name = L["Add Tag To Character"],
					desc = L["Add a tag to a character."],
					usage = "/jamba addtag <character> <tag>",
					get = false,
					set = "AddTagViaCommand",
					guiHidden = true,
				},								
				removeTag = {
					type = "input",
					name = L["Remove Tag From Character"],
					desc = L["Remove a tag from a character."],
					usage = "/jamba removetag <character> <tag>",
					get = false,
					set = "RemoveTagViaCommand",
					guiHidden = true,
				},								
			},
		}
		
	end
	
	return self.configuration
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:GetOptionsFrame
-- Returns the blizzard interface options frame created for JambaComms.
-------------------------------------------------------------------------------------------------------------
function JambaComms:GetOptionsFrame()
	
	return self.optionsFrame
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:RefreshOptions
-- Refresh the JambaTalk options dialog.
-------------------------------------------------------------------------------------------------------------
function JambaComms:RefreshOptions()
	
	LibStub( "AceConfigRegistry-3.0" ):NotifyChange( "Jamba-Comms" )
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:PushSettingsToEnabledCharactersTeam
-- Push the settings for JambaComms to other enabled and connected characters.
-------------------------------------------------------------------------------------------------------------
function JambaComms:PushSettingsToEnabledCharactersTeam()

	-- Push the settings to all.
	self.comms:PushSettings( self.db, "Jamba-Core-Team" )
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:PushSettingsToEnabledCharactersTags
-- Push the settings for JambaComms to other enabled and connected characters.
-------------------------------------------------------------------------------------------------------------
function JambaComms:PushSettingsToEnabledCharactersTags()

	-- Push the settings to all.
	self.comms:PushSettings( self.db, "Jamba-Core-Tags" )
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:PushSettingsToEnabledCharactersChat
-- Push the settings for JambaComms to other enabled and connected characters.
-------------------------------------------------------------------------------------------------------------
function JambaComms:PushSettingsToEnabledCharactersChat()

	-- Push the settings to all.
	self.comms:PushSettings( self.db, "Jamba-Core-Chat" )
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:PushSettingsToEnabledCharactersWarning
-- Push the settings for JambaComms to other enabled and connected characters.
-------------------------------------------------------------------------------------------------------------
function JambaComms:PushSettingsToEnabledCharactersWarning()

	-- Push the settings to all.
	self.comms:PushSettings( self.db, "Jamba-Core-Warning" )
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:ReceiveSettingsTeam
-- Receive settings for JambaComms.
-------------------------------------------------------------------------------------------------------------
function JambaComms:ReceiveSettingsTeam( settings )

	-- Set the new settings.
	self.db.master = settings.master
    
    self.utils:ClearTable( self.db.memberList )
	for memberKey, memberValue in pairs( settings.memberList ) do
		self.db.memberList[memberKey] = memberValue
	end
    
    self.utils:ClearTable( self.db.memberEnabled )
	for memberKey, memberValue in pairs( settings.memberEnabled ) do
		self.db.memberEnabled[memberKey] = memberValue
	end
	
	-- Target and focus options.
	self.db.setFocusToMasterOnMasterChange = settings.setFocusToMasterOnMasterChange
	self.db.setTargetToMasterOnMasterChange = settings.setTargetToMasterOnMasterChange
	self.db.changeMasterOnFocusChange = settings.changeMasterOnFocusChange
	
	-- Make sure this character is a member.  
	self.comms:ConfirmPlayerIsMember()

	-- Make sure there is a master, if none, set this character.
	self.comms:ConfirmThereIsAMaster()

	-- Update master focus and target.
	self.comms:SetMaster( nil, self.db.master )
	
	-- Refresh the options.
	self:RefreshOptions()
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:ReceiveSettingsTags
-- Receive settings for JambaComms.
-------------------------------------------------------------------------------------------------------------
function JambaComms:ReceiveSettingsTags( settings )

	-- Set the new settings.
	self.utils:ClearTableWithInnerTable( self.db.memberTagList )
	for characterName, tagList in pairs( settings.memberTagList ) do
		-- Get the current tag list (new in this case as it was cleared above) for the character.
		local thisTagList = self.comms:GetTagListForCharacter( characterName )
		-- Now set the new tags into place.
		for tagName, tagValue in pairs( tagList ) do	
			thisTagList[tagName] = tagValue
		end
	end                

	-- Refresh the options.
	self:RefreshOptions()
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:ReceiveSettingsChat
-- Receive settings for JambaComms.
-------------------------------------------------------------------------------------------------------------
function JambaComms:ReceiveSettingsChat( settings )

	-- Attempt to leave the current channel.
	if self.db.sendMessageToChannel == true then
		self.comms:LeaveChannel( self.db.messageChannelName )
	end
	
	-- Set the new settings.	
	self.db.displayMessageInDefaultChat = settings.displayMessageInDefaultChat
	self.db.sendMessageToChannel = settings.sendMessageToChannel
	self.db.messageChannelName = settings.messageChannelName
	self.db.messageChannelPassword = settings.messageChannelPassword
	self.db.sendMessageToGuildChat = settings.sendMessageToGuildChat
	self.db.sendMessageToGuildOfficerChat = settings.sendMessageToGuildOfficerChat
	self.db.sendMessageToPartyChat = settings.sendMessageToPartyChat
	self.db.sendMessageToMasterViaWhisper = settings.sendMessageToMasterViaWhisper
	self.db.displaySpecialMessageInDefaultChat = settings.displaySpecialMessageInDefaultChat
			
	-- Leave / join the channels depending on the setting.
	self.comms:ToggleSendMessageToChannel( nil, self.db.sendMessageToChannel )
		
	-- Refresh the options.
	self:RefreshOptions()
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:ReceiveSettingsWarning
-- Receive settings for JambaComms.
-------------------------------------------------------------------------------------------------------------
function JambaComms:ReceiveSettingsWarning( settings )

	-- Attempt to leave the current channel.
	if self.db.sendWarningToChannel == true then
		self.comms:LeaveChannel( self.db.warningChannelName )
	end
	
	-- Set the new settings.
	self.db.displayWarningInDefaultChat = settings.displayWarningInDefaultChat
	self.db.displayWarningInRaidWarning = settings.displayWarningInRaidWarning
	self.db.sendWarningToChannel = settings.sendWarningToChannel
	self.db.warningChannelName = settings.warningChannelName
	self.db.warningChannelPassword = settings.warningChannelPassword
	self.db.sendWarningToGuildChat = settings.sendWarningToGuildChat
	self.db.sendWarningToGuildOfficerChat = settings.sendWarningToGuildOfficerChat
	self.db.sendWarningToPartyChat = settings.sendWarningToPartyChat
	self.db.sendWarningToMasterViaWhisper = settings.sendWarningToMasterViaWhisper

	-- Leave / join the channels depending on the setting.
	self.comms:ToggleSendWarningToChannel( nil, self.db.sendWarningToChannel )
		
	-- Refresh the options.
	self:RefreshOptions()
	
end
	
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- CONFIGURATION CALLBACKS
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
	
-------------------------------------------------------------------------------------------------------------
-- JambaComms: IsCharacterEnabled
-- Return whether or not a character is enabled for communication/selected.
-------------------------------------------------------------------------------------------------------------
function JambaComms:IsCharacterEnabled( info, key )

	local characterName = key
	local enabled = false
	
	if self.comms:CharacterIsAMember( characterName ) then
		enabled = self.db.memberEnabled[characterName]
	end
	
	return enabled
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: ToggleCharacterEnabled
-- Toggle whether or not a character is enabled for communication/selected.
-------------------------------------------------------------------------------------------------------------
function JambaComms:ToggleCharacterEnabled( info, key, state )

	local characterName = key
	local isSelected = state
	
	if self.comms:CharacterIsAMember( characterName ) then
		self.db.memberEnabled[characterName] = isSelected
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: GetMemberToEditTagsFor
-- Get the member to edit tags for.
-------------------------------------------------------------------------------------------------------------
function JambaComms:GetMemberToEditTagsFor( info )
	
	return self.characterToEditTagsFor
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: GetMemberToEditTagsFor
-- Set the member to edit tags for.
-------------------------------------------------------------------------------------------------------------
function JambaComms:SetMemberToEditTagsFor( info, value )

	self.characterToEditTagsFor = value
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: GetTagListForMemberToEditTagsFor
-- Return the tag list of the member to edit tags for.
-------------------------------------------------------------------------------------------------------------
function JambaComms:GetTagListForMemberToEditTagsFor()

	return self.db.memberTagList[self.characterToEditTagsFor]
	
end
-------------------------------------------------------------------------------------------------------------
-- JambaComms: GetMaster
-- Get the master for this character.
-------------------------------------------------------------------------------------------------------------
function JambaComms:GetMaster( info )
	
	return self.db.master
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: SetMaster
-- Set the master for this character; this character must be online.
-------------------------------------------------------------------------------------------------------------
function JambaComms:SetMaster( info, value )

	if value:trim() ~= "" then
	
		local master = self.utils:Capitalise( value )
		
		-- Only allow characters in the member list to be the master.
		if self.comms:CharacterIsAMember( master ) then
			
			self.db.master = master
			
			-- If set focus to master on master change is set; then attempt to do just that.
			if self.db.setFocusToMasterOnMasterChange == true then
				JambaComms:FocusUpdateMacroText( self.db.master )
			end

			-- If set target to master on master change is set; then attempt to do just that.
			if self.db.setTargetToMasterOnMasterChange == true then
				JambaComms:MasterAsTargetUpdateMacroText( self.db.master )
			end

			-- Refresh the options.
			self:RefreshOptions()
			
			-- Tell JambaGroup the master has changed.
			if JambaGroup then
				JambaGroup:OnMasterChange()
			end
				
		else
		
			self.comms:TellSelf( self.playerName, L["Did not set master as that character is not in my member list."] )
			
		end

	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: AddMemberGui
-- Provides a GUI for a user to enter a character name which is added to the member list.
-------------------------------------------------------------------------------------------------------------
function JambaComms:AddMemberGui()
	
	StaticPopup_Show( "JAMBACOMMS_ASK_CHARACTER_NAME" );
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: AddMemberCommand
-- Add member from the command line.
-------------------------------------------------------------------------------------------------------------
function JambaComms:AddMemberCommand( info, parameters )
	
	local character = parameters

	if character and character:trim() ~= "" then
		self:AddMember( nil, self.utils:Capitalise( character ) )
	end	
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: AddMember
-- Add a member to the member list.
-------------------------------------------------------------------------------------------------------------
function JambaComms:AddMember( info, value )
	
	-- Wow names are at least two characters.
	if value and value:trim() ~= "" and value:len() > 1 then
	
		-- Capitalise the name.
		local characterName = self.utils:Capitalise( value )
		
		-- If the character is not already on the list...
		if not self.db.memberList[characterName] then
			-- Yes, add to the member list.
			self.db.memberList[characterName] = characterName
			-- Enable the member by default.
			self.comms:ToggleCharacterEnabled( nil, characterName, true )
		end

		-- Make sure there is a tag of all for this character.
		self.comms:ConfirmCharacterHasAllTag( characterName ) 

		-- Refresh the options.
		self:RefreshOptions()
		
	end
			
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: RemoveSelectedMembersGui
-- Provides a GUI for a user to confirm removing selected members from the member list.
-------------------------------------------------------------------------------------------------------------
function JambaComms:RemoveSelectedMembersGui()
	
	StaticPopup_Show( "JAMBACOMMS_CONFIRM_REMOVE_CHARACTERS" );
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: RemoveMemberCommand
-- Remove member from the command line.
-------------------------------------------------------------------------------------------------------------
function JambaComms:RemoveMemberCommand( info, parameters )
	
	local characterName = parameters

	if characterName and characterName:trim() ~= "" then
	
		-- Remove the character.
		self:RemoveMember( characterName )

		-- Make sure current character being edited for tags is valid.
		if self.utils:Capitalise( self.characterToEditTagsFor ) == self.utils:Capitalise( characterName ) then		
			-- The current player is always on the list.
			self.characterToEditTagsFor = self.playerName
		end
		
		-- Make sure this character is a member.  
		self.comms:ConfirmPlayerIsMember()

		-- Make sure there is a master, if none, set this character.
		self.comms:ConfirmThereIsAMaster()
			
		-- Refresh the options.
		self:RefreshOptions()
			
	end	
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: RemoveMember
-- value - character to remove.
-- Remove a member from the member list.
-------------------------------------------------------------------------------------------------------------
function JambaComms:RemoveMember( value )

	-- Wow names are at least two characters.
	if value and value:trim() ~= "" and value:len() > 1 then
	
		-- Capitalise the name.
		local characterName = self.utils:Capitalise( value )
		
		-- Remove character from list.
		self.db.memberList[characterName] = nil
		self.db.memberEnabled[characterName] = nil
		
		-- Remove tags for this character.
		local tagList = self.comms:GetTagListForCharacter( characterName )
		for tag, enabled in pairs( tagList ) do
			self.comms:RemoveTagFromCharacter( characterName, tag )
			tagList[tag] = nil
		end
		self.db.memberTagList[characterName] = nil

	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: RemoveSelectedMembers
-- Remove selected members from the member list.
-------------------------------------------------------------------------------------------------------------
function JambaComms:RemoveSelectedMembers()
	
	for characterName, enabled in pairs( self.db.memberEnabled ) do
		if enabled then
			self:RemoveMember( characterName )			
			-- Make sure current character being edited for tags is valid.
			if self.utils:Capitalise( self.characterToEditTagsFor ) == self.utils:Capitalise( characterName ) then	
				-- The current player is always on the list.
				self.characterToEditTagsFor = self.playerName
			end
		end
	end

	-- Make sure this character is a member.  
	self.comms:ConfirmPlayerIsMember()

	-- Make sure there is a master, if none, set this character.
	self.comms:ConfirmThereIsAMaster()
		
	-- Refresh the options.
	self:RefreshOptions()
		
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:CanFocusMasterOnMasterChange
-- Return the set focus to master on master change flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:CanFocusMasterOnMasterChange( info )

    return self.db.setFocusToMasterOnMasterChange
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:ToggleFocusMasterOnMasterChange
-- Toggle the set focus to master on master change flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:ToggleFocusMasterOnMasterChange( info, value )

    self.db.setFocusToMasterOnMasterChange = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:CanTargetMasterOnMasterChange
-- Return the set target to master on master change flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:CanTargetMasterOnMasterChange( info )

    return self.db.setTargetToMasterOnMasterChange
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:ToggleTargetMasterOnMasterChange
-- Toggle the set target to master on master change flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:ToggleTargetMasterOnMasterChange( info, value )

    self.db.setTargetToMasterOnMasterChange = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:CanChangeMasterOnFocusChange
-- Return the change master on focus change flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:CanChangeMasterOnFocusChange( info )

    return self.db.changeMasterOnFocusChange
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:ToggleChangeMasterOnFocusChange
-- Toggle the change master on focus change flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:ToggleChangeMasterOnFocusChange( info, value )

    self.db.changeMasterOnFocusChange = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: IsTagSelected
-- Return whether or not a tag is selected.
-------------------------------------------------------------------------------------------------------------
function JambaComms:IsTagSelected( info, key )
	
	local tag = key
	local selected = false
	
	if self.characterToEditTagsFor and tag then
		local tagList = self.comms:GetSelectedTagListForCharacter( self.characterToEditTagsFor )
		if tagList[tag] then
			selected = true
		end
	end
		
	return selected
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: ToggleTagSelected
-- Toggle whether or not a tag is selected.
-------------------------------------------------------------------------------------------------------------
function JambaComms:ToggleTagSelected( info, key, state )

	local tag = key
	local isSelected = state
	
	if self.characterToEditTagsFor and tag then
		local tagList = self.comms:GetSelectedTagListForCharacter( self.characterToEditTagsFor )
		if isSelected then
			tagList[tag] = true
		else
			tagList[tag] = nil
		end
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: AddTagGui
-- Provides a GUI for a user to enter a tag which is added to the tag list.
-------------------------------------------------------------------------------------------------------------
function JambaComms:AddTagGui()
	
	StaticPopup_Show( "JAMBACOMMS_ASK_TAG" );
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: AddTag
-- Add a tag to the tag list for the currently selected edit tag character.
-------------------------------------------------------------------------------------------------------------
function JambaComms:AddTag( info, value )
	
	local tag = value

	-- Add to the tag list for the currently selected edit tag character.
	self.comms:AddTagToCharacter( self.characterToEditTagsFor, tag )
	
	-- Refresh the options.
	self:RefreshOptions()
		
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: RemoveSelectedTagsGui
-- Provides a GUI for a user to confirm removing selected tags from the tag list.
-------------------------------------------------------------------------------------------------------------
function JambaComms:RemoveSelectedTagsGui()
	
	StaticPopup_Show( "JAMBACOMMS_CONFIRM_REMOVE_TAGS" );
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: RemoveSelectedTags
-- Remove selected tags from the tags list.
-------------------------------------------------------------------------------------------------------------
function JambaComms:RemoveSelectedTags()
	
	if self.characterToEditTagsFor then
	
		local tagList = self.comms:GetSelectedTagListForCharacter( self.characterToEditTagsFor )
		for tag, enabled in pairs( tagList ) do
			self.comms:RemoveTagFromCharacter( self.characterToEditTagsFor, tag )
			tagList[tag] = nil
		end
		
		self:RefreshOptions()

	end
			
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: CanDisplayWarningInDefaultChat
-- Return the can display warning in default chat flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:CanDisplayWarningInDefaultChat( info )

    return self.db.displayWarningInDefaultChat
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: ToggleDisplayWarningInDefaultChat
-- Toggle the can display warning in default chat flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:ToggleDisplayWarningInDefaultChat( info, value )

    self.db.displayWarningInDefaultChat = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: CanDisplayWarningInRaidWarning
-- Return the can display warning in raid window flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:CanDisplayWarningInRaidWarning( info )

    return self.db.displayWarningInRaidWarning
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: ToggleDisplayWarningInRaidWarning
-- Toggle the can display warning in raid window flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:ToggleDisplayWarningInRaidWarning( info, value )

    self.db.displayWarningInRaidWarning = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:CanSendWarningToChannel
-- Return the can send warning to channel flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:CanSendWarningToChannel( info )

    return self.db.sendWarningToChannel
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:ToggleSendWarningToChannel
-- Toggle the can send warning to channel flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:ToggleSendWarningToChannel( info, value )

    self.db.sendWarningToChannel = value

	-- If turning off the send warning to channel, leave the current warning channel.
	if value == false then    
		-- Attempt to leave the current channel.
		self.comms:LeaveChannel( self.db.warningChannelName )
	end

	-- If turning on the send warning to channel, join the current warning channel.
	if value == true then    
		-- Attempt to leave the current channel.
		self.comms:JoinWarningChannel()
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:GetWarningChannelName
-- Return the warning channel name.
-------------------------------------------------------------------------------------------------------------
function JambaComms:GetWarningChannelName( info )

    return self.db.warningChannelName
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:SetWarningChannelName
-- Set the warning channel name.
-------------------------------------------------------------------------------------------------------------
function JambaComms:SetWarningChannelName( info, value )

	-- Attempt to leave the current channel.
	self.comms:LeaveChannel( self.db.warningChannelName )
	
    self.db.warningChannelName = value

	-- Attempt to leave the current channel.
	self.comms:JoinWarningChannel()
		    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:GetWarningChannelPassword
-- Return the warning channel password.
-------------------------------------------------------------------------------------------------------------
function JambaComms:GetWarningChannelPassword( info )

    return self.db.warningChannelPassword
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:SetWarningChannelPassword
-- Set the warning channel password.
-------------------------------------------------------------------------------------------------------------
function JambaComms:SetWarningChannelPassword( info, value )

	-- Attempt to leave the current channel.
	self.comms:LeaveChannel( self.db.warningChannelName )
	
    self.db.warningChannelPassword = value
    
    -- Attempt to join the warning channel.
	self.comms:JoinWarningChannel()
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: CanSendWarningToGuildChat
-- Return the can send warning to guild chat flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:CanSendWarningToGuildChat( info )

    return self.db.sendWarningToGuildChat
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: ToggleSendWarningToGuildChat
-- Toggle the can send warning to guild chat flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:ToggleSendWarningToGuildChat( info, value )

    self.db.sendWarningToGuildChat = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: CanSendWarningToGuildOfficerChat
-- Return the can send warning to guild officer chat flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:CanSendWarningToGuildOfficerChat( info )

    return self.db.sendWarningToGuildOfficerChat
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: ToggleSendWarningToGuildOfficerChat
-- Toggle the can send warning to guild officer chat flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:ToggleSendWarningToGuildOfficerChat( info, value )

    self.db.sendWarningToGuildOfficerChat = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: CanSendWarningToPartyChat
-- Return the can send warning to party chat flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:CanSendWarningToPartyChat( info )

    return self.db.sendWarningToPartyChat
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: ToggleSendWarningToPartyChat
-- Toggle the can send warning to party chat flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:ToggleSendWarningToPartyChat( info, value )

    self.db.sendWarningToPartyChat = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: CanSendWarningToMasterViaWhisper
-- Return the send warning to master via whisper flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:CanSendWarningToMasterViaWhisper( info )

    return self.db.sendWarningToMasterViaWhisper
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: ToggleSendWarningToMasterViaWhisper
-- Toggle the send warning to master via whisper flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:ToggleSendWarningToMasterViaWhisper( info, value )

    self.db.sendWarningToMasterViaWhisper = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: CanDisplayMessageInDefaultChat
-- Return the can display message in default chat flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:CanDisplayMessageInDefaultChat( info )

    return self.db.displayMessageInDefaultChat
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: ToggleDisplayMessageInDefaultChat
-- Toggle the can display message in default chat flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:ToggleDisplayMessageInDefaultChat( info, value )

    self.db.displayMessageInDefaultChat = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: CanDisplaySpecialMessageInDefaultChat
-- Return the can display special message in default chat flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:CanDisplaySpecialMessageInDefaultChat( info )

    return self.db.displaySpecialMessageInDefaultChat
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: ToggleDisplaySpecialMessageInDefaultChat
-- Toggle the can display special message in default chat flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:ToggleDisplaySpecialMessageInDefaultChat( info, value )

    self.db.displaySpecialMessageInDefaultChat = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:CanSendMessageToChannel
-- Return the can send message to channel flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:CanSendMessageToChannel( info )

    return self.db.sendMessageToChannel
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:ToggleSendMessageToChannel
-- Toggle the can send message to channel flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:ToggleSendMessageToChannel( info, value )

    self.db.sendMessageToChannel = value

	-- If turning off the send message to channel, leave the current message channel.
	if value == false then    
		-- Attempt to leave the current channel.
		self.comms:LeaveChannel( self.db.messageChannelName )
	end

	-- If turning on the send message to channel, join the current message channel.
	if value == true then    
		-- Attempt to leave the current channel.
		self.comms:JoinMessageChannel()
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:GetMessageChannelName
-- Return the message channel name.
-------------------------------------------------------------------------------------------------------------
function JambaComms:GetMessageChannelName( info )

    return self.db.messageChannelName
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:SetMessageChannelName
-- Set the message channel name.
-------------------------------------------------------------------------------------------------------------
function JambaComms:SetMessageChannelName( info, value )

	-- Attempt to leave the current channel.
	self.comms:LeaveChannel( self.db.messageChannelName )
	
    self.db.messageChannelName = value

	-- Attempt to leave the current channel.
	self.comms:JoinMessageChannel()
		    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:GetMessageChannelPassword
-- Return the message channel password.
-------------------------------------------------------------------------------------------------------------
function JambaComms:GetMessageChannelPassword( info )

    return self.db.messageChannelPassword
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:SetMessageChannelPassword
-- Set the message channel password.
-------------------------------------------------------------------------------------------------------------
function JambaComms:SetMessageChannelPassword( info, value )

	-- Attempt to leave the current channel.
	self.comms:LeaveChannel( self.db.messageChannelName )
	
    self.db.messageChannelPassword = value
    
    -- Attempt to join the message channel.
	self.comms:JoinMessageChannel()
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: CanSendMessageToGuildChat
-- Return the can send message to guild chat flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:CanSendMessageToGuildChat( info )

    return self.db.sendMessageToGuildChat
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: ToggleSendMessageToGuildChat
-- Toggle the can send message to guild chat flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:ToggleSendMessageToGuildChat( info, value )

    self.db.sendMessageToGuildChat = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: CanSendMessageToGuildOfficerChat
-- Return the can send message to guild officer chat flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:CanSendMessageToGuildOfficerChat( info )

    return self.db.sendMessageToGuildOfficerChat
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: ToggleSendMessageToGuildOfficerChat
-- Toggle the can send message to guild officer chat flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:ToggleSendMessageToGuildOfficerChat( info, value )

    self.db.sendMessageToGuildOfficerChat = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: CanSendMessageToPartyChat
-- Return the can send message to party chat flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:CanSendMessageToPartyChat( info )

    return self.db.sendMessageToPartyChat
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: ToggleSendMessageToPartyChat
-- Toggle the can send message to party chat flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:ToggleSendMessageToPartyChat( info, value )

    self.db.sendMessageToPartyChat = value
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: CanSendMessageToMasterViaWhisper
-- Return the send message to master via whisper flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:CanSendMessageToMasterViaWhisper( info )

    return self.db.sendMessageToMasterViaWhisper
    
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: ToggleSendMessageToMasterViaWhisper
-- Toggle the send message to master via whisper flag.
-------------------------------------------------------------------------------------------------------------
function JambaComms:ToggleSendMessageToMasterViaWhisper( info, value )

    self.db.sendMessageToMasterViaWhisper = value
    
end

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- MEMBERS, TAGS AND ENABLED PROCESSING
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-------------------------------------------------------------------------------------------------------------
-- JambaComms: ProcessPlayerFocusChanged
-- Process the player focus changed event.
-------------------------------------------------------------------------------------------------------------
function JambaComms:ProcessPlayerFocusChanged()

	-- Change master on focus change option enabled?
	if self.db.changeMasterOnFocusChange == true then
	
		-- Get the name of the focused unit.
		local targetName, targetRealm = UnitName( "focus" )
		
		-- Attempt to set this name as the master.
		if targetName then
			self.comms:SetMaster( nil, targetName )
		end
		
	end

end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: EnableMemberCommand
-- Enable member from the command line.
-------------------------------------------------------------------------------------------------------------
function JambaComms:EnableMemberCommand( info, parameters )
	
	local character = parameters

	if character and character:trim() ~= "" then
		self.comms:ToggleCharacterEnabled( nil, self.utils:Capitalise( character ), true )
	end	

	self:RefreshOptions()
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: DisableMemberCommand
-- Disable member from the command line.
-------------------------------------------------------------------------------------------------------------
function JambaComms:DisableMemberCommand( info, parameters )
	
	local character = parameters

	if character and character:trim() ~= "" then
		self.comms:ToggleCharacterEnabled( nil, self.utils:Capitalise( character ), false )
	end	

	self:RefreshOptions()
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:GetSelectedTagListForCharacter
-- Returns the selected tag list for a specified character.
-------------------------------------------------------------------------------------------------------------
function JambaComms:GetSelectedTagListForCharacter( character )

	local tagList = self.selectedTags[character]
	if not tagList then
		self.selectedTags[character] = {}
		tagList = self.selectedTags[character]
	end

	return tagList
		
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:GetTagListForCharacter
-- Returns the tag list for a specified character.
-------------------------------------------------------------------------------------------------------------
function JambaComms:GetTagListForCharacter( character )

	-- Get the tag list for this character.
	local tagList = self.db.memberTagList[character]
	-- If there is no list...
	if not tagList then
		-- Create one.
		self.db.memberTagList[character] = {}
		tagList = self.db.memberTagList[character]
	end
		
	return tagList
		
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:AddTagViaCommand
-- Add tag slash command.
-------------------------------------------------------------------------------------------------------------
function JambaComms:AddTagViaCommand( info, parameters )

	local character, tag = strsplit( " ", parameters )
	
	if character and character:trim() ~= "" then
		if tag and tag:trim() ~= "" then
			self.comms:AddTagToCharacter( self.utils:Capitalise( character ), tag )
		end
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:RemoveTagViaCommand
-- Remove tag slash command.
-------------------------------------------------------------------------------------------------------------
function JambaComms:RemoveTagViaCommand( info, parameters )

	local character, tag = strsplit( " ", parameters )
	
	if character and character:trim() ~= "" then
		if tag and tag:trim() ~= "" then
			self.comms:RemoveTagFromCharacter( self.utils:Capitalise( character ), tag )
		end
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: AddTagToCharacter
-- Add a tag to a character.
-------------------------------------------------------------------------------------------------------------
function JambaComms:AddTagToCharacter( character, tag )
		
	-- If the character and tag are not nil then...
	if character and tag then
	
		-- Get the tag list for this character.
		local tagList = self.comms:GetTagListForCharacter( character )
		-- Add to the tag list.
		tagList[tag] = tag
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: RemoveTagFromCharacter
-- Add a tag to a character.
-------------------------------------------------------------------------------------------------------------
function JambaComms:RemoveTagFromCharacter( character, tag )
	
	-- If the character and tag are not nil then...
	if character and tag then
		-- Get the tag list for this character.
		local tagList = self.comms:GetTagListForCharacter( character )
		-- Remove from the tag list.
		tagList[tag] = nil
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:CharacterHasTag
-- character - the character whose tags to inspect.
-- tag - the tag to check for.
-- Returns true if this character has the tag supplied.
-------------------------------------------------------------------------------------------------------------
function JambaComms:CharacterHasTag( character, tag )

	local hasTag = false
	
	if tag then
		local tagList = self.comms:GetTagListForCharacter( character )
		if tagList[tag] then
			hasTag = true
		end
	end
		
	return hasTag
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:ConfirmCharacterHasAllTag
-- character - the character whose tags to inspect.
-- Adds the all tag to the character if it is not already there.
-------------------------------------------------------------------------------------------------------------
function JambaComms:ConfirmCharacterHasAllTag( character )

	-- Make sure there is a tag of all for this character.
	if not self.comms:CharacterHasTag( character, L["all"] ) then
		self.comms:AddTagToCharacter( character, L["all"] )
	end

end
	
-------------------------------------------------------------------------------------------------------------
-- JambaComms: ConfirmPlayerIsMember
-- Makes sure that this player is a team member.  Enables if previously not a member.
-------------------------------------------------------------------------------------------------------------
function JambaComms:ConfirmPlayerIsMember()

	if not self.comms:CharacterIsAMember( self.playerName ) then
		-- Then add as a member and enable.
		self.comms:AddMember( nil, self.playerName )
		self.comms:ToggleCharacterEnabled( nil, self.playerName, true )
	end

	-- Make sure there is a tag of all for this character.
	self.comms:ConfirmCharacterHasAllTag( self.playerName ) 
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: ConfirmThereIsAMaster
-- Make sure there is a master, if none, set this character.
-------------------------------------------------------------------------------------------------------------
function JambaComms:ConfirmThereIsAMaster()

	-- Read the db option for master.  Is it set?
	if not self.db.master or self.db.master:trim() == "" then
		-- No, set it to self.
		self.comms:SetMaster( nil, self.playerName )
	end
	
	-- Is the master in the member list?
	if not self.comms:CharacterIsAMember( self.db.master ) then
		-- No, set self as master.
		self.comms:SetMaster( nil, self.playerName )
	end
	 
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:SetMeAsMaster
-- Set the master to this character.
-------------------------------------------------------------------------------------------------------------
function JambaComms:SetMeAsMaster( info, parameters )

	local tag = parameters
	local target = self.playerName
	
	if tag then 
		self.comms:MasterUpdateSendCommand( target, tag )
	else
		self.comms:MasterUpdateSetMaster( target )
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:SetMasterToTarget
-- Set the master to the target supplied.
-------------------------------------------------------------------------------------------------------------
function JambaComms:SetMasterToTarget( info, parameters )

	local target, tag = strsplit( " ", parameters )
	target = self.utils:Capitalise( target )
	if tag then 
		self.comms:MasterUpdateSendCommand( target, tag )
	else
		self.comms:MasterUpdateSetMaster( target )
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:MasterUpdateSendCommand
-- Send the master update command.
-------------------------------------------------------------------------------------------------------------
function JambaComms:MasterUpdateSendCommand( target, tag )

	self.comms:CommandAll( self.utils.COMMAND_SET_MASTER, target, tag )

end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:MasterUpdateReceiveCommand
-- Receive and handle the master update command.
-------------------------------------------------------------------------------------------------------------
function JambaComms:MasterUpdateReceiveCommand( target, tag )

	-- If this character responds to this tag...
	if self.comms:CharacterHasTag( self.playerName, tag ) then
		-- Then update the macro text for the target specified.
		self.comms:MasterUpdateSetMaster( target )
	end

end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:MasterUpdateSetMaster
-- Set a new master.
-------------------------------------------------------------------------------------------------------------
function JambaComms:MasterUpdateSetMaster( target )

	self.comms:SetMaster( nil, target )

end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:FocusUpdateMacroTextSetMaster
-- Set the Focus to the master
-------------------------------------------------------------------------------------------------------------
function JambaComms:FocusUpdateMacroTextSetMaster( info, parameters )

	-- The only parameter for this command is tag.  If there is a tag, send the command to all
	-- the members, otherwise just this character.
	local tag = parameters
	local target = self.db.master
	
	if tag then
		self.comms:FocusUpdateSendCommand( target, tag )
	else
		self.comms:FocusUpdateMacroText( target )
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:FocusUpdateMacroTextSetTarget
-- Set the Focus to a target.
-------------------------------------------------------------------------------------------------------------
function JambaComms:FocusUpdateMacroTextSetTarget( info, parameters )

	local target, tag = strsplit( " ", parameters )
	target = self.utils:Capitalise( target )
	if tag then 
		self.comms:FocusUpdateSendCommand( target, tag )
	else
		self.comms:FocusUpdateMacroText( target )
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:FocusUpdateSendCommand
-- Send the focus update command.
-------------------------------------------------------------------------------------------------------------
function JambaComms:FocusUpdateSendCommand( target, tag )

	self.comms:CommandAll( self.utils.COMMAND_SET_FOCUS, target, tag )

end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:FocusUpdateReceiveCommand
-- Receive and handle the focus update command.
-------------------------------------------------------------------------------------------------------------
function JambaComms:FocusUpdateReceiveCommand( target, tag )

	-- If this character responds to this tag...
	if self.comms:CharacterHasTag( self.playerName, tag ) then
		-- Then update the macro text for the target specified.
		self.comms:FocusUpdateMacroText( target )
	end

end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:FocusUpdateMacroText
-- focusTarget - the target for the focus command.
-- Updates the text on the set focus macro to the provided target.
-------------------------------------------------------------------------------------------------------------
function JambaComms:FocusUpdateMacroText( focusTarget )

	-- Was a focus target provided?
	if focusTarget then
		-- If not in combat then...
		if not InCombatLockdown() then
			-- Change the macro text on the secure button to set focus.
			JambaSecureButtonSetFocus:SetAttribute( "macrotext", "/focus "..focusTarget )
			-- Get the unit name for reporting purposes.
			local resolvedFocusTarget = UnitName( focusTarget )
			if not resolvedFocusTarget then
				resolvedFocusTarget = L["?"]
			end			
			self.comms:TellMaster( self.playerName, L["Focus target is now W (currently X).  "]( focusTarget, resolvedFocusTarget )..L["__KEY_PRESS_TO_ACTIVATE__"]( self.setFocusKey1, self.setFocusKey2 ) )			
		else
			 -- In combat, cannot change, tell the player.
			 self.comms:TellMaster( self.playerName, L["IN COMBAT - FAILED to update Focus target to X."](focusTarget) )
		end
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:MasterAsTargetUpdateMacroTextSetTarget
-- Set the master target to a target.
-------------------------------------------------------------------------------------------------------------
function JambaComms:MasterAsTargetUpdateMacroTextSetTarget( info, parameters )

	local target, tag = strsplit( " ", parameters )
	if tag then 
		self.comms:MasterAsTargetUpdateSendCommand( target, tag )
	else
		self.comms:MasterAsTargetUpdateMacroText( target )
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:MasterAsTargetUpdateSendCommand
-- Send the master target update command.
-------------------------------------------------------------------------------------------------------------
function JambaComms:MasterAsTargetUpdateSendCommand( target, tag )

	self.comms:CommandAll( self.utils.COMMAND_SET_MASTER_AS_TARGET, target, tag )

end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:MasterAsTargetUpdateReceiveCommand
-- Receive and handle the master target update command.
-------------------------------------------------------------------------------------------------------------
function JambaComms:MasterAsTargetUpdateReceiveCommand( target, tag )

	-- If this character responds to this tag...
	if self.comms:CharacterHasTag( self.playerName, tag ) then
		-- Then update the macro text for the master target specified.
		self.comms:MasterAsTargetUpdateMacroText( target )
	end

end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:MasterAsTargetUpdateMacroText
-- masterTarget - the target for the master target command.
-- Updates the text on the set master target macro to the provided target.
-------------------------------------------------------------------------------------------------------------
function JambaComms:MasterAsTargetUpdateMacroText( masterTarget )

	-- Was a master target provided?
	if masterTarget then
		-- If not in combat then...
		if not InCombatLockdown() then
			-- Change the macro text on the secure button to set ther master target.
			JambaSecureButtonSetMasterAsTarget:SetAttribute( "macrotext", "/target "..masterTarget )	
			self.comms:TellMaster( self.playerName, L["Master target is now X.  "]( masterTarget )..L["__KEY_PRESS_TO_ACTIVATE__"]( self.setMasterAsTargetKey1, self.setMasterAsTargetKey2 ) )
		else
			 -- In combat, cannot change, tell the player.
			 self.comms:TellMaster( self.playerName, L["IN COMBAT - FAILED to update master target to X."](masterTarget) )
		end
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:CharacterIsASlave
-- characterName - the name of the character to check for slave status.
-- Return true if the character is a slave (a member and not the master).
-------------------------------------------------------------------------------------------------------------
function JambaComms:CharacterIsASlave( characterName )
	
	local isASlave = false
	
	if self:CharacterIsAnEnabledMember( characterName ) then
		if not self:CharacterIsTheMaster( characterName ) then
			isASlave = true
		end
	end
	
	return isASlave
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:CharacterIsTheMaster
-- characterName - the name of the character to check for master status.
-- Return true if the character is the master.
-------------------------------------------------------------------------------------------------------------
function JambaComms:CharacterIsTheMaster( characterName )
	
	local isTheMaster = false
	
	if characterName == self:GetMasterName() then
		isTheMaster = true
	end
	
	return isTheMaster
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms: GetMasterName
-- Get the master for this character.
-------------------------------------------------------------------------------------------------------------
function JambaComms:GetMasterName()
	
	-- If no master, then use this character as the master.
	local masterName = self.playerName
	
	-- If there is a stored master...
	if self.db.master and strtrim(self.db.master) ~= "" then
		-- Use the stored master.
		masterName = self.db.master
	end
	
	return masterName
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:CharacterIsAMember
-- Check to see if a character is a member of this character's 'group'.
-- characterName - The name of the character to check for membership.
-------------------------------------------------------------------------------------------------------------
function JambaComms:CharacterIsAMember( characterName ) 

	local isMember = false
	
	if self.db.memberList[characterName] then
		isMember = true
	end
	
	return isMember
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:CharacterIsAnEnabledMember
-- Check to see if a character is a member of this character's 'group' and enabled.
-- characterName - The name of the character to check for membership and being enabled.
-------------------------------------------------------------------------------------------------------------
function JambaComms:CharacterIsAnEnabledMember( characterName ) 

	local isMemberAndEnabled = false

	if self.db.memberEnabled[characterName] then
		isMemberAndEnabled = true
	end
	
	return isMemberAndEnabled
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:CharacterIsAnEnabledMember
-- Return the enabled member list so other modules can iterate members.
-------------------------------------------------------------------------------------------------------------
function JambaComms:EnabledMemberList() 

	return self.db.memberEnabled
	
end

-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------
-- COMMUNICATION PROCESSING
-------------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------------

-------------------------------------------------------------------------------------------------------------
-- JambaComms:UpdateKeyBindings()
-- Update key bindings.
-------------------------------------------------------------------------------------------------------------
function JambaComms:UpdateKeyBindings( keyBindingFrame )

	self.comms:UpdateKeyBindingsSetFocus( keyBindingFrame )
	self.comms:UpdateKeyBindingsSetMasterAsTarget( keyBindingFrame )

end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:UpdateKeyBindingsSetFocus()
-- Update key bindings.
-------------------------------------------------------------------------------------------------------------
function JambaComms:UpdateKeyBindingsSetFocus( keyBindingFrame )
	
	-- Get the keys associated with Jamba Set Focus.
	self.setFocusKey1, self.setFocusKey2 = GetBindingKey( "JAMBASETFOCUS" )		
	
	-- If key 1 is set, then...
	if self.setFocusKey1 then 
		-- Update the set focus button to respond to this key.
		SetOverrideBindingClick( keyBindingFrame, false, self.setFocusKey1, "JambaSecureButtonSetFocus") 
	end
	
	-- If key 2 is set, then...
	if self.setFocusKey2 then 
		-- Update the set focus button to respond to this key.
		SetOverrideBindingClick( keyBindingFrame, false, self.setFocusKey2, "JambaSecureButtonSetFocus") 
	end	
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:UpdateKeyBindingsSetMasterAsTarget()
-- Update key bindings.
-------------------------------------------------------------------------------------------------------------
function JambaComms:UpdateKeyBindingsSetMasterAsTarget( keyBindingFrame )
	
	-- Get the keys associated with Jamba Set Focus.
	self.setMasterAsTargetKey1, self.setMasterAsTargetKey2 = GetBindingKey( "JAMBASETMASTERASTARGET" )		
	
	-- If key 1 is set, then...
	if self.setMasterAsTargetKey1 then 
		-- Update the set focus button to respond to this key.
		SetOverrideBindingClick( keyBindingFrame, false, self.setMasterAsTargetKey1, "JambaSecureButtonSetMasterAsTarget") 
	end
	
	-- If key 2 is set, then...
	if self.setMasterAsTargetKey2 then 
		-- Update the set focus button to respond to this key.
		SetOverrideBindingClick( keyBindingFrame, false, self.setMasterAsTargetKey2, "JambaSecureButtonSetMasterAsTarget") 
	end	
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:TellSelf
-- characterName - The name of the character to telling.
-- message - What to tell the character.
-------------------------------------------------------------------------------------------------------------
function JambaComms:TellSelf( characterName, message )

	LibStub( "AceConsole-3.0" ):Print( characterName..": "..message )

end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:TellMaster
-- characterName - The name of the character doing the telling.
-- message - What to tell the master (normal chat).
-------------------------------------------------------------------------------------------------------------
function JambaComms:TellMaster( characterName, message )
	
	self.comms:ProcessTellMaster( characterName, message, false )
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:TellMasterSpecial
-- characterName - The name of the character doing the telling.
-- message - What to tell the master (special characters).
-------------------------------------------------------------------------------------------------------------
function JambaComms:TellMasterSpecial( characterName, message )
	
	self.comms:ProcessTellMaster( characterName, message, true )
				
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:ProcessTellMaster
-- characterName - The name of the character doing the telling.
-- message - What to tell the master.
-------------------------------------------------------------------------------------------------------------
function JambaComms:ProcessTellMaster( characterName, message, isSpecial )
	
	-- Is the master set?  The master should always be set...
	if not self.db.master or self.db.master:trim() == "" then

		-- No, tell self instead.
		self.comms:TellSelf( characterName, message )
		
	else

		-- Yes, the master is set.
		-- Do stop spam from occurring when the master goes offline and forward system messages is on; only send
		-- a message if it wasn't just sent.
		local canSendMessage = true
		
		-- Is this message the same as the last one?
		if (characterName == self.lastTellMasterCharacterName) and (message == self.lastTellMasterMessage) then
			-- Yes, don't send it.
			canSendMessage = false
		end

		-- If the message can be sent...	
		if canSendMessage then

			-- Send to the master for default chat?
			if self.db.displayMessageInDefaultChat or (self.db.displaySpecialMessageInDefaultChat and isSpecial) then
				Jamba:JambaCommsMessageSend( 
					self.utils.COMMUNICATION_PREFIX,
					characterName..": "..message,
					self.utils.COMMUNICATION_WHISPER,
					self.db.master,
					self.utils.COMMUNICATION_PRIORITY_BULK
				)
			end
			
			if not isSpecial then
			
				-- Send the message to a channel?
				if self.db.sendMessageToChannel then
					-- Do it.
					local channelId, channelName, channelInstance = GetChannelName( self.db.messageChannelName );
					if channelId and channelId > 0 then
						ChatThrottleLib:SendChatMessage( self.utils.COMMUNICATION_PRIORITY_BULK, nil, message, "CHANNEL", nil, channelId )
					end
				end

				-- Send the message to guild chat?
				if self.db.sendMessageToGuildChat then
					if IsInGuild() then
						ChatThrottleLib:SendChatMessage( self.utils.COMMUNICATION_PRIORITY_BULK, nil, message, "GUILD" )
					end
				end
				
				-- Send the message to guild officer chat?
				if self.db.sendMessageToGuildOfficerChat then
					if IsInGuild() then
						ChatThrottleLib:SendChatMessage( self.utils.COMMUNICATION_PRIORITY_BULK, nil, message, "OFFICER" )
					end
				end
				
				-- Send the message to party chat?
				if self.db.sendMessageToPartyChat then
					if GetNumPartyMembers() > 0 then
						ChatThrottleLib:SendChatMessage( self.utils.COMMUNICATION_PRIORITY_BULK, nil, message, "PARTY" )
					end
				end
				
				-- Send the message to master via whisper chat?
				if self.db.sendMessageToMasterViaWhisper then
					-- Don't whisper to yourself; it's not cool.
					if self.playerName ~= self.comms:GetMasterName() then
						ChatThrottleLib:SendChatMessage( self.utils.COMMUNICATION_PRIORITY_BULK, nil, message, "WHISPER", nil, self.comms:GetMasterName() )
					end
				end

			end
															
			-- And update the last sent message saves.
			self.lastTellMasterCharacterName = characterName
			self.lastTellMasterMessage = message

		else
		
			-- Message was not sent.  Reset last sent message saves.
			self.lastTellMasterCharacterName = ""
			self.lastTellMasterMessage = ""
			
		end
		
	end
			
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:WarnMaster
-- message - What to warn the master.
-------------------------------------------------------------------------------------------------------------
function JambaComms:WarnMaster( message )

	local sendWarningTo = self.db.master
	
	-- Is the master set?  The master should always be set...
	if not self.db.master or self.db.master:trim() == "" then
	
		-- No, send warning to self instead.
		sendWarningTo = self.playerName	
		
	end

	-- Send warning via masters default chat or raid warning?
	if self.db.displayWarningInDefaultChat or self.db.displayWarningInRaidWarning then
		-- Yes, send the warning.
		Jamba:JambaCommsMessageSend( 
			self.utils.WARNING_PREFIX,
			message,
			self.utils.COMMUNICATION_WHISPER,
			sendWarningTo,
			self.utils.COMMUNICATION_PRIORITY_NORMAL
		)
	end
	
	-- Send the warning to a channel?
	if self.db.sendWarningToChannel then
		-- Do it.
		local channelId, channelName, channelInstance = GetChannelName( self.db.warningChannelName );
		if channelId and channelId > 0 then
			ChatThrottleLib:SendChatMessage( self.utils.COMMUNICATION_PRIORITY_NORMAL, nil, message, "CHANNEL", nil, channelId )
		end
	end

	-- Send the warning to guild chat?
	if self.db.sendWarningToGuildChat then
		if IsInGuild() then
			ChatThrottleLib:SendChatMessage( self.utils.COMMUNICATION_PRIORITY_NORMAL, nil, message, "GUILD" )
		end
	end
	
	-- Send the warning to guild officer chat?
	if self.db.sendWarningToGuildOfficerChat then
		if IsInGuild() then
			ChatThrottleLib:SendChatMessage( self.utils.COMMUNICATION_PRIORITY_NORMAL, nil, message, "OFFICER" )
		end
	end
	
	-- Send the warning to party chat?
	if self.db.sendWarningToPartyChat then
		if GetNumPartyMembers() > 0 then
			ChatThrottleLib:SendChatMessage( self.utils.COMMUNICATION_PRIORITY_NORMAL, nil, message, "PARTY" )
		end
	end
	
	-- Send the warning to master via whisper chat?
	if self.db.sendWarningToMasterViaWhisper then
		-- Don't whisper to yourself; it's not cool.
		if self.playerName ~= self.comms:GetMasterName() then
			ChatThrottleLib:SendChatMessage( self.utils.COMMUNICATION_PRIORITY_NORMAL, nil, message, "WHISPER", nil, self.comms:GetMasterName() )
		end
	end
			
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:DisplayWarning
-- Display a warning.
-------------------------------------------------------------------------------------------------------------
function JambaComms:DisplayWarning( sender, message )

	if self.db.displayWarningInDefaultChat then
		self.comms:TellSelf( sender, message )
	end

	if self.db.displayWarningInRaidWarning then
		RaidNotice_AddMessage( RaidWarningFrame, sender..": "..message, ChatTypeInfo["RAID_WARNING"] )
		PlaySound( "RaidWarning" )
	end

end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:LeaveChannel
-- Leaves the channel specified.
-------------------------------------------------------------------------------------------------------------
function JambaComms:LeaveChannel( channelName )

	if channelName and channelName:trim() ~= "" then
		LeaveChannelByName( channelName )
	end

end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:JoinWarningChannel
-- Joins the channel specified for warnings if requested.
-------------------------------------------------------------------------------------------------------------
function JambaComms:JoinWarningChannel()

	if self.db.sendWarningToChannel then
		if self.db.warningChannelName and self.db.warningChannelName:trim() ~= "" then
			if self.db.warningChannelPassword and self.db.warningChannelPassword:trim() ~= "" then
				local channelType, channelName = JoinPermanentChannel( self.db.warningChannelName, self.db.warningChannelPassword )
			end
		end
	end

end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:JoinMessageChannel
-- Joins the channel specified for messages if requested.
-------------------------------------------------------------------------------------------------------------
function JambaComms:JoinMessageChannel()

	if self.db.sendMessageToChannel then
		if self.db.messageChannelName and self.db.messageChannelName:trim() ~= "" then
			if self.db.messageChannelPassword and self.db.messageChannelPassword:trim() ~= "" then
				local channelType, channelName = JoinPermanentChannel( self.db.messageChannelName, self.db.messageChannelPassword )
			end
		end
	end

end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:MessageReceived
-- Handle incoming messages on communication channels.
-------------------------------------------------------------------------------------------------------------
function JambaComms:MessageReceived( prefix, message, distribution, sender )
	
	-- Check to see if the message is for JambaComms.
	if prefix == self.utils.COMMUNICATION_PREFIX then		
		-- Yes, print to console.
		LibStub( "AceConsole-3.0" ):Print( message )
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:CommandAll
-- Send a command to all in the team.
-------------------------------------------------------------------------------------------------------------
function JambaComms:CommandAll( command, ... )

	--LibStub( "AceConsole-3.0" ):Print( "SENDING COMMAND ALL:", command )
	
	-- Get the message to send.
	local message = self:CreateCommandToSend( command, ... )

	-- Send the message to all the characters in the team.
	for characterName, enabled in pairs( self.db.memberEnabled ) do
		if enabled then
			Jamba:JambaCommsMessageSend( 
				self.utils.COMMAND_PREFIX,
				message,
				self.utils.COMMUNICATION_WHISPER,
				characterName,
				self.utils.COMMUNICATION_PRIORITY_ALERT
			)
		end
	end
		
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:CommandSlaves
-- Send a command to all in the team except the master.
-------------------------------------------------------------------------------------------------------------
function JambaComms:CommandSlaves( command, ... )

	--LibStub( "AceConsole-3.0" ):Print( "SENDING COMMAND SLAVES:", command, ... )
	
	-- Get the message to send.
	local message = self:CreateCommandToSend( command, ... )

	-- Send the message to all the characters in the team.
	for characterName, enabled in pairs( self.db.memberEnabled ) do
		if enabled then
			if not self:CharacterIsTheMaster( characterName) then
				Jamba:JambaCommsMessageSend( 
					self.utils.COMMAND_PREFIX,
					message,
					self.utils.COMMUNICATION_WHISPER,
					characterName,
					self.utils.COMMUNICATION_PRIORITY_ALERT
				)
			end
		end
	end
		
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:CreateCommandToSend
-- Creates a command to send.
-------------------------------------------------------------------------------------------------------------
function JambaComms:CreateCommandToSend( command, ... )

	-- Start the message with the command and a seperator, followed by the .
	local message = command..self.utils.COMMAND_SEPERATOR

	-- Add any arguments to the message (seperated).
	local numberArguments = select( "#", ... )
	for iterateArguments = 1, numberArguments do
		local argument = select( iterateArguments, ... )
		message = message..argument..self.utils.COMMAND_ARGUMENT_SEPERATOR
	end

	-- Return the command to send.
	return message
			
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:WarningReceived
-- Handle incoming warnings on communication channels.
-------------------------------------------------------------------------------------------------------------
function JambaComms:WarningReceived( prefix, message, distribution, sender )
	
	-- Check to see if the warning is for JambaComms.
	if prefix == self.utils.WARNING_PREFIX then
		-- See if the sender is in the characters group.
		if self:CharacterIsAMember( sender ) then
			-- If this character is the master...
			if self:CharacterIsTheMaster( self.playerName ) then
				self:DisplayWarning( sender, message )
			end
		end
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:CommandReceived
-- Handle incoming commands on communication channels.
-------------------------------------------------------------------------------------------------------------
function JambaComms:CommandReceived( prefix, message, distribution, sender )

	-- Check to see if the command is for JambaComms.
	if prefix == self.utils.COMMAND_PREFIX then
		-- See if the sender is in the characters group.
		if self:CharacterIsAMember( sender ) then
			-- Split the command into its components.
			local command, arguments = strsplit( self.utils.COMMAND_SEPERATOR, message )		
			-- Handle the command.
			self:HandleCommand( sender, command, arguments )
		end
	end
	
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:HandleCommand
-- Handle a recieved command.
-------------------------------------------------------------------------------------------------------------
function JambaComms:HandleCommand( sender, command, arguments )

	--LibStub( "AceConsole-3.0" ):Print( "HANDLING COMMAND:", command, " FROM ", sender )
	
	-- Set focus?
	if command == self.utils.COMMAND_SET_FOCUS then
		local focusTarget, focusTag = strsplit( self.utils.COMMAND_ARGUMENT_SEPERATOR, arguments )
		self.comms:FocusUpdateReceiveCommand( focusTarget, focusTag )
	end

	-- Set master?
	if command == self.utils.COMMAND_SET_MASTER then
		local masterTarget, masterTag = strsplit( self.utils.COMMAND_ARGUMENT_SEPERATOR, arguments )
		self.comms:MasterUpdateReceiveCommand( masterTarget, masterTag )
	end

	-- Take taxi?
	if command == self.utils.COMMAND_TAKE_TAXI then
		local nodeName = strsplit( self.utils.COMMAND_ARGUMENT_SEPERATOR, arguments )
		if JambaTaxi then
			JambaTaxi:DoTakeTaxi( sender, nodeName )
		end
	end

	-- Leave party?
	if command == self.utils.COMMAND_LEAVE_GROUP then
		if JambaGroup then
			JambaGroup:LeaveTheGroup()
		end
	end

	-- Follow target?
	if command == self.utils.COMMAND_FOLLOW_TARGET then
		if JambaFollow then
			local followTarget, followTag = strsplit( self.utils.COMMAND_ARGUMENT_SEPERATOR, arguments )
			JambaFollow:FollowTargetReceiveCommand( followTarget, followTag )
		end
	end

	-- Auto follow after combat?
	if command == self.utils.COMMAND_AUTO_FOLLOW_AFTER_COMBAT then
		if JambaFollow then
			local autoFollowSetToOn, autoFollowTag = strsplit( self.utils.COMMAND_ARGUMENT_SEPERATOR, arguments )
			JambaFollow:AutoFollowAfterCombatReceiveCommand( autoFollowSetToOn, autoFollowTag )
		end
	end

	-- Follow strobe on?
	if command == self.utils.COMMAND_FOLLOW_STROBE_ON then
		if JambaFollow then
			local followStrobeTarget, followStrobeTag = strsplit( self.utils.COMMAND_ARGUMENT_SEPERATOR, arguments )
			JambaFollow:FollowStrobeOnReceiveCommand( followStrobeTarget, followStrobeTag )
		end
	end

	-- Follow strobe off?
	if command == self.utils.COMMAND_FOLLOW_STROBE_OFF then
		if JambaFollow then
			local followStrobeTarget, followStrobeTag = strsplit( self.utils.COMMAND_ARGUMENT_SEPERATOR, arguments )
			JambaFollow:FollowStrobeOffReceiveCommand( followStrobeTarget, followStrobeTag )
		end
	end
		
	-- Quest commands?
	if JambaQuest then
	
		-- Was there a quest index sent?
		local questIndex = strsplit( self.utils.COMMAND_ARGUMENT_SEPERATOR, arguments )
	
		-- Let Jamba-Quest handle these commands.
		JambaQuest:HandleCommand( sender, command, questIndex )
		
	end

	-- Pushed Settings?
	if command == self.utils.COMMAND_PUSH_SETTINGS then
		
		-- Get the settings information.
		local settingsForName, serializedSettings = strsplit( self.utils.COMMAND_ARGUMENT_SEPERATOR, arguments )
		
		-- Deserialize the settings.
		local success, settings = LibStub( "AceSerializer-3.0" ):Deserialize( serializedSettings )

		-- If successful deserializing...
		if success then
			-- And the sender is not this character.
			if sender ~= self.playerName then
			
				-- Are these JambaTalk settings?
				if JambaTalk then
					if settingsForName == "Jamba-Talk" then
						JambaTalk:ReceiveSettings( settings )
					end
				end

				-- Are these JambaTaxi settings?
				if JambaTaxi then
					if settingsForName == "Jamba-Taxi" then
						JambaTaxi:ReceiveSettings( settings )
					end
				end

				-- Are these JambaQuest settings?
				if JambaQuest then
					if settingsForName == "Jamba-Quest" then
						JambaQuest:ReceiveSettings( settings )
					end
				end

				-- Are these JambaNo settings?
				if JambaNo then
					if settingsForName == "Jamba-No" then
						JambaNo:ReceiveSettings( settings )
					end
				end

				-- Are these JambaGroup settings?
				if JambaGroup then
					if settingsForName == "Jamba-Group" then
						JambaGroup:ReceiveSettings( settings )
					end
				end

				-- Are these JambaMerchant settings?
				if JambaMerchant then
					if settingsForName == "Jamba-Merchant" then
						JambaMerchant:ReceiveSettings( settings )
					end
				end
				
				-- Are these JambaComms settings?
				if JambaComms then
					if settingsForName == "Jamba-Core-Team" then
						JambaComms:ReceiveSettingsTeam( settings )
					end
					if settingsForName == "Jamba-Core-Tags" then
						JambaComms:ReceiveSettingsTags( settings )
					end					
					if settingsForName == "Jamba-Core-Chat" then
						JambaComms:ReceiveSettingsChat( settings )
					end					
					if settingsForName == "Jamba-Core-Warning" then
						JambaComms:ReceiveSettingsWarning( settings )
					end						
				end

				-- Are these JambaMe settings?
				if JambaMe then
					if settingsForName == "Jamba-Me" then
						JambaMe:ReceiveSettings( settings )
					end
				end

				-- Are these JambaFollow settings?
				if JambaFollow then
					if settingsForName == "Jamba-Follow" then
						JambaFollow:ReceiveSettings( settings )
					end
				end
																								
				-- Tell the player, settings update successfully.				
				self:TellSelf( self.playerName, settingsForName..L[" settings updated successfully."] )
				
			end
		else
			self:TellSelf( self.playerName, L["Failed to read settings received from "]..sender..L["."] )
		end
	end
		
end

-------------------------------------------------------------------------------------------------------------
-- JambaComms:PushSettings
-- Push settings to other enabled and conncted characters.
-------------------------------------------------------------------------------------------------------------
function JambaComms:PushSettings( settings, settingsForName )

	-- Serialize the settings.
	local serializedSettings = LibStub( "AceSerializer-3.0" ):Serialize( settings )

	-- Send a push settings command to all.
	self:CommandAll( self.utils.COMMAND_PUSH_SETTINGS, settingsForName, serializedSettings )
	
end
