--[[
	### ItemValue ###
	ItemValue Core
	
	Used Libraries:
	Ace2, GratuityLib, Deformat, ItemBonusLib, TipHookerLib, Waterfall-1.0
--]]

ItemValue = AceLibrary("AceAddon-2.0"):new("AceEvent-2.0", "AceConsole-2.0", "AceHook-2.1", "AceDB-2.0", "AceDebug-2.0", "AceComm-2.0", "FuBarPlugin-2.0")
ItemValue.revision = tonumber(string.sub("$Revision: 65280 $", 12, -3))
ItemValue.version = "r" .. ItemValue.revision
ItemValue.date = string.sub("$Date: 2008-03-21 23:20:18 +0100 (Fr, 21 Mrz 2008) $", 8, 17)

-- Libraries
local L = AceLibrary("AceLocale-2.2"):new("ItemValue")
local R = setmetatable({}, {["__index"] = function(t) L:GetReverseTranslation(t) end})

local AceOO = AceLibrary("AceOO-2.0")
local TipHooker = AceLibrary("TipHooker-1.0")
local Gratuity = AceLibrary("Gratuity-2.0")
local Waterfall = AceLibrary("Waterfall-1.0")
local Dewdrop = AceLibrary("Dewdrop-2.0")
local ItemBonusLib = AceLibrary("ItemBonusLib-1.0")
local AceConfigDialog = LibStub("AceConfigDialog-3.0")

local self = ItemValue

-- Stolen from PitBull
local tnew, tdel, tnewHash, tddel
do
	local list = setmetatable({}, {__mode='k'})
	function tnew(...)
		local t = next(list)
		if t then
			list[t] = nil
			for i = 1, select('#', ...) do
				t[i] = select(i, ...)
			end
			return t
		else
			return { ... }
		end
	end
	
	function tnewHash(...)
		local t = next(list)
		if t then
			list[t] = nil
		else
			t = {}
		end
		for i = 1, select('#', ...), 2 do
			t[select(i, ...)] = select(i+1, ...)
		end
		return t
	end
	
	function tdel(t)
		for k in pairs(t) do
			t[k] = nil
		end
		list[t] = true
		return nil
	end

	function tddel(t)
		if type(t) ~= "table" then
			return nil
		end
		for k,v in pairs(t) do
			t[k] = tddel(v)
		end
		return tdel(t)
	end
end
self.tnew, self.tdel, self.tddel, self.tnewHash = tnew, tdel, tddel, tnewHash

-- Table for Sets
local SetByName = {}
self.SetByName = SetByName
local SetById = {}
self.SetById = SetById

local JewelByName
local JewelByEnchantId

-- Table for SetTypes
local SetTypes = {}
self.SetTypes = SetTypes

local PlayerValueFrame
local InspectValueFrame
local SocketAmountCache = setmetatable({}, {__mode = "kv"})
local SocketInfoCache = setmetatable({}, {__mode = "kv"})
local TooltipTextCache = setmetatable({}, {__mode = "kv"})
local TooltipTextCache2 = setmetatable({}, {__mode = "kv"})

local IsValidJewel = function(i, j)
	for JewelName in pairs(self.JewelByName) do
		if j == JewelName then return true end
	end
end

local function optGetter(info)
	local key = info[#info]
	return self.db.profile[key]
end
	
local function optSetter(info, value)
	local key = info[#info]
	self.db.profile[key] = value
	self:Refresh(key)
end

function ItemValue:Refresh()
	if self.db.profile.TooltipEnable then
		TipHooker:Hook(self.AddDataToTooltip, "item")
	else
		TipHooker:Unhook(self.AddDataToTooltip, "item")
	end

	if self.db.profile.EQValueEnabled then
		self:CreatePlayerValueFrame()
		PlayerValueFrame:Show()
	else
		if PlayerValueFrame then
			PlayerValueFrame:Hide()
		end
	end
	
	if self.db.profile.InspectEnabled then
		self:CreateInspectValueFrame()
		InspectValueFrame:Show()
	else
		if InspectValueFrame then
			InspectValueFrame:Hide()
		end
	end
	
	if self.db.profile.CustomSetOrder then
		self.db.profile.SetIds = self.db.profile.SetIds or tnew()
		for SetId, Set in pairs(SetById) do
			self.db.profile.SetIds[Set.Name] = SetId
		end
		self:UpdateSorting()
	else
		if self.db.profile.SetIds and type(self.db.profile.SetIds) == "table" then
			self.db.profile.SetIds = tdel(self.db.profile.SetIds)
		end
	end
	
	ItemValue:ClearCache()
end

local Options = {
	type = "group",
	--childGroups = "tree",
	name = "ItemValue",
	--plugins = {},
	get = optGetter,
	set = optSetter,
	args = {
		NewSets = {
			order = 1,
			type = "group",
			name = "New Set",
			desc = "Creates new Sets.",
			args = {},
		},
		Sets = {
			order	= 2,
			type	= "group",
			name	= "Sets",
			desc	= "Configuration of ItemValue Sets.",
			args	= {},
		},
		Settings = {
			type = "group",
			order = 3,
			name = "Settings",
			desc = "General settings about ItemValue.",
			args = {
				Modifications = {
					order	= 1,
					type	= "group",
					name	= "Item Modifications",
					desc	= "How enchants and jewels should be handled.",
					args	= {
						Enchants = {
							order	= 1,
							type	= "group",
							name	= "Enchants",
							desc	= "Enchant related settings.",
							inline	= true,
							args	= {
								EnchantHandling = {
									order	= 1,
									type	= "select",
									style	= "radio",
									name	= "Enchant Handling",
									desc	= "Defines if item enchants should be included in the calculation.",
									values	= {"Remove enchants", "Keep enchants", "Always use own enchants", "Fill with own enchants"},
								},
							},
						},
						Jewels = {
							order	= 2,
							type	= "group",
							name	= "Jewels",
							desc	= "Jewel related settings.",
							inline	= true,
							args	= {
								JewelHandling = {
									order	= 1,
									type	= "select",
									name	= "Jewel Handling",
									desc	= "Defines if jewels should be included in the calculation.",
									values = {"Remove jewels", "Keep jewels", "Always use default jewels", "Fill with default jewels", "Use best possible jewels"},
								},
								DefaultJewels = {
									order	= 2,
									type	= "group",
									name	= "Default Jewels",
									inline	= true,
									args	= {
										DefaultJewel = {
											order	= 2,
											type	= "input",
											name	= "Default Jewel",
											desc	= "What jewel should be used for \"Always Default\" and \"Fill with Default\" modes.",
											disabled = function()
												return self.db.profile.MultipleDefaultJewels or 
												not(self.db.profile.JewelHandling == 3 --[[ "Always default" ]]or 
												self.db.profile.JewelHandling == 4 --[["Fill with default"]])
											end,
											validate = IsValidJewel,
											usage = "<Jewel Name>",
										},
										MultipleDefaultJewels = {
											order	= 3,
											type	= "toggle",
											name	= "Multiple Default Jewels",
											desc	= "Allows you to define different default jewels for diffently colorored sockets.",
											disabled = function()
												return not(
												 self.db.profile.JewelHandling == 3 or	--Always default"
												 self.db.profile.JewelHandling == 4)	--Fill with default
											end,
										},
										DefaultJewel_red = {
											order	= 4,
											type	= "input",
											name	= "Default Jewel, Red",
											desc	= "What red jewel should be used for \"Always Default\" and \"Fill with Default\" modes.",
											disabled = function()
												return not(self.db.profile.MultipleDefaultJewels and 
												 self.db.profile.JewelHandling == 3 or	--Always default"
												 self.db.profile.JewelHandling == 4)	--Fill with default
											end,
											validate = IsValidJewel,
											usage = "<Jewel Name>",
										},
										DefaultJewel_yellow = {
											order	= 5,
											type	= "input",
											name	= "Default Jewel, Yellow",
											desc	= "What yellow jewel should be used for \"Always Default\" and \"Fill with Default\" modes.",
											disabled = function()
												return not(self.db.profile.MultipleDefaultJewels and
												 self.db.profile.JewelHandling == 3 or	--Always default"
												 self.db.profile.JewelHandling == 4)	--Fill with default
											end,
											validate = IsValidJewel,
											usage = "<Jewel Name>",
										},
										DefaultJewel_blue = {
											order	= 6,
											type	= "input",
											name	= "Default Jewel, Blue",
											desc	= "What blue jewel should be used for \"Always Default\" and \"Fill with Default\" modes.",
											disabled = function()
												return not(self.db.profile.MultipleDefaultJewels and
												 self.db.profile.JewelHandling == 3 or	--Always default"
												 self.db.profile.JewelHandling == 4)	--Fill with default
											end,
											validate = IsValidJewel,
											usage = "<Jewel Name>",
										},
										DefaultJewel_meta = {
											order	= 7,
											type	= "input",
											name	= "Default Jewel, Meta",
											desc	= "What meta jewel should be used for \"Always Default\" and \"Fill with Default\" modes.",
											disabled = function()
												return not(self.db.profile.MultipleDefaultJewels and
												 self.db.profile.JewelHandling == 3 or	--Always default"
												 self.db.profile.JewelHandling == 4)	--Fill with default
											end,
											validate = IsValidJewel,
											usage = "<Jewel Name>",
										},
									},
								},
							},
						},
					},
				},
				Tooltip = {
					order	= 2,
					type	= "group",
					name	= "Tooltip",
					desc	= "Tooltip Display Settings",
					args	= {
						TooltipEnable = {
							order	= 1,
							type	= "toggle",
							name	= "Tooltip Enable",
							desc	= "Enables ALL tooltip display.",
						},
						TooltipShowPoints = {
							order	= 2,
							type	= "toggle",
							name	= "Show Points",
							desc	= "Shows the Value of an item in its tooltip (so you can choose to only show the comparison without the actual value).",
						},
						Comparison = {
							order	= 3,
							type	= "group",
							name	= "Comparison",
							desc	= "Settings about the comparison display in the tooltip.",
							inline	= true,
							args	= {
								TooltipComparisonMode = {
									order	= 3,
									type	= "select",
									name	= "Comparison Mode",
									desc	= "If and how informations of other equipped items should be displayed.",
									values	= {"Don't Compare", "Absolute", "Delta", "Percent"},
								},
								TooltipSwapColor = {
									order	= 4,
									type	= "toggle",
									name	= "Swap Colors",
									desc	= "Swap the comparison colors?",
								},
								TooltipSwapComparison = {
									order	= 4,
									type	= "toggle",
									name	= "Swap Comparison",
									desc	= "Swap the comparison (+5 becomes -5)?",
								},
								TooltipDoubleSided = {
									order	= 5,
									type	= "toggle",
									name	= "Double Sided",
									desc	= "Displays set names on the left side and item values on the right side.",
								},
							},
						},
					},
				},
				EQValue = {
					order = 3,
					type = "group",
					name = "Player Equip Value",
					desc = "All ItemValues of a player",
					args = {
						Self = {
							order = 1,
							type = "group",
							name = "Self",
							desc = "Own equipment value",
							inline = true,
							args = {
								EQValueEnabled = {
									order	= 2,
									type	= "toggle",
									name	= "EQValue Enabled",
									desc	= "Shows own equipment value in the charsheet.",
								},
							}
						},
						Inspect = {
							order = 2,
							type = "group",
							name = "Inspect",
							desc = "Inspect equipment value",
							inline = true,
							args = {
								InspectEnabled = {
									order	= 1,
									type	= "toggle",
									name	= "Inspect Enabled",
									desc	= "Print out targets Equipment Value while inspecting.",
								},
								ShowPoints = {
									order	= 2,
									type	= "toggle",
									name	= "Show Points",
									desc	= "Shows the equipment value of a player.",
									disabled = function() return not (self.db.profile.InspectEnabled) end,
								},
								InspectComparisonMode = {
									order	= 3,
									type	= "select",
									name	= "Comparison Mode",
									desc	= "How your equipment points should be compared to your targets.",
									values	= {"Don't Compare", "Absolute", "Delta", "Percent"},
									disabled = function() return not (self.db.profile.InspectEnabled) end,
								},
								InspectSwapColor = {
									order	= 4,
									type	= "toggle",
									name	= "Swap Colors",
									desc	= "Swap the comparison colors?",
									disabled = function() return not (self.db.profile.InspectEnabled) end,
								},
								InspectSwapComparison = {
									order	= 5,
									type	= "toggle",
									name	= "Swap Comparison",
									desc	= "Swap the comparison (+5 becomes -5)?",
									disabled = function() return not (self.db.profile.InspectEnabled) end,
								},
							},
						},
					},
				},
				Sorting = {
					order = 4,
					type = "group",
					name = "Custom Set Sorting",
					desc = "Settings about custom Set ordering.",
					args = {
						CustomSetOrder = {
							order = 1,
							type = "toggle",
							name = "Enable",
							desc = "Enables or disables custom sorting of Sets.",
						},
					},
				},
			},
		},
		Tools = {
			type = "group",
			order = 4,
			name = "Tools",
			desc = "Tools",
			args = {
				BestJewels = {
					order	= 1,
					type	= "input",
					name	= "Calculate Best Jewels",
					desc	= "Calculates the best possible Jewel combination for an ItemLink.",
					usage 	= "<ItemLink>",
					get		= false,
					set		= function(i,v) self:PrintBestJewels(v) end,
				},
				ListJewels = {
					order	= 2,
					type	= "input",
					name	= L["List Jewels"],
					desc	= L["Lists all known Jewels."],
					get		= false,
					usage	= "\"all\" OR \"enabled\" OR \"disabled\" OR <Part of Jewelname>",
					set	= function(i, v)
						local color = {"|cFFFFFFFF[", "|cFF1EFF00[", "|cFF0070DD[", "|cFFA335EE[", "|cFFFF8000["}
						self:Print(string.format(L["Printing |cffffff78%s|r jewels..."], v))
						local shown = 0
						local total = 0
						for Name, Jewel in pairs(self.JewelByName) do
							if (v == "all") or
							 (v == "enabled" and self.db.profile.EnabledJewels[Name]) or
							 (v == "disabled" and not self.db.profile.EnabledJewels[Name]) or
							 (strfind(strlower(Name), strlower(v))) then
								
								local _, ItemLink = GetItemInfo(Jewel.ItemId)
								if not ItemLink then
									-- Not in local cache -> get color data from db
									ItemLink = color[Jewel.Quality] .. Name .. "]|r (*)"
								end
								self:Print(" ", ItemLink)
								shown = shown + 1
							end
							total = total + 1
						end
						self:Print(string.format(L["%s/%s Jewels shown."], shown, total))
					end,
				},
			}
		}
	},
}

self.Options = Options

local DefaultSettings = {
	PlayerValue = {
		Self = {
			FrameHeight = 105,
			FrameWidth = 200,
			FrameBackgroundColor = {0, 0, 0, 0.3},
			BorderColor = {1, 1, 1, 1},
		},
		Inspect = {
			FrameHeight = 105,
			FrameWidth = 173,
			FrameBackgroundColor = {0, 0, 0, 0.3},
			BorderColor = {1, 1, 1, 1},
		},
	},
	
	TooltipEnable = true,
	TooltipShowPoints = true,
	TooltipComparisonMode = 3,
	TooltipSwapColor = false,
	TooltipSwapComparison = false,
	TooltipDoubleSided = false,
	
	JewelHandling = 2,
	EnchantHandling = 4,
	
	EQValueEnabled = false,
	InspectEnabled = false,
	InspectComparisonMode = 3,
	
	EnabledJewels = {
		["*"] = true,
	},
	
	Jewels = {},
	DisplayInTooltip = {
		["*"] = true,
	},
	Round = {
		["*"] = 1,
	},
	CustomSetSorting = false,
	--SetIds = nil,
}

function ItemValue:OnInitialize()
	self:RegisterDB("ItemValueDB", "ItemValueDBPerChar")
	self:RegisterDefaults("profile", DefaultSettings)
	self:SetCommPrefix("ItemValue")
	self:RegisterComm("ItemValue", "WHISPER")
	
	--self:RegisterChatCommand({"/ivbj"}, )
	LibStub("AceConfig-3.0"):RegisterOptionsTable("ItemValue", Options)
	LibStub("AceConfigDialog-3.0"):AddToBlizOptions("ItemValue")
	
	self:RegisterChatCommand({"/iv", "/itemvalue"}, {
		type = "execute",
		name = "ItemValue Config Dialog",
		desc = "ItemValue Config Dialog",
		func = function() AceConfigDialog:Open("ItemValue") end,
	})
	
	-- FuBar
	self.hasIcon = true
	self.hasNoText = false
	self.defaultPosition = "RIGHT"
	self.cannotDetachTooltip = true
	self.overrideMenu = true
	self.independentProfile = true
	self.hideWithoutStandby = true
	self.hideMenuTitle = true
	self.blizzardTooltip = true
	
	self:SetIcon("Interface\\Icons\\Ability_Warrior_Disarm")
	
	self.OnMenuRequest = {
		type = "group",
		name = "ItemValue",
		desc = "ItemValue",
		args = {}
	}
	
	function self:OnDataUpdate()
		self:UpdateText()
	end
	
	function self:OnTextUpdate()
		self:SetText("ItemValue")
	end
	
	function self:OnClick()
		AceConfigDialog:Open("ItemValue")
	end
	
	function self:OnTooltipUpdate()
		GameTooltip:AddLine("ItemValue")
		GameTooltip:AddLine(" ")
		GameTooltip:AddLine(L["|cffffff00Left-Click|r to change settings via Waterfall."], 0.2, 1, 0.2)
		GameTooltip:AddLine(L["|cffffff00Right-Click|r to change settings via Dewdrop."], 0.2, 1, 0.2)
	end
end

local first = true
function ItemValue:OnEnable()
	if first then
		first = false
		if self.LoadJewels then
			JewelByName, JewelByEnchantId = self:LoadJewels()
			self.JewelByName, self.JewelByEnchantId = JewelByName, JewelByEnchantId
		end
		
		if self.LoadJewelFilters then
			Options.args.Settings.args.Jewels = self:LoadJewelFilters()
		end
		
		-- Load sets
		for _, SetClass in pairs(SetTypes) do
			SetClass.LoadSets()
		end
	end
	
	self:Refresh()
	
	self:RegisterEvent("ItemBonusLib_Update", "UpdatePlayerEquipment")
end

function ItemValue:OnDisable()
	TipHooker:Unhook(self.AddDataToTooltip, "item")
	if PlayerValueFrame then PlayerValueFrame:Hide() end
	if InspectValueFrame then InspectValueFrame:Hide() end
end

function ItemValue:ClearCache()
	for key in pairs(TooltipTextCache) do
		TooltipTextCache[key] = nil
	end
	for key in pairs(TooltipTextCache2) do
		TooltipTextCache2[key] = nil
	end
	
	self:UpdatePlayerValueFrame()
end


--[[
#############################
--    EQ Value Frames
--############################
--]]

function ItemValue:CreatePlayerValueFrame()
	if PlayerValueFrame then return end
	local frame = CreateFrame("Frame", "ItemValuePlayerValueFrame", CharacterModelFrame)
	frame:SetFrameStrata("HIGH")
	frame:SetWidth(200)
	frame:SetHeight(105)
	
	local text = frame:CreateFontString("ItemValuePlayerValueFrameFontstring", "HIGH")
	text:SetFontObject(GameFontHighlight)
	text:SetFont(text:GetFont(), 12)
	text:SetJustifyH("LEFT") 
	text:SetJustifyV("BOTTOM")
	
	text:SetPoint("TOPLEFT", frame, 5, 5)
	text:SetPoint("TOPRIGHT", frame, 5, 5)
	text:SetPoint("BOTTOMRIGHT", frame, 5, 5)
	text:SetPoint("BOTTOMLEFT", frame, 5, 5)
	
	text:SetText("")
	
	frame.text = text
	
	frame:SetBackdrop({
		bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
		tile = false,
		tileSize = 0,
		edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
		edgeSize = 10,
		insets = {left = 2, right = 2, top = 2, bottom = 2},
	})
	frame:SetBackdropColor(0, 0, 0, 0.3)
	
	frame:EnableMouse(true)
	frame:SetPoint("BOTTOMLEFT", 2, 24)
	
	PlayerValueFrame = frame
	self.PlayerValueFrame = frame
	
	--Dewdrop:Register(frame, "children", Options.args.Settings.args.EQValue.args.self)
	
	--self:UpdatePlayerEquipment()
	frame:Show()
end

function ItemValue:UpdatePlayerEquipment()
	self.EquipStats = self:GetUnitEquipment("player")
	
	self:ClearCache()
end

function ItemValue:UpdatePlayerValueFrame()
	if not (PlayerValueFrame) then return end
	local PlayerValueStringTable = tnew()
	
	for SetId, Set in pairs(SetById) do
		if self.db.profile.DisplayInTooltip[Set.Name] then
			local points = floor(Set:GetEquipValue(self.EquipStats))
			table.insert(PlayerValueStringTable, Set.Name .. ": " .. points)
		end
	end
	
	local PlayerValueString = table.concat(PlayerValueStringTable, "\n")
	tdel(PlayerValueStringTable)
	
	PlayerValueFrame.text:SetText(PlayerValueString)
end

function ItemValue:CreateInspectValueFrame()
	if InspectValueFrame then return end
	if not InspectPaperDollFrame then InspectFrame_LoadUI() end
	
	local frame = CreateFrame("Frame", "ItemValueInspectValueFrame", InspectPaperDollFrame)
	frame:SetFrameStrata("HIGH")
	frame:SetWidth(173)
	frame:SetHeight(105)
	
	local text = frame:CreateFontString("ItemValueInspectValueFrameFontstring", "HIGH")
	text:SetFontObject(GameFontHighlight)
	text:SetFont(text:GetFont(), 12)
	text:SetJustifyH("LEFT") 
	text:SetJustifyV("BOTTOM")
	
	text:SetPoint("TOPLEFT", frame, 5, 5)
	text:SetPoint("TOPRIGHT", frame, 5, 5)
	text:SetPoint("BOTTOMRIGHT", frame, 5, 5)
	text:SetPoint("BOTTOMLEFT", frame, 5, 5)
	text:SetText("")
	frame.text = text
	
	frame:SetBackdrop({
		bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
		tile = false,
		tileSize = 0,
		edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
		edgeSize = 10,
		insets = {left = 2, right = 2, top = 2, bottom = 2},
	})
	frame:SetBackdropColor(0, 0, 0, 0.3)
	frame:EnableMouse(true)
	
	frame:SetPoint("BOTTOMLEFT", 75, 140)
	
	InspectValueFrame = frame
	self.InspectValueFrame = frame
	
	Dewdrop:Register(frame, "children", Options.args.Settings.args.EQValue.args.inspect)
	
	frame:SetScript("OnUpdate", self.UpdateInspectValueFrame)
	frame:Show()
end

do -- ItemValue:UpdateInspectValueFrame(force)
	local OldInspectName
	
	function ItemValue:UpdateInspectValueFrame(force)
		if not (InspectValueFrame and InspectPaperDollFrame:IsVisible()) then return end
		local InspectName = UnitName(InspectFrame.unit)
		if (not force) and (InspectName == OldInspectName) or not InspectName then return end
		OldInspectName = InspectName
		
		local UnitEquip = ItemValue:GetInspectEquipment()
		
		ItemValue:Debug("UpdateInspectValueFrame")
		
		local InspectValueStringTable = tnew()
		
		for SetId, Set in pairs(SetById) do
			if ItemValue.db.profile.DisplayInTooltip[Set.Name] then
				local ownpoints = floor(Set:GetEquipValue(ItemValue.EquipStats))
				local unitpoints = floor(Set:GetEquipValue(UnitEquip))
				if unitpoints then
					table.insert(InspectValueStringTable, Set.Name .. ": " .. 
					 ItemValue:Compare(
					  ItemValue.db.profile.InspectComparisonMode,
					  ItemValue.db.profile.InspectShowPoints,
					  ItemValue.db.profile.InspectSwapColors,
					  ItemValue.db.profile.InspectSwapComparison,
					  unitpoints, ownpoints)
					)
				end
			end
		end
		
		local InspectValueString = table.concat(InspectValueStringTable, "\n")
		
		InspectValueFrame.text:SetText(InspectValueString)
		
		tdel(InspectValueStringTable)
	end
end


--[[
#############################
--    Set Handling
--############################
--]]

do -- ItemValue:RegisterSet(Set)
	local validname = function(v) return v and strlen(v) >= 3 end
	
	function ItemValue:RegisterSet(Set)
		local SetName = Set.Name
		if SetByName[SetName] then
			self:Print(string.format(L["Error: Set |cffffff78%s|r already exists!"], SetName))
			return
		end
		
		if not Set.Options.args.General then
			Set.Options.args.General = {
				type = "group",
				order = 1,
				name = "General",
				desc = "General Settings about this StatSet.",
				args = {
					Display = {
						type = "group",
						inline = true,
						order = 2,
						name = "Display",
						desc = "Display and Usage related settings about this StatSet.",
						args = {},
					}
				}
			}
		end
		
		if not Set.Options.args.General.args.Rename then
			Set.Options.args.General.args.Rename = tnewHash(
				"order",	1,
				"type",		"input",
				"name",		"Rename",
				"desc",		"Renames the Set.",
				"width",	"full",
				"set",		function(i, v) self:RenameSet(Set, v) end,
				"get",		function() return SetName end,
				"usage",	"<New Name>"--,
				--"validate",	validname
			)
		end
		if not Set.Options.args.General.args.Delete then
			Set.Options.args.General.args.Delete = tnewHash(
				"type",			"execute",
				"order",		3,
				"name",			"Delete",
				"desc",			"Delets this Set.",
				"width",		"full",
				"func",			function() self:UnregisterSet(Set) end
				--"passValue",	Set
			)
		end
		if not Set.Options.args.General.args.Send then
			Set.Options.args.General.args.Send = tnewHash(
				"type",			"input",
				"order",		2,
				"name",			L["Send To"],
				"desc",			L["Sends this Set to another ItemValue user."],
				"usage",		"<Player Name>",
				"set",			function(i, v) self:SendStatSet(Set, v) end
				--"validate",		validname,
				--"passValue",	Set
			)
		end
		if not Set.Options.args.General.args.Display.args.DisplayInTooltip then
			Set.Options.args.General.args.Display.args.DisplayInTooltip = tnewHash(
				"order",	2,
				"type",		"toggle",
				"name",		L["Display In Tooltip"],
				"desc",		L["Defines if the item value for this set should be displayed in item tooltips."],
				"set",		function(i,v)
								self.db.profile.DisplayInTooltip[SetName] = not not v
								self:ClearCache()
							end,
				"get",		function() return self.db.profile.DisplayInTooltip[SetName] end
			)
		end
		if not Set.Options.args.General.args.Display.args.Round then
		Set.Options.args.General.args.Display.args.Round = tnewHash(
				"type",		"range",
				"order",	4,
				"name",		L["Round"],
				"desc",		L["Round"],
				"min",		-5,
				"max",		5,
				"step",		1,
				"set",		function(i, v)
								self.db.profile.Round[SetName] = floor(v)
								self:ClearCache()
							end,
				"get",		function() return self.db.profile.Round[SetName] end
			)
		end
		
		Options.args.Sets.args[SetName:gsub(" ", "_")] = Set.Options
		
		SetByName[SetName] = Set
		if self.db.profile.SetIds and self.db.profile.SetIds[SetName] then
			local id = self.db.profile.SetIds[SetName]
			if not SetById[id] then
				SetById[id] = Set
				Set.Id = id
			end
		else
			table.insert(SetById, Set)
			Set.Id = #SetById
		end
		
		self:ClearCache()
		self:UpdateSorting()
		AceConfigDialog:ConfigTableChanged(nil, "ItemValue")
		self:TriggerEvent("ItemValue_SetsChanged")
	end
end

function ItemValue:SwapSets(Set1, Set2)
	SetById[Set1.Id] = nil
	SetById[Set2.Id] = nil
	
	SetById[Set2.Id] = SetByName[Set1.Name]
	SetById[Set1.Id] = SetByName[Set2.Name]
	
	Set1.Id, Set2.Id = Set2.Id, Set1.Id
	self.db.profile.SetIds[Set1.Name] = Set1.Id
	self.db.profile.SetIds[Set2.Name] = Set2.Id
	
	self:ClearCache()
	self:UpdateSorting()
	self:TriggerEvent("ItemValue_SetsChanged")
end

function ItemValue:UnregisterSet(Set)
	if not Set then return end
	
	local SetName = Set.Name
	if not SetByName[SetName] then
		return string.format(L["Error: Set \"%s\" doesn't exist!"], SetName)
	end
	
	Set:OnDelete()
	Options.args.Sets.args[SetName:gsub(" ", "_")] = nil
	
	SetByName[SetName] = nil
	
	tddel(SetById[i])
	
	for i=Set.Id,#SetById-1 do
		SetById[i] = SetById[i+1]
		SetById[i].Id = SetById[i].Id - 1
	end
	SetById[#SetById] = nil
	
	self.db.profile.DisplayInTooltip[SetName] = nil
	self.db.profile.Round[SetName] = nil
	if self.db.profile.SetIds and self.db.profile.SetIds[SetName] then self.db.profile.SetIds[SetName] = nil end
	
	self:ClearCache()
	self:UpdateSorting()
	self:TriggerEvent("ItemValue_SetsChanged")
	AceConfigDialog:ConfigTableChanged(nil, "ItemValue")
	if open then Waterfall:Open("ItemValue") end
end

function ItemValue:RenameSet(OldSet, NewName)
	-- Serializes old set, changes the name and deserializes to a new object
	assert(OldSet)
	assert(not SetByName[NewName])
	
	local t = OldSet:Serialize()
	t.Name = NewName
	local NewSet = SetTypes[t.Type]:Deserialize(t)
	assert(NewSet)
	
	self:RegisterSet(NewSet)
	self:UnregisterSet(OldSet)
end

function ItemValue:RegisterSetType(SetClassName, SetClass, SetDesc)
	assert(SetClass)
	assert(SetClassName)
	
	SetTypes[SetClassName] = SetClass
	Options.args.NewSets.args[SetClassName] = {
		type	= "group",
		name	= SetClassName,
		desc	= SetClassName,
		--inline	= true,
		args	= {
			["Desc" .. SetClassName] = {
				order	= 1,
				type	= "description",
				name	= SetDesc or "",
			},
			["New" .. SetClassName] = {
				order	= 2,
				type	= "input",
				name	= string.format("New %s", SetClassName),
				desc	= string.format("Creates a new %s.", SetClassName),
				usage	= "<Name>",
				set		= function(i, name)
					self:Print(string.format("Creating new |cffffff78%s|r: |cffffff78%s|r...", SetClassName, name))
					self:RegisterSet(SetClass:new(name))
					self:ClearCache()
				end,
				validate = function(i, v)
					return v and type(v) == "string" and strlen(v) > 0
				end,
			},
		},
	}
	
	if db then
		SetClass.LoadSets()
	end
end

do -- ItemValue:UpdateSorting()

	local hiddenf = function() return not self.db.profile.SetIds end
	
	local function optGetter(info)
		local key = info[#info]
		return self.db.profile[key]
	end
		
	local function MoveSet(info, value)
		local key = info[#info]
		
		local Set = SetById[tonumber(key)]
		local TargetSet
		if IsShiftKeyDown() then
			TargetSet = SetById[Set.Id + 1]
		else
			TargetSet = SetById[Set.Id - 1]
		end
		
		if not TargetSet or TargetSet == Set then return end
		
		ItemValue:SwapSets(Set, TargetSet)
	end
	
	function ItemValue:UpdateSorting()
		if not self.db.profile.CustomSetOrder then return end
		local n = 100
		
		-- Remove old sets
		for SetIdStr, SetOptions in pairs(Options.args.Settings.args.Sorting.args) do
			if SetOptions.order >= n and not SetById[tonumber(SetIdStr)] then
				Options.args.Settings.args.Sorting.args[SetIdStr] = nil
			end
		end
		
		for SetId, Set in pairs(SetById) do
			local SetIdStr = tostring(SetId)
			if Options.args.Settings.args.Sorting.args[SetIdStr] then
				-- Update order
				Options.args.Settings.args.Sorting.args[SetIdStr].order = n + SetId
				Options.args.Settings.args.Sorting.args[SetIdStr].name = SetId .. ". " .. Set.Name
			else
				-- Add new sets
				Options.args.Settings.args.Sorting.args[SetIdStr] = tnewHash(
					"order",		n + Set.Id,
					"type",			"execute",
					"width",		"full",
					"name",			SetId .. ". " .. Set.Name,
					"desc",			L["Moves the Set one place up (or down if <SHIFT> key is held)"],
					"func",			MoveSet,
					"hidden",		hiddenf
				)
			end
		end
	end
end

-- Set Sharing
function ItemValue:SendStatSet(Set, PlayerName)
	self:Print(string.format(L["Sending Set |cffffff78%s|r to player |cffffff78%s|r..."], Set.Name, PlayerName))
	self:SendCommMessage("WHISPER", PlayerName, Set:Serialize())
end

function ItemValue:OnCommReceive(prefix, sender, distribution, t)
	assert(t and type(t) == "table", "Set invalid.")
	assert(t.Name and type(t.Name) == "string", "Name invalid.")
	assert(t.Type and type(t.Type) == "string", "Type invalid.")
	assert(SetTypes[t.Type], t.Type .. " Type unknown.")
	
	local SetType = t.Type
	self:Print(string.format(L["Received |cffffff78%s|r named |cffffff78%s|r from player |cffffff78%s|r."], SetType, t.Name, sender))
	StaticPopupDialogs["ITEMVALUE_NEW_SET_RECEIVED"].text = string.format(L["Received |cffffff78%s|r named |cffffff78%s|r from player |cffffff78%s|r."], SetType, t.Name, sender)
	t.Name = t.Name .. " - " .. sender
	
	local Set = SetTypes[SetType]:Deserialize(t)
	if not Set then self:Print(string.format(L["Error: Couldn't Deserialize |cffffff78%s|r!"], SetType)) return end
	
	StaticPopupDialogs["ITEMVALUE_NEW_SET_RECEIVED"].OnAccept = function()
		self:RegisterSet(Set)
	end
		
	StaticPopup_Show("ITEMVALUE_NEW_SET_RECEIVED")
end

StaticPopupDialogs["ITEMVALUE_NEW_SET_RECEIVED"] = {
	text = "",
	button1 = L["Accept"],
	button2 = L["Decline"],
	timeout = 0,
	hideOnEscape = 1,
	OnDecline = function()
		StaticPopupDialogs["ITEMVALUE_NEW_SET_RECEIVED"].OnAccept = nil
	end
}


--[[
#############################
--    Utility functions
--############################
--]]

local function round(num, idp)
	local mult = 10^(idp or 0)
	return math.floor(num * mult + 0.5) / mult
end

do -- ItemValue:Compare(CompMode, ShowBase, ReverseColor, SwapComparison, ...)
	local cBetter = "|cFF00FF00"
	local cWorse  = "|cFFFF0000"
	local cEqual = "|cFFFFFF00"
	
	local function Color(reverse, b, v)
		if reverse then
			if (b > v) then
				return cWorse
			elseif (b < v) then
				return cBetter
			else
				return cEqual
			end
		else
			if (b > v) then
				return cBetter
			elseif (b < v) then
				return cWorse
			else
				return cEqual
			end
		end
	end
	
	local function FormatNumber(n)
		if n > 0 then
			return "+" .. round(n, 1)
		else
			return round(n, 1) .. ""
		end
	end
	
	local CompModes = {
		--["Don't Compare"] = nil,
		[2] --[[Absolute]] = function(b, v)
			return v
		end,
		[3] --[[Delta]] = function(b, v)
			return FormatNumber(b-v)
		end,
		[4] --[[Percent]] = function(b, v)
			return (b > 0 and (FormatNumber(v/b*100-100)).."%") or "+"
		end
	}
	
	function ItemValue:Compare(CompMode, ShowBase, ReverseColor, SwapComparison, ...)
		local nargs = select("#", ...)
		local b = select(1, ...)
		
		local lines = tnew()
		
		if ShowBase then
			--if nargs == 2 then table.insert(line, GetColor(select(2, ...), b)) end
			table.insert(lines, b)
			--if nargs == 2 then table.insert(line, "|r") end
		end
		
		local i = 2
		local v = select(2, ...)
		while v do
			if CompModes[CompMode] then
				table.insert(lines, Color(ReverseColor, b, v))
				if ShowBase then
					table.insert(lines, " (")
				elseif i > 2 then
					table.insert(lines, " ")
				end
				if SwapComparison then
					table.insert(lines, CompModes[CompMode](v, b))
				else
					table.insert(lines, CompModes[CompMode](b, v))
				end
				if ShowBase then
					table.insert(lines, ")")
				end
				table.insert(lines, "|r")
			end
			i = i + 1
			v = select(i, ...)
		end
		
		local text = table.concat(lines)
		tdel(lines)
		
		return text
	end
end


--[[
#############################
--    Jewel functions
--############################
--]]

do -- ItemValue:PrintBestJewels(SetName, ItemLink)
	local color = {"|cFFFFFFFF[", "|cFF1EFF00[", "|cFF0070DD[", "|cFFA335EE[", "|cFFFF8000["}
	function ItemValue:PrintBestJewels(SetName, ItemLink)
		local Set = SetByName[SetName]
		if not (ItemLink and Set) then return end
		
		local ItemString = self:ItemLinkToItemString(ItemLink)
		local _, iId, eId = strsplit(":", ItemString)
		
		if self.db.profile.DisplayInTooltip[Set.Name] then
			local ItemLink = self:GetItemLink(iId, eId)
			self:Print(string.format(L["Best jewel combination for |cffffff78%s|r (|cffffff78%s|r Points), Set: |cffffff78%s|r is:"], ItemLink, round(Set:GetItemValue(iId, eId), self.db.profile.Round[Set.Name]), SetName))
			
			local BestJewelCombination = self:CalcBestJewels(Set, ItemString)
			local Jewel1, Jewel2, Jewel3 = BestJewelCombination.Jewel1, BestJewelCombination.Jewel2, BestJewelCombination.Jewel3
			
			local _, ItemId, EnchantId = strsplit(":", ItemString)
			local ItemLink = self:GetItemLink(ItemId, EnchantId, Jewel1.EnchantId, Jewel2.EnchantId, Jewel3.EnchantId)
			
			self:Print(string.format(L[" %s %s Points:"], ItemLink, round(BestJewelCombination.Points, self.db.profile.Round[Set.Name])))
			
			if Jewel1 and Jewel1 ~= self.EmptySocket then
				local _, ItemLinkJ1 = GetItemInfo(Jewel1.ItemId)
				self:Print(" 1. ", (ItemLinkJ1 or color[Jewel1.Quality] .. Jewel1.Name .. "]|r (*)"))
			end
			if Jewel2 and Jewel2 ~= self.EmptySocket then
				local _, ItemLinkJ2 = GetItemInfo(Jewel2.ItemId)
				self:Print(" 2. ", (ItemLinkJ2 or color[Jewel2.Quality] .. Jewel2.Name .. "]|r (*)"))
			end
			if Jewel3 and Jewel3 ~= self.EmptySocket then
				local _, ItemLinkJ3 = GetItemInfo(Jewel3.ItemId)
				self:Print(" 3. ", (ItemLinkJ3 or color[Jewel3.Quality] .. Jewel3.Name .. "]|r (*)"))
			end
		end
	end
end

do -- ItemValue:CalcBestJewels(Set, ItemString)
	local items = {2589, 2592, 4306, 4338, 14047, 21877}
	local item
	
	local function CmpPoints(v1, v2)
		return v1.Points > v2.Points
	end
	
	local BestJewelCombination = {}
	
	function ItemValue:CalcBestJewels(Set, ItemString)
		if not ItemString then return end
		local _, iId, eId, jId1, jId2, jId3, jId4, sId, uId = strsplit(":", ItemString)
		if not iId then iId = ItemString end
		
		local Sockets, RedSockets, YellowSockets, BlueSockets, MetaSocketPos = self:GetSocketInfo(iId)
		
		if not Sockets then return end
		
		-- 3 best jewels for every type are calculated
		local BestJewelRed, BestJewelYellow, BestJewelBlue, BestJewelMeta = tnew(), tnew(), tnew(), tnew()
		
		for JewelName, Jewel in pairs(JewelByName) do
			if self.db.profile.EnabledJewels[JewelName] then
				-- Cant scan jewel's ItemId directly, because its not sure if its in the clients item cache
				-- Workaround: Scan an item from a predefined list
				if not item then
					for _, ItemId in pairs(items) do
						if GetItemInfo(ItemId) or GetItemInfo(ItemId) then -- first GetItemInfo returns sometimes nil 
							item = ItemId
							items = nil
							break
						end
					end
					
					if not item then
						self:Print(L["Warning: No item found to scan jewels with! Please click an itemlink of a cloth type."])
						
						-- Todo: Ask user to retrieve item information
						--[[
						GameTooltip:SetHyperlink("item:"..itemId)
						GameTooltip:Show()
						GameTooltip:Hide()
						--]]
						
						BestJewelCombination.Points = 0
						BestJewelCombination.Jewel1 = self.EmptySocket
						BestJewelCombination.Jewel2 = self.EmptySocket
						BestJewelCombination.Jewel3 = self.EmptySocket
						return BestJewelCombination
					end
				end
				
				-- Used later to sort
				local JewelPoints = tnew()
				JewelPoints.Name = JewelName
				JewelPoints.Points = Set:GetItemValue(self:GetItemString(item, 0, Jewel.EnchantId, 0, 0, 0, 0, 0))
				
				if Jewel.IsRed then
					table.insert(BestJewelRed, JewelPoints)
				end
				if Jewel.IsYellow then
					table.insert(BestJewelYellow, JewelPoints)
				end
				if Jewel.IsBlue then
					table.insert(BestJewelBlue, JewelPoints)
				end
				if Jewel.IsMeta and MetaSocketPos then
					table.insert(BestJewelMeta, JewelPoints)
				end
			end
		end
		
		assert(#BestJewelRed >= 3, "#BestJewelRed is " .. #BestJewelRed)
		assert(#BestJewelYellow >= 3, "#BestJewelYellow is " .. #BestJewelYellow)
		assert(#BestJewelBlue >= 3, "#BestJewelBlue is " .. #BestJewelBlue)
		assert(#BestJewelMeta >= 1, "#BestJewelMeta is " .. #BestJewelMeta)
		
		table.sort(BestJewelRed, CmpPoints)
		table.sort(BestJewelYellow, CmpPoints)
		table.sort(BestJewelBlue, CmpPoints)
		table.sort(BestJewelMeta, CmpPoints)
		
		--self:PrintLiteral(BestJewelRed)
		--self:PrintLiteral(BestJewelYellow)
		--self:PrintLiteral(BestJewelBlue)
		
		-- The up to 9 Gems that are most likely to build the best combination
		local JewelList = tnew()
		
		if RedSockets or BestJewelRed[1].Points > BestJewelYellow[1].Points or BestJewelRed[1].Points > BestJewelBlue[1].Points then
			table.insert(JewelList, JewelByName[BestJewelRed[1].Name])
			if JewelByName[BestJewelRed[1].Name].IsUnique then
				table.insert(JewelList, JewelByName[BestJewelRed[2].Name])
				if JewelByName[BestJewelRed[2].Name].IsUnique then
					table.insert(JewelList, JewelByName[BestJewelRed[3].Name])
				end
			end
		end
		
		if YellowSockets or BestJewelYellow[1].Points > BestJewelRed[1].Points or BestJewelYellow[1].Points > BestJewelBlue[1].Points then
			table.insert(JewelList, JewelByName[BestJewelYellow[1].Name])
			if JewelByName[BestJewelYellow[1].Name].IsUnique then
				table.insert(JewelList, JewelByName[BestJewelYellow[2].Name])
				if JewelByName[BestJewelYellow[2].Name].IsUnique then
					table.insert(JewelList, JewelByName[BestJewelYellow[3].Name])
				end
			end
		end
		
		if BlueSockets or BestJewelBlue[1].Points > BestJewelYellow[1].Points or BestJewelBlue[1].Points > BestJewelRed[1].Points then
			table.insert(JewelList, JewelByName[BestJewelBlue[1].Name])
			if JewelByName[BestJewelBlue[1].Name].IsUnique then
				table.insert(JewelList, JewelByName[BestJewelBlue[2].Name])
				if JewelByName[BestJewelBlue[2].Name].IsUnique then
					table.insert(JewelList, JewelByName[BestJewelBlue[3].Name])
				end
			end
		end
		
		if not DummyJewelTable then DummyJewelTable = {self.EmptySocket} end
		
		local ScanTable1, ScanTable2, ScanTable3
		
		if MetaSocketPos > 0 then
			if not MetaGemTable then MetaGemTable = {} end
			MetaGemTable[1] = JewelByName[BestJewelMeta[1].Name]
		end
		
		ScanTable3 = (MetaSocketPos == 3 and MetaGemTable) or (#Sockets >= 3 and JewelList) or DummyJewelTable
		ScanTable2 = (MetaSocketPos == 2 and MetaGemTable) or (#Sockets >= 2 and JewelList) or DummyJewelTable
		ScanTable1 = (MetaSocketPos == 1 and MetaGemTable) or (JewelList)
		
		--self:PrintLiteral(ScanTable3)
		--self:PrintLiteral(ScanTable2)
		--self:PrintLiteral(ScanTable1)
		
		--self:PrintLiteral(DummyJewelTable)
		
		BestJewelCombination.Points = -1
		
		--self:Print("Test1!")
		for _, Jewel1 in pairs(ScanTable1) do
			--self:Print("Test2!")
			for _, Jewel2 in pairs(ScanTable2) do
				--self:Print("Test3!")
				for _, Jewel3 in pairs(ScanTable3) do
					--self:Print("Test4!")
					--self:Print(Jewel1.Name, Jewel2.Name, Jewel3.Name)
					if Jewel1.IsUnique and (Jewel1==Jewel2 or Jewel1==Jewel3)
					 or Jewel2.IsUnique and (Jewel2==Jewel3) then break end
					local ItemString = self:GetItemString(iId, eId, Jewel1.EnchantId, Jewel2.EnchantId, Jewel3.EnchantId, jId4, sId, uId)
					local Points = Set:GetItemValue(ItemString)
					
					--self:Print(ItemString, Points)
					
					if Points > BestJewelCombination.Points then
						BestJewelCombination.Points = Points
						BestJewelCombination.Jewel1 = Jewel1
						BestJewelCombination.Jewel2 = Jewel2
						BestJewelCombination.Jewel3 = Jewel3
					end
				end
			end
		end
		
		tddel(BestJewelRed)
		tddel(BestJewelYellow)
		tddel(BestJewelBlue)
		tddel(BestJewelMeta)
		tdel(JewelList)
		
		return BestJewelCombination
	end
end

function ItemValue:GetSocketInfo(ItemId)
	Gratuity:SetHyperlink("item:" .. ItemId .. ":0:0:0:0:0:0:0")
	
	local NumLines = Gratuity:NumLines()
	
	if NumLines < 5 then
		return
	end
	
	local RedSockets, YellowSockets, BlueSockets, MetaSocketPos = 0, 0, 0, 0
	
	local Sockets = tnew()
	
	for curLine = 4, NumLines do
		local line = Gratuity:GetLine(curLine)
		if line == L["Red Socket"] then
			table.insert(Sockets, "Red")
			RedSockets = RedSockets + 1
		elseif line == L["Yellow Socket"] then
			table.insert(Sockets, "Yellow")
			YellowSockets = YellowSockets + 1
		elseif line == L["Blue Socket"] then
			table.insert(Sockets, "Blue")
			BlueSockets = BlueSockets + 1
		elseif line == L["Meta Socket"] then
			table.insert(Sockets, "Meta")
			MetaSocketPos = #Sockets
		end
		if #Sockets >= 3 then break end
	end
	
	if #Sockets == 0 then return end
	return Sockets, RedSockets, YellowSockets, BlueSockets, MetaSocketPos
end

function ItemValue:GetSocketCount(itemId)
	if SocketAmountCache[itemId] then
		return SocketAmountCache[itemId]
	else
		Gratuity:SetHyperlink("item:" .. itemId .. ":0:0:0:0:0:0:0")
		
		local NumLines = Gratuity:NumLines()
		
		if NumLines < 5 then
			return 0
		end
		
		local sockets = 0
		for curLine = 4, NumLines do
			local line = Gratuity:GetLine(curLine)
			if line == L["Red Socket"] or line == L["Yellow Socket"]
			 or line == L["Blue Socket"] or line == L["Meta Socket"] then
				sockets = sockets + 1
			end
			if sockets >= 3 then break end
		end
		
		SocketAmountCache[itemId] = sockets
		return sockets
	end
end

function ItemValue:RegisterNewJewel(JewelId)
	local ItemLink = self:GetItemLink(6948, 0, JewelId)
	local GemName, GemItemLink = GetItemGem(ItemLink, 1)
	if GemName then
		local GemItemString = self:ItemLinkToItemString(GemItemLink)
		local _, GemItemId = strsplit(":", GemItemString)
		GemItemId = tonumber(GemItemId) 
		local _, _, GemQuality = GetItemInfo(GemItemId)
		
		--g:SetHyperlink("item:21877:0:" .. cur .. ":0:0:0:0:0")
		--local effect = g:GetLine(2)
		
		Gratuity:SetHyperlink("item:"..GemItemId..":0:0:0:0:0:0:0")
		local numlines = Gratuity:NumLines()
		local ColorString = strlower(Gratuity:GetLine(numlines))
		local red, yellow, blue, meta
		if strfind(ColorString, "red") then red = true end
		if strfind(ColorString, "yellow") then yellow = true end
		if strfind(ColorString, "blue") then blue = true end
		if strfind(ColorString, "meta") then meta = true end 
		
		local GemIsUnique = not not Gratuity:Find("Unique")
		
		local Jewel = {Name = GemName, EnchantId = tonumber(JewelId), ItemId = GemItemId, Quality = GemQuality, IsUnique = GemIsUnique, IsRed = (red == true), IsYellow = (yellow == true), IsBlue = (blue == true), IsMeta = (meta == true),}
		
		self:Print("New Jewel:", GemItemLink)
		
		table.insert(self.db.profile.Jewels, Jewel)
		
		JewelByEnchantId[Jewel.EnchantId] = Jewel
		JewelByName[Jewel.Name] = Jewel
		
		
		Options.args.Settings.args.Jewels.args[Jewel.Name] = {
			order	= 101,
			type	= "toggle",
			name	= Jewel.Name,
			desc	= Jewel.Name,
			get		= function() return self.db.profile.EnabledJewels[Jewel.Name] end,
			set		= function(v) self.db.profile.EnabledJewels[Jewel.Name] = v self:ClearCache() end,
		}
	end
end

function ItemValue:GetDefaultJewel(color)
	if self.db.profile.MultipleDefaultJewels then
		return JewelByName[self.db.profile["DefaultJewel_"..string.tolower(color)]] or self.EmptySocket
	else
		return JewelByName[self.db.profile.DefaultJewel] or self.EmptySocket
	end
end


--[[
#############################
--    Tooltip Handling
--############################
--]]

function ItemValue.AddDataToTooltip(Tooltip, _, argItemLink, argItemString)
	local ItemString = argItemString or ItemValue:ItemLinkToItemString(argItemLink) or ItemValue:ItemLinkToItemString(Tooltip:GetItemLink())
	
	local text = TooltipTextCache[ItemString]
	
	if ItemValue.db.profile.TooltipDoubleSided then
		local text2 = TooltipTextCache2[ItemString]
		if not (text and text2) then
			text, text2 = ItemValue:GetTooltipText(ItemString)
			TooltipTextCache[ItemString] = text
			TooltipTextCache2[ItemString] = text2
		end
		Tooltip:AddDoubleLine(text, text2)
	else
		if not text then
			text = ItemValue:GetTooltipText(ItemString)
			TooltipTextCache[ItemString] = text
		end
		Tooltip:AddLine(text)
	end
	
	Tooltip:Show()
end

do -- ItemValue:GetTooltipText(ItemString)
	local BaseItem = {}
	local CompareItem1 = {}
	local CompareItem2 = {}
	
	function ItemValue:GetTooltipText(ItemString)
		-- Check if enchant link
		if strfind(ItemString, ":") ~= 5 then return "", "" end
		BaseItem.ItemString = ItemString
		
		if not (IsEquippableItem(BaseItem.ItemString) or
		 select(6, GetItemInfo(BaseItem.ItemString)) == L["Gem"]) then -- ???
			return "", ""
		end
		
		local ItemEquipLoc = select(9, GetItemInfo(ItemString))
		local OwnItem = IsEquippedItem(ItemString)
		
		if ItemEquipLoc and not(self.db.profile.TooltipComparisonMode == 1 --[["Don't Compare"]] or OwnItem) then
			local slot1, slot2 = self:GetInventorySlot(BaseItem.ItemString)
			if slot1 then
				CompareItem1.ItemString = self:ItemLinkToItemString(GetInventoryItemLink("player", slot1))
				CompareItem1.Enabled = (type(CompareItem1.ItemString) == "string") and string.len(CompareItem1.ItemString) > 10
			else
				CompareItem1.Enabled = false
			end
			if slot2 then
				CompareItem2.ItemString = self:ItemLinkToItemString(GetInventoryItemLink("player", slot2))
				CompareItem2.Enabled = (type(CompareItem2.ItemString) == "string") and string.len(CompareItem2.ItemString) > 10
			else
				CompareItem2.Enabled = false
			end
		else
			CompareItem1.Enabled = false
			CompareItem2.Enabled = false
		end
		
		self:ApplyItemModifications(BaseItem, CompareItem1.Enabled and CompareItem1, CompareItem2.Enabled and CompareItem2)
		
		local text, text2
		
		for SetId, Set in pairs(SetById) do
			local SetPointsB, SetPoints1, SetPoints2
			
			if self.db.profile.DisplayInTooltip[Set.Name] then
				if self.db.profile.JewelHandling == 5 --[[Best]] and self:GetSocketCount(BaseItem.iId) > 0 then
					local BestJewelCombination = self:CalcBestJewels(Set, BaseItem.iId)
					local Jewel1, Jewel2, Jewel3 = BestJewelCombination.Jewel1, BestJewelCombination.Jewel2, BestJewelCombination.Jewel3
					BaseItem.ItemString = self:GetItemString(BaseItem.iId, BaseItem.eId, Jewel1.EnchantId, Jewel2.EnchantId, Jewel3.EnchantId, BaseItem.jId4, BaseItem.sId, BaseItem.uId)
				end
				SetPointsB = round(Set:GetItemValue(BaseItem.ItemString), self.db.profile.Round[Set.Name])
				if CompareItem1.Enabled then
					if self.db.profile.JewelHandling == 5 --[[Best]] and self:GetSocketCount(CompareItem1.iId) > 0 then
						local BestJewelCombination = self:CalcBestJewels(Set, CompareItem1.iId)
						local Jewel1, Jewel2, Jewel3 = BestJewelCombination.Jewel1, BestJewelCombination.Jewel2, BestJewelCombination.Jewel3
						CompareItem1.ItemString = self:GetItemString(CompareItem1.iId, CompareItem1.eId, Jewel1.EnchantId, Jewel2.EnchantId, Jewel3.EnchantId, CompareItem1.jId4, CompareItem1.sId, CompareItem1.uId)
					end
					SetPoints1 = round(Set:GetItemValue(CompareItem1.ItemString), self.db.profile.Round[Set.Name])
				end
				if CompareItem2.Enabled then
					if self.db.profile.JewelHandling == 5 --[[Best]] and self:GetSocketCount(CompareItem2.iId) > 0 then
						local BestJewelCombination = self:CalcBestJewels(Set, CompareItem2.iId)
						local Jewel1, Jewel2, Jewel3 = BestJewelCombination.Jewel1, BestJewelCombination.Jewel2, BestJewelCombination.Jewel3
						CompareItem2.ItemString = self:GetItemString(CompareItem2.iId, CompareItem2.eId, Jewel1.EnchantId, Jewel2.EnchantId, Jewel3.EnchantId, CompareItem2.jId4, CompareItem2.sId, CompareItem2.uId)
					end
					SetPoints2 = round(Set:GetItemValue(CompareItem2.ItemString), self.db.profile.Round[Set.Name])
				end
				
				if (SetPointsB > 0)
				 or (SetPoints1 and SetPointsB ~= SetPoints1)
				 or (SetPoints2 and SetPointsB ~= SetPoints2) then
					
					local line = (SetPoints1 or SetPoints2) and self:Compare(self.db.profile.TooltipComparisonMode, 
					self.db.profile.TooltipShowPoints, self.db.profile.TooltipSwapColor, self.db.profile.TooltipSwapComparison, 
					SetPointsB, SetPoints1, SetPoints2) or SetPointsB
					
					if self.db.profile.TooltipDoubleSided then
						text = ((text and (text ~= "") and (text .. "\n")) or "") .. Set.Name
						text2 = ((text2 and (text2 ~= "") and (text2 .. "\n")) or "") .. line
					else
						text = ((text and (text ~= "") and (text .. "\n")) or "") .. Set.Name .. ": " .. line
					end
				end
			end
		end
		
		return text or "", text2 or ""
	end
end

do -- ItemValue:ApplyItemModifications(BaseItem, CompareItem1, CompareItem2)
	local function FillItemInfos(Item)
		if not Item then return end
		local _, iId, eId, jId1, jId2, jId3, jId4, sId, uId = strsplit(":", Item.ItemString)
		Item.iId = iId or "0"
		Item.eId = eId or "0"
		Item.jId1 = jId1 or "0"
		Item.jId2 = jId2 or "0"
		Item.jId3 = jId3 or "0"
		Item.jId4 = jId4 or "0"
		Item.sId = sId or "0"
		Item.uId = uId or "0"
		
		-- Unknown Jewel?
		if JewelByEnchantId and type(JewelByEnchantId) == "table" then
			if not JewelByEnchantId[tonumber(jId1)] then ItemValue:RegisterNewJewel(jId1) end
			if not JewelByEnchantId[tonumber(jId2)] then ItemValue:RegisterNewJewel(jId2) end
			if not JewelByEnchantId[tonumber(jId3)] then ItemValue:RegisterNewJewel(jId3) end
		end
	end
	
	function ItemValue:ApplyItemModifications(BaseItem, CompareItem1, CompareItem2)
		FillItemInfos(BaseItem)
		FillItemInfos(CompareItem1)
		FillItemInfos(CompareItem2)
		
		-- Apply Enchant Settings
		if self.db.profile.EnchantHandling == 3 --[["Always own]] then
			if CompareItem1 or CompareItem2 then
				BaseItem.eId = (CompareItem1 and CompareItem1.eId) or CompareItem2.eId
			end
		elseif self.db.profile.EnchantHandling == 4 --[[Fill with own]] then
			if BaseItem.eId == "0" and (CompareItem1 or CompareItem2) then
				BaseItem.eId = (CompareItem1 and CompareItem1.eId) or (CompareItem2 and CompareItem2.eId) or "0"
			end
		elseif self.db.profile.EnchantHandling == 1 --[[Ignore]] then
			BaseItem.eId = "0"
			if CompareItem1 then CompareItem1.eId = "0" end
			if CompareItem2 then CompareItem2.eId = "0" end
		end
		
		-- Apply Jewel Settings
		local SocketsB, Sockets1, Sockets2 = self:GetSocketCount(BaseItem.iId)
		if CompareItem1 then Sockets1 = self:GetSocketCount(CompareItem1.iId) end
		if CompareItem2 then Sockets2 = self:GetSocketCount(CompareItem2.iId) end
		
		if SocketsB > 0 or (Sockets1 and Sockets1 > 0) or (Sockets2 and Sockets2 > 0) then
			local tSocketsB, tSockets1, tSockets2
			
			local DetailedSocketInfo = self.db.profile.MultipleDefaultJewels and 
			 (self.db.profile.JewelHandling == 4 --[[Fill with default]] or
			 self.db.profile.JewelHandling == 3 --[[Always default]])
			 
			if DetailedSocketInfo then
				tSocketsB = self:GetSocketInfo(BaseItem.iId)
				if CompareItem1 then tSockets1 = self:GetSocketInfo(CompareItem1.iId) end
				if CompareItem2 then tSockets2 = self:GetSocketInfo(CompareItem2.iId) end
			end
			
			if self.db.profile.JewelHandling == 4 --[[Fill with default]] then
				BaseItem.jId1 = (SocketsB >= 1 and BaseItem.jId1 == "0" and self:GetDefaultJewel(tSocketsB and tSocketsB[1]).EnchantId) or BaseItem.jId1
				BaseItem.jId2 = (SocketsB >= 2 and BaseItem.jId2 == "0" and self:GetDefaultJewel(tSocketsB and tSocketsB[2]).EnchantId) or BaseItem.jId2
				BaseItem.jId3 = (SocketsB >= 3 and BaseItem.jId3 == "0" and self:GetDefaultJewel(tSocketsB and tSocketsB[3]).EnchantId) or BaseItem.jId3
				if CompareItem1 then
					CompareItem1.jId1 = (Sockets1 >= 1 and CompareItem1.jId1 == "0" and self:GetDefaultJewel(tSockets1 and tSockets1[1]).EnchantId) or CompareItem1.jId1
					CompareItem1.jId2 = (Sockets1 >= 2 and CompareItem1.jId2 == "0" and self:GetDefaultJewel(tSockets1 and tSockets1[2]).EnchantId) or CompareItem1.jId2
					CompareItem1.jId3 = (Sockets1 >= 3 and CompareItem1.jId3 == "0" and self:GetDefaultJewel(tSockets1 and tSockets1[3]).EnchantId) or CompareItem1.jId3
				end
				if CompareItem2 then
					CompareItem2.jId1 = (Sockets2 >= 1 and CompareItem2.jId1 == "0" and self:GetDefaultJewel(tSockets2 and tSockets2[1]).EnchantId) or CompareItem2.jId1
					CompareItem2.jId2 = (Sockets2 >= 2 and CompareItem2.jId2 == "0" and self:GetDefaultJewel(tSockets2 and tSockets2[2]).EnchantId) or CompareItem2.jId2
					CompareItem2.jId3 = (Sockets2 >= 3 and CompareItem2.jId3 == "0" and self:GetDefaultJewel(tSockets2 and tSockets2[3]).EnchantId) or CompareItem2.jId3
				end
			elseif self.db.profile.JewelHandling == 3--[[Always default]] then
				BaseItem.jId1 = (SocketsB >= 1 and self:GetDefaultJewel(tSocketsB and tSocketsB[1]).EnchantId) or BaseItem.jId1
				BaseItem.jId2 = (SocketsB >= 2 and self:GetDefaultJewel(tSocketsB and tSocketsB[2]).EnchantId) or BaseItem.jId2
				BaseItem.jId3 = (SocketsB >= 3 and self:GetDefaultJewel(tSocketsB and tSocketsB[3]).EnchantId) or BaseItem.jId3
				if CompareItem1 then
					CompareItem1.jId1 = (Sockets1 >= 1 and self:GetDefaultJewel(tSockets1 and tSockets1[1]).EnchantId) or CompareItem1.jId1
					CompareItem1.jId2 = (Sockets1 >= 2 and self:GetDefaultJewel(tSockets1 and tSockets1[2]).EnchantId) or CompareItem1.jId2
					CompareItem1.jId3 = (Sockets1 >= 3 and self:GetDefaultJewel(tSockets1 and tSockets1[3]).EnchantId) or CompareItem1.jId3
				end
				if CompareItem2 then
					CompareItem2.jId1 = (Sockets2 >= 1 and self:GetDefaultJewel(tSockets2 and tSockets2[1]).EnchantId) or CompareItem2.jId1
					CompareItem2.jId2 = (Sockets2 >= 2 and self:GetDefaultJewel(tSockets2 and tSockets2[2]).EnchantId) or CompareItem2.jId2
					CompareItem2.jId3 = (Sockets2 >= 3 and self:GetDefaultJewel(tSockets2 and tSockets2[3]).EnchantId) or CompareItem2.jId3
				end
			elseif self.db.profile.JewelHandling == 1 --[[Ignore]] then
				BaseItem.jId1, BaseItem.jId2, BaseItem.jId3 = "0", "0", "0"
				if CompareItem1 then
					CompareItem1.jId1, CompareItem1.jId2, CompareItem1.jId3 = "0", "0", "0"
				end
				if CompareItem2 then
					CompareItem2.jId1, CompareItem2.jId2, CompareItem2.jId3 = "0", "0", "0"
				end
			end
			
			if DetailedSocketInfo then
				if tSocketsB then tdel(tSocketsB) end
				if tSockets1 then tdel(tSockets1) end
				if tSockets2 then tdel(tSockets2) end
			end
		end
		
		BaseItem.ItemString = self:GetItemString(BaseItem.iId, BaseItem.eId, BaseItem.jId1, BaseItem.jId2, BaseItem.jId3, BaseItem.jId4, BaseItem.sId, BaseItem.uId)
		self:Debug(BaseItem.ItemString)
		if CompareItem1 then
			CompareItem1.ItemString = self:GetItemString(CompareItem1.iId, CompareItem1.eId, CompareItem1.jId1, CompareItem1.jId2, CompareItem1.jId3, CompareItem1.jId4, CompareItem1.sId, CompareItem1.uId)
			self:Debug(CompareItem1.ItemString)
		end
		if CompareItem2 then
			CompareItem2.ItemString = self:GetItemString(CompareItem2.iId, CompareItem2.eId, CompareItem2.jId1, CompareItem2.jId2, CompareItem2.jId3, CompareItem2.jId4, CompareItem2.sId, CompareItem2.uId)
			self:Debug(CompareItem2.ItemString)
		end
	end
end


--[[
#############################
--    Item functions
--############################
--]]

do -- ItemValue:GetInventorySlot(ItemLink)
	local slot1 = {
		INVTYPE_AMMO			= 0,
		INVTYPE_GUNPROJECTILE	= 0,
		INVTYPE_BOWPROJECTILE	= 0,
		INVTYPE_HEAD			= 1,
		INVTYPE_NECK			= 2,
		INVTYPE_SHOULDER		= 3,
		INVTYPE_BODY			= 4,
		INVTYPE_CHEST			= 5,
		INVTYPE_ROBE			= 5,
		INVTYPE_WAIST			= 6,
		INVTYPE_LEGS			= 7,
		INVTYPE_FEET			= 8,
		INVTYPE_WRIST			= 9,
		INVTYPE_HAND			= 10,
		INVTYPE_FINGER			= 11,
		INVTYPE_TRINKET			= 13,
		INVTYPE_CLOAK			= 15,
		INVTYPE_WEAPON			= 16,
		INVTYPE_2HWEAPON		= 16,
		INVTYPE_WEAPONMAINHAND	= 16,
		INVTYPE_WEAPONOFFHAND	= 17,
		INVTYPE_SHIELD			= 17,
		INVTYPE_HOLDABLE		= 17,
		INVTYPE_RANGED			= 18,
		INVTYPE_RANGEDRIGHT		= 18,
		INVTYPE_RELIC			= 18,
		INVTYPE_GUN				= 18,
		INVTYPE_CROSSBOW		= 18,
		INVTYPE_WAND			= 18,
		INVTYPE_THROWN			= 18,
		INVTYPE_TABARD			= 19,
	}
	
	local slot2 = {
		INVTYPE_FINGER		= 12,
		INVTYPE_TRINKET		= 14,
		INVTYPE_WEAPON		= 17,
		INVTYPE_2HWEAPON	= 17,
	}
	
	function ItemValue:GetInventorySlot(ItemLink)
		local _, _, _, _, _, _, _, _, ItemEquipLoc = GetItemInfo(ItemLink)
		if not ItemEquipLoc then return end
		
		local s1, s2 = slot1[ItemEquipLoc], slot2[ItemEquipLoc]
		
		return s1, s2
	end
end

function ItemValue:ItemLinkToItemString(ItemLink)
	if not ItemLink then return end
	local il, _, ItemString = strfind(ItemLink, "^|%x+|H(.+)|h%[.+%]")
	
	return il and ItemString or ItemLink
end

function ItemValue:ItemStringToItemLink(ItemString)
	local _, ItemLink = GetItemInfo(ItemString)
	return ItemLink
end

function ItemValue:GetItemString(itemId, enchantId, jewelId1, jewelId2, jewelId3, jewelId4, suffixId, uniqueId)
	return "item:"..itemId..":"..(enchantId or 0)..":"..(jewelId1 or 0)..":"..(jewelId2 or 0)..":"..(jewelId3 or 0)..":"..(jewelId4 or 0)..":"..(suffixId or 0)..":"..(uniqueId or 0)
end

function ItemValue:GetItemLink(itemId, enchantId, jewelId1, jewelId2, jewelId3, jewelId4, suffixId, uniqueId)
	local itemString = self:GetItemString(itemId, enchantId, jewelId1, jewelId2, jewelId3, jewelId4, suffixId, uniqueId)
	local _, itemLink = GetItemInfo(itemString)
	return itemLink
end

function ItemValue:GetUnitEquipment(unit)
	if not unit then return end
	return ItemBonusLib:MergeDetails(
			 ItemBonusLib:BuildBonusSet(
			  ItemBonusLib:GetUnitEquipment(
			   unit
			)))
end

do -- ItemValue:GetInspectEquipment()
	local slots = {
		["Head"] = true,
		["Neck"] = true,
		["Shoulder"] = true,
		["Shirt"] = true,
		["Chest"] = true,
		["Waist"] = true,
		["Legs"] = true,
		["Feet"] = true,
		["Wrist"] = true,
		["Hands"] = true,
		["Finger0"] = true,
		["Finger1"] = true,
		["Trinket0"] = true,
		["Trinket1"] = true,
		["Back"] = true,
		["MainHand"] = true,
		["SecondaryHand"] = true,
		["Ranged"] = true,
		["Tabard"] = true,
	}
	
	for s in pairs(slots) do
		slots[s] = GetInventorySlotInfo(s.."Slot")
	end
	
	local eq = {}
	
	function ItemValue:GetInspectEquipment()
		if not InspectFrame.unit then return end
		
		if UnitName(InspectFrame.unit) == UnitName("player") then return self.EquipStats end
		
		for i in pairs(eq) do
			eq[i] = nil
		end
		
		for slot, id in pairs(slots) do
			eq[slot] = GetInventoryItemLink(InspectFrame.unit, id)
		end
		
		return ItemBonusLib:MergeDetails(ItemBonusLib:BuildBonusSet(eq))
	end
end

--[[
#############################
--    Debug & Test functions
--############################
--]]
do -- ItemValue:Benchmark()
	local t
	local function start()
		t = GetTime()
	end
	local function stop(text)
		local d = floor((GetTime() - t)*1000)
		ItemValue:Print(text .. ":", d .. "ms")
	end
	
	function ItemValue:Benchmark()
		start()
		for i=1, 100 do
			self:RegisterSet(SetTypes.StatSet:new("BenchmarkSet_" .. i))
		end
		stop("Creating 100 StatSets")
		
		start()
		for i=1, 100 do
			self:UnregisterSet(self.SetByName["BenchmarkSet_" .. i])
		end
		stop("Deleting 100 StatSets")
	end
end
