--[[
	### ItemValue ###
	ItemValue StatSet Class
	
--]]

-- Libraries
local L = AceLibrary("AceLocale-2.2"):new("ItemValue")
local AceOO = AceLibrary("AceOO-2.0")
local ItemBonusLib = AceLibrary("ItemBonusLib-1.0")

local db

local ItemBonusLibStats = {
	{"STR", "AGI", "STA", "INT", "SPI", },
	{"ATTACKPOWER", "ATTACKPOWERFERAL", "CR_CRIT", "CR_HIT", "CR_HASTE", "IGNOREARMOR", "CR_EXPERTISE", "RANGEDATTACKPOWER", "RANGEDCRIT"},
	{"DMG", "ARCANEDMG", "FIREDMG", "FROSTDMG", "HOLYDMG", "NATUREDMG", "SHADOWDMG", "CR_SPELLCRIT", "CR_SPELLHIT", "SPELLPEN", "CR_SPELLHASTE"},
	{"HEAL", "MANAREG"},
	{"BASE_ARMOR", "ARMOR_BONUS", "CR_RESILIENCE", "CR_DEFENSE", "CR_BLOCK", "BLOCKVALUE", "CR_DODGE", "CR_PARRY"},
	{"ARCANERES", "FIRERES", "NATURERES", "FROSTRES", "SHADOWRES", "MANA", "HEALTH", "HEALTHREG", "WEAPON_SPEED", "WEAPON_MIN", "WEAPON_MAX", "FISHING", "MINING", "HERBALISM", "SKINNING"},
}

local ItemBonusLibHeaders = {
	"Base_Stats",
	"Physical_DPS",
	"Magical_DPS",
	L["Healing"],
	L["Tanking"],
	L["Misc"],
}

local StatNames = {
	ATTACKPOWERFERAL	= L["Attackpower in Feral Forms"],
	CR_CRIT				= L["Crit Rating"],
	CR_HIT				= L["Hit Rating"],
	CR_HASTE			= L["Haste Rating"],
	RANGEDATTACKPOWER	= L["Ranged Attackpower"],
	RANGEDCRIT			= L["Ranged Crit Rating"],
	DMG					= L["Spelldamage"],
	ARCANEDMG			= L["Arcane Spelldamage"],
	FIREDMG				= L["Fire Spelldamage"],
	FROSTDMG			= L["Frost Spelldamage"],
	HOLYDMG				= L["Holy Spelldamage"],
	NATUREDMG			= L["Nature Spelldamage"],
	SHADOWDMG			= L["Shadow Spelldamage"],
	CR_SPELLCRIT		= L["Spell Crit Rating"],
	CR_SPELLHIT			= L["Spell Hit Rating"],
	SPELLPEN			= L["Spell Penetration"],
	CR_SPELLHASTE		= L["Spell Haste Rating"],
	HEAL				= L["Bonus Healing"],
	MANAREG				= L["Mana/5sec"],
	MANA				= L["Mana"],
	HEALTHREG			= L["Health/5sec"],
	BASE_ARMOR			= L["Base Armor"],
	ARMOR_BONUS			= L["Bonus Armor"],
	CR_EXPERTISE		= L["Weapon Expertise"],
}

local StatDesc = {
	
}

local StatSet = AceOO.Class()

function StatSet.LoadSets()
	db = ItemValue:AcquireDBNamespace("StatSet")
	
	for SetName in pairs(db.profile) do
		ItemValue:RegisterSet(StatSet:new(SetName))
	end
end

function StatSet.prototype:init(Name, Stats)
	StatSet.super.prototype.init(self)
	
	if not db.profile[Name] then
		db.profile[Name] = {
			Stats = Stats or {},
		}
	end
	
	self.Name = Name
	self.Stats = db.profile[Name].Stats
	
	self.Options = {
		type = "group",
		childGroups = "tab",
		name = Name,
		desc = string.format(L["Settings for Set \"%s\"."], Name),
		args = {},
	}
		
	local get = function(info)
		local key = info[#info]
		return self.Stats[key] and tostring(self.Stats[key]) or ""
	end
	local set = function(info, value)
		value = tonumber(value)
		if value == 0 then value = nil end
		
		local key = info[#info]
		self.Stats[key] = value
		ItemValue:ClearCache() 
	end
	
	for h, Stats in pairs(ItemBonusLibStats) do
		local HeaderText = ItemBonusLibHeaders[h]
		self.Options.args[HeaderText] = {
			order = 100 + h,
			type = "group",
			--inline = true,
			name = HeaderText:gsub("_"," "),
			desc = HeaderText:gsub("_"," "),
			args = {},
		}
		
		for n, iblname in pairs(Stats) do
			self.Options.args[HeaderText].args[iblname] = {
				order	= n,
				type	= "input",
				name	= StatNames[iblname] or ItemBonusLib:GetBonusFriendlyName(iblname),
				desc	= StatDesc[iblname] or StatNames[iblname] or ItemBonusLib:GetBonusFriendlyName(iblname),
				get		= get,
				set		= set,
				--validate = tonumber,
				usage	= "<Number>",
				--passValue = iblname,
			}
		end
	end
end

function StatSet.prototype:tostring()
	return "ItemValue StatSet " .. self.Name
end

function StatSet.prototype:OnDelete()
	db.profile[self.Name] = nil
end

function StatSet.prototype:GetEquipValue(eq)
	local Points = 0
	
	if eq then
		for StatName, StatValue in pairs(self.Stats) do
			if eq[StatName] and StatValue then
				Points = Points + (eq[StatName] * StatValue)
			end
		end
	end
	
	return Points
end

function StatSet.prototype:GetItemValue(ItemString)
	return self:GetEquipValue(ItemBonusLib:ScanItem(ItemString, true))
end

function StatSet.prototype:Serialize()
	return {Type = "StatSet", Name = self.Name, Stats = self.Stats}
end

function StatSet:Deserialize(t)
	return StatSet:new(t.Name, t.Stats)
end

ItemValue:RegisterSetType("StatSet", StatSet, "You can use a StatSet to assign a static value to every stat.")
