--[[
	### ItemValue ###
	ItemValue StatSet Class
	
--]]

-- Libraries
local L = AceLibrary("AceLocale-2.2"):new("ItemValue")
local AceOO = AceLibrary("AceOO-2.0")
local ItemBonusLib = AceLibrary("ItemBonusLib-1.0")

local db

local Group = AceOO.Class()

function Group.LoadSets()
	db = ItemValue:AcquireDBNamespace("Group")
	
	for SetName in pairs(db.profile) do
		ItemValue:RegisterSet(Group:new(SetName))
	end
end

function Group.prototype:init(Name, Sets)
    Group.super.prototype.init(self)
	
	if not db.profile[Name] then
		db.profile[Name] = {
			Sets = Sets or {},
		}
	end
	
	self.Name = Name
	self.Sets = db.profile[Name].Sets
	
	self.Options = {
		type = "group",
		name = Name,
		childGroups = "tab",
		desc = string.format(L["Settings for Set \"%s\"."], Name),
		args = {
			Group = {
				type = "group",
				name = "Group",
				desc = "Group",
				args = {
					SetValues = {
						name = "SetValues",
						order = 100,
						type  = "header",
						name  = "Set Values",
					}
				}
			}
		},
	}
	
	ItemValue:RegisterEvent("ItemValue_SetsChanged", function() self:Update() end)
end

function Group.prototype:Update()
	local n = 100
	
	if not self.GetFunc then
		self.GetFunc = function(info)
			local key = info[#info]
			return self.Sets[key] and tostring(self.Sets[key]) or ""
		end
		self.SetFunc = function(info, value)
			value = tonumber(value)
			if value == 0 then value = nil end
			
			local key = info[#info]
			self.Sets[key] = value
			ItemValue:ClearCache() 
		end
	end
	
	-- Remove old sets
	for SetId, Options in pairs(self.Options.args.Group.args) do
		if Options.order > 100 and not ItemValue.SetById[tonumber(SetId)] then
			self.Options.args.Group.args[SetId] = nil
			self.Sets[tonumber(SetId)] = nil
		end
	end
	
	-- Add new sets
	for SetId, Set in pairs(ItemValue.SetById) do
		if Set ~= self and not self.Options.args[SetId] then
			n = n + 1
			self.Options.args.Group.args[tostring(SetId)] = {
				order	= n,
				type	= "input",
				name	= Set.Name,
				desc	= L["Defines the value for this set."],
				get		= self.GetFunc,
				set		= self.SetFunc,
				usage	= "<Number>",
			}
		end
	end
end

function Group.prototype:tostring()
	return "ItemValue Group " .. self.Name
end

function Group.prototype:OnDelete()
	db.profile[self.Name] = nil
end

function Group.prototype:GetEquipValue(eq)
	local Points = 0
	
	if eq then
		for SetId, SetValue in pairs(self.Sets) do
			SetId = tonumber(SetId)
			if ItemValue.SetById[SetId] then
				Points = Points + (ItemValue.SetById[SetId]:GetEquipValue(eq) * SetValue)
			end
		end
	end
	
	return Points
end

function Group.prototype:GetItemValue(ItemString)
	return self:GetEquipValue(ItemBonusLib:ScanItem(ItemString, true))
end

function Group.prototype:Serialize()
	return {Type = "Group", Name = self.Name, Sets = self.Sets}
end

function Group:Deserialize(t)
	return Group:new(t.Name, t.Sets)
end

ItemValue:RegisterSetType("Group", Group)
