--[[
    ### ItemValue ###
    ItemValue StatSetFullEquipmentBasedFormula Class
    
    Thanks to lrdx for posting the idea and codebase for this Module (modhttp://www.wowace.com/forums/index.php?topic=6578.msg191188#msg191188)
--]]

-- Libraries
local L = AceLibrary("AceLocale-2.2"):new("ItemValue")
local AceOO = AceLibrary("AceOO-2.0")
local ItemBonusLib = AceLibrary("ItemBonusLib-1.0")

local db
local FullEquipmentBasedFormula = AceOO.Class()

-- Environment of the Formula, stats will be written into it, allows usage of math functions
local IVFD = {math = math}
setmetatable(IVFD, {["__index"] = function() return 0 end})

function FullEquipmentBasedFormula.LoadSets()
    db = ItemValue:AcquireDBNamespace("FullEquipmentBasedFormula")
    
    for SetName in pairs(db.profile) do
        ItemValue:RegisterSet(FullEquipmentBasedFormula:new(SetName))
    end
end

function FullEquipmentBasedFormula.prototype:init(Name, argFormula, argBaseEquipment)
    FullEquipmentBasedFormula.super.prototype.init(self)

    if not db.profile[Name] then
        db.profile[Name] = {
            Formula = argFormula or "",
            BaseEquipment = argBaseEquipment or ItemBonusLib:GetUnitEquipment("player"),
        }
    end

    self.Name = Name
    self.Formula = db.profile[Name].Formula
	self:UpdatePointsFunc()
    self.BaseEquipment = db.profile[Name].BaseEquipment

	self.Options = {
		type = "group",
		name = Name,
		desc = string.format(L["Settings for Set \"%s\"."], Name),
		childGroups = "tab",
		args = {
			Formula = {
				type = "group",
				name = "Formula",
				order = 10,
				args ={
					Formula = {
						order = 100,
						type  = "input",
						multiline = true,
						width = "full",
						name  = L["Formula"],
						desc  = L["Formula"],
						usage = L["<Formula>"],
						get   = function() return self.Formula or "" end,
						set   = function(i, v)
							self.Formula = v
							self:UpdatePointsFunc()
							db.profile[Name].Formula = v
							ItemValue:ClearCache()
						end
					},
		            ParseEquipment = {
		                order = 200,
		                type = "execute",
		                name = L["Use current equipment"],
		                desc = L["Sets the current equipment as base for calculations."],
		                func = function()
							-- copy the table, because it gets recycled
		                    local currentEquipment = {}
							local eq = ItemBonusLib:GetUnitEquipment("player")
							for k,v in pairs(eq) do
								currentEquipment[k] = v
							end
		                    self.BaseEquipment = currentEquipment
		                    db.profile[Name].BaseEquipment = currentEquipment
		                    ItemValue:ClearCache()
		                end
		            }
				}
			}
		}
	}
	

end

function FullEquipmentBasedFormula.prototype:tostring()
    return "ItemValue FullEquipmentBasedFormula " .. self.Name
end

function FullEquipmentBasedFormula.prototype:OnDelete()
    db.profile[self.Name] = nil
end

function FullEquipmentBasedFormula.prototype:UpdatePointsFunc()
	self.CalcPoints = loadstring("return " .. self.Formula)
	setfenv(self.CalcPoints, IVFD)
end

function FullEquipmentBasedFormula.prototype:GetEquipValue(eq)
	if not (eq and type(eq) == "table") then return 0 end

	for k,v in pairs(eq) do
		IVFD[k] = v
	end
	
	local Points = self:CalcPoints()
	
	for k in pairs(eq) do
		IVFD[k] = nil
	end
	
	return Points or 0
end

local ItemType2EquipLoc = {
    ["INVTYPE_AMMO"] = "Ammo",
    ["INVTYPE_HEAD"] = "Head",
    ["INVTYPE_NECK"] = "Neck",
    ["INVTYPE_SHOULDER"] = "Shoulder",
    ["INVTYPE_BODY"] = "Shirt",
    ["INVTYPE_CHEST"] = "Chest",
    ["INVTYPE_ROBE"] = "Chest",
    ["INVTYPE_WAIST"] = "Waist",
    ["INVTYPE_LEGS"] = "Legs",
    ["INVTYPE_FEET"] = "Feet",
    ["INVTYPE_HAND"] = "Hands",
    ["INVTYPE_FINGER"] = "Finger0",
    ["INVTYPE_TRINKET"] = "Trinket0",
    ["INVTYPE_CLOAK"] = "Back",
    ["INVTYPE_WEAPON"] = "MainHand",
    ["INVTYPE_SHIELD"] = "SecondaryHand",
    ["INVTYPE_2HWEAPON"] = "MainHand",
    ["INVTYPE_WEAPONMAINHAND"] = "MainHand",
    ["INVTYPE_WEAPONOFFHAND"] = "SecondaryHand",
    ["INVTYPE_HOLDABLE"] = "SecondaryHand",
    ["INVTYPE_RANGED"] = "Ranged",
    ["INVTYPE_THROWN"] = "Ranged",
    ["INVTYPE_RANGEDRIGHT"] = "Ranged",
    ["INVTYPE_RELIC"] = "Ranged",
	["INVTYPE_TABARD"] = "Tabard",
}

function FullEquipmentBasedFormula.prototype:GetItemValue(newItemString)
    local _,newItemLink,_,_,_,_,_,_,newItemType,_ = GetItemInfo(newItemString)
    local newItemEquipLoc = ItemType2EquipLoc[newItemType]

	if newItemEquipLoc and self.BaseEquipment and type(self.BaseEquipment) == "table" then
		-- Replace item and switch old item back in after calculation
		local oldItem = self.BaseEquipment[newItemEquipLoc]
		self.BaseEquipment[newItemEquipLoc] = newItemLink
		
		local value = self:GetEquipValue(ItemBonusLib:MergeDetails(ItemBonusLib:BuildBonusSet(self.BaseEquipment)))
		
		self.BaseEquipment[newItemEquipLoc] = oldItem
		return value
	else
		-- behavior like the normal formula if the item isnt equipable (jewels, etc) or no equipment set
		return self:GetEquipValue(ItemBonusLib:ScanItem(newItemString, true))
	end
end

function FullEquipmentBasedFormula.prototype:Serialize()
    return {
        Type = "FullEquipmentBasedFormula",
        Name = self.Name,
        Formula = self.Formula,
        BaseEquipment = self.BaseEquipment,
    }
end

function FullEquipmentBasedFormula:Deserialize(t)
    return FullEquipmentBasedFormula:new(t.Name, t.Formula, t.BaseEquipment)
end

ItemValue:RegisterSetType("FullEquipmentBasedFormula", FullEquipmentBasedFormula)
