--[[
	### ItemValue ###
	ItemValue StatSetFormula Class
	
	Thanks to lrdx for his codebase for the full equipment based valuation (http://www.wowace.com/forums/index.php?topic=6578.msg191188#msg191188)
--]]

-- Libraries
local L = AceLibrary("AceLocale-2.2"):new("ItemValue")
local AceOO = AceLibrary("AceOO-2.0")
local ItemBonusLib = AceLibrary("ItemBonusLib-1.0")

local db
local Formula = AceOO.Class()

-- Environment of the Formula, stats will be written into it, allows usage of math functions
local IVFD = {math = math}
setmetatable(IVFD, {["__index"] = function() return 0 end})

function Formula.LoadSets()
	db = ItemValue:AcquireDBNamespace("Formula")
	
	for SetName in pairs(db.profile) do
		ItemValue:RegisterSet(Formula:new(SetName))
	end
end

function Formula.prototype:init(Name, argFormula)
	Formula.super.prototype.init(self)
	
	if not db.profile[Name] then
		db.profile[Name] = {
			Formula = argFormula or "",
		}
	end
	
	self.Name = Name
	self.Formula = db.profile[Name].Formula
	self:UpdatePointsFunc()
	
	self.Options = {
		type = "group",
		name = Name,
		desc = string.format(L["Settings for Set \"%s\"."], Name),
		childGroups = "tab",
		args = {
			Formula = {
				type = "group",
				name = "Formula",
				order = 10,
				args ={
					Formula = {
						order = 100,
						type  = "input",
						multiline = true,
						width = "full",
						name  = L["Formula"],
						desc  = L["Formula"],
						usage = L["<Formula>"],
						get   = function() return self.Formula or "" end,
						set   = function(i, v)
							self.Formula = v
							self:UpdatePointsFunc()
							db.profile[Name].Formula = v
							ItemValue:ClearCache()
						end
					},
				}
			}
		}
	}

end

function Formula.prototype:tostring()
	return "ItemValue Formula " .. self.Name
end

function Formula.prototype:OnDelete()
	db.profile[self.Name] = nil
end

function Formula.prototype:UpdatePointsFunc()
	self.CalcPoints = loadstring("return " .. self.Formula or "")
	setfenv(self.CalcPoints, IVFD)
end

function Formula.prototype:GetEquipValue(eq)
	if not (eq and type(eq) == "table") then return 0 end
	
	for k,v in pairs(eq) do
		IVFD[k] = v
	end
	
	local Points = self:CalcPoints()
	
	for k in pairs(eq) do
		IVFD[k] = nil
	end
	
	return Points or 0
end

function Formula.prototype:GetItemValue(ItemString)
	return self:GetEquipValue(ItemBonusLib:ScanItem(ItemString, true))
end

function Formula.prototype:Serialize()
	return {Type = "Formula", Name = self.Name, Formula = self.Formula}
end

function Formula:Deserialize(t)
	return Formula:new(t.Name, t.Formula)
end

ItemValue:RegisterSetType("Formula", Formula)
