--[[
	### ItemValue ###
	Best Items
	
--]]

local L = AceLibrary("AceLocale-2.2"):new("ItemValue")
local Gratuity = AceLibrary("Gratuity-2.0")

local tinsert = table.insert

local class = select(2, UnitClass("player"))
local classes = {"DRUID", "HUNTER", "MAGE", "PALADIN", "PRIEST", "ROGUE", "SHAMAN", "WARLOCK", "WARRIOR"}
local classesRev = {}
for k,v in pairs(classes) do
	classesRev[v] = k
end

local Slots = {}

local Default

-- Reuse some tables equal for every class
local Armor = {"Armor"}
local Weapon = {"Weapon"}
local Misc = {"Miscellaneous"}

local INVTYPE_HEAD =		{"INVTYPE_HEAD"}
local INVTYPE_SHOULDER =	{"INVTYPE_SHOULDER"}
local INVTYPE_CHEST =		{"INVTYPE_CHEST", "INVTYPE_ROBE"}
local INVTYPE_WRIST =		{"INVTYPE_WRIST"}
local INVTYPE_HAND =		{"INVTYPE_HAND"}
local INVTYPE_WAIST =		{"INVTYPE_WAIST"}
local INVTYPE_LEGS =		{"INVTYPE_LEGS"}
local INVTYPE_FEET =		{"INVTYPE_FEET"}

local SlotsNeck =		{ItemType = "Armor", ItemSubTypes = Misc, ItemEquipLocs = {"INVTYPE_NECK"}}
local SlotsFinger =		{ItemType = "Armor", ItemSubTypes = Misc, ItemEquipLocs = {"INVTYPE_FINGER"}}
local SlotsTrinket =	{ItemType = "Armor", ItemSubTypes = Misc, ItemEquipLocs = {"INVTYPE_TRINKET"}}

local SlotsBack =		{ItemType = "Armor", ItemSubTypes = {"Cloth"}, ItemEquipLocs = {"INVTYPE_CLOAK"}}

local classProfiles = {
	DRUID = {
		Default = {"Leather", "Cloth"},
		
		Mainhand = 	{ItemType = "Weapon", ItemSubTypes = {"Daggers", "One-Handed Maces", "Fist Weapons"}, ItemEquipLocs = {"INVTYPE_WEAPONMAINHAND", "INVTYPE_WEAPON", "INVTYPE_2HWEAPON"}},
		Offhand = 	{ItemType = "Armor", ItemSubTypes = Misc, ItemEquipLocs = {"INVTYPE_HOLDABLE"}},
		TwoHand =	{ItemType = "Weapon", ItemSubTypes = {"Two-Handed Maces", "Staves"}, ItemEquipLocs = {"INVTYPE_2HWEAPON"}},
		Ranged =	{ItemType = "Armor", ItemSubTypes = {"Idols"}, ItemEquipLocs = {"INVTYPE_RELIC"}},
	},
	HUNTER = {
		Default = {"Mail", "Leather"},
		
		Mainhand = 	{ItemType = "Weapon", ItemSubTypes = {"Daggers", "One-Handed Swords", "One-Handed Axes", "Fist Weapons"}, ItemEquipLocs = {"INVTYPE_WEAPONMAINHAND", "INVTYPE_WEAPON", "INVTYPE_2HWEAPON"}},
		Offhand = 	{ItemType = "Weapon", ItemSubTypes = {"Daggers", "One-Handed Swords", "One-Handed Axes", "Fist Weapons"}, ItemEquipLocs = {"INVTYPE_WEAPONOFFHAND", "INVTYPE_WEAPON"}},
		TwoHand =	{ItemType = "Weapon", ItemSubTypes = {"Two-Handed Swords", "Two-Handed Axes", "Staves", "Polearms"}, ItemEquipLocs = {"INVTYPE_2HWEAPON"}},
		Ranged =	{ItemType = "Weapon", ItemSubTypes = {"Crossbows", "Bows", "Guns", "Thrown"}, ItemEquipLocs = { "INVTYPE_RANGEDRIGHT", "INVTYPE_THROWN", "INVTYPE_RANGED"}},
	},
	MAGE = {
		Default = {"Cloth"},
		
		Mainhand = 	{ItemType = "Weapon", ItemSubTypes = {"Daggers", "One-Handed Swords"}, ItemEquipLocs = {"INVTYPE_WEAPONMAINHAND", "INVTYPE_WEAPON", "INVTYPE_2HWEAPON"}},
		Offhand = 	{ItemType = "Armor", ItemSubTypes = Misc, ItemEquipLocs = {"INVTYPE_HOLDABLE"}},
		TwoHand =	{ItemType = "Weapon", ItemSubTypes = {"Staves"}, ItemEquipLocs = {"INVTYPE_2HWEAPON"}},
		Ranged =	{ItemType = "Weapon", ItemSubTypes = {"Wands"}, ItemEquipLocs = {"INVTYPE_RANGEDRIGHT"}},
	},
	PALADIN = {
		Default = {"Plate", "Mail", "Leather", "Cloth"},
		
		Mainhand = 	{ItemType = "Weapon", ItemSubTypes = {"One-Handed Swords", "One-Handed Maces", "One-Handed Axes"}, ItemEquipLocs = {"INVTYPE_WEAPONMAINHAND", "INVTYPE_WEAPON", "INVTYPE_2HWEAPON"}},
		Offhand = 	{ItemType = "Armor", ItemSubTypes = {"Miscellaneous", "Shields"}, ItemEquipLocs = {"INVTYPE_HOLDABLE", "INVTYPE_SHIELD"}},
		TwoHand =	{ItemType = "Weapon", ItemSubTypes = {"Two-Handed Swords", "Two-Handed Axes", "Two-Handed Maces", "Polearms"}, ItemEquipLocs = {"INVTYPE_2HWEAPON"}},
		Ranged =	{ItemType = "Armor", ItemSubTypes = {"Librams"}, ItemEquipLocs = {"INVTYPE_RELIC"}},
	},
	PRIEST = {
		Default = {"Cloth"},
		
		Mainhand = 	{ItemType = "Weapon", ItemSubTypes = {"Daggers", "One-Handed Maces"}, ItemEquipLocs = {"INVTYPE_WEAPONMAINHAND", "INVTYPE_WEAPON"}},
		Offhand = 	{ItemType = "Armor", ItemSubTypes = Misc, ItemEquipLocs = {"INVTYPE_HOLDABLE"}},
		TwoHand =	{ItemType = "Weapon", ItemSubTypes = {"Staves"}, ItemEquipLocs = {"INVTYPE_2HWEAPON"}},
		Ranged =	{ItemType = "Weapon", ItemSubTypes = {"Wands"}, ItemEquipLocs = {"INVTYPE_RANGEDRIGHT"}},
	},
	ROGUE = {
		Default = {"Leather"},
		
		Mainhand = 	{ItemType = "Weapon", ItemSubTypes = {"Daggers", "One-Handed Maces", "One-Handed Swords", "Fist Weapons"}, ItemEquipLocs = {"INVTYPE_WEAPONMAINHAND", "INVTYPE_WEAPON"}},
		Offhand = 	{ItemType = "Weapon", ItemSubTypes = {"Daggers", "One-Handed Maces", "One-Handed Swords", "Fist Weapons"}, ItemEquipLocs = {"INVTYPE_WEAPONOFFHAND", "INVTYPE_WEAPON"}},
		Ranged =	{ItemType = "Weapon", ItemSubTypes = {"Crossbows", "Bows", "Guns", "Thrown"}, ItemEquipLocs = { "INVTYPE_RANGEDRIGHT", "INVTYPE_THROWN", "INVTYPE_RANGED"}},
	},
	SHAMAN = {
		Default = {"Mail", "Leather", "Cloth"},
		
		Mainhand = 	{ItemType = "Weapon", ItemSubTypes = {"Daggers", "One-Handed Maces", "One-Handed Axes", "Fist Weapons"}, ItemEquipLocs = {"INVTYPE_WEAPONMAINHAND", "INVTYPE_WEAPON"}},
		Offhand = 	{ItemType = "Weapon", ItemSubTypes = {"Daggers", "One-Handed Maces", "One-Handed Axes", "Fist Weapons"}, ItemEquipLocs = {"INVTYPE_WEAPONOFFHAND", "INVTYPE_WEAPON"}},
		OffhandI =	{ItemType = "Armor", ItemSubTypes = {"Shields", "Miscellaneous"}, ItemEquipLocs = {"INVTYPE_SHIELD", "INVTYPE_HOLDABLE"}},
		TwoHand =	{ItemType = "Weapon", ItemSubTypes = {"Two-Handed Maces", "Two-Handed Axes", "Staves"}, ItemEquipLocs = {"INVTYPE_2HWEAPON"}},
		Ranged =	{ItemType = "Armor", ItemSubTypes = {"Crossbows", "Bows", "Guns", "Thrown"}, ItemEquipLocs = { "INVTYPE_RANGEDRIGHT", "INVTYPE_THROWN", "INVTYPE_RANGED"}},
	},
	WARLOCK = {
		Default = {"Cloth"},
		
		Mainhand = 	{ItemType = "Weapon", ItemSubTypes = {"Daggers", "One-Handed Swords"}, ItemEquipLocs = {"INVTYPE_WEAPONMAINHAND", "INVTYPE_WEAPON"}},
		Offhand = 	{ItemType = "Armor", ItemSubTypes = Misc, ItemEquipLocs = {"INVTYPE_HOLDABLE"}},
		TwoHand =	{ItemType = "Weapon", ItemSubTypes = {"Staves"}, ItemEquipLocs = {"INVTYPE_2HWEAPON"}},
		Ranged =	{ItemType = "Weapon", ItemSubTypes = {"Wands"}, ItemEquipLocs = {"INVTYPE_RANGEDRIGHT"}},
	},
	WARRIOR = {
		Default = {"Plate", "Mail", "Leather"},
		
		Mainhand = 	{ItemType = "Weapon", ItemSubTypes = {"Daggers", "One-Handed Swords", "One-Handed Maces", "One-Handed Axes", "Fist Weapons"}, ItemEquipLocs = {"INVTYPE_WEAPONMAINHAND", "INVTYPE_WEAPON"}},
		Offhand = 	{ItemType = "Weapon", ItemSubTypes = {"Daggers", "One-Handed Swords", "One-Handed Maces", "One-Handed Axes", "Fist Weapons"}, ItemEquipLocs = {"INVTYPE_WEAPONOFFHAND", "INVTYPE_WEAPON"}},
		OffhandI =	{ItemType = "Armor", ItemSubTypes = {"Shields"}, ItemEquipLocs = {"INVTYPE_SHIELD"}},
		TwoHand =	{ItemType = "Weapon", ItemSubTypes = {"Two-Handed Swords", "Two-Handed Maces", "Two-Handed Axes", "Staves", "Polearms"}, ItemEquipLocs = {"INVTYPE_2HWEAPON"}},
		Ranged =	{ItemType = "Weapon", ItemSubTypes = {"Crossbows", "Bows", "Guns", "Thrown"}, ItemEquipLocs = { "INVTYPE_RANGEDRIGHT", "INVTYPE_THROWN", "INVTYPE_RANGED"}},
	}
}

for n, c in pairs(classProfiles) do
	c.Head = 		{ItemType = "Armor", ItemSubTypes = c.Default, ItemEquipLocs = INVTYPE_HEAD}
	c.Shoulder =	{ItemType = "Armor", ItemSubTypes = c.Default, ItemEquipLocs = INVTYPE_SHOULDER}
	c.Chest =		{ItemType = "Armor", ItemSubTypes = c.Default, ItemEquipLocs = INVTYPE_CHEST}
	c.Wrist =		{ItemType = "Armor", ItemSubTypes = c.Default, ItemEquipLocs = INVTYPE_WRIST}
	c.Hand =		{ItemType = "Armor", ItemSubTypes = c.Default, ItemEquipLocs = INVTYPE_HAND}
	c.Waist =		{ItemType = "Armor", ItemSubTypes = c.Default, ItemEquipLocs = INVTYPE_WAIST}
	c.Legs =		{ItemType = "Armor", ItemSubTypes = c.Default, ItemEquipLocs = INVTYPE_LEGS}
	c.Feet =		{ItemType = "Armor", ItemSubTypes = c.Default, ItemEquipLocs = INVTYPE_FEET}
	
	c.Default = nil
	
	c.Neck =	SlotsNeck
	c.Finger=	SlotsFinger
	c.Trinket =	SlotsTrinket
	
	c.Back =	SlotsBack
end


-- GUI
local slots = {"Head", "Neck", "Shoulder", "Back", "Chest", "Wrist", "Hand", "Waist", "Legs", "Feet", "Finger", "Trinket", "Mainhand", "Offhand", "OffhandI", "TwoHand", "Ranged"}
local slotId
local OnlyUsable = true
local classProfile = classesRev[class]
local sets = {}
local setId
local onlyUsable = true
ItemValue.Options.args.Tools.args.BestItems = {
	order = 1,
	type = "group",
	name = "Best Items",
	desc = "Best Items",
	args = {
		ClassProfile = {
			order = 1,
			type = "select",
			name = "Class Profile",
			desc = "Class Profile",
			values = classes,
			get = function(i) return classProfile end,
			set = function(i, v) classProfile = v end,
		},
		Slot = {
			order = 2,
			type = "select",
			name = "Slot",
			desc = "Slot",
			get = function(i) return slotId end,
			set = function(i, v) slotId = v end,
			values = slots,
		},
		OnlyUsable = {
			order = 3,
			type = "toggle",
			name = "Only Usable Items",
			desc = "Only Usable Items",
			get = function() return onlyUsable end,
			set = function(i, v) onlyUsable = v end,
		},
		Set = {
			order = 4,
			type = "select",
			name = "Set",
			desc = "Set",
			values = function()
				for k in pairs(sets) do
					sets[k] = nil
				end
				for i, s in pairs(ItemValue.SetById) do
					sets[i] = s.Name
				end
				return sets
			end,
			set = function(i, v) setId = v end,
			get = function() return setId end,
		},
		Go = {
			order = 5,
			type = "execute",
			name = "Go",
			desc = "Go",
			width = "full",
			func = function()
				if ItemValue.co then
					ItemValue:Print("There is already a background BestItems scan running!")
				else
					ItemValue.co = coroutine.create(function() ItemValue:GetBestItemsCR(slots[slotId], classProfiles[classes[classProfile]][slots[slotId]], ItemValue.SetById[setId]) end)
				end
				
				local err
				local n = 0
				local nMax = 1
				
				ItemValue:ScheduleRepeatingEvent("ItemValueBestItemScan", function()
					if coroutine.status(ItemValue.co) == "dead" then
						ItemValue.co = nil
						ItemValue:CancelScheduledEvent("ItemValueBestItemScan")
						ItemValue:CancelScheduledEvent("ItemValueBestItemScanStatus")
					else
						err, n, nMax = coroutine.resume(ItemValue.co)
					end
				end, 0)
				
				ItemValue:ScheduleRepeatingEvent("ItemValueBestItemScanStatus", function()
					ItemValue:Print(string.format("BestItems scan is |cffffff78%.1f%%|r completed.", n/nMax*100))
				end, 1)
				
			end,
		}
	}
}


local function round(num, idp)
	local mult = 10^(idp or 0)
	return math.floor(num * mult + 0.5) / mult
end

local function ItemIsUsable(id)
	Gratuity:SetHyperlink("item:" .. id)
	
	for i=1, Gratuity:NumLines() do
		local cR, cG, cB = Gratuity.vars.Llines[i]:GetTextColor()
		if cR > 0.9 and cG < 0.2 and cB < 0.2 then
			return false
		end
		
		cR, cG, cB = Gratuity.vars.Rlines[i]:GetTextColor()
		if cR > 0.9 and cG < 0.2 and cB < 0.2 then
			return false
		end
	end
	
	return true
end

local Qualities = {3, 4, 5}
function ItemValue:GetBestItems(SlotName, Slot, Set)
	if not type(Slot) == "table" then return end
	
	local Items = ItemValue:GetItems(Qualities, Slot.ItemType, Slot.ItemSubTypes, Slot.ItemEquipLocs, UnitLevel("player"))
	
	GameTooltip:Hide()
	
	ItemValue:Print(string.format(L["Scanned |cffffff78%d|r Items for Set |cffffff78%s|r, which fit into the |cffffff78%s|r slot:"], #Items, Set.Name, SlotName))
	if #Items == 0 then return end
	
	-- Set enchant id
	local eId, i1, i2
	local slot1, slot2 = ItemValue:GetInventorySlot(Items[1])
	
	if slot1 then i1 = ItemValue:ItemLinkToItemString(GetInventoryItemLink("player", slot1)) end
	if slot2 then i2 = ItemValue:ItemLinkToItemString(GetInventoryItemLink("player", slot2)) end
	
	if i1 then eId = select(3, strsplit(":", i1)) end
	if i2 and not eId then eId = select(3, strsplit(":", i2)) end
	
	local ItemsTable = ItemValue.tnew()
	
	for _, ItemId in pairs(Items) do
		local Item = ItemValue.tnew()
		Item.ItemId = ItemId
		
		if ItemValue:GetSocketCount(ItemId) == 0 then
			Item.ItemString = ItemValue:GetItemString(ItemId, eId)
		else
			local bjc = ItemValue:CalcBestJewels(Set, ItemId)
			Item.ItemString = ItemValue:GetItemString(ItemId, eId, bjc.Jewel1.EnchantId, bjc.Jewel2.EnchantId, bjc.Jewel3.EnchantId)
		end
		
		Item.Points = Set:GetItemValue(Item.ItemString)
		
		tinsert(ItemsTable, Item)
	end
	
	table.sort(ItemsTable, function (v1, v2) return v1.Points > v2.Points end)
	
	local c1, c2
	if i1 then c1 = round(Set:GetItemValue(i1), ItemValue.db.profile.Round[Set.Name]) end
	if i2 then c2 = round(Set:GetItemValue(i2), ItemValue.db.profile.Round[Set.Name]) end
	
	for i=1, 15 do
		local Item = ItemsTable[i]
		if not Item then break end
		local b = round(Item.Points, ItemValue.db.profile.Round[Set.Name])
		
		ItemValue:Print(i..".", ItemValue:ItemStringToItemLink(Item.ItemString),
			ItemValue:Compare(ItemValue.db.profile.TooltipComparisonMode, 
			ItemValue.db.profile.TooltipShowPoints, ItemValue.db.profile.TooltipSwapColor, ItemValue.db.profile.TooltipSwapComparison, 
			b, c1, c2))
		
		-- Show only better items, but at least one
		if (c1 and c1 >= b or not c1) and (c2 and c2 >= b or not c2) and (c1 or c2) then break end
	end
	
	ItemValue.tddel(ItemsTable)
	
	ItemValue.tdel(Items)
end

function ItemValue:GetBestItemsCR(SlotName, Slot, Set)
	if not type(Slot) == "table" then return end
	
	coroutine.yield(0, 1)
	local Items = ItemValue:GetItems(Qualities, Slot.ItemType, Slot.ItemSubTypes, Slot.ItemEquipLocs, UnitLevel("player"))
	local n = #Items
	GameTooltip:Hide()
	
	if #Items == 0 then return end
	
	-- Set enchant id
	local eId, i1, i2
	local slot1, slot2 = ItemValue:GetInventorySlot(Items[1])
	
	if slot1 then i1 = ItemValue:ItemLinkToItemString(GetInventoryItemLink("player", slot1)) end
	if slot2 then i2 = ItemValue:ItemLinkToItemString(GetInventoryItemLink("player", slot2)) end
	
	if i1 then eId = select(3, strsplit(":", i1)) end
	if i2 and not eId then eId = select(3, strsplit(":", i2)) end
	
	local startTime = GetTime()
	
	ItemValue:Print(string.format("Scanning |cffffff78%d|r Items for Set |cffffff78%s|r, which fit into the |cffffff78%s|r slot...", n, Set.Name, SlotName))
	
	local ItemsTable = ItemValue.tnew()
	
	for i, ItemId in pairs(Items) do
		local Item = ItemValue.tnew()
		Item.ItemId = ItemId
		
		if ItemValue:GetSocketCount(ItemId) == 0 then
			Item.ItemString = ItemValue:GetItemString(ItemId, eId)
		else
			local bjc = ItemValue:CalcBestJewels(Set, ItemId)
			Item.ItemString = ItemValue:GetItemString(ItemId, eId, bjc.Jewel1.EnchantId, bjc.Jewel2.EnchantId, bjc.Jewel3.EnchantId)
		end
		
		Item.Points = Set:GetItemValue(Item.ItemString)
		
		tinsert(ItemsTable, Item)
		
		--if i % 3 == 0 then
			coroutine.yield(i, n)
		--end
	end
	
	table.sort(ItemsTable, function (v1, v2) return v1.Points > v2.Points end)
	
	local c1, c2
	if i1 then c1 = round(Set:GetItemValue(i1), ItemValue.db.profile.Round[Set.Name]) end
	if i2 then c2 = round(Set:GetItemValue(i2), ItemValue.db.profile.Round[Set.Name]) end
	
	ItemValue:Print(string.format("Scan completed in %.1f seconds, the best |cffffff78%d|r Items for the |cffffff78%s|r slot are:", GetTime()-startTime, 15, SlotName))

	for i=1, 15 do
		local Item = ItemsTable[i]
		if not Item then break end
		local b = round(Item.Points, ItemValue.db.profile.Round[Set.Name])
		
		ItemValue:Print(i..".", ItemValue:ItemStringToItemLink(Item.ItemString),
			ItemValue:Compare(ItemValue.db.profile.TooltipComparisonMode, 
			ItemValue.db.profile.TooltipShowPoints, ItemValue.db.profile.TooltipSwapColor, ItemValue.db.profile.TooltipSwapComparison, 
			b, c1, c2))
		
		-- Show only better items, but at least one
		if (c1 and c1 >= b or not c1) and (c2 and c2 >= b or not c2) and (c1 or c2) then break end
	end
	
	ItemValue.tddel(ItemsTable)
	ItemValue.tdel(Items)
end

function ItemValue:GetItems(filterQuality, filterItemType, filterSubType, filterEquipLoc, filterMaxLevel)
	local items = {}
	
	for i=1, 35000 do
		local itemName, itemLink, itemRarity, itemLevel, itemMinLevel, itemType, itemSubType, itemStackCount, itemEquipLoc = GetItemInfo(i)
		
		local matching = (not not itemName) and (onlyUsable and filterMaxLevel >= itemMinLevel or not onlyUsable) and (filterItemType == itemType)
		local match = false
		
		if matching and filterQuality and type(filterQuality) == "table" then
			for _, q in pairs(filterQuality) do
				if itemRarity == q then
					match = true
					break
				end
			end
			matching = match
		end
		
		if matching and filterSubType and type(filterSubType) == "table" then
			match = false
			for _, t in pairs(filterSubType) do
				if itemSubType == t then
					match = true
					break
				end
			end
			matching = match
		end
		
		if matching and filterEquipLoc and type(filterEquipLoc) == "table" then
			match = false
			for _, l in pairs(filterEquipLoc) do
				if itemEquipLoc == l then
					match = true
					break
				end
			end
			matching = match
		end
		
		if matching then tinsert(items, i) end
	end
	
	if onlyUsable then
		local usableItems = {}
		for _, i in pairs(items) do
			if ItemIsUsable(i) then tinsert(usableItems, i) end
		end
		return usableItems
	else
		return items
	end
end
