--[[

	InventoryOnPar : An Inventory addon that shows when you have outgrown your equipment
		copyright 2006 - 2008 by Pericles

]]

--[[

v0.01 - Initial release

v0.02 - Added /iop version
		Fixed problem with loading of polearms

v0.03 - Fixed problem with loading of thrown weapons

v0.04 - Now shows empty slots in white

v0.10 - Now uses a graphical window to display data

v0.11 - Fixed issue with no Helm giving colour error

v0.20 - Added a button on character inventory screen to open On Par
        Added description field for slots
        Left justified text

v0.21 - Tidied up no longer needed local vars
        Added localization

v0.22 - Changed Item Level routine to take player level and return colour as well as item level

v0.30 - Changed Item Level to account for items with multiple levels

V1.00 - Added Item Level to game tooltip - still need to work out "Required Level" text from item tooltip

v1.10 - now works with chat text tooltip and lootlink tooltips

v1.20 - now also works with EquipCompare comparison tooltips

v1.21 - deals with items that contain " of ..." e.g., "of the boar" and looks up the shortened name instead

v1.30 - deals with items that have multiple levels by reading "Required Level" text from item tooltip after
		v1.21 they were always showing as -3, -5 etc thus always grey.

v1.40 - shows item ITEM_RARITY in brackets after item name

v1.41 - fix bug for lootlink

v1.50 - updated for patch 1.10

v1.51 - fix for missing Item Levels on main tooltip post patch 1.10

v2.00 - added new items content from patch 1.10 includes librams, idols, totems & dungeon sets

v2.01 - fixed item levels of multi level items

v2.02 - added InventoryOnPar Score and Titan Panel display (initial testing version)

v2.03 - modified window textbox to larger size - fixed crash if an empty slot

v2.10 - fixed bugs in TitanButton version ready for public release

v2.20 - added scans of other players

v2.30 - added average itemLevel for players with good kit above lvl 60, added help to the /iop command line and changed the default behaviour to show help, added a fix to try to hide window when character window closes

v3.00 - added UI screen to view captured player data

v3.10 - added Options Editor

v3.20 - added weighted score calculation

v3.30 - replaced name lookups with itemID lookups, enlarged result window

v3.31 - fixed item level zero or 1 (all level items) from dragging down score

v3.40 - added raid and party options to scan all players in a raid/party
	 added level tooltip to atlasloot tooltip
	 fixed sorting options, fixed colours in table, added back in the option to scan or not to scan others

v3.41 - added in data for Librams, Totems and Idols

v3.50 - the view list now shows the kit of a player when you click on their name
	    empty slots that cannot be filled at your level now score zero instead of -level

v3.51 - fixed bug with two-handed weapons

v3.52 - fixed bug with localisations not having weights

v3.53 - added german localisations - thanks to fuersthoelle for the translations

v3.54 - updated toc to patch 1.12

v3.55 - added fix for error on 316 when itemRarity is nil, also moved rarity strings to localisation.lua
           added german localisation for two handed items - thanks to fuersthoelle for the translations
	 changed min levels for empty slots as values in v3.54 seemed too low in actual playtesting giving large negative scores
	 fixed a problem if you started a character on a new realm (name nil line 316 error)
		
v3.60 - added extra items introduced in patch 1.12 and 1.12.1

v3.61 - Added German items to StringData & NumericData files so tooltips work in German version

v3.62 - Enabled cleanup options - needs testing 
        Added kit level display.

v3.63 - Fixed a bug in IOP_GetItemLevelByName where a nil tooltip would return a nil itemLevel causing error

v4.00 - Initial conversion for patch 2.0
		Fixed bug where kitLevel was always a player's kit level - not the scanned player's kit level
		
v4.01 - Removed debugging info left in by mistake

v4.10 - Changed Tooltip to use new patch 2.0.3+ API functions and removed the String Library
        Changed MAX_LEVEL to 70

v5.00 -	Completely changed the score calc for Burning Crusade kit added a bcAdjustment to subtract 10 per item per level above 60
		Changed the colour banding when over lvl 60 to be in bands of 5 based on average kit level
		Fixed bug where max level options were not acknowledged
		Fixed capatilisation error where ticking the show players option didn't change the behaviour
		Added option not to show coloured itemLevel in tooltips
		
v5.10 - Fixed bug where personal IOP score was displaying wrongly - it was using average kit levels above lvl 60 not your level
		Fixed calculation of multiplier 
		A lvl 61 will have a zero on par score if their average kit is lvl 65
		A lvl 62 will have a zero on par score if their average kit is lvl 70
		A lvl 63 will have a zero on par score if their average kit is lvl 75 etc.
		
v5.20 - Altered additional bonus for rare, epic, ledgendary items to be +10, +20, +30 rather than +5, +10, +15

v6.00 - Now use new item weights as per article on wowwiki - http://www.wowwiki.com/Formulas:Item_Values (slot modifiers section)
		Database will be auto cleared on first startup as old values on slightly different calculation basis.
		Added a FuBar plugin
		
v6.01 - Fixed a itemLevel nil error

v6.10 - Added Guild info to display player lists screen
		Changed score colours on display player lists screen now 0-99, 100-249, 250-499, 500-749, 750+
		Added a list only my guild players tick box to display player lists screen
		Moved the ONPar button on character sheet - now only appears on paperdoll sheet and no longer overlaps with Outfitter icon
		Added missing column labels from the display player lists screen
		
v6.11 - Fixed nil error on viewing player you hadn't seen before

v6.50 - Replace blue/epic etc bonus system with one from wowwiki - changed weighting from 1.75 to 1.2 so similar scores retained

v7.01 - Inadvertently called v6.50 - v7.00. 
		Updated database to v7.0 - stores item links for paper dolls
		Added Initial attempt at Paper Doll display
		Changed colour for artifacts to that reported by API

v7.10 - Store race & gender for paperdoll portrait

v7.11 - Forgot to include border texture in zip file

v7.20 - Fixed guild only list clicking showing wrong person
		Fixed opening window showing tick for guild only when option not set
		Fixed Paperdoll window appearing behind the UI frame when clicking on record
		Added a right click delete player option
		Enabled the Cleanup option to delete all players data recorded over 14 days ago
		Now updates UI table if its open and you target a new player
	
v7.30 - PaperDoll frame now updates on equipping a new item or leveling up
		Forgot to have Red border for significantly better items
		Remove announcement of player score to chat window
		Remove word unsafe and add to tooltip instead put blue unsafe border on item 
		Allow shift click of unsafe options to attempt to load tooltip
		
v7.31 - Inadvertently disabled the output of score to chat

v7.40 - Updated TOCs for patch 2.2

v7.50 - Shift left click on paper doll now adds item to chat window
        Ctrl left click on paper doll now shows item in dressing room window
		Shift left click on paper doll forces an item fetch from server - warning may disconnect you
		Fixed own guild only list not scrolling
		Fixed own guild only tick box not clearing when form closed
		Added new average guild score and average lvl 70 score
		
v7.51 - Changed min level for trinket to be 50 - as non profession trinkets only start being available at 40+
           and most are itemLevel 50+ so should be picked up at 45+ 5 levels give time to fill both slots
	 
v7.52 - Changed min level for Relic slot to be 60 - as most start being available at 55+ 
		Fixed German localisation to be 50 as min level for trinket
		Doubled the width of the Fu_Bar tooltip to 4 columns - need feedback from FuBar users that this works.
		
v7.53 - Updated TOC for patch 2.3

v7.60 - Added option to supress score output to chat text
	    Fixed FuBar Plugin tooltip to be full width
		
v7.70 - Added option to limit list to raid/party members only
        Cleanups now show formatted date string not UNIX time number
		
v7.71 - Removed debug info left by mistake

v8.00 - Added support for bonus itemLevels for socketing gems scores based bonuses from itemWeights
			Green quality gems = +6
			Blue quality gems = +8
			Epic quality gems = +10
			Meta gems = +4 (on top of quality)
		Weighting changed to 1.1 to rebalance towards v7 database values
		Database version changed so will clear old data so all values are still directly comparable
		
v8.01 - Fixed bug in epic gem addition

v8.02 - Fixed frame bug where invisible frame prevents clicks on part of chest icon on paperdoll frame

v8.10 - Added text to tooltip to show if items enchanted/buffed
		Updated database to v8.10 which will clear previously stored values to fix issue with gem costs
		Gem costs were adding itemLevels of gems but ignoring item cost of gem slot - fixed.
		Still need to work out how to count number of empty gem sockets
		
v8.11 - Fixed nil error in UI line 362

v8.12 - Fixed missing On Par button on paperdoll frame.

v8.13 - Added average score display to party/raid only lists

v9.00 - Added option to ignore Resilience on items when calculating scores.

v9.01 - Force update of FuBar & Titan on switching ignore resilience on & off

v9.10 - Added default option to ignore Resilience and tick box option to switch between pvp & non pvp
		So now it auto collects both types and stores them this avoids wiping database switching between two options

v9.20 - Added code to ignore calcs during combat

v9.30 - Fix for patch 2.4 requiring an inspect before you can get players items

v9.31 - Fix nil error line 383

v9.40 - Changed InspectUnit to NotifyInspect

v9.41 - Fixed /iop party & /iop raid not giving correct inspections

v9.50 - Added inspect for Opposite faction

v9.51 - REALLY fixed /iop party & /iop raid - DOH!!

v9.60 - Fixed detection of no enchant on "Armor" because it was looking for an enchant on "ARMOUR"!!! 
			WTB enGB localisation from API calls.
		Localised some msg strings
		Removed onload message in chat window as thats out of fashion
		
v9.61 - Additional check to disable storing scores if opposite faction player cannot be scanned

v9.62 - Fixed nil error

v9.63 - Added new artifact colour for IOP above 1000!!
        Fixed display issue for >1000 scores
		
v9.64 - Changed ignore scan value to work with non level 70s

v10.00 - Changed algorithms to cater for higher item levels for WotLK
         Altered BC algorithm formula from ilvl - plvl - 4*(plvl-60)
	 
v10.01 - Altered Rare/Epic ratio bonuses to new wiki values
         Altered slot item values to new wiki values	 
		 Altered weightings - because the new wiki values have a wider range roughly
			600 IOP will be roughly the same as you get further from 600 your score 
			will be slightly higher or slightly lower than before. Having a carrot on stick
			or similar item equipped will now be radically different score now though.

v10.02 - Fixed Options panel crash - Blizzard had renamed generic I18N minutes variable

v10.03 - Fix formula for WotLK

v10.04 - Changed transition levels to give players a level or two to get kit before new formulae kicks in

v10.05 - Added dependency for AceOO-2.0 - thanks to ShaggyForester for spotting this one

--]]

local INVENTORYONPAR_VERSION = 10.05;

local ItemBonusLib = AceLibrary("ItemBonusLib-1.0");

--------------------------------------------------------------------------------------------------
-- Local variables
--------------------------------------------------------------------------------------------------

colourRed    = "ffff0000";	-- red = 5 lvls above char lvl or better
colourOrange = "ffff8000";	-- orange = 2 to 4 lvls above char lvl
colourYellow = "ffffff00";	-- yellow = 1 lvl below to 1 lvl above char lvl
colourGreen  = "ff1eff00";	-- green = 2 to 4 lvls below char lvl
colourGrey   = "ff9d9d9d";	-- gray = 5 lvls below char lvl or worse
colourBlue   = "ff20d0ff";	-- unknown, light blue
colourWhite  = "ffffffff";	-- slot empty - white
colourNone   = "ff808080";  -- default border colour for paper doll

colourDarkBlue = "ff0070dd";   -- item ITEM_RARITY dark blue
colourPurple   = "ffa335ee";   -- item ITEM_RARITY purple
colourGold     = "ffe6cc80";  -- item ITEM_RARITY gold

IOP_reweighting = 1.33; -- value to modify overall score to be similar between versions of formulae
--------------------------------------------------------------------------------------------------
-- Event functions
--------------------------------------------------------------------------------------------------

function InventoryOnPar_Initialise()
	ITEM_RARITY = {};
	ITEM_RARITY[7]={ name=IOP_UNKNOWN, colour=colourBlue };
	ITEM_RARITY[0]={ name=IOP_POOR, colour=colourGrey };
	ITEM_RARITY[1]={ name=IOP_COMMON, colour=colourWhite };
	ITEM_RARITY[2]={ name=IOP_UNCOMMON, colour=colourGreen };
	ITEM_RARITY[3]={ name=IOP_RARE, colour=colourDarkBlue };
	ITEM_RARITY[4]={ name=IOP_EPIC, colour=colourPurple };
	ITEM_RARITY[5]={ name=IOP_LEGENDARY, colour=colourOrange };
	ITEM_RARITY[6]={ name=IOP_ARTIFACT, colour=colourGold };
	IOP.thisRealm = GetRealmName();
	if (IOP.Data == nil) then
		IOP.Data = {};
		IOP.Data.lastUpdated = time();
	end
	if(IOP.Data.version == nil or IOP.Data.version ~= 10.0) then
		IOP.Data[IOP.thisRealm] = {}; -- zap all prior history if not current version
		IOP.Data.version = 10.0;
	end
	if(IOP.Data[IOP.thisRealm]== nil) then
		IOP.Data[IOP.thisRealm] = {};
	end
	if(IOP.Options.CleanupTime == nil) then
		IOP.Options.CleanupTime = 60*60*24*14;
	end
	-- Register our slash command
	SlashCmdList["INVENTORYONPAR"] = function(msg)
		InventoryOnPar_SlashCommandHandler(msg);
	end
	SLASH_INVENTORYONPAR1 = "/iop";

	local _, _, _, buildToc = GetBuildInfo();
	-- DEFAULT_CHAT_FRAME:AddMessage("IOP : buildToc "..buildToc);
	IOP.Data.BCtransitionLevel = 60;  -- set level we expect player have BC gear to be 60 and above
	IOP.Data.WotLKtransitionLevel = 71; -- set level we expect player to have WotLK gear to be 71 and above
	IOP.Data.isEnchanter = IOP_IsEnchanter();
	InventoryOnParOption.InitializeOptions();
	IOP_HookTooltips();
	InventoryOnParFrame:Hide();
end

function InventoryOnPar_OnLoad()
	-- setup data block
	if (IOP == nil) then
		IOP = {};
		IOP.Options = {};
		IOP.Data = {};
	end
	this:RegisterEvent("VARIABLES_LOADED");
	this:RegisterEvent("PLAYER_TARGET_CHANGED");
	this:RegisterEvent("UNIT_INVENTORY_CHANGED");
	this:RegisterEvent("PLAYER_LEVEL_UP");
end

function InventoryOnPar_OnEvent()
	if (event == "VARIABLES_LOADED") then
		InventoryOnPar_Initialise();
	elseif (event == "UNIT_INVENTORY_CHANGED" or event == "PLAYER_LEVEL_UP") then
		if (IOP_PaperDollFrame:IsVisible()) then
			IOP_ShowPlayerDoll();
		else
			IOP_RequestTargetScore("player",1);
		end	
	elseif (event == "PLAYER_TARGET_CHANGED" and not InCombatLockdown()) then
		-- process target's score if not in combat
		IOP.currentTarget = nil;
		if (IOP.Options.scanPlayers == 1) then
			if (UnitExists("target") and UnitIsPlayer("target")) then
				if (CheckInteractDistance("target", 1)) then
					IOP_RequestTargetScore("target", nil);
				end
			end
		end
	end
end

function InventoryOnPar_OnHide()
	getglobal("InventoryOnParFrame"):Hide();
end
--------------------------------------------------------------------------------------------------
-- Other Player functions
--------------------------------------------------------------------------------------------------

function IOP_RequestTargetScore(unit, override)
	if (UnitExists(unit) and UnitIsPlayer(unit)) then
		if (UnitLevel(unit) >= IOP.Options.minLevel and UnitLevel(unit) <= IOP.Options.maxLevel) then
			local playerName = UnitName(unit);
			if (override or (not InCombatLockdown() and IOP_DoesPlayerNeedUpdate(playerName))) then
				-- only update if not in combat
				NotifyInspect(unit) -- fix for patch 2.4 requiring an inspect first
				IOP.currentTarget = {};
				IOP.currentTarget.realmName = IOP.thisRealm;
				IOP.currentTarget.playerName = playerName;
				IOP.currentTarget.playerLevel = UnitLevel(unit);
				IOP.currentTarget.class = UnitClass(unit);
				IOP.currentTarget.gender = UnitSex(unit);
				IOP.currentTarget.race = UnitRace(unit);
				IOP.currentTarget.guild = GetGuildInfo(unit);
				IOP.currentTarget.faction = UnitFactionGroup(unit);
				if (IOP.currentTarget.faction == UnitFactionGroup("player")) then
					IOP.currentTarget.friendly = true;
				else
					IOP.currentTarget.friendly = false;
				end
				if (IOP.currentTarget.guild == nil) then
					IOP.currentTarget.guild = "-No Guild Recorded-";
				end
				IOP.currentTarget.itemList = {};
				IOP.currentTarget.tooltipText, IOP.currentTarget.IOP_Score, IOP.currentTarget.itemList, IOP.currentTarget.pvpScore = IOP_GetItemLevels(unit);
				IOP.currentTarget.recordedTime = time();
				local chatColour = "";
				if(IOP.currentTarget.friendly == false) then
					chatColour = "|c"..colourRed;
				end
				local noScan = -11.5133 * IOP_reweighting * UnitLevel(unit); -- 11.5133 is sum of slot weights
				if (IOP.currentTarget.IOP_Score <= noScan) then 
					-- scan failed - usually inspect blocked so don't save data
					DEFAULT_CHAT_FRAME:AddMessage(chatColour.."InventoryOnPar failed to scan "..IOP.currentTarget.playerName);
				else
					IOP_AddPlayerRecord(IOP.currentTarget);
					IOP.Data.lastUpdated = time();
					if (playerName ~= UnitName("player") and IOP.Options.noChatText == 0) then -- now that we update on item change dont announce to chat if unit is player or no chat option enabled
						if(IOP.Options.pvpValues == 1) then
							DEFAULT_CHAT_FRAME:AddMessage(chatColour.."InventoryOnPar captured Par Score of "..format("%.2f", IOP.currentTarget.pvpScore).." for "..IOP.currentTarget.playerName);
						else
							DEFAULT_CHAT_FRAME:AddMessage(chatColour.."InventoryOnPar captured Par Score of "..format("%.2f", IOP.currentTarget.IOP_Score).." for "..IOP.currentTarget.playerName);
						end
					end
					if(InventoryOnParUIFrame:IsVisible()) then
						IOP_UI_BuildTableIndex();
					end
				end
			end
		end
	end
end

function IOP_DoesPlayerNeedUpdate(playerName)
	local playerInfo = IOP_GetPlayerRecord(playerName);
	local needsUpdate = false;

	if (playerInfo ~= nil) then
		local currentTime = time();
		if (currentTime - playerInfo.recordedTime > IOP.Options.minutesTilUpdate * 60) then
			needsUpdate = true;
		end
	else
		needsUpdate = true;
	end

	return needsUpdate;
end

function IOP_GetPlayerRecord(playerName)
	-- will return player record
	if (playerName == nil) then
		return nil;
	end
	local record = IOP.Data[IOP.thisRealm][playerName];
	if (record) then
		if (record.guild == nil) then 
			record.guild = IOP_NOGUILD;
		end
	end
	return record;
end

function IOP_AddPlayerRecord(playerRecord)
	-- will check if player exists if not adds player to array.
	if (playerRecord ~= nil) then
		if(IOP.Data[IOP.thisRealm][playerRecord.playerName] == nil) then
			IOP.Data[IOP.thisRealm][playerRecord.playerName] = {};
		end
		IOP.Data[IOP.thisRealm][playerRecord.playerName] = playerRecord;
		IOP.Data.lastUpdated = time();
	end
end

function IOP_AddCurrentPlayer()
	IOP.currentTarget = {};
	IOP.currentTarget.realmName = IOP.thisRealm;
	IOP.currentTarget.playerName = UnitName("player");
	IOP.currentTarget.playerLevel = UnitLevel("player");
	IOP.currentTarget.guild = GetGuildInfo("player");
	IOP.currentTarget.class = UnitClass("player");
	IOP.currentTarget.gender = UnitSex("player");
	IOP.currentTarget.race = UnitRace("player");
	IOP.currentTarget.itemList = {};
	IOP.currentTarget.tooltipText, IOP.currentTarget.IOP_Score, IOP.currentTarget.itemList, IOP.currentTarget.pvpScore = IOP_GetItemLevels("player");
	IOP.currentTarget.recordedTime = time();
	IOP_AddPlayerRecord(IOP.currentTarget);
	IOP_UI_tableIndexLastUpdated = time();
end

function IOP_GetPlayers()
	return IOP.Data[IOP.thisRealm];
end

function IOP_HasUpdates(atime)
	if (atime and IOP.Data.lastUpdated) then
		if (atime > IOP.Data.lastUpdated) then
			return true;
		end
	end
	return nil;
end

--------------------------------------------------------------------------------------------------
-- API functions
--------------------------------------------------------------------------------------------------

function IOP_GetItemLevels(unitName)
	local iopValue, pvpValue, iopTotal, pvpTotal, unitLevel, kitLevel, itemList, itemScore
	iopTotal = 0;
	pvpTotal = 0;
	bcAdjustment = 0;
	itemList = {};
	unitLevel = UnitLevel(unitName);
	kitLevel = IOP_GetAverageItemLevel(unitName);
	NotifyInspect(unitName);
	textstring="|c"..colourWhite..IOP_TOOLTIPHEADER1..UnitLevel(unitName)..IOP_TOOLTIPHEADER2..kitLevel..")\n";
	iopTwoHanded = "false";
	for index in ipairs(INVENTORY_SLOT_LIST) do 
		local slotName = INVENTORY_SLOT_LIST[index].name;
		INVENTORY_SLOT_LIST[index].id = GetInventorySlotInfo(slotName);
		local slotLink = GetInventoryItemLink(unitName, INVENTORY_SLOT_LIST[index].id);
		if (slotLink ~= nil) then
			local itemName, itemString, itemRarity, itemLevel, itemMinLevel, itemType, itemSubType = GetItemInfo(slotLink);
			if (itemName == nil) then
				itemName = IOP_UNKNOWN;
			end
			if (itemRarity == nil) then
				itemRarity = 0;
			end
			if (itemLevel == nil) then
				itemLevel = 0;
			end
			itemList[slotName] = {};
			itemList[slotName].slotLink = slotLink;
			local levelColour = IOP_GetLevelColour(itemLevel, unitLevel, kitLevel);
			textstring=textstring.."\n|c"..levelColour..INVENTORY_SLOT_LIST[index].desc.." - |c"..levelColour.." lvl:"..itemLevel.." - "..itemName;
			itemList[slotName].levelColour = levelColour;
			itemList[slotName].itemRarity = itemRarity;
			if(ITEM_RARITY[itemRarity].name) then
				textstring=textstring.."  |c"..ITEM_RARITY[itemRarity].colour.."["..ITEM_RARITY[itemRarity].name.."]";
			end
			iopValue = 0;
			pvpValue = 0;
			local levelOK = IOP_CheckLevel(itemLevel);
			if(levelOK) then
				local wikiBonus = IOP_WikiRarityBonus(itemLevel,itemRarity);
				local gemBonus, missingGemsString = IOP_GemBonus(slotLink, index);
				if (unitLevel <= IOP.Data.BCtransitionLevel) then
					itemScore = itemLevel - unitLevel + wikiBonus + gemBonus;
				elseif (unitLevel > IOP.Data.BCtransitionLevel and unitLevel <= IOP.Data.WotLKtransitionLevel) then
					--itemScore = itemLevel - unitLevel - 4 * (unitLevel - 60) + wikiBonus + gemBonus; -- for Burning Crusade kit subtract 4 levels per level > 60
					itemScore = itemLevel  - 3 * (unitLevel - 64) - 90 + wikiBonus + gemBonus; -- for Burning Crusade kit subtract 3 levels per level > 60 : itemlevel = 3*(reqLvl-60)+90 : use 64 to give 4 levels to get reqLvl gear
				elseif (unitLevel > IOP.Data.WotLKtransitionLevel) then
					itemScore = itemLevel  - 4 * (unitLevel - 70) - 130 + wikiBonus + gemBonus; -- for Wrath of the Litch King kit subtract 4 levels per level > 70 : itemlevel = 4*(reqLvl-67)+130 : use 70 to give 3 levels to get reqLvl gear
				end
				local info = ItemBonusLib:ScanItemLink(slotLink, true);
				local resilienceScore = info.bonuses["CR_RESILIENCE"];
				if (resilienceScore == nil) then
					resilienceScore = 0;
				end
				resilienceScore = 0.94 * resilienceScore; -- added to give a small (6%) weighting to the crit reduction value of resilience in PvE (based on 1.5 defence = 25 resilience).
				itemScore = itemScore - resilienceScore;
				if(itemSubType == IOP_TH_MACE or itemSubType == IOP_TH_SWORD or itemSubType == IOP_TH_AXE or itemSubType == IOP_STAVES or itemSubType == IOP_POLEARMS) then
					-- compensate for two handed weapon by doubling score
					iopValue = itemScore  * IOP_reweighting; -- use 1.00 as two handed weight and reweight values to similar score to before 
					pvpValue = (itemScore + resilienceScore) * IOP_reweighting;
					iopTwoHanded = "true";
				else
					iopValue = itemScore  * INVENTORY_SLOT_LIST[index].weight * IOP_reweighting; -- reweights values to similar score to before
					pvpValue = (itemScore + resilienceScore)  * INVENTORY_SLOT_LIST[index].weight * IOP_reweighting; -- reweights values to similar score to before
				end
				iopTotal = iopTotal + iopValue;
				pvpTotal = pvpTotal + pvpValue;
				if(IOP.Options.pvpValues == 1) then
					textstring=textstring.."  ("..format("%.2f", pvpValue)..")"..missingGemsString;
				else
					textstring=textstring.."  ("..format("%.2f", iopValue)..")"..missingGemsString;
				end
			end
			textstring=textstring..IOP_MissingEnchant(index,itemString,itemType,itemSubType);
			itemList[slotName].itemScore = iopValue;
		else
			textstring=textstring.."\n|c"..colourWhite..INVENTORY_SLOT_LIST[index].desc.." - |c"..colourWhite..IOP_SLOTEMPTY;
			if(slotName == "SecondaryHandSlot" and iopTwoHanded == "true") then
				-- make no adjustment if using two handed weapon
				iopValue = 0;
				pvpValue = 0;
			elseif(IOP_TooLowLevel(index, UnitLevel(unitName), UnitClass(unitName))) then
				iopValue = 0;
				pvpValue = 0;
			else
				iopValue = - UnitLevel(unitName) * INVENTORY_SLOT_LIST[index].weight * IOP_reweighting; -- reweights values to similar score to before
				pvpValue = iopValue;
				iopTotal = iopTotal + iopValue;
				pvpTotal = pvpTotal + pvpValue;
			end
			if(IOP.Options.pvpValues == 1) then
				textstring=textstring.."  ("..format("%.2f", pvpValue)..")";
			else
				textstring=textstring.."  ("..format("%.2f", iopValue)..")";
			end
		end
	end
	if(IOP.Options.pvpValues == 1) then
		textstring = textstring.."\n\n|c"..colourWhite..IOP_SCORETEXT..format("%.2f", pvpTotal);
	else
		textstring = textstring.."\n\n|c"..colourWhite..IOP_SCORETEXT..format("%.2f", iopTotal);
	end
	return textstring, iopTotal, itemList, pvpTotal;
end

function IOP_DEBUG(info)
	for bonus, value in pairs(info.bonuses) do
		DEFAULT_CHAT_FRAME:AddMessage("IOP DEBUG : "..bonus.." - "..ItemBonusLib:GetBonusFriendlyName(bonus).." : "..value)
	end
end

function IOP_WikiRarityBonus(itemLevel, itemRarity) 
	local wikiValue = 0;
	local greenValue = (itemLevel - 8) / 2;
	if(itemRarity < 3) then
		wikiValue = greenValue;
	elseif(itemRarity == 3) then     -- this is blue level 
		wikiValue = (itemLevel - 0.75) / 1.8;
	elseif(itemRarity == 4) then -- this is epic level 
		wikiValue = (itemLevel - 26) / 1.2;
	elseif(itemRarity == 5) then -- this is legendary level
		wikiValue = itemLevel;
	elseif(itemRarity == 6) then -- this is artifact leveld
		wikiValue = (itemLevel + 0.9) / 0.9; -- my guess at artifact level adjustment
	end
	wikiValue = wikiValue - greenValue;
	return wikiValue;
end

function IOP_GemBonus(itemLink, slotId)
	local gemBonus = 0;
	local gemSlotCost = 3;
	local gemName, gemLink, gemType;
	local index;
	for index = 1,3 do
		gemName, gemLink = GetItemGem(itemLink,index);
		if(gemName) then
			local _, _, gemRarity, _, _, _, gemType = GetItemInfo(gemLink);
--			DEFAULT_CHAT_FRAME:AddMessage("IOP: Found gem "..gemName.." in slot "..slotname.." rarity "..gemRarity.." gemtype "..gemType);
			if(gemRarity == 2) then -- this is green quality gem
				gemBonus = gemBonus + 6 - gemSlotCost;
			elseif(gemRarity == 3) then -- this is blue quality gem
				gemBonus = gemBonus + 8 - gemSlotCost;
			elseif(gemRarity == 4) then -- this is epic quality gem
				gemBonus = gemBonus + 10 - gemSlotCost;
			end
			if(gemType == "Meta") then
				gemBonus = gemBonus + 12 - gemSlotCost; -- add 12 item levels for meta gem equipped
			end
		end
	end
	return gemBonus, "";
end

-- This routine attempts to find out how many sockets an item has
function IOP_GemBonusNew(itemLink, slotId)
	local gemBonus = 0;
	local gemSlotCost = 3;
	local gemName, gemLink, gemType;
	local missingGemsString = "";
	HideUIPanel(ItemSocketingFrame);
	UIParent:UnregisterEvent("SOCKET_INFO_UPDATE");
	SocketInventoryItem(INVENTORY_SLOT_LIST[slotId].id);
	if GetSocketItemInfo() then
		local numGems = GetNumSockets();
		DEFAULT_CHAT_FRAME:AddMessage("IOP : in IOP_GemBonus numGems for "..INVENTORY_SLOT_LIST[slotId].name.."("..INVENTORY_SLOT_LIST[slotId].id..") is "..numGems.." - "..GetNumSockets(itemLink));
		if(numGems > 0) then
			local socketedGems = 0;
			for index = 1,3 do
				gemName, gemLink = GetItemGem(itemLink,index);
				if(gemName) then
					socketedGems = socketedGems + 1;
					local _, _, gemRarity, _, _, _, gemType = GetItemInfo(gemLink);
					if(gemRarity == 2) then -- this is green quality gem
						gemBonus = gemBonus + 6;
					elseif(gemRarity == 3) then -- this is blue quality gem
						gemBonus = gemBonus + 8;
					elseif(gemRarity == 4) then -- this is epic quality gem
						gemBonus = gemBonus + 10;
					end
					if(gemType == "Meta") then
						gemBonus = gemBonus + 12; -- add 12 item levels for meta gem equipped
					end
				end
			end
			gemBonus = gemBonus - gemSlotCost * numGems;
			if(socketedGems ~= numGems) then
				missingGemsString = "|c"..colourRed.." - Missing Gems";
			end
			DEFAULT_CHAT_FRAME:AddMessage("IOP : numGems "..numGems.." socketedGems "..socketedGems.." string :"..missingGemsString..":");
		end
	end
	UIParent:RegisterEvent("SOCKET_INFO_UPDATE");
	return gemBonus, missingGemsString;
end

function IOP_MissingEnchant(slotindex,itemString,itemType,itemSubType)
	local enchantText = "";
	if (itemString) then
		-- now add text if item is enchanted
		local _, _, enchantId = strsplit(":", itemString)
		local noEnchant = false;
		if(enchantId == nil or enchantId == "0") then
			if (INVENTORY_SLOT_LIST[slotindex].enchant == 1) then
				noEnchant = true;
			elseif (INVENTORY_SLOT_LIST[slotindex].enchant == 2) then
				-- rings can only be enchanted by enchanter - need to work out if player is enchanter!!!
				if(IOP.Data.isEnchanter == true) then
					noEnchant = true;
				end
			elseif (INVENTORY_SLOT_LIST[slotindex].enchant == 3) then
				-- off-hand/ranged slot only if item can be enchanted ie: not misc, thrown, wand, relic, libram or totem
				if (itemType == IOP_WEAPON and itemSubType ~= IOP_MISC and itemSubType ~= IOP_THROWN and itemSubType ~= IOP_WANDS) then 
					noEnchant = true;
				elseif (itemType == IOP_ARMOUR and itemSubType ~= IOP_IDOL and itemSubType ~= IOP_TOTEM and itemSubType ~= IOP_LIBRAM) then 
					noEnchant = true;
				end
			end
			if (noEnchant == true) then
				enchantText = "|c"..colourRed.." - "..IOP_NOENCHANT;
			end
		end
	end
	return enchantText;
end

function IOP_IsEnchanter()
	for skillIndex = 1, GetNumSkillLines() do
		local skillName = GetSkillLineInfo(skillIndex);
		if (skillName == IOP_ENCHANTING) then
			return true;
		end
	end
	return false;
end

function IOP_CheckLevel(itemLevel)
	if (itemLevel == IOP_NOTFOUND) then
		return nil;
	elseif (type(itemLevel) == "number") then
		if(itemLevel == 0 or itemLevel == 1) then
			return nil;
		end
	elseif (type(itemLevel) == "string") then
		if(itemLevel =="0" or itemLevel == "1") then
			return nil;
		end
	end
	return "itemok";
end

function IOP_TooLowLevel(index, unitLevel, unitClass)
	--DEFAULT_CHAT_FRAME:AddMessage("IOP : Checking slot "..INVENTORY_SLOT_LIST[index].name.." class :"..unitClass.." level :"..unitLevel);
	if (unitLevel >= 60) then
		return nil; -- this unit is level 60+ no reason for empty slot
	end
	if (INVENTORY_SLOT_LIST[index].name == "RangedSlot" and unitLevel < 60) then
		-- paladins, shamans & druids need to be level 52 to use Libram, Totem or Idol - give them to lvl 60 to get something before penalty
		if(unitClass == "Paladin" or unitClass == "Shaman" or unitClass == "Druid") then
			return 1;
		elseif(unitLevel < 10) then
			if (unitClass == "Mage" or unitClass == "Priest" or unitClass == "Warlock") then
				return 1; -- min wand level is 7 give them to level 10 to get something
			else
				return nil; -- no reason for under lvl 7 other class to have this slot empty
			end
		else
			return nil; -- no reason for level >=7 to have this slot empty
		end
	end
	if (INVENTORY_SLOT_LIST[index].name == "SecondaryHandSlot" and unitLevel <5) then
		if (unitClass == "Warrior" or unitClass == "Paladin" or unitClass == "Shaman") then
			return nil; -- no reason for these classes to have no shield
		else
			return 1; -- other classes must be level 5+ to use off-hand item
		end
	end
	if (INVENTORY_SLOT_LIST[index].minLevel > unitLevel) then
		return 1; -- item slot is higher than unit level so they cant use slot
	else
		return nil; -- default no reason for slot to be empty
	end
end

function IOP_GetAverageItemLevel(unitName)
	local totalLevels = 0;
	local itemCount = 0;
	if(unitName == nil) then
		unitName="player";
	end
	for index in ipairs(INVENTORY_SLOT_LIST) do 
		INVENTORY_SLOT_LIST[index].id = GetInventorySlotInfo(INVENTORY_SLOT_LIST[index].name);
		local slotLink = GetInventoryItemLink(unitName, INVENTORY_SLOT_LIST[index].id);
		if (slotLink ~= nil) then
			local itemName, _, itemRarity, itemLevel, itemMinLevel, itemType, itemSubType = GetItemInfo(slotLink);
			if(itemLevel and itemLevel ~= 0) then
				totalLevels = totalLevels + itemLevel;
				itemCount = itemCount +1;
			end
		end
	end
	if (itemCount > 0) then
		return math.floor(totalLevels / itemCount);
	else
		return 0;
	end
end

function IOP_ShowItemLevels()
	InventoryOnParText:SetJustifyH("LEFT");
	InventoryOnParText:SetJustifyV("MIDDLE");
	InventoryOnParText:SetText(IOP_GetItemLevels("player"));
	InventoryOnParFrame:Show();
end

function IOP_ShowPlayerData()
	if(IOP.Data[IOP.thisRealm] ~= nil) then
		for index, player in pairs(IOP.Data[IOP.thisRealm]) do
			DEFAULT_CHAT_FRAME:AddMessage(player.playerName.." a lvl "..player.playerLevel.." "..player.class.." on "..date(IOP.Options.dateFormat, player.recordedTime).." had par of : "..player.IOP_Score);
		end
	else
		DEFAULT_CHAT_FRAME:AddMessage("No data recorded for "..IOP.thisRealm.." realm yet.");
	end
end

function IOP_ScanMembers(partyType)
	if(InCombatLockdown()) then
		DEFAULT_CHAT_FRAME:AddMessage("Cannot check scores in combat.");
	else
		local numberInParty = 0;
		if(partyType=="party") then
			numberInParty = GetNumPartyMembers();
		elseif(partyType=="raid") then
			numberInParty = GetNumRaidMembers();
		end
		if (numberInParty==0) then
			DEFAULT_CHAT_FRAME:AddMessage("[IOP Scan] Error you are not in a "..partyType,1,0,0);
		else
			for i=1, numberInParty, 1 do
				if (CheckInteractDistance(partyType..i, 1)) then
					IOP_RequestTargetScore(partyType..i, 1);
				else
					DEFAULT_CHAT_FRAME:AddMessage(UnitName(partyType..i).." is out of inspect range.");
				end
			end
		end
	end
end

function IOP_ShowPvPStatus()
	if (IOP.Options.pvpValues == 0) then
		DEFAULT_CHAT_FRAME:AddMessage("IOP : Resilience is currently IGNORED in displayed calculation.");
	else
		DEFAULT_CHAT_FRAME:AddMessage("IOP : Resilience is currently INCLUDED in displayed calculation.");
	end
end

function IOP_DebugItemIDs()
	for index in ipairs(INVENTORY_SLOT_LIST) do 
		INVENTORY_SLOT_LIST[index].id = GetInventorySlotInfo(INVENTORY_SLOT_LIST[index].name);
		local slotLink = GetInventoryItemLink("player", INVENTORY_SLOT_LIST[index].id);
		if (slotLink ~= nil) then
			local itemName = GetItemInfo(slotLink);
			DEFAULT_CHAT_FRAME:AddMessage(INVENTORY_SLOT_LIST[index].name.." - "..slotLink.." : "..itemName);
		end
	end
end

--------------------------------------------------------------------------------------------------
-- Handler functions
--------------------------------------------------------------------------------------------------

function InventoryOnPar_SlashCommandHandler(msg)
	if(msg == "version") then
		DEFAULT_CHAT_FRAME:AddMessage("InventoryOnPar Version : "..INVENTORYONPAR_VERSION);
	elseif(msg == "show") then
		ShowUIPanel(InventoryOnParUIFrame);
	elseif(msg == "data") then
		IOP_ShowItemLevels();
	elseif(msg == "options") then
		ShowUIPanel(InventoryOnParOptionFrame)
	elseif(msg == "doll") then 
		IOP_ShowPlayerDoll();
	elseif(msg == "debug") then
		IOP_DebugItemIDs();
	elseif(msg == "raid") then
		IOP_ScanMembers("raid");
	elseif(msg == "party") then
		IOP_ScanMembers("party");
	elseif(msg == "pvp status") then
		IOP_ShowPvPStatus();
	else
		DEFAULT_CHAT_FRAME:AddMessage("InventoryOnPar Commands");
		DEFAULT_CHAT_FRAME:AddMessage("/iop version - shows version information");
		DEFAULT_CHAT_FRAME:AddMessage("/iop show - displays other player data collected");
		DEFAULT_CHAT_FRAME:AddMessage("/iop data - displays current character data");
		DEFAULT_CHAT_FRAME:AddMessage("/iop doll - displays current character sheet");
		DEFAULT_CHAT_FRAME:AddMessage("/iop options - displays options setup panel");
		DEFAULT_CHAT_FRAME:AddMessage("/iop party - scans all party members in inspect range");
		DEFAULT_CHAT_FRAME:AddMessage("/iop raid - scans all raid members in inspect range");
		DEFAULT_CHAT_FRAME:AddMessage("/iop pvp status - shows whether resilience is included in calculation");
	end
end

function IOP_HookTooltips()
	-- hook the item tooltip
	IOP_GameTooltip_Orig = GameTooltip:GetScript("OnShow");
	GameTooltip:SetScript("OnShow", IOP_GameTooltip_OnShow);
	IOP_ItemRefTooltip_Orig = ItemRefTooltip:GetScript("OnShow");
	ItemRefTooltip:SetScript("OnShow", IOP_ItemRefTooltip_OnShow);
	if (LootLinkTooltip) then
		IOP_LootLinkTooltip_Orig = LootLinkTooltip:GetScript("OnShow");
		LootLinkTooltip:SetScript("OnShow", IOP_LootLinkTooltip_OnShow);
	end
	-- now check if EquipCompare Comparison tooltips loaded
	if (ComparisonTooltip1) then
		IOP_ComparisonTooltip1_Orig = ComparisonTooltip1:GetScript("OnShow");
		ComparisonTooltip1:SetScript("OnShow", IOP_ComparisonTooltip1_OnShow);
	end
	if (ComparisonTooltip2) then
		IOP_ComparisonTooltip2_Orig = ComparisonTooltip2:GetScript("OnShow");
		ComparisonTooltip2:SetScript("OnShow", IOP_ComparisonTooltip2_OnShow);
	end
	if (AtlasLootTooltip) then
		IOP_AtlasLootTooltip_Orig = AtlasLootTooltip:GetScript("OnShow");
		AtlasLootTooltip:SetScript("OnShow", IOP_AtlasLootTooltip_OnShow);
	end
end

function IOP_GameTooltip_OnShow()
	-- call original event for GameTooltip:OnShow()
	if IOP_GameTooltip_Orig then
		IOP_GameTooltip_Orig(event)
	end

	-- add Item Level info to tooltip if the tooltip has at least 1 line
	local IOP_tooltip = getglobal("GameTooltipTextLeft1");
	if (IOP_tooltip) then
		local reqLevel = IOP_GetReqLevel("GameTooltip");
		local itemLevel, levelColour = IOP_GetItemLevelByName(GameTooltip);
		if (levelColour ~= colourBlue and itemLevel ~= nil) then
			-- suppress tooltip line if unknown
			GameTooltip:AddLine("|c"..levelColour.."Item Level : "..itemLevel);
			GameTooltip:Show()
		end
	else
		DEFAULT_CHAT_FRAME:AddMessage("|c00bfffff hmm it is null");
	end
end

function IOP_ItemRefTooltip_OnShow()
	-- call original event for ItemRefTooltip:OnShow()
	if IOP_ItemRefTooltip_Orig then
		IOP_ItemRefTooltip_Orig(event)
	end

	-- add Item Level info to tooltip if the tooltip has at least 1 line
	local IOP_tooltip = getglobal("ItemRefTooltipTextLeft1");
	if (IOP_tooltip) then
		local reqLevel = IOP_GetReqLevel("ItemRefTooltip");
		local itemLevel, levelColour = IOP_GetItemLevelByName(ItemRefTooltip);
		if (levelColour ~= colourBlue and itemLevel ~= nil) then
			-- suppress tooltip line if unknown
			ItemRefTooltip:AddLine("|c"..levelColour.."Item Level : "..itemLevel);
			ItemRefTooltip:Show()
		end
	end
end

function IOP_LootLinkTooltip_OnShow()
	-- call original event for LootLinkTooltip:OnShow()
	if IOP_LootLinkTooltip_Orig then
		IOP_LootLinkTooltip_Orig(event)
	end

	-- add Item Level info to tooltip if the tooltip has at least 1 line
	local IOP_tooltip = getglobal("LootLinkTooltipTextLeft1");
	if (IOP_tooltip) then
		local reqLevel = IOP_GetReqLevel("LootLinkTooltip");
		local itemLevel, levelColour = IOP_GetItemLevelByName(LootLinkTooltip);
		if (levelColour ~= colourBlue and itemLevel ~= nil) then
			-- suppress tooltip line if unknown
			LootLinkTooltip:AddLine("|c"..levelColour.."Item Level : "..itemLevel);
			LootLinkTooltip:Show()
		end
	end
end

function IOP_ComparisonTooltip1_OnShow()
	-- call original event for ComparisonTooltip1TextLeft1:OnShow()
	if IOP_ComparisonTooltip1_Orig then
		IOP_ComparisonTooltip1_Orig(event)
	end

	-- add Item Level info to tooltip if the tooltip has at least 1 line
	local IOP_tooltip = getglobal("ComparisonTooltip1TextLeft1");
	if (IOP_tooltip) then
		local reqLevel = IOP_GetReqLevel("ComparisonTooltip1");
		local itemLevel, levelColour = IOP_GetItemLevelByName(ComparisonTooltip1);
		if (levelColour ~= colourBlue and itemLevel ~= nil) then
			-- suppress tooltip line if unknown
			ComparisonTooltip1:AddLine("|c"..levelColour.."Item Level : "..itemLevel);
			ComparisonTooltip1:Show()
		end
	end
end

function IOP_ComparisonTooltip2_OnShow()
	-- call original event for ComparisonTooltip1TextLeft2:OnShow()
	if IOP_ComparisonTooltip2_Orig then
		IOP_ComparisonTooltip2_Orig(event)
	end

	-- add Item Level info to tooltip if the tooltip has at least 1 line
	local IOP_tooltip = getglobal("ComparisonTooltip2TextLeft1");
	if (IOP_tooltip) then
		local reqLevel = IOP_GetReqLevel("ComparisonTooltip2");
		local itemLevel, levelColour = IOP_GetItemLevelByName(ComparisonTooltip2);
		if (levelColour ~= colourBlue and itemLevel ~= nil) then
			-- suppress tooltip line if unknown
			ComparisonTooltip2:AddLine("|c"..levelColour.."Item Level : "..itemLevel);
			ComparisonTooltip2:Show()
		end
	end
end

function IOP_AtlasLootTooltip_OnShow()
	-- call original event for ComparisonTooltip1TextLeft2:OnShow()
	if IOP_AtlasLootTooltip_Orig then
		IOP_AtlasLootTooltip_Orig(event)
	end

	-- add Item Level info to tooltip if the tooltip has at least 1 line
	local IOP_tooltip = getglobal("AtlasLootTooltipTextLeft1");
	if (IOP_tooltip) then
		local reqLevel = IOP_GetReqLevel("AtlasLootTooltip");
		local itemLevel, levelColour = IOP_GetItemLevelByName(AtlasLootTooltip);
		if (levelColour ~= colourBlue and itemLevel ~= nil) then
			-- suppress tooltip line if unknown
			AtlasLootTooltip:AddLine("|c"..levelColour.."Item Level : "..itemLevel);
			AtlasLootTooltip:Show()
		end
	end
end


function xIOP_ShowTooltip()
	-- add Item Level info to tooltip if the tooltip has at least 1 line
	local IOP_tooltip = getglobal(this:GetName().."TextLeft1");
	if (IOP_tooltip) then
		local reqLevel = IOP_GetReqLevel(this:GetName());
		local itemLevel, levelColour = IOP_GetItemLevelByName(this);
		if (levelColour ~= colourBlue and itemLevel ~= nil) then
			-- suppress tooltip line if unknown
			this:AddLine("|c"..levelColour.."Item Level : "..itemLevel);
			this:Show()
		end
	else
		DEFAULT_CHAT_FRAME:AddMessage("|c00bfffff hmm it is null");
	end
end

function xIOP_Tooltip_OnShow()          ----------------Possible Generic Call - Test it
	local IOP_Tooltip = getglobal("IOP_"..this:GetName().."_Orig");
	if (IOP_Tooltip) then
		IOP_Tooltip(event)
	end
	IOP_ShowTooltip();
end

function xIOP_GameTooltip_OnShow()
	-- call original event for GameTooltip:OnShow()
	if IOP_GameTooltip_Orig then
		IOP_GameTooltip_Orig(event)
	end
	IOP_ShowTooltip();
end

function xIOP_ItemRefTooltip_OnShow()
	-- call original event for ItemRefTooltip:OnShow()
	if IOP_ItemRefTooltip_Orig then
		IOP_ItemRefTooltip_Orig(event)
	end
	IOP_ShowTooltip();
end

function xIOP_LootLinkTooltip_OnShow()
	-- call original event for LootLinkTooltip:OnShow()
	if IOP_LootLinkTooltip_Orig then
		IOP_LootLinkTooltip_Orig(event)
	end
	IOP_ShowTooltip();
end

function xIOP_ComparisonTooltip1_OnShow()
	-- call original event for ComparisonTooltip1TextLeft1:OnShow()
	if IOP_ComparisonTooltip1_Orig then
		IOP_ComparisonTooltip1_Orig(event)
	end
	IOP_ShowTooltip();
end

function xIOP_ComparisonTooltip2_OnShow()
	-- call original event for ComparisonTooltip1TextLeft2:OnShow()
	if IOP_ComparisonTooltip2_Orig then
		IOP_ComparisonTooltip2_Orig(event)
	end
	IOP_ShowTooltip();
end

--------------------------------------------------------------------------------------------------
-- Main functions
--------------------------------------------------------------------------------------------------

function IOP_GetReqLevel(fieldname)
	local iStart, iEnd, value
	for index = 2, 30 do
		field = getglobal(fieldname.."TextLeft"..index);
		if(field and field:IsVisible()) then
			linetext = field:GetText();
			if (linetext) then
				iStart, iEnd, value = string.find(linetext, "Requires Level (%d+)");
				if (value) then
					return value;
				end
			end
		end
	end
	return 0;
end

function IOP_GetItemLevelByName(tooltip)
    local playerLevel, kitLevel
	playerLevel = UnitLevel("player");
	kitLevel = IOP_GetAverageItemLevel("player");
	if (IOP.Options.showLevelInTooltip == 1) then 
		if (tooltip) then
			local _, itemLink = tooltip:GetItem();
			if (itemLink) then
				local _, _, _, itemLevel = GetItemInfo(itemLink);
				local colour = IOP_GetLevelColour(itemLevel, playerLevel, kitLevel);
				return itemLevel, colour;
			else
				return IOP_NOTFOUND, colourBlue;
			end
		end
		return IOP_NOTFOUND, colourBlue;
	end
	return "dont show", colourBlue;
end

function IOP_GetLevelColour(itemLevel, playerLevel, kitLevel)
	local levelDiff, bcAdjustment, colour
    levelDiff = 0;
	bcAdjustment = 0;
	colour = colourWhite;
	if (itemLevel == nil) then
		return 0, colourWhite;
	end
	if (itemLevel == "0" or itemLevel == "-1") then
		return 0, colourWhite;
	end
	if (playerLevel >= 60 and kitLevel >=playerLevel) then -- for level 60+ chars use average kit level for colours
		levelDiff = itemLevel - kitLevel;
	else
		levelDiff = itemLevel - playerLevel;
	end
	if (playerLevel > 60) then
	    -- in Burning Crusade there is wide differences in item level 
		-- so this will scale the colour bands in blocks of 5 levels
		-- rather than the pre lvl 60 scale per level
		levelDiff = floor(levelDiff / 5);
	end
	if (levelDiff >= 5) then
		colour = colourRed;
	end
	if (levelDiff <= 4 and levelDiff >=2) then
		colour = colourOrange;
	end
	if (levelDiff <= 1 and levelDiff >=-1) then
		colour = colourYellow;
	end
	if (levelDiff <= -2 and levelDiff >=-4) then
		colour = colourGreen;
	end
	if (levelDiff <= -5) then
		colour = colourGrey;
	end
	return colour;
end
