﻿--[[
Name: IncomingHealsLib-1.0
Revision: $Revision: 80013 $
Author: Tattersail (joep.moritz@wur.nl)
Website:
Documentation:
SVN:
Description: Communicates healing spells to the raid (and resses too!)
Dependencies: Ace2, Babble-2.2, CompostLib, GratuityLib, RosterLib, SpecialEventsEmbed

TODO: Fix party heals + target buffs like Sanctity Aura and Fel Armor
]]

local MAJOR_VERSION, MINOR_VERSION = "1.0", "$Revision: 80013 $"
local LIBNAME = "IncomingHealsLib-"..MAJOR_VERSION

if not AceLibrary then error(LIBNAME .. " requires AceLibrary.") end
if not AceLibrary:IsNewVersion(LIBNAME, MINOR_VERSION) then return end
if not AceLibrary:HasInstance("AceEvent-2.0") then error(LIBNAME .. " requires AceEvent-2.0.") end
if not AceLibrary:HasInstance("AceComm-2.0") then error(LIBNAME .. " requires AceComm-2.0.") end
if not AceLibrary:HasInstance("AceLocale-2.2") then error(LIBNAME .. " requires AceLocale-2.2.") end
if not AceLibrary:HasInstance("Babble-Spell-2.2") then error(LIBNAME .. " requires Babble-Spell-2.2.") end
if not AceLibrary:HasInstance("Gratuity-2.0") then error(LIBNAME .. " requires Gratuity-2.0.") end
if not AceLibrary:HasInstance("Roster-2.1") then error(LIBNAME .. " requires Roster-2.1.") end
if not AceLibrary:HasInstance("SpecialEvents-Aura-2.0") then error(LIBNAME .. " requires SpecialEvents-Aura-2.0.") end

local L = AceLibrary("AceLocale-2.2"):new("IncomingHealsLib")

L:RegisterTranslations("enUS", function() return {

  ["tooltip_regexp_hot"] = "for (%d+) over (%d+) sec",
  ["tooltip_regexp_direct"] = "for (%d+) to (%d+)",
  ["tooltip_regexp_direct_hot"] = "for (%d+) to (%d+) and another (%d+) over (%d+) sec",
  ["tooltip_regexp_hot_direct"] = "for (%d+) over (%d+) sec.+heals themself for (%d)",
  ["tooltip_regexp_channel"] = "for (%d+) to (%d+) every (%d+) seconds for (%d+) sec",
  ["tooltip_regexp_swiftmend"] = "",
  ["tooltip_regexp_resurrect"] = "",
  ["tooltip_regexp_instant_cast"] = "Instant cast",

  ["tooltip_regexp_hot_arg1"] = "arg1",
  ["tooltip_regexp_hot_arg2"] = "arg2",
  ["tooltip_regexp_direct_hot_arg1"] = "arg1",
  ["tooltip_regexp_direct_hot_arg2"] = "arg2",
  ["tooltip_regexp_direct_hot_arg3"] = "arg3",
  ["tooltip_regexp_direct_hot_arg4"] = "arg4",
  ["tooltip_regexp_hot_direct_arg1"] = "arg1",
  ["tooltip_regexp_hot_direct_arg2"] = "arg2",
  ["tooltip_regexp_hot_direct_arg3"] = "arg3",
  ["tooltip_regexp_channel_arg1"] = "arg1",
  ["tooltip_regexp_channel_arg2"] = "arg2",
  ["tooltip_regexp_channel_arg3"] = "arg3",

} end)

L:RegisterTranslations("deDE", function() return {


  ["tooltip_regexp_hot"] = "(%d+) Sek.+insgesamt (%d+)",
  ["tooltip_regexp_direct"] = "um (%d+) bis (%d+)",
  ["tooltip_regexp_direct_hot"] = "um (%d+) bis (%d+) und \195\188ber (%d+) Sek. um weitere (%d+)",
  ["tooltip_regexp_hot_direct"] = "(%d+) Sek. lang um insgesamt (%d+)+um (%d) geheilt",
--  ["tooltip_regexp_channel"] = "(%d+) Sek. lang alle (%d+) Sek. (%d+) Gesundheit",
  ["tooltip_regexp_channel"] = "",
  ["tooltip_regexp_swiftmend"] = "",
  ["tooltip_regexp_resurrect"] = "",
  ["tooltip_regexp_instant_cast"] = "Sofort",

  ["tooltip_regexp_hot_arg1"] = "arg2",
  ["tooltip_regexp_hot_arg2"] = "arg1",
  ["tooltip_regexp_direct_hot_arg1"] = "arg1",
  ["tooltip_regexp_direct_hot_arg2"] = "arg2",
  ["tooltip_regexp_direct_hot_arg3"] = "arg4",
  ["tooltip_regexp_direct_hot_arg4"] = "arg3",
  ["tooltip_regexp_hot_direct_arg1"] = "arg2",
  ["tooltip_regexp_hot_direct_arg2"] = "arg1",
  ["tooltip_regexp_hot_direct_arg3"] = "arg3",
  ["tooltip_regexp_channel_arg1"] = "arg3",
  ["tooltip_regexp_channel_arg2"] = "arg1",
  ["tooltip_regexp_channel_arg3"] = "arg2",


} end)

L:RegisterTranslations("koKR", function() return {

  ["tooltip_regexp_hot"] = "(%d+)에 걸쳐 (%d+)",
  ["tooltip_regexp_direct"] = "(%d+)~(%d+)",
  ["tooltip_regexp_direct_hot"] = "(%d+)~(%d+)만큼 회복시키고 추가로 (%d+)에 걸쳐 총 (%d+)",
  ["tooltip_regexp_hot_direct"] = "(%d+)에 걸쳐 (%d+). 즉시 (%d)의 생명력을 회복",
  ["tooltip_regexp_channel"] = "(%d+) 동안 매 (%d+)초마다 주위의 모든 파티원의 생명력을 (%d+)~(%d+)",
  ["tooltip_regexp_swiftmend"] = "",
  ["tooltip_regexp_resurrect"] = "",
  ["tooltip_regexp_instant_cast"] = "즉시 시전",

  ["tooltip_regexp_hot_arg1"] = "arg1",
  ["tooltip_regexp_hot_arg2"] = "arg2",
  ["tooltip_regexp_direct_hot_arg1"] = "arg1",
  ["tooltip_regexp_direct_hot_arg2"] = "arg2",
  ["tooltip_regexp_direct_hot_arg3"] = "arg3",
  ["tooltip_regexp_direct_hot_arg4"] = "arg4",
  ["tooltip_regexp_hot_direct_arg1"] = "arg1",
  ["tooltip_regexp_hot_direct_arg2"] = "arg2",
  ["tooltip_regexp_hot_direct_arg3"] = "arg3",
  ["tooltip_regexp_channel_arg1"] = "arg1",
  ["tooltip_regexp_channel_arg2"] = "arg2",
  ["tooltip_regexp_channel_arg3"] = "arg3",

} end)

L:RegisterTranslations("ruRU", function() return {

  ["tooltip_regexp_hot"] = "на (%d+) ед. в течение (%d+) сек",
  ["tooltip_regexp_direct"] = "на (%d+) - (%d+) ед.",
  ["tooltip_regexp_direct_hot"] = "на (%d+) - (%d+) ед. и еще (%d+) в течение (%d+) сек",
  ["tooltip_regexp_hot_direct"] = "на (%d+) ед. в течение (%d+) сек.+heals themself на (%d)",
  ["tooltip_regexp_channel"] = "на (%d+) - (%d+) ед. каждые (%d+) секунд в течение (%d+) сек",
  ["tooltip_regexp_swiftmend"] = "",
  ["tooltip_regexp_resurrect"] = "",
  ["tooltip_regexp_instant_cast"] = "Мнгновенно",

  ["tooltip_regexp_hot_arg1"] = "arg1",
  ["tooltip_regexp_hot_arg2"] = "arg2",
  ["tooltip_regexp_direct_hot_arg1"] = "arg1",
  ["tooltip_regexp_direct_hot_arg2"] = "arg2",
  ["tooltip_regexp_direct_hot_arg3"] = "arg3",
  ["tooltip_regexp_direct_hot_arg4"] = "arg4",
  ["tooltip_regexp_hot_direct_arg1"] = "arg1",
  ["tooltip_regexp_hot_direct_arg2"] = "arg2",
  ["tooltip_regexp_hot_direct_arg3"] = "arg3",
  ["tooltip_regexp_channel_arg1"] = "arg1",
  ["tooltip_regexp_channel_arg2"] = "arg2",
  ["tooltip_regexp_channel_arg3"] = "arg3",

} end)


local lib = { }
AceLibrary("AceEvent-2.0"):embed(lib)
AceLibrary("AceComm-2.0"):embed(lib)

local BS = AceLibrary("Babble-Spell-2.2")
local Gratuity = AceLibrary("Gratuity-2.0")
local Roster = AceLibrary("Roster-2.1")
local Aura = AceLibrary("SpecialEvents-Aura-2.0")

local PLAYERNAME = UnitName("player")
local START_TRY_TIME_DIFF = 1.0 -- max time diff between sent and start for a real cast
local SENDVERSION_TIME_INTERVAL = 5
local SHOW_KNOWN_USERS_DELAY = 3

local COMM_PREFIX = "IHL"

local SPELLS_BY_ID = {
   [1] = { name = BS["Renew"], type = "hot", casttime = 0 },
   [2] = { name = BS["Flash Heal"], type = "direct", casttime = 1.5 },
   [3] = { name = BS["Heal"], type = "direct", casttime = 3.0 },
   [4] = { name = BS["Greater Heal"], type = "direct", casttime = 3.0 },
   [5] = { name = BS["Prayer of Healing"], type = "direct", casttime = 3.0, target = "party" },
   [6] = { name = BS["Binding Heal"], type = "direct", casttime = 1.5, target = "binding" },
   [7] = { name = BS["Rejuvenation"], type = "hot", casttime = 0 },
   [8] = { name = BS["Regrowth"], type = "direct_hot", casttime = 2.0 },
   [9] = { name = BS["Lifebloom"], type = "hot_direct", casttime = 0 },
  [10] = { name = BS["Healing Touch"], type = "direct", casttime = 3.5 },
  [11] = { name = BS["Tranquility"], type = "channel", casttime = 8, target = "party" },
  [12] = { name = BS["Swiftmend"], type = "swiftmend", casttime = 0 },
  [13] = { name = BS["Holy Light"], type = "direct", casttime = 2.5, bol = 400 },
  [14] = { name = BS["Flash of Light"], type = "direct", casttime = 1.5, bol = 115 },
  [15] = { name = BS["Lesser Healing Wave"], type = "direct", casttime = 1.5 },
  [16] = { name = BS["Healing Wave"], type = "direct", casttime = 3.0 },
  [17] = { name = BS["Chain Heal"], type = "direct", casttime = 2.5 },
  [18] = { name = BS["Resurrection"], type = "resurrect", casttime = 10 },
  [19] = { name = BS["Redemption"], type = "resurrect", casttime = 10 },
  [20] = { name = BS["Rebirth"], type = "resurrect", casttime = 3 },
}

local SPELLS_BY_NAME = { }
for i, spell in ipairs(SPELLS_BY_ID) do
  SPELLS_BY_NAME[spell.name] = spell
  spell.id = i
end


-- List of talents we want to know about
local TALENTS_FUN = {
  [BS["Spiritual Healing"]] = true,
  [BS["Empowered Healing"]] = true,
  [BS["Improved Renew"]] = true,
  [BS["Healing Light"]] = true,
  [BS["Empowered Touch"]] = true,
  [BS["Empowered Rejuvenation"]] = true,
  [BS["Gift of Nature"]] = true,
  [BS["Improved Rejuvenation"]] = true,
  [BS["Purification"]] = true,
  [BS["Improved Chain Heal"]] = true,
}



-- Activate a new instance of this library
local function activate(self, oldLib, oldDeactivate)
  if oldLib then  -- if upgrading
    oldLib:UnregisterAllEvents()
  end

  self.version = MAJOR_VERSION.." "..MINOR_VERSION:sub(12, -2)
  self:Reset()
  self.knownUsers = { [PLAYERNAME] = MAJOR_VERSION }

  self:SetCommPrefix(COMM_PREFIX)

  if AceLibrary("AceEvent-2.0"):IsFullyInitialized() then
    self:OnFullyInitialized()
  else
    self:RegisterEvent("AceEvent_FullyInitialized", "OnFullyInitialized")
  end

  if oldDeactivate then  -- clean up the old library
    oldDeactivate(oldLib)
  end
end

function lib:OnFullyInitialized()
  self:RegisterEvent("UNIT_SPELLCAST_SENT", "OnSpellCastSent")
  self:RegisterEvent("UNIT_SPELLCAST_SUCCEEDED", "OnSpellCastSucceeded")
  self:RegisterEvent("UNIT_SPELLCAST_START", "OnSpellCastStart")
  self:RegisterEvent("UNIT_SPELLCAST_STOP", "OnSpellCastStop")
  self:RegisterEvent("UNIT_SPELLCAST_CHANNEL_START", "OnSpellCastStart")
  self:RegisterEvent("UNIT_SPELLCAST_CHANNEL_STOP", "OnSpellCastStop")
--  self:RegisterEvent("UNIT_SPELLCAST_FAILED", "OnSpellCastFailed")
--  self:RegisterEvent("UNIT_SPELLCAST_INTERRUPTED", "OnSpellCastInterrupted")
--  self:RegisterEvent("UNIT_SPELLCAST_DELAYED", "OnSpellCastDelayed")
  self:RegisterEvent("TRAINER_UPDATE", "ScanTalents")

  self:RegisterComm(COMM_PREFIX, "GROUP", "OnCommReceive")

  self:ScanTalents()
  self:SendVersion()
end























---------------------------------------------------
--      Spell communication, receiving side      --
---------------------------------------------------


function lib:SendComm(ctype, ...)
  self:SendPrioritizedCommMessage("ALERT", "GROUP", ctype, ...)
  self["OnReceive"..ctype](self, PLAYERNAME, ...)
end

function lib:OnCommReceive(prefix, source, distribution, ctype, ...)
  if not self.knownUsers[source] then
    self.knownUsers[source] = "1.0"
    self:SendVersion()
  end

  if ctype ~= "Version" and self.knownUsers[source] < MAJOR_VERSION then
    return
  end

  if ctype == nil then
    return
  end

  if self["OnReceive"..ctype] then
    self["OnReceive"..ctype](self, source, ...)
  end
end




--  Normal heals

function lib:OnReceiveHeal(source, target, amount, hotamount, hottime)
  local unitid = self:GetUnitIDFromName(source)
  local targetunitid = self:GetUnitIDFromName(target)
  local spellname, rank, displayName, icon, startTime, endTime, isTradeSkill = UnitCastingInfo(unitid)
  if not unitid then return end -- MCd players?
  if not spellname then spellname, rank, displayName, icon, startTime, endTime, isTradeSkill = UnitChannelInfo(unitid) end
  if not spellname then return end
  local spell = SPELLS_BY_NAME[spellname]

  if spell.type == "channel" then return end -- not implemented yet, no big deal anyway

  local ch = self.castingHeals[source]
  ch.target = target
  ch.spell = spell
  ch.amount = amount
  ch.hotamount = hotamount
  ch.hottime = hottime
  ch.endtime = endTime / 1000

  -- Get a list of targets for a spelltype (prayer of healing, binding heal, or normal)
  if next(ch.targets) then
    for k in pairs(ch.targets) do
      ch.targets[k] = nil
    end
  end
  
  if spell.target == "party" then
    local subgroup = Roster:GetUnitObjectFromName(source).subgroup
    for unit in Roster:IterateRoster() do
      if unit.subgroup == subgroup then
        ch.targets[unit.name] = true
      end
    end
  elseif spell.target == "binding" then
    ch.targets[target] = true
    ch.targets[source] = true
  else
    ch.targets[target] = true
  end

  -- Blessing of Light
  if targetunitid and spell.bol and (Aura:UnitHasBuff(targetunitid, BS["Blessing of Light"]) or Aura:UnitHasBuff(targetunitid, BS["Greater Blessing of Light"])) then
    ch.amount = ch.amount + spell.bol
  end

  -- Healing Way
  if targetunitid and (spell.id == 15 or spell.id == 16) then
    local index, count, icon, rank = Aura:UnitHasBuff(targetunitid, BS["Healing Way"])
    ch.amount = ch.amount + ch.amount * (count or 0) * 0.06
  end

  -- Sanctity Aura (we assume it's always improved by talents, cuz in raids it usually is)
  if targetunitid and Aura:UnitHasBuff(targetunitid, BS["Sanctity Aura"]) then
    ch.amount = ch.amount * 1.06
  end

  -- Fel Armor
  if targetunitid and Aura:UnitHasBuff(targetunitid, BS["Fel Armor"]) then
    ch.amount = ch.amount * 1.20
  end

  self:ScheduleEvent("IHL_OnHealEnd_"..source, self.OnHealEnd, (endTime - startTime) / 1000, self, source)

  for target in pairs(ch.targets) do
    self:TriggerEvent("IncomingHealsLib_HealStart", target, source)
  end
end

function lib:OnReceiveCancelHeal(source)
  local ch = self.castingHeals[source]
  ch.endtime = 0
  self:CancelScheduledEvent("IHL_OnHealEnd_"..source)
  for target in pairs(ch.targets) do
    self:TriggerEvent("IncomingHealsLib_HealCancel", target, source)
  end
end

function lib:OnHealEnd(source)
  local ch = self.castingHeals[source]
  local spell = ch.spell

  if spell.type == "direct_hot" then
    for target in pairs(ch.targets) do
      self:StartHot(source, target, spell.id, ch.hotamount, ch.hottime)
    end
  end

  for target in pairs(ch.targets) do
    self:TriggerEvent("IncomingHealsLib_HealEnd", target, source)
  end
end





-- Resurrects

function lib:OnReceiveResurrect(source, target)
  local unitid = self:GetUnitIDFromName(source)
  local spellname, rank, displayName, icon, startTime, endTime, isTradeSkill = UnitCastingInfo(unitid)
  if not spellname then return end
  local spell = SPELLS_BY_NAME[spellname]

  local cr = self.castingResses[source]
  cr.target = target
  cr.spell = spell
  cr.endtime = endTime / 1000

  self:ScheduleEvent("IH_OnResEnd_"..source, self.OnResEnd, (endTime - startTime) / 1000, self, source)

  self:TriggerEvent("IncomingHealsLib_ResStart", target, source)
end

function lib:OnReceiveCancelRes(source)
  local cr = self.castingResses[source]
  cr.endtime = 0
  self:CancelScheduledEvent("IH_OnResEnd_"..source)
  self:TriggerEvent("IncomingHealsLib_ResEnd", cr.target, source)
end

function lib:OnResEnd(source)
local cr = self.castingResses[source]
  self:TriggerEvent("IncomingHealsLib_ResEnd", cr.target, source)
end





-- HoTs

function lib:OnReceiveHot(source, target, id, amount, runtime)
  self:StartHot(source, target, id, amount, runtime)
end

function lib:StartHot(source, target, id, amount, runtime) -- amount is hotamount
  local spell = SPELLS_BY_ID[id]
  local targetunitid = self:GetUnitIDFromName(target)

  local rh = self.runningHots[target][source..id]

  -- Lifebloom can stack 3 times per player
  if spell.id == 9 then
    if rh.endtime < GetTime() then
      rh.stacks = 0
    end
    rh.stacks = min(3, (rh.stacks or 0) + 1)
    rh.amount = rh.stacks * amount

  -- all other hots
  else
    rh.amount = amount

  end

  rh.runtime = runtime
  rh.endtime = GetTime() + runtime -- don't move this up, need old endtime for lifebloom code (see above)
  rh.spell = spell
  rh.source = source


  -- Sanctity Aura (we assume it's always improved by talents, cuz in raids it usually is)
  if targetunitid and Aura:UnitHasBuff(targetunitid, BS["Sanctity Aura"]) then
    rh.amount = rh.amount * 1.06
  end

  -- Fel Armor
  if targetunitid and Aura:UnitHasBuff(targetunitid, BS["Fel Armor"]) then
    rh.amount = rh.amount * 1.20
  end

  self:ScheduleEvent("IHL_OnHotEnd_"..source..target..id, self.OnHotEnd, runtime, self, source, target, id)
  self:TriggerEvent("IncomingHealsLib_HotStart", target, source, id)
end

function lib:OnHotEnd(source, target, id)
  self:TriggerEvent("IncomingHealsLib_HotEnd", target, source, id)
end

--[[ Remove lowest duration rejuvenation or regrowth ]]--
function lib:OnReceiveSwiftmend(source, target)
  local hots = self.runningHots[target]
  local index_to_remove = nil
  local endtime = nil
  local rejuv = SPELLS_BY_ID[7]
  local regrowth = SPELLS_BY_ID[8]
  for index, rh in pairs(hots) do
    if (rh.spell == rejuv or rh.spell == regrowth) and (not endtime or rh.endtime < endtime) then index_to_remove = index end
  end
  if index_to_remove then hots[index_to_remove].endtime = 0 end

  self:CancelScheduledEvent("IHL_OnHotEnd_"..source..index_to_remove)
  self:TriggerEvent("IncomingHealsLib_Swiftmend", target, source)
end



















-----------------------------------------------------
-- Spell detection and communication, sending side --
-----------------------------------------------------


local senttime = 0
local senttarget = nil
local sentspellname = nil
local sentrank = nil
local sentcasttime = 0
local sentspell = nil
local cancancel =  false

function lib:OnSpellCastSent(unit, spellname, rank, target)
  local spell = SPELLS_BY_NAME[spellname]
  if not spell then
    cancancel = false
    sentcasttime = 0
    sentspell = nil
    return
  end

  senttime = GetTime()
  senttarget = target
  sentspellname = spellname
  sentrank = rank
  sentcasttime = spell.casttime
  sentspell = spell
  cancancel = false

  if Aura:UnitHasBuff("player", BS["Nature's Swiftness"]) then sentcasttime = 0 end
end


function lib:OnSpellCastStart(unitid)
  if unitid ~= "player" then return end

  local spellname, rank, displayName, icon, startTime, endTime, isTradeSkill = UnitCastingInfo(unitid)
  if not spellname then spellname, rank, displayName, icon, startTime, endTime, isTradeSkill = UnitChannelInfo(unitid) end
  local spell = SPELLS_BY_NAME[spellname]
  if not spell then return end

  if spell.type == "resurrect" then
    self:SendComm("Resurrect", senttarget)
  else
    local amount, hotamount, hottime = self:GetHealData(spellname, rank)
    self:SendComm("Heal", senttarget, amount, hotamount, hottime)
  end

  cancancel = true
end

function lib:OnSpellCastStop(unitid)
  if unitid ~= "player" then return end

  if cancancel and sentcasttime > 0 then
    cancancel = false
    if sentspell.type == "resurrect" then
      self:SendComm("CancelRes")
    else
      self:SendComm("CancelHeal")
    end
  end
end

function lib:OnSpellCastSucceeded()
  cancancel = false
  if sentspell and sentcasttime == 0 then

    if sentspell.type == "swiftmend" then
      self:SendComm("Swiftmend", senttarget)
    else
      local amount, hotamount, hottime = self:GetHealData(sentspellname, sentrank)
      if hotamount and hotamount > 0 then
        self:SendComm("Hot", senttarget, sentspell.id, hotamount, hottime)
      end
    end
  end
end




















--[[ Resets all incoming heals and hots ]]--
function lib:Reset()
  -- index is name of source unit
  self.castingHeals = setmetatable({}, { __index = function(self, key)
    local t = {
      targets = {},
      target = 0,
      spell = 0,
      amount = 0,
      hotamount = 0,
      hottime = 0,
      endtime = 0,
    }
    self[key] = t
    return t
  end})

  -- index is name of source unit
  self.castingResses = setmetatable({ }, { __index = function(self, key)
    local t = {
      target = 0,
      spell = 0,
      endtime = 0,
    }
    self[key] = t
    return t
  end})

   -- incomingHots["Ciel"]["Arani_6"] = { } --> Arani has a Rejuvenation on Ciel
  self.runningHots = setmetatable({ }, { __index = function(self, key)
    local t = setmetatable({ }, { __index = function(self, key)
      local t = {
--        stacks = 0, -- Lifebloom, not adding it here to save some space
        amount = 0,
        runtime = 0,
        endtime = 0,
        spell = 0,
        source = 0,
      }
      self[key] = t
      return t
    end})
    self[key] = t
    return t
  end})
end



function lib:GetHealData(spellname, rank)
  local spell = SPELLS_BY_NAME[spellname]
  local id = spell.id
  local stype = spell.type
  local spellindex = 1
  local sName, sRank

  repeat
    sName, sRank = GetSpellName(spellindex, BOOKTYPE_SPELL)
    if sName == spellname and sRank == rank then
      break
    end
    spellindex = spellindex + 1
  until not sName

  Gratuity:SetSpell(spellindex, BOOKTYPE_SPELL)
  local found, _, arg1, arg2, arg3, arg4 = Gratuity:Find(L["tooltip_regexp_"..stype], 3, 4)

  if not found or not arg1 then return end

  arg1 = tonumber(arg1)
  arg2 = tonumber(arg2)
  arg3 = tonumber(arg3)
  arg4 = tonumber(arg4)

  if L:HasTranslation("tooltip_regexp_"..stype.."_arg1") then
    local args = {
      arg1 = arg1,
      arg2 = arg2,
      arg3 = arg3,
      arg4 = arg4,
    }
    if L:HasTranslation("tooltip_regexp_"..stype.."_arg1") then arg1 = args[L["tooltip_regexp_"..stype.."_arg1"]] end
    if L:HasTranslation("tooltip_regexp_"..stype.."_arg2") then arg2 = args[L["tooltip_regexp_"..stype.."_arg2"]] end
    if L:HasTranslation("tooltip_regexp_"..stype.."_arg3") then arg3 = args[L["tooltip_regexp_"..stype.."_arg3"]] end
    if L:HasTranslation("tooltip_regexp_"..stype.."_arg4") then arg4 = args[L["tooltip_regexp_"..stype.."_arg4"]] end
  end

  local amount, hotamount, hottime = 0, 0, 0

  local plusheal = GetSpellBonusHealing()
  if spell.target == "party" then plusheal = floor(plusheal * 0.333) end

  if stype == "direct" then
    amount = ((arg1 + arg2) / 2 + plusheal * spell.casttime / 3.5)
  elseif stype == "hot" then
    hotamount = arg1 + plusheal * arg2 / 15 -- QQ for druids
    hottime = arg2
  elseif stype == "hot_direct" then -- lifebloom
    hotamount = arg1 + plusheal * 0.52
    hottime = arg2
    amount = arg3 + plusheal / 2
  elseif stype == "direct_hot" then
    amount = (arg1 + arg2) / 2 + plusheal / 2 * spell.casttime / 3.5 -- Can't find any reference to regrowth coefficient that matches my tests
    hotamount = arg3 + plusheal / 2 * arg4 / 15
    hottime = arg4
  elseif stype == "channel" then -- "for (%d+) to (%d+) every (%d+) seconds for (%d+) sec",
    hotamount = (arg1 + arg2) / 2 / arg3 * arg5 + plusheal
    hottime = arg4
  end

  -- Empowered Healing
  if id == 2 then amount = amount + plusheal * self:GetTalentRank("Empowered Healing") * 0.02 end
  if id == 4 then amount = amount + plusheal * self:GetTalentRank("Empowered Healing") * 0.04 end

  -- Empowered Touch
  if id == 10 then amount = amount + plusheal * self:GetTalentRank("Empowered Touch") * 0.10 end

  -- Empowered Rejuvenation
  if id == 7 then hotamount = hotamount + plusheal * self:GetTalentRank("Empowered Rejuvenation") * 0.04 end
  if id == 8 then hotamount = hotamount + plusheal * self:GetTalentRank("Empowered Rejuvenation") * 0.04 / 2 * hottime / 15 end
  if id == 8 then amount = amount + plusheal * self:GetTalentRank("Empowered Rejuvenation") * 0.04 / 2 * spell.casttime / 3.5 end
  if id == 9 then hotamount = hotamount + plusheal * self:GetTalentRank("Empowered Rejuvenation") * 0.04 * 0.52 end

  -- Improved Renew
  if id == 1 then hotamount = hotamount + hotamount * self:GetTalentRank(BS["Improved Renew"]) * 0.05 end

  -- Improved Rejuvenation
  if id == 7 then hotamount = hotamount + hotamount * self:GetTalentRank(BS["Improved Rejuvenation"]) * 0.05 end

  -- Spiritual Healing
  hotamount = hotamount + hotamount * self:GetTalentRank(BS["Spiritual Healing"]) * 0.02
  amount = amount + amount * self:GetTalentRank(BS["Spiritual Healing"]) * 0.02

  -- Gift of Nature
  hotamount = hotamount + hotamount * self:GetTalentRank(BS["Gift of Nature"]) * 0.02
  amount = amount + amount * self:GetTalentRank(BS["Gift of Nature"]) * 0.02

  -- Purification
  hotamount = hotamount + hotamount * self:GetTalentRank(BS["Purification"]) * 0.02
  amount = amount + amount * self:GetTalentRank(BS["Purification"]) * 0.02

  -- Improved Chain Heal
  if id == 17 then amount = amount + amount * self:GetTalentRank(BS["Improved Chain Heal"]) * 0.10 end

  -- Healing Light and Divine Favor
  if id == 13 or id == 14 then
    amount = amount + amount * self:GetTalentRank(BS["Healing Light"]) * 0.04
    if Aura:UnitHasBuff("player", BS["Divine Favor"]) then amount = amount * 1.5 end
  end

  -- Check heals
  if amount > 0 and self.checkheals then DEFAULT_CHAT_FRAME:AddMessage("Amount: "..floor(amount)) end
  if hotamount > 0 and self.checkheals then DEFAULT_CHAT_FRAME:AddMessage("HoT per tick: "..floor(hotamount/hottime*3)) end

  return amount, hotamount, hottime
end
























------------------------------
--      Version info        --
------------------------------

local last_sendversion_time = 0

function lib:SendVersion()
  last_sendversion_time = GetTime()
  self:SendComm("Version", self.version)
end

function lib:OnReceiveVersion(source, version)
  if not last_sendversion_time or last_sendversion_time + SENDVERSION_TIME_INTERVAL < GetTime() then
    self:SendVersion()
  end
  self.knownUsers[source] = version
end













------------------------------
--      Talents             --
------------------------------

local talents = { }

function lib:ScanTalents()
  for tabindex = 1, GetNumTalentTabs() do
    for talentindex = 1, GetNumTalents(tabindex) do
      local name, _, _, _, rank = GetTalentInfo(tabindex, talentindex)
      if TALENTS_FUN[name] then talents[name] = rank end
    end
  end
end

function lib:GetTalentRank(name)
  return talents[name] or 0
end





















------------------------------
--     Utilities            --
------------------------------

function lib:GetUnitIDFromName(name)
  if not name then return nil end
  local unitid = Roster:GetUnitIDFromName(name)
  if not unitid then
    if UnitName("target") == name then unitid = "target"
--    elseif UnitName("targettarget") == name then unitid = "targettarget"  -- doesn't update correctly on PLYAER_TARGET_CHANGED
    elseif UnitName("focus") == name then unitid = "focus"
    elseif UnitName("focustarget") == name then unitid = "focustarget"
    end

    if not unitid then
      for i = 1, GetNumRaidMembers() do
        if UnitName("raid"..i.."target") == name then unitid = "raid"..i.."target" end
        if UnitName("raid"..i.."targettarget") == name then unitid = "raid"..i.."targettarget" end
      end
    end
  end
  return unitid
end













------------------------------
--         API              --
------------------------------

-- /x local IHL = AceLibrary("IncomingHealsLib-1.0"); print(IHL.castingHeals);
-- /x local IHL = AceLibrary("IncomingHealsLib-1.0"); local amount, overheal, myamount, myoverheal = IHL:GetTargetIncomingHealInfo("player"); echo(amount.."--"..overheal.."--"..myamount.."--"..myoverheal);
-- returns: amount, overheal amount, my amount, my overheal amount
function lib:GetTargetIncomingHealInfo(unitname)
  if not unitname then return 0, 0, 0, 0 end
  local amount = 0
  local mych = self.castingHeals[PLAYERNAME]
  local myamount = mych.amount or 0
  local mytime = mych.endtime or 0
  local amountbeforemyheal = 0

  local unitid = self:GetUnitIDFromName(unitname)
  local unitmissinghp = 100000
  if unitid and UnitHealthMax(unitid) ~= 100 then
    unitmissinghp = UnitHealthMax(unitid) - UnitHealth(unitid)
  end

  if mych.endtime < GetTime() or not mych.targets[unitname] then
    myamount = 0
    mytime = 0
  end

  for source, ch in pairs(self.castingHeals) do
    if ch.endtime > GetTime() and ch.spell.type ~= "channel" and ch.targets[unitname] then
      amount = amount + ch.amount
      if ch.endtime < mytime then
        amountbeforemyheal = amountbeforemyheal + ch.amount
      end
    end
  end

  return amount, max(amount - unitmissinghp, 0), myamount, max(min(myamount + amountbeforemyheal - unitmissinghp, myamount), 0)
end

-- /x local IHL = AceLibrary("IncomingHealsLib-1.0"); local hottick, myhottick = IHL:GetTargetRunningHotInfo("player"); echo(hottick.."--"..myhottick);
-- returns: hottick, myhottick
function lib:GetTargetRunningHotInfo(unitname)
  if not unitname then return 0, 0 end
  local hottick = 0
  local myhottick = 0

  for _, rh in pairs(self.runningHots[unitname]) do
    if rh.endtime > GetTime() then
      hottick = hottick + rh.amount / rh.runtime
      if rh.source == PLAYERNAME then
        myhottick = myhottick + rh.amount / rh.runtime
      end
    end
  end

  for source, ch in pairs(self.castingHeals) do
    if ch.endtime > GetTime() and ch.spell.type == "channel" then
      if ch.targets[unitname] then
        hottick = hottick + ch.hotamount / ch.hottime
      end
    end
  end

  return hottick * 3, myhottick * 3
end


-- returns: number of incoming resses, myres
function lib:GetTargetIncomingResInfo(unitname)
  if not unitname then return 0, false, false end
  local incresses = 0
  local mycr = self.castingResses[PLAYERNAME]
  local myres = self.castingResses[PLAYERNAME].target == PLAYERNAME
  local myoverres = false

  for source, cr in pairs(self.castingResses) do
    if cr.endtime > GetTime() and cr.target == unitname then
      incresses = incresses + 1
      if myres and cr.endtime < mycr.endtime then
        myoverres = true
      end
    end
  end

  return incresses, myres, myoverres
end

-- returns table with data if casting, nil if not
function lib:GetCastingInfo(unitname)

  local ch = self.castingHeals[unitname]
  if ch.endtime > GetTime() then
    return ch
  end

  local cr = self.castingResses[unitname]
  if cr.endtime > GetTime() then
    return cr
  end

  return nil
end




















--------------------------------
--      Load this bitch!      --
--------------------------------
AceLibrary:Register(lib, LIBNAME, MINOR_VERSION, activate)
lib = nil
