----------------------------
--      Locals            --
----------------------------

local DD = AceLibrary("Dewdrop-2.0")
local IHL = AceLibrary("IncomingHealsLib-1.0")
local L = AceLibrary("AceLocale-2.2"):new("IncomingHeals")
local Roster = AceLibrary("Roster-2.1")
local PLAYERNAME = UnitName("player")

IncomingHeals = AceLibrary("AceAddon-2.0"):new("AceEvent-2.0", "AceConsole-2.0", "AceDB-2.0", "AceHook-2.1")

local new, del
do
	local list = setmetatable({}, {__mode='k'})
	function new(...)
		local t = next(list)
		if t then
			list[t] = nil
			for i = 1, select('#', ...) do
				t[i] = select(i, ...)
			end
			return t
		else
			return {...}
		end
	end
	function del(t)
		setmetatable(t, nil)
		for k in pairs(t) do
			t[k] = nil
		end
		list[t] = true
		return nil
	end
end




















----------------------------
--      Constants         --
----------------------------

local STARTUP_MESSAGE_DELAY = 2.0
local COLOR_CODE = "00FF00"
local SENDVERSION_TIME_INTERVAL = 5
local SHOW_KNOWN_USERS_DELAY = 3



BackdropEmptyBlack = { bgFile = "Interface\\ChatFrame\\ChatFrameBackground", tile = true, tileSize = 0, insets = { left = 0, right = 0, top = 0, bottom = 0 } }


IncomingHeals.COLORS = {
	white	= { r = 1, g = 1, b = 1, a = 1 },
	black	= { r = 0, g = 0, b = 0, a = 1 },
	red		= { r = 1, g = 0.1, b = 0.1, a = 1 },
	green	= { r = 0.1, g = 1, b = 0.1, a = 1 },
	blue	= { r = 0.4, g = 0.4, b = 1, a = 1 },
	yellow	= { r = 1, g = 1, b = 0.1, a = 1 },
}

IncomingHeals.COLORS.heal		= IncomingHeals.COLORS.white
IncomingHeals.COLORS.overheal	= IncomingHeals.COLORS.white
IncomingHeals.COLORS.myheal		= IncomingHeals.COLORS.green
IncomingHeals.COLORS.myoverheal = IncomingHeals.COLORS.red
IncomingHeals.COLORS.hot		= IncomingHeals.COLORS.white
IncomingHeals.COLORS.overhot	= IncomingHeals.COLORS.white
IncomingHeals.COLORS.myhot		= IncomingHeals.COLORS.green
IncomingHeals.COLORS.myoverhot	= IncomingHeals.COLORS.green
IncomingHeals.COLORS.res		= IncomingHeals.COLORS.yellow
IncomingHeals.COLORS.myres		= IncomingHeals.COLORS.green
IncomingHeals.COLORS.myoverres	= IncomingHeals.COLORS.red

























------------------------------
--      Visuals             --
------------------------------

function IncomingHeals:UpdateVisuals(target)
	--echo(GetTime().." UpdateVisuals")

	-- Update graphs
	for index, graph in pairs(self.Graph.graphs) do
		if graph.unitname == target then
			graph:Update()
		end
	end

	local unitid = IHL:GetUnitIDFromName(target)
	if not unitid then return end
	local amount, overheal, myamount, myoverheal = IHL:GetTargetIncomingHealInfo(target)
	local hottick, myhottick = IHL:GetTargetRunningHotInfo(target)
	local incresses, myres = IHL:GetTargetIncomingResInfo(target)

	if UnitIsDeadOrGhost(unitid) or not UnitIsConnected(unitid) then
		amount = 0
		hotamount = 0
	end

	local frames = self:GetUnitFrames(unitid)
	for _, frame in ipairs(frames) do
		if frame then
			local bgcolor = self.COLORS.black
			if PriestMonitor then
				bgcolor = PriestMonitor:GetCircleColor(unitid) or self.COLORS.black
			end
			self:UpdateUnitFrame(frame, amount, overheal, myamount, myoverheal, hottick, myhottick, incresses, myres, bgcolor)
		end
	end
end

function IncomingHeals:GetUnitFrames(unitid)
	local frames = { }

	if sRaidFrames then
		for _, f in pairs(sRaidFrames:FindUnitFrames(unitid)) do
			if not f.heal then self:srf_CreateDisplay(f) end
			table.insert(frames, f)
		end
	end

	if CT_RA_UnitIDFrameMap and CT_RA_UnitIDFrameMap[unitid] then
		local f = CT_RA_UnitIDFrameMap[unitid]
		if not f.heal then
			self:ctra_CreateDisplay(f)
			f.title = f.Name
		end
		table.insert(frames, f)
	end

	if self.XPerlFrames and self.XPerlFrames[unitid] then
		local f = self.XPerlFrames[unitid]
		if not f.heal then self:xperl_CreateDisplay(f) end
		table.insert(frames, f)
	end

	return frames
end


function IncomingHeals:UpdateUnitFrame(f, amount, overheal, myamount, myoverheal, hottick, myhottick, incresses, myres, bgcolor)
	local colors = self.COLORS

	bgcolor = bgcolor or colors.black

	f.healframe:Hide()
	f.hotframe:Hide()
	f.heal:Hide()
	f.hot:Hide()

	if incresses > 0 then
		f.heal:SetTextColor(1, 1, 0, 1)
		f.heal:SetText(incresses.."RI")
		f.heal:Show()
		f.healframe:Show()
		return
	end

	if amount > 0 then
		local color = colors.heal
		local r, g, b = color.r, color.g, color.b
		if myamount > 0 then
			r, g, b = IncomingHeals:MixColors(colors.myheal, colors.myoverheal, myoverheal / myamount)
		else
			r, g, b = IncomingHeals:MixColors(colors.heal, colors.overheal, overheal / amount)
		end

		f.heal:SetTextColor(r, g, b, 1)
		if amount < 10000 then f.heal:SetText(string.format("%.1f", amount / 1000))
		else f.heal:SetText(string.format("%.0fk", amount / 1000)) end
		f.heal:Show()
		f.healframe:Show()
	end

	if hottick > 0 then
		local color = colors.hot
		if myhottick > 0 then
			color = colors.myhot
		end

		f.hot:SetTextColor(color.r, color.g, color.b, color.a)
		f.hot:SetText(string.format("%.1f", hottick / 1000))
		f.hot:Show()
		f.hotframe:Show()
	end

	if bgcolor then
		f.hotframe:SetBackdropColor(bgcolor.r, bgcolor.g, bgcolor.b, bgcolor.a)
		f.healframe:SetBackdropColor(bgcolor.r, bgcolor.g, bgcolor.b, bgcolor.a)
		if bgcolor ~= self.COLORS.black then
			f.hotframe:Show()
			f.healframe:Show()
		end
	end
end


--[[ Resets all visuals ]]--
function IncomingHeals:ResetVisuals()
	if sRaidFrames then
		for _, f in ipairs(sRaidFrames.frames) do
			if f.heal then
				f.heal:Hide()
				f.hot:Hide()
				f.healframe:Hide()
				f.hotframe:Hide()
			end
		end
	end

	if CT_RA_UnitIDFrameMap then
		for _, f in pairs(CT_RA_UnitIDFrameMap) do
			if f.heal then
				f.heal:Hide()
				f.hot:Hide()
				f.healframe:Hide()
				f.hotframe:Hide()
			end
		end
	end

	if self.XPerlFrames then
		for unitid, f in pairs(self.XPerlFrames) do
			if f.heal then
				f.heal:Hide()
				f.hot:Hide()
				f.healframe:Hide()
				f.hotframe:Hide()
			end
		end
	end
end


function IncomingHeals:CreateDisplay(f)
	f.hotframe = CreateFrame("Frame", nil, f)
	f.hotframe:SetBackdrop(BackdropEmptyBlack)
	f.hotframe:SetBackdropColor(0, 0, 0, 1)
--	f.hotframe:SetFrameLevel(100000)
	f.hotframe:SetFrameStrata("HIGH")
	self:SetWHP(f.hotframe, 23, 15, "TOPRIGHT", f.buff2, "TOPLEFT", -2, -1)

	f.hot = f.hotframe:CreateFontString(nil, "ARTWORK", "GameFontNormal")
	f.hot:SetJustifyH("RIGHT")
	self:SetWHP(f.hot, 23, 15, "TOPRIGHT", f.hotframe, "TOPRIGHT",	0, 1)

	f.healframe = CreateFrame("Frame", nil, f)
	f.healframe:SetBackdrop(BackdropEmptyBlack)
	f.healframe:SetBackdropColor(0, 0, 0, 1)
--	f.healframe:SetFrameLevel(f.healframe:GetFrameLevel()+10)
	f.healframe:SetFrameStrata("HIGH")
	self:SetWHP(f.healframe, 23, 15, "TOPRIGHT", f.hotframe, "TOPLEFT", 0, 0)

	f.heal = f.healframe:CreateFontString(nil, "ARTWORK", "GameFontNormal")
	f.heal:SetJustifyH("RIGHT")
	self:SetWHP(f.heal, 23, 15, "TOPRIGHT", f.healframe, "TOPRIGHT",	0, 1)
end

function IncomingHeals:srf_CreateDisplay(f)
	self:CreateDisplay(f)

	self:SetWHP(f.buff1, 10, 10, "TOPRIGHT", f, "TOPRIGHT", -4, -4)
	self:SetWHP(f.buff2, 10, 10, "RIGHT", f.buff1, "LEFT", 0, 0)
	self:SetWHP(f.buff3, 10, 10, "TOP", f.buff1, "BOTTOM", 0, 0)
	self:SetWHP(f.buff4, 10, 10, "RIGHT", f.buff3, "LEFT", 0, 0)

	f.buff1:SetFrameLevel(f.buff1:GetFrameLevel()+1)
	f.buff2:SetFrameLevel(f.buff2:GetFrameLevel()+1)
	f.buff3:SetFrameLevel(f.buff3:GetFrameLevel()+1)
	f.buff4:SetFrameLevel(f.buff4:GetFrameLevel()+1)
end

function IncomingHeals:ctra_CreateDisplay(f)
	self:CreateDisplay(f)
	self:SetWHP(f.hotframe, 23, 15, "TOPRIGHT", f, "TOPRIGHT", -25, -5)
end

function IncomingHeals:xperl_CreateDisplay(f)
	self:CreateDisplay(f)
	self:SetWHP(f.hotframe, 23, 15, "TOPRIGHT", f, "TOPRIGHT", -5, -5)
	f.nameFrame.text:SetJustifyH("LEFT")
	f.nameFrame.text:ClearAllPoints()
	f.nameFrame.text:SetPoint("TOPLEFT", f, "TOPLEFT", 3, -3)
end

if sRaidFrames and sRaidFrames.SetWHP then
	IncomingHeals.SetWHP = sRaidFrames.SetWHP
else
	function IncomingHeals:SetWHP(frame, width, height, p1, relative, p2, x, y)
		frame:SetWidth(width)
		frame:SetHeight(height)

		if (p1) then
			frame:ClearAllPoints()
			frame:SetPoint(p1, relative, p2, x, y)
		end
	end
end

function IncomingHeals:srf_UpdateUnit(srf, units)
	if units then
		if type(units) == "string" then
			self:UpdateVisuals(UnitName(units))
		else
			for unit in pairs(units) do
				self:UpdateVisuals(UnitName(unit))
			end
		end
	end
end

function IncomingHeals:CT_RA_UpdateRaidGroup(updateType)
	for i=1, MAX_RAID_MEMBERS do
		self:UpdateVisuals(UnitName("raid"..i))
	end
end

function IncomingHeals:CT_RA_UpdateUnitStatus(f)
	if f then self:UpdateVisuals(f.unitName) end
end

function IncomingHeals:CT_RA_UpdateUnitDead(f)
	if f then self:UpdateVisuals(f.unitName) end
end

function IncomingHeals:XPerl_Raid_UpdateDisplay(f)
	if f then
		local unitid = f:GetAttribute("unit")
		self.XPerlFrames[unitid] = f --.nameFrame
		self:UpdateVisuals(UnitName(unitid))
	end
end

function IncomingHeals:OnUnitHealth(units)
	for unitid in pairs(units) do
		self:UpdateVisuals(UnitName(unitid))
		if PriestMonitor then
			local extra_units = PriestMonitor:GetCircleUnits(unitid)
			if extra_units then
				for i, unitid2 in ipairs(extra_units) do
					self:UpdateVisuals(UnitName(unitid2))
				end
			end
		end
	end
end























------------------------------
--      Version info        --
------------------------------

function IncomingHeals:ShowKnownUsers()
	DEFAULT_CHAT_FRAME:AddMessage("|c00"..COLOR_CODE..self.name.."|r - "..L["known users"]..":")

	local users = new()
	for playername, version in pairs(IHL.knownUsers) do
		table.insert(users, { name = playername, version = version })
	end
	table.sort(users, function(t1, t2) return t1.name < t2.name end)
	for index, user in ipairs(users) do
		DEFAULT_CHAT_FRAME:AddMessage(user.name.." "..L["using version"].." "..user.version)
	end
	del(users)

	DEFAULT_CHAT_FRAME:AddMessage("|c00"..COLOR_CODE..self.name.."|r - "..L["raid healers NOT using IncomingHealsLib"]..":")

	for u in Roster:IterateRoster() do
		if (u.class == "PRIEST" or u.class == "DRUID" or u.class == "PALADIN" or u.class == "SHAMAN") and not IHL.knownUsers[u.name] then
			DEFAULT_CHAT_FRAME:AddMessage(u.name)
		end
	end
end














function IncomingHeals:MixColors(c1, c2, ratio)
	return c1.r * (1 - ratio) + c2.r * ratio, c1.g * (1 - ratio) + c2.g * ratio, c1.b * (1 - ratio) + c2.b * ratio
end













------------------------------
--      Initialization      --
------------------------------

function IncomingHeals:OnInitialize()

	self:RegisterDB("IncomingHealsDBPerChar")
	self:RegisterDefaults("profile", {
		checkheals = false,
		graphs = { },
	})

	if AceLibrary("AceEvent-2.0"):IsFullyInitialized() then
		self:OnFullyInitialized()
	else
		self:RegisterEvent("AceEvent_FullyInitialized", "OnFullyInitialized")
	end



	self.initialized = true

	IHL.version = IHL.version.." - IH version "..self.version

end

function IncomingHeals:OnFullyInitialized()
	self:RegisterEvent("IncomingHealsLib_HealStart", "UpdateVisuals")
	self:RegisterEvent("IncomingHealsLib_HealCancel", "UpdateVisuals")
	self:RegisterEvent("IncomingHealsLib_HealEnd", "UpdateVisuals")
	self:RegisterEvent("IncomingHealsLib_ResStart", "UpdateVisuals")
	self:RegisterEvent("IncomingHealsLib_ResCancel", "UpdateVisuals")
	self:RegisterEvent("IncomingHealsLib_ResEnd", "UpdateVisuals")
	self:RegisterEvent("IncomingHealsLib_HotStart", "UpdateVisuals")
	self:RegisterEvent("IncomingHealsLib_HotEnd", "UpdateVisuals")
	self:RegisterEvent("IncomingHealsLib_Swiftmend", "UpdateVisuals")
	self:RegisterBucketEvent("UNIT_HEALTH", 0.2, "OnUnitHealth") -- srf_UpdateUnit does this too

	self:RegisterChatCommand({"/ih", "/incomingheals"}, self.options)

	if sRaidFrames then
		self:SecureHook(sRaidFrames, "UpdateUnitDetails", "srf_UpdateUnit")
		self:SecureHook(sRaidFrames, "UpdateUnitHealth", "srf_UpdateUnit")
	end

	if CT_RA_UnitIDFrameMap then
		self:SecureHook("CT_RA_UpdateRaidGroup", "CT_RA_UpdateRaidGroup")
		self:SecureHook("CT_RA_UpdateUnitStatus", "CT_RA_UpdateUnitStatus")
		self:SecureHook("CT_RA_UpdateUnitDead", "CT_RA_UpdateUnitDead")
	end

	if XPerl_Raid_UpdateDisplay then
		self.XPerlFrames = { }
		self:SecureHook("XPerl_Raid_UpdateDisplay", "XPerl_Raid_UpdateDisplay")
		XPerl_Raid_UpdateDisplayAll()
	end

	self.Graph.Initialize()

	self:ScheduleEvent(DEFAULT_CHAT_FRAME.AddMessage, STARTUP_MESSAGE_DELAY, DEFAULT_CHAT_FRAME, "|c00"..COLOR_CODE..self.name.."|r "..L["version"].." "..self.version.." "..L["initialized"])
end







function IncomingHeals:OnEnable()
	if not self.initialized then
		self:OnInitialize()
	end

	self.enabled = true
end

function IncomingHeals:OnDisable()
	self.enabled = false
	self:Reset()
end
