
-----------------------------
--     SCANNING MODULE     --
-----------------------------

-- Create module table
InTheBuff.Scan = { };
local Scan = InTheBuff.Scan;

-- temporary variables used during scans
Scan.timer = nil;				-- timer checking scan timeouts
Scan.scanning_unit = nil;		-- current unit being scanned
Scan.scan_start = nil;			-- time current scan began
Scan.notify_start = nil;		-- time current unit was notified for inspect

-- scan = {
--		name = "scan name",
--		units = {"raid1", "raid2", "raid3", ... },
--		
--		Process = function (scan, unit, error) to call on INSPECT_TALENT_READY, passes current scan, unit string, and error string (or nil if no error)
--		Finished = function (scan) to call when finished, passes current scan
--		
--		results = { },		the results of the scan, filled in as it is run
-- }



-- Create a new scan table
function Scan:NewScan(name, process, finished, units)

	-- check arguments
	if type(name) ~= "string" then return nil, "Scan name must be a string."; end
	if type(process) ~= "function" then return nil, "Scan process must be a function."; end
	-- if type(finished) ~= "function" then return nil, "Scan finished must be a function."; end
	if type(units) == "string" then units = {units}; end	-- allow single-unit calls
	if type(units) ~= "table" then return nil, "Scan units must be a string or table."; end
	if ( #units < 1 ) then return nil, "Scan must cover at least one unit."; end
	
	local scan = { };
	scan.name = name;
	scan.Process = process;
	scan.Finished = finished;
	scan.units = units;

Scan.units_total = 0;
Scan.units_OOR = 0;
Scan.units_OOR_list = { };

	return scan;
	
end



-- Check whether a target is in range or not
local function UnitInRange(unit) return CheckInteractDistance(unit, 1) or false; end


-- Checks the timeouts on the current scan
local function CheckScanTimeouts()
	
	local self = Scan;
	if not self.scan then return; end
	
	local now = GetTime();
	
	-- interrupt current scan if it exceeds 60 seconds
	if ( ( now - self.scan_start ) > 60 ) then self:FinishScan(); return; end
	
	-- interrupt current inspect wait if it exceeds 5 seconds
	if self.notify_start and ( ( now - self.notify_start ) > 5 ) then 
		self.scan:Process(self.scanning_unit, "Inspect timed out.");
		self:ScanNext();
	end
	
end


-- Event handler that calls the 'process' function for the current scan
local function TalentsReady()

	local scan = Scan.scan;

	-- Abort if we did not just scan a unit (or don't know what it was)
	if not Scan.scanning_unit then return; end
	
	-- Abort if no scan in progress
	if not scan then return; end

	-- Call the process function on the just-scanned unit and clear it
	scan:Process(Scan.scanning_unit);

	Scan.scanning_unit = nil;

	-- Move on to next unit
	Scan:ScanNext();
	
end


-- Run a scan
function Scan:RunScan(scan)
	
	-- Abort if scan in progress
	if self.scan then return nil, "Cannot perform multiple scans simultaneously."; end
	
	-- Abort if player is dead
	if UnitIsDead("player") then return nil, "Cannot perform scans while dead."; end
	
	-- Abort if no units to scan
	if ( #scan.units < 1 ) then return nil, "Scan must cover at least one unit."; end
	
	-- Set scan
	self.scan = scan;

	-- Register for talent and timeout events
	InTheBuff:RegisterEvent("INSPECT_TALENT_READY", TalentsReady);
	self.timer = InTheBuff:ScheduleRepeatingTimer(CheckScanTimeouts, 1);
	
	-- Set up first unit and go
	self.scan_start = GetTime();
	self:ScanNext();
	
end


-- Scan the next unit
function Scan:ScanNext()

	-- Abort if no scan in progress
	local scan = self.scan;
	if not scan then return; end
	
	-- Clear previous unit
	self.scanning_unit = nil;
	
	-- If no more units left, finish scan
	if ( #scan.units == 0 ) then return self:FinishScan(); end
	
	-- Pop the next unit off of the unit list
	self.scanning_unit = tremove(scan.units);
	local unit = self.scanning_unit;
	
	-- Check that unit exists
	if not UnitExists(unit) then
		scan:Process(unit, "Unit does not exist.");
		return self:ScanNext();
	end
	
	-- Ensure unit is a player
	if not UnitIsPlayer(unit) then
		scan:Process(unit, "Unit is not a player.");
		return self:ScanNext();
	end
	
	-- Check range
	if not UnitInRange(unit) then
	        Scan.units_OOR = Scan.units_OOR + 1;

		local name = UnitName(unit);
		local _, class = UnitClass(unit);
		local color;
		-- temporary for live due to no DK's
		if class == "DEATH KNIGHT" then
		   classcolor = "FF0000";
		else
		   classcolor = format("%02X%02X%02X", 255*RAID_CLASS_COLORS[class].r, 255*RAID_CLASS_COLORS[class].g, 255*RAID_CLASS_COLORS[class].b);
		end
		name = "|cFF" .. classcolor .. name .. "|r";

		tinsert(Scan.units_OOR_list, name);

		scan:Process(unit, "Unit out of range.");
		return self:ScanNext();
	end
	
	-- Check hostility
	if UnitCanAttack("player", unit) then
		scan:Process(unit, "Unit is hostile.");
		return self:ScanNext();
	end
	
	-- Skip the inspect on yourself, just record talents
	if UnitIsUnit("player", unit) then
		Scan.units_total = Scan.units_total + 1;
		scan:Process(unit);
		return self:ScanNext();
	end
	
	-- Notify inspection
	Scan.units_total = Scan.units_total + 1;
	self.notify_start = GetTime();
	NotifyInspect(unit);

end


-- Finish a scan
function Scan:FinishScan()
	-- unregister events
	InTheBuff:UnregisterEvent("INSPECT_TALENT_READY");
	InTheBuff:CancelTimer(self.timer, true);
	
	-- if scan is nil, we're done
	if not self.scan then return; end
	
	-- abort any remaining units
	if self.scanning_unit then self.scan:Process(self.scanning_unit, "Scan aborted."); end
	for _, unit in ipairs(self.scan.units) do self.scan:Process(unit, "Scan aborted."); end

	-- finish the scan
	local scan = self.scan;
	self.scan = nil;
	
	if ( type(scan.Finished) == "function" ) then scan:Finished(); end

end


------------------------------
--     TALENT RECORDING     --
------------------------------

-- Store talent information once it becomes available
local function RecordTalents(scan, unit, err)
	-- don't bother with nonexistant units
	if not UnitExists(unit) then return; end

	-- work properly if we are the current target
	local inspect = 1;
	if UnitIsUnit(unit,"player") then
	   inspect = 0;
	end

	-- create scan result table if not found
	scan.results = scan.results or { };
	
	-- set up default table values
	local result = {
		name = UnitName(unit),
		class = select(2, UnitClass(unit)),
		level = UnitLevel(unit),
	};
	
	-- error check
	if err then
		result.err = err;
		tinsert(scan.results, result);
		return;
	end
	-- for each talent tab
	for tab = 1, GetNumTalentTabs(inspect) do
		
		-- record tab
		local _, _, spent = GetTalentTabInfo(tab, inspect);
		result[tab] = {spent=spent};

		-- record talent choices
--		for talent = 1, GetNumTalents(tab, inspect) do
		for talent = 1, 51 do
			result[tab][talent] = select(5, GetTalentInfo(tab, talent, inspect));
		end
	
	end

	tinsert(scan.results, result);
end


-- Handle the results of a scan once it is complete
local function ProcessScan(scan)

	if not scan.results then return; end

	for _, unit in ipairs(scan.results) do
	    local color = format("%02X%02X%02X", 255*RAID_CLASS_COLORS[unit.class].r, 255*RAID_CLASS_COLORS[unit.class].g, 255*RAID_CLASS_COLORS[unit.class].b);
	    if unit.err then 
--	       InTheBuff:Print(format("|cFF%s%s|r %d: ?/?/? (%s)", color, unit.name, unit.level, unit.err)); 
             else 
--               InTheBuff:Print(format("|cFF%s%s|r %d: %d/%d/%d", color, unit.name, unit.level, unit[1].spent, unit[2].spent, unit[3].spent)); 
	       Scan:AnalyzeUnit(unit);
             end 
        end

	InTheBuff.Report:Show();
end

function Scan:AnalyzeUnit(unit)

	 local buffs   = InTheBuff.buffs;
	 local results = InTheBuff.results;

	 for class, classbuffs in pairs(buffs) do
	     local success, result;
	     if unit.class == class then
	     	for _, details in ipairs(classbuffs) do

		    -- run check in protected mode
		    if details.check then
    		      success, result = pcall(details.check, unit);
		    else
		      InTheBuff:Print("ERROR!  Check not defined for", unit.name, details.ability);
		    end

		    -- catch errors
    		    if not success then
		       if result then
        	       	  InTheBuff:Print("|cFFFF0000Error|r executing talent check |cFFFFFF00["..details.name.."]|r Please report to Jynxx: "..result);
		       else
			  InTheBuff:Print("failure but no result");
		       end
		    else
			if result then
			   results[details.name] = results[details.name] or { };
			   tinsert(results[details.name], {player=unit.name, class=class, type=details.type, category=details.category, ability=details.ability, value=details.value, tree=details.tree});
			end
		    end
   
		end
	     end	     		     
	 end
end


-----------------------------
--     USAGE FUNCTIONS     --
-----------------------------

-- Scan the current target
function InTheBuff:InitiateTargetScan()
	
	if not UnitExists("target") then
		self:Print("You have no target");
		return;
	end
	
	if not UnitInRange("target") then
		self:Print("You are too far away");
		return;
	end

	-- reset any previous results
	InTheBuff.results = nil;
	InTheBuff.results = { };

	_, err = Scan:RunScan(Scan:NewScan("Target Scan", RecordTalents, ProcessScan, "target"));
	
	-- if err then self:Print(err); end

end


-- Scan the current target
function InTheBuff:InitiatePartyScan()
	
	-- abort unless we are in a party
	if ( GetNumPartyMembers() == 0 ) then
		self:Print("You are not in a group!");
		return;
	end
	
	-- generate unit list
	local units = { };
	for i = 1, 4 do
		if UnitExists("party"..i) then tinsert(units, "party"..i); end
	end
	
	-- reset any previous results
	InTheBuff.results = nil;
	InTheBuff.results = { };

	_, err = Scan:RunScan(Scan:NewScan("Party Scan", RecordTalents, ProcessScan, units));
	
	if err then self:Print(err); end
	
end


function InTheBuff:InitiateRaidScan10()
	 local groups = { true, true };
	 InTheBuff:InitiateRaidScan(groups);
end

function InTheBuff:InitiateRaidScan25()
	 local groups = { true, true, true, true, true };
	 InTheBuff:InitiateRaidScan(groups);
end

function InTheBuff:InitiateRaidScan40()
	 local groups = { true, true, true, true, true, true, true, true };
	 InTheBuff:InitiateRaidScan(groups);
end

function InTheBuff:InitiateGroupScan34()
	 local groups = { false, false, true, true };
	 InTheBuff:InitiateRaidScan(groups);
end

function InTheBuff:InitiateGroupScan56()
	 local groups = { false, false, false, false, true, true };
	 InTheBuff:InitiateRaidScan(groups);
end

function InTheBuff:InitiateGroupScan78()
	 local groups = { false, false, false, false, false, false, true, true };
	 InTheBuff:InitiateRaidScan(groups);
end


-- Scan the current target
function InTheBuff:InitiateRaidScan(selected_groups)

	-- abort unless we are in a raid
	if ( GetNumRaidMembers() == 0 ) then
		self:Print("You are not in a raid group!");
		return;
	end

	-- generate unit list
	local units = { };
	for i = 1, 40 do
	        local num = select(3, GetRaidRosterInfo(i));
	      	if selected_groups[num] then
		  if UnitExists("raid"..i) then tinsert(units, "raid"..i); end
		end
	end

	-- reset any previous results
	InTheBuff.results = nil;
	InTheBuff.results = { };
	
	_, err = Scan:RunScan(Scan:NewScan("Raid Scan", RecordTalents, ProcessScan, units));

	if err then self:Print(err); end

end
