local AceOO = AceLibrary("AceOO-2.0")

local MobHealth = nil
local FocusHealth = AceOO.Class(IceUnitBar)

FocusHealth.prototype.color = nil


-- Constructor --
function FocusHealth.prototype:init()
	FocusHealth.super.prototype.init(self, "FocusHealth", "focus")
	
	self:SetDefaultColor("FocusHealthHostile", 231, 31, 36)
	self:SetDefaultColor("FocusHealthFriendly", 46, 223, 37)
	self:SetDefaultColor("FocusHealthNeutral", 210, 219, 87)

	MobHealth = AceLibrary:HasInstance("LibMobHealth-4.0") and AceLibrary("LibMobHealth-4.0") or nil
end


function FocusHealth.prototype:GetDefaultSettings()
	local settings = FocusHealth.super.prototype.GetDefaultSettings(self)

	settings["enabled"] = false
	settings["side"] = IceCore.Side.Right
	settings["offset"] = -1
	settings["scale"] = 0.7
	settings["mobhealth"] = (MobHealth3 ~= nil)
	settings["classColor"] = false
--	settings["hideBlizz"] = true
	settings["upperText"] = "[PercentHP:Round]"
	settings["lowerText"] = ""
	settings["raidIconOnTop"] = true
	settings["showRaidIcon"] = true
	settings["raidIconXOffset"] = 12
	settings["raidIconYOffset"] = 0
	settings["lockIconAlpha"] = false
	settings["abbreviateHealth"] = true
	settings["barVerticalOffset"] = 35

	return settings
end


-- OVERRIDE
function FocusHealth.prototype:GetOptions()
	local opts = FocusHealth.super.prototype.GetOptions(self)

	opts["mobhealth"] = {
		type = "toggle",
		name = "MobHealth3 support",
		desc = "Enable/disable MobHealth3 Focus HP data. If this option is gray, you do not have MobHealth3.",
		get = function()
			return self.moduleSettings.mobhealth
		end,
		set = function(value)
			self.moduleSettings.mobhealth = value
			self:Update(self.unit)
		end,
		disabled = function()
			return (not self.moduleSettings.enabled) and (MobHealth3 == nil)
		end,
		order = 40
	}

	opts["classColor"] = {
		type = "toggle",
		name = "Class color bar",
		desc = "Use class color as the bar color instead of reaction color",
		get = function()
			return self.moduleSettings.classColor
		end,
		set = function(value)
			self.moduleSettings.classColor = value
			self:Update(self.unit)
		end,
		disabled = function()
			return not self.moduleSettings.enabled
		end,
		order = 41
	}
--[[
	opts["hideBlizz"] = {
		type = "toggle",
		name = "Hide Blizzard Frame",
		desc = "Hides Blizzard Focus frame and disables all events related to it",
		get = function()
			return self.moduleSettings.hideBlizz
		end,
		set = function(value)
			self.moduleSettings.hideBlizz = value
			if (value) then
				self:HideBlizz()
			else
				self:ShowBlizz()
			end
		end,
		disabled = function()
			return not self.moduleSettings.enabled
		end,
		order = 42
	}
]]
	opts["scaleHealthColor"] = {
		type = "toggle",
		name = "Color bar by health %",
		desc = "Colors the health bar from MaxHealthColor to MinHealthColor based on current health %",
		get = function()
			return self.moduleSettings.scaleHealthColor
		end,
		set = function(value)
			self.moduleSettings.scaleHealthColor = value
			self:Redraw()
		end,
		disabled = function()
			return not self.moduleSettings.enabled
		end,
		order = 43
	}

	opts["showRaidIcon"] = {
		type = "toggle",
		name = "Show Raid Icon",
		desc = "Whether or not to show the raid icon above this bar",
		get = function()
			return self.moduleSettings.showRaidIcon
		end,
		set = function(value)
			self.moduleSettings.showRaidIcon = value
			self:UpdateRaidFocusIcon()
		end,
		disabled = function()
			return not self.moduleSettings.enabled
		end,
		order = 50
	}

	opts["lockIconAlpha"] = {
		type = "toggle",
		name = "Lock raid icon to 100% alpha",
		desc = "With this enabled, the raid icon is always 100% alpha, regardless of the bar's alpha. Otherwise, it assumes the bar's alpha level.",
		get = function()
			return self.moduleSettings.lockIconAlpha
		end,
		set = function(value)
			self.moduleSettings.lockIconAlpha = value
			self:Redraw()
		end,
		disabled = function()
			return not self.moduleSettings.enabled
		end,
		order = 51
	}

	opts["raidIconOnTop"] = {
		type = "toggle",
		name = "Draw Raid Icon On Top",
		desc = "Whether to draw the raid icon in front of or behind this bar",
		get = function()
			return self.moduleSettings.raidIconOnTop
		end,
		set = function(value)
			self.moduleSettings.raidIconOnTop = value
			self:UpdateRaidFocusIcon()
		end,
		disabled = function()
			return not self.moduleSettings.enabled or not self.moduleSettings.showRaidIcon
		end,
		order = 52
	}

	opts["raidIconXOffset"] = {
		type = "range",
		name = "Raid Icon X Offset",
		desc = "How far to push the raid icon right or left",
		min = -300,
		max = 300,
		step = 1,
		get = function()
			return self.moduleSettings.raidIconXOffset
		end,
		set = function(value)
			self.moduleSettings.raidIconXOffset = value
			self:SetRaidIconPlacement()
			self:Redraw()
		end,
		disabled = function()
			return not self.moduleSettings.enabled
		end,
		order = 53
	}

	opts["raidIconYOffset"] = {
		type = "range",
		name = "Raid Icon Y Offset",
		desc = "How far to push the raid icon up or down",
		min = -300,
		max = 300,
		step = 1,
		get = function()
			return self.moduleSettings.raidIconYOffset
		end,
		set = function(value)
			self.moduleSettings.raidIconYOffset = value
			self:SetRaidIconPlacement()
			self:Redraw()
		end,
		disabled = function()
			return not self.moduleSettings.enabled
		end,
		order = 54
	}

	opts["shortenHealth"] = {
		type = 'toggle',
		name = 'Abbreviate estimated health',
		desc = 'If this is checked, then a health value of 1100 will display as 1.1k, otherwise it shows the number\n\nNote: this only applies if you are NOT using DogTag',
		get = function()
			return self.moduleSettings.abbreviateHealth
		end,
		set = function(v)
			self.moduleSettings.abbreviateHealth = v
		end,
		disabled = function()
			return not self.moduleSettings.enabled
		end,
		order = 40.1
	}

	return opts
end


function FocusHealth.prototype:Enable(core)
	FocusHealth.super.prototype.Enable(self, core)

	self:RegisterEvent("UNIT_HEALTH", "Update")
	self:RegisterEvent("UNIT_MAXHEALTH", "Update")
	self:RegisterEvent("UNIT_FLAGS", "Update")
	self:RegisterEvent("UNIT_FACTION", "Update")
	self:RegisterEvent("RAID_TARGET_UPDATE", "UpdateRaidFocusIcon")
	self:RegisterEvent("PLAYER_FOCUS_CHANGED", "UpdateFocus")

--	if (self.moduleSettings.hideBlizz) then
--		self:HideBlizz()
--	end

	self:CreateRaidIconFrame()

	self:Update(self.unit)
end

function FocusHealth.prototype:UpdateFocus()
	self:Update(self.unit)
end

function FocusHealth.prototype:Disable(core)
	FocusHealth.super.prototype.Disable(self, core)
end



function FocusHealth.prototype:Update(unit)
	FocusHealth.super.prototype.Update(self)
	if (unit and (unit ~= self.unit)) then
		return
	end

	if not (UnitExists(unit)) then
		self:Show(false)
		return
	else	
		self:Show(true)
	end

	self:UpdateRaidFocusIcon()

	self.color = "FocusHealthFriendly" -- friendly > 4

	local reaction = UnitReaction(self.unit, "player")
	if (reaction and (reaction == 4)) then
		self.color = "FocusHealthNeutral"
	elseif (reaction and (reaction < 4)) then
		self.color = "FocusHealthHostile"
	end
	
	if (self.moduleSettings.classColor) then
		self.color = self.unitClass
	end

	if (self.moduleSettings.scaleHealthColor) then
		self.color = "ScaledHealthColor"
	end

	if (self.tapped) then
		self.color = "Tapped"
	end

	self:UpdateBar(self.health/self.maxHealth, self.color)

	if not AceLibrary:HasInstance("LibDogTag-3.0") then
		self:SetBottomText1(math.floor(self.healthPercentage * 100))

		-- first see if we have LibMobHealth that we can piggyback on
		if MobHealth then
			self.health = MobHealth:GetUnitCurrentHP(self.unit)
			self.maxHealth = MobHealth:GetUnitMaxHP(self.unit)
		elseif (self.maxHealth == 100 and self.moduleSettings.mobhealth and MobHealth3) then
			-- assumption that if a unit's max health is 100, it's not actual amount
			-- but rather a percentage - this obviously has one caveat though

			self.health, self.maxHealth, _ = MobHealth3:GetUnitHealth(self.unit, self.health, self.maxHealth)
		end

		if self.moduleSettings.abbreviateHealth then
			self.health = self:Round(self.health)
			self.maxHealth = self:Round(self.maxHealth)
		end

		if (self.maxHealth ~= 100) then
			self:SetBottomText2(self:GetFormattedText(self.health, self.maxHealth), self.color)
		else
			self:SetBottomText2()
		end
	end

	if self.frame.raidIcon then
		self.frame.raidIcon:SetAlpha(self.moduleSettings.lockIconAlpha and 1 or self.alpha)
	end
end


function FocusHealth.prototype:CreateRaidIconFrame()
	if (not self.frame.raidIcon) then
		self.frame.raidIcon = CreateFrame("Frame", nil, self.frame)
	end

	if (not self.frame.raidIcon.icon) then
		self.frame.raidIcon.icon = self.frame.raidIcon:CreateTexture(nil, "BACKGROUND")
		self.frame.raidIcon.icon:SetTexture("Interface\\TargetingFrame\\UI-RaidTargetingIcons")
	end

	self:SetRaidIconPlacement()
	self.frame.raidIcon:SetWidth(16)
	self.frame.raidIcon:SetHeight(16)

	self.frame.raidIcon.icon:SetAllPoints(self.frame.raidIcon)
	SetRaidTargetIconTexture(self.frame.raidIcon.icon, 0)
	self.frame.raidIcon:Hide()
end

function FocusHealth.prototype:SetRaidIconPlacement()
	self.frame.raidIcon:ClearAllPoints()
	self.frame.raidIcon:SetPoint("BOTTOM", self.frame, "TOPLEFT", self.moduleSettings.raidIconXOffset, self.moduleSettings.raidIconYOffset)
end


function FocusHealth.prototype:UpdateRaidFocusIcon()
	if self.moduleSettings.raidIconOnTop then
		self.frame.raidIcon:SetFrameStrata("MEDIUM")
	else
		self.frame.raidIcon:SetFrameStrata("LOW")
	end

	if not (UnitExists(self.unit)) or not self.moduleSettings.showRaidIcon then
		self.frame.raidIcon:Hide()
		return
	end

	local index = GetRaidTargetIndex(self.unit);

	if (index and (index > 0)) then
		SetRaidTargetIconTexture(self.frame.raidIcon.icon, index)
		self.frame.raidIcon:Show()
	else
		self.frame.raidIcon:Hide()
	end
end


function FocusHealth.prototype:Round(health)
	if (health > 1000000) then
		return self:MathRound(health/1000000, 1) .. "M"
	end
	if (health > 1000) then
		return self:MathRound(health/1000, 1) .. "k"
	end
	return health
end


function FocusHealth.prototype:MathRound(num, idp)
	local mult = 10^(idp or 0)
	return math.floor(num  * mult + 0.5) / mult
end

--[[
function FocusHealth.prototype:ShowBlizz()
	FocusFrame:Show()
	FocusFrame:RegisterEvent("PLAYER_TARGET_CHANGED")
	FocusFrame:RegisterEvent("UNIT_HEALTH")
	FocusFrame:RegisterEvent("UNIT_LEVEL")
	FocusFrame:RegisterEvent("UNIT_FACTION")
	FocusFrame:RegisterEvent("UNIT_CLASSIFICATION_CHANGED")
	FocusFrame:RegisterEvent("UNIT_AURA")
	FocusFrame:RegisterEvent("PLAYER_FLAGS_CHANGED")
	FocusFrame:RegisterEvent("PARTY_MEMBERS_CHANGED")
	FocusFrame:RegisterEvent("RAID_TARGET_UPDATE")
end


function FocusHealth.prototype:HideBlizz()
	FocusFrame:Hide()
	FocusFrame:UnregisterAllEvents()
	end
]]

-- Load us up
IceHUD.FocusHealth = FocusHealth:new()
