local MAX_TABS = 6;

local LS = AceLibrary("AceLocale-2.2"):new("Historian-Shared");
local f = AceLibrary("HistorianFormatting-1.0");
local abacus = AceLibrary("Abacus-2.0");

-- Tabs
HISTORIANGUI_FRAME_HABITS = LS["STAT_CATEGORY_HABITS"];
HISTORIANGUI_FRAME_ACHIEVEMENTS = LS["STAT_CATEGORY_ACHIEVEMENTS"];
HISTORIANGUI_FRAME_DISAPPOINTMENTS = LS["STAT_CATEGORY_DISAPPOINTMENTS"];
HISTORIANGUI_FRAME_FORTUNE = LS["STAT_CATEGORY_FORTUNE"];
HISTORIANGUI_FRAME_KILLS = LS["STAT_CATEGORY_KILL_DETAILS"];
HISTORIANGUI_FRAME_SPELLS = LS["STAT_CATEGORY_SPELL_DETAILS"];
HISTORIANGUI_FRAME_LEVELS = LS["STAT_CATEGORY_LEVEL_DETAILS"];

-- Headers
HISTORIANGUI_HEADER_ITEMS_LOOTED = LS["STAT_CATEGORY_ITEMS_LOOTED"];

-- Habits tab labels
HISTORIANGUI_LABEL_FOLLOWED = LS["STAT_MOST_FOLLOWED"];
HISTORIANGUI_LABEL_FAVORITEPREY = LS["STAT_FAVORITE_PREY"];
HISTORIANGUI_LABEL_FAVORITESPELL = LS["STAT_FAVORITE_SPELL"];
HISTORIANGUI_LABEL_TOTALSPELLS = LS["STAT_TOTAL_SPELLS"];
HISTORIANGUI_LABEL_JUMPS = LS["STAT_JUMPS"];
HISTORIANGUI_LABEL_SNACKS = LS["STAT_FOOD"];
HISTORIANGUI_LABEL_DRINKS = LS["STAT_DRINK"];
HISTORIANGUI_LABEL_TIPSY = LS["STAT_TIPSY"];
HISTORIANGUI_LABEL_DRUNK = LS["STAT_DRUNK"];
HISTORIANGUI_LABEL_SMASHED = LS["STAT_SMASHED"];


-- Achievements tab labels
HISTORIANGUI_LABEL_QUESTS = LS["STAT_QUESTS_COMPLETED"];
HISTORIANGUI_LABEL_RECIPES = LS["STAT_RECIPES_LEARNED"];
HISTORIANGUI_LABEL_AREAS = LS["STAT_AREAS_DISCOVERED"];
HISTORIANGUI_LABEL_KILLS = LS["STAT_TOTAL_KILLS"];


-- Disappointments tab labels
HISTORIANGUI_LABEL_ENV_DROWNING = LS["STAT_ENV_DROWNING"];
HISTORIANGUI_LABEL_ENV_FATIGUE = LS["STAT_ENV_FATIGUE"];
HISTORIANGUI_LABEL_ENV_FALL = LS["STAT_ENV_FALL"];
HISTORIANGUI_LABEL_ENV_FIRE = LS["STAT_ENV_FIRE"];
HISTORIANGUI_LABEL_ENV_LAVA = LS["STAT_ENV_LAVA"];
HISTORIANGUI_LABEL_ENV_SLIME = LS["STAT_ENV_SLIME"];
HISTORIANGUI_LABEL_DEATHS = LS["STAT_DEATHS"];


-- Fortune tab labels
HISTORIANGUI_LABEL_MAX_MONEY = LS["STAT_MAX_MONEY"]
HISTORIANGUI_LABEL_LOOTED_COMMON = LS["STAT_LOOTED_COMMON"];
HISTORIANGUI_LABEL_LOOTED_UNCOMMON = LS["STAT_LOOTED_UNCOMMON"];
HISTORIANGUI_LABEL_LOOTED_RARE = LS["STAT_LOOTED_RARE"];
HISTORIANGUI_LABEL_LOOTED_EPIC = LS["STAT_LOOTED_EPIC"];
HISTORIANGUI_LABEL_LOOTED_LEGENDARY = LS["STAT_LOOTED_LEGENDARY"];
HISTORIANGUI_LABEL_LOOTED_ARTIFACT = LS["STAT_LOOTED_ARTIFACT"];


local leftText = "";
local rightText = "";
local leftCell = nil;
local rightCell = nil;

-- *************************************
-- * Utility functions for data display
-- *************************************
local function GetNumberStat(valueLabel, number)
	if (number == nil) or (number < 1) then
		return LS["NOT_AVAILABLE"];
	end;

	if (valueLabel ~= nil) then
		return string.format(valueLabel, f:IntToStr(number));
	else
		return f:IntToStr(number);
	end;
end;

local function GetNameAndNumberStat(valueLabel, name, number)
	if (number == nil) or (number < 1) then
		return LS["NOT_AVAILABLE"];
	end;

	return string.format(valueLabel, name, f:IntToStr(number));
end;


HistorianGUI = AceLibrary("AceAddon-2.0"):new("AceEvent-2.0");


-- *************************************
-- * OnLoad
-- *************************************
function HistorianGUI:OnLoad()
	PanelTemplates_SetNumTabs(HistorianTabContainer, MAX_TABS);
	PanelTemplates_SetTab(HistorianTabContainer, 1);
	self.currentTab = 1;
	
	-- Enable closing of the window using ESC
	tinsert(UISpecialFrames,"HistorianTabContainer");
end;


-- *************************************
-- * OnShow
-- *************************************
function HistorianGUI:OnShow()
	HistorianTabContainer.IsShowing = true;
	PlaySound("UChatScrollButton");
	
	self:ReloadCurrentTab();
	self:RegisterEvent("Historian_Updated", "OnStatsUpdated");
end;


-- *************************************
-- * ReloadCurrentTab
-- *************************************
function HistorianGUI:ReloadCurrentTab()
	self:ShowTab(self.currentTab);
end;


-- *************************************
-- * OnStatsUpdated
-- *************************************
function HistorianGUI:OnStatsUpdated()
	self:ReloadCurrentTab();
end;


-- *************************************
-- * OnHide
-- *************************************
function HistorianGUI:OnHide()
	HistorianTabContainer.IsShowing = false;
	PlaySound("UChatScrollButton");
end;


-- *************************************
-- * OnHeaderMouseDown
-- *************************************
function HistorianGUI:OnHeaderMouseDown()
	local frame = this:GetParent();
	
	if (arg1 == "LeftButton") then
		frame:StartMoving();
	elseif (arg1 == "RightButton") then
		frame:ClearAllPoints();
		frame:SetPoint("CENTER", "UIParent", "CENTER");
	end
end;


-- *************************************
-- * OnHeaderMouseUp
-- *************************************
function HistorianGUI:OnHeaderMouseUp()
	local frame = this:GetParent();
	frame:StopMovingOrSizing();
end;


-- *************************************
-- * ShowTab
-- *************************************
function HistorianGUI:ShowTab(tabID)
	PanelTemplates_SetTab(HistorianTabContainer, tabID);

	for i = 1, MAX_TABS do
		local tab = getglobal("HistorianTabPage" .. i);
		if (tab ~= nil) then
			if (i == tabID) then
				tab:Show();
			else
				tab:Hide();
			end;
		end;
	end;
	
	self.currentTab = tabID;
	self:ShowContent(tabID);
end;


-- *************************************
-- * OnTabClicked
-- *************************************
function HistorianGUI:OnTabClicked(tabID)
	self:ShowTab(tabID);
end;



-- *************************************
-- * ShowContent
-- *************************************
function HistorianGUI:ShowContent(tabID)
	if (tabID == 1) then
		self:ShowHabits();
	elseif (tabID == 2) then
		self:ShowAchievements();
		self:ShowFortune();
	elseif (tabID == 3) then
		self:ShowDisappointments();
	elseif (tabID == 4) then
		self:ShowKills();
	elseif (tabID == 5) then
		self:ShowSpells();
	elseif (tabID == 6) then
		self:ShowLevels();
	end;
end;


-- *************************************
-- * ShowHabits
-- *************************************
function HistorianGUI:ShowHabits(leftCell, rightCell)
	local data = Historian:GetHabitStats();
	
	HistorianGUI_Value_MostFollowed:SetText(GetNameAndNumberStat(LS["STAT_MOST_FOLLOWED_VALUE"], data.MostFollowed.Name, data.MostFollowed.Count));
	HistorianGUI_Value_FavoritePrey:SetText(GetNameAndNumberStat(LS["STAT_FAVORITE_PREY_VALUE"], data.FavoritePrey.Name, data.FavoritePrey.Count));
	HistorianGUI_Value_FavoriteSpell:SetText(GetNameAndNumberStat(LS["STAT_FAVORITE_SPELL_VALUE"], data.FavoriteSpell.Name, data.FavoriteSpell.Count));
	
	HistorianGUI_Value_Jumps:SetText(GetNumberStat(nil, data.Jumps));

	HistorianGUI_Value_Snacks:SetText(GetNumberStat(LS["STAT_FOOD_VALUE"], data.Food));
	HistorianGUI_Value_Drinks:SetText(GetNumberStat(LS["STAT_DRINK_VALUE"], data.Drink));

	HistorianGUI_Value_Tipsy:SetText(GetNumberStat(LS["STAT_TIPSY_VALUE"], data.Tipsy));
	HistorianGUI_Value_Drunk:SetText(GetNumberStat(LS["STAT_DRUNK_VALUE"], data.Drunk));
	HistorianGUI_Value_Smashed:SetText(GetNumberStat(LS["STAT_SMASHED_VALUE"], data.Smashed));
	
	HistorianTabPage1:UpdateScrollChildRect();
end;


-- *************************************
-- * ShowAchievements
-- *************************************
function HistorianGUI:ShowAchievements()
	local data = Historian:GetAchievementStats();
	
	HistorianGUI_Value_Quests:SetText(GetNumberStat(nil, data.QuestsCompleted));
	HistorianGUI_Value_Recipes:SetText(GetNumberStat(nil, data.RecipesLearned));
	HistorianGUI_Value_AreasDiscovered:SetText(GetNumberStat(nil, data.AreasDiscovered));
	HistorianGUI_Value_SpellsCast:SetText(GetNumberStat(nil, data.TotalSpellsCast));
	HistorianGUI_Value_TotalKills:SetText(GetNumberStat(nil, data.TotalKills));

	HistorianTabPage2:UpdateScrollChildRect();
end;



-- *************************************
-- * ShowDisappointments
-- *************************************
function HistorianGUI:ShowDisappointments()
	local data = Historian:GetDisappointmentStats();
	
	local function GetEnvDamageStat(times, totalDamage)
		if (times == nil) or (times < 1) then
			return LS["NOT_AVAILABLE"];
		end;

		local timesText = f:IntToStr(times);
		local damageText = f:IntToStr(totalDamage);

		if (times == 1) then
			return string.format(LS["STAT_ENV_VALUE_SINGLE"], damageText);
		else
			return string.format(LS["STAT_ENV_VALUE"], timesText, damageText);
		end;
	end;
	
	HistorianGUI_Value_Drowning:SetText(GetEnvDamageStat(data.Drowning.Times, data.Drowning.TotalDamage));
	HistorianGUI_Value_Fatigued:SetText(GetEnvDamageStat(data.Fatique.Times, data.Fatique.TotalDamage));
	HistorianGUI_Value_FallenDown:SetText(GetEnvDamageStat(data.Falling.Times, data.Falling.TotalDamage));
	HistorianGUI_Value_Burnt:SetText(GetEnvDamageStat(data.Fire.Times, data.Fire.TotalDamage));
	HistorianGUI_Value_Lava:SetText(GetEnvDamageStat(data.Lava.Times, data.Lava.TotalDamage));
	HistorianGUI_Value_Slime:SetText(GetEnvDamageStat(data.Slime.Times, data.Slime.TotalDamage));

	HistorianGUI_Value_Deaths:SetText(GetNumberStat(nil, data.TotalDeaths));

	HistorianTabPage3:UpdateScrollChildRect();
end;


-- *************************************
-- * ShowFortune
-- *************************************
function HistorianGUI:ShowFortune()
	local data = Historian:GetFortuneStats();

	HistorianGUI_Value_MoneyAtPeak:SetText(abacus:FormatMoneyFull(data.MostMoneyEver, true));
	
	HistorianGUI_Value_LootedCommon:SetText(GetNumberStat(nil, data.LootedCommon));
	HistorianGUI_Value_LootedUncommon:SetText(GetNumberStat(nil, data.LootedUncommon));
	HistorianGUI_Value_LootedRare:SetText(GetNumberStat(nil, data.LootedRare));
	HistorianGUI_Value_LootedEpic:SetText(GetNumberStat(nil, data.LootedEpic));
	HistorianGUI_Value_LootedLegendary:SetText(GetNumberStat(nil, data.LootedLegendary));
	HistorianGUI_Value_LootedArtifact:SetText(GetNumberStat(nil, data.LootedArtifact));
end;


-- *************************************
-- * SortStatsByCount
-- *************************************
local function SortStatsByCount(data, statsFormatFunc)
	-- Sorts the data in descending order based on the "Count" property returned by statsFormatFunc
	local result = {};
	
	for name, stats in pairs(data) do
		result[#result+1] = statsFormatFunc(name, stats);
	end
	
	local function sortFunc(val1, val2)
		-- Sorting in descending order for the value, ascending order for the name
		if (val1.Count == val2.Count) then
			return (val1.Name:lower() < val2.Name:lower()); 
		end;
		
		return (val2.Count < val1.Count);
	end;

	table.sort(result, sortFunc);

	return result;
end;


-- *************************************
-- * ShowKills
-- *************************************
function HistorianGUI:ShowKills()
	local parentFrameName = "HistorianTabPage4ChildFrame";
	local parentFrame = getglobal(parentFrameName);
	
	self:RemoveAllLabels(parentFrame);
	
	local function formatStats(name, stats)
		return { Name = name, Count = stats.Any, KB = stats.KillingBlows };
	end;
	
	local allKills = SortStatsByCount(Historian:GetAllKillStats(), formatStats);

	local noInfoAvailable = true;
	
	for index, stats in ipairs(allKills) do
		noInfoAvailable = false;
		
		local id = parentFrameName .. "_KillStat" .. tostring(index);
		local anchorID = parentFrameName .. "_Header";
		
		if (index > 1) then 
			anchorID = "$parent_KillStat" .. tostring(index - 1);
		end;

		if (index > 500) then
			-- 500 entries is enough - anything more is too slow to load :)
			break;
		end;
		
		self:CreateStatLabel(id, parentFrame, anchorID, tostring(index) .. ".  " .. stats.Name, f:IntToStr(stats.Count) .. " (" .. f:IntToStr(stats.KB) .. ")");
	end;
	
	if (noInfoAvailable) then
		self:CreateNotAvailableLabel(parentFrameName .. "NoDataAvailable", parentFrame, parentFrameName .. "_Header", LS["STAT_KILLS_NONE"]);
	end;
end;


-- *************************************
-- * ShowSpells
-- *************************************
function HistorianGUI:ShowSpells()
	local parentFrameName = "HistorianTabPage5ChildFrame";
	local parentFrame = getglobal(parentFrameName);
	
	self:RemoveAllLabels(parentFrame);
	
	local function formatStats(name, stats)
		return { Name = name, Count = stats.Succeeded };
	end;
	
	local allSpells = SortStatsByCount(Historian:GetAllSpellStats(), formatStats);

	local noInfoAvailable = true;
	
	for index, stats in ipairs(allSpells) do
		noInfoAvailable = false;
		
		local id = parentFrameName .. "_SpellStat" .. tostring(index);
		local anchorID = parentFrameName .. "_Header";
		
		if (index > 1) then 
			anchorID = "$parent_SpellStat" .. tostring(index - 1);
		end;

		self:CreateStatLabel(id, parentFrame, anchorID, tostring(index) .. ".  " .. stats.Name, f:IntToStr(stats.Count));
	end;
	
	if (noInfoAvailable) then
		self:CreateNotAvailableLabel(parentFrameName .. "NoDataAvailable", parentFrame, parentFrameName .. "_Header", LS["STAT_SPELLS_NONE"]);
	end;
end;


-- *************************************
-- * ShowLevels
-- *************************************
function HistorianGUI:ShowLevels()
	local parentFrameName = "HistorianTabPage6ChildFrame";
	local parentFrame = getglobal(parentFrameName);
	
	self:RemoveAllLabels(parentFrame);
	
	local levels = Historian:GetAllLevelStats();
	
	local index = 1;
	local last = 2;
	
	local noInfoAvailable = true;
	
	for level = 70, 2, -1 do
		local levelInfo = levels[level];
		if (levelInfo ~= nil) then
			noInfoAvailable = false;
			
			local id = parentFrameName .. "_LevelStat" .. tostring(index);
			local anchorID = parentFrameName .. "_Header";

			if (index > 1) then 
				anchorID = "$parent_LevelStat" .. tostring(index - 1);
			end;
			
			local area = levelInfo.Zone;
			if (levelInfo.SubZone ~= "") then
				area = area .. " - " .. levelInfo.SubZone;
			end;

			self:CreateStatLabel(id, parentFrame, anchorID, string.format(LS["STAT_LEVEL_X"], tostring(level)), "(" .. area .. ")  " .. levelInfo.Date);
			index = index + 1;
			
		end;
	end;
	
	if (noInfoAvailable) then
		self:CreateNotAvailableLabel(parentFrameName .. "NoDataAvailable", parentFrame, parentFrameName .. "_Header", LS["STAT_LEVELS_NONE"]);
	end;
end;


-- *************************************
-- * CreateNotAvailableLabel
-- *************************************
function HistorianGUI:CreateNotAvailableLabel(name, parent, anchoredTo, text)
	local frame = getglobal(name);
	local label = nil;
	
	if (frame == nil) then
		-- Creating frame from scratch
		frame = CreateFrame("Frame", name, parent, "HistorianHeaderFrame");
		label = frame:CreateFontString(name .. "_Label", "ARTWORK", "HistorianDataNotAvailable");
		frame:SetPoint("TOPLEFT", anchoredTo, "BOTTOMLEFT", 0, 0);
	else
		-- Reusing existing frame
		label = getglobal(name .. "_Label");
	end;
	
	label:SetText(text);

	frame:Show();
end;


-- *************************************
-- * CreateStatLabel
-- *************************************
function HistorianGUI:CreateStatLabel(name, parent, anchoredTo, label, value, onClick)
	-- Create a frame with two FontStrings: the label on the left and the value on the right.
	-- The frame may be clickable.
	local frame = getglobal(name);
	local labelObj = nil;
	local valueObj = nil;
	
	if (frame == nil) then
		-- Creating frame from scratch
		frame = CreateFrame("Frame", name, parent, "HistorianStatisticFrame");
		labelObj = frame:CreateFontString(name .. "_LeftCell", "ARTWORK", "HistorianDataLeftCellTemplate");
		valueObj = frame:CreateFontString(name .. "_RightCell", "ARTWORK", "HistorianDataRightCellTemplate");
		
		frame:SetScript("OnMouseUp", onClick);
		frame:SetPoint("TOPLEFT", anchoredTo, "BOTTOMLEFT", 0, 0);
	else
		-- Reusing existing frame
		labelObj = getglobal(name .. "_LeftCell");
		valueObj = getglobal(name .. "_RightCell");
	end;
	
	labelObj:SetText(label);
	valueObj:SetText(value);

	frame:Show();
end;


-- *************************************
-- * RemoveAllLabels
-- *************************************
function HistorianGUI:RemoveAllLabels(frame)
	-- Hide all labels created using CreateStatLabel.
	local children = { frame:GetChildren() };
	
	for _, child in ipairs(children) do
		if child:IsObjectType("Frame") then
			if (string.find(child:GetName(), "_Header") == nil) then
				child:Hide();
			end;
		end;
	end;
end;


-- *************************************
-- * OnEnter
-- * Called when the mouse is over an option
-- *************************************
function HistorianGUI:OnEnter(option, ...)
	if (option.highlight == nil) then
		local highlight = option:CreateTexture(nil, "BACKGROUND")
		highlight:SetTexture("Interface\\QuestFrame\\UI-QuestTitleHighlight")
		option.highlight = highlight
		highlight:SetBlendMode("ADD")
		highlight:SetAllPoints(option)
	end;
	
	option.highlight:Show();
end;


-- *************************************
-- * OnLeave
-- * Called when the mouse leaves an option
-- *************************************
function HistorianGUI:OnLeave(option, ...)
	if (option.highlight ~= nil) then
		option.highlight:Hide();
	end;
end;


-- *************************************
-- * InsertIntoChat
-- *************************************
function HistorianGUI:InsertIntoChat(...)
	if (ChatFrameEditBox == nil) then 
		return; 
	end;
	
	if not ChatFrameEditBox:IsVisible() then 
		ChatFrameEditBox:Show();
	end;
	
	ChatFrameEditBox:Insert(string.format(...) .." ");
end;


-- *************************************
-- * OnClick
-- * Called when the user clicks on an option
-- *************************************
function HistorianGUI:OnClick(option, mouseButton)
	if (mouseButton == "LeftButton") then
		--if (IsShiftKeyDown()) then
			local name = option:GetName();
			if (name ~= nil) and (string.len(name) > 29) then
				name = string.sub(name, 29);
				
				-- Habits
				if (name == "MostFollowed") then
					self:InsertIntoChat(LS["CHAT_INSERT_MOST_FOLLOWED"], Historian.db.char.Follow.MostFollowed.Name, f:IntToStr(Historian.db.char.Follow.MostFollowed.Count)); 
				elseif (name == "FavoritePrey") then
					self:InsertIntoChat(LS["CHAT_INSERT_FAVORITE_PREY"], Historian.db.char.Kills.Favorite.Name, f:IntToStr(Historian.db.char.Kills.Favorite.Count));
				elseif (name == "FavoriteSpell") then
					self:InsertIntoChat(LS["CHAT_INSERT_FAVORITE_SPELL"], Historian.db.char.Spells.Favorite.Name, f:IntToStr(Historian.db.char.Spells.Favorite.Count));
				elseif (name == "Jumps") then
					self:InsertIntoChat(LS["CHAT_INSERT_JUMPS"], f:IntToStr(Historian.db.char.Jumps));
				elseif (name == "Snacks") then
					self:InsertIntoChat(LS["CHAT_INSERT_FOOD"], f:IntToStr(Historian.db.char.Consumption.Food));
				elseif (name == "Drinks") then
					self:InsertIntoChat(LS["CHAT_INSERT_DRINK"], f:IntToStr(Historian.db.char.Consumption.Drink));
				elseif (name == "Tipsy") then
					self:InsertIntoChat(LS["CHAT_INSERT_TIPSY"], Historian.db.char.Consumption.Tipsy);
				elseif (name == "Drunk") then
					self:InsertIntoChat(LS["CHAT_INSERT_DRUNK"], Historian.db.char.Consumption.Drunk);
				elseif (name == "Smashed") then
					self:InsertIntoChat(LS["CHAT_INSERT_SMASHED"], Historian.db.char.Consumption.Smashed);
					
				-- Achievements
				elseif (name == "Quests") then
					self:InsertIntoChat(LS["CHAT_INSERT_QUESTS_COMPLETED"], f:IntToStr(Historian.db.char.Advancement.QuestsCompleted));
				elseif (name == "Recipes") then
					self:InsertIntoChat(LS["CHAT_INSERT_RECIPES_LEARNED"], f:IntToStr(Historian.db.char.Advancement.RecipesLearned));
				elseif (name == "AreasDiscovered") then
					self:InsertIntoChat(LS["CHAT_INSERT_AREAS_DISCOVERED"], f:IntToStr(Historian.db.char.Discoveries.Areas));
				elseif (name == "SpellsCast") then
					self:InsertIntoChat(LS["CHAT_INSERT_TOTAL_SPELLS"], f:IntToStr(Historian.db.char.Spells.Total));
				elseif (name == "TotalKills") then
					self:InsertIntoChat(LS["CHAT_INSERT_TOTAL_KILLS"], f:IntToStr(Historian.db.char.Kills.Total));
				elseif (name == "MoneyAtPeak") then
					self:InsertIntoChat(LS["CHAT_INSERT_MAX_MONEY"], abacus:FormatMoneyFull(Historian.db.char.Loot.MostMoneyEver, false));
				elseif (name == "LootedCommon") then
					self:InsertIntoChat(LS["CHAT_INSERT_LOOTED_COMMON"], f:IntToStr(Historian.db.char.Loot.ItemsOfQuality[1]));
				elseif (name == "LootedUncommon") then
					self:InsertIntoChat(LS["CHAT_INSERT_LOOTED_UNCOMMON"], f:IntToStr(Historian.db.char.Loot.ItemsOfQuality[2]));
				elseif (name == "LootedRare") then
					self:InsertIntoChat(LS["CHAT_INSERT_LOOTED_RARE"], f:IntToStr(Historian.db.char.Loot.ItemsOfQuality[3]));
				elseif (name == "LootedEpic") then
					self:InsertIntoChat(LS["CHAT_INSERT_LOOTED_EPIC"], f:IntToStr(Historian.db.char.Loot.ItemsOfQuality[4]));
				elseif (name == "LootedLegendary") then
					self:InsertIntoChat(LS["CHAT_INSERT_LOOTED_LEGENDARY"], f:IntToStr(Historian.db.char.Loot.ItemsOfQuality[5]));
				elseif (name == "LootedArtifact") then
					self:InsertIntoChat(LS["CHAT_INSERT_LOOTED_ARTIFACT"], f:IntToStr(Historian.db.char.Loot.ItemsOfQuality[6]));
					
				-- Disappointments
				elseif (name == "Drowning") then
					self:InsertIntoChat(LS["CHAT_INSERT_ENV_DROWNING"], f:IntToStr(Historian.db.char.EnvironmentalDamage.Drowning.Times), f:IntToStr(Historian.db.char.EnvironmentalDamage.Drowning.TotalDamage));
				elseif (name == "Fatigued") then
					self:InsertIntoChat(LS["CHAT_INSERT_ENV_FATIGUE"], f:IntToStr(Historian.db.char.EnvironmentalDamage.Fatique.Times), f:IntToStr(Historian.db.char.EnvironmentalDamage.Fatique.TotalDamage));
				elseif (name == "FallenDown") then
					self:InsertIntoChat(LS["CHAT_INSERT_ENV_FALL"], f:IntToStr(Historian.db.char.EnvironmentalDamage.Falling.Times), f:IntToStr(Historian.db.char.EnvironmentalDamage.Falling.TotalDamage));
				elseif (name == "Burnt") then
					self:InsertIntoChat(LS["CHAT_INSERT_ENV_FIRE"], f:IntToStr(Historian.db.char.EnvironmentalDamage.Fire.Times), f:IntToStr(Historian.db.char.EnvironmentalDamage.Fire.TotalDamage));
				elseif (name == "Lava") then
					self:InsertIntoChat(LS["CHAT_INSERT_ENV_LAVA"], f:IntToStr(Historian.db.char.EnvironmentalDamage.Lava.Times), f:IntToStr(Historian.db.char.EnvironmentalDamage.Lava.TotalDamage));
				elseif (name == "Slime") then
					self:InsertIntoChat(LS["CHAT_INSERT_ENV_SLIME"], f:IntToStr(Historian.db.char.EnvironmentalDamage.Slime.Times), f:IntToStr(Historian.db.char.EnvironmentalDamage.Slime.TotalDamage));
				elseif (name == "Deaths") then
					self:InsertIntoChat(LS["CHAT_INSERT_DEATHS"], f:IntToStr(Historian.db.char.Advancement.TotalDeaths));
				end;
			end;
		--end;
	end;
end;