local L = AceLibrary("AceLocale-2.2"):new("Historian");
local LS = AceLibrary("AceLocale-2.2"):new("Historian-Shared");
local f = AceLibrary("HistorianFormatting-1.0");
local deformat = AceLibrary("Deformat-2.0");


-- *************************************
-- * RegisterForEvents
-- *************************************
function Historian:RegisterForEvents()
	self:RegisterEvent("AUTOFOLLOW_BEGIN");
	self:RegisterEvent("CHAT_MSG_LOOT");
	self:RegisterEvent("CHAT_MSG_SYSTEM");
	self:RegisterEvent("LEARNED_SPELL_IN_TAB");
	self:RegisterEvent("PLAYER_DEAD");
	self:RegisterEvent("PLAYER_LEVEL_UP");
	self:RegisterEvent("PLAYER_MONEY");
	self:RegisterEvent("TRADE_SKILL_SHOW");
	self:RegisterEvent("UI_INFO_MESSAGE");
	self:RegisterEvent("UNIT_AURA");
	self:RegisterEvent("UNIT_SPELLCAST_SUCCEEDED");
	self:RegisterEvent("UPDATE_BATTLEFIELD_SCORE");
	self:RegisterEvent("UPDATE_MOUSEOVER_UNIT");
	
	self:RegisterEvent("COMBAT_LOG_EVENT_UNFILTERED", "CombatLogEvent");
	
	self:SecureHook("AscendStop");

	self.playerGUID = UnitGUID("player");
end;


-- *************************************
-- * IsCurrentPlayer
-- *************************************
function Historian:IsCurrentPlayer(guid)
	return guid == self.playerGUID;
end;


-- *************************************
-- * EventUnitDeath
-- *************************************
function Historian:EventUnitDeath(timestamp, eventType, sourceGUID, sourceName, sourceFlags, destGUID, destName, destFlags)
	if (self.lastKill ~= nil) and (self.lastKill == destGUID) then return; end;

	self.lastKill = destGUID;

	local inInstance, instanceType = IsInInstance();
	if (inInstance) then	
		local mobName = destName;

		if (mobName == nil) then
			return;
		end;

		if self:IsCurrentPlayer(sourceGUID) then
			self:RecordKill(mobName, true);
		else
			if (self:IsCountingInstanceKills()) then
				self:RecordKill(mobName, false);
			end;
		end;
	end;
end;


-- *************************************
-- * EventPartyKill
-- *************************************
function Historian:EventPartyKill(timestamp, eventType, sourceGUID, sourceName, sourceFlags, destGUID, destName, destFlags, ...)
	local mobName = destName;

	if (mobName == nil) then
		return;
	end;
	
	self.lastKill = destGUID;

	if (self:IsCurrentPlayer(sourceGUID)) then
		self:RecordKill(mobName, true);
	else
		if (self:IsCountingPartyKills()) then
			self:RecordKill(mobName, false);
		end;
	end;
end;


-- *************************************
-- * EventEnvironmentalDamage
-- *************************************
function Historian:EventEnvironmentalDamage(timestamp, eventType, sourceGUID, sourceName, sourceFlags, destGUID, destName, destFlags, environmentalType, amount, ...)
	if (self:IsCurrentPlayer(destGUID)) then
		local lookup = {
			["DROWNING"] 	= "Drowning",
			["FALLING"] 	= "Falling",
			["FATIGUE"] 	= "Fatique",
			["FIRE"] 		= "Fire",
			["LAVA"] 		= "Lava",
			["SLIME"] 		= "Slime",
		}
		
		local type = lookup[environmentalType];
		if (type ~= nil) then
			self:RegisterEnvironmentDamage(type, amount);
		end;		
	end;
end;

-- *************************************
-- * CombatLogEvent
-- * Called whenever a new combat message is fired.
-- *************************************
function Historian:CombatLogEvent(timestamp, eventType, sourceGUID, sourceName, sourceFlags, destGUID, destName, destFlags, ...)
	local EventHandlers = {
		["ENVIRONMENTAL_DAMAGE"] = Historian.EventEnvironmentalDamage,
		["UNIT_DIED"] = Historian.EventUnitDeath,
		["UNIT_DESTROYED"] = Historian.EventUnitDeath,
		["PARTY_KILL"] = Historian.EventPartyKill,
	}

	local handler = EventHandlers[eventType];
	
	if handler then
		handler(self, timestamp, eventType, sourceGUID, sourceName, sourceFlags, destGUID, destName, destFlags, ...)
	end
end;


-- *************************************
-- * GetValuesFromChatMessage
-- * Returns the values stored in 'message' based on the Blizzard constant 'localizedMessage'.
-- * For example:
-- * 	local mobKilled = self:GetValuesFromChatMessage(chatMessage, SELFKILLOTHER);
-- * where chatMessage contains 'You have slain SomeEnemy!', returns 'SomeEnemy'.
-- *************************************
function Historian:GetValuesFromChatMessage(message, localizedMessage)
	return deformat:Deformat(message, localizedMessage);
end;


-- *************************************
-- * IsWoWMessage
-- * Returns true if 'message' is of the type 'constant' 
-- * ('constant' contains placeholders so a straight comparison would not work)
-- *************************************
function Historian:IsWoWMessage(message, constant)
	local formatted, replaced, replacedDig;

	formatted, replaced = string.gsub(constant, "%%s", "(.+)");
	formatted, replacedDig = string.gsub(formatted, "%%d", "(%%d%+)");

	return (string.find(message, formatted) == 1);
end;



-- *************************************
-- * Tooltip
-- *************************************
function Historian:UPDATE_MOUSEOVER_UNIT()
	if GameTooltip:IsVisible() and GameTooltipTextLeft1:GetText() ~= nil then
		if (not Historian:IsShowingTooltip()) then return; end;

		if (UnitIsPlayer("mouseover")) or (not UnitCreatureType("mouseover")) or (UnitIsFriend("player","mouseover")) then 
			return;
		end;

		local mobName = UnitName("mouseover");

		if (mobName ~= nil) then
			local killStats = Historian:GetKillStatsForEnemy(mobName);
			if (killStats ~= nil) then
				local numberOfKills = f:AsNumber(killStats.Any);
				local numberOfKBs = f:AsNumber(killStats.KillingBlows);

				if (numberOfKills > 0) then
					local red = 1;
					local green = 1;
					local blue = 0;

					if (numberOfKills == 1) then
						GameTooltip:AddLine(L["MSG_TOOLTIP_KILLED_SINGLE"], red, green, blue);
					else
						if (numberOfKills ~= numberOfKBs) then
							GameTooltip:AddLine(string.format(L["MSG_TOOLTIP_KILLED_WITH_KB"], f:IntToStr(numberOfKills), f:IntToStr(numberOfKBs)), red, green, blue);
						else
							GameTooltip:AddLine(string.format(L["MSG_TOOLTIP_KILLED"], f:IntToStr(numberOfKills)), red, green, blue);
						end;
					end;
					GameTooltip:Show();
				end;
			end;
		end;
	end
end;



-- *************************************
-- * PLAYER_MONEY (Remember the biggest fortune the player ever had at any point in time)
-- *************************************
function Historian:PLAYER_MONEY()
	Historian:OnMoneyChanged();
end;


-- *************************************
-- * CHAT_MSG_LOOT (Player received an item)
-- *************************************
function Historian:CHAT_MSG_LOOT(chatMessage)
	local count = 1;
	local item = nil;
	
	item = self:GetValuesFromChatMessage(chatMessage, LOOT_ITEM_SELF);
	if (item == nil) then
		item, count = self:GetValuesFromChatMessage(chatMessage, LOOT_ITEM_SELF_MULTIPLE);
	end;

	if (item == nil) then
		item = self:GetValuesFromChatMessage(chatMessage, LOOT_ITEM_CREATED_SELF);
	end;
	
	if (item == nil) then
		item, count = self:GetValuesFromChatMessage(chatMessage, LOOT_ITEM_CREATED_SELF_MULTIPLE);
	end;
	
	if (item == nil) then
		item = self:GetValuesFromChatMessage(chatMessage, LOOT_ITEM_PUSHED_SELF);
	end;
	
	if (item == nil) then
		item = self:GetValuesFromChatMessage(chatMessage, LOOT_ITEM_PUSHED_SELF_MULTIPLE);
	end;
	
	if (count == nil) then count = 1; end;

	if (item ~= nil) then
		Historian:RecordItemLooted(item);
	end;
end;


-- *************************************
-- * PLAYER_DEAD (Player was killed)
-- *************************************
function Historian:PLAYER_DEAD()
	Historian:OnDeath();
end;


-- *************************************
-- * PLAYER_LEVEL_UP (Player gained another level)
-- *************************************
function Historian:PLAYER_LEVEL_UP(newLevel)
	local levelInfo = { };
	levelInfo["Zone"] = GetRealZoneText(); 
	levelInfo["SubZone"] = GetSubZoneText();
	levelInfo["Date"] = date();
	
	Historian:RecordLevelUp(newLevel, levelInfo);
end;


-- *************************************
-- * CHAT_MSG_SYSTEM (Misc. chat messages)
-- *************************************
function Historian:CHAT_MSG_SYSTEM(chatMessage)
	if (self:IsWoWMessage(chatMessage, ERR_QUEST_COMPLETE_S)) then
		return Historian:OnQuestCompleted();
	end;

	if (self:IsWoWMessage(chatMessage, ERR_LEARN_RECIPE_S)) then
		return Historian:OnRecipeLearned();
	end;

	if (self:IsWoWMessage(chatMessage, DRUNK_MESSAGE_ITEM_SELF2)) then
		return Historian:OnPlayerTipsy();
	end;

	if (self:IsWoWMessage(chatMessage, DRUNK_MESSAGE_ITEM_SELF3)) then
		return Historian:OnPlayerDrunk();
	end;

	if (self:IsWoWMessage(chatMessage, DRUNK_MESSAGE_ITEM_SELF4)) then
		return Historian:OnPlayerCompletelySmashed();
	end;

	local count;
	local item = self:GetValuesFromChatMessage(chatMessage, ERR_QUEST_REWARD_ITEM_S);
	
	if (item == nil) then
		count, item = self:GetValuesFromChatMessage(chatMessage, ERR_QUEST_REWARD_ITEM_MULT_IS);
	end;
	
	if (item ~= nil) then
		Historian:RecordItemLooted(item);
	end;
end;


-- *************************************
-- * AUTOFOLLOW_BEGIN (Player started following another player)
-- *************************************
function Historian:AUTOFOLLOW_BEGIN(playerFollowed)
	return Historian:OnFollow(playerFollowed);
end;


-- *************************************
-- * TradeSkillShow
-- *************************************
function Historian:TRADE_SKILL_SHOW()
	local tradeSkillName = GetTradeSkillLine();
	local withHeaders = GetNumTradeSkills();
	
	local currentTotal, currentWithHeaders = Historian:GetTradeskillStats(tradeSkillName);
	
	if (withHeaders == currentWithHeaders) then
		-- No updates required
		return;
	end;
	
	-- Remove headers from the equation
	local total = 0;
	local index = 0;
	
	for index = GetFirstTradeSkill(), withHeaders, 1 do
		local skillName, skillType = GetTradeSkillInfo(index);
		if (skillName ~= nil) and (skillType ~= "header") then
			total = total + 1;
		end;
	end;
	
	Historian:SetTradeskillStats(tradeSkillName, total, withHeaders);
end;


-- *************************************
-- * UNIT_AURA
-- *************************************
function Historian:UNIT_AURA(unit)
	local wasEating = self.isEating;
	local wasDrinking = self.isDrinking;
	
	self.isEating = false;
	self.isDrinking = false;
	
	if (UnitIsUnit("player", unit)) then
		local i = 1;
		while UnitBuff("player", i) do
			buffName, _, icon, _, duration, timeleft = UnitBuff("player", i);
			
			if (buffName == L["BUFF_FOOD"]) then
				self.isEating = true;
				if (not wasEating) then
					Historian:OnEating();
				end;
			else 
				if (buffName == L["BUFF_DRINK"]) then
					self.isDrinking = true;
					if (not wasDrinking) then
						Historian:OnDrinking();
					end;
				end;
			end;
			
			i = i + 1;
		end
	end;
end;


-- *************************************
-- * AscendStop
-- *************************************
function Historian:AscendStop()
	if (IsFlying() or IsSwimming()) then return; end;
	local now = GetTime();
	
	if (self.lastJump ~= nil) and ((now - self.lastJump) < 0.9) then
		-- A real jump can only occur so often.
		return;
	end;

	self.lastJump = now;	
	Historian:OnJump();
end;


-- *************************************
-- * UI_INFO_MESSAGE
-- * Called whenever an info message is displayed at the top of the screen
-- *************************************
function Historian:UI_INFO_MESSAGE(message)
	if (self:IsWoWMessage(message, ERR_ZONE_EXPLORED)) then
		return Historian:OnDiscovery();
	end;
end;


-- *************************************
-- * LEARNED_SPELL_IN_TAB
-- * Called whenever the player learns a new spell
-- *************************************
function Historian:LEARNED_SPELL_IN_TAB(tabID)
	Historian:CreateSpellTable();
end;


-- *************************************
-- * UNIT_SPELLCAST_SUCCEEDED
-- * Called whenever someone successfully casts a spell nearby (incl. the player)
-- *************************************
function Historian:UNIT_SPELLCAST_SUCCEEDED(unit, spellName, spellRank)
	if (unit ~= "player") then return; end;
	
	Historian:RecordSpellcast(spellName, (self.realSpells[spellName] ~= nil));
end;


-- *************************************
-- * UPDATE_BATTLEFIELD_SCORE
-- * Called whenever the battlefield scores are updated
-- *************************************
function Historian:UPDATE_BATTLEFIELD_SCORE()
	local winner = GetBattlefieldWinner();
	
	-- We are only interested in the stats when the battle is over
	if (winner == nil) then return;	end;

	local inInstance, instanceType = IsInInstance();
	if (instanceType == nil) then return; end;
	if (instanceType ~= "pvp") and (instanceType ~= "arena") then return; end;
	
	local playerName = UnitName("player");
	local scoresAvailable = GetNumBattlefieldScores();

	for i = 1, scoresAvailable do
		local name, killingBlows, honorKills, deaths, honorGained, faction, rank, race, class, filename, damageDone, healingDone = GetBattlefieldScore(i);
		if (name == playerName) then
			-- These are the stats for the current player
			local stats = {};
			stats.killingBlows = killingBlows;
			stats.honorKills = honorKills;
			stats.deaths = deaths;
			stats.honorGained = honorGained;
			stats.damageDone = damageDone;
			stats.healingDone = healingDone;
			stats.winner = winner;
			stats.isArenaMatch = (instanceType == "arena");
			
			-- Save the statistics for this battleground or arena
			self:RecordPvPBattle(GetRealZoneText(), stats);
			return;
		end;
	end;		
end;
