
--[[
  Name:     Hellbender DKP
  Author:   Earthbane @ Hellscream
  Contact:  earthbane.9yq@gishpuppy.com
  
  Description:
            Hellbender DKP is a replacement for HoB_DKP and works initially using the same
            formulas as HoB_DKP. It will dynamically generate DKP values for items based on
            the item's atributes. The importance of item atributes can be customized to tweek
            the importance of certain stats to certain classes.
            
            The aim for Hellbender is to be easier to customize the DKP weights and make it
            easier for others to hack to their needs
  
  Complimentary Addons:
            CT_Raidtracker - Hellbender should be able to send it's DKP values to
            CTRA when it records the item in the raid.        
--]]--

-------------------
-- Localize globals
-------------------
local L = AceLibrary("AceLocale-2.2"):new("Hellbender");
local BC = AceLibrary("Babble-Class-2.2");
local BI = AceLibrary("Babble-Inventory-2.2");
local StatModifiers = HBDKPStatModifiers;
local ItemSlotWeights = HBDKPItemSlotWeights;
local WeaponTypeWeights = HBDKPWeaponTypeWeights;
local WarriorWeaponsWeights = HBDKPWarriorWeaponsWeights;
local RogueWeaponsWeights = HBDKPRogueWeaponsWeights;
local CasterWeaponsWeights = HBDKPCasterWeaponsWeights;
local HunterWeaponsWeights = HBDKPHunterWeaponsWeights;
local RangedWeaponsWeights = HBDKPRangedWeaponsWeights;
local GemWeights = HBDKPGemWeights;

------------------
-- Library Loading
------------------
local ibl = AceLibrary("ItemBonusLib-1.0");
local Gratuity = AceLibrary("Gratuity-2.0");
Hellbender = AceLibrary("AceAddon-2.0"):new("AceConsole-2.0", "AceEvent-2.0", "AceDB-2.0","AceHook-2.1");

------------------
-- Local Variables
------------------
local Hellbender_Old_HDKP_GetDKP = nil;
local lastDKP = {};
local ignoreSlots = {
  ["INVTYPE_RELIC"] = true,
  ["INVTYPE_TABARD"] = true,
  ["INVTYPE_BAG"] = true,
  ["INVTYPE_BODY"] = true, 
};

-----------------------------
-- Configure Default Settings
-----------------------------
local options = { 
  type='group',
  args = {
    debug = {
      type = 'range',
      name = L['Debug'],
      desc = L['Enable debug output'],
      get = "GetDebug",
      set = "SetDebug",
      min = 0,
      max = 3,
    },
    
    showEligibleClasses = {
      type = 'toggle',
      name = L['Eligibility'],
      desc = L['Toggle to show in tooltip those classes who are able to use the weapon'],
      get = "ShowEligibleClasses",
      set = "ToggleShowEligibleClasses",
    },
    
    showClass = {
      type = 'toggle',
      name = L['Class'],
      desc = L["Toggle to show in tooltip which class's DKP modifiers were used for final DKP value"],
      get = "ShowDKPClass",
      set = "ToggleShowDKPClass",
    },
    
    dkpPrecision = {
      type = 'range',
      name = L['Precision'],
      desc = L['Set precision in DKP result, how many decimal places to show.'],
      min = 0,
      max = 6,
      get = "GetDKPPrecision",
      set = "SetDKPPrecision",
    },
    
    dkpMultiplier = {
      type = 'range',
      name = L['multiplier'],
      desc = L['Multiply the final DKP calculation by this amount'],
      min = 1,
      max = 100,
      get = "GetDKPMultiplier",
      set = "SetDKPMultiplier",
    },
    
    CTRA_Hook = {
      type = 'toggle',
      name = L['CTRA_Hook'],
      desc = L["Should Hellbender hook into CT_Raidassist for DKP values. Note: This will override HoB_DKP if installed"],
      get = "GetHookCTRA",
      set = "SetHookCTRA",
    },
  },
}

--[[
  Debugging code

  from options array:
        statDump = {
          type = 'execute',
          name = 'dumpIt',
          desc = "Dump ItemBonusLib Pairs",
          func = "dumpNameList",
        },

function Hellbender:dumpNameList()
  local list = ibl:GetNameList();
  for key,value in pairs(list) do
    self:Print("IBL: "..value.." ==> "..ibl:GetBonusFriendlyName(value));
  end
end

--]]--

Hellbender:RegisterChatCommand(L["Slash-Commands"], options);

Hellbender:RegisterDB("HellbenderDB","HellbenderDBPC");
Hellbender:RegisterDefaults("profile", {
  debug = 0,
  showEligibleClasses = true,
  showClass = true,
  dkpPrecision = 2,
  dkpMultiplier = 1,
  CTRA_Hook = true,
} )

-----------------------------------
-- Slash Command Property Functions
-----------------------------------
function Hellbender:GetDKPMultiplier()
  return self.db.profile.dkpMultiplier;
end

function Hellbender:SetDKPMultiplier(NewValue)
  self.db.profile.dkpMultiplier = NewValue;
end

function Hellbender:GetDKPPrecision()
  return self.db.profile.dkpPrecision;
end

function Hellbender:SetDKPPrecision(NewValue)
  self.db.profile.dkpPrecision = NewValue;
end

function Hellbender:ShowDKPClass()
  return self.db.profile.showClass;
end

function Hellbender:ToggleShowDKPClass()
  self.db.profile.showClass = not self.db.profile.showClass;
end

function Hellbender:ShowEligibleClasses()
  return self.db.profile.showEligibleClasses;
end

function Hellbender:ToggleShowEligibleClasses()
  self.db.profile.showEligibleClasses = not self.db.profile.showEligibleClasses;
end

function Hellbender:GetDebug()
  return self.db.profile.debug;
end

function Hellbender:SetDebug(NewValue)
  self.db.profile.debug = NewValue;
end

function Hellbender:SetHookCTRA()
  self.db.profile.hookCTRA = not self.db.profile.hookCTRA;
  self:toggleCTRAHook();
end

function Hellbender:GetHookCTRA()
  return self.db.profile.hookCTRA;
end

-----------------
-- Event Handlers
-----------------
function Hellbender:OnInitialize()

end

function Hellbender:OnEnable()

  -- Borrowed basic code and concept from TipHooker-1.0
  tooltips = {
    ["GameTooltip"] = true,
		["ItemRefTooltip"] = true,
		["ShoppingTooltip"] = true,
		-- EquipCompare support
		["ComparisonTooltip"] = true,
		-- EQCompare support
		["EQCompareTooltip"] = true,
		-- takKompare support
		["tekKompareTooltip"] = true,
		-- LinkWrangler support
		["IRR_"] = true,
		["LinkWrangler"] = true,
		-- MultiTips support
		-- Links support
		["LinksTooltip"] = true,
		-- AtlasLoot support
		["AtlasLootTooltip"] = true,
		-- ItemMagic support
		["ItemMagicTooltip"]= true,
		-- Sniff support
		["SniffTooltip"] = true,
	};
	
	-- Change from TipHooker, hooking to OnTooltipSetItem for
	-- scanning purposes
	local tooltip = EnumerateFrames();
	while tooltip do
    if tooltip:GetObjectType() == "GameTooltip" then
      local name = tooltip:GetName();
      if name then
        self:Debug("Found GameTooltip: "..name,1);
        if tooltips[name] then
          self:Debug("InitializeHook(item) = "..name,1);
          self:HookScript(tooltip, "OnTooltipSetItem")
  			end
      end
    end
    tooltip = EnumerateFrames(tooltip)
	end

  if self.db.profile.debug >= 1 then
    self:HookReport();
  end
  
  self:toggleCTRAHook();
--[[
  if self.db.profile.hookCTRA then
    if HDKP_GetDKP then
      Hellbender__Old_HDKP_GetDKP = HDKP_GetDKP;
    -- Replace call to Hob_DKP with our own magic juice
    HDKP_GetDKP = Hellbender_HDKP_GetDKP_Hack;
    self:Debug("HoB_DKP hook overridden",1);
  end
]]--
  
  self:Print("Enabled and registered");

end

function Hellbender:OnDisable()

end


function Hellbender:Debug(message, level)
  if self.db.profile.debug == nil then
    self:Print("Debug failed, no debug value! "..message);
    return;
  end
  
  if self.db.profile.debug >= level then
    self:Print(message)
  end
end

--------
-- Hooks
--------

function Hellbender:toggleCTRAHook()
  if self.db.profile.hookCTRA then
    if HDKP_GetDKP then
      -- Replace call to Hob_DKP with our own magic juice
      Hellbender_Old_HDKP_GetDKP = HDKP_GetDKP;
      self:Debug("HoB_DKP discovered and overridden for CTRA DKP values",1);
    end
    
    HDKP_GetDKP = Hellbender_HDKP_GetDKP_Hack;
    self:Debug("CTRA hook set",1);
  else
    if Hellbender_Old_HDKP_GetDKP then
      HDKP_GetDKP = Hellbender_Old_HDKP_GetDKP;
      self:Debug("HoB_DKP now set for CTRA DKP values",1);
    else
      HDKP_GetDKP = function() return 0 end;
      self:Debug("CTRA hook unset",1);
    end
  end
end 

---------------------------------------------
-- OnTooltipSetItem
--
-- A tooltip we hooked has recieved an item to
-- display. We will extract the item, get the
-- DKP value, and display it.
---------------------------------------------
function Hellbender:OnTooltipSetItem(tooltip,...)
  if self.db.profile.debug >= 1 then
    self:Print("In OnTooltipSetItem");
  end
  local item, link = tooltip:GetItem();
  if not item then 
    self:Debug("No item given from "..tooltip:GetName(),2);
    return self.hooks[tooltip].OnTooltipSetItem(tooltip, ...) 
  else
    self:Debug("Got item ("..item.." , "..link..") from "..tooltip:GetName(),2);
  end

  local dkp, highClass, damageClass, weaponUsers = self:GimmieDKP(link);
  
  if highClass ~= nil then
    self:Debug("Printing results.",2);
    
    local dkpString = dkp;
    if self.db.profile.showClass then
      dkpString = dkpString.." ("..highClass;
      if damageClass ~= nil and damageClass ~= "" then
        dkpString = dkpString.."/"..damageClass;
      end
      dkpString = dkpString..")";
    end
    tooltip:AddDoubleLine("DKP:",dkpString,1,1,1,1,1,1);
    
    if self.db.profile.showEligibleClasses and weaponUsers ~= "" then
      tooltip:AddDoubleLine("Eligible Classes:",weaponUsers,1,1,1,1,1,1);
    end
  else
    self:Debug("No DKP calculated. Please come again.",2);
  end

  return self.hooks[tooltip].OnTooltipSetItem(tooltip, ...);
end

function Hellbender:GimmieDKP(itemLinkOrString)

  local itemName, itemLink, itemRarity, itemLevel, itemMinLevel, itemType, itemSubType, itemStackCount,itemEquipLoc, itemTexture = GetItemInfo(itemLinkOrString);
  
  if itemRarity < 3 then
    return 0, nil, nil, nil;
  end
  
  local found1, _1, itemString = string.find(itemLink, "^|c%x+|H(.+)|h%[.+%]");
  self:Debug("Item string extracted: "..itemString,2);
  local found2, _2, itemID, enchantID, jewelID1, jewelID2, jewelID3, jewelID4, suffixID, uniqueID = string.find(itemString, "^item:(%d+):(%d+):(%d+):(%d+):(%d+):(%d+):(%-?%d+):(%-?%d+)");
  self:Debug("Item ID extracted: "..itemID,2);
  
  -----------------------------------------------------------------------------
  -- Here we remove the jem details from the item. DKP values should only be on
  -- factory original equipment, not after market addons
  -----------------------------------------------------------------------------
  itemName, itemLink, itemRarity, itemLevel, itemMinLevel, itemType, itemSubType, itemStackCount,itemEquipLoc, itemTexture = GetItemInfo(itemID);
  self:Debug("W/O Gems: "..itemName..":"..itemLink..":"..itemRarity..":"..itemLevel..":"..itemMinLevel..":"..itemType..":"..itemSubType..":"..itemStackCount..":"..itemEquipLoc..":"..itemTexture,1);

  -- Cowardly refuse to work on DKPs on a few slot types
  if itemEquipLoc == ""  or ignoreSlots[itemEquipLoc] then
    return 0, nil, nil, nil;
  end
  
  if itemType ~= L["Weapon"] and itemType ~= L["Armor"] then
    return 0, nil, nil, nil;
  end

  -- Check if we out last calculation was the same item
  if lastDKP['link'] ~= nil and lastDKP['link'] == itemLink then
    if lastDKP['damageClass'] == nil then
      lastDKP['damageClass'] = "";
    end
    if lastDKP['statClass'] == nil then
      self:Print("STAT CLASS MISSING FOR "..itemLink);
      for key,value in pairs(lastDKP) do
        self:Print("** "..key.." ==> "..value);
      end
      return 0,"ERROR","ERROR","ERROR";
    end
    self:Debug("Returning cached DKP results for "..itemLink..": "..lastDKP['dkp']..","..lastDKP['statClass']..","..lastDKP['damageClass'],2);
    return lastDKP['dkp'], lastDKP['statClass'], lastDKP['damageClass'],lastDKP['weaponUsers'];
  end

  local bonuses = ibl:ScanItem(itemLink,true);
  
  self:Debug(itemName..":"..itemLink..":"..itemRarity..":"..itemLevel..":"..itemMinLevel..":"..itemType..":"..itemSubType..":"..itemStackCount..":"..itemEquipLoc..":"..itemTexture,2);

  if self.db.profile.debug >= 1 then
    for stat,value in pairs(bonuses) do
      self:Print(ibl:GetBonusFriendlyName(stat).."("..stat..") => "..value);
    end
  end

  local specialModifier = 1;
  local DKPValues = {};
  local tempDKP = 0;
  local highClass = nil;
  local eligible = false;
  local weaponUsers = "";
  local isCaster = false;
  local users = {}; -- List of classes that can use item
  
  if itemType == L['Armor'] and ItemSlotWeights[itemEquipLoc] ~= nil then
    self:Debug("Slot Weight Adjusted. "..itemEquipLoc.." with weight "..ItemSlotWeights[itemEquipLoc],2);
    specialModifier = ItemSlotWeights[itemEquipLoc];
  end
  
  
  if itemType == L['Weapon'] then
  
    -- Check if it's a caster weapon, ignore bows, xbows, and guns
    if NonCasterRangedLocations[itemEquipLoc] == nil then
      if itemEqipLoc == "INVTYPE_RANGEDRIGHT" then
        isCaster = true;
      else
        for stat,_ in pairs(bonuses) do
          if CasterFlags[stat] ~= nil then
            isCaster = true;
            break;
          end
        end
      end
    end
    
    if WeaponTypeWeights[itemSubType] ~= nil then
      specialModifier = WeaponTypeWeights[itemSubType];
    end
  end
  
  
  --[[
    Work out the DKP value for the stats of the item.
    
    If it's a weapon, we will later check it's damage output
    and add the Weapon Damage DKP to the stat DKP
  ]]--
  local class, weights = nil,nil;
  for class,weights in pairs(StatModifiers) do
    eligible = false;
    if class == nil then
      self:Debug("Nil class found. running next loop",1);
    else
      self:Debug("Class checked: "..class,1);
      DKPValues[class] = 0;
      if itemType == L['Armor'] then
        -- Work out if eligible class
        self:Debug("Armor check. Class: ("..class..") Subtype: ("..itemSubType..")",2);
        
        -- FIX - Cloak is cloth, but *all* cloaks are cloth and dkp calcs should take
        --        everyone into account
        if itemEquipLoc == "INVTYPE_CLOAK" then
          self:Debug("Armor is cloak, marking all classes eligible",2);
          eligible = true;
        elseif ArmorUse[itemSubType] ~= nil and ArmorUse[itemSubType][class] ~= nil then
          self:Debug(class.." is able to use "..itemSubType,2);
          eligible = true;
        end
      end
      
      if itemType == L['Weapon'] then
        self:Debug("Weapon check. Class: ("..class..") Subtype: "..itemSubType,2);
        if WeaponUse[itemSubType] ~= nil and WeaponUse[itemSubType][class] ~= nil then
          self:Debug(class.." is able to use "..itemSubType,2);
          if weaponUsers == "" then
            weaponUsers = class;
          else
            weaponUsers = weaponUsers..", "..class;
          end
          eligible = true;
        end
      end
    
      if eligible then
        for stat,value in pairs(bonuses) do 
      
          if DKPValues[class] == nil then
            DKPValues[class] = 0;
          end
          
          if weights[stat] ~= nil then
            DKPValues[class] = DKPValues[class] + value * weights[stat] * specialModifier;
          end
        end
        
        if highClass == nil then
          highClass = class
        else
          if DKPValues[class] >= DKPValues[highClass] then
            highClass = class;
          end
        end
        --self:Debug("DKP for class "..class.." = "..DKPValues[class],3);
      end
      --self:Print(key.." => "..value) 
      --tooltip:AddDoubleLine(ibl:GetBonusFriendlyName(key),value,1,1,1,r,g,b);
    end
  end
  
  -- Now workout Weapon Damage DKP value
  local damageClass = "No DMG";
  local typeWeight = 1;
  local weaponDKP = -1;
  local sub = 0;
  local dpsWeight = 1;
  local topDamageWeight = 1;
  local dps = 0;
  
  if itemType == L['Weapon'] then
  
    -- 2 Handed caster weapons will have Damage DKP included. All other caster items (save wand) have
    -- Damage dkp values ignored
    if isCaster and itemEquipLoc == "INVTYPE_2HWEAPON" and CasterWeaponsWeights[itemEquipLoc][itemSubType] ~= nil then
      
      sub = CasterWeaponsWeights[itemEquipLoc][itemSubType].sub;
      dpsWeight = CasterWeaponsWeights[itemEquipLoc][itemSubType].dps;
      topDamageWeight = CasterWeaponsWeights[itemEquipLoc][itemSubType].top;
      
      dps = ((bonuses["WEAPON_MAX"] + bonuses['WEAPON_MIN'])/2) / bonuses['WEAPON_SPEED'];
      weaponDKP = (((dps * dpsWeight) + (bonuses["WEAPON_MAX"] * topDamageWeight) - sub) * 2 ) * 0.35;
      
      -- Since only Shaman and Pallies can use these caster, we will figure out which of their
      -- stat DKPs we will use
      if DKPValues[BC["Shaman"]] > DKPValues[BC["Paladin"]] then
        damageClass = BC["Shaman"];
      else
        damageClass = BC["Paladin"];
      end
      
      highClass = damageClass;
      
      self:Debug("Caster 2H Results: "..damageClass.." ("..weaponDKP..") "..bonuses["WEAPON_MAX"]..","..bonuses["WEAPON_MIN"]..","..bonuses["WEAPON_SPEED"],2);
    end
    
    -- All other non-ranged weapon checks
    -- Weapons are evaluated from a Warrior (Tank) standpoint and Rogue (Melee DPS) standpoint
    -- The higher dkp result is the winner
    if not isCaster and NonCasterRangedLocations[itemEquipLoc] == nil and itemEquipLoc ~= "INVTYPE_RANGEDRIGHT" then
      -- IDEA Could possibly make this smoother. some of below is copy and pasted from above
      local warDKP, rogueDKP = 0,0
      dps = ((bonuses["WEAPON_MAX"] + bonuses['WEAPON_MIN'])/2) / bonuses['WEAPON_SPEED'];
      
      if WarriorWeaponsWeights[itemEquipLoc][itemSubType] ~= nil then
        sub = WarriorWeaponsWeights[itemEquipLoc][itemSubType].sub;
        dpsWeight = WarriorWeaponsWeights[itemEquipLoc][itemSubType].dps;
        topDamageWeight = WarriorWeaponsWeights[itemEquipLoc][itemSubType].top;
        
        self:Debug("Warrior modifiers: "..sub..","..dpsWeight..","..topDamageWeight,2);
        
        warDKP = ((dps * dpsWeight) + (bonuses["WEAPON_MAX"] * topDamageWeight) - sub) * 0.8;
        if itemEquipLoc == "INVTYPE_2HWEAPON" then
          warDKP = warDKP * 2;
        end
      end
      
      if RogueWeaponsWeights[itemEquipLoc] ~= nil and RogueWeaponsWeights[itemEquipLoc][itemSubType] ~= nil then
        sub = RogueWeaponsWeights[itemEquipLoc][itemSubType].sub;
        dpsWeight = RogueWeaponsWeights[itemEquipLoc][itemSubType].dps;
        topDamageWeight = RogueWeaponsWeights[itemEquipLoc][itemSubType].top;

        self:Debug("Rogue modifiers: "..sub..","..dpsWeight..","..topDamageWeight,2);

        rogueDKP = (dps * dpsWeight) + (bonuses["WEAPON_MAX"] * topDamageWeight) - sub;
      end
      
      if warDKP > rogueDKP then
        weaponDKP = warDKP;
        damageClass = BC["Warrior"];
      else
        damageClass = BC["Rogue"];
        weaponDKP = rogueDKP;
      end
      
      self:Debug("Non-ranged melee Results: "..damageClass.." ("..weaponDKP..") "..bonuses["WEAPON_MAX"]..","..bonuses["WEAPON_MIN"]..","..bonuses["WEAPON_SPEED"],2);
    end
    
    -- Finish off weapon DKP calcs with ranged
    if NonCasterRangedLocations[itemEquipLoc] ~= nil or itemEquipLoc == "INVTYPE_RANGEDRIGHT" then
      sub = RangedWeaponsWeights[itemSubType].sub;
      dpsWeight = RangedWeaponsWeights[itemSubType].dps;
      topDamageWeight = RangedWeaponsWeights[itemSubType].top;
      
      dps = ((bonuses["WEAPON_MAX"] + bonuses['WEAPON_MIN'])/2) / bonuses['WEAPON_SPEED'];
      weaponDKP = (dps * dpsWeight) + (bonuses["WEAPON_MAX"] * topDamageWeight) - sub;
     
      damageClass = L['Ranged'];
      
      self:Debug("Ranged Results: "..damageClass.." ("..weaponDKP..") "..bonuses["WEAPON_MAX"]..","..bonuses["WEAPON_MIN"]..","..bonuses["WEAPON_SPEED"],2);
      
    end
    if WeaponTypeWeights[itemType] ~= nil then
      weaponDKP = WeaponTypeWeights[itemType] * weaponDKP;
    end

  end 
  
  if itemType ~= L['Weapon'] then
    damageClass = nil;
  end
  
  if weaponDKP > 0 then
    DKPValues[highClass] = DKPValues[highClass] + weaponDKP;
  end
  
  self:Debug("Running gem details on item("..itemID..")",1);
  
  -- Gem DKP Check
  local gemDKP = 0;
  local gemData = self:GemDetails(itemID);
  
  gemDKP = gemData.red * GemWeights[L['Red Socket']] + gemData.yellow * GemWeights[L['Yellow Socket']] + gemData.blue * GemWeights[L['Blue Socket']] + gemData.meta * GemWeights[L['Meta Socket']];
  
  self:Debug("Gem DKP: "..gemDKP,1);
  
  if self.db.profile.debug >= 1 then
    for key,value in pairs (gemData) do
      self:Print("GemData:  "..key.." ==> "..value);
    end
  end
  
  DKPValues[highClass] = self:Round( (DKPValues[highClass] + gemDKP) * self.db.profile.dkpMultiplier);
  
  --[[
    Save results in lastDKP cache
    
    Moving the mouse over an item link or icon can call this
    event many times. Why do all the calcs over again? Lazness wins again
  ]]--
  lastDKP['link'] = itemLink;
  lastDKP['dkp'] = DKPValues[highClass];
  lastDKP['statClass'] = highClass;
  lastDKP['damageClass'] = damageClass;
  lastDKP['weaponUsers'] = weaponUsers;
  
  return DKPValues[highClass], highClass, damageClass, weaponUsers;
end

--[[
  Taking function from HoB_DKP to work with CT_Raidassist
]]--
function Hellbender_HDKP_GetDKP_Hack(itemid, enchantid, subid, extra)
  -- item:itemid:enchantid:subid:extra
  Hellbender:Debug("In CTRT Hook: item:"..itemid..":"..enchantid..":"..subid..":"..extra,1);
  local itemString = "item:"..itemid..":"..enchantid..":"..subid..":"..extra;
  local dkp = Hellbender:GimmieDKP(itemString);
  if dkp then
    Hellbender:Debug("CTRA DKP Returned: "..dkp,1);
  else
    Hellbender:Debug("I got nothing back.",1);
    dkp = 0;
  end
  return dkp;
end

function Hellbender:GemDetails(itemID)
	Gratuity:SetHyperlink("item:" .. itemID .. ":0:0:0:0:0:0:0");
	
	local count = Gratuity:NumLines();
	local found = false;
	
	local sockets = {
    red = 0,
    yellow = 0,
    blue = 0,
    meta = 0,
    total = 0,
  };
	
	if count < 5 then
		return sockets;
	end
	
	for lineX = 4, count do
		local line = Gratuity:GetLine(lineX);
		if line == L["Red Socket"] then
			sockets.red = sockets.red + 1;
			found = true;
		elseif line == L["Yellow Socket"] then
			sockets.yellow = sockets.yellow + 1;
			found = true;
		elseif line == L["Blue Socket"] then
			sockets.blue = sockets.blue + 1;
			found = true;
		elseif line == L["Meta Socket"] then
      sockets.meta = sockets.meta + 1;
      found = true;
		end
		
		if found then
		  sockets.total = sockets.total + 1;
		  found = false;
		end
		
		-- There's a 3 socket limit on this ride
		if sockets.total >= 3 then
		  break;
		end
	end
  
  return sockets;	
end

function Hellbender:Round(number)
  return format("%."..self.db.profile.dkpPrecision.."f",number);
end