-- HealAssign
-- by Greltok of Aerie Peak
-- Copyright (c) 2006-2008 Simon Ward

local HEALASSIGN_VERSION                = HA_version:new( 0, 12, 0 );

local HEALASSIGN_COMMUNICATION_PREFIX   = "[HA]";

local HEALASSIGN_MAP_RADIUS             = 80;

local banzai = LibStub and LibStub:GetLibrary( "LibBanzai-2.0", true );
local healComm = LibStub and LibStub:GetLibrary( "LibHealComm-3.0", true );

HEALASSIGN_TITLE = string.format( HEALASSIGN_TITLE, tostring( HEALASSIGN_VERSION ) );

local HEALASSIGN_CODE_WARRIOR           = "@W";
local HEALASSIGN_CODE_ROGUE             = "@R";
local HEALASSIGN_CODE_HUNTER            = "@H";
local HEALASSIGN_CODE_MAGE              = "@M";
local HEALASSIGN_CODE_WARLOCK           = "@L";
local HEALASSIGN_CODE_DRUID             = "@D";
local HEALASSIGN_CODE_PRIEST            = "@P";
local HEALASSIGN_CODE_SHAMAN            = "@S";
local HEALASSIGN_CODE_PALADIN           = "@A";

local HEALASSIGN_CODE_GROUP1            = "@1";
local HEALASSIGN_CODE_GROUP2            = "@2";
local HEALASSIGN_CODE_GROUP3            = "@3";
local HEALASSIGN_CODE_GROUP4            = "@4";
local HEALASSIGN_CODE_GROUP5            = "@5";
local HEALASSIGN_CODE_GROUP6            = "@6";
local HEALASSIGN_CODE_GROUP7            = "@7";
local HEALASSIGN_CODE_GROUP8            = "@8";

local HEALASSIGN_KEYWORD_TO_CODE =
{
    [HEALASSIGN_KEYWORD_WARRIOR]        = HEALASSIGN_CODE_WARRIOR,
    [HEALASSIGN_KEYWORD_ROGUE]          = HEALASSIGN_CODE_ROGUE,
    [HEALASSIGN_KEYWORD_HUNTER]         = HEALASSIGN_CODE_HUNTER,
    [HEALASSIGN_KEYWORD_MAGE]           = HEALASSIGN_CODE_MAGE,
    [HEALASSIGN_KEYWORD_WARLOCK]        = HEALASSIGN_CODE_WARLOCK,
    [HEALASSIGN_KEYWORD_DRUID]          = HEALASSIGN_CODE_DRUID,
    [HEALASSIGN_KEYWORD_PRIEST]         = HEALASSIGN_CODE_PRIEST,
    [HEALASSIGN_KEYWORD_SHAMAN]         = HEALASSIGN_CODE_SHAMAN,
    [HEALASSIGN_KEYWORD_PALADIN]        = HEALASSIGN_CODE_PALADIN,
    
    [HEALASSIGN_KEYWORD_GROUP1]         = HEALASSIGN_CODE_GROUP1,
    [HEALASSIGN_KEYWORD_GROUP2]         = HEALASSIGN_CODE_GROUP2,
    [HEALASSIGN_KEYWORD_GROUP3]         = HEALASSIGN_CODE_GROUP3,
    [HEALASSIGN_KEYWORD_GROUP4]         = HEALASSIGN_CODE_GROUP4,
    [HEALASSIGN_KEYWORD_GROUP5]         = HEALASSIGN_CODE_GROUP5,
    [HEALASSIGN_KEYWORD_GROUP6]         = HEALASSIGN_CODE_GROUP6,
    [HEALASSIGN_KEYWORD_GROUP7]         = HEALASSIGN_CODE_GROUP7,
    [HEALASSIGN_KEYWORD_GROUP8]         = HEALASSIGN_CODE_GROUP8,
    
    [HEALASSIGN_KEYWORD_GROUP1_ALT]     = HEALASSIGN_CODE_GROUP1,
    [HEALASSIGN_KEYWORD_GROUP2_ALT]     = HEALASSIGN_CODE_GROUP2,
    [HEALASSIGN_KEYWORD_GROUP3_ALT]     = HEALASSIGN_CODE_GROUP3,
    [HEALASSIGN_KEYWORD_GROUP4_ALT]     = HEALASSIGN_CODE_GROUP4,
    [HEALASSIGN_KEYWORD_GROUP5_ALT]     = HEALASSIGN_CODE_GROUP5,
    [HEALASSIGN_KEYWORD_GROUP6_ALT]     = HEALASSIGN_CODE_GROUP6,
    [HEALASSIGN_KEYWORD_GROUP7_ALT]     = HEALASSIGN_CODE_GROUP7,
    [HEALASSIGN_KEYWORD_GROUP8_ALT]     = HEALASSIGN_CODE_GROUP8
};

local HEALASSIGN_CODE_TO_KEYWORD =
{
    [HEALASSIGN_CODE_WARRIOR]           = HEALASSIGN_KEYWORD_WARRIOR,
    [HEALASSIGN_CODE_ROGUE]             = HEALASSIGN_KEYWORD_ROGUE,
    [HEALASSIGN_CODE_HUNTER]            = HEALASSIGN_KEYWORD_HUNTER,
    [HEALASSIGN_CODE_MAGE]              = HEALASSIGN_KEYWORD_MAGE,
    [HEALASSIGN_CODE_WARLOCK]           = HEALASSIGN_KEYWORD_WARLOCK,
    [HEALASSIGN_CODE_DRUID]             = HEALASSIGN_KEYWORD_DRUID,
    [HEALASSIGN_CODE_PRIEST]            = HEALASSIGN_KEYWORD_PRIEST,
    [HEALASSIGN_CODE_SHAMAN]            = HEALASSIGN_KEYWORD_SHAMAN,
    [HEALASSIGN_CODE_PALADIN]           = HEALASSIGN_KEYWORD_PALADIN,
    
    [HEALASSIGN_CODE_GROUP1]            = HEALASSIGN_KEYWORD_GROUP1,
    [HEALASSIGN_CODE_GROUP2]            = HEALASSIGN_KEYWORD_GROUP2,
    [HEALASSIGN_CODE_GROUP3]            = HEALASSIGN_KEYWORD_GROUP3,
    [HEALASSIGN_CODE_GROUP4]            = HEALASSIGN_KEYWORD_GROUP4,
    [HEALASSIGN_CODE_GROUP5]            = HEALASSIGN_KEYWORD_GROUP5,
    [HEALASSIGN_CODE_GROUP6]            = HEALASSIGN_KEYWORD_GROUP6,
    [HEALASSIGN_CODE_GROUP7]            = HEALASSIGN_KEYWORD_GROUP7,
    [HEALASSIGN_CODE_GROUP8]            = HEALASSIGN_KEYWORD_GROUP8
};

local HEALASSIGN_CODE_TO_CLASS =
{
    [HEALASSIGN_CODE_WARRIOR]           = "WARRIOR",
    [HEALASSIGN_CODE_ROGUE]             = "ROGUE",
    [HEALASSIGN_CODE_HUNTER]            = "HUNTER",
    [HEALASSIGN_CODE_MAGE]              = "MAGE",
    [HEALASSIGN_CODE_WARLOCK]           = "WARLOCK",
    [HEALASSIGN_CODE_DRUID]             = "DRUID",
    [HEALASSIGN_CODE_PRIEST]            = "PRIEST",
    [HEALASSIGN_CODE_SHAMAN]            = "SHAMAN",
    [HEALASSIGN_CODE_PALADIN]           = "PALADIN"
};

local HEALASSIGN_CODE_TO_GROUP =
{
    [HEALASSIGN_CODE_GROUP1]            = 1,
    [HEALASSIGN_CODE_GROUP2]            = 2,
    [HEALASSIGN_CODE_GROUP3]            = 3,
    [HEALASSIGN_CODE_GROUP4]            = 4,
    [HEALASSIGN_CODE_GROUP5]            = 5,
    [HEALASSIGN_CODE_GROUP6]            = 6,
    [HEALASSIGN_CODE_GROUP7]            = 7,
    [HEALASSIGN_CODE_GROUP8]            = 8,
};

HealAssign_Saved =
{
    savedVersion                        = nil,
    showMapIcon                         = true,
    notifyMemberAdded                   = true,
    notifyMemberOffline                 = true,
    notifyAssignmentsChanged            = true,
    playSoundMyAssignmentsChanged       = true,
    showAssignments                     = true,
    locked                              = nil,
    mapIconAngle                        = 0,
    mapSquare                           = nil,
    healingAssignments                  = nil,
    savedSets                           = nil,
    channelName                         = nil,
    ignoreOwnHeals                      = nil,
    opacity                             = 1,
    scale                               = 1,
    posX                                = 200,
    posY                                = -200
};

HealAssign =
{
    serverName                          = nil,
    
    playerName                          = nil,
    playerLanguage                      = nil,
    spellRange                          = nil,
    playerIsHealer                      = false,
    
    eventHandlers                       = {},
    
    raidMembers                         = nil,
    raidHealers                         = nil,
    raidVersions                        = nil,
    
    whisperHandlers                     = {},
    
    slashCommandHandlers                = {},
    
    raidAddonMessageHandlers            = {},
    
    ownHeals                            = {},
    
    playerAssignments                   = nil,
    
    mapIconDragTimer                    = nil,
    
    configureUnitButtonsTimer           = nil,
    
    updateRangeTimer                    = nil,
    
    unitButtons                         = nil,
    unitButtonsPool                     = nil,
    
    syncFragments                       = nil,
    syncSender                          = nil,
    
    inLockdown                          = false,
    pendingConfigureUnitButtons         = false,
    
    updatedVersion                      = nil
};

function HealAssign.InvertTable( t )
    local result = nil;
    if t then
        result = {};
        for k, v in pairs( t ) do
            result[v] = k;
        end
    end
	return result;
end

local HEALASSIGN_CLASS_TO_CODE = HealAssign.InvertTable( HEALASSIGN_CODE_TO_CLASS );
local HEALASSIGN_GROUP_TO_CODE = HealAssign.InvertTable( HEALASSIGN_CODE_TO_GROUP );

HealAssign.eventHandlers["PLAYER_LOGIN"] = function()
    HealAssign.Init();
end

HealAssign.eventHandlers["PLAYER_ENTERING_WORLD"] = function()
    HealAssign.inLockdown = false;
    HealAssign.UpdateRaidRoster();
end

HealAssign.eventHandlers["VARIABLES_LOADED"] = function()
    HealAssign_Saved.savedVersion = HEALASSIGN_VERSION;
end

HealAssign.eventHandlers["RAID_ROSTER_UPDATE"] = function()
    HealAssign.UpdateRaidRoster();
end

HealAssign.eventHandlers["UNIT_HEALTH"] = function()
    HealAssign.HealthChanged( arg1 );
end

HealAssign.eventHandlers["UNIT_MAXHEALTH"] = function()
    HealAssign.HealthChanged( arg1 );
end

HealAssign.eventHandlers["CHAT_MSG_WHISPER"] = function()
    HealAssign.RespondToWhisper( arg2, arg1 );
end

HealAssign.eventHandlers["CHAT_MSG_ADDON"] = function()
    if arg1 == HEALASSIGN_COMMUNICATION_PREFIX and arg3 == "RAID" then
        HealAssign.HandleRaidAddonMessage( arg4, arg2 );
    end
end

HealAssign.eventHandlers["PLAYER_REGEN_DISABLED"] = function()
	HealAssign.inLockdown = true;
end

HealAssign.eventHandlers["PLAYER_REGEN_ENABLED"] = function()
	HealAssign.inLockdown = false;
	
	if HealAssign.pendingConfigureUnitButtons then
	   HealAssign.HandleConfigureUnitButtons();
	end
end

HealAssign.whisperHandlers[HEALASSIGN_WHISPER_COMMAND_TARGETS] = function( from, message )
    if HealAssign.IsHealer( from ) then
        local targetsString = HealAssign.TargetsStringForHealer( from, false );
        if targetsString then
            HealAssign.MessageWhisper( from, targetsString );
        else
            HealAssign.MessageWhisper( from, HEALASSIGN_MESSAGE_NO_TARGETS );
        end
    else
        HealAssign.MessageWhisper( from, HEALASSIGN_MESSAGE_NOT_A_HEALER );
    end
end

HealAssign.whisperHandlers[HEALASSIGN_WHISPER_COMMAND_HEALERS] = function( from, message )
    local attributes = HealAssign.raidMembers and HealAssign.raidMembers[from];
    if attributes then
        local healerString = HealAssign.HealersStringForTarget( from, false );
        local classString = HealAssign.HealersStringForTarget( HEALASSIGN_CLASS_TO_CODE[attributes.class], false );
        local groupString = HealAssign.HealersStringForTarget( HEALASSIGN_GROUP_TO_CODE[attributes.subgroup], false );
        if healerString or classString or groupString then
            if healerString then
                HealAssign.MessageWhisper( from, healerString );
            end
            if classString then
                HealAssign.MessageWhisper( from, classString );
            end
            if groupString then
                HealAssign.MessageWhisper( from, groupString );
            end
        else
            HealAssign.MessageWhisper( from, HEALASSIGN_MESSAGE_NO_HEALERS );
        end
    end
end

-- Full sync request
HealAssign.raidAddonMessageHandlers["R"] = function( from, message )
    HealAssign.HandleVersion( from, message );
    HealAssign.SendVersion();
    
    if HealAssign.IsServer( HealAssign.playerName ) then
        HealAssign.SendSync();
    elseif from == HealAssign.serverName then
        HealAssign.serverName = nil;
        HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_SERVER_LOST, HealAssign.ColorizeName( from ) ) );
    end
end

-- Version number
HealAssign.raidAddonMessageHandlers["V"] = function( from, message )
    HealAssign.HandleVersion( from, message );
    if from == HealAssign.serverName then
        HealAssign.serverName = nil;
        HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_SERVER_LOST, HealAssign.ColorizeName( from ) ) );
    end
end

-- Version number from server
HealAssign.raidAddonMessageHandlers["*V"] = function( from, message )
    HealAssign.HandleVersion( from, message );
    if HealAssign.IsPromoted( from ) and HealAssign.serverName ~= from then
        HealAssign.serverName = from;
        HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_SERVER_SET, HealAssign.ColorizeName( from ) ) );
    end
end

-- Clear all from client
HealAssign.raidAddonMessageHandlers["C"] = function( from, message )
    if HealAssign.IsServer( HealAssign.playerName ) then
        if not message or string.len( message ) == 0 then
            if HealAssign.IsPromoted( from ) then
                if HealAssign.ClearImp( from ) then
                    HealAssign.MessageRaidAddon( string.format( "*C %s", from ) );
                end
            end
        end
    end
end

-- Clear all from server
HealAssign.raidAddonMessageHandlers["*C"] = function( from, message )
    if HealAssign.IsServer( from ) then
        HealAssign.ClearImp( message );
    end
end

-- Assign from client
HealAssign.raidAddonMessageHandlers["A"] = function( from, message )
    if HealAssign.IsServer( HealAssign.playerName ) then
        local _, _, healerName, targetName = string.find( message, "^(%S+)%s(%S+)$" );
        if healerName and targetName then
            if HealAssign.IsPromoted( from ) or ( from == healerName ) then
                if HealAssign.AssignImp( from, healerName, targetName ) then
                    HealAssign.MessageRaidAddon( string.format( "*A %s %s %s", from, healerName, targetName ) );
                end
            end
        end
    end
end

-- Assign from server
HealAssign.raidAddonMessageHandlers["*A"] = function( from, message )
    if HealAssign.IsServer( from ) then
        local _, _, who, healerName, targetName = string.find( message, "^(%S+)%s(%S+)%s(%S+)$" );
        HealAssign.AssignImp( who, healerName, targetName );
    end
end

-- Unassign from client
HealAssign.raidAddonMessageHandlers["U"] = function( from, message )
    if HealAssign.IsServer( HealAssign.playerName ) then
        local _, _, healerName, targetName = string.find( message, "^(%S+)%s?(%S*)$" );
        if healerName then
            if HealAssign.IsPromoted( from ) or ( from == healerName ) then
                if HealAssign.UnassignImp( from, healerName, targetName ) then
                    if targetName and string.len( targetName ) > 0 then
                        HealAssign.MessageRaidAddon( string.format( "*U %s %s %s", from, healerName, targetName ) );
                    else
                        HealAssign.MessageRaidAddon( string.format( "*U %s %s", from, healerName ) );
                    end
                end
            end
        end
    end
end

-- Unassign from server
HealAssign.raidAddonMessageHandlers["*U"] = function( from, message )
    if HealAssign.IsServer( from ) then
        local _, _, who, healerName, targetName = string.find( message, "^(%S+)%s(%S+)%s?(%S*)$" );
        HealAssign.UnassignImp( who, healerName, targetName );
    end
end

-- Sync from server
HealAssign.raidAddonMessageHandlers["*S"] = function( from, message )
    if HealAssign.IsServer( from ) then
        local _, _, index, count, fragment = string.find( message, "^(%x+)/(%x+)%s?(.*)$" );
        index = tonumber( index, 16 );
        count = tonumber( count, 16 );
        
        if index == 1 then
            if count == 1 then
                HealAssign.ParseSync( from, fragment or "" );
                HealAssign.syncFragments = nil;
                HealAssign.syncSender = nil;
                if HealAssign.syncTimer then
                    HealAssign.syncTimer:stop();
                end
            else
                HealAssign.syncFragments = { fragment };
                HealAssign.syncSender = from;
                
                if not HealAssign.syncTimer then
                    HealAssign.syncTimer = HA_timer:new( 5, HealAssign.HandleSyncTimout, nil, false );
                else
                    HealAssign.syncTimer:reset();
                end
                HealAssign.syncTimer:run();
            end
        else
            if HealAssign.syncSender and from == HealAssign.syncSender then
                if HealAssign.syncFragments then
                    if ( # HealAssign.syncFragments + 1 ) == index then
                        table.insert( HealAssign.syncFragments, fragment );
                        if index == count then
                            HealAssign.ParseSync( from, table.concat( HealAssign.syncFragments ) );
                            HealAssign.syncFragments = nil;
                            HealAssign.syncSender = nil;
                            if HealAssign.syncTimer then
                                HealAssign.syncTimer:stop();
                            end
                        elseif HealAssign.syncTimer then
                            HealAssign.syncTimer:reset();
                        end
                    else
                        if HealAssign.syncTimer then
                            HealAssign.syncTimer:stop();
                        end
                        HealAssign.SyncFailed( HEALASSIGN_MESSAGE_SYNC_OUT_OF_ORDER );
                    end
                else
                    if HealAssign.syncTimer then
                        HealAssign.syncTimer:stop();
                    end
                    HealAssign.SyncFailed( HEALASSIGN_MESSAGE_SYNC_PARTIAL );
                end
            else
                HealAssign.SyncFailed( HEALASSIGN_MESSAGE_SYNC_UNEXPECTED_SENDER );
            end
        end
    end
end

function HealAssign.HandleSyncTimout()
    HealAssign.SyncFailed( HEALASSIGN_MESSAGE_SYNC_TIMEOUT );
end

function HealAssign.HandleVersion( from, message )
    if HealAssign.raidVersions then
        local fromVersion = HA_version:newFromString( message );
        HealAssign.raidVersions[from] = fromVersion;
        
        if fromVersion > HEALASSIGN_VERSION and ( not HealAssign.updatedVersion or fromVersion > HealAssign.updatedVersion ) then
            HealAssign.updatedVersion = fromVersion;
            HealAssign.MessagePlayer( HealAssign.ColorizeText( string.format( HEALASSIGN_MESSAGE_VERSION_UPDATED, tostring( fromVersion ) ), 1, 0.5, 0 ) );
        end
    end
end

function HealAssign.IsWhisperCommand( from, message )
    return HealAssign.raidMembers and HealAssign.raidMembers[from] and HealAssign.whisperHandlers and HealAssign.whisperHandlers[string.lower( message )];
end

function HealAssign.RespondToWhisper( from, message )
    local lowerMessage = string.lower( message );
    if HealAssign.IsWhisperCommand( from, lowerMessage ) then
        HealAssign.whisperHandlers[lowerMessage]( from, lowerMessage );
    end
end

function HealAssign.OnMouseDown( arg1 )
    if arg1 == "LeftButton" then
        this:StartMoving();
    end
end

function HealAssign.OnMouseUp( arg1 )
    if arg1 == "LeftButton" then
        this:StopMovingOrSizing();
    end
end

function HealAssign.AssignmentsOnMouseDown( arg1 )
    if arg1 == "LeftButton" and ( not HealAssign_Saved.locked or IsShiftKeyDown() ) then
        this:StartMoving();
    end
end

function HealAssign.AssignmentsOnMouseUp( arg1 )
    if arg1 == "LeftButton" then
        this:StopMovingOrSizing();
        HealAssign.SaveWindowPosition();
    end
end

function HealAssign.OnShow()
    PlaySound( "igMainMenuOpen" );
end

function HealAssign.OnHide()
    PlaySound( "igMainMenuClose" );
end

function HealAssign.OnLoad()
	tinsert( UISpecialFrames, "HealAssign_OptionsFrame" );
	
	SlashCmdList["HealAssign"] = function( msg )
		HealAssign.SlashCommand( msg );
	end
	
	SLASH_HealAssign1 = HEALASSIGN_SLASH_COMMAND1;
	SLASH_HealAssign2 = HEALASSIGN_SLASH_COMMAND2;
	
	for key in pairs( HealAssign.eventHandlers ) do
	  --DEFAULT_CHAT_FRAME:AddMessage( "Registering " .. key );
  	  this:RegisterEvent( key );
	end
	
	local oldChatFrame_OnEvent = ChatFrame_OnEvent;
	
	function ChatFrame_OnEvent( event )
        local passUpChatFrameEvent = true;
        
        if HealAssign.raidMembers then
            if event == "CHAT_MSG_WHISPER" then
                passUpChatFrameEvent = not HealAssign.IsWhisperCommand( arg2, arg1 );
            elseif event == "CHAT_MSG_WHISPER_INFORM" then
                passUpChatFrameEvent = string.find( arg1, HEALASSIGN_MESSAGE_PREFIX ) ~= 1;
            end
        end
        
        if passUpChatFrameEvent then
            oldChatFrame_OnEvent( event );
            --oldChatFrame_OnEvent( event, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9 );
        end
    end
    
	DEFAULT_CHAT_FRAME:AddMessage( string.format( HEALASSIGN_LOADED, tostring( HEALASSIGN_VERSION ) ) .. " " .. HEALASSIGN_BASIC_HELP );
end

function HealAssign.OnEvent( event )
  if HealAssign.eventHandlers[event] then
    HealAssign.eventHandlers[event]( arg1, arg2, arg3, arg4 );
  end
end

function HealAssign.OnUpdate( arg1 )
    HA_timermanager.update( arg1 );
end

function HealAssign.Init()
	HealAssign.playerName = UnitName( "player" );
	    
    local _, nonLocClass = UnitClass( "player" );
    
    HealAssign.spellRange = HEALASSIGN_SPELL_RANGE[nonLocClass];
    
    if nonLocClass == "DRUID" or nonLocClass == "PALADIN" or nonLocClass == "PRIEST" or nonLocClass == "SHAMAN" then
        HealAssign.playerIsHealer = true;
    else
        HealAssign.playerIsHealer = false;
    end
    
    HealAssign.SetMapIconAngle();
    
    if HealAssign_Saved.showMapIcon then
        HealAssign_IconFrame:Show();
    else
        HealAssign_IconFrame:Hide();
    end
    
    HealAssign.RestoreWindowPosition();
    HealAssign.ScaleWindow();
    
    PanelTemplates_SetNumTabs( HealAssign_OptionsFrame, 2 );
	PanelTemplates_SetTab( HealAssign_OptionsFrame, HealAssign_OptionsFrameTab1:GetID() );
	HealAssign_OptionsFrameAssignmentsFrame:Show();
    HealAssign_OptionsFrameOptionsFrame:Hide();
	PanelTemplates_UpdateTabs( HealAssign_OptionsFrame );
	
	HealAssign_OptionsShowMapButtonCheckbox:SetChecked( HealAssign_Saved.showMapIcon );
	HealAssign_OptionsSquareMapCheckbox:SetChecked( HealAssign_Saved.mapSquare );
	HealAssign_OptionsNotifyMemberAddedCheckbox:SetChecked( HealAssign_Saved.notifyMemberAdded );
	HealAssign_OptionsNotifyMemberOfflineCheckbox:SetChecked( HealAssign_Saved.notifyMemberOffline );
	HealAssign_OptionsNotifyAssignmentsChangedCheckbox:SetChecked( HealAssign_Saved.notifyAssignmentsChanged );
	
    if HealAssign.playerIsHealer then
        HealAssign_OptionsNotifyMyAssignmentsChangedCheckbox:SetChecked( HealAssign_Saved.playSoundMyAssignmentsChanged );
        HealAssign_OptionsIgnoreOwnHealsCheckbox:SetChecked( HealAssign_Saved.ignoreOwnHeals );
        HealAssign_OptionsShowAssignmentsCheckbox:SetChecked( HealAssign_Saved.showAssignments );
        HealAssign_OptionsLockAssignmentsCheckbox:SetChecked( HealAssign_Saved.locked );
    else
        HealAssign_OptionsNotifyMyAssignmentsChangedCheckbox:SetChecked( false );
        HealAssign_OptionsNotifyMyAssignmentsChangedCheckbox:Disable();
        getglobal( HealAssign_OptionsNotifyMyAssignmentsChangedCheckbox:GetName() .. "Text" ):SetTextColor( GRAY_FONT_COLOR.r, GRAY_FONT_COLOR.g, GRAY_FONT_COLOR.b );
        
        HealAssign_OptionsIgnoreOwnHealsCheckbox:SetChecked( false );
        HealAssign_OptionsIgnoreOwnHealsCheckbox:Disable();
        getglobal( HealAssign_OptionsIgnoreOwnHealsCheckbox:GetName() .. "Text" ):SetTextColor( GRAY_FONT_COLOR.r, GRAY_FONT_COLOR.g, GRAY_FONT_COLOR.b );
        
        HealAssign_OptionsShowAssignmentsCheckbox:SetChecked( false );
        HealAssign_OptionsShowAssignmentsCheckbox:Disable();
        getglobal( HealAssign_OptionsShowAssignmentsCheckbox:GetName() .. "Text" ):SetTextColor( GRAY_FONT_COLOR.r, GRAY_FONT_COLOR.g, GRAY_FONT_COLOR.b );
        
        HealAssign_OptionsLockAssignmentsCheckbox:SetChecked( false );
        HealAssign_OptionsLockAssignmentsCheckbox:Disable();
        getglobal( HealAssign_OptionsLockAssignmentsCheckbox:GetName() .. "Text" ):SetTextColor( GRAY_FONT_COLOR.r, GRAY_FONT_COLOR.g, GRAY_FONT_COLOR.b );
    end
    
    HealAssign_OptionsOpacitySlider:SetValue( HealAssign_Saved.opacity or 1 );
    HealAssign_OptionsScaleSlider:SetValue( HealAssign_Saved.scale or 1 );
end

function HealAssign.ExtractCommand( commandString )
    local command = nil;
    local arguments = nil;
    for arg in string.gmatch( string.lower( commandString ), "%S+" ) do
        if not command then
            command = arg;
        else
            if not arguments then
                arguments = {};
            end
            table.insert( arguments, arg );
        end
    end
    return command, arguments;
end

HealAssign.slashCommandHandlers[HEALASSIGN_SLASH_COMMAND_HELP] = function( arguments )
    HealAssign.Help( arguments );
end

HealAssign.slashCommandHandlers[HEALASSIGN_SLASH_COMMAND_OPTIONS] = function( arguments )
    HealAssign.OptionsToggle();
end

HealAssign.slashCommandHandlers[HEALASSIGN_SLASH_COMMAND_ASSIGN] = function( arguments )
    HealAssign.AssignCommand( arguments );
end

HealAssign.slashCommandHandlers[HEALASSIGN_SLASH_COMMAND_ASSIGN_TARGET] = function( arguments )
    HealAssign.AssignTargetCommand( arguments );
end

HealAssign.slashCommandHandlers[HEALASSIGN_SLASH_COMMAND_UNASSIGN] = function( arguments )
    HealAssign.UnassignCommand( arguments );
end

HealAssign.slashCommandHandlers[HEALASSIGN_SLASH_COMMAND_CLEAR] = function( arguments )
    HealAssign.ClearCommand( arguments );
end

HealAssign.slashCommandHandlers[HEALASSIGN_SLASH_COMMAND_LIST] = function( arguments )
    local argumentCount = ( arguments and # arguments ) or 0;
    
    if argumentCount == 0 then
        HealAssign.ListToPlayer();
    elseif argumentCount == 1 then
        local subCommand = arguments[1];
        
        if subCommand == HEALASSIGN_SLASH_COMMAND_SUB_LIST_RAID then
            HealAssign.ListToRaid();
        elseif subCommand == HEALASSIGN_SLASH_COMMAND_SUB_LIST_CHANNEL then
            HealAssign.ListToChannel();
        elseif subCommand == HEALASSIGN_SLASH_COMMAND_SUB_LIST_OFFICER then
            HealAssign.ListToOfficer();
        elseif subCommand == HEALASSIGN_SLASH_COMMAND_SUB_LIST_WHISPER then
            HealAssign.ListToWhisper();
        else
            HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_INVALID_ARGUMENT, subCommand ) );
        end
    else
        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_INVALID_ARGUMENT_COUNT );
    end
end

HealAssign.slashCommandHandlers[HEALASSIGN_SLASH_COMMAND_LISTTARGETS] = function( arguments )
    local argumentCount = ( arguments and # arguments ) or 0;
    
    if argumentCount == 0 then
        HealAssign.ListTargetsToPlayer();
    elseif argumentCount == 1 then
        local subCommand = arguments[1];
        
        if subCommand == HEALASSIGN_SLASH_COMMAND_SUB_LIST_RAID then
            HealAssign.ListTargetsToRaid();
        elseif subCommand == HEALASSIGN_SLASH_COMMAND_SUB_LIST_CHANNEL then
            HealAssign.ListTargetsToChannel();
        elseif subCommand == HEALASSIGN_SLASH_COMMAND_SUB_LIST_OFFICER then
            HealAssign.ListTargetsToOfficer();
        else
            HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_INVALID_ARGUMENT, subCommand ) );
        end
    else
        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_INVALID_ARGUMENT_COUNT );
    end
end

HealAssign.slashCommandHandlers[HEALASSIGN_SLASH_COMMAND_SETCHANNEL] = function( arguments )
    HealAssign.SetChannelCommand( arguments );
end

HealAssign.slashCommandHandlers[HEALASSIGN_SLASH_COMMAND_UNASSIGNED] = function( arguments )
    HealAssign.ListUnassignedHealers();
end

HealAssign.slashCommandHandlers[HEALASSIGN_SLASH_COMMAND_VERSIONCHECK] = function( arguments )
    HealAssign.VersionCheck();
end

HealAssign.slashCommandHandlers[HEALASSIGN_SLASH_COMMAND_VERSIONLIST] = function( arguments )
    HealAssign.VersionList();
end

HealAssign.slashCommandHandlers[HEALASSIGN_SLASH_COMMAND_SYNC] = function( arguments )
    HealAssign.SyncCommand();
end

HealAssign.slashCommandHandlers[HEALASSIGN_SLASH_COMMAND_SERVER] = function( arguments )
    HealAssign.ServerCommand();
end

HealAssign.slashCommandHandlers[HEALASSIGN_SLASH_COMMAND_SET] = function( arguments )
    local argumentCount = ( arguments and # arguments ) or 0;
    
    if argumentCount > 0 then
        local subCommand = arguments[1];
        
        if subCommand == HEALASSIGN_SLASH_COMMAND_SUB_SET_SAVE then
            if argumentCount == 2 then
                HealAssign.SaveSet( arguments[2] )
            else
                HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_INVALID_ARGUMENT_COUNT );
            end
        elseif subCommand == HEALASSIGN_SLASH_COMMAND_SUB_SET_LOAD then
            if argumentCount == 2 then
                HealAssign.LoadSetFromPartialName( arguments[2] )
            else
                HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_INVALID_ARGUMENT_COUNT );
            end
        elseif subCommand == HEALASSIGN_SLASH_COMMAND_SUB_SET_ERASE then
            if argumentCount >= 2 then
                for i = 2, argumentCount do
                    HealAssign.EraseSet( arguments[i] );
                end
            else
                HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_INVALID_ARGUMENT_COUNT );
            end
        elseif subCommand == HEALASSIGN_SLASH_COMMAND_SUB_SET_LIST then
            if argumentCount == 1 then
                HealAssign.ListSets();
            else
                HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_INVALID_ARGUMENT_COUNT );
            end
        else
            HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_INVALID_ARGUMENT, subCommand ) );
        end
    else
        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_INVALID_ARGUMENT_COUNT );
    end
end

--[[
HealAssign.slashCommandHandlers["dumpraid"] = function( arguments )
    HealAssign.DumpRaid();
end

HealAssign.slashCommandHandlers["dumphealers"] = function( arguments )
    HealAssign.DumpHealers();
end
--]]

function HealAssign.SlashCommand( msg )
    local command, arguments = HealAssign.ExtractCommand( msg );
    
    if HealAssign.slashCommandHandlers and HealAssign.slashCommandHandlers[command] then
        HealAssign.slashCommandHandlers[command]( arguments );
    else
        if command and string.len( command ) > 0 then
            HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_UNKNOWN_COMMAND, command ) );
        else
            DEFAULT_CHAT_FRAME:AddMessage( HEALASSIGN_TITLE );
        end
        DEFAULT_CHAT_FRAME:AddMessage( HEALASSIGN_BASIC_HELP );
    end
end

function HealAssign.Help( arguments )
    local argumentCount = ( arguments and # arguments ) or 0;
    if argumentCount == 0 then
        DEFAULT_CHAT_FRAME:AddMessage( HEALASSIGN_TITLE );
        DEFAULT_CHAT_FRAME:AddMessage( HEALASSIGN_HELP_INTRO );
        for cmd in HealAssign.PairsByKeys( HealAssign.slashCommandHandlers ) do
            DEFAULT_CHAT_FRAME:AddMessage( string.format( HEALASSIGN_HELP_FORMATTER, HealAssign.ColorizeText( cmd, 1, 1, 0 ), HEALASSIGN_HELP_BASIC_LISTING[cmd] or HEALASSIGN_HELP_NOT_AVAILABLE ) );
        end
    else
        local cmd = arguments[1];
        local help = HEALASSIGN_HELP_BASIC_LISTING[cmd];
        if help then
            DEFAULT_CHAT_FRAME:AddMessage( string.format( HEALASSIGN_HELP_FORMATTER, HealAssign.ColorizeText( cmd, 1, 1, 0 ), help ) );
            local help2 = HEALASSIGN_HELP_LISTING[cmd];
            if help2 then
                DEFAULT_CHAT_FRAME:AddMessage( help2 );
            end
        else
            HealAssign.MessagePlayer( string.format( HEALASSIGN_HELP_NOT_AVAILABLE_FOR_TOPIC, cmd ) );
        end
    end
end

function HealAssign.OptionsShow()
    HealAssign_OptionsFrame:Show();
end

function HealAssign.OptionsHide()
    HealAssign_OptionsFrame:Hide();
end

function HealAssign.OptionsToggle()
    if HealAssign_OptionsFrame:IsShown() then
        HealAssign.OptionsHide();
    else
        HealAssign.OptionsShow();
    end
end

function HealAssign.CopyAssignments( assignments )
    local result = nil;
    if assignments and next( assignments ) then
        result = {};
        for healer, targets in pairs( assignments ) do
            local newTargets = {};
            for k, v in pairs( targets ) do
                newTargets[k] = v;
            end
            result[healer] = newTargets;
        end
    end
	return result;
end

function HealAssign.AnyAssignments()
    local result = false;
    if HealAssign_Saved.healingAssignments then
        for healer, assignments in pairs( HealAssign_Saved.healingAssignments ) do
            if assignments and next( assignments ) then
                result = true;
                break;
            end
        end
    end
    return result;
end

function HealAssign.CanClear()
    return HealAssign.raidMembers and HealAssign.IsPromoted( HealAssign.playerName ) and HealAssign.AnyAssignments();
end

function HealAssign.CanAssign( healerName, targetName )
    return HealAssign.raidMembers and ( HealAssign.IsPromoted( HealAssign.playerName ) or ( healerName == HealAssign.playerName ) ) and not HealAssign.IsAssigned( healerName, targetName );
end

function HealAssign.CanUnassign( healerName, targetName )
    local result = false;
    if HealAssign.raidMembers and ( HealAssign.IsPromoted( HealAssign.playerName ) or ( healerName == HealAssign.playerName ) ) then
        result = HealAssign.IsAssigned( healerName, targetName );
    end
    return result;
end

function HealAssign.ClearImp( who )
    local success = false;
    
    if who and HealAssign_Saved.healingAssignments then
        
        local myAssignmentsChanged = HealAssign.playerAssignments and next( HealAssign.playerAssignments );
        
        HealAssign_Saved.healingAssignments = nil;
        
        HealAssign.playerAssignments = nil;
        
        if HealAssign_Saved.notifyAssignmentsChanged then
            HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_ASSIGNMENTS_CLEARED, HealAssign.ColorizeName( who ) ) );
        end
        
        if myAssignmentsChanged then
            HealAssign.MyAssignmentsChanged();
        end
        
        success = true;
    end
    
    return success;
end

function HealAssign.AssignImp( who, healerName, targetName )
    local success = false;
    
    if who and healerName and targetName then
        if not HealAssign_Saved.healingAssignments then
            HealAssign_Saved.healingAssignments = {};
        end
        
        if not HealAssign_Saved.healingAssignments[healerName] then
            HealAssign_Saved.healingAssignments[healerName] = {};
        end
        
        if not HealAssign_Saved.healingAssignments[healerName][targetName] then
            HealAssign_Saved.healingAssignments[healerName][targetName] = true;
            
            if HealAssign_Saved.notifyAssignmentsChanged then
                HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_HEALER_ASSIGNED, HealAssign.ColorizeName( who ), HealAssign.FormatName( healerName, true ), HealAssign.FormatName( targetName, true ) ) );
            end
            
            if healerName == HealAssign.playerName then
                if HealAssign.AddPlayerAssignment( targetName ) then
                    HealAssign.MyAssignmentsChanged();
                end
            end
            
            success = true;
        end
    end
    
    return success;
end

function HealAssign.UnassignImp( who, healerName, targetName )
    local success = false;
    
    if who and healerName then
        
        local unassignSingle = targetName and string.len( targetName ) > 0;
        
        if unassignSingle then
            if HealAssign.IsAssigned( healerName, targetName ) then
                HealAssign_Saved.healingAssignments[healerName][targetName] = nil;
                
                if HealAssign_Saved.notifyAssignmentsChanged then
                    HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_HEALER_UNASSIGNED, HealAssign.ColorizeName( who ), HealAssign.FormatName( healerName, true ), HealAssign.FormatName( targetName, true ) ) );
                end
                
                success = true;
            end
        else
            if HealAssign.IsAssigned( healerName ) then
                HealAssign_Saved.healingAssignments[healerName] = nil;
                
                if HealAssign_Saved.notifyAssignmentsChanged then
                    HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_HEALER_UNASSIGNED_ALL, HealAssign.ColorizeName( who ), HealAssign.FormatName( healerName, true ) ) );
                end
                
                success = true;
            end
        end
        
        if success and ( healerName == HealAssign.playerName ) then
            
            local myAssignmentsChanged = false;
            
            if HealAssign.playerAssignments then
                if unassignSingle then
                    myAssignmentsChanged = HealAssign.RemovePlayerAssignment( targetName );
                else
                    HealAssign.playerAssignments = nil;
                    myAssignmentsChanged = true;
                end
            end
            
            if myAssignmentsChanged then
                HealAssign.MyAssignmentsChanged();
            end
        end
    end
    
    return success;
end

function HealAssign.SendVersion()
    if HealAssign.IsServer( HealAssign.playerName ) then
        HealAssign.MessageRaidAddon( string.format( "*V %s", tostring( HEALASSIGN_VERSION ) ) );
    else
        HealAssign.MessageRaidAddon( string.format( "V %s", tostring( HEALASSIGN_VERSION ) ) );
    end
end

function HealAssign.SendSyncRequest()
    HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_REQUESTING_SYNC );
    HealAssign.MessageRaidAddon( string.format( "R %s", tostring( HEALASSIGN_VERSION ) ) );
end

function HealAssign.SyncCommand()
    if HealAssign.raidMembers then
        if HealAssign.IsServer( HealAssign.playerName ) then
            HealAssign.SendSync();
        else
            HealAssign.SendSyncRequest();
        end
    else
        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_IN_RAID );
    end
end

function HealAssign.SendSync()
    if HealAssign.raidMembers and HealAssign.IsServer( HealAssign.playerName ) then
        local syncList = {};
        if HealAssign_Saved.healingAssignments then
            for healer, assignments in pairs( HealAssign_Saved.healingAssignments ) do
                if assignments and next( assignments ) then
                    local targetsList = {};
                    for target in pairs( assignments ) do
                        table.insert( targetsList, target );
                    end
                    table.insert( syncList, string.format( "%s:%s.", healer, table.concat( targetsList, "," ) ) );
                end
            end
        end
        
        local fullSyncString = table.concat( syncList );
        
        local splits = HealAssign.SplitString( fullSyncString, 200 );
        
        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_SENDING_SYNC );
        
        if splits then
            local splitCount = # splits;
            for i, v in ipairs( splits ) do
                HealAssign.MessageRaidAddon( string.format( "*S %X/%X %s", i, splitCount, v ) );
            end
        else
            HealAssign.MessageRaidAddon( string.format( "*S 1/1" ) );
        end
    end
end

function HealAssign.ParseSync( who, syncMessage )
    local newAssignments = {};
    for healerString in string.gmatch( syncMessage, "[^%.]+" ) do
        local _, _, healerName, targetList = string.find( healerString, "^(%S+):(%S+)$" );
        if healerName and targetList then
            local healerSet = {};
            newAssignments[healerName] = healerSet;
            for targetName in string.gmatch( targetList, "[^,]+" ) do
                if targetName then
                    healerSet[targetName] = true;
                end
            end
        end
    end
    
    HealAssign_Saved.healingAssignments = newAssignments;
    
    HealAssign.ResetPlayerAssignments();
    
    HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_ASSIGNMENTS_SYNCED, HealAssign.ColorizeName( who ) ) );
end

function HealAssign.SyncFailed( reason )
    HealAssign.syncFragments = nil;
    HealAssign.syncSender = nil;
    HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_SYNC_FAILED, reason or HEALASSIGN_MESSAGE_SYNC_UNKNOWN ) );
end

function HealAssign.IsServer( name )
    return HealAssign.raidMembers and ( name == HealAssign.serverName );
end

function HealAssign.HasServer()
    return HealAssign.raidMembers and ( HealAssign.serverName ~= nil );
end

function HealAssign.IsHealer( name )
    local result = false;
    if HealAssign.raidHealers then
        local lowerName = string.lower( name );
        for _, healers in pairs( HealAssign.raidHealers ) do
            for healerName in pairs( healers ) do
                if lowerName == string.lower( healerName ) then
                    result = true;
                    break;
                end
            end
            if result then
                break;
            end
        end
    end
    return result;
end

function HealAssign.MyAssignmentsChanged()
    HealAssign.ConfigureUnitButtons();
    
    if HealAssign_Saved.playSoundMyAssignmentsChanged then
        PlaySoundFile( "Sound\\interface\\MagicClick.wav" );
    end
end

function HealAssign.StartMapIconDrag()
    if not HealAssign.mapIconDragTimer then
        HealAssign.mapIconDragTimer = HA_timer:new( 0.01, HealAssign.HandleMapIconDrag, nil, true );
    end
    HealAssign.mapIconDragTimer:run();
end

function HealAssign.StopMapIconDrag()
    if HealAssign.mapIconDragTimer then
        HealAssign.mapIconDragTimer:stop();
    end
end

function HealAssign.HandleMapIconDrag()
	local xpos,ypos = GetCursorPosition();
	local xmin,ymin = Minimap:GetLeft(), Minimap:GetBottom();

	xpos = xmin-xpos/Minimap:GetEffectiveScale() + 70;
	ypos = ypos/Minimap:GetEffectiveScale() - ymin - 70;

	HealAssign_Saved.mapIconAngle = math.deg( math.atan2( ypos, xpos ) );
	HealAssign.SetMapIconAngle();
end

function HealAssign.SetMapIconAngle()
	local angle = HealAssign_Saved.mapIconAngle or 0;
	local xpos = cos( angle );
	local ypos = sin( angle );
	if HealAssign_Saved.mapSquare then
        local tempx = HEALASSIGN_MAP_RADIUS;
        local tempy = HEALASSIGN_MAP_RADIUS;
        if ypos ~= 0 then
            local ratio = math.abs( xpos / ypos );
            if ratio > 1 then
                tempy = tempy / ratio;
            elseif ratio < 1 then
                tempx = tempx * ratio;
            end
        else
            tempy = 0;
        end
        xpos = ( xpos >= 0 and tempx ) or -tempx;
        ypos = ( ypos >= 0 and tempy ) or -tempy;
    else
        xpos = xpos * HEALASSIGN_MAP_RADIUS;
        ypos = ypos * HEALASSIGN_MAP_RADIUS;
    end
	HealAssign_IconFrame:SetPoint( "TOPLEFT", "Minimap", "TOPLEFT", 52 - xpos, ypos - 52 );
end

function HealAssign.ShowMapIconTooltip()
    if ( this:GetLeft() or 0 ) < 400 then
        GameTooltip:SetOwner(this,"ANCHOR_RIGHT");
    else
        GameTooltip:SetOwner(this,"ANCHOR_LEFT");
    end
    
    GameTooltip:AddLine( HEALASSIGN_TITLE );
    GameTooltip:AddLine( HEALASSIGN_TOOLTIP, 0.8, 0.8, 0.8, 1 );
    GameTooltip:Show();
end

function HealAssign.SplitString( stringToSplit, maxLength )
    local result = nil;
    if stringToSplit and maxLength and maxLength > 0 then
        for index = 1, string.len( stringToSplit ), maxLength do
            local chunk = string.sub( stringToSplit, index, index + maxLength - 1 );
            if not result then
                result = {};
            end
            table.insert( result, chunk );
        end
    end
    return result;
end

function HealAssign.ListToPlayer()
    if HealAssign.raidMembers then
        local assignments = HealAssign.AllAssignments( true );
		if assignments then
        	for i, v in ipairs( assignments ) do
				HealAssign.MessagePlayer( v );
			end
        else
            HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NO_ASSIGNMENTS );
        end
    else
        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_IN_RAID );
    end
end

function HealAssign.ListTargetsToPlayer()
    if HealAssign.raidMembers then
        local targets = HealAssign.AllTargets( true );
        if targets then
        	for i, v in ipairs( targets ) do
				HealAssign.MessagePlayer( v );
			end
        else
            HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NO_ASSIGNMENTS );
        end
    else
        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_IN_RAID );
    end
end

function HealAssign.ListToRaid()
    HealAssign.ListTo( "RAID" );
end

function HealAssign.ListTargetsToRaid()
    HealAssign.ListTargetsTo( "RAID" );
end

function HealAssign.ListToOfficer()
    if CanGuildInvite() then
        HealAssign.ListTo( "OFFICER" );
    else
        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_GUILD_OFFICER );
    end
end

function HealAssign.ListTargetsToOfficer()
    if CanGuildInvite() then
        HealAssign.ListTargetsTo( "OFFICER" );
    else
        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_GUILD_OFFICER );
    end
end

function HealAssign.ListTo( distribution, target )
    if HealAssign.raidMembers then
        if HealAssign.IsPromoted( HealAssign.playerName ) then
            local assignments = HealAssign.AllAssignments( false );
            if assignments then
                for i, v in ipairs( assignments ) do
                    HealAssign.SendMessage( v, distribution, target );
                end
            else
                HealAssign.SendMessage( HEALASSIGN_MESSAGE_NO_ASSIGNMENTS, distribution, target );
            end
        else
            HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_PROMOTED );
        end
    else
        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_IN_RAID );
    end
end

function HealAssign.ListTargetsTo( distribution, target )
    if HealAssign.raidMembers then
        if HealAssign.IsPromoted( HealAssign.playerName ) then
            local targets = HealAssign.AllTargets( false );
            if targets then
                for i, v in ipairs( targets ) do
                    HealAssign.SendMessage( v, distribution, target );
                end
            else
                HealAssign.SendMessage( HEALASSIGN_MESSAGE_NO_ASSIGNMENTS, distribution, target );
            end
        else
            HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_PROMOTED );
        end
    else
        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_IN_RAID );
    end
end

function HealAssign.ListToWhisper()
    if HealAssign.raidMembers then
        if HealAssign.IsPromoted( HealAssign.playerName ) then
            HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_WHISPERING_ASSIGNMENTS );
            for _, healers in pairs( HealAssign.raidHealers ) do
                for healerName in pairs( healers ) do
                    if HealAssign.raidMembers[healerName] and HealAssign.raidMembers[healerName].online then
                        local targetsString = HealAssign.TargetsStringForHealer( healerName, false );
                        if targetsString then
                            HealAssign.MessageWhisper( healerName, targetsString );
                        else
                            HealAssign.MessageWhisper( healerName, HEALASSIGN_MESSAGE_NO_TARGETS );
                        end
                    end
                end
            end
        else
            HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_PROMOTED );
        end
    else
        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_IN_RAID );
    end
end

function HealAssign.ListToChannel()
    if HealAssign.raidMembers then
        if HealAssign.IsPromoted( HealAssign.playerName ) then
            local assignments = HealAssign.AllAssignments( false );
            if assignments then
                for i, v in ipairs( assignments ) do
                    HealAssign.MessageChannel( v );
                end
            else
                HealAssign.MessageChannel( HEALASSIGN_MESSAGE_NO_ASSIGNMENTS );
            end
        else
            HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_PROMOTED );
        end
    else
        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_IN_RAID );
    end
end

function HealAssign.ListTargetsToChannel()
    if HealAssign.raidMembers then
        if HealAssign.IsPromoted( HealAssign.playerName ) then
            local targets = HealAssign.AllTargets( false );
            if targets then
                for i, v in ipairs( targets ) do
                    HealAssign.MessageChannel( v );
                end
            else
                HealAssign.MessageChannel( HEALASSIGN_MESSAGE_NO_ASSIGNMENTS );
            end
        else
            HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_PROMOTED );
        end
    else
        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_IN_RAID );
    end
end

function HealAssign.TargetSorter( a, b )
    local result = true;
    
    local aKeyWord = HEALASSIGN_CODE_TO_KEYWORD[a];
    local bKeyWord = HEALASSIGN_CODE_TO_KEYWORD[b];
    
    if not aKeyWord then
        if not bKeyWord then
            result = a < b;
        else
            result = true;
        end
    else
        if not bKeyWord then
            result = false;
        else
            result = aKeyWord < bKeyWord;
        end
    end
    
    return result;
end

function HealAssign.AllAssignments( colorize )
    local result = nil;
	
	if HealAssign.raidMembers and HealAssign_Saved.healingAssignments then
        for healer in HealAssign.PairsByKeys( HealAssign_Saved.healingAssignments ) do
            local targetsString = HealAssign.TargetsStringForHealer( healer, colorize );
            if targetsString then
				if result then
					table.insert( result, targetsString );
				else
					result = { targetsString };
				end
            end
        end
    end
    
	return result;
end

function HealAssign.AllTargets( colorize )
    local result = nil;
    
    if HealAssign.raidMembers and HealAssign_Saved.healingAssignments then
        
        for target in HealAssign.PairsByKeys( HealAssign.raidMembers ) do
            local healersString = HealAssign.HealersStringForTarget( target, colorize );
            if healersString then
                if result then
					table.insert( result, healersString );
				else
					result = { healersString };
				end
            end
        end
        
        for codeword in HealAssign.PairsByKeys( HEALASSIGN_CODE_TO_KEYWORD, HealAssign.TargetSorter ) do
            local healersString = HealAssign.HealersStringForTarget( codeword, colorize );
            if healersString then
                if result then
					table.insert( result, healersString );
				else
					result = { healersString };
				end
            end
        end
    end
    
    return result;
end

function HealAssign.HealersStringForTarget( target, colorize )
    local resultString = nil;
    
    local healerList = HealAssign.HealersForTarget( target, colorize );
    
    if healerList then
        resultString = string.format( "%s%s%s%s",
                                        HealAssign.FormatName( target, colorize ),
                                        HEALASSIGN_MESSAGE_TARGET_SEPARATOR,
                                        table.concat( healerList, HEALASSIGN_MESSAGE_LIST_SEPARATOR ),
                                        HEALASSIGN_MESSAGE_LIST_TERMINATOR );
    end
    
    return resultString;
end

function HealAssign.TargetsStringForHealer( healer, colorize )
    local resultString = nil;
    
    local targetsList = HealAssign.TargetsForHealer( healer, colorize );
    
    if targetsList then
        resultString = string.format( "%s%s%s%s",
                                        HealAssign.FormatName( healer, colorize ),
                                        HEALASSIGN_MESSAGE_HEALER_SEPARATOR,
                                        table.concat( targetsList, HEALASSIGN_MESSAGE_LIST_SEPARATOR ),
                                        HEALASSIGN_MESSAGE_LIST_TERMINATOR );
    end
        
    return resultString;
end

function HealAssign.TargetsForHealer( healer, colorize )
    local result = nil;
    if HealAssign.raidMembers and HealAssign_Saved.healingAssignments and HealAssign.raidMembers[healer] then
        local assignments = HealAssign_Saved.healingAssignments[healer];
        if assignments then
            for target in HealAssign.PairsByKeys( assignments, HealAssign.TargetSorter ) do
                if HealAssign.raidMembers[target] or HEALASSIGN_CODE_TO_KEYWORD[target] then
                    
                    local targetName = HealAssign.FormatName( target, colorize );
                    
                    if result then
                        table.insert( result, targetName );
                    else
                        result = { targetName };
                    end
                end
            end
        end
    end
    return result;
end

function HealAssign.HealersForTarget( target, colorize )
    local result = nil;
    if HealAssign.raidMembers and HealAssign_Saved.healingAssignments and ( HealAssign.raidMembers[target] or HEALASSIGN_CODE_TO_KEYWORD[target] ) then
        for healer, assignments in HealAssign.PairsByKeys( HealAssign_Saved.healingAssignments ) do
            if HealAssign.raidMembers[healer] and assignments[target] then
                
                local healerName = HealAssign.FormatName( healer, colorize );
                
                if result then
                    table.insert( result, healerName );
                else
                    result = { healerName };
                end
            end
        end
    end
    return result;
end

function HealAssign.IsAssigned( healerName, targetName )
    local result = false;
    if HealAssign.raidMembers and HealAssign_Saved.healingAssignments then
        local assignments = HealAssign_Saved.healingAssignments[healerName];
        if assignments then
            if targetName then
                result = assignments[targetName];
            else
                result = next( assignments );
            end
        end
    end
    return result;
end

function HealAssign.ServerCommand( arguments )
    if HealAssign.raidMembers then
        if not arguments then
            if HealAssign.IsPromoted( HealAssign.playerName ) then
                HealAssign.serverName = HealAssign.playerName;
                HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_SERVER_SET, HealAssign.ColorizeName( HealAssign.playerName ) ) );
                HealAssign.SendVersion();
                HealAssign.SendSync();
            else
                HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_PROMOTED );
            end
        else
            HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_INVALID_ARGUMENT_COUNT );
        end
    else
        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_IN_RAID );
    end
end

function HealAssign.ClearCommand( arguments )
    if HealAssign.raidMembers then
        if not arguments then
            HealAssign.Clear();
        else
            HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_INVALID_ARGUMENT_COUNT );
        end
    else
        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_IN_RAID );
    end
end

function HealAssign.Clear()
    if HealAssign.raidMembers then
        if HealAssign.IsPromoted( HealAssign.playerName ) then
            if HealAssign.AnyAssignments() then
                if HealAssign.IsServer( HealAssign.playerName ) then
                    if HealAssign.ClearImp( HealAssign.playerName ) then
                        HealAssign.MessageRaidAddon( string.format( "*C %s", HealAssign.playerName ) );
                    end
                else
                    if HealAssign.HasServer() then
                        HealAssign.MessageRaidAddon( "C" );
                    else
                        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NO_SERVER );
                    end
                end
            else
                HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NO_ASSIGNMENTS );
            end
        else
            HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_PROMOTED );
        end
    else
        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_IN_RAID );
    end
end

function HealAssign.AssignCommand( arguments )
    if HealAssign.raidMembers then
        local argumentCount = ( arguments and # arguments ) or 0;
        if argumentCount >= 2 then
            local healerName = HealAssign.FindRaidHealer( arguments[1] );
            if healerName then
                for i = 2, argumentCount do
                    local targetName = HealAssign.FindRaidMember( arguments[i] );
                    if targetName then
                        HealAssign.Assign( healerName, targetName );
                    end
                end
            end
        else
            HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_INVALID_ARGUMENT_COUNT );
        end
    else
        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_IN_RAID );
    end
end

function HealAssign.AssignTargetCommand( arguments )
    if HealAssign.raidMembers then
        local argumentCount = ( arguments and # arguments ) or 0;
        if argumentCount >= 2 then
            local targetName = HealAssign.FindRaidMember( arguments[1] );
            if targetName then
                for i = 2, argumentCount do
                    local healerName = HealAssign.FindRaidHealer( arguments[i] );
                    if healerName then
                        HealAssign.Assign( healerName, targetName );
                    end
                end
            end
        else
            HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_INVALID_ARGUMENT_COUNT );
        end
    else
        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_IN_RAID );
    end
end

function HealAssign.Assign( healerName, targetName )
    if HealAssign.raidMembers then
        if HealAssign.IsPromoted( HealAssign.playerName ) or ( healerName == HealAssign.playerName ) then
            if not HealAssign.IsAssigned( healerName, targetName ) then
                if HealAssign.IsServer( HealAssign.playerName ) then
                    if HealAssign.AssignImp( HealAssign.playerName, healerName, targetName ) then
                        HealAssign.MessageRaidAddon( string.format( "*A %s %s %s", HealAssign.playerName, healerName, targetName ) );
                    end
                else
                    if HealAssign.HasServer() then
                        HealAssign.MessageRaidAddon( string.format( "A %s %s", healerName, targetName ) );
                    else
                        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NO_SERVER );
                    end
                end
            else
                HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_HEALER_ALREADY_ASSIGNED, HealAssign.FormatName( healerName, true ), HealAssign.FormatName( targetName, true ) ) );
            end
        else
            HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_PROMOTED );
        end
    else
        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_IN_RAID );
    end
end

function HealAssign.UnassignCommand( arguments )
    if HealAssign.raidMembers then
        local argumentCount = ( arguments and # arguments ) or 0;
        if argumentCount >= 1 then
            local healerName = HealAssign.FindRaidHealer( arguments[1] );
            if healerName then
                if argumentCount > 1 then
                    for i = 2, argumentCount do
                        local targetName = HealAssign.FindRaidMember( arguments[i] );
                        if targetName then
                            HealAssign.Unassign( healerName, targetName );
                        end
                    end
                else
                    HealAssign.Unassign( healerName, nil );
                end
            end
        else
            HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_INVALID_ARGUMENT_COUNT );
        end
    else
        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_IN_RAID );
    end
end

function HealAssign.Unassign( healerName, targetName )
    if HealAssign.raidMembers then
        if HealAssign.IsPromoted( HealAssign.playerName ) or ( healerName == HealAssign.playerName ) then
            if targetName and string.len( targetName ) > 0 then
                if HealAssign.IsAssigned( healerName, targetName ) then
                    if HealAssign.IsServer( HealAssign.playerName ) then
                        if HealAssign.UnassignImp( HealAssign.playerName, healerName, targetName ) then
                            HealAssign.MessageRaidAddon( string.format( "*U %s %s %s", HealAssign.playerName, healerName, targetName ) );
                        end
                    else
                        if HealAssign.HasServer() then
                            HealAssign.MessageRaidAddon( string.format( "U %s %s", healerName, targetName ) );
                        else
                            HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NO_SERVER );
                        end
                    end
                else
                    HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_HEALER_NOT_ASSIGNED_TARGET, HealAssign.FormatName( healerName, true ), HealAssign.FormatName( targetName, true ) ) );
                end
            else
                if HealAssign.IsAssigned( healerName ) then
                    if HealAssign.IsServer( HealAssign.playerName ) then
                        if HealAssign.UnassignImp( HealAssign.playerName, healerName, nil ) then
                            HealAssign.MessageRaidAddon( string.format( "*U %s %s", HealAssign.playerName, healerName ) );
                        end
                    else
                        if HealAssign.HasServer() then
                            HealAssign.MessageRaidAddon( string.format( "U %s", healerName ) );
                        else
                            HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NO_SERVER );
                        end
                    end
                else
                    HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_HEALER_NOT_ASSIGNED, HealAssign.FormatName( healerName, true ) ) );
                end
            end
        else
            HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_PROMOTED );
        end
    else
        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_IN_RAID );
    end
end

function HealAssign.UnassignedHealers( colorize )
    local result = nil;
	
	if HealAssign.raidMembers and HealAssign.raidHealers then
		local temp = {};
		
		for _, healers in pairs( HealAssign.raidHealers ) do
			for name in pairs( healers ) do
				if not HealAssign.IsAssigned( name ) then
					temp[name] = true;
				end
			end
		end
		
        if next( temp ) then
            result = {};
            for healer in HealAssign.PairsByKeys( temp ) do
                table.insert( result, HealAssign.FormatName( healer, colorize ) );
            end
        end
	end
	
    return result;
end

function HealAssign.ListUnassignedHealers()
	if HealAssign.raidMembers then
		local unassigned = HealAssign.UnassignedHealers( true );
		if unassigned then
	        local resultString = string.format( "%s%s%s",
	                                               HEALASSIGN_MESSAGE_UNASSIGNED_HEALERS_PREFIX,
	                                               table.concat( unassigned, HEALASSIGN_MESSAGE_LIST_SEPARATOR ),
	                                               HEALASSIGN_MESSAGE_LIST_TERMINATOR );
	        
			HealAssign.MessagePlayer( resultString );
		else
			HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NO_UNASSIGNED_HEALERS );
		end
    else
        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_IN_RAID );
    end
end

function HealAssign.PrintVersion( name, vers )
    local colorName = HealAssign.ColorizeName( name );
    if vers then
        local versionString = tostring( vers );
        if vers < HEALASSIGN_VERSION then
            versionString = HealAssign.ColorizeText( versionString, 1, 0, 0 );
        elseif vers > HEALASSIGN_VERSION then
            versionString = HealAssign.ColorizeText( versionString, 0, 1, 0 );
        end
        
        if HealAssign.IsServer( name ) then
            HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_VERSION_FORMATTER_SERVER, colorName, versionString ) );
        else
            HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_VERSION_FORMATTER_USER, colorName, versionString ) );
        end
    else
        local attributes = HealAssign.raidMembers and HealAssign.raidMembers[name];
        if attributes and not attributes.online then
            HealAssign.MessagePlayer( string.format( "%s %s", colorName, HealAssign.ColorizeText( HEALASSIGN_MESSAGE_VERSION_OFFLINE, 1, 1, 0 ) ) );
        else
            HealAssign.MessagePlayer( string.format( "%s %s", colorName, HealAssign.ColorizeText( HEALASSIGN_MESSAGE_VERSION_NONE, 1, 0, 0 ) ) );
        end
    end
end

function HealAssign.VersionCheck()
    if HealAssign.raidMembers then
        local temp = {};
        
        if HealAssign.raidHealers then
            for _, healers in pairs( HealAssign.raidHealers ) do
                for name in pairs( healers ) do
                    local healerVersion = HealAssign.raidVersions and HealAssign.raidVersions[name];
                    if not healerVersion or healerVersion ~= HEALASSIGN_VERSION then
                        temp[name] = true;
                    end
                end
            end
        end
        
        if HealAssign.raidVersions and next( HealAssign.raidVersions ) then
            for name, vers in pairs( HealAssign.raidVersions ) do
                if vers ~= HEALASSIGN_VERSION then
                    temp[name] = true;
                end
            end
        end
        
        if next( temp ) then
            for name in HealAssign.PairsByKeys( temp ) do
                local vers = HealAssign.raidVersions and HealAssign.raidVersions[name];
                HealAssign.PrintVersion( name, vers );
            end
        else
            HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_VERSIONS_IDENTICAL );
        end
	else
        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_IN_RAID );
    end
end

function HealAssign.VersionList()
    if HealAssign.raidMembers then
        if HealAssign.raidVersions and next( HealAssign.raidVersions ) then
            for member, vers in HealAssign.PairsByKeys( HealAssign.raidVersions ) do
                HealAssign.PrintVersion( member, vers );
            end
        else
            HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NO_VERSIONS );
        end
    else
        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_IN_RAID );
    end
end

function HealAssign.RaidIndexForUnitID( unitID )
    if not unitIDMap then
        unitIDMap = {}; -- global
        for i = 1, MAX_RAID_MEMBERS do
            unitIDMap[HealAssign.UnitIDForRaidIndex( i )] = i;
        end
    end
    return unitIDMap[unitID];
end

function HealAssign.UnitIDForRaidIndex( index )
    return string.format( "raid%d", index );
end

function HealAssign.FindRaidMember( partialName )
    local result = nil;
    
    if HealAssign.raidMembers and partialName and string.len( partialName ) > 0 then
        local nameCandidates = {};
        result = HealAssign.KeyFromPartialKey( partialName, HealAssign.raidMembers, nameCandidates );
        if not result then
            if not next( nameCandidates ) then
                result = HealAssign.KeyFromPartialKey( partialName, HEALASSIGN_KEYWORD_TO_CODE, nameCandidates );
            end
            
            if not result then
                result = HealAssign.KeyFromCandidates( partialName, nameCandidates, HEALASSIGN_MESSAGE_MEMBER_NOT_FOUND, HEALASSIGN_MESSAGE_MEMBER_MULTIPLE_FOUND, HealAssign.KeywordFormatter );
            end
            
            if result then
                local keyword = HEALASSIGN_KEYWORD_TO_CODE[result];
                if keyword then
                    result = keyword;
                end
            end
        end
    end
    
    return result;
end

function HealAssign.FindRaidHealer( partialName )
    local result = nil;
    
    if HealAssign.raidHealers and partialName and string.len( partialName ) > 0 then
        local nameCandidates = {};
        for nonLocClass, members in pairs( HealAssign.raidHealers ) do
            result = HealAssign.KeyFromPartialKey( partialName, members, nameCandidates );
            if result then
                break;
            end
        end
        if not result then
            result = HealAssign.KeyFromCandidates( partialName, nameCandidates, HEALASSIGN_MESSAGE_HEALER_NOT_FOUND, HEALASSIGN_MESSAGE_HEALER_MULTIPLE_FOUND, HealAssign.KeywordFormatter );
        end
    end
    
    return result;
end

function HealAssign.KeywordFormatter( str )
    local result = nil;
    local codeword = HEALASSIGN_KEYWORD_TO_CODE[str];
    if codeword then
        local nonLocClass = HEALASSIGN_CODE_TO_CLASS[codeword];
        if nonLocClass then
            result = HealAssign.ColorizeName( str, nonLocClass );
        else
            result = str;
        end
    else
        result = HealAssign.FormatName( str, true );
    end
    return result;
end

function HealAssign.SetNameFormatter( str )
    return string.format( HEALASSIGN_SET_MULTIPLE_FORMATTER, str );
end

function HealAssign.KeyFromPartialKey( partialKey, keyList, keyCandidates )
    local result = nil;
    
    if partialKey and keyList and keyCandidates then
        for k in pairs( keyList ) do
            local match = HealAssign.MatchKey( partialKey, k );
            if match == 2 then
                result = k;
                break;
            elseif match == 1 then
                table.insert( keyCandidates, k );
            end
        end
    end
    
    return result;
end

function HealAssign.KeyFromCandidates( partialKey, keyCandidates, noneFoundErrorString, multipleFoundErrorString, multipleFormatter )
    local result = nil;
    
    local numberCandidates = # keyCandidates;
    if numberCandidates == 1 then
        result = keyCandidates[1];
    elseif numberCandidates > 1 then
        HealAssign.MessagePlayer( string.format( multipleFoundErrorString, partialKey ) );
        table.sort( keyCandidates );
        for i, v in ipairs( keyCandidates ) do
            if multipleFormatter then
                HealAssign.MessagePlayer( multipleFormatter( v ) );
            else
                HealAssign.MessagePlayer( v );
            end
        end
    else
        HealAssign.MessagePlayer( string.format( noneFoundErrorString, partialKey ) );
    end
    
    return result;
end

function HealAssign.SendMessage( message, distribution, target )
    if GetNumRaidMembers() > 0 and message then
        ChatThrottleLib:SendChatMessage( "NORMAL", HEALASSIGN_COMMUNICATION_PREFIX, HEALASSIGN_MESSAGE_PREFIX .. message, distribution, nil, target );
    end
end

function HealAssign.MessageRaidAddon( message )
    if GetNumRaidMembers() > 0 and message then
        ChatThrottleLib:SendAddonMessage( "ALERT", HEALASSIGN_COMMUNICATION_PREFIX, message, "RAID" );
    end
end

function HealAssign.MessagePlayer( message )
    if message then
        DEFAULT_CHAT_FRAME:AddMessage( HEALASSIGN_MESSAGE_PREFIX .. message );
    end
end

function HealAssign.MessageChannel( message )
    if message then
        local channel = ( HealAssign_Saved.channelName and GetChannelName( HealAssign_Saved.channelName ) );
        if channel and channel > 0 then
            ChatThrottleLib:SendChatMessage( "NORMAL", HEALASSIGN_COMMUNICATION_PREFIX, HEALASSIGN_MESSAGE_PREFIX .. message, "CHANNEL", nil, channel );
        else
            HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NO_CHANNEL );
        end
    end
end

function HealAssign.MessageWhisper( to, message )
    if to and message then
        ChatThrottleLib:SendChatMessage( "NORMAL", HEALASSIGN_COMMUNICATION_PREFIX, HEALASSIGN_MESSAGE_PREFIX .. message, "WHISPER", nil, to );
    end
end

function HealAssign.HandleRaidAddonMessage( from, message )
    --HealAssign.MessagePlayer( string.format( "%s: %s", from, message ) );
    
    if from and message and from ~= HealAssign.playerName then
        local _, _, cmd, arg = string.find( message, "^(%S+)%s*(.*)" );
        
        if HealAssign.raidAddonMessageHandlers and HealAssign.raidAddonMessageHandlers[cmd] then
            HealAssign.raidAddonMessageHandlers[cmd]( from, arg );
        else
            HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_UNKNOWN_MESSAGE_TYPE, HealAssign.ColorizeName( from ) ) );
        end
    end
end

function HealAssign.IsPromoted( name )
    local result = false;
    if HealAssign.raidMembers then
        local attributes = HealAssign.raidMembers[name];
        if attributes and attributes.rank and attributes.rank > 0 then
            result = true;
        end
    end
    return result;
end

function HealAssign.TabClick()
    PanelTemplates_Tab_OnClick( HealAssign_OptionsFrame );
    
	if this:GetName() == "HealAssign_OptionsFrameTab1" then
		PanelTemplates_SetTab( HealAssign_OptionsFrame, this:GetID() );
		HealAssign_OptionsFrameAssignmentsFrame:Show();
		HealAssign_OptionsFrameOptionsFrame:Hide();
		PlaySound( "igCharacterInfoTab" );
	elseif this:GetName() == "HealAssign_OptionsFrameTab2" then
		PanelTemplates_SetTab( HealAssign_OptionsFrame, this:GetID() );
		HealAssign_OptionsFrameAssignmentsFrame:Hide();
		HealAssign_OptionsFrameOptionsFrame:Show();
		PlaySound( "igCharacterInfoTab" );
	end
end

function HealAssign.UpdateRaidRoster()
    local numRaidMembers = GetNumRaidMembers();
    
    if numRaidMembers > 0 then
        local joinedRaid = not HealAssign.raidMembers;
        
        if not HealAssign.raidVersions then
            HealAssign.raidVersions = { [HealAssign.playerName] = HEALASSIGN_VERSION };
        end
        
        local newRaidMembers = {};
        
        local newRaidHealers = { DRUID = {}, PALADIN = {}, PRIEST = {}, SHAMAN = {} };
        
        for index = 1, numRaidMembers do
            local name, rank, subgroup, level, class, nonLocClass, zone, online, dead = GetRaidRosterInfo( index );
            if name then
                newRaidMembers[name] = { rank = rank, subgroup = subgroup, class = nonLocClass, online = online, dead = dead, index = index };
                if newRaidHealers[nonLocClass] then
                    newRaidHealers[nonLocClass][name] = true;
                end
            end
        end
        
        if HealAssign.raidMembers and HealAssign.playerName then
            local myNewAttributes = newRaidMembers[HealAssign.playerName];
            if myNewAttributes then
                local myOldAttributes = HealAssign.raidMembers[HealAssign.playerName];
                if myOldAttributes then
                    if myNewAttributes.rank ~= myOldAttributes.rank then
                        HealAssign.RankChanged( myNewAttributes.rank, myOldAttributes.rank );
                    end
                end
            end
        end
        
        if HealAssign.raidMembers then
            -- Track group changes
            for name, attributes in pairs( newRaidMembers ) do
                local oldAttributes = HealAssign.raidMembers[name];
                if oldAttributes then
                    local oldGroup = oldAttributes.subgroup;
                    if attributes.subgroup ~= oldGroup then
                        -- Slam the new group.
                        oldAttributes.subgroup = attributes.subgroup;
                        HealAssign.RaidMemberChangedSubGroup( name, attributes.subgroup, oldGroup );
                    end
                end
            end
            -- Track added raid members
            for name, attributes in pairs( newRaidMembers ) do
                local oldAttributes = HealAssign.raidMembers[name];
                if oldAttributes then
                    if attributes.online then
                        if not oldAttributes.online then
                            HealAssign.RaidMemberCameOnline( name, attributes.class );
                        end
                    else
                        if oldAttributes.online then
                            HealAssign.RaidMemberWentOffline( name, attributes.class );
                        end
                    end
                end
                
                local wasAdded = not HealAssign.raidMembers[name];
                HealAssign.raidMembers[name] = attributes;
                if wasAdded then
                    HealAssign.RaidMemberWasAdded( name, attributes.class, attributes.subgroup );
                end
            end
            -- Track removed raid members
            for name, attributes in pairs( HealAssign.raidMembers ) do
                if not newRaidMembers[name] and name ~= HealAssign.playerName then
                    HealAssign.raidMembers[name] = nil;
                    HealAssign.RaidMemberWasRemoved( name, attributes.class, attributes.subgroup );
                end
            end
        else
            HealAssign.raidMembers = newRaidMembers;
        end
        
        if HealAssign.raidHealers then
            -- Track added raid healers
            for nonLocClass, members in pairs( newRaidHealers ) do
                for name, attributes in pairs( members ) do
                    local wasAdded = not HealAssign.raidHealers[nonLocClass][name];
                    HealAssign.raidHealers[nonLocClass][name] = attributes;
                    if wasAdded then
                        HealAssign.RaidHealerWasAdded( name, nonLocClass );
                    end
                end
            end
            -- Track removed raid healers
            for nonLocClass, members in pairs( HealAssign.raidHealers ) do
                for name in pairs( members ) do
                    if not newRaidHealers[nonLocClass][name] and name ~= HealAssign.playerName then
                        HealAssign.raidHealers[nonLocClass][name] = nil;
                        HealAssign.RaidHealerWasRemoved( name, nonLocClass );
                    end
                end
            end
        else
            HealAssign.raidHealers = newRaidHealers;
        end
        
        if joinedRaid then
            HealAssign.JoinedRaid();
        end
    else
        local leftRaid = HealAssign.raidMembers;
        
        HealAssign.raidMembers = nil;
        HealAssign.raidHealers = nil;
        HealAssign.raidVersions = nil;
        HealAssign.playerAssignments = nil;
        HealAssign.serverName = nil;
        
        if leftRaid then
            HealAssign.LeftRaid();
        end
    end
    
    HealAssign.ConfigureUnitButtons();
end

function HealAssign.AddPlayerAssignment( name )
    local result = false;
    
    local class = HEALASSIGN_CODE_TO_CLASS[name];
    if class then
        for raidName, attributes in pairs( HealAssign.raidMembers ) do
            if attributes.class == class then
                if HealAssign.AddPlayerAssignmentImp( raidName ) then
                    result = true;
                end
            end
        end
    else
        local group = HEALASSIGN_CODE_TO_GROUP[name];
        if group then
            for raidName, attributes in pairs( HealAssign.raidMembers ) do
                if attributes.subgroup == group then
                    if HealAssign.AddPlayerAssignmentImp( raidName ) then
                        result = true;
                    end
                end
            end
        else
            result = HealAssign.AddPlayerAssignmentImp( name );
        end
    end
    
    return result;
end

function HealAssign.RemovePlayerAssignment( name )
    local result = false;
    
    local class = HEALASSIGN_CODE_TO_CLASS[name];
    if class then
        for raidName, attributes in pairs( HealAssign.raidMembers ) do
            if attributes.class == class then
                if not HealAssign.IsAssigned( HealAssign.playerName, raidName ) and not HealAssign.IsAssigned( HealAssign.playerName, HEALASSIGN_GROUP_TO_CODE[attributes.subgroup] ) then
                    if HealAssign.RemovePlayerAssignmentImp( raidName ) then
                        result = true;
                    end
                end
            end
        end
    else
        local group = HEALASSIGN_CODE_TO_GROUP[name];
        if group then
            for raidName, attributes in pairs( HealAssign.raidMembers ) do
                if attributes.subgroup == group then
                    if not HealAssign.IsAssigned( HealAssign.playerName, raidName ) and not HealAssign.IsAssigned( HealAssign.playerName, HEALASSIGN_CLASS_TO_CODE[attributes.class] ) then
                        if HealAssign.RemovePlayerAssignmentImp( raidName ) then
                            result = true;
                        end
                    end
                end
            end
        else
            local attributes = HealAssign.raidMembers and HealAssign.raidMembers[name];
            
            if attributes then
                if not HealAssign.IsAssigned( HealAssign.playerName, HEALASSIGN_CLASS_TO_CODE[attributes.class] ) and not
                        HealAssign.IsAssigned( HealAssign.playerName, HEALASSIGN_GROUP_TO_CODE[attributes.subgroup] ) then
                    result = HealAssign.RemovePlayerAssignmentImp( name );
                end
            else
                result = HealAssign.RemovePlayerAssignmentImp( name );
            end
        end
    end
    
    return result;
end

function HealAssign.AddPlayerAssignmentImp( name )
    local result = false;
    
    local attributes = HealAssign.raidMembers and HealAssign.raidMembers[name];
    if attributes then
    
        if not HealAssign.playerAssignments then
            HealAssign.playerAssignments = {};
        end
        
        local found = false;
        for i, v in ipairs( HealAssign.playerAssignments ) do
            if v == name then
                found = true;
                break;
            end
        end
        
        if not found then
            table.insert( HealAssign.playerAssignments, name );
            result = true;
        end
    end
    
    return result;
end

function HealAssign.RemovePlayerAssignmentImp( name )
    local result = false;
    
    if HealAssign.playerAssignments then
        for i, v in ipairs( HealAssign.playerAssignments ) do
            if v == name then
                table.remove( HealAssign.playerAssignments, i );
                result = true;
                break;
            end
        end
    end
    
    return result;
end

function HealAssign.JoinedRaid()
    HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_JOINED_RAID );
    HealAssign.ResetPlayerAssignments();
    HealAssign.SendSyncRequest();
end

function HealAssign.LeftRaid()
    HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_LEFT_RAID );
    for k in pairs( HealAssign.ownHeals ) do
        HealAssign.ownHeals[k] = nil;
    end
end

function HealAssign.RaidMemberWasAdded( name, nonLocClass, subgroup )
    if HealAssign_Saved.notifyMemberAdded then
        HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_MEMBER_JOINED_RAID, HealAssign.ColorizeName( name, nonLocClass ), HEALASSIGN_CLASS_NAMES[nonLocClass] ) );
    end
    
    if HealAssign.IsAssigned( HealAssign.playerName, name ) or
            HealAssign.IsAssigned( HealAssign.playerName, HEALASSIGN_CLASS_TO_CODE[nonLocClass] ) or
            HealAssign.IsAssigned( HealAssign.playerName, HEALASSIGN_GROUP_TO_CODE[subgroup] ) then
        
        if HealAssign.AddPlayerAssignment( name ) then
            HealAssign.MyAssignmentsChanged();
        end
    end
end

function HealAssign.RaidMemberWasRemoved( name, nonLocClass, subgroup )
    if HealAssign.raidVersions then
        HealAssign.raidVersions[name] = nil;
    end
    if HealAssign_Saved.notifyMemberAdded then
        HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_MEMBER_LEFT_RAID, HealAssign.ColorizeName( name, nonLocClass ), HEALASSIGN_CLASS_NAMES[nonLocClass] ) );
    end
    
    if HealAssign.RemovePlayerAssignment( name ) then
        HealAssign.MyAssignmentsChanged();
    end
    
    if name == HealAssign.serverName then
        HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_SERVER_LOST, HealAssign.ColorizeName( name, nonLocClass ), HEALASSIGN_CLASS_NAMES[nonLocClass] ) );
        HealAssign.serverName = nil;
    end
end

function HealAssign.RaidMemberCameOnline( name, nonLocClass )
    if HealAssign_Saved.notifyMemberOffline then
        HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_MEMBER_ONLINE, HealAssign.ColorizeName( name, nonLocClass ), HEALASSIGN_CLASS_NAMES[nonLocClass] ) );
    end
end

function HealAssign.RaidMemberWentOffline( name, nonLocClass )
    if HealAssign.raidVersions then
        HealAssign.raidVersions[name] = nil;
    end
    if HealAssign_Saved.notifyMemberOffline then
        HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_MEMBER_OFFLINE, HealAssign.ColorizeName( name, nonLocClass ), HEALASSIGN_CLASS_NAMES[nonLocClass] ) );
    end
    if name == HealAssign.serverName then
        HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_SERVER_LOST, HealAssign.ColorizeName( name, nonLocClass ), HEALASSIGN_CLASS_NAMES[nonLocClass] ) );
        HealAssign.serverName = nil;
    end
end

function HealAssign.RaidHealerWasAdded( name, nonLocClass )
end

function HealAssign.RaidHealerWasRemoved( name, nonLocClass )
end

function HealAssign.RaidMemberChangedSubGroup( name, newGroup, oldGroup )
    --HealAssign.MessagePlayer( string.format( "%s changed to %d from %d", name, newGroup, oldGroup ) );
    
    local attributes = HealAssign.raidMembers and HealAssign.raidMembers[name];
    if attributes then
        local myAssignmentsChanged = false;
        
        if HealAssign.IsAssigned( HealAssign.playerName, name ) or
                HealAssign.IsAssigned( HealAssign.playerName, HEALASSIGN_CLASS_TO_CODE[attributes.class] ) or
                HealAssign.IsAssigned( HealAssign.playerName, HEALASSIGN_GROUP_TO_CODE[newGroup] ) then
        
            myAssignmentsChanged = HealAssign.AddPlayerAssignment( name );
        else
            myAssignmentsChanged = HealAssign.RemovePlayerAssignment( name );
        end
        
        if myAssignmentsChanged then
            HealAssign.MyAssignmentsChanged();
        end
    end
end

function HealAssign.RankChanged( newRank, oldRank )
    local rankString = "";
    if newRank == 0 then
        rankString = HEALASSIGN_RAID_RANK_NORMAL;
    elseif newRank == 1 then
        rankString = HEALASSIGN_RAID_RANK_ASSISTANT;
    elseif newRank == 2 then
        rankString = HEALASSIGN_RAID_RANK_LEADER;
    end
    HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_RAID_RANK_CHANGED, rankString ) );
end

function HealAssign.EscapeString( str )
    if str then
        str = string.gsub( str, "%%", "%%%%" );
        str = string.gsub( str, "%(", "%%(" );
        str = string.gsub( str, "%)", "%%)" );
        str = string.gsub( str, "%.", "%%." );
        str = string.gsub( str, "%+", "%%+" );
        str = string.gsub( str, "%-", "%%-" );
        str = string.gsub( str, "%*", "%%*" );
        str = string.gsub( str, "%?", "%%?" );
        str = string.gsub( str, "%[", "%%[" );
        str = string.gsub( str, "%^", "%%^" );
        str = string.gsub( str, "%$", "%%$" );
    end
    return str;
end

-- Returns 0 for no match, 1 for possible match, 2 for exact match
function HealAssign.MatchKey( key, candidate )
	local result = 0;
	if key and candidate then
		candidate = string.lower( candidate );
		key = HealAssign.EscapeString( string.lower( key ) );
		local i, j = string.find( candidate, "^" .. key );
		if i then
			if i == 1 and j == string.len( candidate ) then
				result = 2;
			else
				result = 1;
			end
		end
	end
	return result;
end

function HealAssign.FormatName( name, colorize )
    local keyword = HEALASSIGN_CODE_TO_KEYWORD[name];
    if keyword then
        if colorize then
            local nonLocClass = HEALASSIGN_CODE_TO_CLASS[name];
            if nonLocClass then
                name = HealAssign.ColorizeName( keyword, nonLocClass )
            else
                name = keyword;
            end
        else
            name = keyword;
        end
    elseif colorize then
        name = HealAssign.ColorizeName( name );
    end
    return name;
end

function HealAssign.ColorizeName( name, class )
    if not class and HealAssign.raidMembers then
        local attributes = HealAssign.raidMembers[name];
        if attributes then
            class = attributes.class;
        end
    end
    
    local color;
    if class then
        color = RAID_CLASS_COLORS[class];
    end
    if not color then
        color = GRAY_FONT_COLOR;
    end
    
    return HealAssign.ColorizeText( name, color.r, color.g, color.b );
end

--[[
function HealAssign.DumpRaid()
    local numRaidMembers = GetNumRaidMembers();
    if numRaidMembers > 0 then
        DEFAULT_CHAT_FRAME:AddMessage( string.format( "Number raid members: %d", numRaidMembers ) );
        if HealAssign.raidMembers then
            for name, attributes in HealAssign.PairsByKeys( HealAssign.raidMembers ) do
                DEFAULT_CHAT_FRAME:AddMessage( string.format( "%s (%s) Group: %d", name, attributes.class, attributes.subgroup ) );
            end
        end
    else
        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_IN_RAID );
    end
end

function HealAssign.DumpHealers()
    if GetNumRaidMembers() > 0 then
        if HealAssign.raidHealers then
            for nonLocClass, members in HealAssign.PairsByKeys( HealAssign.raidHealers ) do
                if next( members ) then
                    local color = RAID_CLASS_COLORS[nonLocClass];
                    DEFAULT_CHAT_FRAME:AddMessage( string.format( "-- %s --", HEALASSIGN_CLASS_NAMES[nonLocClass] ), color.r, color.g, color.b );
                    for name in HealAssign.PairsByKeys( members ) do
                        DEFAULT_CHAT_FRAME:AddMessage( name );
                    end
                end
            end
        end
    else
        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_IN_RAID );
    end
end
--]]

function HealAssign.SetChannelCommand( arguments )
    if arguments and # arguments == 1 then
        HealAssign.SetChannel( arguments[1] );
    else
        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_INVALID_ARGUMENT_COUNT );
    end
end

function HealAssign.ResetPlayerAssignments()
    local myAssignmentsChanged = false;
    
    if HealAssign.raidMembers then
        local myAssignments = HealAssign_Saved.healingAssignments and HealAssign_Saved.healingAssignments[HealAssign.playerName];
        if myAssignments and next( myAssignments ) then
            
            for targetName in pairs( myAssignments ) do
                if ( HealAssign.raidMembers[targetName] or HEALASSIGN_CODE_TO_KEYWORD[targetName] ) and HealAssign.AddPlayerAssignment( targetName ) then
                    myAssignmentsChanged = true;
                end
            end
            
            if HealAssign.playerAssignments then
                for i, v in HealAssign.ipairsreverse( HealAssign.playerAssignments ) do
                    local attributes = HealAssign.raidMembers[v];
                    if not attributes or ( not myAssignments[v] and
                            not myAssignments[HEALASSIGN_CLASS_TO_CODE[attributes.class]] and
                            not myAssignments[HEALASSIGN_GROUP_TO_CODE[attributes.subgroup]] ) then
                        
                        table.remove( HealAssign.playerAssignments, i );
                        myAssignmentsChanged = true;
                    end
                end
            end
        
        elseif HealAssign.playerAssignments then
            if # HealAssign.playerAssignments > 0 then
                myAssignmentsChanged = true;
            end
            HealAssign.playerAssignments = nil;
        end
    end
    
    if myAssignmentsChanged then
        HealAssign.MyAssignmentsChanged();
    end
end

function HealAssign.OpacityChanged( value )
    HealAssign_Saved.opacity = value;
    if HealAssign_Window then
        HealAssign_Window:SetAlpha( HealAssign_Saved.opacity or 1 );
    end
end

function HealAssign.ScaleChanged( value )
    if not HealAssign.inLockdown then
        HealAssign_Saved.scale = value;
        HealAssign.ScaleWindow();
    else
        HealAssign_OptionsScaleSlider:SetValue( HealAssign_Saved.scale or 1 );
    end
end

function HealAssign.SaveWindowPosition()
	if HealAssign_Window then
        local x, y = HealAssign_Window:GetLeft(), HealAssign_Window:GetTop();
        local s = HealAssign_Window:GetEffectiveScale();
        local parentHeight = UIParent:GetHeight() * UIParent:GetEffectiveScale();
        HealAssign_Saved.posX = x*s;
        HealAssign_Saved.posY = y*s - parentHeight;
	end
end

function HealAssign.RestoreWindowPosition()
	if HealAssign_Window then
        local x = HealAssign_Saved.posX or 0;
        local y = HealAssign_Saved.posY or 0;
        local s = HealAssign_Window:GetEffectiveScale();
        HealAssign_Window:ClearAllPoints();
        HealAssign_Window:SetPoint( "TOPLEFT", UIParent, "TOPLEFT", x/s, y/s );
    end
end

function HealAssign.ScaleWindow()
    if HealAssign_Window then
        HealAssign_Window:SetClampedToScreen( false );
        HealAssign.SaveWindowPosition();
        HealAssign_Window:SetScale( HealAssign_Saved.scale or 1 );
        HealAssign.RestoreWindowPosition();
        HealAssign_Window:SetClampedToScreen( true );
    end
end

function HealAssign.ResetWindow()
    HealAssign_Saved.opacity = 1;
    HealAssign_Window:SetAlpha( 1 );
    HealAssign_OptionsOpacitySlider:SetValue( 1 );
    
    if not HealAssign.inLockdown then
        HealAssign_Saved.posX = 200;
        HealAssign_Saved.posY = -200;
        HealAssign_Saved.scale = 1;
        
        HealAssign_Window:SetScale( 1 );
        
        HealAssign.RestoreWindowPosition();
        
        HealAssign_OptionsScaleSlider:SetValue( 1 );
    end
end

function HealAssign.UnitHasBuff( unit, buffName )
    local i = 1;
    while true do
        local buffAtIndex = UnitBuff( unit, i );
        if buffAtIndex then
            if buffAtIndex == buffName then
                return true;
            end
        else
            break;
        end
        i = i + 1;
    end
    return false;
end

function HealAssign.HealthChanged( unitID )
    if GetNumRaidMembers() > 0 and HealAssign.playerIsHealer and HealAssign.unitButtons then
        for k, v in pairs( HealAssign.unitButtons ) do
            if unitID == v.raidID then
                HealAssign.UpdateUnitButton( v );
                break;
            end
        end
    end
end

function HealAssign.HandleUnitButtonRanges()
    if HealAssign.unitButtons then
        for k, v in pairs( HealAssign.unitButtons ) do            
            if v.raidID then
                if HealAssign.UpdateUnitButtonRange( v ) then
                    HealAssign.UpdateUnitButton( v );
                end
            end
        end
    end
end

function HealAssign.UpdateUnitButtonRange( unitButton )
    local changed = false;
    if unitButton then
        local oldValue = unitButton.oor;
        if unitButton.raidID and HealAssign.spellRange and ( IsSpellInRange( HealAssign.spellRange, unitButton.raidID ) ~= 1 ) then
            unitButton.oor = true;
        else
            unitButton.oor = nil;
        end
        if unitButton.oor ~= oldValue then
            changed = true;
        end
    end
    return changed;
end

function HealAssign.UpdateUnitButton( unitButton )
    if unitButton then
        if unitButton.raidID then
            local name = UnitName( unitButton.raidID );
            local _, class = UnitClass( unitButton.raidID )
            local health = UnitHealth( unitButton.raidID );
            local healthMax = UnitHealthMax( unitButton.raidID );
            local online = UnitIsConnected( unitButton.raidID );
            
            unitButton.name:SetText( name );
            
            local color = RAID_CLASS_COLORS[class] or GRAY_FONT_COLOR;
            
            unitButton.name:SetTextColor( color.r, color.g, color.b );
            
            unitButton.healthBar:SetMinMaxValues( 0, healthMax );
            unitButton.healthBar:SetValue( health );
            
            if unitButton.incomingBar then
                unitButton.incomingBar:SetMinMaxValues( 0, healthMax );
                if unitButton.incomingHeals then
                    unitButton.incomingBar:SetValue( health + unitButton.incomingHeals );
                else
                    unitButton.incomingBar:SetValue( 0 );
                end
            end
            
            if online and healthMax > 0 then
                local ratio = health / healthMax;
                                
                if ratio >= 0.5 then
                    unitButton.healthBar:SetStatusBarColor( 2 - ratio * 2, 1, 0, 1 );
                else
                    unitButton.healthBar:SetStatusBarColor( 1, ratio * 2, 0, 1 );
                end
            else
                unitButton.healthBar:SetStatusBarColor( GRAY_FONT_COLOR.r, GRAY_FONT_COLOR.g, GRAY_FONT_COLOR.b );
            end
            
            local statusText;
                
            if online then
                if class == "PRIEST" and HealAssign.UnitHasBuff( unitButton.raidID, HEALASSIGN_BUFF_SPIRIT_OF_REDEMPTION ) then
                    statusText = HEALASSIGN_STATUS_SPIRIT_OF_REDEMPTION;
                else
                    if UnitIsDead( unitButton.raidID ) then
                        if class == "HUNTER" and HealAssign.UnitHasBuff( unitButton.raidID, HEALASSIGN_BUFF_FEIGN_DEATH ) then
                            statusText = HEALASSIGN_STATUS_FEIGN_DEATH;
                        else
                            statusText = HEALASSIGN_STATUS_DEAD;
                        end
                    elseif UnitIsGhost( unitButton.raidID ) then
                        statusText = HEALASSIGN_STATUS_GHOST;
                    else
                        local deficit = health - healthMax;
                        if deficit < 0 then
                            statusText = tostring( deficit );
                        end
                        if unitButton.aggro then
                            if not statusText then
                                statusText = HEALASSIGN_STATUS_AGGRO;
                            end
                            statusText = HealAssign.ColorizeText( statusText, 1, 0, 0 );
                        end
                    end
                end
            else
                statusText = HEALASSIGN_STATUS_OFFLINE;
            end
            
            unitButton.status:SetText( statusText or "" );
            
            if online and unitButton.oor then
                unitButton:SetAlpha( 0.35 );
            else
                unitButton:SetAlpha( 1 );
            end
        else
            unitButton.name:SetText( "" );
            unitButton.status:SetText( "" );
            unitButton.healthBar:SetMinMaxValues( 0, 1 );
            unitButton.healthBar:SetValue( 0 );
            if unitButton.incomingBar then
                unitButton.incomingBar:SetMinMaxValues( 0, 1 );
                unitButton.incomingBar:SetValue( 0 );
            end
        end
    end
end

function HealAssign.ConfigureUnitButtons()
    if not HealAssign.configureUnitButtonsTimer then
        HealAssign.configureUnitButtonsTimer = HA_timer:new( 0.1, HealAssign.HandleConfigureUnitButtons, nil, false );
    end
    HealAssign.configureUnitButtonsTimer:run();
end

function HealAssign.HandleConfigureUnitButtons()
    
    if HealAssign.inLockdown then
        HealAssign.pendingConfigureUnitButtons = true;
    else
        if HealAssign.playerAssignments then
            local count = # HealAssign.playerAssignments;
            
            HealAssign.SetUnitButtonCount( count );
            
            if count > 0 and HealAssign_Window:IsShown() then
                table.sort( HealAssign.playerAssignments );
                
                -- Clique support
                if not ClickCastFrames then
                    ClickCastFrames = {};
                end
                
                for i, v in ipairs( HealAssign.playerAssignments ) do
                    local unitButton = HealAssign.unitButtons[i];
                    
                    local attributes = HealAssign.raidMembers and HealAssign.raidMembers[v];
                    
                    if attributes then
                        unitButton.raidID = HealAssign.UnitIDForRaidIndex( attributes.index );
                    else
                        unitButton.raidID = nil;
                    end
                    
                    if unitButton.raidID and banzai then
                        unitButton.aggro = banzai:GetUnitAggroByUnitId( unitButton.raidID );
                    else
                        unitButton.aggro = nil;
                    end
                    
                    HealAssign.UpdateUnitButtonRange( unitButton );
                    HealAssign.UpdateUnitButton( unitButton );
                    
                    unitButton:SetAttribute( "unit", unitButton.raidID );
                    unitButton:SetAttribute( "type", "target" );
                    
                    ClickCastFrames[unitButton] = true;
                end
            end
        else
            HealAssign.SetUnitButtonCount( 0 );
        end
        
        HealAssign.pendingConfigureUnitButtons = false;
    end
end

function HealAssign.RegisterAssignmentCallbacks()
    if banzai then
        banzai:RegisterCallback( HealAssign.BanzaiCallback );
    end
    
    if healComm then
        healComm.RegisterCallback( "HealAssign", "HealComm_DirectHealStart", HealAssign.HealComm_DirectHealStart );
        healComm.RegisterCallback( "HealAssign", "HealComm_DirectHealDelayed", HealAssign.HealComm_DirectHealDelayed );
        healComm.RegisterCallback( "HealAssign", "HealComm_DirectHealStop", HealAssign.HealComm_DirectHealStop );
        healComm.RegisterCallback( "HealAssign", "HealComm_HealModifierUpdate", HealAssign.HealComm_HealModifierUpdate );
    end
end

function HealAssign.UnregisterAssignmentCallbacks()
    if banzai then
        banzai:UnregisterCallback( HealAssign.BanzaiCallback );
    end
    
    if healComm then
        healComm.UnregisterCallback( "HealAssign", "HealComm_DirectHealStart" );
        healComm.UnregisterCallback( "HealAssign", "HealComm_DirectHealDelayed" );
        healComm.UnregisterCallback( "HealAssign", "HealComm_DirectHealStop" );
        healComm.UnregisterCallback( "HealAssign", "HealComm_HealModifierUpdate" );
    end
end

function HealAssign.SetUnitButtonCount( count )
    if not visibleUnitButtonCount then
        visibleUnitButtonCount = 0; -- global
    end
    
    if count > 0 and HealAssign_Saved.showAssignments then
        if not HealAssign_Window:IsShown() then
            HealAssign.RegisterAssignmentCallbacks();
            HealAssign_Window:Show();
        end
    else
        if HealAssign_Window:IsShown() then
            HealAssign.UnregisterAssignmentCallbacks()
            HealAssign_Window:Hide();
        end
    end
    
    if count ~= visibleUnitButtonCount then
        
        if not HealAssign.unitButtons then
            HealAssign.unitButtons = {};
        end
        
        if count > visibleUnitButtonCount then
            
            local gridFrameLevel = HealAssign_WindowGrid:GetFrameLevel();
            
            for i = visibleUnitButtonCount + 1, count do
                local unitButton = HealAssign.unitButtons[i];
                
                if unitButton then
                    unitButton:SetParent( HealAssign_WindowGrid );
                else
                    local newUnitButtonName = "HealAssign_UnitButton" .. i;
                    
                    unitButton = CreateFrame( "Button", newUnitButtonName, HealAssign_WindowGrid, "HealAssign_UnitButton" );
                    
                    if healComm then
                        unitButton.incomingBar = CreateFrame( "StatusBar", nil, unitButton );
                        unitButton.incomingBar:SetStatusBarTexture( "Interface\\TargetingFrame\\UI-StatusBar" );
                        unitButton.incomingBar:SetAllPoints( unitButton );
                        unitButton.incomingBar:SetStatusBarColor( 0.3, 0.4, 1, 1 );
                    end
                    
                    unitButton.healthBar = CreateFrame( "StatusBar", nil, unitButton.incomingBar or unitButton );
                    unitButton.healthBar:SetStatusBarTexture( "Interface\\TargetingFrame\\UI-StatusBar" );
                    unitButton.healthBar:SetAllPoints( unitButton );
                    
                    unitButton.name = unitButton.healthBar:CreateFontString( nil, "ARTWORK", "HealAssign_UnitFont" );
                    unitButton.name:SetJustifyH( "LEFT" );
                    unitButton.name:SetJustifyV( "MIDDLE" );
                    unitButton.name:SetAllPoints( unitButton.healthBar );
                    
                    unitButton.status = unitButton.healthBar:CreateFontString( nil, "OVERLAY", "HealAssign_UnitFont" );
                    unitButton.status:SetJustifyH( "RIGHT" );
                    unitButton.status:SetJustifyV( "MIDDLE" );
                    unitButton.status:SetAllPoints( unitButton.healthBar );
                    
                    table.insert( HealAssign.unitButtons, unitButton );
                end
                
                unitButton:SetFrameLevel( gridFrameLevel + 2 );
                
                unitButton:ClearAllPoints();
                
                if i == 1 then
                    unitButton:SetPoint( "TOPLEFT", HealAssign_WindowGrid, "TOPLEFT", 0, 0 );
                else
                    local previousButton = HealAssign.unitButtons[i-1];
                    if previousButton then
                        unitButton:SetPoint( "TOPLEFT", previousButton, "BOTTOMLEFT", 0, 0 );
                    end
                end
                
                unitButton:SetPoint( "RIGHT", HealAssign_WindowGrid, "RIGHT", 0, 0 );
                
                unitButton:Show();
            end
            
            -- Have to set this twice, or the bars don't have correct alpha.
            HealAssign_Window:SetAlpha( 0 );
            HealAssign_Window:SetAlpha( HealAssign_Saved.opacity or 1 );
        
        elseif count < visibleUnitButtonCount then
            for i = visibleUnitButtonCount, count + 1, -1 do
                local unitButton = HealAssign.unitButtons[i];
                if unitButton then
                    unitButton:Hide();
                    unitButton:SetParent( nil );
                    unitButton.raidID = nil;
                    unitButton.aggro = nil;
                end
            end
        end
        
        local firstUnitButton = HealAssign.unitButtons[1];
        if firstUnitButton then
            HealAssign_Window:SetHeight( HealAssign_Window:GetHeight() + ( ( count - visibleUnitButtonCount ) * firstUnitButton:GetHeight() ) );
        end
        
        visibleUnitButtonCount = count;
        
        if visibleUnitButtonCount > 0 then
            if not HealAssign.updateRangeTimer then
                HealAssign.updateRangeTimer = HA_timer:new( 0.5, HealAssign.HandleUnitButtonRanges, nil, true );
            end
            HealAssign.updateRangeTimer:run();
        else
            if HealAssign.updateRangeTimer then
                HealAssign.updateRangeTimer:stop();
            end
        end
    end
end

function HealAssign.SetChannel( channel, leaveOld )
    if HealAssign_Saved.channelName and GetChannelName( HealAssign_Saved.channelName ) == 0 then
        HealAssign_Saved.channelName = nil;
    end
    
    if channel ~= HealAssign_Saved.channelName then
        if HealAssign_Saved.channelName then
            if leaveOld then
                LeaveChannelByName( HealAssign_Saved.channelName );
            end
            HealAssign_Saved.channelName = nil;
        end
        
        if channel then
            if GetChannelName( channel ) == 0 then
                local zoneChannel, channelName = JoinChannelByName( channel, nil, DEFAULT_CHAT_FRAME:GetID() );
                --RemoveChatWindowChannel( DEFAULT_CHAT_FRAME:GetID(), channelName );
                
                if channelName then
                    HealAssign_Saved.channelName = channelName;
                end
            else
                HealAssign_Saved.channelName = channel;
            end
        end
    end
    
    if HealAssign_Saved.channelName then
        HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_CHANNEL_SET, HealAssign_Saved.channelName ) );
    else
        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NO_CHANNEL );
    end
end

function HealAssign.SaveSet( name )
    if HealAssign.AnyAssignments() then
        
        if not HealAssign_Saved.savedSets then
            HealAssign_Saved.savedSets = {};
        end
        
        local newSave = { saveTime = time(), assignments = HealAssign.CopyAssignments( HealAssign_Saved.healingAssignments ) };
        
        local overwrote = HealAssign_Saved.savedSets[name];
        
        HealAssign_Saved.savedSets[name] = newSave;
        
        if overwrote then
            HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_SET_OVERWROTE, name ) );
        else
            HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_SET_SAVED, name ) );
        end
    else
        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NO_ASSIGNMENTS );
    end
end

function HealAssign.LoadSetFromPartialName( partialName )
    if HealAssign.raidMembers then
        if HealAssign.IsPromoted( HealAssign.playerName ) then
            if HealAssign.IsServer( HealAssign.playerName ) then
                local name = nil;
                
                if partialName and string.len( partialName ) > 0 then
                    local nameCandidates = {};
                    name = HealAssign.KeyFromPartialKey( partialName, HealAssign_Saved.savedSets, nameCandidates );
                    if not name then
                        name = HealAssign.KeyFromCandidates( partialName, nameCandidates, HEALASSIGN_MESSAGE_SET_NOT_FOUND, HEALASSIGN_MESSAGE_SET_MULTIPLE_FOUND, HealAssign.SetNameFormatter );
                    end
                    
                    if name then
                        HealAssign.LoadSet( name );
                    end
                end
            else
                HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_SERVER );
            end
        else
            HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_PROMOTED );
        end
    else
        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_IN_RAID );
    end
end

function HealAssign.LoadSet( name )
    if HealAssign.raidMembers then
        if HealAssign.IsPromoted( HealAssign.playerName ) then
            if HealAssign.IsServer( HealAssign.playerName ) then
            
                local loadAssignments = HealAssign_Saved.savedSets and HealAssign_Saved.savedSets[name];
                
                if loadAssignments then
                    HealAssign_Saved.healingAssignments = HealAssign.CopyAssignments( loadAssignments.assignments );
                    HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_SET_LOADED, name ) );
                    HealAssign.ResetPlayerAssignments();
                    HealAssign.SendSync();
                else
                    HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_SET_NOT_FOUND, name ) );
                end
            else
                HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_SERVER );
            end
        else
            HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_PROMOTED );
        end
    else
        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_NOT_IN_RAID );
    end
end

function HealAssign.EraseSet( name )
    if HealAssign_Saved.savedSets and HealAssign_Saved.savedSets[name] then
        HealAssign_Saved.savedSets[name] = nil;
        HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_SET_ERASED, name ) );
    else
        HealAssign.MessagePlayer( string.format( HEALASSIGN_MESSAGE_SET_NOT_FOUND, name ) );
    end
end

function HealAssign.ListSets()
    if HealAssign_Saved.savedSets and next( HealAssign_Saved.savedSets ) then
        for name, save in HealAssign.PairsByKeys( HealAssign_Saved.savedSets ) do
            HealAssign.MessagePlayer( string.format( HEALASSIGN_SET_FORMATTER, name, date( HEALASSIGN_SET_DATE_FORMATTER, save.saveTime ) ) );
        end
    else
        HealAssign.MessagePlayer( HEALASSIGN_MESSAGE_SET_NONE );
    end
end

function HealAssign.BanzaiCallback( aggro, name, ... )
    if GetNumRaidMembers() > 0 and HealAssign.playerIsHealer and HealAssign.unitButtons then
        for k, v in pairs( HealAssign.unitButtons ) do
            if v.raidID then
                for i = 1, select( "#", ... ) do
                    local unitID = select( i, ... );
                    if unitID == v.raidID then
                        if aggro == 1 then
                            v.aggro = true;
                        elseif aggro == 0 then
                            v.aggro = nil;
                        end
                        HealAssign.UpdateUnitButton( v );
                        return;
                    end
                end
            end
        end
    end
end

function HealAssign.UpdateIncomingHeals( unitButton )
    if healComm and unitButton then
        local raidID = unitButton.raidID
        if raidID then
            local incoming = healComm:UnitIncomingHealGet( raidID, GetTime() + 100 ) or 0;
            if not HealAssign_Saved.ignoreOwnHeals then
                incoming = incoming + ( HealAssign.ownHeals[UnitName( raidID )] or 0 );
            end
            local effectiveIncoming;
            if incoming > 0 then
                effectiveIncoming = incoming * healComm:UnitHealModifierGet( raidID );
            end
            if unitButton.incomingHeals ~= effectiveIncoming then
                unitButton.incomingHeals = effectiveIncoming;
                HealAssign.UpdateUnitButton( unitButton );
            end
        end
    end
end

function HealAssign.HealComm_DirectHealStart( event, healerName, healSize, endTime, ... )
    local isOwnHeal = healerName == HealAssign.playerName;
    if isOwnHeal or HealAssign.unitButtons then
        for i = 1, select( "#", ... ) do
            local targetName = select( i, ... );
            if isOwnHeal then
                HealAssign.ownHeals[targetName] = healSize;
            end
            if HealAssign.unitButtons then
                for k, v in pairs( HealAssign.unitButtons ) do            
                    local raidID = v.raidID;
                    if raidID and UnitName( raidID ) == targetName then
                        HealAssign.UpdateIncomingHeals( v );
                        break;
                    end
                end
            end
        end
    end
end

function HealAssign.HealComm_DirectHealDelayed( event, healerName, healSize, endTime, ... )
    if HealAssign.unitButtons then
        for i = 1, select( "#", ... ) do
            local targetName = select( i, ... );
            for k, v in pairs( HealAssign.unitButtons ) do            
                local raidID = v.raidID;
                if raidID and UnitName( raidID ) == targetName then
                    HealAssign.UpdateIncomingHeals( v );
                    break;
                end
            end
        end
    end
end

function HealAssign.HealComm_DirectHealStop( event, healerName, healSize, succeeded, ... )
    if healerName == HealAssign.playerName then
        for k in pairs( HealAssign.ownHeals ) do
            HealAssign.ownHeals[k] = nil;
        end
    end
    if HealAssign.unitButtons then
        for i = 1, select( "#", ... ) do
            local targetName = select( i, ... );
            for k, v in pairs( HealAssign.unitButtons ) do            
                local raidID = v.raidID;
                if raidID and UnitName( raidID ) == targetName then
                    HealAssign.UpdateIncomingHeals( v );
                    break;
                end
            end
        end
    end
end

function HealAssign.HealComm_HealModifierUpdate( event, unit, targetName, healModifier )
    if HealAssign.unitButtons then
        for k, v in pairs( HealAssign.unitButtons ) do            
            local raidID = v.raidID;
            if raidID and UnitIsUnit( raidID, unit ) then
                HealAssign.UpdateIncomingHeals( v );
                break;
            end
        end
    end
end

function HealAssign.ScrollBar_Update()
    FauxScrollFrame_Update( HealAssign_ScrollBar, 50, 5, 16 );
    -- DEFAULT_CHAT_FRAME:AddMessage( "We're at " .. FauxScrollFrame_GetOffset( HealAssign_ScrollBar ) );
    -- 50 is max entries, 5 is number of lines, 16 is pixel height of each line
end

function HealAssign.ipairsreverse( a )
    return function ( a, i ) i = i - 1; if i > 0 then return i, a[i]; end; end, a, # a + 1;
end

function HealAssign.PairsByKeys( t, f )
    local a = {}
    for n in pairs( t ) do
        table.insert( a, n )
    end
    table.sort( a, f )
    local i = 0      -- iterator variable
    local iter = function()   -- iterator function
        i = i + 1
        if a[i] == nil then
            return nil
        else
            return a[i], t[a[i]]
        end
    end
    return iter
end

function HealAssign.ColorizeText( text, red, green, blue )
    return string.format( "|c00%.2x%.2x%.2x%s|r",
                            math.floor( red * 255 ),
                            math.floor( green * 255 ),
                            math.floor( blue * 255 ),
                            text );
end
