-- GuildMap by Svetlania on Azjol-Nerub(US) - wow@docbru.de
-- Original design by Bru
-- reference(s) to Ace Libraries
local Astrolabe = AceLibrary("Astrolabe-GM-0.2")
local L = AceLibrary("AceLocale-2.2"):new("GuildMap")
local BZ = AceLibrary("Babble-Zone-2.2")
local playerName = UnitName("player")

-- construct addon and add mixins
GuildMap = AceLibrary("AceAddon-2.0"):new("AceEvent-2.0")

-- seconds after which the UI shall be repainted
local UPDATE_INTERVAL = 0.50

-- seconds to wait between announces when a player is moving
local EARLIEST_ANNOUNCE = 3.0
-- seconds after which an announce must take place when a player is standing still
local LATEST_ANNOUNCE = 15.0

-- seconds after which a player without updated status info will be removed
local PLAYER_TIMEOUT = 20.0

-- constants (do not modify)
local MAX_MINIMAP_ICONS = 75
local MAX_WORLDMAP_ICONS = 250

local lastAnnounce = 0
local versionWarning = true
local announceVersion = true
local pendingAnnounce = false
local players = {}
local position = {continent = 0, zone = 0, x = 0, y = 0}

-- Called when the addon is loaded
function GuildMap:OnInitialize()
	self:InitRegionData()
	
	local loadedMessage = string.format(L["GuildMap (rev. %s) by Svetlania loaded"], self.version);
	UIErrorsFrame:AddMessage(loadedMessage, 0.0, 1.0, 0.0, 1.0, UIERRORS_HOLD_TIME);
end

-- Called when the addon is enabled
function GuildMap:OnEnable()
	self:RegisterEvent("CHAT_MSG_ADDON")
	self:RegisterEvent("WORLD_MAP_UPDATE")
	self:RegisterEvent("RAID_ROSTER_UPDATE", "UpdatePartyOrRaid")
	self:RegisterEvent("PARTY_MEMBERS_CHANGED", "UpdatePartyOrRaid")

	self:ScheduleRepeatingEvent(self.Update, UPDATE_INTERVAL, self)
end

local function handleSplitMessage(sender, ...)
	for i = 1, select("#", ...) do
		GuildMap:HandleMessage(sender, (select(i, ...)))
	end
end

function GuildMap:CHAT_MSG_ADDON(prefix, message, distribution, sender)
	if prefix ~= "GUILDMAP" or playerName == sender then return end

	if message:find("#") then
		handleSplitMessage(sender, strsplit("#", message))
	else
		self:HandleMessage(sender, message)
	end
end

function GuildMap:WORLD_MAP_UPDATE()
	if WorldMapFrame:IsVisible() then
		self:UpdateWorldMap()
	end
end

function GuildMap:UpdateWorldMap()
	local currentContinent = GetCurrentMapContinent()
	-- there was a crash for battlegrounds (which return -1 for continent I think)
	if currentContinent < 0 or currentContinent > 3 then
		for i = 1, MAX_WORLDMAP_ICONS do
			local poi = getglobal("GuildMapMain" .. i)
			poi:Hide()
		end
		return
	end

	local w, h = WorldMapDetailFrame:GetWidth(), WorldMapDetailFrame:GetHeight()

	local currentZone = GetCurrentMapZone()
	local count = 1
	for player, playerData in pairs(players) do
		if playerData.show then
			local playerX = playerData.x / 10000
			local playerY = playerData.y / 10000

			local mapx, mapy = Astrolabe:TranslateWorldMapPosition(playerData.continent, playerData.zone, playerX, playerY, currentContinent, currentZone)
			if mapx and mapy and mapx >= 0 and mapx <= 1 and mapy >= 0 and mapy <= 1 then
				local mnX = mapx * w
				local mnY = -mapy * h

				local poi = getglobal("GuildMapMain" .. count)
				poi:SetPoint("CENTER", "WorldMapDetailFrame", "TOPLEFT", mnX, mnY)
				poi:Show()

				poi.unit = player
				count = count + 1
			end
		end

		-- no more than 250 players on world map
		if count >= MAX_WORLDMAP_ICONS then break end
	end

	for i = count, MAX_WORLDMAP_ICONS do
		local poi = getglobal("GuildMapMain" .. i)
		poi:Hide()
	end
end


function GuildMap:UpdateMiniMap()
	local continent = position.continent
	if continent < 1 or continent > 3 then
		for i = 1, MAX_MINIMAP_ICONS do
			local poi = getglobal("GuildMapMini" .. i)
			poi:Hide()
		end
		return
	end

	local zone = position.zone
	local currFrame = 1
	for player, playerData in pairs(players) do
		if playerData.show and continent == playerData.continent then
			local x = position.x / 10000
			local y = position.y / 10000
			local playerX = playerData.x / 10000
			local playerY = playerData.y / 10000

			local distFromCenter, deltax, deltay = Astrolabe:ComputeMiniMapDistance(continent, zone, x, y, playerData.continent, playerData.zone, playerX, playerY)
			if distFromCenter then
				local poi = getglobal("GuildMapMini" .. currFrame)
				local poiTexture = getglobal("GuildMapMini" .. currFrame .. "Texture")

				local inSight = false
				if Squeenix or (GetMinimapShape and GetMinimapShape() == "SQUARE") then
					-- rectangular map
					inSight = abs(deltax) < 64 and abs(deltay) < 64
				else
					inSight = distFromCenter <= 56.5
				end

				if inSight then
					poiTexture:SetTexture("Interface\\AddOns\\GuildMap\\Images\\MiniMapIcon")
					poi:SetPoint("CENTER", "Minimap", "CENTER", deltax, deltay)
					poi:Show()
					poi.unit = player
					currFrame = currFrame + 1
				elseif distFromCenter < 400 then
					local roundDeltax = deltax * 56.5 / distFromCenter
					local roundDeltay = deltay * 56.5 / distFromCenter

					if Squeenix or (GetMinimapShape and GetMinimapShape() == "SQUARE") then
						if deltax > 64 then
							deltax = 64
						elseif deltax < -64 then
							deltax = -64
						end
						if deltay > 64 then
							deltay = 64
						elseif deltay < -64 then
							deltay = -64
						end
					else
						deltax = roundDeltax
						deltay = roundDeltay
					end

					poiTexture:SetTexture(self:Angle(roundDeltax, roundDeltay))

					poi:SetPoint("CENTER", "Minimap", "CENTER", deltax, deltay)
					poi:Show()
					poi.unit = player
					currFrame = currFrame + 1
				end
			end
		end

		-- no more than 75 players on mini map
		if currFrame >= MAX_MINIMAP_ICONS then break end
	end
	for i = currFrame, MAX_MINIMAP_ICONS do
		local poi = getglobal("GuildMapMini" .. i)
		poi:Hide()
	end
end

function GuildMap:Update()
	self:UpdatePartyOrRaid()

	local announceDelta = GetTime() - lastAnnounce

	local positionChanged = self:UpdatePlayerPos()
	-- an announce is scheduled if the player position has changed or 
	-- the maximum time without an announce was exceeded
	pendingAnnounce =
		pendingAnnounce or
		positionChanged or
		(announceDelta > LATEST_ANNOUNCE)

	if pendingAnnounce and announceDelta > EARLIEST_ANNOUNCE then
		self:AnnouncePosition()
	end

	-- garbage collect timed out players
	self:RemoveOldPlayers()

	if WorldMapFrame:IsVisible() then
		self:UpdateWorldMap()
	else
		self:EnsureZoneMap()
		self:UpdateMiniMap()
	end
end

function GuildMap:UpdatePlayerPos()
	local continent, zone, x, y = self:GetCurrentPlayerPosition()

	x = math.floor(x * 10000)
	y = math.floor(y * 10000)

	if position.x ~= x or position.y ~= y then
		position.continent = continent
		position.zone = zone
		position.x = x
		position.y = y
		return true
	end
end

function GuildMap:AnnouncePosition()
	if self:IsValidPosition(position.continent, position.zone, position.x, position.y) then
		pendingAnnounce = nil
		lastAnnounce = GetTime()
		local outZone = self:ConvertZoneOut(position.continent, position.zone)
		local message = "POS " .. position.continent .. " " .. outZone .. " " .. position.x .. " " .. position.y
		if announceVersion then
			message = message .. "#V " .. self.version
			announceVersion = nil
		end
		self:Broadcast(message)
	end
end

function GuildMap:Broadcast(message)
	if IsInGuild() then
		local clearAFK = GetCVar("autoClearAFK")
		SetCVar("autoClearAFK", 0)
		SendAddonMessage("GUILDMAP", message, "GUILD")
		SetCVar("autoClearAFK", clearAFK)
	end
end

function GuildMap:HandleMessage(sender, message)
	if message:sub(1, 4) == "POS " then
		self:HandlePosition(sender, message)
	elseif message:sub(1, 2) == "V " then
		self:HandleVersion(sender, message)
	end
end

function GuildMap:HandlePosition(sender, message)
	local tmpcont, tmpzone, tmpx, tmpy = select(3, message:find("POS (%d+) (%d+) (%d+) (%d+)"))
	if tmpcont and tmpzone and tmpx and tmpy then
		-- perform some validations to prevent crashes later -)
		tmpcont = tonumber(tmpcont)
		tmpzone = tonumber(tmpzone)
		tmpx = tonumber(tmpx)
		tmpy = tonumber(tmpy)

		if self:IsValidPosition(tmpcont, tmpzone, tmpx, tmpy) then
			tmpzone = self:ConvertZoneIn(tmpcont, tmpzone)
			if not players[sender] then
				-- new player: announce
				announceVersion = true
				pendingAnnounce = true
				players[sender] = { continent = tmpcont, zone = tmpzone, x = tmpx, y = tmpy, lastUpdate = GetTime(), show = true}
			else
				-- if we have the player already, do update him only to not reset the show flag
				players[sender].continent = tmpcont
				players[sender].zone = tmpzone
				players[sender].x = tmpx
				players[sender].y = tmpy
				players[sender].lastUpdate = GetTime()
			end
		end
	end
end

function GuildMap:HandleVersion(sender, message)
	if players[sender] then
		local version = message:sub(3, message:len())
		players[sender].version = version
		local revision = tonumber(version)
		if revision and versionWarning and revision > tonumber(self.version) then
			local updateMessage = string.format(L["%s is using a newer version (rev. %s) of GuildMap. Please update."], sender, revision)
			DEFAULT_CHAT_FRAME:AddMessage("|cff00ff00" .. updateMessage)
			versionWarning = nil
		end
	end
end

function GuildMap:MiniMap_OnClick(arg1)
	local x, y = GetCursorPosition()
	x = x / Minimap:GetEffectiveScale()
	y = y / Minimap:GetEffectiveScale()

	local cx, cy = Minimap:GetCenter()
	x = x + CURSOR_OFFSET_X - cx
	y = y + CURSOR_OFFSET_Y - cy
	Minimap:PingLocation(x, y)
end

function GuildMap:RemoveOldPlayers()
	for player, playerData in pairs(players) do
		if (GetTime() - playerData.lastUpdate > PLAYER_TIMEOUT) then
			players[player] = nil
		end
	end
end

function GuildMap:IsValidPosition(continent, zone, x, y)
	if continent < 1 or continent > 3 then return false end
	if continent == 1 and (zone < 1 or zone > 24) then return false end
	if continent == 2 and (zone < 1 or zone > 29) then return false end
	if continent == 3 and (zone < 1 or zone > 8) then return false end
	if x < 1 or x > 9999 then return false end
	if y < 1 or y > 9999 then return false end
	return true
end

function GuildMap:ShowMiniMapToolTip()
	GameTooltip:SetOwner(this, "ANCHOR_BOTTOMLEFT")

	local newLineString = ""
	local tooltipText = ""

	for i = 1, 25 do
		local unitButton = getglobal("GuildMapMini"..i)
		if unitButton:IsVisible() and MouseIsOver(unitButton) then
			tooltipText = tooltipText .. newLineString .. unitButton.unit
			newLineString = "\n"
		elseif not unitButton:IsVisible() then
			break
		end
	end

	GameTooltip:SetText(tooltipText, 1.0, 0.82, 0.0, 1,1)
	GameTooltip:Show()
end

function GuildMap:ShowWorldMapToolTip()
	-- determine tooltip anchor
	local x, y = this:GetCenter()
	local parentX, parentY = this:GetParent():GetCenter()
	if x > parentX then
		WorldMapTooltip:SetOwner(this, "ANCHOR_LEFT")
	else
		WorldMapTooltip:SetOwner(this, "ANCHOR_RIGHT")
	end

	local newLineString = ""
	local tooltipText = ""

	for i = 1, 250 do
		local unitButton = getglobal("GuildMapMain"..i)
		if unitButton:IsVisible() and MouseIsOver(unitButton) then
			tooltipText = tooltipText .. newLineString .. unitButton.unit
			newLineString = "\n"
		elseif not unitButton:IsVisible() then
			break
		end
	end

	WorldMapTooltip:SetText(tooltipText, 1.0, 0.82, 0.0, 1,1)
	WorldMapTooltip:Show()
end

function GuildMap:UpdatePartyOrRaid()
	for player, playerData in pairs(players) do
		playerData.show = true
	end

	local numRaidMembers = GetNumRaidMembers()
	for i = 1, numRaidMembers do
		local name = GetRaidRosterInfo(i)
		if name and players[name] then
			players[name].show = false
		end
	end

	local numPartyMembers = GetNumPartyMembers()
	for i = 1, numPartyMembers do
		local name = UnitName("party" .. i)
		if name and players[name] then
			players[name].show = false
		end
	end
end

function GuildMap:Angle(x, y)
	local angle = asin(x / 56.5)
	if x <= 0 and y <= 0 then
		angle = 180 - angle
	elseif x <= 0 and y > 0 then
		angle = 360 + angle
	elseif x > 0 and y >= 0 then
		angle = angle
	else
		angle = 180 - angle
	end
	local fileNumber = math.floor((angle / 10) + 0.5) * 10
	if fileNumber == 360 then
		fileNumber = 0
	end
	return "Interface\\Addons\\GuildMap\\images\\MiniMapArrow" .. fileNumber
end

function GuildMap:GetCurrentPlayerPosition()
	local continent = GetCurrentMapContinent()
	local zone = GetCurrentMapZone()
	local x, y = GetPlayerMapPosition("player")
	continent = continent or -1
	zone = zone or -1
	x = x or -1
	y = y or -1
	return continent, zone, x, y
end

local function CurrentZoneFix_SetupFix()
	-- CurrentZoneFix v2.0 by Legorol
	-- email: legorol@cosmosui.org
	local versionID = 20
	local id = CurrentZoneFix_FixInPlace
	if id then
		id = tonumber(id)
		if id and id >= versionID then
			return
		elseif not SetMapToCurrentZone(true) then
			DEFAULT_CHAT_FRAME:AddMessage("Warning! Obsolete version "..
				"of CurrentZoneFix function detected. The old "..
				"version is being loaded either as a standalone "..
				"AddOn, or as code embedded inside another AddOn. "..
				"You must update it to avoid bugs with the map!",1,0,0)
			return
		end
	end

	local zoneID = {}
	for continent in ipairs{GetMapContinents()} do
		for zone,name in ipairs{GetMapZones(continent)} do
			zoneID[name] = zone
		end
	end

	local orig_SetMapToCurrentZone = SetMapToCurrentZone
	SetMapToCurrentZone = function(deactivate)
		if deactivate then
			SetMapToCurrentZone = orig_SetMapToCurrentZone
			CurrentZoneFix_FixInPlace = nil
			return true
		end
		orig_SetMapToCurrentZone()
		if GetCurrentMapZone() == 0 and GetCurrentMapContinent() > 0 then
			SetMapZoom(GetCurrentMapContinent(), zoneID[GetRealZoneText()])
		else
			SetMapToCurrentZone = orig_SetMapToCurrentZone
			CurrentZoneFix_FixInPlace = "Deactivated "..versionID
		end
	end
	CurrentZoneFix_FixInPlace = versionID
end

local function iterateZones(continent, ...)
	for i = 1, select("#", ...) do
		local zone = select(i, ...)
		local translatedEN = BZ:GetReverseTranslation(zone)
		local indexEN = ZoneIndexesEN[translatedEN]
		ZoneIndexesOut[continent][i] = indexEN
		ZoneIndexesIn[continent][indexEN] = i
	end
end

local function iterateContinents(...)
	for i = 1, select("#", ...) do
		local c = select(i, ...)
		ZoneIndexesOut[i] = {}
		ZoneIndexesIn[i] = {}
		iterateZones(i, GetMapZones(i))
	end
end

function GuildMap:InitRegionData()
	iterateContinents(GetMapContinents())
	CurrentZoneFix_SetupFix()
end

function GuildMap:ConvertZoneOut(continent, zone)
	if continent >= 1 and continent <= 3 and zone ~= 0 then
		return ZoneIndexesOut[continent][zone]
	else
		return zone
	end
end

function GuildMap:ConvertZoneIn(continent, zone)
	if continent >= 1 and continent <= 3 and zone ~= 0 then
		return ZoneIndexesIn[continent][zone]
	else
		return zone
	end
end

function GuildMap:SetupDummies()
	local continents = { GetMapContinents() }
	for c in ipairs({GetMapContinents()}) do
		local zones = { GetMapZones(c) }
		for z, zone in ipairs(zones) do
			players[zone] = { continent = c, zone = z, x = 5000, y = 5000, lastUpdate = 10000000, show = true }
		end
	end
end

function GuildMap:EnsureZoneMap()
	local mapContinent = GetCurrentMapContinent()
	local mapZone = GetCurrentMapZone()

	local playerContinent, playerZone = Astrolabe:GetCurrentPlayerPosition()

	if playerContinent ~= 0 and playerZone ~= 0 and (mapContinent ~= playerContinent or mapZone ~= playerZone) then
		SetMapToCurrentZone()
	end
end

