--[[--------------------------------------------------------
-- GuildGreet, a World of Warcraft social guild assistant --
------------------------------------------------------------
CODE INDEX (search on index for fast access):
_01_ Addon Variables
_02_ Addon Startup
_03_ Event Handler
_04_ Addon Initialization
_05_ Guild Roster Import
_06_ Monitor System Messages
_07_ Display Greetlist
_08_ Display Greeting Tooltip
_09_ Greet Players
_10_ Slash Handler
_11_ Tab Changer
_12_ Settings Tab Update
_13_ Greetings Tab Update
_14_ Players Tab Update
_15_ General Helper Functions
_16_ List frame
_17_ player menu
_18_ Friends list parsing
_19_ Friends Tab Update
_20_ Channel handling
_21_ Testing
_22_ Channel Tab Update
_23_ Colour picker handling
_24_ Colour handling
_25_ Display Help Tooltip
_26_ Chat name extension
_27_ Debug dumping
_28_ Interface reloading
_29_ urbin addon listing
_30_ big brother
]]----------------------------------------------------------

-- Addon constants
GLDG_NAME 	= "GuildGreet"
GLDG_GUI	= "GuildGreetFrame"		-- Name of GUI config window
GLDG_LIST	= "GuildGreetList"		-- Name of GUI player list
GLDG_COLOUR	= "GuildGreetColourFrame"	-- Name of colour picker addition
GDLG_VNMBR	= 201011			-- Number code for this version

-- Table linking tabs to frames
GLDG_Tab2Frame = {}
GLDG_Tab2Frame.Tab1 = "Settings"
GLDG_Tab2Frame.Tab2 = "Greetings"
GLDG_Tab2Frame.Tab3 = "Players"
GLDG_Tab2Frame.Tab4 = "Friends"
GLDG_Tab2Frame.Tab5 = "Channel"
GLDG_Tab2Frame.Tab6 = "Colour"

GLDG_SubTab2Frame = {}
GLDG_SubTab2Frame.Tab1 = "General"
GLDG_SubTab2Frame.Tab2 = "Chat"
GLDG_SubTab2Frame.Tab3 = "Greeting"
GLDG_SubTab2Frame.Tab4 = "Debug"
GLDG_SubTab2Frame.Tab5 = "Other"


-- Strings we look for
GLDG_ONLINE 	= ".*%[(.+)%]%S*"..string.sub(ERR_FRIEND_ONLINE_SS, 20)
GLDG_OFFLINE	= string.format(ERR_FRIEND_OFFLINE_S, "(.+)")
GLDG_JOINED	= string.format(ERR_GUILD_JOIN_S, "(.+)")
GLDG_PROMO	= string.format(ERR_GUILD_PROMOTE_SSS, "(.+)", "(.+)", "(.+)")
GLDG_DEMOTE	= string.format(ERR_GUILD_DEMOTE_SSS, ".+", "(.+)", "(.+)")

GLDG_DEFAULT_ONLINE_COLOUR		= "|cFFA0FFA0"
GLDG_DEFAULT_IS_OFFLINE_COLOUR		= "|cFFFFFFFF"
GLDG_DEFAULT_GOES_OFFLINE_COLOUR	= "|cFF7F7F7F"
GLDG_DEFAULT_HELP_COLOUR		= "|cFFFFFF7F"
GLDG_DEFAULT_ALIAS_COLOUR		= "|cFFFFA0A0"
GLDG_DEFAULT_HEADER_COLOUR		= "|c7FFF0000"

GLDG_DEFAULT_LIST_COLOUR		= "|cFFFF7F00"
GLDG_DEFAULT_NEW_COLOUR			= "|cFFFF3F3F"
GLDG_DEFAULT_LVL_COLOUR			= "|cFF7F7F7F"
GLDG_DEFAULT_RANK_COLOUR		= "|cFFCC00CC"
GLDG_DEFAULT_RELOG_COLOUR		= "|cFF3FFF3F"

GLDG_ONLINE_COLOUR		= GLDG_DEFAULT_ONLINE_COLOUR
GLDG_IS_OFFLINE_COLOUR		= GLDG_DEFAULT_IS_OFFLINE_COLOUR
GLDG_GOES_OFFLINE_COLOUR	= GLDG_DEFAULT_GOES_OFFLINE_COLOUR
GLDG_ALIAS_COLOUR		= GLDG_DEFAULT_ALIAS_COLOUR

--------------------------
-- _01_ Addon Variables --
--------------------------

-- Stored data
GLDG_Data = {}			-- Data saved between sessions
GLDG_DataGuild = nil		-- Pointer to relevant guild section in GLDG_Data
GLDG_DataGreet = nil		-- Pointer to relevant greeting section in GLDG_Data
GLDG_DataFriends = nil		-- Pointer to relevant friends section in GLDG_Data
GLDG_DataChannel = nil		-- Pointer to relevant channel section in GLDG_Data
-- Initialization
GLDG_Main = nil			-- Main program window
GLDG_Realm = nil		-- Name of the current realm
GLDG_Player = nil		-- Name of the current player
GLDG_GuildName = nil		-- Name of your guild
GLDG_GuildAlias = nil		-- Alias of your guild
GLDG_NewGuild = nil		-- Set if initializing a new guild
GLDG_InitialGuildUpdate = nil	-- To make sure we get at least one update
GLDG_InitialFriendsUpdate = nil	-- To make sure we get at least one update
GLDG_UpdateRequest = 0		-- If set with time, update will be performed
GLDG_UpdateRequestFriends = 0	-- If set with time, update will be performed
GLDG_InitComplete = nil		-- Set in initialization is done

-- Various
GLDG_Debug = false		-- Show debugging
-- Core variables
GLDG_Online = {}		-- Time of player going online
GLDG_Offline = {}		-- Time of player going offline
GLDG_RankUpdate = {}		-- Set with time for all players getting promoted during the session
GLDG_Queue = {}			-- List of players waiting to be greeted
-- Configuration: greetings tab
GLDG_SelColName = nil		-- Name of the currently selected collection
GLDG_NumColRows = 5		-- Maximum number of collections that can be displayed
GLDG_ChangeName = nil		-- Name of the setting to be changed
GLDG_Selection = "Greet"	-- Selected greeting category
GLDG_SelMsgNum = nil		-- Number of the currently selected message
GLDG_NumSelRows = 5		-- Maximum number of greetings that can be displayed
GLDG_GreetOffset = 0		-- Offset for displaying greetings
-- Configuration: players tab
GLDG_SelPlrName = nil		-- Name of the currently selected player
GLDG_NumPlrRows = 20		-- Maximum number of players that can be displayed
GLDG_SortedList = {}		-- Sorted list of members of your guild, excluding your own characters
GLDG_PlayerOffset = 0		-- Offset for displaying players
GLDG_NumMain = 0		-- Number of players defined as main
GLDG_NumAlts = 0		-- Number of players that are alts for current selected player
GLDG_NumSubRows = 9		-- Maximum number of mains that can be displayed on subframe
-- Configuration: friends tab
GLDG_SelPlrName_f = nil		-- Name of the currently selected friend
GLDG_NumPlrRows_f = 20		-- Maximum number of friend that can be displayed
GLDG_SortedList_f = {}		-- Sorted list of friend
GLDG_PlayerOffset_f = 0		-- Offset for displaying friend
GLDG_NumMain_f = 0		-- Number of friend defined as main
GLDG_NumAlts_f = 0		-- Number of friend that are alts for current selected friend
GLDG_NumSubRows_f = 9		-- Maximum number of mains that can be displayed on subframe
-- Configuration: channel tab
GLDG_SelPlrName_c = nil		-- Name of the currently selected friend
GLDG_NumPlrRows_c = 20		-- Maximum number of friend that can be displayed
GLDG_SortedList_c = {}		-- Sorted list of friend
GLDG_PlayerOffset_c = 0		-- Offset for displaying friend
GLDG_NumMain_c = 0		-- Number of friend defined as main
GLDG_NumAlts_c = 0		-- Number of friend that are alts for current selected friend
GLDG_NumSubRows_c = 9		-- Maximum number of mains that can be displayed on subframe
-- update timer
GLDG_UPDATE_TIME = 10		-- Number of seconds to query guild and friends list (default)
-- channel parse counter
GLDG_unregister = 0		-- Number of pending requests
-- auto greet flag
GLDG_autoGreeted = 0		-- To make sure auto greet is only done once per login
-- chat event handler
GLDG_originalChatEventHandler = nil

------------------------
-- _02_ Addon Startup --
------------------------

------------------------------------------------------------
function GLDG_OnLoad()
	-- Events monitored by Event Handler
	GLDG_Main = this
	this:RegisterEvent("ADDON_LOADED")
	this:RegisterEvent("VARIABLES_LOADED")
	this:RegisterEvent("PLAYER_ENTERING_WORLD")

	-- Slash commands for CLI
	SLASH_GLDG1 = "/guildgreet"
	SLASH_GLDG2 = "/gg"
	SlashCmdList.GLDG = GLDG_SlashHandler
end

------------------------------------------------------------
function GLDG_myAddons()
	-- Register addon with myAddons
	if not (myAddOnsFrame_Register) then return end
	local version = GetAddOnMetadata("GuildGreet", "Version");
	local date = GetAddOnMetadata("GuildGreet", "X-Date");
	local author = GetAddOnMetadata("GuildGreet", "Author");
	local web = GetAddOnMetadata("GuildGreet", "X-Website");
	if (version == nil) then
		version = "unknown";
	end
	if (date == nil) then
		date = "unknown";
	end
	if (author == nil) then
		author = "unknown";
	end
	if (web == nil) then
		web = "unknown";
	end

	myAddOnsFrame_Register({
		name = GLDG_NAME,
		version = version,
		releaseDate = date,
		author = author,
		email = "none",
		website = web,
		category = MYADDONS_CATEGORY_GUILD,
		optionsframe = GLDG_GUI,
	})
end


------------------------
-- _03_ Event Handler --
------------------------

function GLDG_OnEvent()
	-- Distribute events to appropriate functions
	if (event == "ADDON_LOADED") and (arg1 == GLDG_NAME) then
		GLDG_myAddons()
		GLDG_Init()
		GLDG_RegisterUrbinAddon(GLDG_NAME, GLDG_About)
		GLDG_Main:UnregisterEvent("ADDON_LOADED")

		GLDG_originalChatEventHandler = ChatFrame_OnEvent
		ChatFrame_OnEvent = GLDG_ChatOnEvent

	elseif (event == "VARIABLES_LOADED") then
		GLDG_Main:UnregisterEvent("VARIABLES_LOADED")

		-- add menu to player frame and chat menu
		GLDG_AddPopUpButtons();

	elseif (event == "PLAYER_ENTERING_WORLD") then
		if not GLDG_InitComplete then
			GLDG_Init()
		end
		GLDG_InitRoster()
		GLDG_Main:UnregisterEvent("PLAYER_ENTERING_WORLD")
		GLDG_Main:RegisterEvent("GUILD_ROSTER_UPDATE")
		GLDG_Main:RegisterEvent("FRIENDLIST_UPDATE")
		GLDG_Main:RegisterEvent("FRIENDLIST_SHOW")
		GLDG_Main:RegisterEvent("CHAT_MSG_CHANNEL_JOIN")
		GLDG_Main:RegisterEvent("CHAT_MSG_CHANNEL_LEAVE")
		GLDG_Main:RegisterEvent("CHAT_MSG_CHANNEL_NOTICE")
		--GLDG_Main:RegisterEvent("CHAT_MSG_CHANNEL_LIST")
		GLDG_Main:RegisterEvent("CHAT_MSG_ADDON")
		GLDG_Main:RegisterEvent("PARTY_MEMBERS_CHANGED")
		GLDG_Main:RegisterEvent("RAID_ROSTER_UPDATE")

	elseif (event == "GUILD_ROSTER_UPDATE") then
		if IsInGuild() then
			if (GLDG_GuildName and GLDG_Realm and GLDG_GuildName~="") then
				-- guild name known -> treat guild info
				GLDG_RosterImport()
			else
				-- guild name not yet known -> try reinitialisation
				GLDG_InitRoster()
			end
		else
			-- nothing to do if not in guild
		end

	-- guild members and/or friends joining/leaving
	elseif (event == "CHAT_MSG_SYSTEM") then
		GLDG_SystemMsg(arg1)

	elseif ((event == "FRIENDLIST_UPDATE") or (event == "FRIENDLIST_SHOW")) then
		if (GLDG_Realm and GLDG_Player) then
			GLDG_FriendsUpdate()
		else
			GLDG_InitRoster()
		end

	-- entering special channel
	elseif (event == "CHAT_MSG_CHANNEL_NOTICE") then
		if (arg9 and arg9 ~="" and arg9 == GLDG_ChannelName) then
			if (arg1 and arg1 == "YOU_JOINED") then
				GLDG_CheckChannel()
			elseif (arg1 and arg1 == "YOU_LEFT") then
				GLDG_inChannel = false;
				GLDG_UpdateTabs()
			end
		end

	-- parsing update of special channel
	elseif (event == "CHAT_MSG_CHANNEL_LIST") then
		if (arg9 and arg9~="" and arg9==GLDG_ChannelName) then
			GLDG_InitChannel(arg1)
		end

	-- catching people entering and leaving channel
	elseif (event == "CHAT_MSG_CHANNEL_JOIN") then
		GLDG_UpdateChannel(true, arg2)
	elseif (event == "CHAT_MSG_CHANNEL_LEAVE") then
		GLDG_UpdateChannel(false, arg2)

	elseif (event == "CHAT_MSG_ADDON") then
		-- arg1 = prefix, arg2 = msg, arg3 = channel, arg4 = sender
		if (arg1 == "GLDG") then
			--GLDG_Print("--> "..event..": "..arg1.." ["..arg2.."] - "..arg3..": "..arg4)

			if (string.sub(arg2, 1, 4)=="VER:") then
				if (GLDG_Data.ShowNewerVersions) then
					local myVersion	= GetAddOnMetadata("GuildGreet", "Version")
					local hisVersion = string.sub(arg2, 5)

					if (not GLDG_HighestVersion) then
						GLDG_HighestVersion = myVersion
					end

					if (GLDG_Data.BigBrother and (arg4 ~= GLDG_Player)) then
						if (not GLDG_BigBrother) then
							GLDG_BigBrother = {}
						end
						if (not GLDG_BigBrother[arg4]) then
							GLDG_BigBrother[arg4] = hisVersion
							GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..arg4..GLDG_TXT.bigBrother..hisVersion)
						elseif (GLDG_BigBrother[arg4] ~= hisVersion) then
							GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..arg4..GLDG_TXT.bigBrother2..GLDG_BigBrother[arg4]..GLDG_TXT.bigBrother3..hisVersion..GLDG_TXT.bigBrother4)
							GLDG_BigBrother[arg4] = hisVersion
						end
					end

					--GLDG_Print("    My Version ["..myVersion.."] - Other Version ["..hisVersion.."]")
					if (hisVersion > myVersion) then
						if (hisVersion > GLDG_HighestVersion) then
							-- to make sure, we only warn once
							GLDG_HighestVersion = hisVersion

							GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_TXT.newer)
							GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_TXT.newer1..myVersion..GLDG_TXT.newer4)
							GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_TXT.newer2..arg4..GLDG_TXT.newer3..hisVersion..GLDG_TXT.newer4)
						else
							--GLDG_Print("    "..arg4.." has version "..hisVersion.." - you have "..myVersion.." - highest seen "..GLDG_HighestVersion)
						end
					end
				end
			end
		end

	-- catching people leaving or entering party/raid
	elseif (event == "PARTY_MEMBERS_CHANGED") then
		if (GetNumPartyMembers() > 0) then
			SendAddonMessage("GLDG", "VER:"..GetAddOnMetadata("GuildGreet", "Version"), "PARTY")
		end
	elseif (event == "RAID_ROSTER_UPDATE") then
		if (GetNumRaidMembers() > 0) then
			SendAddonMessage("GLDG", "VER:"..GetAddOnMetadata("GuildGreet", "Version"), "RAID")
		end

	end
end

-------------------------------
-- _04_ Addon Initialization --
-------------------------------

function GLDG_Init()
	local version = GetAddOnMetadata("GuildGreet", "Version");
	if (version == nil) then
		version = "unknown";
	end

	-- store realm and player names
	if not GLDG_Realm then GLDG_Realm = GetCVar("realmName") end
	if not GLDG_Player then GLDG_Player = UnitName("player") end

	-- Clear obsolete options
	GLDG_Data.EnableContextMenu = nil

	-- Set defaults for missing settings
	if not GLDG_Data.RelogTime then GLDG_Data.RelogTime = 0 end
	if not GLDG_Data.MinLevelUp then GLDG_Data.MinLevelUp = 0 end
	if not GLDG_Data.UpdateTime then GLDG_Data.UpdateTime = GLDG_UPDATE_TIME end
	if not GLDG_Data.ListSize then GLDG_Data.ListSize = 5 end
	if not GLDG_Data.ChatFrame then GLDG_Data.ChatFrame = 0 end

	if not GLDG_Data.Greet then GLDG_Data.Greet = GLDG_GREET end
	if not GLDG_Data.GreetBack then GLDG_Data.GreetBack = GLDG_GREETBACK end
	if not GLDG_Data.Welcome then GLDG_Data.Welcome = GLDG_WELCOME end
	if not GLDG_Data.NewRank then GLDG_Data.NewRank = GLDG_RANK end
	if not GLDG_Data.NewLevel then GLDG_Data.NewLevel = GLDG_LEVEL end
	if not GLDG_Data.Bye then GLDG_Data.Bye = GLDG_BYE end
	if not GLDG_Data.Night then GLDG_Data.Night = GLDG_NIGHT end
	if not GLDG_Data.Guild then GLDG_Data.Guild = GLDG_GUILD end
	if not GLDG_Data.Channel then GLDG_Data.Channel = GLDG_CHANNEL end
	if not GLDG_Data.ByeGuild then GLDG_Data.ByeGuild = GLDG_BYE_GUILD end
	if not GLDG_Data.NightGuild then GLDG_Data.NightGuild = GLDG_NIGHT_GUILD end
	if not GLDG_Data.ByeChannel then GLDG_Data.ByeChannel = GLDG_BYE_CHANNEL end
	if not GLDG_Data.NightChannel then GLDG_Data.NightChannel = GLDG_NIGHT_CHANNEL end
	if not GLDG_Data.LaterGuild then GLDG_Data.LaterGuild = GLDG_LATER_GUILD end
	if not GLDG_Data.LaterChannel then GLDG_Data.LaterChannel = GLDG_LATER_CHANNEL end

	if not GLDG_Data.Collections then GLDG_Data.Collections = {} end
	if not GLDG_Data.Custom then GLDG_Data.Custom = {} end
	if not GLDG_Data.Ranks then GLDG_Data.Ranks = {} end

	if not GLDG_Data.ChannelNames then GLDG_Data.ChannelNames = {} end
	if not GLDG_Data.GuildAlias then GLDG_Data.GuildAlias = {} end

	if not GLDG_Data.GreetAsMain	then GLDG_Data.GreetAsMain = nil end
	if not GLDG_Data.Randomize	then GLDG_Data.Randomize = nil end
	if not GLDG_Data.Whisper    	then GLDG_Data.Whisper = nil end
	if not GLDG_Data.IncludeOwn 	then GLDG_Data.IncludeOwn = nil end
	if not GLDG_Data.ListNames  	then GLDG_Data.ListNames = nil end
	if not GLDG_Data.ListNamesOff	then GLDG_Data.ListNamesOff = nil end
	if not GLDG_Data.ListLevelUp	then GLDG_Data.ListLevelUp = nil end
	if not GLDG_Data.ListLevelUpOff	then GLDG_Data.ListLevelUpOff = nil end
	if not GLDG_Data.ListQuit	then GLDG_Data.ListQuit = nil end
	if not GLDG_Data.ExtendChat	then GLDG_Data.ExtendChat = nil end
	if not GLDG_Data.ExtendAlias	then GLDG_Data.ExtendAlias = nil end
	if not GLDG_Data.ExtendMain	then GLDG_Data.ExtendMain = nil end
	if not GLDG_Data.AutoAssign 	then GLDG_Data.AutoAssign = nil end
	if not GLDG_Data.UseFriends 	then GLDG_Data.UseFriends = nil end
	if not GLDG_Data.ListUp     	then GLDG_Data.ListUp = nil end
	if not GLDG_Data.ListVisible	then GLDG_Data.ListVisible = nil end

	if not GLDG_Data.GreetGuild	then GLDG_Data.GreetGuild = {} end
	if not GLDG_Data.GreetChannel	then GLDG_Data.GreetChannel = {} end
	if not GLDG_Data.AutoGreet	then GLDG_Data.AutoGreet = {} end

	if not GLDG_Data.SupressGreet		then GLDG_Data.SupressGreet = nil end
	if not GLDG_Data.SupressJoin		then GLDG_Data.SupressJoin = nil end
	if not GLDG_Data.SupressLevel		then GLDG_Data.SupressLevel = nil end
	if not GLDG_Data.SupressRank		then GLDG_Data.SupressRank = nil end
	if not GLDG_Data.NoGratsOnLogin		then GLDG_Data.NoGratsOnLogin = true end

	if not GLDG_Data.UseLocalTime      then GLDG_Data.UseLocalTime = nil end
	if not GLDG_Data.ShowNewerVersions then GLDG_Data.ShowNewerVersions = nil end

	if not GLDG_Data.colours		then GLDG_Data.colours = {} end
	if not GLDG_Data.colours.guild		then GLDG_Data.colours.guild = {} end
	if not GLDG_Data.colours.friends	then GLDG_Data.colours.friends = {} end
	if not GLDG_Data.colours.channel	then GLDG_Data.colours.channel = {} end
	if not GLDG_Data.colours.help 		then GLDG_Data.colours.help = GLDG_DEFAULT_HELP_COLOUR end
	if not GLDG_Data.colours.header		then GLDG_Data.colours.header = GLDG_DEFAULT_HEADER_COLOUR end

	if not GLDG_Data.colours.guild.online	then GLDG_Data.colours.guild.online = GLDG_DEFAULT_ONLINE_COLOUR end
	if not GLDG_Data.colours.guild.isOff	then GLDG_Data.colours.guild.isOff = GLDG_DEFAULT_IS_OFFLINE_COLOUR end
	if not GLDG_Data.colours.guild.goOff	then GLDG_Data.colours.guild.goOff = GLDG_DEFAULT_GOES_OFFLINE_COLOUR end
	if not GLDG_Data.colours.guild.alias	then GLDG_Data.colours.guild.alias = GLDG_DEFAULT_ALIAS_COLOUR end
	if not GLDG_Data.colours.guild.list	then GLDG_Data.colours.guild.list = GLDG_DEFAULT_LIST_COLOUR end
	if not GLDG_Data.colours.guild.new	then GLDG_Data.colours.guild.new = GLDG_DEFAULT_NEW_COLOUR end
	if not GLDG_Data.colours.guild.lvl	then GLDG_Data.colours.guild.lvl = GLDG_DEFAULT_LVL_COLOUR end
	if not GLDG_Data.colours.guild.rank	then GLDG_Data.colours.guild.rank = GLDG_DEFAULT_RANK_COLOUR end
	if not GLDG_Data.colours.guild.relog	then GLDG_Data.colours.guild.relog = GLDG_DEFAULT_RELOG_COLOUR end

	if not GLDG_Data.colours.friends.online	then GLDG_Data.colours.friends.online = GLDG_DEFAULT_ONLINE_COLOUR end
	if not GLDG_Data.colours.friends.isOff	then GLDG_Data.colours.friends.isOff = GLDG_DEFAULT_IS_OFFLINE_COLOUR end
	if not GLDG_Data.colours.friends.goOff	then GLDG_Data.colours.friends.goOff = GLDG_DEFAULT_GOES_OFFLINE_COLOUR end
	if not GLDG_Data.colours.friends.alias	then GLDG_Data.colours.friends.alias = GLDG_DEFAULT_ALIAS_COLOUR end
	if not GLDG_Data.colours.friends.list	then GLDG_Data.colours.friends.list = GLDG_DEFAULT_LIST_COLOUR end
	if not GLDG_Data.colours.friends.new	then GLDG_Data.colours.friends.new = GLDG_DEFAULT_NEW_COLOUR end
	if not GLDG_Data.colours.friends.lvl	then GLDG_Data.colours.friends.lvl = GLDG_DEFAULT_LVL_COLOUR end
	if not GLDG_Data.colours.friends.rank	then GLDG_Data.colours.friends.rank = GLDG_DEFAULT_RANK_COLOUR end
	if not GLDG_Data.colours.friends.relog	then GLDG_Data.colours.friends.relog = GLDG_DEFAULT_RELOG_COLOUR end

	if not GLDG_Data.colours.channel.online	then GLDG_Data.colours.channel.online = GLDG_DEFAULT_ONLINE_COLOUR end
	if not GLDG_Data.colours.channel.isOff	then GLDG_Data.colours.channel.isOff = GLDG_DEFAULT_IS_OFFLINE_COLOUR end
	if not GLDG_Data.colours.channel.goOff	then GLDG_Data.colours.channel.goOff = GLDG_DEFAULT_GOES_OFFLINE_COLOUR end
	if not GLDG_Data.colours.channel.alias	then GLDG_Data.colours.channel.alias = GLDG_DEFAULT_ALIAS_COLOUR end
	if not GLDG_Data.colours.channel.list	then GLDG_Data.colours.channel.list = GLDG_DEFAULT_LIST_COLOUR end
	if not GLDG_Data.colours.channel.new	then GLDG_Data.colours.channel.new = GLDG_DEFAULT_NEW_COLOUR end
	if not GLDG_Data.colours.channel.lvl	then GLDG_Data.colours.channel.lvl = GLDG_DEFAULT_LVL_COLOUR end
	if not GLDG_Data.colours.channel.rank	then GLDG_Data.colours.channel.rank = GLDG_DEFAULT_RANK_COLOUR end
	if not GLDG_Data.colours.channel.relog	then GLDG_Data.colours.channel.relog = GLDG_DEFAULT_RELOG_COLOUR end


	-- Set initial pointers to avoid errors (hack!)
	GLDG_DataGuild = {}
	GLDG_DataGreet = {}
	GLDG_DataFriends = {}
	GLDG_DataChannel = {}

	GLDG_ChannelName = ""

	-- Keep version in configuration file
	GLDG_Data.Version = GDLG_VNMBR

	-- Prepare popup dialogs
	GLDG_PrepareReloadQuestion()

	-- Initialize the list GUI
	getglobal(GLDG_LIST.."TitleText"):SetText(GLDG_NAME.." "..version)

	-- Initialize the config GUI
	getglobal(GLDG_GUI.."Title"):SetText(GLDG_NAME.." "..version)

	-- Initialize the colour picker frame
	getglobal(GLDG_COLOUR.."RedText"):SetText(GLDG_TXT.red)
	getglobal(GLDG_COLOUR.."GreenText"):SetText(GLDG_TXT.green)
	getglobal(GLDG_COLOUR.."BlueText"):SetText(GLDG_TXT.blue)
	getglobal(GLDG_COLOUR.."OpacityText"):SetText(GLDG_TXT.opactiy)

	-- Make GUI close on escape
	tinsert(UISpecialFrames, GLDG_GUI)

	-- Initialize tabs and set the first one active
	local frame = getglobal(GLDG_GUI)
	PanelTemplates_SetNumTabs(frame, GLDG_TableSize(GLDG_Tab2Frame))
	PanelTemplates_SetTab(frame, 1)

	-- Set tab names and initialize tabframes
	for tabNum = 1, GLDG_TableSize(GLDG_Tab2Frame) do
		local tab = getglobal(GLDG_GUI.."Tab"..tabNum)
		local frameName = GLDG_Tab2Frame["Tab"..tabNum]
		if frameName then
			local label = GLDG_TXT["Tab"..frameName]
			if label then
				-- tab has label: initialize frame
				tab:SetText(label)
				tab:Show()
				GLDG_InitFrame(frameName)
			end
		end
	end

	-- Initialize subtabs and set the first one active
	local frame = getglobal(GLDG_GUI.."Settings")
	PanelTemplates_SetNumTabs(frame, GLDG_TableSize(GLDG_SubTab2Frame))
	PanelTemplates_SetTab(frame, 1)

	-- Set subtab names and initialize tabframes
	for tabNum = 1, GLDG_TableSize(GLDG_SubTab2Frame) do
		local tab = getglobal(GLDG_GUI.."SettingsTab"..tabNum)
		local subFrame = GLDG_SubTab2Frame["Tab"..tabNum]
		local frameName = "Settings"..subFrame
		if frameName then
			local label = GLDG_TXT["SubTab"..subFrame]
			if label then
				-- tab has label: initialize frame
				tab:SetText(label)
				tab:Show()
				GLDG_InitFrame(frameName)
			end
		end
	end

	GLDG_UpdateTabs()
	GLDG_InitComplete = true
end

------------------------------------------------------------
function GLDG_InitFrame(frameName)
	-- Set full name for frame
	local name = GLDG_GUI..frameName

	-- Configure the frames
	if (frameName == "Settings") then
		-- nothing to set
	elseif (frameName == "Greetings") then
		getglobal(name.."Header"):SetText(GLDG_TXT.greetheader)
		getglobal(name.."CollectHeader"):SetText(GLDG_TXT.collectheader)
		getglobal(name.."ColRealm"):SetText(GLDG_TXT.cbrealm)
		getglobal(name.."ColGuild"):SetText(GLDG_TXT.cbguild)
		getglobal(name.."ColPlayer"):SetText(GLDG_TXT.cbplayer)
		getglobal(name.."SubCustomHeader"):SetText(GLDG_TXT.custheader)
		getglobal(name.."SubNewHeader"):SetText(GLDG_TXT.cbnew)
		getglobal(name.."SubNewAdd"):SetText(GLDG_TXT.add)
		getglobal(name.."SubNewCancel"):SetText(GLDG_TXT.cancel)
		getglobal(name.."SubChangeSelection"):SetText(GLDG_TXT.selection)
		getglobal(name.."SubChangeGlobal"):SetText(GLDG_TXT.colglobal)
		getglobal(name.."SubChangeClear"):SetText(GLDG_TXT.nodef)
		getglobal(name.."SubChangeCancel"):SetText(GLDG_TXT.cancel)
		getglobal(name.."SelDefault"):SetText(GLDG_TXT.default)
		getglobal(name.."SelRelog"):SetText(GLDG_TXT.relog)
		getglobal(name.."SelJoin"):SetText(GLDG_TXT.joining)
		getglobal(name.."SelRank"):SetText(GLDG_TXT.newrank)
		getglobal(name.."SelLevel"):SetText(GLDG_TXT.newlevel)
		getglobal(name.."SelBye"):SetText(GLDG_TXT.bye)
		getglobal(name.."SelNight"):SetText(GLDG_TXT.night)
		getglobal(name.."SelGuild"):SetText(GLDG_TXT.guild)
		getglobal(name.."SelChannel"):SetText(GLDG_TXT.channel)
		getglobal(name.."SelByeGuild"):SetText(GLDG_TXT.byeGuild)
		getglobal(name.."SelNightGuild"):SetText(GLDG_TXT.nightGuild)
		getglobal(name.."SelByeChannel"):SetText(GLDG_TXT.byeChannel)
		getglobal(name.."SelNightChannel"):SetText(GLDG_TXT.nightChannel)
		getglobal(name.."SelLaterGuild"):SetText(GLDG_TXT.laterGuild)
		getglobal(name.."SelLaterChannel"):SetText(GLDG_TXT.laterChannel)
	elseif (frameName == "Players") then
		-- Header and option texts
		getglobal(name.."Header"):SetText(GLDG_TXT.playersheader)
		getglobal(name.."IgnoreText"):SetText(GLDG_TXT.ignores)
		getglobal(name.."AltText"):SetText(GLDG_TXT.showalt)
		getglobal(name.."Alt2Text"):SetText(GLDG_TXT.groupalt)
		getglobal(name.."UnassignedText"):SetText(GLDG_TXT.filterUnassigned)
		-- Button text
		getglobal(name.."Alias"):SetText(GLDG_TXT.pbalias)
		-- Set value for option checkboxes
		GLDG_UpdatePlayerCheckboxes()
	elseif (frameName == "Friends") then
		-- Header and option texts
		getglobal(name.."Header"):SetText(GLDG_TXT.friendsheader)
		getglobal(name.."IgnoreText"):SetText(GLDG_TXT.ignores)
		getglobal(name.."AltText"):SetText(GLDG_TXT.showalt)
		getglobal(name.."Alt2Text"):SetText(GLDG_TXT.groupalt)
		getglobal(name.."UnassignedText"):SetText(GLDG_TXT.filterUnassigned)
		-- Button text
		getglobal(name.."Alias"):SetText(GLDG_TXT.pbalias)
		-- Set value for option checkboxes
		GLDG_UpdatePlayerCheckboxes()
	elseif (frameName == "Channel") then
		-- Header and option texts
		getglobal(name.."Header"):SetText(GLDG_TXT.channelheader)
		getglobal(name.."IgnoreText"):SetText(GLDG_TXT.ignores)
		getglobal(name.."AltText"):SetText(GLDG_TXT.showalt)
		getglobal(name.."Alt2Text"):SetText(GLDG_TXT.groupalt)
		getglobal(name.."UnassignedText"):SetText(GLDG_TXT.filterUnassigned)
		-- Button text
		getglobal(name.."Alias"):SetText(GLDG_TXT.pbalias)
		-- Set value for option checkboxes
		GLDG_UpdatePlayerCheckboxes()
	elseif (frameName == "Colour") then
		-- Column header
		getglobal(name.."ColHeaderGuild"):SetText(GLDG_TXT.colGuild)
		getglobal(name.."ColHeaderFriends"):SetText(GLDG_TXT.colFriends)
		getglobal(name.."ColHeaderChannel"):SetText(GLDG_TXT.colChannel)
		-- Chat header and options
		getglobal(name.."HeaderChat"):SetText(GLDG_TXT.colChatHeader)
		getglobal(name.."ComingOnline"):SetText(GLDG_TXT.colOn)
		getglobal(name.."GoingOffline"):SetText(GLDG_TXT.colGoOff)
		getglobal(name.."IsOffline"):SetText(GLDG_TXT.colIsOff)
		getglobal(name.."Alias"):SetText(GLDG_TXT.colAlias)
		-- List header and options
		getglobal(name.."HeaderList"):SetText(GLDG_TXT.colListHeader)
		getglobal(name.."List"):SetText(GLDG_TXT.colList)
		getglobal(name.."Relog"):SetText(GLDG_TXT.colRelog)
		getglobal(name.."New"):SetText(GLDG_TXT.colNew)
		getglobal(name.."Level"):SetText(GLDG_TXT.colLevel)
		getglobal(name.."Rank"):SetText(GLDG_TXT.colRank)
		-- Common header and options
		getglobal(name.."HeaderCommon"):SetText(GLDG_TXT.colCommonHeader)
		getglobal(name.."Help"):SetText(GLDG_TXT.colHelp)
		getglobal(name.."Header"):SetText(GLDG_TXT.colHeader)
		-- Default button
		getglobal(name.."DefaultButton"):SetText(GLDG_TXT.colDefault)
		-- Hide unused colours
		getglobal(name.."FriendsNewButton"):Hide()
		getglobal(name.."FriendsNewColour"):Hide()
		getglobal(name.."FriendsRankButton"):Hide()
		getglobal(name.."FriendsRankColour"):Hide()
		getglobal(name.."FriendsLevelButton"):Hide()
		getglobal(name.."FriendsLevelColour"):Hide()
		getglobal(name.."ChannelNewButton"):Hide()
		getglobal(name.."ChannelNewColour"):Hide()
		getglobal(name.."ChannelRankButton"):Hide()
		getglobal(name.."ChannelRankColour"):Hide()
		getglobal(name.."ChannelLevelButton"):Hide()
		getglobal(name.."ChannelLevelColour"):Hide()
	elseif (frameName == "SettingsGeneral") then
		-- Greeting options texts
		getglobal(name.."Header"):SetText(GLDG_TXT.optheader)
		getglobal(name.."GreetAsMainText"):SetText(GLDG_TXT.greetasmain)
		getglobal(name.."RandomizeText"):SetText(GLDG_TXT.randomize)
		getglobal(name.."WhisperText"):SetText(GLDG_TXT.whisper)
		getglobal(name.."IncludeOwnText"):SetText(GLDG_TXT.includeOwn)
		getglobal(name.."AutoAssignText"):SetText(GLDG_TXT.autoAssign)
		getglobal(name.."UseFriendsText"):SetText(GLDG_TXT.useFriends)
		getglobal(name.."ChannelNameText"):SetText(GLDG_TXT.channelName)
		-- Queued greetings list texts
		getglobal(name.."ListHeader"):SetText(GLDG_TXT.listheader)
		getglobal(name.."ListdirectText"):SetText(GLDG_TXT.listdirect)
		getglobal(name.."ListheaderText"):SetText(GLDG_TXT.listvisible)
		-- Set value for checkboxes
		getglobal(name.."GreetAsMainBox"):SetChecked(GLDG_Data.GreetAsMain)
		getglobal(name.."RandomizeBox"):SetChecked(GLDG_Data.Randomize)
		getglobal(name.."WhisperBox"):SetChecked(GLDG_Data.Whisper)
		getglobal(name.."IncludeOwnBox"):SetChecked(GLDG_Data.IncludeOwn)
		getglobal(name.."AutoAssignBox"):SetChecked(GLDG_Data.AutoAssign)
		getglobal(name.."UseFriendsBox"):SetChecked(GLDG_Data.UseFriends)
		getglobal(name.."ListdirectBox"):SetChecked(GLDG_Data.ListUp)
		getglobal(name.."ListheaderBox"):SetChecked(GLDG_Data.ListVisible)
		-- Set values for Relog and Listsize sliders
		getglobal(name.."RelogSlider"):SetValue(GLDG_Data.RelogTime)
		getglobal(name.."MinLevelUpSlider"):SetValue(GLDG_Data.MinLevelUp)
		getglobal(name.."UpdateTimeSlider"):SetValue(GLDG_Data.UpdateTime/10)
		getglobal(name.."ListsizeSlider"):SetValue(GLDG_Data.ListSize)
	elseif (frameName == "SettingsChat") then
		-- List settings
		getglobal(name.."Header"):SetText(GLDG_TXT.chatheader)
		getglobal(name.."ChatFrameSlider"):SetValue(GLDG_Data.ChatFrame)
		getglobal(name.."ListNamesBox"):SetChecked(GLDG_Data.ListNames)
		getglobal(name.."ListNamesOffBox"):SetChecked(GLDG_Data.ListNamesOff)
		getglobal(name.."ListLevelUpBox"):SetChecked(GLDG_Data.ListLevelUp)
		getglobal(name.."ListLevelUpOffBox"):SetChecked(GLDG_Data.ListLevelUpOff)
		getglobal(name.."ListQuitBox"):SetChecked(GLDG_Data.ListQuit)
		getglobal(name.."ExtendChatBox"):SetChecked(GLDG_Data.ExtendChat)
		getglobal(name.."ExtendAliasBox"):SetChecked(GLDG_Data.ExtendAlias)
		getglobal(name.."ExtendMainBox"):SetChecked(GLDG_Data.ExtendMain)
		getglobal(name.."ListNamesText"):SetText(GLDG_TXT.listNames)
		getglobal(name.."ListNamesOffText"):SetText(GLDG_TXT.listNamesOff)
		getglobal(name.."ListLevelUpText"):SetText(GLDG_TXT.listLevelUp)
		getglobal(name.."ListLevelUpOffText"):SetText(GLDG_TXT.listLevelUpOff)
		getglobal(name.."ListQuitText"):SetText(GLDG_TXT.listQuit)
		getglobal(name.."ExtendChatText"):SetText(GLDG_TXT.extendChat)
		getglobal(name.."ExtendAliasText"):SetText(GLDG_TXT.extendAlias)
		getglobal(name.."ExtendMainText"):SetText(GLDG_TXT.extendMain)
	elseif (frameName == "SettingsGreeting") then
		-- header
		getglobal(name.."Header"):SetText(GLDG_TXT.greetingheader)
		getglobal(name.."SubHeader"):SetText(GLDG_TXT.greetingsubheader)
		-- greet options
		getglobal(name.."GreetGuildBox"):SetChecked(GLDG_Data.GreetGuild[GLDG_Realm.." - "..GLDG_Player])
		getglobal(name.."GreetChannelBox"):SetChecked(GLDG_Data.GreetChannel[GLDG_Realm.." - "..GLDG_Player])
		getglobal(name.."AutoGreetBox"):SetChecked(GLDG_Data.AutoGreet[GLDG_Realm.." - "..GLDG_Player])
		getglobal(name.."GreetGuildText"):SetText(GLDG_TXT.greetGuild)
		getglobal(name.."GreetChannelText"):SetText(GLDG_TXT.greetChannel)
		getglobal(name.."AutoGreetText"):SetText(GLDG_TXT.autoGreet)
		-- suppress options
		GLDG_UpdateSupressed()
		getglobal(name.."SupressAll"):SetText(GLDG_TXT.supressAll)
		getglobal(name.."SupressNone"):SetText(GLDG_TXT.supressNone)

		getglobal(name.."SupressGreetText"):SetText(GLDG_TXT.supressGreet)
		getglobal(name.."SupressJoinText"):SetText(GLDG_TXT.supressJoin)
		getglobal(name.."SupressLevelText"):SetText(GLDG_TXT.supressLevel)
		getglobal(name.."SupressRankText"):SetText(GLDG_TXT.supressRank)

		getglobal(name.."NoGratsOnLoginBox"):SetChecked(GLDG_Data.NoGratsOnLogin)
		getglobal(name.."NoGratsOnLoginText"):SetText(GLDG_TXT.noGratsOnLogin)
		-- guild alias
		getglobal(name.."GuildAliasSet"):SetText(GLDG_TXT.set)
		getglobal(name.."GuildAliasClear"):SetText(GLDG_TXT.clear)
	elseif (frameName == "SettingsDebug") then
		getglobal(name.."Header"):SetText(GLDG_TXT.dumpHeader)
		GLDG_ShowDump()
	elseif (frameName == "SettingsOther") then
		getglobal(name.."Header"):SetText(GLDG_TXT.otherheader)
		getglobal(name.."UseLocalTimeBox"):SetChecked(GLDG_Data.UseLocalTime)
		getglobal(name.."UseLocalTimeText"):SetText(GLDG_TXT.useLocalTime)
		getglobal(name.."ShowNewerVersionsBox"):SetChecked(GLDG_Data.ShowNewerVersions)
		getglobal(name.."ShowNewerVersionsText"):SetText(GLDG_TXT.showNewerVersions)
	end
end

------------------------------------------------------------
function GLDG_UpdateTabs()
	local friends = getglobal(GLDG_GUI.."Tab4")
	if GLDG_Data.UseFriends then
		friends:Enable()
		friends:Show()
	else
		friends:Disable()
		friends:Hide()
	end

	local channel = getglobal(GLDG_GUI.."Tab5")
	if GLDG_inChannel then
		channel:Enable()
		channel:Show()
	else
		channel:Disable()
		channel:Hide()
	end
end

------------------------------------------------------------
function GLDG_UpdatePlayerCheckboxes()
	local name = GLDG_GUI.."Players"
	getglobal(name.."IgnoreBox"):SetChecked(GLDG_Data.ShowIgnore)
	getglobal(name.."AltBox"):SetChecked(GLDG_Data.ShowAlt)
	getglobal(name.."Alt2Box"):SetChecked(GLDG_Data.GroupAlt)
	getglobal(name.."UnassignedBox"):SetChecked(GLDG_Data.FilterUnassigned)

	name = GLDG_GUI.."Friends"
	getglobal(name.."IgnoreBox"):SetChecked(GLDG_Data.ShowIgnore)
	getglobal(name.."AltBox"):SetChecked(GLDG_Data.ShowAlt)
	getglobal(name.."Alt2Box"):SetChecked(GLDG_Data.GroupAlt)
	getglobal(name.."UnassignedBox"):SetChecked(GLDG_Data.FilterUnassigned)

	name = GLDG_GUI.."Channel"
	getglobal(name.."IgnoreBox"):SetChecked(GLDG_Data.ShowIgnore)
	getglobal(name.."AltBox"):SetChecked(GLDG_Data.ShowAlt)
	getglobal(name.."Alt2Box"):SetChecked(GLDG_Data.GroupAlt)
	getglobal(name.."UnassignedBox"):SetChecked(GLDG_Data.FilterUnassigned)
end

------------------------------------------------------------
function GLDG_InitGreet(section)
	-- Set greetings section pointer if section exists
	local t = GLDG_Data.Custom
	if not t[section] then return true
	elseif (t[section] == "") then GLGG_DataGreet = GLDG_Data
	else	GLDG_SelColName = t[section]
		GLDG_DataGreet = GLDG_Data.Collections[t[section]] end
end

------------------------------------------------------------
function GLDG_InitRoster()
	-- Retreive realm, player and guild name if needed
	if not GLDG_Realm then GLDG_Realm = GetCVar("realmName") end
	if not GLDG_Player then GLDG_Player = UnitName("player") end
	if not GLDG_GuildName or GLDG_GuildName == "" then GLDG_GuildName = GetGuildInfo("player") end
	if not GLDG_GuildName then GLDG_GuildName = "" end
	if (GLDG_GuildName == "") then
		GLDG_GuildAlias = ""
	else
		if not GLDG_Data.GuildAlias[GLDG_Realm.." - "..GLDG_GuildName] then
			GLDG_Data.GuildAlias[GLDG_Realm.." - "..GLDG_GuildName] = GLDG_GuildName
		end
		GLDG_GuildAlias = GLDG_Data.GuildAlias[GLDG_Realm.." - "..GLDG_GuildName]
	end

	if not (GLDG_Realm and GLDG_Player) then return end

	if (GLDG_Realm and GLDG_Player) then
		-- Create the friends store if needed
		if not GLDG_Data[GLDG_Realm.." -- friends"] then
			GLDG_Data[GLDG_Realm.." -- friends"] = {}
		end
		if not GLDG_Data[GLDG_Realm.." -- friends"].chars then
			GLDG_Data[GLDG_Realm.." -- friends"].chars = {}
		end
		if not GLDG_Data[GLDG_Realm.." -- friends"].chars[GLDG_Player] then
			GLDG_Data[GLDG_Realm.." -- friends"].chars[GLDG_Player] = {}
		end
		if not GLDG_Data[GLDG_Realm.." -- friends"].friends then
			GLDG_Data[GLDG_Realm.." -- friends"].friends = {}
		end
		-- Set friends section pointer
		GLDG_DataFriends = GLDG_Data[GLDG_Realm.." -- friends"]
	else
		GLDG_DataFriends = {}
	end

	if (GLDG_Realm and GLDG_Player) then
		-- create channel name store if needed
		if not GLDG_Data.ChannelNames then
			GLDG_Data.ChannelNames = {}
		end
		if not GLDG_Data.ChannelNames[GLDG_Realm.." - "..GLDG_Player] then
			GLDG_Data.ChannelNames[GLDG_Realm.." - "..GLDG_Player] = ""
		end
		-- set channel name pointer
		GLDG_ChannelName = GLDG_Data.ChannelNames[GLDG_Realm.." - "..GLDG_Player]

		-- create channel store if needed
		if (GLDG_ChannelName ~= "") then
			if not GLDG_Data[GLDG_Realm.." - channel - "..GLDG_ChannelName] then
				GLDG_Data[GLDG_Realm.." - channel - "..GLDG_ChannelName] = {}
			end
			-- Set channel section pointer
			GLDG_DataChannel = GLDG_Data[GLDG_Realm.." - channel - "..GLDG_ChannelName]
		else
			GLDG_DataChannel = {}
		end
	else
		GLDG_ChannelName = ""
		GLDG_DataChannel = {}
	end
	--getglobal(GLDG_GUI.."Settings".."ChannelNameEditbox"):SetText(GLDG_ChannelName)

	if (GLDG_Realm and GLDG_GuildName and GLDG_GuildName~="") then
		-- Create the guild store if needed
		if not GLDG_Data[GLDG_Realm.." - "..GLDG_GuildName] then
			GLDG_NewGuild = true
			GLDG_Data[GLDG_Realm.." - "..GLDG_GuildName] = {}
		end

		-- Set guild section pointer
		GLDG_DataGuild = GLDG_Data[GLDG_Realm.." - "..GLDG_GuildName]
	else
		GLDG_DataGuild = {}

	end

	if (GLDG_Realm and GLDG_GuildName and GLDG_GuildName~="" and GLDG_Player) then
		-- Set greetings section pointer
		if GLDG_InitGreet(GLDG_Realm.."-"..GLDG_Player) and
		   GLDG_InitGreet(GLDG_Realm.."-"..GLDG_GuildName) and
		   GLDG_InitGreet(GLDG_Realm) then
			-- No custom collections are used
			GLDG_DataGreet = GLDG_Data
		end

		-- Set initial custom collection settings
		GLDG_ShowCustom(GLDG_GUI.."Greetings")
	else
		GLDG_DataGreet = GLDG_Data
	end

	-- Update config dialog (can't be done in InitFrame())
	local name = GLDG_GUI.."SettingsGreeting"
	if (GLDG_GuildName~="") then

		getglobal(name.."GuildAliasHeader"):SetText(GLDG_TXT.guildAlias.."|cFFFFFF7F"..GLDG_GuildName.."|r")
		if (GLDG_GuildAlias ~= GLDG_GuildName) then
			getglobal(name.."GuildAliasEditbox"):SetText(GLDG_GuildAlias)
		else
			getglobal(name.."GuildAliasEditbox"):SetText("")
		end

		getglobal(name.."GuildAliasEditbox"):Show()
		getglobal(name.."GuildAliasWarning"):Hide()
		getglobal(name.."GuildAliasSet"):Enable("")
		getglobal(name.."GuildAliasClear"):Enable("")
	else
		getglobal(name.."GuildAliasHeader"):SetText(GLDG_TXT.guildAlias.."|cFFFFFF7F"..GLDG_GuildName.."|r")
		getglobal(name.."GuildAliasWarning"):SetText(GLDG_TXT.guildNoAlias)
		getglobal(name.."GuildAliasEditbox"):SetText("")

		getglobal(name.."GuildAliasEditbox"):Hide()
		getglobal(name.."GuildAliasWarning"):Show()
		getglobal(name.."GuildAliasSet"):Disable("")
		getglobal(name.."GuildAliasClear"):Disable("")
	end

	-- Launch request for full guild roster
	if (IsInGuild()) then
		local offline = GetGuildRosterShowOffline()
		SetGuildRosterShowOffline(true)
		if not offline then SetGuildRosterShowOffline(false) end

		-- GLDG_RosterImport will call GLDG_CheckChannel
	else
		-- GLDG_RosterImport won't be called
		GLDG_CheckChannel()
	end

	-- Launch request for friends info
	ShowFriends()
end

------------------------------------------------------------
function GLDG_ClickGuildAliasSet()
	local name = GLDG_GUI.."SettingsGreeting"
	GLDG_GuildAlias = getglobal(name.."GuildAliasEditbox"):GetText()
	GLDG_Data.GuildAlias[GLDG_Realm.." - "..GLDG_GuildName] = GLDG_GuildAlias

	getglobal(name.."GuildAliasEditbox"):ClearFocus()
end

------------------------------------------------------------
function GLDG_ClickGuildAliasClear()
	GLDG_GuildAlias = GLDG_GuildName
	GLDG_Data.GuildAlias[GLDG_Realm.." - "..GLDG_GuildName] = GLDG_GuildAlias

	local name = GLDG_GUI.."SettingsGreeting"
	getglobal(name.."GuildAliasEditbox"):SetText("")
	getglobal(name.."GuildAliasEditbox"):ClearFocus()
end

------------------------------------------------------------
function GLDG_OnUpdate()
	-- if we're not interested in updates, return
	if ((not GLDG_UpdateRequest) and (not GLDG_UpdateRequestFriends)) then return end

	-- are we interested in guild updates?
	if (GLDG_UpdateRequest) then
		-- is it time, yet?
		if (GetTime() >= GLDG_UpdateRequest) then

			-- yes: treat guild information
			if not (GLDG_Realm and GLDG_GuildName and GLDG_GuildName~="") then
				-- some information is still missing - try to get it
				GLDG_InitRoster()

				-- trigger next update
				if not GLDG_updateCount then GLDG_updateCount = 0 end
				GLDG_updateCount = GLDG_updateCount + 1
				if (GLDG_updateCount < 10) then
					-- first 10 updates in quick succession (no matter if we're in a guild or not)
					GLDG_UpdateRequest = GetTime() + 1
				elseif (IsInGuild()) then
					-- we're in a guild and the information is still missing, keep polling
					GLDG_UpdateRequest = GetTime() + GLDG_UPDATE_TIME
				else
					-- we're not in a guild, stop polling
					GLDG_UpdateRequest = nil
				end
			else
				-- we've got all the base information -> renew full roster
				local setting = GetGuildRosterShowOffline()
				SetGuildRosterShowOffline(true)
				GuildRoster()
				if not setting then SetGuildRosterShowOffline(false) end

				-- trigger next update
				if (GLDG_Data.UpdateTime > 0) then
					GLDG_UpdateRequest = GetTime() + GLDG_Data.UpdateTime
				else
					GLDG_UpdateRequest = nil
				end
			end
		end
	end

	-- are we interested in friend updates (i.e. are we in phase 1)?
	if (GLDG_UpdateRequestFriends) then
		if (GLDG_Data.UseFriends) then
			-- is it time, yet?
			if (GetTime() >= GLDG_UpdateRequestFriends) then

				-- yes: treat friend information
				if not GLDG_Realm then
					-- some information is still missing - try to get it
					GLDG_Print("--- OnUpdate: friends update -> reinit roster ---")
					GLDG_InitRoster()

					-- trigger next update
					GLDG_UpdateRequestFriends = GetTime() + 1
				else
					-- we've got all the base information -> check friends
					ShowFriends()

					-- this will cause GLDG_FriendsUpdate() to be called
					-- which will retrigger the update or not depending on
					-- the configuration, until then, turn it off
					GLDG_UpdateRequestFriends = nil
				end
			end
		else
			GLDG_UpdateRequestFriends = nil
		end
	end
end


------------------------------
-- _05_ Guild Roster Import --
------------------------------

function GLDG_RosterImport()
	if (not GLDG_GuildName or GLDG_GuildName == "") then
		return
	end

	GLDG_SetActiveColourSet("guild")

	-- Update guildrank names
	GLDG_Data.Ranks[GLDG_Realm.."-"..GLDG_GuildName] = {}
	for i = 1, GuildControlGetNumRanks() do GLDG_Data.Ranks[GLDG_Realm.."-"..GLDG_GuildName][GuildControlGetRankName(i)] = i end
	-- Add info about all players of your guild
	local mains = {}
	local alts = {}
	local cnt = 0
	local update = false
	local complete = false
	local maxMembers = GetNumGuildMembers()
	if maxMembers == nil then maxMembers = nil end
	for i = 1, maxMembers do
		local pl, rn, ri, lv, cl, zn, pn, on, ol, st = GetGuildRosterInfo(i)
		if pl then
			cnt = cnt +1
			if not GLDG_DataGuild[pl] then
				-- New player
				GLDG_DataGuild[pl] = {}
				if not GLDG_NewGuild and not GLDG_Data.SupressJoin and (GLDG_TableSize(GLDG_DataGreet.Welcome) > 0) then
					GLDG_DataGuild[pl].new = true
				end
			end
			if (pl == UnitName("player")) then
				-- This player is our own: ignore completely
				GLDG_DataGuild[pl].own = true
				--GLDG_DataGuild[pl].ignore = true
			end

			if (GLDG_Data.AutoAssign) then
				-- detect if note contains "Main[ <discardable text>]"
				if (pn == "Main" or string.sub(pn, 1, 5)=="Main ") then
					mains[pl] = "true"
				elseif (on and (on == "Main" or string.sub(on, 1, 5)=="Main ")) then
					mains[pl] = "true"
				end

				-- detect if note contains "alt-<main name>[ <discardable text]"
				if (string.sub(pn, 1, 4)=="alt-") then
					local main = string.sub(pn, 5)

					local a,b,c=strfind(main, "(%S+)"); --contiguous string of non-space characters
					if a then
						main = c
					end

					alts[pl] = main
				elseif (on and string.sub(on, 1, 4)=="alt-") then
					local main = string.sub(on, 5)

					local a,b,c=strfind(main, "(%S+)"); --contiguous string of non-space characters
					if a then
						main = c
					end

					alts[pl] = main
				end
			end

			if not GLDG_DataGuild[pl].lvl then
				GLDG_DataGuild[pl].lvl = lv
			end

			if (type(GLDG_DataGuild[pl].newrank) ~= "number") then
				GLDG_DataGuild[pl].newrank = nil
			end
			if not GLDG_RankUpdate[pl] then
				if GLDG_DataGuild[pl].newrank and (ri > GLDG_DataGuild[pl].newrank) then
					-- Player got demoted again
					GLDG_DataGuild[pl].promotor = nil
					GLDG_DataGuild[pl].newrank = nil
				elseif GLDG_DataGuild[pl].newrank and (ri < GLDG_DataGuild[pl].newrank) then
					-- Player got promoted again
					GLDG_DataGuild[pl].promotor = nil
					GLDG_DataGuild[pl].newrank = ri
				elseif GLDG_DataGuild[pl].rank and (ri < GLDG_DataGuild[pl].rank) then
					-- Player got promoted
					GLDG_DataGuild[pl].newrank = ri
				end
				GLDG_DataGuild[pl].rank = ri
				GLDG_DataGuild[pl].rankname = rn
			end

			if ol then
				-- Remove rank info if we're not interested in it
				if GLDG_DataGuild[pl].newrank and (GLDG_Data.SupressRank or (GLDG_TableSize(GLDG_DataGreet.NewRank) == 0) or GLDG_DataGuild[pl].ignore) then
					GLDG_DataGuild[pl].promotor = nil
					GLDG_DataGuild[pl].newrank = nil
				end
				-- Add promoted player to queue if hour is known
				if GLDG_DataGuild[pl].newrank and GLDG_RankUpdate[pl] and (not GLDG_Queue[pl]) then
					GLDG_Queue[pl] = GLDG_RankUpdate[pl]
					update = true
				end

				-- Update player level
				if (lv > GLDG_DataGuild[pl].lvl) then
					if (not GLDG_DataGuild[pl].own) then
						if GLDG_Data.ListLevelUp then
							GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r ["..pl..": "..cl.."] "..GLDG_TXT.lvlIncrease1.." "..GLDG_DataGuild[pl].lvl.." "..GLDG_TXT.lvlIncrease2.." "..lv.." "..GLDG_TXT.lvlIncrease3);
						end
						if ( (GLDG_TableSize(GLDG_DataGreet.NewLevel) > 0) and not GLDG_Data.SupressLevel and not GLDG_DataGuild[pl].ignore and (lv > GLDG_Data.MinLevelUp)) then
							if (GLDG_Online[pl] or not GLDG_Data.NoGratsOnLogin) then
								GLDG_DataGuild[pl].newlvl = true
							end
						end
					end
				end

				-- Update queue with all changes still missing
				if (GLDG_DataGuild[pl].new or GLDG_DataGuild[pl].newlvl or GLDG_DataGuild[pl].newrank) and (not GLDG_Queue[pl]) then
					GLDG_Queue[pl] = "[??:??] "
					update = true;
				end
				GLDG_DataGuild[pl].lvl = lv
			else
				-- offline players are only shown if the corresponding option has been set
				-- this is forced for the initial update. if there is a guild roster update
				-- event before this, only the online players may be set
				-- this is the reason for setting complete only if at least one offline player
				-- has been seen
				complete = true
				if (not GLDG_DataGuild[pl].own) then
					if (lv > GLDG_DataGuild[pl].lvl) then
						if ((GLDG_DataGuild[pl].storedLvl == nil) or (lv > GLDG_DataGuild[pl].storedLvl)) then
							if GLDG_Data.ListLevelUpOff then
								GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":"..GLDG_GOES_OFFLINE_COLOUR.." ["..pl.."] "..GLDG_TXT.lvlIncrease1.." "..GLDG_DataGuild[pl].lvl.." "..GLDG_TXT.lvlIncrease2.." "..lv.." "..GLDG_TXT.lvlIncrease3.." (off)");
							end

							GLDG_DataGuild[pl].storedLvl = lv
						end
					end
				end
			end
			if not GLDG_Offline[pl] then
				GLDG_Offline[pl] = false
			end
			if (GLDG_Online[pl] == nil) then
				GLDG_Online[pl] = ol
			end
		end
	end

	-- Parse guild notes for auto assignation
	if (GLDG_Data.AutoAssign) then
		for p in pairs(mains) do
			if (GLDG_DataGuild[p] and GLDG_DataGuild[p].alt==nil and GLDG_DataGuild[p].main==nil) then
				GLDG_DataGuild[p].main = true
			end
		end
		for p in pairs(alts) do
			if (GLDG_DataGuild[p] and GLDG_DataGuild[p].alt==nil and GLDG_DataGuild[p].main==nil) then
				if (GLDG_DataGuild[alts[p]] and GLDG_DataGuild[alts[p]].main) then
					GLDG_DataGuild[p].alt = alts[p]
					GLDG_DataGuild[p].main = nil
				end
			end
		end
	end

	if not GLDG_checkedChannel then
		GLDG_CheckChannel()
	end

	-- If we got our info, switch to the next phase
	if (cnt > 0) then
		if (complete and not GLDG_InitialGuildUpdate) then
			GLDG_InitialGuildUpdate = true
			GLDG_NewGuild = false

			if (GLDG_Data.UpdateTime > 0) then
				GLDG_UpdateRequest = GetTime() + GLDG_Data.UpdateTime
			else
				GLDG_UpdateRequest = nil
			end

			GLDG_RosterPurge()
			GLDG_Main:RegisterEvent("CHAT_MSG_SYSTEM")
		end
		if (update) then
			GLDG_ShowQueue()
		end

		if (GLDG_Data.AutoGreet[GLDG_Realm.." - "..GLDG_Player] and (GLDG_autoGreeted == 0)) then
			GLDG_KeyGreet()
			GLDG_autoGreeted = 1
		end
	end
end

------------------------------------------------------------
function GLDG_RosterPurge()
	-- Don't purge if list is not complete
	if not GetGuildRosterShowOffline() then return end
	-- Set guildlist
	local purge = {}
	-- Purge old members from database
	for p in pairs(GLDG_DataGuild) do if (GLDG_Offline[p] == nil) then purge[p] = true end end
	for p in pairs(purge) do
		if GLDG_Data.ListQuit then
			GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r ["..p.."] "..GLDG_TXT.leftguild);
		end
		GLDG_DataGuild[p] = nil

		-- check if we're deleting the main to an alt
		for q in pairs(GLDG_DataGuild) do
			if (GLDG_DataGuild[q].alt and GLDG_DataGuild[q].alt==p) then
				GLDG_DataGuild[q].alt = nil
			end
		end
	end
end


------------------------------------------
-- assemble list of alts of a main char --
------------------------------------------
-- bee
function GLDG_FindAlts(mainName, playerName, colourise)
	local altList = "";

	local details = GLDG_findPlayerDetails(playerName)

	for player in pairs(GLDG_DataGuild) do
		local color = GLDG_IS_OFFLINE_COLOUR;
		local endcolor = "|r";
		if (colourise == 0) then
			color = GLDG_GOES_OFFLINE_COLOUR;
		elseif ( player == playerName) then
			color = GLDG_ONLINE_COLOUR;
			endcolor = "|r";
		end
		if ( GLDG_DataGuild[player].main and (player == mainName) ) then
			if (player == playerName and details~="") then
				altList = altList..color.."["..player..": "..details.."]"..endcolor.." ";
			else
				altList = altList..color.."["..player.."]"..endcolor.." ";
			end
		elseif ( GLDG_DataGuild[player].alt == mainName ) then
			if (player == playerName and details~="") then
				altList = altList..color.."("..player..": "..details..")"..endcolor.." ";
			else
				altList = altList..color..player..endcolor.." ";
			end
		end
	end

	altList = altList..GLDG_findAlias(playerName, colourise);

	return altList
end

------------------------------------------
-- bee
function GLDG_FindAlts_f(mainName, playerName, colourise)
	local altList = "";

	local details = GLDG_findPlayerDetails_f(playerName)

	for player in pairs(GLDG_DataFriends.friends) do
		local color = GLDG_IS_OFFLINE_COLOUR;
		local endcolor = "|r";
		if (colourise == 0) then
			color = GLDG_GOES_OFFLINE_COLOUR;
		elseif ( player == playerName) then
			color = GLDG_ONLINE_COLOUR;
			endcolor = "|r";
		end
		if ( GLDG_DataFriends.friends[player].main and (player == mainName) ) then
			if (player == playerName and details~="") then
				altList = altList..color.."["..player..": "..details.."]"..endcolor.." ";
			else
				altList = altList..color.."["..player.."]"..endcolor.." ";
			end
		elseif ( GLDG_DataFriends.friends[player].alt == mainName ) then
			if (player == playerName and details~="") then
				altList = altList..color.."("..player..": "..details..")"..endcolor.." ";
			else
				altList = altList..color..player..endcolor.." ";
			end
		end
	end

	altList = altList..GLDG_findAlias_f(playerName, colourise);

	altList = altList.." {f}"

	return altList
end

------------------------------------------
-- bee
function GLDG_FindAlts_c(mainName, playerName, colourise)
	local altList = "";

	for player in pairs(GLDG_DataChannel) do
		local color = GLDG_IS_OFFLINE_COLOUR;
		local endcolor = "|r";
		if (colourise == 0) then
			color = GLDG_GOES_OFFLINE_COLOUR;
		elseif ( player == playerName) then
			color = GLDG_ONLINE_COLOUR;
			endcolor = "|r";
		end
		if ( GLDG_DataChannel[player].main and (player == mainName) ) then
			--if (player == playerName) then
				altList = altList..color.."["..player.."]"..endcolor.." ";

			--end
		elseif ( GLDG_DataChannel[player].alt == mainName ) then
			--if (player == playerName) then
				altList = altList..color..player..endcolor.." ";
			--end
		end
	end

	altList = altList..GLDG_findAlias_c(playerName, colourise);

	altList = altList.." {c}"

	return altList
end


------------------------------------------
-- get a list of players that are on --
------------------------------------------
-- bee
function GLDG_getOnlineList()
	local onList = {};
	GuildRoster();
	local numTotal = GetNumGuildMembers(true);
	for i = 0, numTotal do
		local name, rank, rankIndex, level, class, zone, note, officernote, online, status = GetGuildRosterInfo(i);
		if ((name ~= nil) and ( online ~= nil)) then
			onList[name] = true;
		end
	end

	return onList;
end

------------------------------------------
-- bee
function GLDG_getOnlineList_f()
	local onList = {};

	local numTotal = GetNumFriends()
	for i = 1, numTotal do
		local name, level, class, loc, connected, status = GetFriendInfo(i);
		if ((name ~= nil) and connected) then
			onList[name] = true;
		end
	end

	return onList;
end

------------------------------------------
-- bee
function GLDG_getOnlineList_c()
	local onList = {};

	for p in pairs(GLDG_DataChannel) do
		if (GLDG_Online[p]) then
			onList[p] = true
		end
	end

	return onList;
end

------------------------------------------
-- find player details for a player --
------------------------------------------
-- bee
function GLDG_findPlayerDetails(playerName)
	local result = "";

	GuildRoster();
	local numTotal = GetNumGuildMembers(true);
	for i = 0, numTotal do
		local name, rank, rankIndex, level, class, zone, note, officernote, online, status = GetGuildRosterInfo(i);
		if ((name ~= nil) and ( name == playerName)) then
			result = class..", "..level
		end
	end

	return result;
end

------------------------------------------
-- bee
function GLDG_findPlayerDetails_f(playerName)
	local result = "";

	local numTotal = GetNumFriends()
	for i = 1, numTotal do
		local name, level, class, loc, connected, status = GetFriendInfo(i);
		if ((name ~= nil) and ( name == playerName) and connected) then
			result = class..", "..level
		end
	end

	return result;
end

------------------------------------------
-- find alias for current player  --
------------------------------------------
-- bee
function GLDG_findAlias(playerName, colourise)
	local result = "";

	local alias = "";

	local aliasColour = "";

	if (colourise == 0) then
		aliasColour = GLDG_GOES_OFFLINE_COLOUR;
	elseif (colourise == 1) then
		aliasColour = GLDG_ALIAS_COLOUR;
	end

	if (playerName) then
		if (GLDG_DataGuild[playerName]) then
			if (GLDG_DataGuild[playerName].alias) then
				alias = GLDG_DataGuild[playerName].alias;
			else
				alias = playerName;
			end
			if (GLDG_Data.GreetAsMain and GLDG_DataGuild[playerName].alt) then
				if (GLDG_DataGuild[GLDG_DataGuild[playerName].alt].alias) then
					alias = GLDG_DataGuild[GLDG_DataGuild[playerName].alt].alias;
				else
					alias = GLDG_DataGuild[playerName].alt;
				end
			end
		end
	end

	if (alias ~= "") then
		result = result.." - "..aliasColour..alias.."|r";
	end

	return result;
end

------------------------------------------
-- bee
function GLDG_findAlias_f(playerName, colourise)
	local result = "";

	local alias = "";

	local aliasColour = "";

	if (colourise == 0) then
		aliasColour = GLDG_GOES_OFFLINE_COLOUR;
	elseif (colourise == 1) then
		aliasColour = GLDG_ALIAS_COLOUR;
	end

	if (playerName) then
		if (GLDG_DataFriends.friends[playerName]) then
			if (GLDG_DataFriends.friends[playerName].alias) then
				alias = GLDG_DataFriends.friends[playerName].alias;
			else
				alias = playerName;
			end
			if (GLDG_Data.GreetAsMain and GLDG_DataFriends.friends[playerName].alt) then
				if (GLDG_DataFriends.friends[GLDG_DataFriends.friends[playerName].alt].alias) then
					alias = GLDG_DataFriends.friends[GLDG_DataFriends.friends[playerName].alt].alias;
				else
					alias = GLDG_DataFriends.friends[playerName].alt;
				end
			end
		end
	end

	if (alias ~= "") then
		result = result.." - "..aliasColour..alias.."|r";
	end

	return result;
end

------------------------------------------
-- bee
function GLDG_findAlias_c(playerName, colourise)
	local result = "";

	local alias = "";

	local aliasColour = "";

	if (colourise == 0) then
		aliasColour = GLDG_GOES_OFFLINE_COLOUR;
	elseif (colourise == 1) then
		aliasColour = GLDG_ALIAS_COLOUR;
	end

	if (playerName) then
		if (GLDG_DataChannel[playerName]) then
			if (GLDG_DataChannel[playerName].alias) then
				alias = GLDG_DataChannel[playerName].alias;
			else
				alias = playerName;
			end
			if (GLDG_Data.GreetAsMain and GLDG_DataChannel[playerName].alt) then
				if (GLDG_DataChannel[GLDG_DataChannel[playerName].alt].alias) then
					alias = GLDG_DataChannel[GLDG_DataChannel[playerName].alt].alias;
				else
					alias = GLDG_DataChannel[playerName].alt;
				end
			end
		end
	end

	if (alias ~= "") then
		result = result.." - "..aliasColour..alias.."|r";
	end

	return result;
end

------------------------------------------
-- list of alts and main of a char --
------------------------------------------
-- bee
function GLDG_ListForPlayer(playerName, allDetails, onList, print)
	local result = "";

	if (print == nil) then
		print = true;
	end
	if (allDetails == nil) then
		allDetails = false
	end
	if (playerName ~= nil) then
		if GLDG_DataGuild[playerName] ~= nil then
			GLDG_SetActiveColourSet("guild")

			if ( onList == nil) then
				onList = GLDG_getOnlineList();
			end

			local mainName;
			if ( GLDG_DataGuild[playerName].main ) then
				mainName = playerName
			elseif ( GLDG_DataGuild[playerName].alt ~= nil ) then
				mainName = GLDG_DataGuild[playerName].alt;
			else
				mainName = "-";
			end

			local details = GLDG_findPlayerDetails(playerName);
			local playerDetails = "";

			for player in pairs(GLDG_DataGuild) do
				local color = "";
				local endcolor = "";
				if (print) then
					color = GLDG_IS_OFFLINE_COLOUR;
					endcolor = "|r";
				end
				if ( onList[player] and print) then
					color = GLDG_ONLINE_COLOUR;
					endcolor = "|r";
				end

				if (allDetails) then
					playerDetails = GLDG_findPlayerDetails(player)
				end

				if ( GLDG_DataGuild[player].main and (player == mainName) ) then
					if ((player == playerName) and print and details ~= "") then
						result = result..color.."["..player..": "..details.."]"..endcolor.." ";
					elseif (player == playerName) then
						result = result..color.."["..player.."]"..endcolor.." ";
					elseif (playerDetails~="" and print) then
						result = result..color.."["..player..": "..playerDetails.."]"..endcolor.." ";
					else
						result = result..color.."["..player.."]"..endcolor.." ";
					end
				elseif ( GLDG_DataGuild[player].alt == mainName ) then
					if ((player == playerName) and print and details ~= "") then
						result = result..color.."("..player..": "..details..")"..endcolor.." ";
					elseif (player == playerName) then
						result = result..color..player..endcolor.." ";
					elseif (playerDetails~="" and print) then
						result = result..color.."("..player..": "..playerDetails..")"..endcolor.." ";
					else
						result = result..color..player..endcolor.." ";
					end
				elseif ((player == playerName) and print and details ~= "") then
					result = result..color.."("..player..": "..details..")"..endcolor.." ";
				elseif (player == playerName) then
					result = result..color..player..endcolor.." ";
				end
			end

			if (print) then
				result = result..GLDG_findAlias(playerName, 1);
			else
				result = result..GLDG_findAlias(playerName, 2);
			end
		elseif GLDG_Data.UseFriends and GLDG_DataFriends.friends[playerName] ~= nil then
			GLDG_SetActiveColourSet("friends")

			if ( onList == nil) then
				onList = GLDG_getOnlineList_f();
			end

			local mainName;
			if ( GLDG_DataFriends.friends[playerName].main ) then
				mainName = playerName
			elseif ( GLDG_DataFriends.friends[playerName].alt ~= nil ) then
				mainName = GLDG_DataFriends.friends[playerName].alt;
			else
				mainName = "-";
			end

			local details = GLDG_findPlayerDetails_f(playerName);
			local playerDetails = ""

			for player in pairs(GLDG_DataFriends.friends) do
				local color = "";
				local endcolor = "";
				if (print) then
					color = GLDG_IS_OFFLINE_COLOUR;
					endcolor = "|r";
				end
				if ( onList[player] and print) then
					color = GLDG_ONLINE_COLOUR;
					endcolor = "|r";
				end

				if (allDetails) then
					playerDetails = GLDG_findPlayerDetails_f(player)
				end

				if ( GLDG_DataFriends.friends[player].main and (player == mainName) ) then
					if ((player == playerName) and print and details ~= "") then
						result = result..color.."["..player..": "..details.."]"..endcolor.." ";
					elseif (player == playerName) then
						result = result..color.."["..player.."]"..endcolor.." ";
					elseif (playerDetails~="" and print) then
						result = result..color.."["..player..": "..playerDetails.."]"..endcolor.." ";
					else
						result = result..color.."["..player.."]"..endcolor.." ";
					end
				elseif ( GLDG_DataFriends.friends[player].alt == mainName ) then
					if ((player == playerName) and print and details ~= "") then
						result = result..color.."("..player..": "..details..")"..endcolor.." ";
					elseif (player == playerName) then
						result = result..color..player..endcolor.." ";
					elseif (playerDetails~="" and print) then
						result = result..color.."("..player..": "..playerDetails..")"..endcolor.." ";
					else
						result = result..color..player..endcolor.." ";
					end
				elseif ((player == playerName) and print and details ~= "") then
					result = result..color.."("..player..": "..details..")"..endcolor.." ";
				elseif (player == playerName) then
					result = result..color..player..endcolor.." ";
				end
			end

			if (print) then
				result = result..GLDG_findAlias_f(playerName, 1).." {f}";
			else
				result = result..GLDG_findAlias_f(playerName, 2).." {f}";
			end
		elseif GLDG_inChannel and GLDG_DataChannel[playerName] ~= nil then
			GLDG_SetActiveColourSet("channel")

			if ( onList == nil) then
				onList = GLDG_getOnlineList_c();
			end

			local mainName;
			if ( GLDG_DataChannel[playerName].main ) then
				mainName = playerName
			elseif ( GLDG_DataChannel[playerName].alt ~= nil ) then
				mainName = GLDG_DataChannel[playerName].alt;
			else
				mainName = "-";
			end

			local details = "";
			local playerDetails = "";

			for player in pairs(GLDG_DataChannel) do
				local color = "";
				local endcolor = "";
				if (print) then
					color = GLDG_IS_OFFLINE_COLOUR;
					endcolor = "|r";
				end
				if ( onList[player] and print) then
					color = GLDG_ONLINE_COLOUR;
					endcolor = "|r";
				end

				if ( GLDG_DataChannel[player].main and (player == mainName) ) then
					if ((player == playerName) and print and details ~= "") then
						result = result..color.."["..player..": "..details.."]"..endcolor.." ";
					elseif (player == playerName) then
						result = result..color.."["..player.."]"..endcolor.." ";
					elseif (playerDetails~="" and print) then
						result = result..color.."["..player..": "..playerDetails.."]"..endcolor.." ";
					else
						result = result..color.."["..player.."]"..endcolor.." ";
					end
				elseif ( GLDG_DataChannel[player].alt == mainName ) then
					if ((player == playerName) and print and details ~= "") then
						result = result..color.."("..player..": "..details..")"..endcolor.." ";
					elseif (player == playerName) then
						result = result..color..player..endcolor.." ";
					elseif (playerDetails~="" and print) then
						result = result..color.."("..player..": "..playerDetails..")"..endcolor.." ";
					else
						result = result..color..player..endcolor.." ";
					end
				elseif ((player == playerName) and print and details ~= "") then
					result = result..color.."("..player..": "..details..")"..endcolor.." ";
				elseif (player == playerName) then
					result = result..color..player..endcolor.." ";
				end
			end

			if (print) then
				result = result..GLDG_findAlias_c(playerName, 1).." {c}";
			else
				result = result..GLDG_findAlias_c(playerName, 2).." {c}";
			end
		else
			result = playerName.." "..GLDG_TXT.notinguild;
			if GLDG_Data.UseFriends then
				result = result.." "..GLDG_TXT.notfriend;
			end
			if GLDG_inChannel then
				result = result.." "..GLDG_TXT.notchannel;
			end
		end
		if (print) then
			GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..result);
		end
	end

	return result;
end

------------------------------------------
-- list of alts and main of a char --
------------------------------------------
-- bee
function GLDG_ListAllPlayers(offline, print)
	local onList = GLDG_getOnlineList();
	local line = "";
	local result = {};
	local i = 0;

	GLDG_SetActiveColourSet("guild")

	if offline then
		if (print) then
			GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_TXT.listall)
		end
		for player in pairs(GLDG_DataGuild) do
			if ( GLDG_DataGuild[player].main or not GLDG_DataGuild[player].alt ) then
				line = GLDG_ListForPlayer(player, false, onList, print);
				result[i] = line;
				i = i+1;
			end
		end
	else
		if (print) then
			GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_TXT.listonline)
		end
		for player in pairs(onList) do
			line = GLDG_ListForPlayer(player, false, onList, print);
			result[i] = line;
			i = i+1;
		end
	end

	return result;
end

------------------------------------------
-- bee
function GLDG_ListAllPlayers_f(offline, print)
	local onList = GLDG_getOnlineList_f();
	local line = "";
	local result = {};
	local i = 0;

	GLDG_SetActiveColourSet("friends")

	if offline then
		if (print) then
			GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_TXT.listall_friends)
		end
		for player in pairs(GLDG_DataFriends.friends) do
			if (GLDG_DataFriends.chars[GLDG_Player][player]) then
				if ( GLDG_DataFriends.friends[player].main or not GLDG_DataFriends.friends[player].alt ) then
					line = GLDG_ListForPlayer(player, false, onList, print);
					result[i] = line;
					i = i+1;
				end
			end
		end
	else
		if (print) then
			GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_TXT.listonline_friends)
		end
		for player in pairs(onList) do
			line = GLDG_ListForPlayer(player, false, onList, print);
			result[i] = line;
			i = i+1;
		end
	end

	return result;
end

------------------------------------------
-- bee
function GLDG_ListAllPlayers_c(offline, print)
	local onList = GLDG_getOnlineList_c();
	local line = "";
	local result = {};
	local i = 0;

	GLDG_SetActiveColourSet("channel")

	if offline then
		if (print) then
			GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_TXT.listall_channel)
		end
		for player in pairs(GLDG_DataChannel) do
			if (( GLDG_DataChannel[player].main or not GLDG_DataChannel[player].alt ) and
			    ( not GLDG_DataGuild[player])) then
				line = GLDG_ListForPlayer(player, false, onList, print);
				result[i] = line;
				i = i+1;
			end
		end
	else
		if (print) then
			GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_TXT.listonline_channel)
		end
		for player in pairs(onList) do
			if not GLDG_DataGuild[player] then
				line = GLDG_ListForPlayer(player, false, onList, print);
				result[i] = line;
				i = i+1;
			end
		end
	end

	return result;
end


----------------------------------
-- _06_ Monitor System Messages --
----------------------------------

function GLDG_SystemMsg(msg)
	-- Receiving system message
	GLDG_DebugPrint("incoming system message: "..msg)

	-- Check players coming online
	local _, _, player = string.find(msg, GLDG_ONLINE)
	if player then
		GLDG_DebugPrint("detected player coming online: "..player)
		if (GLDG_DataGuild[player] and not GLDG_DataGuild[player].ignore) then
			SendAddonMessage("GLDG", "VER:"..GetAddOnMetadata("GuildGreet", "Version"), "GUILD")

			GLDG_DebugPrint("player "..player.." is a member of our guild")
			GLDG_Online[player] = GetTime()

			-- bee
			if GLDG_Data.ListNames then
				if GLDG_DataGuild[player].alt then
					--
					-- Alt von Main
					--
					local main = GLDG_DataGuild[player].alt;
					local altsList = GLDG_FindAlts(main, player, 1)
					GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..altsList)
				else
					if GLDG_DataGuild[player].main then
						--
						-- Main
						--
						local main = player;
						local altsList = GLDG_FindAlts(main, player, 1)
						GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..altsList)
					else
						--
						-- Hat keinen Alt
						--
						local details = GLDG_findPlayerDetails(player);
						local alias = GLDG_findAlias(player, 1);

						if (details ~= "") then
							GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_ONLINE_COLOUR..player..": "..details.."|r"..alias)
						else
							GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_ONLINE_COLOUR..player.."|r"..alias)
						end
					end
				end
			end

			if ((GLDG_DataGuild[player].storedLvl ~= nil) and (GLDG_DataGuild[player].storedLvl > GLDG_DataGuild[player].lvl)) then
				if (not GLDG_DataGuild[player].own) then
					if ( (GLDG_TableSize(GLDG_DataGreet.NewLevel) > 0) and not GLDG_Data.SupressLevel and not GLDG_DataGuild[player].ignore and (GLDG_DataGuild[player].storedLvl > GLDG_Data.MinLevelUp)) then

						if (not GLDG_Data.NoGratsOnLogin) then
							GLDG_DataGuild[player].newlvl = true
						end
					end
				end
				GLDG_DataGuild[player].lvl = GLDG_DataGuild[player].storedLvl
				GLDG_DataGuild[player].storedLvl = nil
			end

			if GLDG_DataGuild[player].alt then
				GLDG_Offline[player] = GLDG_Offline[GLDG_DataGuild[player].alt]
			end
			if GLDG_Offline[player] and (GLDG_Online[player] - GLDG_Offline[player] < GLDG_Data.RelogTime * 60) then
				return
			end
			GLDG_DebugPrint("player "..player.." is not been online in the last "..GLDG_Data.RelogTime.." minutes.")
			if GLDG_Offline[player] and ((GLDG_TableSize(GLDG_DataGreet.GreetBack) == 0) or (GLDG_Data.SupressGreet)) then
				return
			end
			GLDG_DebugPrint("player "..player.." is not been online before")
			if not (GLDG_Offline[player] or GLDG_DataGuild[player].new or GLDG_DataGuild[player].newlvl or GLDG_DataGuild[player].newrank) and (GLDG_Data.SupressGreet or (GLDG_TableSize(GLDG_DataGreet.Greet) == 0)) then
				return
			end
			GLDG_DebugPrint("player "..player.." should be greeted")
			GLDG_Queue[player] = GLDG_GetLogtime(player)

			GLDG_ShowQueue()

		elseif (GLDG_Data.UseFriends and GLDG_DataFriends.friends[player] and not GLDG_DataFriends.friends[player].ignore) then

			GLDG_DebugPrint("player "..player.." is a friend")
			GLDG_Online[player] = GetTime()

			-- bee
			if GLDG_Data.ListNames then
				if GLDG_DataFriends.friends[player].alt then
					--
					-- Alt von Main
					--
					local main = GLDG_DataFriends.friends[player].alt;
					local altsList = GLDG_FindAlts_f(main, player, 1)
					GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..altsList)
				else
					if GLDG_DataFriends.friends[player].main then
						--
						-- Main
						--
						local main = player;
						local altsList = GLDG_FindAlts_f(main, player, 1)
						GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..altsList)
					else
						--
						-- Hat keinen Alt
						--
						local details = GLDG_findPlayerDetails_f(player);
						local alias = GLDG_findAlias_f(player, 1);

						if (details ~= "") then
							GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_ONLINE_COLOUR..player..": "..details.."|r"..alias.." {f}")
						else
							GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_ONLINE_COLOUR..player.."|r"..alias.." {f}")
						end
					end
				end
			end

			if GLDG_DataFriends.friends[player].alt then GLDG_Offline[player] = GLDG_Offline[GLDG_DataFriends.friends[player].alt] end
			if GLDG_Offline[player] and (GLDG_Online[player] - GLDG_Offline[player] < GLDG_Data.RelogTime * 60) then return end
			GLDG_DebugPrint("player "..player.." is not been online in the last "..GLDG_Data.RelogTime.." minutes.")
			if GLDG_Offline[player] and (GLDG_Data.SupressGreet or (GLDG_TableSize(GLDG_DataGreet.GreetBack) == 0)) then return end
			GLDG_DebugPrint("player "..player.." is not been online before")
			if not (GLDG_Offline[player] or GLDG_DataFriends.friends[player].new or GLDG_DataFriends.friends[player].newlvl or GLDG_DataFriends.friends[player].newrank) and (GLDG_Data.SupressGreet or (GLDG_TableSize(GLDG_DataGreet.Greet) == 0)) then return end
			GLDG_DebugPrint("player "..player.." should be greeted")
			GLDG_Queue[player] = GLDG_GetLogtime(player)
			GLDG_ShowQueue()
		end
		return end

	-- Check players going offline
	local _, _, player = string.find(msg, GLDG_OFFLINE)
	if player then
		GLDG_DebugPrint("detected player going offline: "..player)
		if (GLDG_DataGuild[player] and not GLDG_DataGuild[player].ignore) then
			GLDG_DebugPrint("player "..player.." is a member of our guild")
			GLDG_Online[player] = nil
			GLDG_RankUpdate[player] = nil
			if GLDG_DataGuild[player].alt then
				local main = GLDG_DataGuild[player].alt
				GLDG_DataGuild[main].last = player
				GLDG_Offline[main] = GetTime()
			elseif GLDG_DataGuild[player].main then
				GLDG_DataGuild[player].last = player
				GLDG_Offline[player] = GetTime()
			else GLDG_Offline[player] = GetTime() end
			if GLDG_Queue[player] then
				GLDG_Queue[player] = nil
				GLDG_ShowQueue() end

			-- bee
			if GLDG_Data.ListNamesOff then
				if GLDG_DataGuild[player].alt then
					--
					-- Alt von Main
					--
					local main = GLDG_DataGuild[player].alt;
					local altsList = GLDG_FindAlts(main, player, 0)
					GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..altsList)
				else
					if GLDG_DataGuild[player].main then
						--
						-- Main
						--
						local main = player;
						local altsList = GLDG_FindAlts(main, player, 0)
						GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..altsList)
					else
						--
						-- Hat keinen Alt
						--
						local details = GLDG_findPlayerDetails(player);
						local alias = GLDG_findAlias(player, 0);

						if (details ~= "") then
							GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_GOES_OFFLINE_COLOUR..player..": "..details..alias)
						else
							GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_GOES_OFFLINE_COLOUR..player..alias)
						end
					end
				end
			end

		elseif (GLDG_Data.UseFriends and GLDG_DataFriends.friends[player] and not GLDG_DataFriends.friends[player].ignore) then

			GLDG_DebugPrint("player "..player.." is a friend")
			GLDG_Online[player] = nil
			GLDG_RankUpdate[player] = nil
			if GLDG_DataFriends.friends[player].alt then
				local main = GLDG_DataFriends.friends[player].alt
				GLDG_DataFriends.friends[main].last = player
				GLDG_Offline[main] = GetTime()
			elseif GLDG_DataFriends.friends[player].main then
				GLDG_DataFriends.friends[player].last = player
				GLDG_Offline[player] = GetTime()
			else GLDG_Offline[player] = GetTime() end
			if GLDG_Queue[player] then
				GLDG_Queue[player] = nil
				GLDG_ShowQueue() end

			-- bee
			if GLDG_Data.ListNamesOff then
				if GLDG_DataFriends.friends[player].alt then
					--
					-- Alt von Main
					--
					local main = GLDG_DataFriends.friends[player].alt;
					local altsList = GLDG_FindAlts_f(main, player, 0)
					GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..altsList)
				else
					if GLDG_DataFriends.friends[player].main then
						--
						-- Main
						--
						local main = player;
						local altsList = GLDG_FindAlts_f(main, player, 0)
						GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..altsList)
					else
						--
						-- Hat keinen Alt
						--
						local details = GLDG_findPlayerDetails_f(player);
						local alias = GLDG_findAlias_f(player, 0);

						if (details ~= "") then
							GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_GOES_OFFLINE_COLOUR..player..": "..details..alias.." {f}")
						else
							GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_GOES_OFFLINE_COLOUR..player..alias.." {f}")
						end
					end
				end
			end
		end

		return end

	-- Check players joining the guild
	local _, _, player = string.find(msg, GLDG_JOINED)
	if player and not GLDG_Data.SupressJoin and (GLDG_TableSize(GLDG_DataGreet.Welcome) > 0) then
		GLDG_DebugPrint("detected player joining guild: "..player)
		GLDG_DataGuild[player] = {}
		GLDG_Online[player] = GetTime()
		GLDG_Offline[player] = false
		GLDG_Queue[player] = GLDG_GetLogtime(player)
		GLDG_DataGuild[player].new = true
		GLDG_ShowQueue()

		SendAddonMessage("GLDG", "VER:"..GetAddOnMetadata("GuildGreet", "Version"), "GUILD")
		return end

	-- Check for promotions
	local _, _, promo, player, rank = string.find(msg, GLDG_PROMO)
	if player and (GLDG_TableSize(GLDG_DataGreet.NewRank) > 0) and not GLDG_Data.SupressRank and not GLDG_DataGuild[player].ignore then
		GLDG_DebugPrint("detected player getting promotion: "..player.." -> "..rank)
		GLDG_DataGuild[player].promoter = promo
		GLDG_DataGuild[player].rankname = rank
		if GLDG_Data.Ranks[GLDG_Realm.."-"..GLDG_GuildName] and GLDG_Data.Ranks[GLDG_Realm.."-"..GLDG_GuildName][rank] then
			GLDG_DataGuild[player].rank = GLDG_Data.Ranks[GLDG_Realm.."-"..GLDG_GuildName][rank] - 1 end
		GLDG_RankUpdate[player] = GLDG_GetLogtime(player)
		GLDG_DataGuild[player].newrank = GLDG_DataGuild[player].rank
		if GLDG_Online[player] then
			GLDG_Queue[player] = GLDG_GetLogtime(player)
			GLDG_ShowQueue() end
		return end

	-- Check for demotions
	local _, _, player, rank = string.find(msg, GLDG_DEMOTE)
	if player then
		GLDG_DebugPrint("detected player getting demotion: "..player.." -> "..rank)
		GLDG_DataGuild[player].promoter = nil
		GLDG_DataGuild[player].rankname = rank
		if GLDG_Data.Ranks[GLDG_Realm.."-"..GLDG_GuildName] and GLDG_Data.Ranks[GLDG_Realm.."-"..GLDG_GuildName][rank] then
			GLDG_DataGuild[player].rank = GLDG_Data.Ranks[GLDG_Realm.."-"..GLDG_GuildName][rank] - 1 end
		GLDG_RankUpdate[player] = GLDG_GetLogtime(player)
		if GLDG_DataGuild[player].newrank then
			GLDG_DataGuild[player].newrank = nil
			GLDG_Queue[player] = nil
			GLDG_ShowQueue() end
		return end
end

------------------------------------------------------------
function GLDG_GetLogtime(player)
	-- Helper function: mark playing coming online
	local hour, minute = GLDG_GetTime()
	if (hour < 10) then hour = "[0"..hour else hour = "["..hour end
	if (minute < 10) then minute = ":0"..minute.."] " else minute = ":"..minute.."] " end
	return hour..minute
end

------------------------------------------------------------
function GLDG_GetTime()
	local hour, minute

	if GLDG_Data.UseLocalTime then
		local t = date('*t')		-- returns local time
		hour = t.hour
		minute = t.min
	else
		hour, minute = GetGameTime()	-- returns server time
	end

	return hour, minute
end


----------------------------
-- _07_ Display Greetlist --
----------------------------

function GLDG_ShowQueue()
	-- Sort queue according logon time
	local queue = {}
	local total = 0
	for p in pairs(GLDG_Queue) do
		-- Look for the position in the list we need
		local loc = 1
		while queue[loc] and (GLDG_Online[queue[loc]] <= GLDG_Online[p]) do
			loc = loc + 1
		end
		-- We found the position: move everything beyond it
		for cnt = total, loc, -1 do
			queue[cnt+1] = queue[cnt]
		end
		-- Insert the player
		total = total + 1
		queue[loc] = p
	end
	-- Display the list as needed
	if (total == 0) and not (GLDG_Debug or GLDG_Data.ListVisible) then
		-- No lines and not debugging: hide main window
		getglobal(GLDG_LIST):Hide()
		return end
	-- Hide the unused direction
	local dir = "ULine"
	if GLDG_Data.ListUp then dir = "Line" end
	cnt = 1
	while getglobal(GLDG_LIST..dir..cnt) do
		getglobal(GLDG_LIST..dir..cnt):Hide()
		cnt = cnt + 1 end
	if GLDG_Data.ListUp then dir = "ULine" else dir = "Line" end
	-- Show the used direction
	for cnt = 1, math.min(total, GLDG_Data.ListSize) do
		local line = GLDG_LIST..dir..cnt
		local colorName = "list"
		local setName = "guild"
		getglobal(line.."Text"):SetText(GLDG_Queue[queue[cnt]]..queue[cnt])
		getglobal(line.."Text2"):SetText("")
		if (GLDG_DataGuild[queue[cnt]]) then
			if GLDG_DataGuild[queue[cnt]].new then
				colorName = "new"
				getglobal(line.."Text2"):SetText(GLDG_TXT.new)
			elseif GLDG_DataGuild[queue[cnt]].newlvl then
				colorName = "lvl"
				getglobal(line.."Text2"):SetText(GLDG_TXT.lvl)
			elseif GLDG_DataGuild[queue[cnt]].newrank then
				colorName = "rank"
				getglobal(line.."Text2"):SetText(GLDG_TXT.rank)
			elseif GLDG_Offline[queue[cnt]] then
				colorName = "relog" end
		elseif (GLDG_Data.UseFriends and GLDG_DataFriends.friends[queue[cnt]]) then
			setName = "friends"
			getglobal(line.."Text2"):SetText("{f}")
			if GLDG_DataFriends.friends[queue[cnt]].new then
				colorName = "new"
				getglobal(line.."Text2"):SetText(GLDG_TXT.new.." {f}")
			elseif GLDG_DataFriends.friends[queue[cnt]].newlvl then
				colorName = "lvl"
				getglobal(line.."Text2"):SetText(GLDG_TXT.lvl.." {f}")
			elseif GLDG_DataFriends.friends[queue[cnt]].newrank then
				colorName = "rank"
				getglobal(line.."Text2"):SetText(GLDG_TXT.rank.." {f}")
			elseif GLDG_Offline[queue[cnt]] then
				colorName = "relog"
			end
		elseif (GLDG_inChannel and GLDG_DataChannel[queue[cnt]]) then
			setName = "channel"
			getglobal(line.."Text2"):SetText("{c}")
			if GLDG_DataChannel[queue[cnt]].new then
				colorName = "new"
				getglobal(line.."Text2"):SetText(GLDG_TXT.new.." {c}")
			elseif GLDG_DataChannel[queue[cnt]].newlvl then
				colorName = "lvl"
				getglobal(line.."Text2"):SetText(GLDG_TXT.lvl.." {c}")
			elseif GLDG_DataChannel[queue[cnt]].newrank then
				colorName = "rank"
				getglobal(line.."Text2"):SetText(GLDG_TXT.rank.." {c}")
			elseif GLDG_Offline[queue[cnt]] then
				colorName = "relog"
			end

		-- handle test entries
		elseif (string.sub(queue[cnt], 1, 6)=="Tester") then
			local number = string.sub(queue[cnt], 7)
			setName = "guild"
			if (number == "1") then
				getglobal(line.."Text2"):SetText("{t}")
				colorName = "list"
			elseif (number == "2") then
				getglobal(line.."Text2"):SetText(GLDG_TXT.new.." {t}")
				colorName = "new"
			elseif (number == "3") then
				getglobal(line.."Text2"):SetText(GLDG_TXT.lvl.." {t}")
				colorName = "lvl"
			elseif (number == "4") then
				getglobal(line.."Text2"):SetText(GLDG_TXT.rank.." {t}")
				colorName = "rank"
			elseif (number == "5") then
				getglobal(line.."Text2"):SetText("{t}")
				colorName = "relog"
			end
		end
		GLDG_SetTextColor(line.."Text", setName, colorName)
		GLDG_SetTextColor(line.."Text2", setName, colorName)
		getglobal(line):Show() end
	-- Hide lines we don't need
	cnt = total + 1
	while getglobal(GLDG_LIST..dir..cnt) do
		getglobal(GLDG_LIST..dir..cnt):Hide()
		cnt = cnt + 1 end

	-- Set header colour
	local a,r,g,b = GLDG_ColourToRGB_perc(GLDG_Data.colours.header)
	local f = 1-((r+g+b)/3)
	GuildGreetListTitleText:SetTextColor(f,f,f,1)
	GuildGreetListTitleTexture:SetTexture(r,g,b,a)

	-- Show main window
	getglobal(GLDG_LIST):Show()
end


--------------------------
-- _08_ Display Greeting Tooltip --
--------------------------

function GLDG_ShowToolTip(buttonName)
	-- Tooltip title: use player name and color
	local name = string.sub(getglobal(buttonName.."Text"):GetText(), 9)
	local logtime = string.sub(getglobal(buttonName.."Text"):GetText(), 2, 6)
	local r, g, b = getglobal(buttonName.."Text"):GetTextColor()
	-- Construct tip
	local tip = string.format(GLDG_TXT.tipdef, logtime)
	if GLDG_DataGuild[name] then
		if GLDG_DataGuild[name].new then
			if (logtime == "??:??") then
				tip = GLDG_TXT.tipnew2
			else
				tip = string.format(GLDG_TXT.tipnew, logtime)
			end

		elseif GLDG_DataGuild[name].newlvl then
			tip = string.format(GLDG_TXT.tiplvl, GLDG_DataGuild[name].lvl)

		elseif GLDG_DataGuild[name].newrank then
			if GLDG_DataGuild[name].promoter then
				 if (logtime == "??:??") then
				 	tip = string.format(GLDG_TXT.tiprank, GLDG_DataGuild[name].promoter, GLDG_DataGuild[name].rankname)
				 else
				 	tip = string.format(GLDG_TXT.tiprank2, logtime, GLDG_DataGuild[name].promoter, GLDG_DataGuild[name].rankname)
				 end
			else
				tip = string.format(GLDG_TXT.tiprank3, GLDG_DataGuild[name].rankname)
			end

		elseif GLDG_Offline[name] then
			local minoff = math.ceil((GLDG_Online[name] - GLDG_Offline[name]) / 60)
			local hrsoff = math.floor(minoff / 60)
			minoff = math.fmod(minoff, 60)
			local timestr = ""
			if (hrsoff > 0) then timestr = string.format(GLDG_TXT.hour, hrsoff) end
			if (minoff > 0) then timestr = timestr..string.format(GLDG_TXT.minute, minoff) end
			tip = string.format(GLDG_TXT.tiprelog, logtime, timestr) end

		-- If this is a main, add last used character to tip
		if GLDG_DataGuild[name].main and GLDG_DataGuild[name].last then
			tip = tip..string.format(GLDG_TXT.tiprelogalt, GLDG_DataGuild[name].last)

		-- If this is not the main, add last used character and main information to tip
		elseif GLDG_DataGuild[name].alt then
			local main = GLDG_DataGuild[name].alt
			if GLDG_DataGuild[main].last then
				tip = tip..string.format(GLDG_TXT.tiprelogalt, GLDG_DataGuild[main].last)
			end
			tip = tip..string.format(GLDG_TXT.tipalt, GLDG_DataGuild[name].alt)
		end

		-- If player has alias, add to name
		if GLDG_DataGuild[name].alias then
			name = name.." ("..GLDG_DataGuild[name].alias..")"
		end

	elseif GLDG_Data.UseFriends and GLDG_DataFriends.friends[name] then
		tip = string.format(GLDG_TXT.tipdef_f, logtime)
		if GLDG_DataFriends.friends[name].new then
			if (logtime == "??:??") then
				tip = GLDG_TXT.tipnew2
			else
				tip = string.format(GLDG_TXT.tipnew, logtime)
			end

		elseif GLDG_DataFriends.friends[name].newlvl then
			tip = string.format(GLDG_TXT.tiplvl_f, GLDG_DataFriends.friends[name].lvl)

		elseif GLDG_DataFriends.friends[name].newrank then
			if GLDG_DataFriends.friends[name].promoter then
				 if (logtime == "??:??") then
				 	tip = string.format(GLDG_TXT.tiprank, GLDG_DataFriends.friends[name].promoter, GLDG_DataFriends.friends[name].rankname)
				 else
				 	tip = string.format(GLDG_TXT.tiprank2, logtime, GLDG_DataFriends.friends[name].promoter, GLDG_DataFriends.friends[name].rankname)
				 end
			else
				tip = string.format(GLDG_TXT.tiprank3, GLDG_DataFriends.friends[name].rankname)
			end

		elseif GLDG_Offline[name] then
			local minoff = math.ceil((GLDG_Online[name] - GLDG_Offline[name]) / 60)
			local hrsoff = math.floor(minoff / 60)
			minoff = math.fmod(minoff, 60)
			local timestr = ""
			if (hrsoff > 0) then timestr = string.format(GLDG_TXT.hour, hrsoff) end
			if (minoff > 0) then timestr = timestr..string.format(GLDG_TXT.minute, minoff) end
			tip = string.format(GLDG_TXT.tiprelog_f, logtime, timestr)
		end

		-- If this is a main, add last used character to tip
		if GLDG_DataFriends.friends[name].main and GLDG_DataFriends.friends[name].last then
			tip = tip..string.format(GLDG_TXT.tiprelogalt, GLDG_DataFriends.friends[name].last)

		-- If this is not the main, add last used character and main information to tip
		elseif GLDG_DataFriends.friends[name].alt then
			local main = GLDG_DataFriends.friends[name].alt
			if GLDG_DataFriends.friends[main].last then
				tip = tip..string.format(GLDG_TXT.tiprelogalt, GLDG_DataFriends.friends[main].last)
			end
			tip = tip..string.format(GLDG_TXT.tipalt, GLDG_DataFriends.friends[name].alt)
		end

		-- If player has alias, add to name
		if GLDG_DataFriends.friends[name].alias then
			name = name.." ("..GLDG_DataFriends.friends[name].alias..")"
		end
		name = name.." {f}"


	elseif GLDG_DataChannel[name] then
		tip = string.format(GLDG_TXT.tipdef_c, logtime)
		if GLDG_DataChannel[name].new then
			if (logtime == "??:??") then
				tip = GLDG_TXT.tipnew2
			else
				tip = string.format(GLDG_TXT.tipnew, logtime)
			end

		elseif GLDG_DataChannel[name].newlvl then
			tip = string.format(GLDG_TXT.tiplvl_c, GLDG_DataChannel[name].lvl)

		elseif GLDG_DataChannel[name].newrank then
			if GLDG_DataChannel[name].promoter then
				 if (logtime == "??:??") then
				 	tip = string.format(GLDG_TXT.tiprank, GLDG_DataChannel[name].promoter, GLDG_DataChannel[name].rankname)
				 else
				 	tip = string.format(GLDG_TXT.tiprank2, logtime, GLDG_DataChannel[name].promoter, GLDG_DataChannel[name].rankname)
				 end
			else
				tip = string.format(GLDG_TXT.tiprank3, GLDG_DataChannel[name].rankname)
			end

		elseif GLDG_Offline[name] then
			local minoff = math.ceil((GLDG_Online[name] - GLDG_Offline[name]) / 60)
			local hrsoff = math.floor(minoff / 60)
			minoff = math.fmod(minoff, 60)
			local timestr = ""
			if (hrsoff > 0) then timestr = string.format(GLDG_TXT.hour, hrsoff) end
			if (minoff > 0) then timestr = timestr..string.format(GLDG_TXT.minute, minoff) end
			tip = string.format(GLDG_TXT.tiprelog_c, logtime, timestr)
		end

		-- If this is a main, add last used character to tip
		if GLDG_DataChannel[name].main and GLDG_DataChannel[name].last then
			tip = tip..string.format(GLDG_TXT.tiprelogalt, GLDG_DataChannel[name].last)

		-- If this is not the main, add last used character and main information to tip
		elseif GLDG_DataChannel[name].alt then
			local main = GLDG_DataChannel[name].alt
			if GLDG_DataChannel[main].last then
				tip = tip..string.format(GLDG_TXT.tiprelogalt, GLDG_DataChannel[main].last)
			end
			tip = tip..string.format(GLDG_TXT.tipalt, GLDG_DataChannel[name].alt)
		end

		-- If player has alias, add to name
		if GLDG_DataChannel[name].alias then
			name = name.." ("..GLDG_DataChannel[name].alias..")"
		end
		name = name.." {c}"

	-- handle test entries
	elseif (string.sub(name, 1, 6)=="Tester") then
		local number = string.sub(name, 7)
		if (number == "1") then
			tip = string.format(GLDG_TXT.tipdef, logtime)
		elseif (number == "2") then
			tip = string.format(GLDG_TXT.tipnew, logtime)
		elseif (number == "3") then
			tip = string.format(GLDG_TXT.tiplvl, math.random(70))
		elseif (number == "4") then
			tip = string.format(GLDG_TXT.tiprank2, logtime, "Supertester", "Gopher-Tester")
		elseif (number == "5") then
			tip = string.format(GLDG_TXT.tiprelog, logtime, "some time")
		end
		name = name.." {t}"
	end

	-- Show tooltip
	GameTooltip_SetDefaultAnchor(GameTooltip, this)
	GameTooltip:SetText(name, r, g, b, 1.0, 1)
	GameTooltip:AddLine(tip, 1, 1, 1, 1.0, 1)
	GameTooltip:Show()
end


------------------------
-- _09_ Greet Players --
------------------------

function GLDG_ClickName(button, name)
	-- Strip timestamp from name
	name = string.sub(name, 9)
	-- Greet the player if left click
	if (button == "LeftButton") then GLDG_SendGreet(name) end
	-- Cleanup where needed
	GLDG_CleanupPlayer(name)
	-- Remove name from queue and refresh
	GLDG_Queue[name] = nil
	GLDG_ShowQueue()
end

------------------------------------------------------------
function GLDG_ClearList()
	for name in pairs(GLDG_Queue) do
		GLDG_CleanupPlayer(name)
	end
	GLDG_Queue = {}
	GLDG_ShowQueue()
end

------------------------------------------------------------
function GLDG_CleanupPlayer(name)
	if not name then return end
	if (GLDG_DataGuild[name]) then
		local player = GLDG_DataGuild[name]
		player.new = nil
		player.newlvl = nil
		player.newrank = nil
		player.promoter = nil
	elseif (GLDG_DataFriends.friends[name]) then
		local player = GLDG_DataFriends.friends[name]
		player.new = nil
		player.newlvl = nil
		player.newrank = nil
		player.promoter = nil
	elseif (GLDG_DataChannel[name]) then
		local player = GLDG_DataChannel[name]
		player.new = nil
		player.newlvl = nil
		player.newrank = nil
		player.promoter = nil
	else
		-- take care of test chars
		GLDG_Online[name] = nil
		GLDG_Offline[name] = nil
	end
end

------------------------------------------------------------
function GLDG_SendGreet(name, testing)
	-- find player entry
	local useGuild = true;
	local useFriends = false;
	local option = ""
	local list = GLDG_DataGreet.Greet
	local player = GLDG_DataGuild[name]
	local names = {}

	if (not player) then
		useGuild = false;
		useFriends = true;
		if (GLDG_Data.UseFriends) then
			player = GLDG_DataFriends.friends[name]
		end
		if (not player) then
			useFriends = false;
			if GLDG_inChannel then
				player = GLDG_DataChannel[name]
			end
			if (not player) then
				return
			end
		end
	end
	local cname = name
	table.insert(names, name)

	-- Choose appropriate greeting list
	if player.new then list = GLDG_DataGreet.Welcome
	elseif player.newrank then
		list = GLDG_DataGreet.NewRank
		option = player.rankname
	elseif player.newlvl then
		list = GLDG_DataGreet.NewLevel
		option = player.lvl
	elseif GLDG_Offline[name] then list = GLDG_DataGreet.GreetBack end

	-- Switch to alias if defined
	local usedAlias = false
	if player.alias then
		name = player.alias
		table.insert(names, name)
		usedAlias = true
	end

	-- Use name of main if alt and option is set
	if player.alt then
		local lname = ""
		local lalias = ""
		if (useGuild) then
			if GLDG_DataGuild[player.alt].alias then
				lalias = GLDG_DataGuild[player.alt].alias
			end
			lname = player.alt
		elseif (useFriends) then
			if GLDG_DataFriends.friends[player.alt].alias then
				lalias = GLDG_DataFriends.friends[player.alt].alias
			end
			lname = player.alt
		else
			if GLDG_DataChannel[player.alt].alias then
				lalias = GLDG_DataChannel[player.alt].alias
			end
			lname = player.alt
		end
		if (lname~="") then
			table.insert(names, lname)
		end
		if (lalias~="") then
			table.insert(names, lalias)
			lname = lalias
		end
		if (not usedAlias and GLDG_Data.GreetAsMain and lname~="") then
			name = lname;
		end
	end

	-- Select a random greeting
	local msg = list[math.random(GLDG_TableSize(list))]

	-- Select a random name
	if GLDG_Data.Randomize then
		name = names[math.random(GLDG_TableSize(names))]
	end

	-- Send greeting
	if (testing) then
		GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r ["..string.format(msg, name, option).."]")
	else
		if GLDG_Data.Whisper or useFriends then
			SendChatMessage(string.format(msg, name, option), "WHISPER", nil, cname)
		elseif useGuild then
			SendChatMessage(string.format(msg, name, option), "GUILD")
		else
			local channel = GetChannelName(GLDG_ChannelName)
			if (channel) then
				SendChatMessage(string.format(msg, name, option), "CHANNEL", nil, channel)
			else
				SendChatMessage(string.format(msg, name, option), "WHISPER", nil, cname)
			end
		end
	end
end

------------------------------------------
-- say goodbye to person, using main name
------------------------------------------
function GLDG_SendBye(name, testing)
	if (name == nil) then return end

	-- Choose appropriate greeting list
	local option = ""
	local list = GLDG_DataGreet.Bye

	-- if time is between 20:00 and 06:00 use night mode
	local hour,min = GLDG_GetTime();
	if ((hour >= 20) or (hour <=5)) then
		list = GLDG_DataGreet.Night;
	end

	-- find player entry
	local useGuild = true;
	local useFriends = false
	local player = GLDG_DataGuild[name]
	local names = {}

	if (not player) then
		useGuild = false
		useFriends = true
		if GLDG_Data.UseFriends then
			player = GLDG_DataFriends.friends[name]
		end
		if (not player) then
			useFriends = false;
			if GLDG_inChannel then
				player = GLDG_DataChannel[name]
			end
			if (not player) then
				return
			end
		end
	end

	local cname = name
	table.insert(names, name)

	-- Switch to alias if defined
	local usedAlias = false
	if player.alias then
		name = player.alias
		table.insert(names, name)
		usedAlias = true
	end

	-- Use name of main if alt and option is set
	if player.alt then
		local lname = ""
		local lalias = ""
		if (useGuild) then
			if GLDG_DataGuild[player.alt].alias then
				lalias = GLDG_DataGuild[player.alt].alias
			end
			lname = player.alt
		elseif (useFriends) then
			if GLDG_DataFriends.friends[player.alt].alias then
				lalias = GLDG_DataFriends.friends[player.alt].alias
			end
			lname = player.alt
		else
			if GLDG_DataChannel[player.alt].alias then
				lalias = GLDG_DataChannel[player.alt].alias
			end
			lname = player.alt
		end
		if (lname~="") then
			table.insert(names, lname)
		end
		if (lalias~="") then
			table.insert(names, lalias)
			lname = lalias
		end
		if (not usedAlias and GLDG_Data.GreetAsMain and lname~="") then
			name = lname;
		end
	end

	-- Select a random good bye
	local msg = list[math.random(GLDG_TableSize(list))]

	-- Select a random name
	if GLDG_Data.Randomize then
		name = names[math.random(GLDG_TableSize(names))]
	end

	-- Send good bye
	if (testing) then
		GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r ["..string.format(msg, name, option).."]")
	else
		if GLDG_Data.Whisper or useFriends then
			SendChatMessage(string.format(msg, name, option), "WHISPER", nil, cname)
		elseif useGuild then
			SendChatMessage(string.format(msg, name, option), "GUILD")
		else
			local channel = GetChannelName(GLDG_ChannelName)
			if (channel) then
				SendChatMessage(string.format(msg, name, option), "CHANNEL", nil, channel)
			else
				SendChatMessage(string.format(msg, name, option), "WHISPER", nil, cname)
			end
		end
	end
end

------------------------------------------------------------
function GLDG_KeyGreet()
	if (GLDG_Data.GreetGuild[GLDG_Realm.." - "..GLDG_Player]) then
		GLDG_GreetGuild()
	end
	if (GLDG_Data.GreetChannel[GLDG_Realm.." - "..GLDG_Player]) then
		GLDG_GreetChannel()
	end
end

------------------------------------------------------------
function GLDG_GreetGuild()
	if (GLDG_GuildName=="") then return end

	local msg = GLDG_DataGreet.Guild[math.random(GLDG_TableSize(GLDG_DataGreet.Guild))]

	-- Send greeting
	SendChatMessage(string.format(msg, GLDG_GuildAlias), "GUILD")
end

------------------------------------------------------------
function GLDG_GreetChannel()
	local msg = GLDG_DataGreet.Channel[math.random(GLDG_TableSize(GLDG_DataGreet.Channel))]

	-- Send greeting
	local channel = GetChannelName(GLDG_ChannelName)
	if (channel) then
		SendChatMessage(msg, "CHANNEL", nil, channel)
	end
end

------------------------------------------------------------
function GLDG_KeyBye()
	if (GLDG_Data.GreetGuild[GLDG_Realm.." - "..GLDG_Player]) then
		GLDG_ByeGuild()
	end
	if (GLDG_Data.GreetChannel[GLDG_Realm.." - "..GLDG_Player]) then
		GLDG_ByeChannel()
	end
end

------------------------------------------------------------
function GLDG_ByeGuild()
	if (GLDG_GuildName=="") then return end

	local list = GLDG_DataGreet.ByeGuild
	-- if time is between 20:00 and 06:00 use night mode
	local hour,min = GLDG_GetTime();
	if ((hour >= 20) or (hour <=5)) then
		list = GLDG_DataGreet.NightGuild;
	end

	local msg = list[math.random(GLDG_TableSize(list))]

	-- Send greeting
	SendChatMessage(string.format(msg, GLDG_GuildAlias), "GUILD")
end

------------------------------------------------------------
function GLDG_ByeChannel()
	local list = GLDG_DataGreet.ByeChannel
	-- if time is between 20:00 and 06:00 use night mode
	local hour,min = GLDG_GetTime();
	if ((hour >= 20) or (hour <=5)) then
		list = GLDG_DataGreet.NightChannel;
	end

	local msg = list[math.random(GLDG_TableSize(GLDG_Data.ByeChannel))]

	-- Send greeting
	local channel = GetChannelName(GLDG_ChannelName)
	if (channel) then
		SendChatMessage(msg, "CHANNEL", nil, channel)
	end
end

------------------------------------------------------------
function GLDG_KeyLater()
	if (GLDG_Data.GreetGuild[GLDG_Realm.." - "..GLDG_Player]) then
		GLDG_LaterGuild()
	end
	if (GLDG_Data.GreetChannel[GLDG_Realm.." - "..GLDG_Player]) then
		GLDG_LaterChannel()
	end
end

------------------------------------------------------------
function GLDG_LaterGuild()
	if (GLDG_GuildName=="") then return end

	local msg = GLDG_DataGreet.LaterGuild[math.random(GLDG_TableSize(GLDG_DataGreet.LaterGuild))]

	-- Send greeting
	SendChatMessage(string.format(msg, GLDG_GuildAlias), "GUILD")
end

------------------------------------------------------------
function GLDG_LaterChannel()
	local msg = GLDG_DataGreet.LaterChannel[math.random(GLDG_TableSize(GLDG_DataGreet.LaterChannel))]

	-- Send greeting
	local channel = GetChannelName(GLDG_ChannelName)
	if (channel) then
		SendChatMessage(msg, "CHANNEL", nil, channel)
	end
end

------------------------
-- _10_ Slash Handler --
------------------------

function GLDG_SlashHandler(msg)
	if not msg then
		getglobal(GLDG_GUI):Show()
	else
		local msgLower = string.lower(msg)
		local words = GLDG_GetWords(msg)
		local wordsLower = GLDG_GetWords(msgLower)
		local size = GLDG_TableSize(wordsLower)

		if (size>0) then
			if (wordsLower[0]=="debug") then
				if (size>1) then
					if (wordsLower[1]=="off") then
						GLDG_Debug = false
					elseif (wordsLower[1]=="on") then
						GLDG_Debug = true
					elseif (wordsLower[1]=="toggle") then
						GLDG_Debug = not GLDG_Debug
					else
						GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_TXT.command.." ["..GLDG_Data.colours.help..msgLower.."|r]")
						GLDG_Help()
					end
				end
				local state = "OFF"
				if GLDG_Debug then state = "ON" end
				DEFAULT_CHAT_FRAME:AddMessage("GUILDGREET DEBUG IS NOW "..state)

			elseif (wordsLower[0]=="force") then
				GLDG_ForceChatlist()
			elseif (wordsLower[0]=="clear") then
				GLDG_ClearList()
			elseif (wordsLower[0]=="guild") then
				if (size>1) then
					if (wordsLower[1]=="show") then
						GLDG_ListAllPlayers(false, true)
					elseif (wordsLower[1]=="all") then
						GLDG_ListAllPlayers(true, true)
					elseif (wordsLower[1]=="list") then
						GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_TXT.pasting);
						GLDG_list = GLDG_ListAllPlayers(true, false);
						GLDG_PasteList.List:Show();
					else
						GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_TXT.command.." ["..GLDG_Data.colours.help..msgLower.."|r]")
						GLDG_Help()
					end
				else
					GLDG_ListAllPlayers(false, true)
				end
			elseif (wordsLower[0]=="friends") then
				if GLDG_Data.UseFriends then
					if (size>1) then
						if (wordsLower[1]=="show") then
							GLDG_ListAllPlayers_f(false, true)
						elseif (wordsLower[1]=="all") then
							GLDG_ListAllPlayers_f(true, true)
						elseif (wordsLower[1]=="list") then
							GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_TXT.pasting);
							GLDG_list = GLDG_ListAllPlayers_f(true, false);
							GLDG_PasteList.List:Show();
						else
							GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_TXT.command.." ["..GLDG_Data.colours.help..msgLower.."|r]")
							GLDG_Help()
						end
					else
						GLDG_ListAllPlayers_f(false, true)
					end
				else
					GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_TXT.disabled)
				end
			elseif (wordsLower[0]=="channel") then
				if GLDG_inChannel then
					if (size>1) then
						if (wordsLower[1]=="show") then
							GLDG_ListAllPlayers_c(false, true)
						elseif (wordsLower[1]=="all") then
							GLDG_ListAllPlayers_c(true, true)
						elseif (wordsLower[1]=="list") then
							GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_TXT.pasting);
							GLDG_list = GLDG_ListAllPlayers_c(true, false);
							GLDG_PasteList.List:Show();
						else
							GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_TXT.command.." ["..GLDG_Data.colours.help..msgLower.."|r]")
							GLDG_Help()
						end
					else
						GLDG_ListAllPlayers_c(false, true)
					end
				else
					GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_TXT.disabled_c)
				end
			elseif (wordsLower[0]=="all") then
				if GLDG_inChannel then
					if (size>1) then
						if (wordsLower[1]=="show") then
							GLDG_ListAllPlayers(false, true)
							GLDG_ListAllPlayers_f(false, true)
							GLDG_ListAllPlayers_c(false, true)
						elseif (wordsLower[1]=="all") then
							GLDG_ListAllPlayers(true, true)
							GLDG_ListAllPlayers_f(true, true)
							GLDG_ListAllPlayers_c(true, true)
						--elseif (wordsLower[1]=="list") then
						--	GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_TXT.pasting);
						--	GLDG_list = GLDG_ListAllPlayers_c(true, false);
						--	GLDG_PasteList.List:Show();
						else
							GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_TXT.command.." ["..GLDG_Data.colours.help..msgLower.."|r]")
							GLDG_Help()
						end
					else
						GLDG_ListAllPlayers(false, true)
						GLDG_ListAllPlayers_f(false, true)
						GLDG_ListAllPlayers_c(false, true)
					end
				else
					GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_TXT.disabled_c)
				end

			elseif (wordsLower[0]=="show") then
				if (size>1) then
					GLDG_ListForPlayer(words[1], false)
				else
					GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_TXT.argmissing.." ["..GLDG_Data.colours.help..wordsLower[0].."|r]")
					GLDG_Help()
				end
			elseif (wordsLower[0]=="full") then
				if (size>1) then
					GLDG_ListForPlayer(words[1], true)
				else
					GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_TXT.argmissing.." ["..GLDG_Data.colours.help..wordsLower[0].."|r]")
					GLDG_Help()
				end

			elseif (wordsLower[0]=="greet") then
				if (size>1) then
					if (wordsLower[1]=="guild") then
						GLDG_GreetGuild()
					elseif (wordsLower[1]=="channel") then
						GLDG_GreetChannel()
					elseif (wordsLower[1]=="all") then
						GLDG_GreetGuild()
						GLDG_GreetChannel()
					else
						GLDG_SendGreet(words[1], true)
					end
				else
					GLDG_GreetGuild()
					GLDG_GreetChannel()
				end
			elseif (wordsLower[0]=="bye") then
				if (size>1) then
					if (wordsLower[1]=="guild") then
						GLDG_ByeGuild()
					elseif (wordsLower[1]=="channel") then
						GLDG_ByeChannel()
					elseif (wordsLower[1]=="all") then
						GLDG_ByeGuild()
						GLDG_ByeChannel()
					else
						GLDG_SendBye(words[1], true)
					end
				else
					GLDG_ByeGuild()
					GLDG_ByeChannel()
				end

			elseif (wordsLower[0]=="later") then
				if (size>1) then
					if (wordsLower[1]=="guild") then
						GLDG_LaterGuild()
					elseif (wordsLower[1]=="channel") then
						GLDG_LaterChannel()
					elseif (wordsLower[1]=="all") then
						GLDG_LaterGuild()
						GLDG_LaterChannel()
					else
						GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_TXT.command.." ["..GLDG_Data.colours.help..msgLower.."|r]")
						GLDG_Help()
					end
				else
					GLDG_LaterGuild()
					GLDG_LaterChannel()
				end

			elseif (wordsLower[0]=="help") then
				GLDG_Help()
			elseif (wordsLower[0]=="about") then
				GLDG_About()
			elseif (wordsLower[0]=="urbin") then
				GLDG_ListUrbinAddonDetails()
			elseif (wordsLower[0]=="test") then
				GLDG_Test()
			elseif (wordsLower[0]=="config") then
				getglobal(GLDG_GUI):Show()

			else
				GLDG_ListForPlayer(words[0], false)
			end
		else
			getglobal(GLDG_GUI):Show()
		end
	end
end


----------------------
-- _11_ Tab Changer --
----------------------

function GLDG_ClickTab(tabName)
	-- Actions to perform when a tab is clicked
	PanelTemplates_Tab_OnClick(getglobal(GLDG_GUI))
	-- Show frame linked to tab, hide all others
	local tabNum = 1
	while getglobal(GLDG_GUI.."Tab"..tabNum) do
		local frame = nil
		if GLDG_Tab2Frame["Tab"..tabNum] then frame = getglobal (GLDG_GUI..GLDG_Tab2Frame["Tab"..tabNum]) end
		if frame then if (tabName == GLDG_GUI.."Tab"..tabNum) then frame:Show() else frame:Hide() end end
		tabNum = tabNum + 1
	end
end

function GLDG_ClickSubTab(tabName)
	-- Actions to perform when a tab is clicked
	PanelTemplates_Tab_OnClick(getglobal(GLDG_GUI.."Settings"))
	-- Show frame linked to tab, hide all others
	local tabNum = 1
	while getglobal(GLDG_GUI.."SettingsTab"..tabNum) do
		local frame = nil
		if GLDG_SubTab2Frame["Tab"..tabNum] then frame = getglobal (GLDG_GUI.."Settings"..GLDG_SubTab2Frame["Tab"..tabNum]) end
		if frame then if (tabName == GLDG_GUI.."SettingsTab"..tabNum) then frame:Show() else frame:Hide() end end
		tabNum = tabNum + 1
	end
end

-----------------------------
-- _12_ General Tab Update --
-----------------------------

------------------------------------------------------------
function GLDG_UpdateRelog()
	-- Store the new value
	GLDG_Data.RelogTime = this:GetValue()
	-- Update display
	local text = getglobal(this:GetParent():GetName().."RelogText")
	if (GLDG_Data.RelogTime == 0) then text:SetText(GLDG_TXT.relogalways)
	else text:SetText(string.format(GLDG_TXT.reloglimit, GLDG_Data.RelogTime)) end
end

------------------------------------------------------------
function GLDG_UpdateMinLevelUp()
	-- Store the new value
	GLDG_Data.MinLevelUp = this:GetValue()
	-- Update display
	local text = getglobal(this:GetParent():GetName().."MinLevelUpText")
	if (GLDG_Data.MinLevelUp == 0) then text:SetText(GLDG_TXT.minlevelupalways)
	else text:SetText(string.format(GLDG_TXT.minlevelup, GLDG_Data.MinLevelUp)) end
end

------------------------------------------------------------
function GLDG_UpdateUpdateTime()
	-- Store the new value
	GLDG_Data.UpdateTime = this:GetValue() * 10
	-- Update display
	local text = getglobal(this:GetParent():GetName().."UpdateTimeText")
	if (GLDG_Data.UpdateTime == 0) then
		text:SetText(GLDG_TXT.eventonly)
	else
		text:SetText(string.format(GLDG_TXT.updatetime, GLDG_Data.UpdateTime))
	end

	if GLDG_InitialGuildUpdate then
		if (GLDG_Data.UpdateTime >= 0) then
			GLDG_UpdateRequest = GetTime() + GLDG_Data.UpdateTime
		else
			GLDG_UpdateRequest = nil
		end
	end
	if GLDG_InitialFriendsUpdate then
		if (GLDG_Data.UpdateTime >= 0) then
			GLDG_UpdateRequestFriends = GetTime() + GLDG_Data.UpdateTime
		else
			GLDG_UpdateRequestFriends = nil
		end
	end
end

------------------------------------------------------------
function GLDG_UpdateListsize()
	-- Store the new value
	GLDG_Data.ListSize = this:GetValue()
	-- Update display
	local text = getglobal(this:GetParent():GetName().."ListsizeText")
	text:SetText(string.format(GLDG_TXT.listsize, GLDG_Data.ListSize))
	-- Update queue
	GLDG_ShowQueue()
end

------------------------------------------------------------
function GLDG_UpdateChatFrame()
	if not GLDG_updatingChatFrame then
		GLDG_updatingChatFrame = true

		-- Store the new value
		GLDG_Data.ChatFrame = this:GetValue()

		--if (GLDG_Data.ChatFrame > NUM_CHAT_WINDOWS) then
		--	local oldValue = GLDG_Data.ChatFrame
		--	GLDG_Data.ChatFrame = NUM_CHAT_WINDOWS
		--	GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r Chat frame "..oldValue.." does not exist, using chat frame "..GLDG_Data.ChatFrame.." instead")
		--	this:SetValue(GLDG_Data.ChatFrame)
		--end

		-- Update display
		local text = getglobal(this:GetParent():GetName().."ChatFrameText")
		if (GLDG_Data.ChatFrame == 0) then
			text:SetText(GLDG_TXT.defaultChatFrame)
			GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r Now using default chat frame")
		else
			local name, fontSize, r, g, b, alpha, shown, locked, docked = GetChatWindowInfo(GLDG_Data.ChatFrame)
			text:SetText(string.format(GLDG_TXT.chatFrame, GLDG_Data.ChatFrame, name))
			GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r Now using chat frame "..GLDG_Data.ChatFrame.." ("..name..")")
		end

		GLDG_updatingChatFrame = nil
	end
end

------------------------------------------------------------
function GLDG_DropDownTemplate_OnShow()
	UIDropDownMenu_Initialize(this, GLDG_DropDown_Initialize);
	UIDropDownMenu_SetWidth(150);
end

------------------------------------------------------------
function GLDG_DropDown_Initialize()

	GLDG_dropDownData = {}

	local info
	info = UIDropDownMenu_CreateInfo();
	info.func = GLDG_DropDown_OnClick
	local j = 1

	for i = 0, 10 do
		local id, name = GetChannelName(i)
		if (i==0 or id>0 and name) then
			if (i==0) then
				name = GLDG_TXT.none
			end

			info.checked = nil
			info.text = name
			UIDropDownMenu_AddButton(info);
			GLDG_dropDownData[j] = name
			if (name == GLDG_ChannelName) or
			   (name == GLDG_TXT.none and GLDG_ChannelName=="") then
				UIDropDownMenu_SetSelectedID(getglobal(GLDG_GUI.."SettingsGeneral".."ChannelNameDropboxButton"), j);
			end
			j = j + 1
		end
	end
end

------------------------------------------------------------
function GLDG_DropDown_OnClick(list)

	if not GLDG_dropDownData then GLDG_dropDownData = {} end
	local i = this:GetID();
	local name = GLDG_dropDownData[i]
	if not name then name = "" end
	if name == GLDG_TXT.none then name = "" end

	local oldChannelName = GLDG_ChannelName
	GLDG_ChannelName = name;
	if (GLDG_ChannelName == "") then
		GLDG_inChannel = false
		GLDG_UpdateTabs()
	else
		if GLDG_CheckChannel() then
			UIDropDownMenu_SetSelectedID(getglobal(GLDG_GUI.."SettingsGeneral".."ChannelNameDropboxButton"), i);
		else
			GLDG_ChannelName = oldChannelName
			if GLDG_CheckChannel() then
				GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r Channel ["..GLDG_ChannelName.."] does not exist, setting back to ["..oldChannelName.."]")
			else
				GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r Channel ["..GLDG_ChannelName.."] does not exist, nor does the previous channel ["..oldChannelName.."], clearing channel name")
				UIDropDownMenu_SetSelectedID(getglobal(GLDG_GUI.."SettingsGeneral".."ChannelNameDropboxButton"), 1);
				GLDG_ChannelName = ""
				GLDG_inChannel = false
				GLDG_UpdateTabs()
			end
		end
	end

	-- store new channel name
	if not GLDG_Data.ChannelNames[GLDG_Realm.." - "..GLDG_Player] then
		GLDG_Data.ChannelNames[GLDG_Realm.." - "..GLDG_Player] = ""
	end
	-- set channel name pointer
	GLDG_Data.ChannelNames[GLDG_Realm.." - "..GLDG_Player] = GLDG_ChannelName

	-- update channel section pointer
	if (GLDG_ChannelName~="") then
		if not GLDG_Data[GLDG_Realm.." - channel - "..GLDG_ChannelName] then
			GLDG_Data[GLDG_Realm.." - channel - "..GLDG_ChannelName] = {}
		end
		GLDG_DataChannel = GLDG_Data[GLDG_Realm.." - channel - "..GLDG_ChannelName]
	else
		GLDG_DataChannel = {}
	end
end

------------------------------------------------------------
function GLDG_SupressAll()
	GLDG_Data.SupressGreet = 1
	GLDG_Data.SupressJoin = 1
	GLDG_Data.SupressLevel = 1
	GLDG_Data.SupressRank = 1

	GLDG_UpdateSupressed()
end

------------------------------------------------------------
function GLDG_SupressNone()
	GLDG_Data.SupressGreet = nil
	GLDG_Data.SupressJoin = nil
	GLDG_Data.SupressLevel = nil
	GLDG_Data.SupressRank = nil

	GLDG_UpdateSupressed()
end

------------------------------------------------------------
function GLDG_UpdateSupressed()
	local name = GLDG_GUI.."SettingsGreeting";
	getglobal(name.."SupressGreetBox"):SetChecked(GLDG_Data.SupressGreet)
	getglobal(name.."SupressJoinBox"):SetChecked(GLDG_Data.SupressJoin)
	getglobal(name.."SupressLevelBox"):SetChecked(GLDG_Data.SupressLevel)
	getglobal(name.."SupressRankBox"):SetChecked(GLDG_Data.SupressRank)
end

-------------------------------
-- _13_ Greetings Tab Update --
-------------------------------

function GLDG_ShowCollections(frame)
	-- Set frame
	if not frame then frame = this:GetParent():GetName() end
	-- Update scrollbar
	local bar = getglobal(frame.."Collectbar")
	FauxScrollFrame_Update(bar, GLDG_TableSize(GLDG_Data.Collections), GLDG_NumColRows, 15)
	local offset = FauxScrollFrame_GetOffset(bar)
	-- Sort the list of existing custom collections
	local sorted = {}
	local total = 0
	for col in pairs(GLDG_Data.Collections) do
		-- Look for the position in the list we need
		local loc = 1
		while sorted[loc] and (sorted[loc] < col) do loc = loc + 1 end
		-- We found the position: move everything beyond it
		for cnt = total, loc, -1 do sorted[cnt+1] = sorted[cnt] end
		-- Insert the collection
		total = total + 1
		sorted[loc] = col end
	-- Show the collections visible
	for i = 1, GLDG_NumColRows do
		getglobal(frame.."Collect"..i):UnlockHighlight()
		if sorted[i + offset] then
			getglobal(frame.."Collect"..i.."Text"):SetText(sorted[i + offset])
			getglobal(frame.."Collect"..i):Enable()
			if GLDG_SelColName and (GLDG_SelColName == sorted[i + offset]) then getglobal(frame.."Collect"..i):LockHighlight() end
		else	getglobal(frame.."Collect"..i.."Text"):SetText("")
			getglobal(frame.."Collect"..i):Disable() end end
	-- Set buttons and text depending the selected collection
	local colhead = GLDG_TXT.colglobal
	if GLDG_SelColName then
		getglobal(frame.."ColNewDel"):SetText(GLDG_TXT.cbdel)
		colhead = string.format(GLDG_TXT.colcustom, GLDG_SelColName)
	else 	getglobal(frame.."ColNewDel"):SetText(GLDG_TXT.cbnew) end
	getglobal(frame.."GreetHeader"):SetText(string.format(GLDG_TXT.msgheader, colhead))
end

------------------------------------------------------------
function GLDG_ShowGreetings(frame)
	-- Show messages from selected category
	local list = GLDG_Data[GLDG_Selection]
	if GLDG_SelColName then
		list = GLDG_Data.Collections[GLDG_SelColName][GLDG_Selection]
	end
	local cnt = 1
	local line = frame.."Line"
	while list[cnt + GLDG_GreetOffset] and getglobal(line..cnt) do
		getglobal(line..cnt.."Text"):SetText(list[cnt + GLDG_GreetOffset])
		getglobal(line..cnt):Enable()
		cnt = cnt + 1 end
	while getglobal(line..cnt) do
		getglobal(line..cnt.."Text"):SetText("")
		getglobal(line..cnt):Disable()
		cnt = cnt + 1 end
	-- Set highlight
	for cnt = 1, GLDG_NumSelRows do
		if GLDG_SelMsgNum and (GLDG_SelMsgNum == GLDG_GreetOffset + cnt) then getglobal(line..cnt):LockHighlight()
		else getglobal(line..cnt):UnlockHighlight() end end
	-- Set editbox
	if GLDG_SelMsgNum then getglobal(frame.."Editbox"):SetText(list[GLDG_SelMsgNum])
	else getglobal(frame.."Editbox"):SetText("") end
	-- Set editbox buttons
	if GLDG_SelMsgNum then
		getglobal(frame.."MsgAdd"):SetText(GLDG_TXT.update)
		getglobal(frame.."MsgDel"):SetText(GLDG_TXT.delete)
	else	getglobal(frame.."MsgAdd"):SetText(GLDG_TXT.add)
		getglobal(frame.."MsgDel"):SetText(GLDG_TXT.clear) end
end

------------------------------------------------------------
function GLDG_ShowCustom(frame)
	-- Display the current values
	local d = GLDG_Data.Custom[GLDG_Realm]
	if d and (d ~= "") and not GLDG_Data.Collections[d] then
		-- Collection no longer exists
		GLDG_Data.Custom[GLDG_Realm] = nil
		d = nil end
	local f = getglobal(frame.."SubCustomRealm")
	if not d then f:SetText(GLDG_TXT.nodef)
	elseif (d == "") then f:SetText(GLDG_TXT.colglobal)
	else f:SetText(d) end
	d = GLDG_Data.Custom[GLDG_Realm.."-"..GLDG_GuildName]
	if d and (d ~= "") and not GLDG_Data.Collections[d] then
		-- Collection no longer exists
		GLDG_Data.Custom[GLDG_Realm.."-"..GLDG_GuildName] = nil
		d = nil end
	f = getglobal(frame.."SubCustomGuild")
	if not d then f:SetText(GLDG_TXT.nodef)
	elseif (d == "") then f:SetText(GLDG_TXT.colglobal)
	else f:SetText(d) end
	d = GLDG_Data.Custom[GLDG_Realm.."-"..UnitName("player")]
	if d and (d ~= "") and not GLDG_Data.Collections[d] then
		-- Collection no longer exists
		GLDG_Data.Custom[GLDG_Realm.."-"..UnitName("player")] = nil
		d = nil end
	f = getglobal(frame.."SubCustomChar")
	if not d then f:SetText(GLDG_TXT.nodef)
	elseif (d == "") then f:SetText(GLDG_TXT.colglobal)
	else f:SetText(d) end
	-- Set greetings section pointer at is possibly changed
	if GLDG_InitGreet(GLDG_Realm.."-"..UnitName("player")) and
	   GLDG_InitGreet(GLDG_Realm.."-"..GLDG_GuildName) and
	   GLDG_InitGreet(GLDG_Realm) then
		-- No custom collections are used
		GLDG_DataGreet = GLDG_Data end
	-- Show the frame
	GLDG_ColButtonPressed(frame)
	getglobal(frame.."SubCustom"):Show()
end

------------------------------------------------------------
function GLDG_ClickCollection(col)
	-- Set new selected collection
	if GLDG_SelColName and (GLDG_SelColName == col) then GLDG_SelColName = nil
	else GLDG_SelColName = col end
	-- Update change selection button
	local button = getglobal(this:GetParent():GetName().."SubChangeSelection")
	if GLDG_SelColName then button:Enable() else button:Disable() end
	-- Refresh display
	GLDG_ShowCollections()
	GLDG_ShowGreetings(this:GetParent():GetName())
end

------------------------------------------------------------
function GLDG_ColButtonPressed(frame)
	-- Enable all collections buttons
	getglobal(frame.."ColNewDel"):Enable()
	getglobal(frame.."ColRealm"):Enable()
	getglobal(frame.."ColGuild"):Enable()
	getglobal(frame.."ColPlayer"):Enable()
	-- Hide all subframes
	getglobal(frame.."SubCustom"):Hide()
	getglobal(frame.."SubNew"):Hide()
	getglobal(frame.."SubChange"):Hide()
end

------------------------------------------------------------
function GLDG_ClickNewCol(frame)
	-- If collection is selected, remove it
	if GLDG_SelColName then
		GLDG_Data.Collections[GLDG_SelColName] = nil
		GLDG_SelColName = nil
		GLDG_ShowCustom(frame)
		GLDG_ShowCollections(frame)
		return end
	-- Disable this button and show the subframe
	GLDG_ColButtonPressed(frame)
	this:Disable()
	getglobal(frame.."SubNew"):Show()
end

------------------------------------------------------------
function GLDG_ClickChangeCustom(setting, frame)
	-- Set the value to be updated
	GLDG_ChangeName = GLDG_Realm
	if (setting == "guild") then
		getglobal(frame.."SubChangeHeader"):SetText(GLDG_TXT.cbguild)
		GLDG_ChangeName = GLDG_ChangeName.."-"..GLDG_GuildName
	elseif (setting == "char") then
		getglobal(frame.."SubChangeHeader"):SetText(GLDG_TXT.cbplayer)
		GLDG_ChangeName = GLDG_ChangeName.."-"..UnitName("player")
	else	getglobal(frame.."SubChangeHeader"):SetText(GLDG_TXT.cbrealm) end
	-- Disable this button and show the subframe
	GLDG_ColButtonPressed(frame)
	this:Disable()
	getglobal(frame.."SubChange"):Show()
end

------------------------------------------------------------
function GLDG_ClickNewColAdd(frame)
	-- Don't do anything if no name is given or name already exists
	local box = getglobal(this:GetParent():GetName().."Editbox")
	local name = box:GetText()
	if (name == "") or GLDG_Data.Collections[name] then return end
	box:SetText("")
	-- Create the new collection
	GLDG_Data.Collections[name] = {}
	local n = GLDG_Data.Collections[name]
	n.Greet = {}
	n.GreetBack = {}
	n.Welcome = {}
	n.NewRank = {}
	n.NewLevel = {}
	n.Bye = {}
	n.Night = {}
	n.Guild = {}
	n.Channel = {}
	n.ByeGuild = {}
	n.NightGuild = {}
	n.ByeChannel = {}
	n.NightChannel = {}
	n.LaterGuild = {}
	n.LaterChannel = {}
	-- Refresh list and hide new collection frame
	GLDG_ShowCustom(frame)
	GLDG_ShowCollections(frame)
end

------------------------------------------------------------
function GLDG_ChangeCollect(setting, frame)
	-- Set new setting
	if (setting == "clear") then GLDG_Data.Custom[GLDG_ChangeName] = nil
	elseif (setting == "global") then GLDG_Data.Custom[GLDG_ChangeName] = ""
	elseif (setting == "selection") then GLDG_Data.Custom[GLDG_ChangeName] = GLDG_SelColName end
	-- Hide frame
	GLDG_ShowCustom(frame)
end

------------------------------------------------------------
function GLDG_ClickGreetButton(id)
	-- Set frame
	local frame = this:GetName()
	-- Change selection if id is given
	if id then
		frame = this:GetParent():GetName()
		GLDG_SelMsgNum = nil
		getglobal(frame.."Editbox"):SetText("")
		local map = {}
		map[1] = "Greet"
		map[2] = "GreetBack"
		map[3] = "Welcome"
		map[4] = "NewRank"
		map[5] = "NewLevel"
		map[6] = "Bye"
		map[7] = "Night"
		map[8] = "Guild"
		map[9] = "Channel"
		map[10] = "ByeGuild"
		map[11] = "NightGuild"
		map[12] = "ByeChannel"
		map[13] = "NightChannel"
		map[14] = "LaterGuild"
		map[15] = "LaterChannel"
		GLDG_Selection = map[id]
	end
	-- Enable all buttons except active
	if (GLDG_Selection == "Greet") then getglobal(frame.."SelDefault"):Disable() else getglobal(frame.."SelDefault"):Enable() end
	if (GLDG_Selection == "GreetBack") then getglobal(frame.."SelRelog"):Disable() else getglobal(frame.."SelRelog"):Enable() end
	if (GLDG_Selection == "Welcome") then getglobal(frame.."SelJoin"):Disable() else getglobal(frame.."SelJoin"):Enable() end
	if (GLDG_Selection == "NewRank") then getglobal(frame.."SelRank"):Disable() else getglobal(frame.."SelRank"):Enable() end
	if (GLDG_Selection == "NewLevel") then getglobal(frame.."SelLevel"):Disable() else getglobal(frame.."SelLevel"):Enable() end
	if (GLDG_Selection == "Bye") then getglobal(frame.."SelBye"):Disable() else getglobal(frame.."SelBye"):Enable() end
	if (GLDG_Selection == "Night") then getglobal(frame.."SelNight"):Disable() else getglobal(frame.."SelNight"):Enable() end
	if (GLDG_Selection == "Guild") then getglobal(frame.."SelGuild"):Disable() else getglobal(frame.."SelGuild"):Enable() end
	if (GLDG_Selection == "Channel") then getglobal(frame.."SelChannel"):Disable() else getglobal(frame.."SelChannel"):Enable() end
	if (GLDG_Selection == "ByeGuild") then getglobal(frame.."SelByeGuild"):Disable() else getglobal(frame.."SelByeGuild"):Enable() end
	if (GLDG_Selection == "NightGuild") then getglobal(frame.."SelNightGuild"):Disable() else getglobal(frame.."SelNightGuild"):Enable() end
	if (GLDG_Selection == "ByeChannel") then getglobal(frame.."SelByeChannel"):Disable() else getglobal(frame.."SelByeChannel"):Enable() end
	if (GLDG_Selection == "NightChannel") then getglobal(frame.."SelNightChannel"):Disable() else getglobal(frame.."SelNightChannel"):Enable() end
	if (GLDG_Selection == "LaterGuild") then getglobal(frame.."SelLaterGuild"):Disable() else getglobal(frame.."SelLaterGuild"):Enable() end
	if (GLDG_Selection == "LaterChannel") then getglobal(frame.."SelLaterChannel"):Disable() else getglobal(frame.."SelLaterChannel"):Enable() end
	-- Update editbox header
	getglobal(frame.."EditboxHeader"):SetText(GLDG_TXT[GLDG_Selection])
	-- Refresh scrollbar
	GLDG_ClickGreetScrollBar(getglobal(frame.."Scrollbar"))
end

------------------------------------------------------------
function GLDG_ClickGreetScrollBar(frame)
	if not frame then
		frame = this
	end
	GLDG_GreetOffset = FauxScrollFrame_GetOffset(frame)
	local list = GLDG_Data[GLDG_Selection]
	if GLDG_SelColName then
		list = GLDG_Data.Collections[GLDG_SelColName][GLDG_Selection]
	end
	FauxScrollFrame_Update(frame, GLDG_TableSize(list), GLDG_NumSelRows, 15)
	GLDG_ShowGreetings(frame:GetParent():GetName())
end

------------------------------------------------------------
function GLDG_ClickGreeting(id)
	-- Set new selected message
	if GLDG_SelMsgNum and (GLDG_SelMsgNum == GLDG_GreetOffset + id) then GLDG_SelMsgNum = nil
	else GLDG_SelMsgNum = GLDG_GreetOffset + id end
	-- Refresh greetings list
	GLDG_ShowGreetings(this:GetParent():GetName())
end

------------------------------------------------------------
function GLDG_ClickGreetAdd(frame)
	-- Clear focus
	getglobal(frame.."Editbox"):ClearFocus()
	-- Refuse if no input in editbox
	local text = getglobal(frame.."Editbox"):GetText()
	if (text == "") then return end
	-- Update message if one is selected
	local list = GLDG_Data[GLDG_Selection]
	if GLDG_SelColName then list = GLDG_Data.Collections[GLDG_SelColName][GLDG_Selection] end
	if GLDG_SelMsgNum then list[GLDG_SelMsgNum] = text
	-- Add message if none is selected
	else 	getglobal(frame.."Editbox"):SetText("")
		list[GLDG_TableSize(list) + 1] = text end
	-- Update display
	GLDG_ClickGreetScrollBar(getglobal(frame.."Scrollbar"))
end

------------------------------------------------------------
function GLDG_ClickGreetRemove(frame)
	-- Clear focus
	getglobal(frame.."Editbox"):ClearFocus()
	-- Remove message if one is selected
	local list = GLDG_Data[GLDG_Selection]
	if GLDG_SelColName then list = GLDG_Data.Collections[GLDG_SelColName][GLDG_Selection] end
	if GLDG_SelMsgNum then
		local cnt = GLDG_SelMsgNum
		while list[cnt + 1] do
			list[cnt] = list[cnt + 1]
			cnt = cnt + 1 end
		GLDG_SelMsgNum = nil
		list[cnt] = nil end
	-- Update display
	GLDG_ClickGreetScrollBar(getglobal(frame.."Scrollbar"))
end


-----------------------------
-- _14_ Players Tab Update --
-----------------------------

function GLDG_SortString(player)
	-- Helper function: returns string that should be used for sorting
	local result = player
	if (GLDG_DataGuild[player].alt and GLDG_Data.GroupAlt) then result = GLDG_DataGuild[player].alt..player end
	return string.lower(result)
end

------------------------------------------------------------
function GLDG_ListPlayers()
	-- Abort if addon is not initialized
	if not GLDG_DataGuild then return end
	-- Check if there is a reason to clear the selected player
	if GLDG_SelPlrName then
		if (not GLDG_DataGuild[GLDG_SelPlrName]) or
		   (GLDG_DataGuild[GLDG_SelPlrName].ignore and not GLDG_Data.ShowIgnore) then
			GLDG_SelPlrName = nil end end
	-- Prepare for list creation
	GLDG_SortedList = {}
	local total = 0
	GLDG_NumMain = 0
	GLDG_NumAlts = 0
	local s = {}
	if GLDG_SelPlrName then s = GLDG_DataGuild[GLDG_SelPlrName] end
	-- Create the list of players to display
	for player in pairs(GLDG_DataGuild) do
		local p = GLDG_DataGuild[player]
		if ((not p.own) or GLDG_Data.IncludeOwn) then
			-- Update counters for main and alt
			if p.main then GLDG_NumMain = GLDG_NumMain + 1 end
			if p.alt and (p.alt == GLDG_SelPlrName) then GLDG_NumAlts = GLDG_NumAlts + 1 end
			-- See if we are supposed to add this player in our list
			local show = GLDG_Data.ShowIgnore or not p.ignore
			show = show and ((p.alt == GLDG_SelPlrName) or (p.alt == s.alt) or not p.alt or GLDG_Data.ShowAlt)
			show = show and (((not p.alt) and (not p.main)) or not GLDG_Data.FilterUnassigned)
			if show then
				-- Look for the position in the list we need
				local loc = 1
				while GLDG_SortedList[loc] and (GLDG_SortString(player) > GLDG_SortString(GLDG_SortedList[loc])) do loc = loc + 1 end
				-- We found our position: move everything beyond it
				for cnt = total, loc, -1 do GLDG_SortedList[cnt + 1] = GLDG_SortedList[cnt] end
				-- Insert our player
				total = total + 1
				GLDG_SortedList[loc] = player end end
		end
	-- Update the scrollbar
	GLDG_ClickPlayerBar()
end

------------------------------------------------------------
function GLDG_ShowPlayers()
	-- Display the players visible in the frame
	local cnt = 1
	local line = GLDG_GUI.."PlayersLine"
	while GLDG_SortedList[cnt + GLDG_PlayerOffset] and getglobal(line..cnt) do
		local text = getglobal(line..cnt.."Text")
		local text2 = getglobal(line..cnt.."Text2")
		local p = GLDG_DataGuild[GLDG_SortedList[cnt + GLDG_PlayerOffset]]
		if p.alias then text:SetText(GLDG_SortedList[cnt + GLDG_PlayerOffset].."*")
		else text:SetText(GLDG_SortedList[cnt + GLDG_PlayerOffset]) end
		text2:SetText("")
		if p.ignore then
			text:SetTextColor(1, 0.25, 0.25)
			text2:SetText(GLDG_TXT.markIGN)
		elseif p.main then
			text:SetTextColor(0.25, 1, 0.25)
			text2:SetText(GLDG_TXT.markMAIN)
		elseif p.alt then
			text:SetTextColor(0.25, 0.25, 1)
			text2:SetText(GLDG_TXT.markALT)
		elseif p.alias then
			text:SetTextColor(0.68, 0.8, 1)
		else text:SetTextColor(1, 1, 1) end
		text2:SetTextColor(text:GetTextColor())
		getglobal(line..cnt):Enable()
		cnt = cnt +1 end
	-- Disable any rows left unused
	for cnt2 = cnt, GLDG_NumPlrRows do
		getglobal(line..cnt2.."Text"):SetText("")
		getglobal(line..cnt2.."Text2"):SetText("")
		getglobal(line..cnt2):Disable() end
	-- Set/clear highlight
	for cnt = 1, GLDG_NumPlrRows do
		if GLDG_SelPlrName and (GLDG_SortedList[cnt + GLDG_PlayerOffset] == GLDG_SelPlrName) then getglobal(line..cnt):LockHighlight()
		else getglobal(line..cnt):UnlockHighlight() end end
	-- Update buttons
	GLDG_ShowPlayerButtons()
end

------------------------------------------------------------
function GLDG_ClickPlayer(playerName)
	-- Prepare for scrollbar adjustment if needed
	local p = {}
	if GLDG_SelPlrName then p = GLDG_DataGuild[GLDG_SelPlrName] end
	local old = nil
	if (not GLDG_Data.ShowAlt) and (p.main or p.alt) then old = GLDG_GetPlayerOffset(playerName) end
	-- Set new selected player
	if (playerName == GLDG_SelPlrName) then GLDG_SelPlrName = nil
	else GLDG_SelPlrName = playerName end
	-- Refresh list
	GLDG_ListPlayers()
	-- Adjust scrollbar if needed
	GLDG_CorrectPlayerOffset(old, playerName)
end

------------------------------------------------------------
function GLDG_ClickPlayerBar(frame)
	-- Update offset
	if not frame then frame = getglobal(GLDG_GUI.."PlayersPlayerbar") end
	FauxScrollFrame_Update(frame, GLDG_TableSize(GLDG_SortedList), GLDG_NumPlrRows, 15)
	GLDG_PlayerOffset = FauxScrollFrame_GetOffset(frame)
	GLDG_ShowPlayers()
end

------------------------------------------------------------
function GLDG_ShowPlayerButtons()
	-- Set frame
	local frame = GLDG_GUI.."Players"
	-- Hide subframes
	getglobal(frame.."SubAlias"):Hide()
	getglobal(frame.."SubMainAlt"):Hide()
	-- If no player is selected: hide all buttons
	if not GLDG_SelPlrName then
		getglobal(frame.."Ignore"):Hide()
		getglobal(frame.."Alias"):Hide()
		getglobal(frame.."Main"):Hide()
		getglobal(frame.."Alt"):Hide()
		return end
	-- Set selected player
	local p = GLDG_DataGuild[GLDG_SelPlrName]
	-- Ignore button
	local button = getglobal(frame.."Ignore")
	if p.ignore then button:SetText(GLDG_TXT.pbrign) else button:SetText(GLDG_TXT.pbsign) end
	button:Show()
	-- Alias button
	button = getglobal(frame.."Alias")
	button:Show()
	button:Enable()
	-- Main button
	button = getglobal(frame.."Main")
	button:Show()
	if p.alt then button:SetText(GLDG_TXT.pbpmain)
	elseif not p.main then button:SetText(GLDG_TXT.pbsmain)
	elseif (GLDG_NumAlts == 0) then button:SetText(GLDG_TXT.pbrmain)
	else button:Hide() end
	-- Alt button
	button = getglobal(frame.."Alt")
	button:Show()
	button:Enable()
	if p.alt then button:SetText(GLDG_TXT.pbralt)
	elseif p.main then button:Hide()
	else button:SetText(GLDG_TXT.pbsalt) end
end

------------------------------------------------------------
function GLDG_ClickIgnore()
	-- Toggle ignore state for selected player
	local p = GLDG_DataGuild[GLDG_SelPlrName]
	local main = p.alt
	if p.main then main = GLDG_SelPlrName end
	local newval = nil
	if not p.ignore then newval = true end
	-- If main or alt, toggle ignore for all characters of this player
	if main then for q in pairs(GLDG_DataGuild) do if (q == main) or (GLDG_DataGuild[q].alt == main) then GLDG_DataGuild[q].ignore = newval end end
	-- Otherwise, simply toggle ignore for this one character
	else p.ignore = newval end
	-- Show updated list
	GLDG_ListPlayers()
end

------------------------------------------------------------
function GLDG_ClickAlias()
	-- Activate alias subframe
	GLDG_ShowPlayerButtons()
	this:Disable()
	getglobal(this:GetParent():GetName().."SubAlias"):Show()
end

------------------------------------------------------------
function GLDG_ClickMain()
	-- Toggle main status for selected character or promote an alt
	local p = GLDG_DataGuild[GLDG_SelPlrName]
	if p.main then p.main = nil
	elseif p.alt then
		local oldmain = p.alt
		p.alt = nil
		p.main = true
		for q in pairs(GLDG_DataGuild) do
			if (q == oldmain) then
				GLDG_DataGuild[q].main = nil
				GLDG_DataGuild[q].alt = GLDG_SelPlrName
			elseif (GLDG_DataGuild[q].alt == oldmain) then
				GLDG_DataGuild[q].alt = GLDG_SelPlrName end end
	else p.main = true end
	-- Show updated list
	GLDG_ListPlayers()
end

------------------------------------------------------------
function GLDG_ClickAlt()
	-- Prepare for scrollbar adjustment if needed
	local p = {}
	if GLDG_SelPlrName then p = GLDG_DataGuild[GLDG_SelPlrName] end
	local old = nil
	if GLDG_Data.GroupAlt then old = GLDG_GetPlayerOffset(GLDG_SelPlrName) end
	-- Toggle alt status for selected character
	local p = GLDG_DataGuild[GLDG_SelPlrName]
	if p.alt then p.alt = nil
	elseif (GLDG_NumMain == 1) then
		for q in pairs(GLDG_DataGuild) do
			if GLDG_DataGuild[q].main then
				p.alt = q
				p.ignore = GLDG_DataGuild[q].ignore
				break end end
	else	GLDG_ShowPlayerButtons()
		this:Disable()
		getglobal(this:GetParent():GetName().."SubMainAlt"):Show()
		return end
	-- Show updated list
	GLDG_ListPlayers()
	-- Adjust scrollbar if needed
	GLDG_CorrectPlayerOffset(old, GLDG_SelPlrName)
end

------------------------------------------------------------
function GLDG_ShowPlayerAlias(frame)
	-- Set title
	getglobal(frame.."Header"):SetText(string.format(GLDG_TXT.aliashead, GLDG_SelPlrName))
	-- Set editbox and buttons text
	local p = GLDG_DataGuild[GLDG_SelPlrName]
	if p.alias then
		getglobal(frame.."Set"):SetText(GLDG_TXT.update)
		getglobal(frame.."Del"):SetText(GLDG_TXT.delete)
		getglobal(frame.."Editbox"):SetText(p.alias)
	else	getglobal(frame.."Set"):SetText(GLDG_TXT.set)
		getglobal(frame.."Del"):SetText(GLDG_TXT.cancel)
		getglobal(frame.."Editbox"):SetText("") end
end

------------------------------------------------------------
function GLDG_ClickAliasSet(alias)
	if (alias == "") then return end
	GLDG_DataGuild[GLDG_SelPlrName].alias = alias
	GLDG_ListPlayers()
end

------------------------------------------------------------
function GLDG_ClickAliasRemove()
	GLDG_DataGuild[GLDG_SelPlrName].alias = nil
	GLDG_ListPlayers()
end

------------------------------------------------------------
function GLDG_ShowMainAlt(frame)
	-- Set frame and linename
	if not frame then frame = this end
	local name = frame:GetParent():GetName()
	-- Set title
	getglobal(name.."Header"):SetText(string.format(GLDG_TXT.mainhead, GLDG_SelPlrName))
	-- Create a sorted list of all mains
	local mainlist = {}
	local total = 0
	for p in pairs(GLDG_DataGuild) do
		if GLDG_DataGuild[p].main then
			local loc = 1
			while mainlist[loc] and (mainlist[loc] < p) do loc = loc + 1 end
			for cnt = total, loc, -1 do mainlist[cnt + 1] = mainlist[cnt] end
			total = total + 1
			mainlist[loc] = p end end
	-- Configure scrollbar
	FauxScrollFrame_Update(frame, GLDG_TableSize(mainlist), GLDG_NumSubRows, 15)
	local offset = FauxScrollFrame_GetOffset(frame)
	-- Set all rows
	name = name.."Line"
	for line = 1, GLDG_NumSubRows do
		if mainlist[line + offset] then
			getglobal(name..line.."Text"):SetText(mainlist[line + offset])
			getglobal(name..line):Enable()
		else	getglobal(name..line.."Text"):SetText("")
			getglobal(name..line):Disable() end end
end

------------------------------------------------------------
function GLDG_ClickMainAlt(player)
	-- Mark position of current selected player
	local old = GLDG_GetPlayerOffset(GLDG_SelPlrName)
	-- Make this the main for the currently selected character
	GLDG_DataGuild[GLDG_SelPlrName].alt = player
	GLDG_DataGuild[GLDG_SelPlrName].ignore = GLDG_DataGuild[player].ignore
	-- Hide the subframe window
	this:GetParent():Hide()
	-- Refresh the playerlist
	GLDG_ListPlayers()
	-- Adjust scrollbar if needed
	GLDG_CorrectPlayerOffset(old, GLDG_SelPlrName)
end

------------------------------------------------------------
function GLDG_GetPlayerOffset(playerName)
	-- Find current position number on list
	local old = nil
	for i = 1, GLDG_TableSize(GLDG_SortedList) do
		if (GLDG_SortedList[i] == playerName) then
			old = i
			break end end
	return old
end

------------------------------------------------------------
function GLDG_CorrectPlayerOffset(old, playerName)
	-- Abort if no value given
	if not old then return end
	local new = nil
	-- Find new position number on list for this selection
	for i = 1, GLDG_TableSize(GLDG_SortedList) do
		if (GLDG_SortedList[i] == playerName) then
			new = i
			break end end
	-- Abort if no longer on list or position didn't change
	if (not new) or (old == new) then return end
	-- Calculate new offset and set it
	GLDG_PlayerOffset = math.max(0, math.min(GLDG_PlayerOffset + new - old, GLDG_TableSize(GLDG_SortedList) - GLDG_NumPlrRows))
	FauxScrollFrame_SetOffset(getglobal(GLDG_GUI.."PlayersPlayerbar"), GLDG_PlayerOffset)
	GLDG_ListPlayers()
end


-----------------------------------
-- _15_ General Helper Functions --
-----------------------------------

function GLDG_SetTextColor(textName, setName, colourName)
	if not (textName and setName and colourName) then return end
	if not GLDG_Data.colours[setName] then return end
	if not GLDG_Data.colours[setName][colourName] then return end

	local colour = GLDG_Data.colours[setName][colourName]

	local a,r,g,b = GLDG_ColourToRGB_perc(colour)

	local text = getglobal(textName)
	if text then text:SetTextColor(r, g, b) end
end

------------------------------------------------------------
function GLDG_DebugPrint(msg)
	if not (GLDG_Debug and msg) then return end
	DEFAULT_CHAT_FRAME:AddMessage("GUILDGREET DEBUG: "..msg)
end

------------------------------------------------------------
function GLDG_Print(msg, forceDefault)
	if not (msg) then return end
	if ((GLDG_Data.ChatFrame==0) or forceDefault) then
		DEFAULT_CHAT_FRAME:AddMessage(msg)
	else
		getglobal("ChatFrame"..GLDG_Data.ChatFrame):AddMessage(msg)
	end
end

------------------------------------------------------------
function GLDG_TableSize(info)
	local result = 0
	if info then
		for item in pairs(info) do result = result + 1 end
	end
	return result
end

------------------------------------------------------------
function GLDG_GetWords(str)
	local ret = {};
	local pos=0;
	local index=0
	while(true) do
		local word;
		_,pos,word=string.find(str, "^ *([^%s]+) *", pos+1);
		if(not word) then
			return ret;
		end
		ret[index]=word
		index = index+1
	end
end

------------------------------------------------------------
function GLDG_Help()
	GLDG_Print(" ", true)
	GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_TXT.help, true)
	GLDG_Print(GLDG_Data.colours.help..GLDG_TXT.usage..":|r /gg guild "..GLDG_Data.colours.help.."[|rshow"..GLDG_Data.colours.help.."] | |rguild all "..GLDG_Data.colours.help.."| |rguild list", true)
	GLDG_Print(GLDG_Data.colours.help..GLDG_TXT.usage..":|r /gg friends "..GLDG_Data.colours.help.."[|rshow"..GLDG_Data.colours.help.."] | |rfriends all "..GLDG_Data.colours.help.."| |rfriends list", true)
	GLDG_Print(GLDG_Data.colours.help..GLDG_TXT.usage..":|r /gg channel "..GLDG_Data.colours.help.."[|rshow"..GLDG_Data.colours.help.."] | |rchannel all "..GLDG_Data.colours.help.."| |rchannel list", true)
	GLDG_Print(GLDG_Data.colours.help..GLDG_TXT.usage..":|r /gg all "..GLDG_Data.colours.help.."[|rshow"..GLDG_Data.colours.help.."] | |rguild all "..GLDG_Data.colours.help.."| |rguild list", true)
	GLDG_Print(GLDG_Data.colours.help..GLDG_TXT.usage..":|r /gg "..GLDG_Data.colours.help.."[|rshow"..GLDG_Data.colours.help.."] <|r"..GLDG_TXT.name..GLDG_Data.colours.help.."> | |rfull "..GLDG_Data.colours.help.."<|r"..GLDG_TXT.name..GLDG_Data.colours.help..">|r", true)
	GLDG_Print(GLDG_Data.colours.help..GLDG_TXT.usage..":|r /gg help "..GLDG_Data.colours.help.."| |rabout "..GLDG_Data.colours.help.."| |rconfig "..GLDG_Data.colours.help.."| |rtest", true)
	GLDG_Print(GLDG_Data.colours.help..GLDG_TXT.usage..":|r /gg urbin|r "..GLDG_TXT.help_urbin, true);
	GLDG_Print(GLDG_Data.colours.help..GLDG_TXT.usage..":|r /gg clear", true)
	GLDG_Print(GLDG_Data.colours.help..GLDG_TXT.usage..":|r /gg greet "..GLDG_Data.colours.help.."| |rbye "..GLDG_Data.colours.help.."| |rlater "..GLDG_Data.colours.help.."[|r guild "..GLDG_Data.colours.help.."| |rchannel "..GLDG_Data.colours.help.."| |rall "..GLDG_Data.colours.help.."| <|rname"..GLDG_Data.colours.help.."> ]|r", true)
	GLDG_Print(" - "..GLDG_Data.colours.help.."guild/friends/channel [show]:|r "..GLDG_TXT.help_all, true);
	GLDG_Print(" - "..GLDG_Data.colours.help.."guild/friends/channel all:|r "..GLDG_TXT.help_online, true);
	GLDG_Print(" - "..GLDG_Data.colours.help.."guild/friends/channel list:|r "..GLDG_TXT.help_list, true);
	GLDG_Print(" - "..GLDG_Data.colours.help.."[show] <name>:|r "..GLDG_TXT.help_name, true);
	GLDG_Print(" - "..GLDG_Data.colours.help.."full <name>:|r "..GLDG_TXT.help_name_full, true);
	GLDG_Print(" - "..GLDG_Data.colours.help.."clear:|r "..GLDG_TXT.help_clear, true);
	GLDG_Print(" - "..GLDG_TXT.noargs, true);
	GLDG_Print(" ", true)
end

------------------------------------------------------------
function GLDG_About(urbin)
	local ver = GetAddOnMetadata("GuildGreet", "Version");
	local date = GetAddOnMetadata("GuildGreet", "X-Date");
	local author = GetAddOnMetadata("GuildGreet", "Author");
	local web = GetAddOnMetadata("GuildGreet", "X-Website");

	if (author ~= nil) then
		GLDG_Print(GLDG_NAME.." "..GLDG_TXT.by.." "..GLDG_Data.colours.help..author.."|r", true);
	end
	if (ver ~= nil) then
		GLDG_Print("  "..GLDG_TXT.version..": "..GLDG_Data.colours.help..ver.."|r", true);
	end
	if (date ~= nil) then
		GLDG_Print("  "..GLDG_TXT.date..": "..GLDG_Data.colours.help..date.."|r", true);
	end
	if (web ~= nil) then
		GLDG_Print("  "..GLDG_TXT.web..": "..GLDG_Data.colours.help..web.."|r", true);
	end

	if (urbin) then
		GLDG_Print("  "..GLDG_TXT.slash..": "..GLDG_Data.colours.help..SLASH_GLDG2.."|r", true);
	else
		GLDG_ListUrbinAddons(GLDG_NAME)
	end
end


---------------------
-- _16_ List frame --
---------------------
-- bee
GLDG_PasteList = {
	nilFrame = {
		GetName = function() return "Global" end
	},
}

------------------------------------------------------------
-- Show function for list frame
------------------------------------------------------------
-- bee
function GLDG_PasteList.Show()
	GLDG_PasteList.List.lastEntry = "";

	local list = GLDG_list;
	if (list) then
		for i in pairs(list) do
			if (list[i]) then
				if (GLDG_PasteList.List.lastEntry ~= "") then
					GLDG_PasteList.List.lastEntry = GLDG_PasteList.List.lastEntry.."\n";
				end
				GLDG_PasteList.List.lastEntry = GLDG_PasteList.List.lastEntry..list[i];
			end
		end
	end

	GLDG_PasteList.List.Box:SetText(GLDG_PasteList.List.lastEntry)
end

------------------------------------------------------------
-- Close function for list frame
------------------------------------------------------------
-- bee
function GLDG_PasteList.Done()
	GLDG_PasteList.List:Hide()
end

------------------------------------------------------------
-- Update function for list frame
------------------------------------------------------------
-- bee
function GLDG_PasteList.Update()
	GLDG_PasteList.List.Box:SetText(GLDG_PasteList.List.lastEntry)
	GLDG_PasteList.List.Scroll:UpdateScrollChildRect()
	GLDG_PasteList.List.Box:ClearFocus()
end

------------------------------------------------------------
-- Create our list frame
------------------------------------------------------------
-- bee
GLDG_PasteList.List = CreateFrame("Frame", "", UIParent)
GLDG_PasteList.List:Hide()
GLDG_PasteList.List:SetPoint("CENTER", "UIParent", "CENTER")
GLDG_PasteList.List:SetFrameStrata("DIALOG")
GLDG_PasteList.List:SetHeight(560)
GLDG_PasteList.List:SetWidth(500)
GLDG_PasteList.List:SetBackdrop({
	bgFile = "Interface/Tooltips/UI-Tooltip-Background",
	edgeFile = "Interface/Tooltips/UI-Tooltip-Border",
	tile = true, tileSize = 32, edgeSize = 32,
	insets = { left = 9, right = 9, top = 9, bottom = 9 }
})
GLDG_PasteList.List:SetBackdropColor(0,0,0, 0.8)
GLDG_PasteList.List:SetScript("OnShow", GLDG_PasteList.Show)

GLDG_PasteList.List.Done = CreateFrame("Button", "", GLDG_PasteList.List, "OptionsButtonTemplate")
GLDG_PasteList.List.Done:SetText("Close")
GLDG_PasteList.List.Done:SetPoint("BOTTOMRIGHT", GLDG_PasteList.List, "BOTTOMRIGHT", -10, 10)
GLDG_PasteList.List.Done:SetScript("OnClick", GLDG_PasteList.Done)

GLDG_PasteList.List.Scroll = CreateFrame("ScrollFrame", "GLDG_PasteListInputScroll", GLDG_PasteList.List, "UIPanelScrollFrameTemplate")
GLDG_PasteList.List.Scroll:SetPoint("TOPLEFT", GLDG_PasteList.List, "TOPLEFT", 20, -20)
GLDG_PasteList.List.Scroll:SetPoint("RIGHT", GLDG_PasteList.List, "RIGHT", -30, 0)
GLDG_PasteList.List.Scroll:SetPoint("BOTTOM", GLDG_PasteList.List.Done, "TOP", 0, 10)

GLDG_PasteList.List.Box = CreateFrame("EditBox", "GLDG_PasteListEditBox", GLDG_PasteList.List.Scroll)
GLDG_PasteList.List.Box:SetWidth(460)
GLDG_PasteList.List.Box:SetHeight(85)
GLDG_PasteList.List.Box:SetMultiLine(true)
GLDG_PasteList.List.Box:SetAutoFocus(false)
GLDG_PasteList.List.Box:SetFontObject(GameFontHighlight)
GLDG_PasteList.List.Box:SetScript("OnEscapePressed", GLDG_PasteList.Done)
GLDG_PasteList.List.Box:SetScript("OnTextChanged", GLDG_PasteList.Update)

GLDG_PasteList.List.Scroll:SetScrollChild(GLDG_PasteList.List.Box)
------------------------------------------------------------


---------------------
-- _17_ player menu --
---------------------

------------------------------------------------------------
function GLDG_AddPopUpButtons()
	UnitPopupMenus["GLDG"] = {"GLDG_LOOKUP", "GLDG_BYE", "CANCEL"};
	UnitPopupButtons["GLDG"] = { text = GLDG_TXT.menu, dist = 0, nested = 1};
	UnitPopupButtons["GLDG_LOOKUP"] = { text = GLDG_TXT.lookup, dist = 0,};
	UnitPopupButtons["GLDG_BYE"] = { text = GLDG_TXT.goodbye, dist = 0 };

	table.insert(UnitPopupMenus["PLAYER"], #UnitPopupMenus["PLAYER"]-1, "GLDG");
	table.insert(UnitPopupMenus["FRIEND"], #UnitPopupMenus["FRIEND"]-1, "GLDG");
	table.insert(UnitPopupMenus["PARTY"], #UnitPopupMenus["PARTY"]-1, "GLDG");

	hooksecurefunc("UnitPopup_OnClick", GLDG_UnitPopupOnClick);
	hooksecurefunc("UnitPopup_HideButtons", GLDG_UnitPopupHideButtons);
end

------------------------------------------------------------
-- Hooked function for a unit popup
function GLDG_UnitPopupOnClick()
	local dropdownFrame = getglobal(UIDROPDOWNMENU_INIT_MENU);
	local button = this.value;
	local name = dropdownFrame.name;

	if button == "GLDG_LOOKUP" then
		GLDG_ListForPlayer(name, false);
		ToggleDropDownMenu(1, nil, dropdownFrame, "cursor");
	elseif button == "GLDG_BYE" then
		GLDG_SendBye(name);
		ToggleDropDownMenu(1, nil, dropdownFrame, "cursor");
	end

	return;
end

------------------------------------------------------------
-- Hooked function to hide buttons for non-friendly players
function GLDG_UnitPopupHideButtons()
	local dropdownFrame = getglobal(UIDROPDOWNMENU_INIT_MENU);
	local coop = dropdownFrame.unit and UnitCanCooperate("player", dropdownFrame.unit)
	for index, value in ipairs(UnitPopupMenus[dropdownFrame.which]) do
		if (((value == "GLDG_LOOKUP") or (value == "GLDG_BYE")) and not coop) then
			UnitPopupShown[UIDROPDOWNMENU_MENU_LEVEL][index] = 0;
			return;
		end
	end
end

---------------------
-- _18_ friends roster update --
---------------------

function GLDG_FriendsUpdate()
	if (not GLDG_Data.UseFriends) then
		return
	end

	-- prepare purge marks
	for p in pairs(GLDG_DataFriends.chars[GLDG_Player]) do
		GLDG_DataFriends.chars[GLDG_Player][p].friend = "false";
	end

	-- parse friends list
	local cnt = 0
	local complete = false
	local numTotal = GetNumFriends()
	for i = 1, numTotal do
		local name, level, class, loc, connected, status = GetFriendInfo(i);
		if (name) then
			cnt = cnt + 1
			-- mark this friend as "active" so it won't be purged below
			if (GLDG_DataFriends.chars[GLDG_Player][name] == nil) then
				GLDG_DataFriends.chars[GLDG_Player][name] = {}
			end
			GLDG_DataFriends.chars[GLDG_Player][name].friend = "true"

			-- update data for this friend
			if (GLDG_DataFriends.friends[name] == nil) then
				GLDG_DataFriends.friends[name] = {}
			end
			if (GLDG_DataFriends.friends[name].lvl == nil) then
				GLDG_DataFriends.friends[name].lvl = level
			end

			if (GLDG_Data.UseFriends) then
				if (connected) then
					-- Update player level
					if (level > GLDG_DataFriends.friends[name].lvl) then
						if GLDG_Data.ListLevelUp then
							GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r ["..name.."] "..GLDG_TXT.lvlIncrease1.." "..GLDG_DataFriends.friends[name].lvl.." "..GLDG_TXT.lvlIncrease2.." "..level.." "..GLDG_TXT.lvlIncrease3);
						end
						if (GLDG_TableSize(GLDG_DataGreet.NewLevel) > 0) and not GLDG_Data.SupressLevel and not GLDG_DataFriends.friends[name].ignore and (level > GLDG_Data.MinLevelUp) then
							if (GLDG_Online[pl] or not GLDG_Data.NoGratsOnLogin) then
								GLDG_DataFriends.friends[name].newlvl = true
							end
						end
					end
					-- Update queue with all changes still missing
					if (GLDG_DataFriends.friends[name].new or GLDG_DataFriends.friends[name].newlvl or GLDG_DataFriends.friends[name].newrank) and (not GLDG_Queue[name]) then
						GLDG_Queue[name] = "[??:??] "
					end

					GLDG_DataFriends.friends[name].lvl = level
				end
				if (not GLDG_Offline[name]) then
					GLDG_Offline[name] = false
				end
				if (GLDG_Online[name] == nil) then
					GLDG_Online[name] = connected
				end
			end
		end
	end

	-- purge "removed friends" for this char
	for p in pairs(GLDG_DataFriends.chars[GLDG_Player]) do
		if (GLDG_DataFriends.chars[GLDG_Player][p].friend ~= "true") then
			GLDG_DataFriends.chars[GLDG_Player][p] = nil;
		end
	end

	-- purge friends that have been removed from all chars
	local purgeList = {}
	for p in pairs(GLDG_DataFriends.friends) do
		local friend = false
		for q in pairs(GLDG_DataFriends.chars) do
			if (GLDG_DataFriends.chars[q][p] and GLDG_DataFriends.chars[q][p].friend == "true") then
				friend = true
			end
		end
		-- mark for deletion
		if (not friend) then
			purgeList[p] = "true"
		end
	end
	for p in pairs(purgeList) do
		GLDG_DataFriends.friends[p] = nil;

		-- check if we're deleting the main of an alt
		for q in pairs(GLDG_DataFriends.friends) do
			if (GLDG_DataFriends.friends[q].alt and GLDG_DataFriends.friends[q].alt==p) then
				GLDG_DataFriends.friends[q].alt = nil
			end
		end
	end


	-- If we got our info, switch to the next phase
	if (cnt > 0) then
		if not GLDG_InitialFriendsUpdate then
			GLDG_InitialFriendsUpdate = true
			GLDG_Main:RegisterEvent("CHAT_MSG_SYSTEM")
		end

		-- retrigger friends update
		if (GLDG_Data.UpdateTime > 0) then
			GLDG_UpdateRequestFriends = GetTime() + GLDG_Data.UpdateTime
		else
			GLDG_UpdateRequestFriends = nil
		end

		GLDG_ShowQueue()
	end
end


-----------------------------
-- _19_ Friends Tab Update --
-----------------------------
function GLDG_SortString_f(player)
	-- Helper function: returns string that should be used for sorting
	local result = player
	if (GLDG_DataFriends.friends[player].alt and GLDG_Data.GroupAlt) then result = GLDG_DataFriends.friends[player].alt..player end
	return string.lower(result)
end

------------------------------------------------------------
function GLDG_ListPlayers_f()
	local abort = false
	-- Abort if addon is not initialized
	if not GLDG_DataFriends then abort = true end
	if not GLDG_DataFriends.friends then abort = true  end
	-- Abort if friend support is disabled
	if (not GLDG_Data.UseFriends) then abort = true end

	local frame = GLDG_GUI.."Friends"
	if (abort == true) then
		GLDG_SelPlrName_f = nil
		getglobal(frame.."SubAlias"):Hide()
		getglobal(frame.."SubMainAlt"):Hide()
		getglobal(frame.."Ignore"):Hide()
		getglobal(frame.."Alias"):Hide()
		getglobal(frame.."Main"):Hide()
		getglobal(frame.."Alt"):Hide()

		local line = GLDG_GUI.."FriendsLine"
		for cnt = 1, GLDG_NumPlrRows_f do
			getglobal(line..cnt.."Text"):SetText("")
			getglobal(line..cnt.."Text2"):SetText("")
			getglobal(line..cnt):Disable()
		end

		return
	end

	-- Check if there is a reason to clear the selected player
	if GLDG_SelPlrName_f then
		if (not GLDG_DataFriends.friends[GLDG_SelPlrName_f]) or
		   (GLDG_DataFriends.friends[GLDG_SelPlrName_f].ignore and not GLDG_Data.ShowIgnore) or
		   (GLDG_DataGuild and GLDG_DataGuild[GLDG_SelPlrName_f])  or
		   (not GLDG_DataFriends.chars[GLDG_Player][GLDG_SelPlrName_f]) then
			GLDG_SelPlrName_f = nil end end
	-- Prepare for list creation
	GLDG_SortedList_f = {}
	local total = 0
	GLDG_NumMain_f = 0
	GLDG_NumAlts_f = 0
	local s = {}
	if GLDG_SelPlrName_f then s = GLDG_DataFriends.friends[GLDG_SelPlrName_f] end
	-- Create the list of players to display
	for player in pairs(GLDG_DataFriends.friends) do
		local p = GLDG_DataFriends.friends[player]
		-- Update counters for main and alt
		if p.main then GLDG_NumMain_f = GLDG_NumMain_f + 1 end
		if p.alt and (p.alt == GLDG_SelPlrName_f) then GLDG_NumAlts_f = GLDG_NumAlts_f + 1 end
		-- See if we are supposed to add this player in our list
		local show = GLDG_Data.ShowIgnore or not p.ignore
		show = show and ((p.alt == GLDG_SelPlrName_f) or (p.alt == s.alt) or not p.alt or GLDG_Data.ShowAlt)
		show = show and (GLDG_DataFriends.chars[GLDG_Player][player])
		show = show and not GLDG_DataGuild[player] -- don't show guild members
		show = show and (((not p.alt) and (not p.main)) or not GLDG_Data.FilterUnassigned)
		if show then
			-- Look for the position in the list we need
			local loc = 1
			while GLDG_SortedList_f[loc] and (GLDG_SortString_f(player) > GLDG_SortString_f(GLDG_SortedList_f[loc])) do loc = loc + 1 end
			-- We found our position: move everything beyond it
			for cnt = total, loc, -1 do GLDG_SortedList_f[cnt + 1] = GLDG_SortedList_f[cnt] end
			-- Insert our player
			total = total + 1
			GLDG_SortedList_f[loc] = player end end
	-- Update the scrollbar
	GLDG_ClickPlayerBar_f()
end

------------------------------------------------------------
function GLDG_ShowPlayers_f()
	if (not GLDG_Data.UseFriends) then return end
	-- Display the players visible in the frame
	local cnt = 1
	local line = GLDG_GUI.."FriendsLine"
	while GLDG_SortedList_f[cnt + GLDG_PlayerOffset_f] and getglobal(line..cnt) do
		local text = getglobal(line..cnt.."Text")
		local text2 = getglobal(line..cnt.."Text2")
		local p = GLDG_DataFriends.friends[GLDG_SortedList_f[cnt + GLDG_PlayerOffset_f]]
		if p.alias then text:SetText(GLDG_SortedList_f[cnt + GLDG_PlayerOffset_f].."*")
		else text:SetText(GLDG_SortedList_f[cnt + GLDG_PlayerOffset_f]) end
		text2:SetText("")
		if p.ignore then
			text:SetTextColor(1, 0.25, 0.25)
			text2:SetText(GLDG_TXT.markIGN)
		elseif p.main then
			text:SetTextColor(0.25, 1, 0.25)
			text2:SetText(GLDG_TXT.markMAIN)
		elseif p.alt then
			text:SetTextColor(0.25, 0.25, 1)
			text2:SetText(GLDG_TXT.markALT)
		elseif p.alias then
			text:SetTextColor(0.68, 0.8, 1)
		else text:SetTextColor(1, 1, 1) end
		text2:SetTextColor(text:GetTextColor())
		getglobal(line..cnt):Enable()
		cnt = cnt +1 end
	-- Disable any rows left unused
	for cnt2 = cnt, GLDG_NumPlrRows_f do
		getglobal(line..cnt2.."Text"):SetText("")
		getglobal(line..cnt2.."Text2"):SetText("")
		getglobal(line..cnt2):Disable() end
	-- Set/clear highlight
	for cnt = 1, GLDG_NumPlrRows_f do
		if GLDG_SelPlrName_f and (GLDG_SortedList_f[cnt + GLDG_PlayerOffset_f] == GLDG_SelPlrName_f) then getglobal(line..cnt):LockHighlight()
		else getglobal(line..cnt):UnlockHighlight() end end
	-- Update buttons
	GLDG_ShowPlayerButtons_f()
end

------------------------------------------------------------
function GLDG_ClickPlayer_f(playerName)
	if (not GLDG_Data.UseFriends) then return end
	-- Prepare for scrollbar adjustment if needed
	local p = {}
	if GLDG_SelPlrName_f then p = GLDG_DataFriends.friends[GLDG_SelPlrName_f] end
	local old = nil
	if (not GLDG_Data.ShowAlt) and (p.main or p.alt) then old = GLDG_GetPlayerOffset_f(playerName) end
	-- Set new selected player
	if (playerName == GLDG_SelPlrName_f) then GLDG_SelPlrName_f = nil
	else GLDG_SelPlrName_f = playerName end
	-- Refresh list
	GLDG_ListPlayers_f()
	-- Adjust scrollbar if needed
	GLDG_CorrectPlayerOffset_f(old, playerName)
end

------------------------------------------------------------
function GLDG_ClickPlayerBar_f(frame)
	if (not GLDG_Data.UseFriends) then return end
	-- Update offset
	if not frame then frame = getglobal(GLDG_GUI.."FriendsPlayerbar") end
	FauxScrollFrame_Update(frame, GLDG_TableSize(GLDG_SortedList_f), GLDG_NumPlrRows_f, 15)
	GLDG_PlayerOffset_f = FauxScrollFrame_GetOffset(frame)
	GLDG_ShowPlayers_f()
end

------------------------------------------------------------
function GLDG_ShowPlayerButtons_f()
	-- Set frame
	local frame = GLDG_GUI.."Friends"
	-- Hide subframes
	getglobal(frame.."SubAlias"):Hide()
	getglobal(frame.."SubMainAlt"):Hide()
	-- If no player is selected: hide all buttons
	if not GLDG_SelPlrName_f then
		getglobal(frame.."Ignore"):Hide()
		getglobal(frame.."Alias"):Hide()
		getglobal(frame.."Main"):Hide()
		getglobal(frame.."Alt"):Hide()
		return end
	if (not GLDG_Data.UseFriends) then return end
	-- Set selected player
	local p = GLDG_DataFriends.friends[GLDG_SelPlrName_f]
	-- Ignore button
	local button = getglobal(frame.."Ignore")
	if p.ignore then button:SetText(GLDG_TXT.pbrign) else button:SetText(GLDG_TXT.pbsign) end
	button:Show()
	-- Alias button
	button = getglobal(frame.."Alias")
	button:Show()
	button:Enable()
	-- Main button
	button = getglobal(frame.."Main")
	button:Show()
	if p.alt then button:SetText(GLDG_TXT.pbpmain)
	elseif not p.main then button:SetText(GLDG_TXT.pbsmain)
	elseif (GLDG_NumAlts_f == 0) then button:SetText(GLDG_TXT.pbrmain)
	else button:Hide() end
	-- Alt button
	button = getglobal(frame.."Alt")
	button:Show()
	button:Enable()
	if p.alt then button:SetText(GLDG_TXT.pbralt)
	elseif p.main then button:Hide()
	else button:SetText(GLDG_TXT.pbsalt) end
end

------------------------------------------------------------
function GLDG_ClickIgnore_f()
	if (not GLDG_Data.UseFriends) then return end
	-- Toggle ignore state for selected player
	local p = GLDG_DataFriends.friends[GLDG_SelPlrName_f]
	local main = p.alt
	if p.main then main = GLDG_SelPlrName_f end
	local newval = nil
	if not p.ignore then newval = true end
	-- If main or alt, toggle ignore for all characters of this player
	if main then for q in pairs(GLDG_DataFriends.friends) do if (q == main) or (GLDG_DataFriends.friends[q].alt == main) then GLDG_DataFriends.friends[q].ignore = newval end end
	-- Otherwise, simply toggle ignore for this one character
	else p.ignore = newval end
	-- Show updated list
	GLDG_ListPlayers_f()
end

------------------------------------------------------------
function GLDG_ClickAlias_f()
	if (not GLDG_Data.UseFriends) then return end
	-- Activate alias subframe
	GLDG_ShowPlayerButtons_f()
	this:Disable()
	getglobal(this:GetParent():GetName().."SubAlias"):Show()
end

------------------------------------------------------------
function GLDG_ClickMain_f()
	if (not GLDG_Data.UseFriends) then return end
	-- Toggle main status for selected character or promote an alt
	local p = GLDG_DataFriends.friends[GLDG_SelPlrName_f]
	if p.main then p.main = nil
	elseif p.alt then
		local oldmain = p.alt
		p.alt = nil
		p.main = true
		for q in pairs(GLDG_DataFriends.friends) do
			if (q == oldmain) then
				GLDG_DataFriends.friends[q].main = nil
				GLDG_DataFriends.friends[q].alt = GLDG_SelPlrName_f
			elseif (GLDG_DataFriends.friends[q].alt == oldmain) then
				GLDG_DataFriends.friends[q].alt = GLDG_SelPlrName_f end end
	else p.main = true end
	-- Show updated list
	GLDG_ListPlayers_f()
end

------------------------------------------------------------
function GLDG_ClickAlt_f()
	if (not GLDG_Data.UseFriends) then return end
	-- Prepare for scrollbar adjustment if needed
	local p = {}
	if GLDG_SelPlrName_f then p = GLDG_DataFriends.friends[GLDG_SelPlrName_f] end
	local old = nil
	if GLDG_Data.GroupAlt then old = GLDG_GetPlayerOffset_f(GLDG_SelPlrName_f) end
	-- Toggle alt status for selected character
	local p = GLDG_DataFriends.friends[GLDG_SelPlrName_f]
	if p.alt then p.alt = nil
	elseif (GLDG_NumMain_f == 1) then
		for q in pairs(GLDG_DataFriends.friends) do
			if GLDG_DataFriends.friends[q].main then
				p.alt = q
				p.ignore = GLDG_DataFriends.friends[q].ignore
				break end end
	else	GLDG_ShowPlayerButtons_f()
		this:Disable()
		getglobal(this:GetParent():GetName().."SubMainAlt"):Show()
		return end
	-- Show updated list
	GLDG_ListPlayers_f()
	-- Adjust scrollbar if needed
	GLDG_CorrectPlayerOffset_f(old, GLDG_SelPlrName_f)
end

------------------------------------------------------------
function GLDG_ShowPlayerAlias_f(frame)
	if (not GLDG_Data.UseFriends) then return end
	-- Set title
	getglobal(frame.."Header"):SetText(string.format(GLDG_TXT.aliashead, GLDG_SelPlrName_f))
	-- Set editbox and buttons text
	local p = GLDG_DataFriends.friends[GLDG_SelPlrName_f]
	if p.alias then
		getglobal(frame.."Set"):SetText(GLDG_TXT.update)
		getglobal(frame.."Del"):SetText(GLDG_TXT.delete)
		getglobal(frame.."Editbox"):SetText(p.alias)
	else	getglobal(frame.."Set"):SetText(GLDG_TXT.set)
		getglobal(frame.."Del"):SetText(GLDG_TXT.cancel)
		getglobal(frame.."Editbox"):SetText("") end
end

------------------------------------------------------------
function GLDG_ClickAliasSet_f(alias)
	if (not GLDG_Data.UseFriends) then return end
	if (alias == "") then return end
	GLDG_DataFriends.friends[GLDG_SelPlrName_f].alias = alias
	GLDG_ListPlayers_f()
end

------------------------------------------------------------
function GLDG_ClickAliasRemove_f()
	if (not GLDG_Data.UseFriends) then return end
	GLDG_DataFriends.friends[GLDG_SelPlrName_f].alias = nil
	GLDG_ListPlayers_f()
end

------------------------------------------------------------
function GLDG_ShowMainAlt_f(frame)
	if (not GLDG_Data.UseFriends) then return end
	-- Set frame and linename
	if not frame then frame = this end
	local name = frame:GetParent():GetName()
	-- Set title
	getglobal(name.."Header"):SetText(string.format(GLDG_TXT.mainhead, GLDG_SelPlrName_f))
	-- Create a sorted list of all mains
	local mainlist = {}
	local total = 0
	for p in pairs(GLDG_DataFriends.friends) do
		if (GLDG_DataFriends.friends[p].main and
		    GLDG_DataFriends.chars[GLDG_Player][p] and
		    (GLDG_DataGuild and not GLDG_DataGuild[p])) then
			local loc = 1
			while mainlist[loc] and (mainlist[loc] < p) do loc = loc + 1 end
			for cnt = total, loc, -1 do mainlist[cnt + 1] = mainlist[cnt] end
			total = total + 1
			mainlist[loc] = p
		end
	end
	-- Configure scrollbar
	FauxScrollFrame_Update(frame, GLDG_TableSize(mainlist), GLDG_NumSubRows_f, 15)
	local offset = FauxScrollFrame_GetOffset(frame)
	-- Set all rows
	name = name.."Line"
	for line = 1, GLDG_NumSubRows_f do
		if mainlist[line + offset] then
			getglobal(name..line.."Text"):SetText(mainlist[line + offset])
			getglobal(name..line):Enable()
		else	getglobal(name..line.."Text"):SetText("")
			getglobal(name..line):Disable() end end
end

------------------------------------------------------------
function GLDG_ClickMainAlt_f(player)
	if (not GLDG_Data.UseFriends) then return end
	-- Mark position of current selected player
	local old = GLDG_GetPlayerOffset_f(GLDG_SelPlrName_f)
	-- Make this the main for the currently selected character
	GLDG_DataFriends.friends[GLDG_SelPlrName_f].alt = player
	GLDG_DataFriends.friends[GLDG_SelPlrName_f].ignore = GLDG_DataFriends.friends[player].ignore
	-- Hide the subframe window
	this:GetParent():Hide()
	-- Refresh the playerlist
	GLDG_ListPlayers_f()
	-- Adjust scrollbar if needed
	GLDG_CorrectPlayerOffset_f(old, GLDG_SelPlrName_f)
end

------------------------------------------------------------
function GLDG_GetPlayerOffset_f(playerName)
	if (not GLDG_Data.UseFriends) then return nil end
	-- Find current position number on list
	local old = nil
	for i = 1, GLDG_TableSize(GLDG_SortedList_f) do
		if (GLDG_SortedList_f[i] == playerName) then
			old = i
			break end end
	return old
end

------------------------------------------------------------
function GLDG_CorrectPlayerOffset_f(old, playerName)
	if (not GLDG_Data.UseFriends) then return end
	-- Abort if no value given
	if not old then return end
	local new = nil
	-- Find new position number on list for this selection
	for i = 1, GLDG_TableSize(GLDG_SortedList_f) do
		if (GLDG_SortedList_f[i] == playerName) then
			new = i
			break end end
	-- Abort if no longer on list or position didn't change
	if (not new) or (old == new) then return end
	-- Calculate new offset and set it
	GLDG_PlayerOffset_f = math.max(0, math.min(GLDG_PlayerOffset_f + new - old, GLDG_TableSize(GLDG_SortedList_f) - GLDG_NumPlrRows_f))
	FauxScrollFrame_SetOffset(getglobal(GLDG_GUI.."FriendsPlayerbar"), GLDG_PlayerOffset_f)
	GLDG_ListPlayers_f()
end


-----------------------------
-- _20_ Channel handling   --
-----------------------------

------------------------------------------------------------
function GLDG_CheckChannel()
	GLDG_inChannel = false;
	if (GLDG_ChannelName and GLDG_ChannelName ~= "") then

		local id, name = GetChannelName(GLDG_ChannelName)
		local id2, name2 = GetChannelName(id)
		if ((id>0) and (id2>0) and (id==id2) and (name2==GLDG_ChannelName)) then
			GLDG_DebugPrint("Channel ["..GLDG_ChannelName.."] exists and has id ["..id.."]")

			GLDG_inChannel = true;

			-- get members of channel
			GLDG_Main:RegisterEvent("CHAT_MSG_CHANNEL_LIST")
			DEFAULT_CHAT_FRAME:UnregisterEvent("CHAT_MSG_CHANNEL_LIST")
			GLDG_unregister = GLDG_unregister + 1
			ListChannelByName(GLDG_ChannelName)
		end
	end
	GLDG_UpdateTabs()

	return GLDG_inChannel
end

------------------------------------------------------------
function GLDG_InitChannel(data)
	-- now that we got a list of all people in the channel, we do not need to monitor this event
	if (GLDG_unregister > 0) then
		GLDG_unregister = GLDG_unregister - 1
		if (GLDG_unregister == 0) then
			GLDG_Main:UnregisterEvent("CHAT_MSG_CHANNEL_LIST")
			DEFAULT_CHAT_FRAME:RegisterEvent("CHAT_MSG_CHANNEL_LIST")
		end
	end


	if (GLDG_DataGuild) then
		GLDG_checkedChannel = true
	end

	if (GLDG_inChannel and GLDG_DataGuild) then
		if (data) then
			local names = GLDG_GetWords(data)

			for p in pairs(names) do
				local name = names[p]
				local a,b,c = strfind(name, ',', 0, true)
				if (a) then
					name = string.sub(name, 1, a-1)
				end
				local a,b,c = strfind(name, '*', 0, true)
				if (a) then
					name = string.sub(name, a+1)
				end

				GLDG_DebugPrint(p..": Name ["..names[p].."] retrieved -> ["..name.."] extracted");
				if (name and not GLDG_DataGuild[name]) then
					if (GLDG_DataChannel[name] == nil) then
						GLDG_DataChannel[name] = {}
					end

					if (not GLDG_Offline[name]) then
						GLDG_Offline[name] = false
					end
					if (GLDG_Online[name] == nil) then
						GLDG_Online[name] = true
					end
				end
			end
		end
	end
end

------------------------------------------------------------
function GLDG_UpdateChannel(joined, player)
	if not player then return end

	if (GLDG_inChannel and GLDG_DataGuild and not GLDG_DataGuild[player]) then
		if not GLDG_DataChannel[player] then
			GLDG_DataChannel[player] = {}
		end

		if (joined) then
			GLDG_DebugPrint("Player ["..player.."] joined channel ["..GLDG_ChannelName.."]")

			if (GLDG_DataChannel[player] and not GLDG_DataChannel[player].ignore) then
				GLDG_DebugPrint("player "..player.." is a member of our channel")
				GLDG_Online[player] = GetTime()

				-- bee
				if GLDG_Data.ListNames then
					if GLDG_DataChannel[player].alt then
						--
						-- Alt von Main
						--
						local main = GLDG_DataChannel[player].alt;
						local altsList = GLDG_FindAlts_c(main, player, 1)
						GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..altsList)
					else
						if GLDG_DataChannel[player].main then
							--
							-- Main
							--
							local main = player;
							local altsList = GLDG_FindAlts_c(main, player, 1)
							GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..altsList)
						else
							--
							-- Hat keinen Alt
							--
							local alias = GLDG_findAlias_c(player, 1);

							GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_ONLINE_COLOUR..player.."|r"..alias.." {c}")
						end
					end
				end

				if GLDG_DataChannel[player].alt then GLDG_Offline[player] = GLDG_Offline[GLDG_DataChannel[player].alt] end
				if GLDG_Offline[player] and (GLDG_Online[player] - GLDG_Offline[player] < GLDG_Data.RelogTime * 60) then return end
				GLDG_DebugPrint("player "..player.." is not been online in the last "..GLDG_Data.RelogTime.." minutes.")
				if GLDG_Offline[player] and (GLDG_Data.SupressGreet or (GLDG_TableSize(GLDG_DataGreet.GreetBack) == 0)) then return end
				GLDG_DebugPrint("player "..player.." is not been online before")
				if not GLDG_Offline[player] and (GLDG_Data.SupressGreet or (GLDG_TableSize(GLDG_DataGreet.Greet) == 0)) then return end
				GLDG_DebugPrint("player "..player.." should be greeted")
				GLDG_Queue[player] = GLDG_GetLogtime(player)
				GLDG_ShowQueue()
			end

		else
			GLDG_DebugPrint("Player ["..player.."] left channel ["..GLDG_ChannelName.."]")

			if (GLDG_DataChannel[player] and not GLDG_DataChannel[player].ignore) then
				GLDG_DebugPrint("player "..player.." is a member of our channel")
				GLDG_Online[player] = nil
				GLDG_RankUpdate[player] = nil
				if GLDG_DataChannel[player].alt then
					local main = GLDG_DataChannel[player].alt
					GLDG_DataChannel[main].last = player
					GLDG_Offline[main] = GetTime()
				elseif GLDG_DataChannel[player].main then
					GLDG_DataChannel[player].last = player
					GLDG_Offline[player] = GetTime()
				else GLDG_Offline[player] = GetTime() end
				if GLDG_Queue[player] then
					GLDG_Queue[player] = nil
					GLDG_ShowQueue() end

				-- bee
				if GLDG_Data.ListNamesOff then
					if GLDG_DataChannel[player].alt then
						--
						-- Alt von Main
						--
						local main = GLDG_DataChannel[player].alt;
						local altsList = GLDG_FindAlts_c(main, player, 0)
						GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..altsList)
					else
						if GLDG_DataChannel[player].main then
							--
							-- Main
							--
							local main = player;
							local altsList = GLDG_FindAlts_c(main, player, 0)
							GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..altsList)
						else
							--
							-- Hat keinen Alt
							--
							local alias = GLDG_findAlias_c(player, 0);

							GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_GOES_OFFLINE_COLOUR..player..alias.." {c}")
						end
					end
				end
			end
		end
	end
end

------------------------------------------------------------
function GLDG_ForceChatlist()
	GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_TXT.forceChatList)
	GLDG_unregister = 0
	DEFAULT_CHAT_FRAME:RegisterEvent("CHAT_MSG_CHANNEL_LIST")
end


-----------------------------
-- _21_ Testing            --
-----------------------------

------------------------------------------------------------
function GLDG_Test()
	GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r Test")

	--SendAddonMessage("GLDG", "Nur so ein Text", "WHISPER", "Urbin")

	SendAddonMessage("GLDG", "VER:"..GetAddOnMetadata("GuildGreet", "Version"), "GUILD")
	SendAddonMessage("GLDG", "VER:"..GetAddOnMetadata("GuildGreet", "Version"), "PARTY")
	SendAddonMessage("GLDG", "VER:"..GetAddOnMetadata("GuildGreet", "Version"), "RAID")

	SendAddonMessage("GLDG", "VER:"..GetAddOnMetadata("GuildGreet", "Version").."2", "GUILD")

	--GLDG_CreateTestChars()

	--[[ -- used for screenshot in order not to use real guild names --
	GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r Creating list")
	local list = {}
	list[1] = "[Magam] Lyb Venum - Mag"
	list[2] = "[Kelthuzad] Ganad Tholie - Kel"
	list[3] = "[Ronnie] Radon Rumag Renid - Ron"
	list[4] = "[Tom] Saka Luna - Tom"
	list[5] = "[Gundor] Hilada Vorix - Gundor"
	list[6] = "[Luator] Tibbor Hagadu - Luator"
	list[7] = "[Zapator] Mula Jeno Jilador Koradin Wegamul - Zappie"
	list[8] = "[Hilane] Mulase Mimidor - Hil"
	list[9] = "[Hugena] Fiodan - Huggie"
	list[10] = "[Kyreena] Kalomur - Kyreena"
	list[11] = "[Dinosa] Stib Lokinat - Dino"
	list[12] = "[Xynasa] Wugedan - Xyn"
	list[13] = "[Remolo] Quagal - Remo"
	list[14] = "[Anima] Arista Olibeth - Ani"
	list[15] = "[Pug] Thomas Charles - Pug"
	list[16] = "[Dores] Frazon Wiligom - Dori"
	list[17] = "[Numan] Jomin Yaldor - Numan"
	list[18] = "[Haxxor] - Hax"
	list[19] = "[Pwner] Gank0r - L00s0r"
	list[20] = "[Nadel] Sabtise - Nad"
	list[21] = "[Bunjipp] - Bun"
	list[22] = "[Haruna] Jorador - Haru"
	list[23] = "[Olibado] - Oli"
	list[24] = "[Sumada] - Sum"
	list[25] = "[Irida] - Iri"
	list[26] = "[Manolo] - Mano"
	list[27] = "[Tiara] - Tiara"
	list[28] = "[Tonina] - Toni"
	list[29] = "[Tamara] Solana Illidur - Tami"
	list[30] = "[Ugida] - Ugida"
	list[31] = "[Soleno] Somvitg - Sol"
	list[32] = "[Frog] - Frog"
	list[33] = "[Glazzi] - Glazzi"
	list[34] = "[Weinbär] Brummbär Waschbär- Bär"
	list[35] = "[Xhorosh] - Xor"
	list[36] = "[Grubosh] - Grubosh"
	list[37] = "[Mechtild] - Mechtild"
	list[38] = "[Munher] - Muni"
	list[39] = "[Liv] - Liv"
	list[40] = "[Spifosa] - Spif"
	list[41] = "[Hobbus] Calvadur - Hob"
	list[42] = "[Taran] - Taran"
	GLDG_list = list
	GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r Showing list")
	GLDG_PasteList.List:Show();
	GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r Shown list")
	]]--

end

------------------------------------------------------------
function GLDG_CreateTestChars()
	GLDG_Queue["Tester1"] = GLDG_GetLogtime("Tester1")
	GLDG_Queue["Tester2"] = GLDG_GetLogtime("Tester2")
	GLDG_Queue["Tester3"] = GLDG_GetLogtime("Tester3")
	GLDG_Queue["Tester4"] = GLDG_GetLogtime("Tester4")
	GLDG_Queue["Tester5"] = GLDG_GetLogtime("Tester5")

	GLDG_Online["Tester1"] = GLDG_GetLogtime("Tester1")
	GLDG_Online["Tester2"] = GLDG_GetLogtime("Tester2")
	GLDG_Online["Tester3"] = GLDG_GetLogtime("Tester3")
	GLDG_Online["Tester4"] = GLDG_GetLogtime("Tester4")
	GLDG_Online["Tester5"] = GLDG_GetLogtime("Tester5")

	GLDG_ShowQueue()
	-- todo
end

-----------------------------
-- _22_ Channel Tab Update --
-----------------------------
function GLDG_SortString_c(player)
	-- Helper function: returns string that should be used for sorting
	local result = player
	if (GLDG_DataChannel[player].alt and GLDG_Data.GroupAlt) then result = GLDG_DataChannel[player].alt..player end
	return string.lower(result)
end

------------------------------------------------------------
function GLDG_ListPlayers_c()
	local abort = false
	-- Abort if addon is not initialized
	if not GLDG_DataChannel then abort = true  end
	-- Abort if friend support is disabled
	if (not GLDG_inChannel) then abort = true end

	local frame = GLDG_GUI.."Channel"
	if (abort == true) then
		GLDG_SelPlrName_c = nil
		getglobal(frame.."SubAlias"):Hide()
		getglobal(frame.."SubMainAlt"):Hide()
		getglobal(frame.."Ignore"):Hide()
		getglobal(frame.."Alias"):Hide()
		getglobal(frame.."Main"):Hide()
		getglobal(frame.."Alt"):Hide()
		getglobal(frame.."Delete"):Hide()

		local line = GLDG_GUI.."ChannelLine"
		for cnt = 1, GLDG_NumPlrRows_c do
			getglobal(line..cnt.."Text"):SetText("")
			getglobal(line..cnt.."Text2"):SetText("")
			getglobal(line..cnt):Disable()
		end

		return
	end

	-- Check if there is a reason to clear the selected player
	if GLDG_SelPlrName_c then
		if (not GLDG_DataChannel[GLDG_SelPlrName_c]) or
		   (GLDG_DataChannel[GLDG_SelPlrName_c].ignore and not GLDG_Data.ShowIgnore) or
		   (GLDG_DataGuild and GLDG_DataGuild[GLDG_SelPlrName_c]) then
			GLDG_SelPlrName_c = nil end end
	-- Prepare for list creation
	GLDG_SortedList_c = {}
	local total = 0
	GLDG_NumMain_c = 0
	GLDG_NumAlts_c = 0
	local s = {}
	if GLDG_SelPlrName_c then s = GLDG_DataChannel[GLDG_SelPlrName_c] end
	-- Create the list of players to display
	for player in pairs(GLDG_DataChannel) do
		local p = GLDG_DataChannel[player]
		-- Update counters for main and alt
		if p.main then GLDG_NumMain_c = GLDG_NumMain_c + 1 end
		if p.alt and (p.alt == GLDG_SelPlrName_c) then GLDG_NumAlts_c = GLDG_NumAlts_c + 1 end
		-- See if we are supposed to add this player in our list
		local show = GLDG_Data.ShowIgnore or not p.ignore
		show = show and ((p.alt == GLDG_SelPlrName_c) or (p.alt == s.alt) or not p.alt or GLDG_Data.ShowAlt)
		show = show and not GLDG_DataGuild[player] -- don't show guild members
		show = show and (((not p.alt) and (not p.main)) or not GLDG_Data.FilterUnassigned)
		if show then
			-- Look for the position in the list we need
			local loc = 1
			while GLDG_SortedList_c[loc] and (GLDG_SortString_c(player) > GLDG_SortString_c(GLDG_SortedList_c[loc])) do loc = loc + 1 end
			-- We found our position: move everything beyond it
			for cnt = total, loc, -1 do GLDG_SortedList_c[cnt + 1] = GLDG_SortedList_c[cnt] end
			-- Insert our player
			total = total + 1
			GLDG_SortedList_c[loc] = player end end
	-- Update the scrollbar
	GLDG_ClickPlayerBar_c()
end

------------------------------------------------------------
function GLDG_ShowPlayers_c()
	if (not GLDG_inChannel) then return end
	-- Display the players visible in the frame
	local cnt = 1
	local line = GLDG_GUI.."ChannelLine"
	while GLDG_SortedList_c[cnt + GLDG_PlayerOffset_c] and getglobal(line..cnt) do
		local text = getglobal(line..cnt.."Text")
		local text2 = getglobal(line..cnt.."Text2")
		local p = GLDG_DataChannel[GLDG_SortedList_c[cnt + GLDG_PlayerOffset_c]]
		if p.alias then text:SetText(GLDG_SortedList_c[cnt + GLDG_PlayerOffset_c].."*")
		else text:SetText(GLDG_SortedList_c[cnt + GLDG_PlayerOffset_c]) end
		text2:SetText("")
		if p.ignore then
			text:SetTextColor(1, 0.25, 0.25)
			text2:SetText(GLDG_TXT.markIGN)
		elseif p.main then
			text:SetTextColor(0.25, 1, 0.25)
			text2:SetText(GLDG_TXT.markMAIN)
		elseif p.alt then
			text:SetTextColor(0.25, 0.25, 1)
			text2:SetText(GLDG_TXT.markALT)
		elseif p.alias then
			text:SetTextColor(0.68, 0.8, 1)
		else text:SetTextColor(1, 1, 1) end
		text2:SetTextColor(text:GetTextColor())
		getglobal(line..cnt):Enable()
		cnt = cnt +1 end
	-- Disable any rows left unused
	for cnt2 = cnt, GLDG_NumPlrRows_c do
		getglobal(line..cnt2.."Text"):SetText("")
		getglobal(line..cnt2.."Text2"):SetText("")
		getglobal(line..cnt2):Disable() end
	-- Set/clear highlight
	for cnt = 1, GLDG_NumPlrRows_c do
		if GLDG_SelPlrName_c and (GLDG_SortedList_c[cnt + GLDG_PlayerOffset_c] == GLDG_SelPlrName_c) then getglobal(line..cnt):LockHighlight()
		else getglobal(line..cnt):UnlockHighlight() end end
	-- Update buttons
	GLDG_ShowPlayerButtons_c()
end

------------------------------------------------------------
function GLDG_ClickPlayer_c(playerName)
	if (not GLDG_inChannel) then return end
	-- Prepare for scrollbar adjustment if needed
	local p = {}
	if GLDG_SelPlrName_c then p = GLDG_DataChannel[GLDG_SelPlrName_c] end
	local old = nil
	if (not GLDG_Data.ShowAlt) and (p.main or p.alt) then old = GLDG_GetPlayerOffset_c(playerName) end
	-- Set new selected player
	if (playerName == GLDG_SelPlrName_c) then GLDG_SelPlrName_c = nil
	else GLDG_SelPlrName_c = playerName end
	-- Refresh list
	GLDG_ListPlayers_c()
	-- Adjust scrollbar if needed
	GLDG_CorrectPlayerOffset_c(old, playerName)
end

------------------------------------------------------------
function GLDG_ClickPlayerBar_c(frame)
	if (not GLDG_inChannel) then return end
	-- Update offset
	if not frame then frame = getglobal(GLDG_GUI.."ChannelPlayerbar") end
	FauxScrollFrame_Update(frame, GLDG_TableSize(GLDG_SortedList_c), GLDG_NumPlrRows_c, 15)
	GLDG_PlayerOffset_c = FauxScrollFrame_GetOffset(frame)
	GLDG_ShowPlayers_c()
end

------------------------------------------------------------
function GLDG_ShowPlayerButtons_c()
	-- Set frame
	local frame = GLDG_GUI.."Channel"
	-- Hide subframes
	getglobal(frame.."SubAlias"):Hide()
	getglobal(frame.."SubMainAlt"):Hide()
	-- If no player is selected: hide all buttons
	if not GLDG_SelPlrName_c then
		getglobal(frame.."Ignore"):Hide()
		getglobal(frame.."Alias"):Hide()
		getglobal(frame.."Main"):Hide()
		getglobal(frame.."Alt"):Hide()
		getglobal(frame.."Delete"):Hide()
		return end
	if (not GLDG_inChannel) then return end
	-- Set selected player
	local p = GLDG_DataChannel[GLDG_SelPlrName_c]
	-- Ignore button
	local button = getglobal(frame.."Ignore")
	if p.ignore then button:SetText(GLDG_TXT.pbrign) else button:SetText(GLDG_TXT.pbsign) end
	button:Show()
	-- Alias button
	button = getglobal(frame.."Alias")
	button:Show()
	button:Enable()
	-- Main button
	button = getglobal(frame.."Main")
	button:Show()
	if p.alt then button:SetText(GLDG_TXT.pbpmain)
	elseif not p.main then button:SetText(GLDG_TXT.pbsmain)
	elseif (GLDG_NumAlts_c == 0) then button:SetText(GLDG_TXT.pbrmain)
	else button:Hide() end
	-- Alt button
	button = getglobal(frame.."Alt")
	button:Show()
	button:Enable()
	if p.alt then button:SetText(GLDG_TXT.pbralt)
	elseif p.main then button:Hide()
	else button:SetText(GLDG_TXT.pbsalt) end
	-- Delete button
	button = getglobal(frame.."Delete")
	button:Show()
	button:Enable()
	button:SetText(GLDG_TXT.pbdelete)
end

------------------------------------------------------------
function GLDG_ClickIgnore_c()
	if (not GLDG_inChannel) then return end
	-- Toggle ignore state for selected player
	local p = GLDG_DataChannel[GLDG_SelPlrName_c]
	local main = p.alt
	if p.main then main = GLDG_SelPlrName_c end
	local newval = nil
	if not p.ignore then newval = true end
	-- If main or alt, toggle ignore for all characters of this player
	if main then for q in pairs(GLDG_DataChannel) do if (q == main) or (GLDG_DataChannel[q].alt == main) then GLDG_DataChannel[q].ignore = newval end end
	-- Otherwise, simply toggle ignore for this one character
	else p.ignore = newval end
	-- Show updated list
	GLDG_ListPlayers_c()
end

------------------------------------------------------------
function GLDG_ClickAlias_c()
	if (not GLDG_inChannel) then return end
	-- Activate alias subframe
	GLDG_ShowPlayerButtons_c()
	this:Disable()
	getglobal(this:GetParent():GetName().."SubAlias"):Show()
end

------------------------------------------------------------
function GLDG_ClickMain_c()
	if (not GLDG_inChannel) then return end
	-- Toggle main status for selected character or promote an alt
	local p = GLDG_DataChannel[GLDG_SelPlrName_c]
	if p.main then p.main = nil
	elseif p.alt then
		local oldmain = p.alt
		p.alt = nil
		p.main = true
		for q in pairs(GLDG_DataChannel) do
			if (q == oldmain) then
				GLDG_DataChannel[q].main = nil
				GLDG_DataChannel[q].alt = GLDG_SelPlrName_c
			elseif (GLDG_DataChannel[q].alt == oldmain) then
				GLDG_DataChannel[q].alt = GLDG_SelPlrName_c end end
	else p.main = true end
	-- Show updated list
	GLDG_ListPlayers_c()
end

------------------------------------------------------------
function GLDG_ClickAlt_c()
	if (not GLDG_inChannel) then return end
	-- Prepare for scrollbar adjustment if needed
	local p = {}
	if GLDG_SelPlrName_c then p = GLDG_DataChannel[GLDG_SelPlrName_c] end
	local old = nil
	if GLDG_Data.GroupAlt then old = GLDG_GetPlayerOffset_c(GLDG_SelPlrName_c) end
	-- Toggle alt status for selected character
	local p = GLDG_DataChannel[GLDG_SelPlrName_c]
	if p.alt then p.alt = nil
	elseif (GLDG_NumMain_c == 1) then
		for q in pairs(GLDG_DataChannel) do
			if GLDG_DataChannel[q].main then
				p.alt = q
				p.ignore = GLDG_DataChannel[q].ignore
				break end end
	else	GLDG_ShowPlayerButtons_c()
		this:Disable()
		getglobal(this:GetParent():GetName().."SubMainAlt"):Show()
		return end
	-- Show updated list
	GLDG_ListPlayers_c()
	-- Adjust scrollbar if needed
	GLDG_CorrectPlayerOffset_c(old, GLDG_SelPlrName_c)
end

------------------------------------------------------------
function GLDG_ShowPlayerAlias_c(frame)
	if (not GLDG_inChannel) then return end
	-- Set title
	getglobal(frame.."Header"):SetText(string.format(GLDG_TXT.aliashead, GLDG_SelPlrName_c))
	-- Set editbox and buttons text
	local p = GLDG_DataChannel[GLDG_SelPlrName_c]
	if p.alias then
		getglobal(frame.."Set"):SetText(GLDG_TXT.update)
		getglobal(frame.."Del"):SetText(GLDG_TXT.delete)
		getglobal(frame.."Editbox"):SetText(p.alias)
	else	getglobal(frame.."Set"):SetText(GLDG_TXT.set)
		getglobal(frame.."Del"):SetText(GLDG_TXT.cancel)
		getglobal(frame.."Editbox"):SetText("") end
end

------------------------------------------------------------
function GLDG_ClickAliasSet_c(alias)
	if (not GLDG_inChannel) then return end
	if (alias == "") then return end
	GLDG_DataChannel[GLDG_SelPlrName_c].alias = alias
	GLDG_ListPlayers_c()
end

------------------------------------------------------------
function GLDG_ClickAliasRemove_c()
	if (not GLDG_inChannel) then return end
	GLDG_DataChannel[GLDG_SelPlrName_c].alias = nil
	GLDG_ListPlayers_c()
end

------------------------------------------------------------
function GLDG_ShowMainAlt_c(frame)
	if (not GLDG_inChannel) then return end
	-- Set frame and linename
	if not frame then frame = this end
	local name = frame:GetParent():GetName()
	-- Set title
	getglobal(name.."Header"):SetText(string.format(GLDG_TXT.mainhead, GLDG_SelPlrName_c))
	-- Create a sorted list of all mains
	local mainlist = {}
	local total = 0
	for p in pairs(GLDG_DataChannel) do
		if (GLDG_DataChannel[p].main and
		    (GLDG_DataGuild and not GLDG_DataGuild[p])) then
			local loc = 1
			while mainlist[loc] and (mainlist[loc] < p) do loc = loc + 1 end
			for cnt = total, loc, -1 do mainlist[cnt + 1] = mainlist[cnt] end
			total = total + 1
			mainlist[loc] = p end end
	-- Configure scrollbar
	FauxScrollFrame_Update(frame, GLDG_TableSize(mainlist), GLDG_NumSubRows_c, 15)
	local offset = FauxScrollFrame_GetOffset(frame)
	-- Set all rows
	name = name.."Line"
	for line = 1, GLDG_NumSubRows_c do
		if mainlist[line + offset] then
			getglobal(name..line.."Text"):SetText(mainlist[line + offset])
			getglobal(name..line):Enable()
		else	getglobal(name..line.."Text"):SetText("")
			getglobal(name..line):Disable() end end
end

------------------------------------------------------------
function GLDG_ClickMainAlt_c(player)
	if (not GLDG_inChannel) then return end
	-- Mark position of current selected player
	local old = GLDG_GetPlayerOffset_c(GLDG_SelPlrName_c)
	-- Make this the main for the currently selected character
	GLDG_DataChannel[GLDG_SelPlrName_c].alt = player
	GLDG_DataChannel[GLDG_SelPlrName_c].ignore = GLDG_DataChannel[player].ignore
	-- Hide the subframe window
	this:GetParent():Hide()
	-- Refresh the playerlist
	GLDG_ListPlayers_c()
	-- Adjust scrollbar if needed
	GLDG_CorrectPlayerOffset_c(old, GLDG_SelPlrName_c)
end

------------------------------------------------------------
function GLDG_GetPlayerOffset_c(playerName)
	if (not GLDG_inChannel) then return nil end
	-- Find current position number on list
	local old = nil
	for i = 1, GLDG_TableSize(GLDG_SortedList_c) do
		if (GLDG_SortedList_c[i] == playerName) then
			old = i
			break end end
	return old
end

------------------------------------------------------------
function GLDG_CorrectPlayerOffset_c(old, playerName)
	if (not GLDG_inChannel) then return end
	-- Abort if no value given
	if not old then return end
	local new = nil
	-- Find new position number on list for this selection
	for i = 1, GLDG_TableSize(GLDG_SortedList_c) do
		if (GLDG_SortedList_c[i] == playerName) then
			new = i
			break end end
	-- Abort if no longer on list or position didn't change
	if (not new) or (old == new) then return end
	-- Calculate new offset and set it
	GLDG_PlayerOffset_c = math.max(0, math.min(GLDG_PlayerOffset_c + new - old, GLDG_TableSize(GLDG_SortedList_c) - GLDG_NumPlrRows_c))
	FauxScrollFrame_SetOffset(getglobal(GLDG_GUI.."ChannelPlayerbar"), GLDG_PlayerOffset_c)
	GLDG_ListPlayers_c()
end

------------------------------------------------------------
function GLDG_ClickDelete_c()
	if (not GLDG_inChannel) then return end

	if (GLDG_SelPlrName_c) then
		-- go through list, remove this char as main for all alts
		for p in pairs(GLDG_DataChannel) do
			if (GLDG_DataChannel[p].alt and GLDG_DataChannel[p].alt == GLDG_SelPlrName_c) then
				GLDG_DataChannel[p].alt = nil
			end
		end

		-- remove char from list
		GLDG_DataChannel[GLDG_SelPlrName_c] = nil
	end

	-- Clear current name
	GLDG_SelPlrName_c = nil

	-- Show updated list
	GLDG_ListPlayers_c()
end

-----------------------------
-- _23_ Colour picker handling    --
-----------------------------
function GLDG_ColourToRGB(colour)
	local hex = string.sub(colour, 3)
	local alpha, rhex, ghex, bhex = string.sub(hex, 1,2), string.sub(hex, 3, 4), string.sub(hex, 5, 6), string.sub(hex, 7, 8)
	return tonumber(alpha, 16), tonumber(rhex, 16), tonumber(ghex, 16), tonumber(bhex, 16)
end

------------------------------------------------------------
function GLDG_ColourToRGB_perc(colour)
	local hex = string.sub(colour, 3)
	local alpha, rhex, ghex, bhex = string.sub(hex, 1,2), string.sub(hex, 3, 4), string.sub(hex, 5, 6), string.sub(hex, 7, 8)
	local a,r,g,b = tonumber(alpha, 16), tonumber(rhex, 16), tonumber(ghex, 16), tonumber(bhex, 16)
	return a/255, r/255, g/255, b/255
end

------------------------------------------------------------
function GLDG_RGBToColour(a, r, g, b)
	return string.format("|c%02X%02X%02X%02X", a, r, g, b)
end

------------------------------------------------------------
function GLDG_RGBToColour_perc(a, r, g, b)
	return string.format("|c%02X%02X%02X%02X", a*255, r*255, g*255, b*255)
end

------------------------------------------------------------
function GLDG_ShowColourPicker()
	if not GLDG_colour then GLDG_colour = "|cFFFFFFFF" end

	-- store old callbacks
	GLDG_oldColourPicked = GLDG_ColorPickerFrame.func
	GLDG_oldColourCancel = GLDG_ColorPickerFrame.cancelFunc
	GLDG_oldOpacityChanged = GLDG_ColorPickerFrame.opacityFunc

	-- define our own callbacks
	GLDG_ColorPickerFrame.func = GLDG_ColourPicked
	GLDG_ColorPickerFrame.opacityFunc = GLDG_ColourPicked
	GLDG_ColorPickerFrame.cancelFunc = GLDG_ColourCancel

	-- set up window
	local a,r,g,b = GLDG_ColourToRGB_perc(GLDG_colour)
	if (GLDG_colourName=="header") then
		GLDG_ColorPickerFrame.hasOpacity = true;
	else
		GLDG_ColorPickerFrame.hasOpacity = false;
	end
	GLDG_ColorPickerFrame.opacity = a
	GLDG_ColorPickerFrame.previousValues = {r, g, b}
	GLDG_oldOpacity = a

	GLDG_updating = true
	GLDG_UpdateColoursSwatch()
	GLDG_UpdateColoursNumbers()
	GLDG_updating = nil

	-- show window
	--GuildGreetColourFrame:Show()
	GLDG_ColorPickerFrame:Show();
	--GuildGreetColourFrame:Lower()

	GLDG_colorPickerShown = true

end

------------------------------------------------------------
function GLDG_HideColourPicker()
	-- hide window
	GLDG_ColorPickerFrame:Hide();
	--GuildGreetColourFrame:Hide()
	GLDG_colorPickerShown = nil

	-- restore old callbacks
	GLDG_ColorPickerFrame.func = GLDG_oldColourPicked
	GLDG_ColorPickerFrame.cancelFunc = GLDG_oldColourCancel
	GLDG_ColorPickerFrame.opacityFunc = GLDG_oldOpacityChanged
	GLDG_oldColourPicked = nil
	GLDG_oldColourCancel = nil
	GLDG_oldOpacityChanged = nil

	-- clear temp values
	GLDG_current_red = nil
	GLDG_current_green = nil
	GLDG_current_blue = nil
	GLDG_current_opacity = nil
	GLDG_oldOpacity = nil
end

------------------------------------------------------------
function GLDG_ColourPicked()
	if GLDG_colorPickerShown and not GLDG_updating then
		-- get values
		local r, g, b = GLDG_ColorPickerFrame:GetColorRGB();

		-- use values
		local a = 1
		if (GLDG_colourName=="header") then
			a = GLDG_OpacitySliderFrame:GetValue()
		end
		GLDG_colour = GLDG_RGBToColour_perc(a, r, g, b)
		GLDG_UpdateCurrentColour()

		if (this:GetName() == "GLDG_ColorPickerOkayButton") then
			GLDG_HideColourPicker()
		else
			GLDG_updating = true
			GLDG_UpdateColoursNumbers()
			GLDG_updating = nil
		end
	end
end

------------------------------------------------------------
function GLDG_ColourCancel(prevvals)
	-- restore values
	local r, g, b = unpack(prevvals)
	local a = GLDG_oldOpacity


	GLDG_colour = GLDG_RGBToColour_perc(a, r, g, b)
	--GLDG_Print("Restored colour is "..GLDG_colour..string.sub(GLDG_colour, 3).."|r")
	GLDG_UpdateCurrentColour()

	GLDG_HideColourPicker()
end

------------------------------------------------------------
function GLDG_UpdateColoursSwatch()
	local a,r,g,b = GLDG_ColourToRGB_perc(GLDG_colour)

	GLDG_ColorSwatch:SetTexture(r, g, b);
	GLDG_ColorPickerFrame:SetColorRGB(r, g, b);

	GLDG_ColorPickerFrame.opacity = a
	GLDG_OpacitySliderFrame:SetValue(a);
end

------------------------------------------------------------
function GLDG_UpdateColoursNumbers()
	local a,r,g,b = GLDG_ColourToRGB(GLDG_colour)

	GLDG_current_red = r
	GLDG_current_green = g
	GLDG_current_blue = b
	GLDG_current_opacity = a

	GuildGreetColourFrameRed:SetNumber(r)
	GuildGreetColourFrameGreen:SetNumber(g)
	GuildGreetColourFrameBlue:SetNumber(b)
	GuildGreetColourFrameOpacity:SetNumber(a)
end

------------------------------------------------------------
function GLDG_ColourCancelEdit(element)
	if not GLDG_updating then
		if this then
			this:HighlightText(this:GetNumLetters())
		end
		GLDG_updating = true
		GLDG_UpdateColoursSwatch()
		GLDG_updating = nil
	end
end

------------------------------------------------------------
function GLDG_ColourEnter(element, number)
	if not GLDG_updating then
		if this then
			this:HighlightText(this:GetNumLetters())
		end
		if (number >= 0 and number<=255) then
			local a,r,g,b = GLDG_ColourToRGB(GLDG_colour)

			if (element == "GuildGreetColourFrameRed") then
				GLDG_current_red = number
				GLDG_colour = GLDG_RGBToColour(a, number, g, b)
				GLDG_UpdateCurrentColour()
			elseif (element == "GuildGreetColourFrameGreen") then
				GLDG_current_green = number
				GLDG_colour = GLDG_RGBToColour(a, r, number, b)
				GLDG_UpdateCurrentColour()
			elseif (element == "GuildGreetColourFrameBlue") then
				GLDG_current_blue = number
				GLDG_colour = GLDG_RGBToColour(a, r, g, number)
				GLDG_UpdateCurrentColour()
			elseif (element == "GuildGreetColourFrameOpacity") then
				GLDG_current_opacity = number
				GLDG_colour = GLDG_RGBToColour(number, r, g, b)
				GLDG_UpdateCurrentColour()
			end
		else
			if (element == "GuildGreetColourFrameRed") then
				GuildGreetColourFrameRed:SetNumber(GLDG_current_red)
			elseif (element == "GuildGreetColourFrameGreen") then
				GuildGreetColourFrameGreen:SetNumber(GLDG_current_green)
			elseif (element == "GuildGreetColourFrameBlue") then
				GuildGreetColourFrameBlue:SetNumber(GLDG_current_blue)
			elseif (element == "GuildGreetColourFrameOpacity") then
				GuildGreetColourFrameOpacity:SetNumber(GLDG_current_opacity)
			end
		end

		GLDG_updating = true
		GLDG_UpdateColoursSwatch()
		GLDG_updating = nil
	end

end

------------------------------------------------------------
function GLDG_ColourTab(element)
	if this then
		this:HighlightText(this:GetNumLetters())
	end
	if (element == "GuildGreetColourFrameRed") then
		GuildGreetColourFrameGreen:SetFocus()
		GuildGreetColourFrameGreen:HighlightText()
	elseif (element == "GuildGreetColourFrameGreen") then
		GuildGreetColourFrameBlue:SetFocus()
		GuildGreetColourFrameBlue:HighlightText()
	elseif (element == "GuildGreetColourFrameBlue") then
		if (GLDG_ColorPickerFrame.hasOpacity) then
			GuildGreetColourFrameOpacity:SetFocus()
			GuildGreetColourFrameOpacity:HighlightText()
		else
			GuildGreetColourFrameRed:SetFocus()
			GuildGreetColourFrameRed:HighlightText()
		end
	elseif (element == "GuildGreetColourFrameOpacity") then
		GuildGreetColourFrameRed:SetFocus()
		GuildGreetColourFrameRed:HighlightText()
	end
end

------------------------------------------------------------
function GLDG_ColourUpdate(element, number)
	--GLDG_Print("Update: element ["..element.."].. - number ["..number.."]")
end


-----------------------------
-- _24_ Colour handling    --
-----------------------------
function GLDG_SetActiveColourSet(set)
	GLDG_ONLINE_COLOUR		= GLDG_DEFAULT_ONLINE_COLOUR
	GLDG_IS_OFFLINE_COLOUR		= GLDG_DEFAULT_IS_OFFLINE_COLOUR
	GLDG_GOES_OFFLINE_COLOUR	= GLDG_DEFAULT_GOES_OFFLINE_COLOUR
	GLDG_ALIAS_COLOUR		= GLDG_DEFAULT_ALIAS_COLOUR

	if (set and GLDG_Data.colours[set]) then
		GLDG_ONLINE_COLOUR		= GLDG_Data.colours[set].online
		GLDG_IS_OFFLINE_COLOUR		= GLDG_Data.colours[set].isOff
		GLDG_GOES_OFFLINE_COLOUR	= GLDG_Data.colours[set].goOff
		GLDG_ALIAS_COLOUR		= GLDG_Data.colours[set].alias
	end
end

------------------------------------------------------------
function GLDG_ColoursShow()
	local name = GLDG_GUI.."Colour"

	-- only build these lists the first time they are used to avoid excessive garbage creation
	if not GLDG_set_list then
		GLDG_set_list = {}
		GLDG_set_list["Guild"] = "guild"
		GLDG_set_list["Friends"] = "friends"
		GLDG_set_list["Channel"] = "channel"
	end
	if not GLDG_element_list then
		GLDG_element_list = {}
		GLDG_element_list["Online"] = "online"
		GLDG_element_list["IsOffline"] = "isOff"
		GLDG_element_list["GoOffline"] = "goOff"
		GLDG_element_list["Alias"] = "alias"
		GLDG_element_list["List"] = "list"
		GLDG_element_list["Relog"] = "relog"
		GLDG_element_list["New"] = "new"
		GLDG_element_list["Level"] = "lvl"
		GLDG_element_list["Rank"] = "rank"
	end

	for s in pairs(GLDG_set_list) do
		for e in pairs(GLDG_element_list) do
			local colour = GLDG_Data.colours[GLDG_set_list[s]][GLDG_element_list[e]]
			local a,r,g,b = GLDG_ColourToRGB_perc(colour)
			local texture = getglobal(name..s..e.."ColourTexture")
			texture:SetTexture(r,g,b,a)
			local text = getglobal(name..s..e.."Button")
			text:SetText(string.sub(colour, 5))
		end
	end

	local a,r,g,b = GLDG_ColourToRGB_perc(GLDG_Data.colours.help)
	local texture = getglobal(name.."HelpColourTexture")
	texture:SetTexture(r,g,b,a)
	local text = getglobal(name.."HelpButton")
	text:SetText(string.sub(GLDG_Data.colours.help, 5))

	a,r,g,b = GLDG_ColourToRGB_perc(GLDG_Data.colours.header)
	texture = getglobal(name.."HeaderColourTexture")
	texture:SetTexture(r,g,b,a)
	local text = getglobal(name.."HeaderButton")
	text:SetText(string.sub(GLDG_Data.colours.header, 3))
end

------------------------------------------------------------
function GLDG_ColourClick(name)
	if not GLDG_colorPickerShown then
		-- which element was clicked
		GLDG_setName = ""
		if (string.find(name, GLDG_GUI.."Colour".."Guild")) then
			GLDG_setName = "guild"
		elseif (string.find(name, GLDG_GUI.."Colour".."Friends")) then
			GLDG_setName = "friends"
		elseif (string.find(name, GLDG_GUI.."Colour".."Channel")) then
			GLDG_setName = "channel"
		end

		GLDG_colourName = ""
		if (string.find(name, "Online")) then
			GLDG_colourName = "online"
		elseif (string.find(name, "IsOffline")) then
			GLDG_colourName = "isOff"
		elseif (string.find(name, "GoOffline")) then
			GLDG_colourName = "goOff"
		elseif (string.find(name, "Alias")) then
			GLDG_colourName = "alias"

		elseif (string.find(name, "List")) then
			GLDG_colourName = "list"
		elseif (string.find(name, "New")) then
			GLDG_colourName = "new"
		elseif (string.find(name, "Level")) then
			GLDG_colourName = "lvl"
		elseif (string.find(name, "Rank")) then
			GLDG_colourName = "rank"
		elseif (string.find(name, "Relog")) then
			GLDG_colourName = "relog"

		elseif (string.find(name, "Help")) then
			GLDG_colourName = "help"
		elseif (string.find(name, "Header")) then
			GLDG_colourName = "header"
		end

		if (string.find(name, "Default")) then
			GLDG_ColourRestoreDefaults()
		else
			if ((GLDG_setName~="" or GLDG_colourName=="help" or GLDG_colourName=="header") and (GLDG_colourName ~="")) then
				-- get current colour
				if (GLDG_colourName=="help") then
					GLDG_colour = GLDG_Data.colours.help
				elseif (GLDG_colourName=="header") then
					GLDG_colour = GLDG_Data.colours.header
				else
					GLDG_colour = GLDG_Data.colours[GLDG_setName][GLDG_colourName]
				end

				-- open color picker
				GLDG_ShowColourPicker()
			else
				GLDG_Print("--- unknown button ["..name.."] pressed ---")
			end
		end
	else
		GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_TXT.colourConfig)
	end
end

------------------------------------------------------------
function  GLDG_UpdateCurrentColour()
	--GLDG_Print("Colour is "..GLDG_colour..string.sub(GLDG_colour, 3).."|r")

	if (not GLDG_setName) or (not GLDG_colourName) then return end

	if ((GLDG_setName~="" or GLDG_colourName=="help" or GLDG_colourName=="header") and (GLDG_colourName ~="")) then
		-- set value
		if (GLDG_colourName=="help") then
			GLDG_Data.colours.help = GLDG_colour
		elseif (GLDG_colourName=="header") then
			GLDG_Data.colours.header = GLDG_colour
		else
			GLDG_Data.colours[GLDG_setName][GLDG_colourName] = GLDG_colour
		end

		-- update colour cache
		GLDG_SetActiveColourSet("guild")
	end

	GLDG_ColoursShow()
end


------------------------------------------------------------
function GLDG_ColourRestoreDefaults()
	if not GLDG_colorPickerShown then
		GLDG_Data.colours.help = GLDG_DEFAULT_HELP_COLOUR
		GLDG_Data.colours.header = GLDG_DEFAULT_HEADER_COLOUR

		GLDG_Data.colours.guild.online = GLDG_DEFAULT_ONLINE_COLOUR
		GLDG_Data.colours.guild.isOff = GLDG_DEFAULT_IS_OFFLINE_COLOUR
		GLDG_Data.colours.guild.goOff = GLDG_DEFAULT_GOES_OFFLINE_COLOUR
		GLDG_Data.colours.guild.alias = GLDG_DEFAULT_ALIAS_COLOUR
		GLDG_Data.colours.guild.list = GLDG_DEFAULT_LIST_COLOUR
		GLDG_Data.colours.guild.new = GLDG_DEFAULT_NEW_COLOUR
		GLDG_Data.colours.guild.lvl = GLDG_DEFAULT_LVL_COLOUR
		GLDG_Data.colours.guild.rank = GLDG_DEFAULT_RANK_COLOUR
		GLDG_Data.colours.guild.relog = GLDG_DEFAULT_RELOG_COLOUR

		GLDG_Data.colours.friends.online = GLDG_DEFAULT_ONLINE_COLOUR
		GLDG_Data.colours.friends.isOff = GLDG_DEFAULT_IS_OFFLINE_COLOUR
		GLDG_Data.colours.friends.goOff = GLDG_DEFAULT_GOES_OFFLINE_COLOUR
		GLDG_Data.colours.friends.alias = GLDG_DEFAULT_ALIAS_COLOUR
		GLDG_Data.colours.friends.list = GLDG_DEFAULT_LIST_COLOUR
		GLDG_Data.colours.friends.new = GLDG_DEFAULT_NEW_COLOUR
		GLDG_Data.colours.friends.lvl = GLDG_DEFAULT_LVL_COLOUR
		GLDG_Data.colours.friends.rank = GLDG_DEFAULT_RANK_COLOUR
		GLDG_Data.colours.friends.relog = GLDG_DEFAULT_RELOG_COLOUR

		GLDG_Data.colours.channel.online = GLDG_DEFAULT_ONLINE_COLOUR
		GLDG_Data.colours.channel.isOff = GLDG_DEFAULT_IS_OFFLINE_COLOUR
		GLDG_Data.colours.channel.goOff = GLDG_DEFAULT_GOES_OFFLINE_COLOUR
		GLDG_Data.colours.channel.alias = GLDG_DEFAULT_ALIAS_COLOUR
		GLDG_Data.colours.channel.list = GLDG_DEFAULT_LIST_COLOUR
		GLDG_Data.colours.channel.new = GLDG_DEFAULT_NEW_COLOUR
		GLDG_Data.colours.channel.lvl = GLDG_DEFAULT_LVL_COLOUR
		GLDG_Data.colours.channel.rank = GLDG_DEFAULT_RANK_COLOUR
		GLDG_Data.colours.channel.relog = GLDG_DEFAULT_RELOG_COLOUR

		-- update colour cache
		GLDG_SetActiveColourSet("guild")

		GLDG_ColoursShow()
	else
		GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_TXT.colourDefault)
	end
end


--------------------------
-- _25_ Display Help Tooltip
--------------------------

function GLDG_ShowHelpToolTip(element)
	if not element then return end

	local name = ""

	-- cut off leading frame name
	if (string.find(element, GLDG_GUI)) then
		name = string.sub(element, string.len(GLDG_GUI)+1)
	elseif (string.find(element, GLDG_COLOUR)) then
		name = string.sub(element, string.len(GLDG_COLOUR)+1)
	elseif (string.find(element, GLDG_LIST)) then
		name = element
	end

	-- cut off trailing number in case of line and collect
	local s,e = string.find(name, "Line");
	if (s and e) then
		name = string.sub(name, 0, e)
	end
	s,e = string.find(name, "Collect");
	if (s and e) then
		name = string.sub(name, 0, e)
	end

	-- cut off colour button/texture
	if (string.find(name, "Colour") == 1) then
		-- ["ColourGuildNewButton"] = true,
		s,e = string.find(name, "Button");
		if (s and e) then
			name = string.sub(name, 0, s-1)
		end
		-- ["ColourGuildNewColour"] = true,
		s,e = string.find(name, "Colour", 2);	-- start at 2 to skip the initial Colour
		if (s and e) then
			name = string.sub(name, 0, s-1)
		end
	end


	local tip = ""
	local head = ""
	if (GLDG_TXT.elements and
	    GLDG_TXT.elements.name and
	    GLDG_TXT.elements.tip and
	    GLDG_TXT.elements.name[name] and
	    GLDG_TXT.elements.tip[name]) then
		tip = GLDG_TXT.elements.tip[name]
		head = GLDG_TXT.elements.name[name]

		if (GLDG_Data.needsTip and GLDG_Data.needsTip[name]) then
			GLDG_Data.needsTip[name] = nil
		end
	else
		if (not GLDG_Data.needsTip) then
			GLDG_Data.needsTip = {}
		end
		GLDG_Data.needsTip[name] = true
	end

	GameTooltip_SetDefaultAnchor(GameTooltip, this)
	if (head ~= "") then
		GameTooltip:SetText(head, 1, 1, 0.5, 1.0, 1)
--		GameTooltip:AddLine(name, 1, 0, 0, 1.0, 1)
		GameTooltip:AddLine(tip, 1, 1, 1, 1.0, 1)
--	else
--		GameTooltip:SetText(element, 1, 1, 0.5, 1.0, 1)
--		GameTooltip:AddLine(name, 1, 1, 1, 1.0, 1)
	end

	GameTooltip:Show()
end


--------------------------
-- _26_ Chat name extension
--------------------------

------------------------------------------------------------
function GLDG_ChatOnEvent(event)
--[[
	taken from "http://www.wowwiki.com/Events_C_(Cancel,_Character,_Chat,_Cinematic,_Clear,_Close,_Confirm,_Corpse,_Craft,_Current,_Cursor,_CVar)"

	"CHAT_MSG_CHANNEL"
	Called when the client recieves a channel message.
	arg1: chat message
	arg2: author
	arg3: language
	arg4: channel name with number
	arg5: unknown (empty string)
	arg6: unknown (empty string)
	arg7: unknown (appears to be channel number)
	arg8: channel number
	arg9: channel name without number

	"CHAT_MSG_GUILD"
	Fired when a message is sent or received in the Guild channel.
	arg1: Message that was sent
	arg2: Author
	arg3: Language that the message was sent in

	"CHAT_MSG_OFFICER"
	Fired when a message is sent or received in the Guild Officer channel.
	arg1: Message that was received
	arg2: Author
	arg3: Language used

	"CHAT_MSG_WHISPER"
	Fired when a whisper is received from another player.
	The rest of the arguments appear to be nil
	arg1: Message received
	arg2: Author
	arg3: Language (or nil if universal, like messages from GM)
	arg6: status (like "DND" or "GM")
]]

	if GLDG_Data.ExtendChat then
		local main = ""
		if (event) then
			-- guild members can post to all three (officers even four) channels
			if (((event == "CHAT_MSG_GUILD") and arg2) or
			    ((event == "CHAT_MSG_OFFICER") and arg2) or
			    ((event == "CHAT_MSG_CHANNEL") and arg9 and (arg9 == GLDG_ChannelName) and arg2) or
			    ((event == "CHAT_MSG_WHISPER") and arg2)) then
				--GLDG_Print("Event ["..event.."] from player ["..arg2.."]")
				if (GLDG_DataGuild[arg2] and not GLDG_DataGuild[arg2].ignore) then
					if GLDG_DataGuild[arg2].alt then
						main = GLDG_DataGuild[arg2].alt;
						if GLDG_Data.ExtendAlias then
							if GLDG_DataGuild[main] and GLDG_DataGuild[main].alias then
								main = GLDG_DataGuild[main].alias
							end
						end
					elseif GLDG_Data.ExtendMain and GLDG_DataGuild[arg2].main then
						main = arg2
						if GLDG_Data.ExtendAlias then
							if GLDG_DataGuild[main] and GLDG_DataGuild[main].alias then
								main = GLDG_DataGuild[main].alias
							end
						end
					end
				end
			end
			-- channel members can use two of three channels
			if ((main=="") and
			    (((event == "CHAT_MSG_CHANNEL") and arg9 and (arg9 == GLDG_ChannelName) and arg2) or
			     ((event == "CHAT_MSG_WHISPER") and arg2))) then
				--GLDG_Print("Event ["..event.."] from player ["..arg2.."]")
				if (GLDG_inChannel and GLDG_DataChannel[arg2] and not GLDG_DataChannel[arg2].ignore) then
					if GLDG_DataChannel[arg2].alt then
						main = GLDG_DataChannel[arg2].alt;
						if GLDG_Data.ExtendAlias then
							if GLDG_DataChannel[main] and GLDG_DataChannel[main].alias then
								main = GLDG_DataChannel[main].alias
							end
						end
					elseif GLDG_Data.ExtendMain and GLDG_DataChannel[arg2].main then
						main = arg2
						if GLDG_Data.ExtendAlias then
							if GLDG_DataChannel[main] and GLDG_DataChannel[main].alias then
								main = GLDG_DataChannel[main].alias
							end
						end
					end
				end
			end
			-- friends can only whisper
			if ((main=="") and
			    ((event == "CHAT_MSG_WHISPER") and arg2)) then
				--GLDG_Print("Event ["..event.."] from player ["..arg2.."]")
				if (GLDG_Data.UseFriends and GLDG_DataFriends.friends[arg2] and not GLDG_DataFriends.friends[arg2].ignore) then
					if GLDG_DataFriends.friends[arg2].alt then
						main = GLDG_DataFriends.friends[arg2].alt;
						if GLDG_Data.ExtendAlias then
							if GLDG_DataFriends.friends[main] and GLDG_DataFriends.friends[main].alias then
								main = GLDG_DataFriends.friends[main].alias
							end
						end
					elseif GLDG_Data.ExtendMain and GLDG_DataFriends.friends[arg2].main then
						main = arg2
						if GLDG_Data.ExtendAlias then
							if GLDG_DataFriends.friends[main] and GLDG_DataFriends.friends[main].alias then
								main = GLDG_DataFriends.friends[main].alias
							end
						end
					end
				end
			end

			-- use this to discover any other events being sent
			if ((event ~= "CHAT_MSG_GUILD") and
			    (event ~= "CHAT_MSG_OFFICER") and
			    (event ~= "CHAT_MSG_CHANNEL") and
			    (event ~= "CHAT_MSG_WHISPER")) then
				--GLDG_Print("Event ["..event.."] is boring")
			end
		end

		if ((main ~= "") and arg1 and arg2 and (arg2 ~= GLDG_Player)) then
			arg1 = GLDG_Data.colours.help.."{"..main.."}|r "..arg1
		end
	end

	if (event) then
		GLDG_originalChatEventHandler(event)
	end
end

--------------------------
-- _27_ Debug dumping
--------------------------

------------------------------------------------------------
function GLDG_Dump(msg)
	if not msg then return end
	if not GLDGD_Dump then return end

	if GLDGD_InitComplete then
		if GLDG_Backlog_Index and (GLDG_Backlog_Index > 0) then
			GLDGD_DumpMsg("--- Starting to log backlog ---")
				for index in pairs(GLDG_Backlog) do
					GLDGD_DumpMsg("{"..GLDG_Backlog[index].."}")
				end
			GLDGD_DumpMsg("--- Done with logging backlog ---")
			GLDG_Backlog = nil
			GLDG_Backlog_Index = nil
		end
		GLDGD_DumpMsg(msg)
	else
		if not GLDG_Backlog then
			GLDG_Backlog = {}
			GLDG_Backlog_Index = 0
		end
		GLDG_Backlog_Index = GLDG_Backlog_Index + 1
		GLDG_Backlog[GLDG_Backlog_Index] = msg
	end
end

------------------------------------------------------------
function GLDG_ShowDump()
	if GLDGD_InitComplete and GLDGD_Dump then
		getglobal(GLDG_GUI.."SettingsDebug".."EnableDumpText"):SetText(GLDG_TXT.enableDump)
		getglobal(GLDG_GUI.."SettingsDebug".."VerboseDumpText"):SetText(GLDG_TXT.verboseDump)
		getglobal(GLDG_GUI.."SettingsDebug".."DumpSetText"):SetText(string.format(GLDG_TXT.dumpSet, GLDGD_Dump.CurrentDumpSet, option))
		getglobal(GLDG_GUI.."SettingsDebug".."CurrentIndexText"):SetText(string.format(GLDG_TXT.currentIndex, GLDGD_Dump.CurrentSetIndex-1, GLDGD_Dump.CurrentDumpSet, option))
		getglobal(GLDG_GUI.."SettingsDebug".."ClearButton"):SetText(string.format(GLDG_TXT.btnClear, GLDGD_Dump.CurrentDumpSet, option))
		getglobal(GLDG_GUI.."SettingsDebug".."ClearAllButton"):SetText(GLDG_TXT.btnClearAll)
		getglobal(GLDG_GUI.."SettingsDebug".."NewButton"):SetText(GLDG_TXT.btnNew)

		getglobal(GLDG_GUI.."SettingsDebug".."EnableDumpBox"):SetChecked(GLDGD_Dump.Enabled)
		getglobal(GLDG_GUI.."SettingsDebug".."VerboseDumpBox"):SetChecked(GLDGD_Dump.Verbose)
		getglobal(GLDG_GUI.."SettingsDebug".."EnableDumpBox"):Enable()
		getglobal(GLDG_GUI.."SettingsDebug".."VerboseDumpBox"):Enable()
		getglobal(GLDG_GUI.."SettingsDebug".."ClearButton"):Enable()
		getglobal(GLDG_GUI.."SettingsDebug".."ClearAllButton"):Enable()
		getglobal(GLDG_GUI.."SettingsDebug".."NewButton"):Enable()
	else
		getglobal(GLDG_GUI.."SettingsDebug".."EnableDumpText"):SetText(GLDG_TXT.noDumping)
		getglobal(GLDG_GUI.."SettingsDebug".."VerboseDumpText"):SetText(GLDG_TXT.noDumping)
		getglobal(GLDG_GUI.."SettingsDebug".."DumpSetText"):SetText(GLDG_TXT.noDumpingShort)
		getglobal(GLDG_GUI.."SettingsDebug".."CurrentIndexText"):SetText(GLDG_TXT.noDumpingShort)
		getglobal(GLDG_GUI.."SettingsDebug".."ClearButton"):SetText(GLDG_TXT.noDumpingShort)
		getglobal(GLDG_GUI.."SettingsDebug".."ClearAllButton"):SetText(GLDG_TXT.noDumpingShort)
		getglobal(GLDG_GUI.."SettingsDebug".."NewButton"):SetText(GLDG_TXT.noDumpingShort)

		getglobal(GLDG_GUI.."SettingsDebug".."EnableDumpBox"):SetChecked(false)
		getglobal(GLDG_GUI.."SettingsDebug".."VerboseDumpBox"):SetChecked(false)
		getglobal(GLDG_GUI.."SettingsDebug".."EnableDumpBox"):Disable()
		getglobal(GLDG_GUI.."SettingsDebug".."VerboseDumpBox"):Disable()
		getglobal(GLDG_GUI.."SettingsDebug".."ClearButton"):Disable()
		getglobal(GLDG_GUI.."SettingsDebug".."ClearAllButton"):Disable()
		getglobal(GLDG_GUI.."SettingsDebug".."NewButton"):Disable()
	end
end

------------------------------------------------------------
function GLDG_ClickDumpEnable(checked)
	if GLDGD_InitComplete and GLDGD_Dump then
		GLDGD_Dump.Enabled = checked

		if GLDGD_Status() then
			GLDGD_Status()
		end
	end
end

------------------------------------------------------------
function GLDG_ClickDumpVerbose(checked)
	if GLDGD_InitComplete and GLDGD_Dump then
		GLDGD_Dump.Verbose = checked

		if GLDGD_Status() then
			GLDGD_Status()
		end
	end
end

------------------------------------------------------------
function GLDG_ClickDumpClear()
	if GLDGD_InitComplete and GLDGD_Clear() then
		GLDGD_Clear()
	end
	GLDG_ShowDump()
end

------------------------------------------------------------
function GLDG_ClickDumpClearAll()
	if GLDGD_InitComplete and GLDGD_ClearAll() then
		GLDGD_ClearAll()
	end
	GLDG_ShowDump()
end

------------------------------------------------------------
function GLDG_ClickDumpNew()
	if GLDGD_InitComplete and GLDGD_New() then
		GLDGD_New()
	end
	GLDG_ShowDump()
end

--------------------------
-- _28_ Interface reloading (currently unused but left in code for future use)
--------------------------
function GLDG_QueryReloadUI()
	StaticPopup_Show("GLDG_RELOAD");
end

------------------------------------------------------------
function GLDG_PrepareReloadQuestion()
	StaticPopupDialogs["GLDG_RELOAD"] = {
		text = GLDG_TXT.reloadQuestion,
		button1 = GLDG_TXT.reloadNow,
		button2 = GLDG_TXT.later,
		OnAccept = function()
			ReloadUI();
			end,
		OnCancel = function()
			GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|cFFFF0000 "..GLDG_TXT.reload)
			end,
		timeout = 0,
		whileDead = 1,
		hideOnEscape = 1,
		sound = "igQuestFailed",
	};
end

--------------------------
-- _29_ urbin addon listing
--------------------------
function GLDG_RegisterUrbinAddon(name, about)
	if (not name) then
		return
	end
	if (not URBIN_AddonList) then
		URBIN_AddonList = {}
	end
	URBIN_AddonList[name] = {}
	URBIN_AddonList[name].name = name
	URBIN_AddonList[name].about = about
end

function GLDG_ListUrbinAddons(name)
	if (not URBIN_AddonList) then
		return
	end

	local addons = ""
	for p in pairs(URBIN_AddonList) do
		if (URBIN_AddonList[p].name ~= name) then
			if (addons ~= "") then
				addons = addons..", "
			end
			addons = addons..URBIN_AddonList[p].name
		end
	end

	if (addons ~= "") then
		GLDG_Print(" ", true);
		GLDG_Print("  "..GLDG_TXT.urbin..": "..GLDG_Data.colours.help..addons.."|r", true);
	end
end

function GLDG_ListUrbinAddonDetails()
	for p in pairs(URBIN_AddonList) do
		if (URBIN_AddonList[p].about) then
			URBIN_AddonList[p].about(true)
		end
	end
end

--------------------------
-- _30_ big brother
--------------------------
function GLDG_BB(flag)
	GLDG_Data.BigBrother = flag
	if (not flag) then
		GLDG_BigBrother = nil
	end
end

function GLDG_ShowBigBrother()
	if (GLDG_Data.BigBrother and GLDG_BigBrother) then
		GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_TXT.bigBrother5)
		for p in pairs(GLDG_BigBrother) do
			GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..p..": "..GLDG_BigBrother[p])
		end
	else
		GLDG_Print(GLDG_Data.colours.help..GLDG_NAME..":|r "..GLDG_TXT.bigBrother6)
	end
end
