--[[
  Guild Event Manager (GEM) - Version 3
  by Kiki from European "Conseil des Ombres" (Horde)
  ----------------------------------------
  Default Sorting plugin - Uses subscription TimeStamp as priority, and Minimum limits
]]

if ( GetLocale() == "frFR" ) then
GEM3_SORT_NAME = "Date et minimas";
GEM3_SORT_HELP = "Tri les joueurs en fonction de leur date d'inscription en respectant les minimas de classe/role.";
--[[elseif ( GetLocale() == "deDE" ) then
GEM3_SORT_NAME = "Datum";
GEM3_SORT_HELP = "Sortiert Spieler unter Beachtung ihrer Anmeldezeit.";]]
else
GEM3_SORT_NAME = "Date and min";
GEM3_SORT_HELP = "Sort players using their subscription time, filling minimums first.";
end
--------------- Internal functions ---------------

local function _GEM3_SORT_DpsRoomForMe(event,player)
  if(player.role == GEM3_ROLE_CDPS or player.role == GEM3_ROLE_RDPS)
  then
    local dps_lim = event.roles[GEM3_ROLE_DPS];
    local count = event.roles[GEM3_ROLE_CDPS].sort_stamp_count + event.roles[GEM3_ROLE_RDPS].sort_stamp_count;
    if(dps_lim.max and count > dps_lim.max) -- Dps limit reached
    then
      return false;
    end
  end
  return true;
end

local function _GEM3_SORT_CheckLimits(event,player,count)
  local clas_lim = event.classes[player.class];
  local rol_lim = event.roles[player.role];

  -- Check if exceeding class limit
  if(clas_lim.max and clas_lim.sort_stamp_count > clas_lim.max) -- Class limit reached
  then
    GEM3_ChatDebug(GEM3_DEBUG_SORT,"_GEM3_SORT_CheckLimits: Class limit reached, check minimas");
    if((clas_lim.min and clas_lim.sort_stamp_count <= clas_lim.min)
       or (rol_lim and rol_lim.min and rol_lim.sort_stamp_count <= rol_lim.min))
    then
      GEM3_ChatDebug(GEM3_DEBUG_SORT,"_GEM3_SORT_CheckLimits: Minima fails if I remove this player, keeping him as titular");
      return true;
    end
    return false;
  elseif(rol_lim and rol_lim.max and rol_lim.sort_stamp_count > rol_lim.max) -- Role limit reached
  then
    GEM3_ChatDebug(GEM3_DEBUG_SORT,"_GEM3_SORT_CheckLimits: Role limit reached, check minimas");
    if((rol_lim.min and rol_lim.sort_stamp_count <= rol_lim.min)
       or (clas_lim.min and clas_lim.sort_stamp_count <= clas_lim.min))
    then
      GEM3_ChatDebug(GEM3_DEBUG_SORT,"_GEM3_SORT_CheckLimits: Minima fails if I remove this player, keeping him as titular");
      return true;
    end
    return false;
  elseif(not _GEM3_SORT_DpsRoomForMe(event,player))
  then
    GEM3_ChatDebug(GEM3_DEBUG_SORT,"_GEM3_SORT_CheckLimits: DPS limit reached");
    return false;
  elseif(count > event.max_count) -- Global limit reached
  then
    if((clas_lim.min and clas_lim.sort_stamp_count <= clas_lim.min)
       or (rol_lim and rol_lim.min and rol_lim.sort_stamp_count <= rol_lim.min))
    then
      GEM3_ChatDebug(GEM3_DEBUG_SORT,"_GEM3_SORT_CheckLimits: Global limit reached but minima fails if I remove this player, keeping him as titular");
      return true;
    end
    GEM3_ChatDebug(GEM3_DEBUG_SORT,"_GEM3_SORT_CheckLimits: Global limit reached");
    return false;
  end
  return true; -- Check is OK
end

local function _GEM3_SORT_CheckMaxs(event,player,count)
  local clas_lim = event.classes[player.class];
  local rol_lim = event.roles[player.role];

  -- Check if exceeding class limit
  if(clas_lim.max and clas_lim.sort_stamp_count > clas_lim.max) -- Class limit reached
  then
    return false;
  elseif(rol_lim and rol_lim.max and rol_lim.sort_stamp_count > rol_lim.max) -- Role limit reached
  then
    return false;
  elseif(not _GEM3_SORT_DpsRoomForMe(event,player))
  then
    return false;
  elseif(count > event.max_count) -- Global limit reached
  then
    return false;
  end
  return true; -- Check is OK
end

local function _GEM3_SORT_SortList(tab1,tab2)
  if(tab1.lead_force_queue == GEM3_SUB_FORCE_QUEUE_TITULAR) -- 1 forced titular
  then
    if(tab2.lead_force_queue ~= GEM3_SUB_FORCE_QUEUE_TITULAR) -- but 2 not forced
    then
      return true; -- Sort 1 before
    end
  elseif(tab2.lead_force_queue == GEM3_SUB_FORCE_QUEUE_TITULAR) -- 2 forced titular (but not 1, otherwise it would have been trapped before)
  then
    return false; -- Sort 2 before
  end
  return tab1.main_time < tab2.main_time;
end

local function _GEM3_SORT_SetTitular(event,tits,tab)
  local clas_lim = event.classes[tab.class];
  local rol_lim = event.roles[tab.role];

  clas_lim.sort_stamp_count = clas_lim.sort_stamp_count + 1; -- Increase class limit
  if(rol_lim)
  then
    rol_lim.sort_stamp_count = rol_lim.sort_stamp_count + 1; -- Increase role limit
  end
  tinsert(tits,tab);
end

local function _GEM3_SORT_SetSubstitute(event,subs,tab)
  tinsert(subs,tab);
end

local function _GEM3_SORT_MoveTitularToSubstitute(event,tits,subs,tab,pos)
  local clas_lim = event.classes[tab.class];
  local rol_lim = event.roles[tab.role];

  clas_lim.sort_stamp_count = clas_lim.sort_stamp_count - 1; -- Decrease class limit
  if(rol_lim)
  then
    rol_lim.sort_stamp_count = rol_lim.sort_stamp_count - 1; -- Decrease role limit
  end

  tinsert(subs,tab);
  tremove(tits,pos);
end

local function _GEM3_SORT_Sort(event)
  local tits = GA_GetTable();
  local subs = GA_GetTable();
  local repl = GA_GetTable();
  local rejs = GA_GetTable();

  GEM3_ChatDebug(GEM3_DEBUG_SORT,"Plugin Stamp: Entering Sorting function");
  
  -- Reset some internal data
  for n,lim in pairs(event.classes)
  do
    lim.sort_stamp_count = 0;
  end
  for n,lim in pairs(event.roles)
  do
    lim.sort_stamp_count = 0;
  end

  -- Create list of all players eligible for Titular and Substitutes queues
  for name,tab in pairs(event.players)
  do
    GEM3_ChatDebug(GEM3_DEBUG_SORT,"Plugin Stamp: Processing player "..name..": LeadForced="..tostring(tab.lead_force_queue).." Forced="..tostring(tab.force_queue));
    if(tab.lead_force_queue == GEM3_SUB_FORCE_QUEUE_TITULAR and tab.force_queue ~= GEM3_SUB_FORCE_QUEUE_NOT_COMING) -- Forced Tit by leader, and still coming
    then
      _GEM3_SORT_SetTitular(event,tits,tab);
      GEM3_ChatDebug(GEM3_DEBUG_SORT,"Plugin Stamp: Forced titular by leader");
    elseif(tab.lead_force_queue == GEM3_SUB_FORCE_QUEUE_SUBSTITUTE and tab.force_queue ~= GEM3_SUB_FORCE_QUEUE_NOT_COMING) -- Forced Sub by leader, and still coming
    then
      _GEM3_SORT_SetSubstitute(event,subs,tab);
      GEM3_ChatDebug(GEM3_DEBUG_SORT,"Plugin Stamp: Forced substiture by leader");
    elseif(tab.lead_force_queue == GEM3_SUB_FORCE_QUEUE_REPLACEMENT and tab.force_queue ~= GEM3_SUB_FORCE_QUEUE_NOT_COMING) -- Forced Repl by leader, and still coming
    then
      tinsert(repl,tab);
      GEM3_ChatDebug(GEM3_DEBUG_SORT,"Plugin Stamp: Forced replacement by leader");
    elseif(tab.lead_force_queue == GEM3_SUB_FORCE_QUEUE_NOT_COMING) -- Forced NC by leader
    then
      tinsert(rejs,tab);
      GEM3_ChatDebug(GEM3_DEBUG_SORT,"Plugin Stamp: Forced reject by leader");
    elseif(tab.lead_force_queue == nil)
    then
      GEM3_ChatWarning("Plugin Stamp: tab.lead_force_queue is nil for "..name);
    else -- Use player's queue (not forced by leader, or player choose NOT_COMING)
      if(tab.force_queue == GEM3_SUB_FORCE_QUEUE_TITULAR) -- Player wants to be titular
      then
        GEM3_ChatDebug(GEM3_DEBUG_SORT,"Plugin Stamp: Player wants any available queue");
        _GEM3_SORT_SetTitular(event,tits,tab);
      elseif(tab.force_queue == GEM3_SUB_FORCE_QUEUE_SUBSTITUTE) -- Player want to be forced into Sub
      then
        _GEM3_SORT_SetSubstitute(event,subs,tab);
        GEM3_ChatDebug(GEM3_DEBUG_SORT,"Plugin Stamp: Forced substitute by player");
      elseif(tab.force_queue == GEM3_SUB_FORCE_QUEUE_REPLACEMENT) -- Player want to be forced into Repl
      then
        tinsert(repl,tab);
        GEM3_ChatDebug(GEM3_DEBUG_SORT,"Plugin Stamp: Forced replacement by player");
      elseif(tab.force_queue == GEM3_SUB_FORCE_QUEUE_NOT_COMING) -- Player don't want to come
      then
        tinsert(rejs,tab);
        GEM3_ChatDebug(GEM3_DEBUG_SORT,"Plugin Stamp: Forced reject by player");
      end
    end
  end

  -- Sort tits by stamp
  table.sort(tits,_GEM3_SORT_SortList);
  
  local i = #tits; -- Start from last
  
  -- Move limit exceeding Tits intos Subs - Should only happen if too many "forced tit by leader" players
  while(i ~= 0)
  do
    local tab = tits[i];
    if(not _GEM3_SORT_CheckLimits(event,tab,#tits))
    then
      _GEM3_SORT_MoveTitularToSubstitute(event,tits,subs,tab,i);
    end
    i = i - 1; -- Next!
  end
  
  -- Sort other queues by stamp
  table.sort(subs,_GEM3_SORT_SortList);
  table.sort(repl,_GEM3_SORT_SortList);
  table.sort(rejs,_GEM3_SORT_SortList);
  
  local consistency = true;
  -- Set queue pos for all queues
  for i,tab in ipairs(tits)
  do
    tab.current_queue = GEM3_SUB_STATE_TITULAR;
    tab.queue_pos = i;
    GEM3_ChatDebug(GEM3_DEBUG_SORT,"Plugin Stamp: Titular #"..i.." is "..tab.name);
    if(not _GEM3_SORT_CheckMaxs(event,tab,#tits))
    then
      GEM3_ChatDebug(GEM3_DEBUG_SORT,"Plugin Stamp: Min/Max consistency failure detected");
      consistency = false;
    end
  end
  for i,tab in ipairs(subs)
  do
    tab.current_queue = GEM3_SUB_STATE_SUBSTITUTE;
    tab.queue_pos = i;
    GEM3_ChatDebug(GEM3_DEBUG_SORT,"Plugin Stamp: Substitute #"..i.." is "..tab.name);
  end
  for i,tab in ipairs(repl)
  do
    tab.current_queue = GEM3_SUB_STATE_REPLACEMENT;
    tab.queue_pos = i;
    GEM3_ChatDebug(GEM3_DEBUG_SORT,"Plugin Stamp: Replacement #"..i.." is "..tab.name);
  end
  for i,tab in ipairs(rejs)
  do
    tab.current_queue = GEM3_SUB_STATE_NOT_COMING;
    tab.queue_pos = i;
    GEM3_ChatDebug(GEM3_DEBUG_SORT,"Plugin Stamp: NotComing #"..i.." is "..tab.name);
  end

  -- Release tables
  GA_ReleaseTable(tits);
  GA_ReleaseTable(subs);
  GA_ReleaseTable(repl);
  GA_ReleaseTable(rejs);

  if(not consistency)
  then
    -- GUI Callback
    GEM3_TriggerCallback("OnSortingInconsistency",event);
  end
  GEM3_ChatDebug(GEM3_DEBUG_SORT,"Plugin Stamp: Leaving Sorting function");
end


--------------- Plugin structure ---------------

GEM3_SORT_Stamp = {
  --[[
    Name parameter. [MANDATORY]
     Displayed name of the sorting plugin.
  ]]
  Name = GEM3_SORT_NAME;
  
  --[[
    SortType parameter. [MANDATORY]
     Internal code for the sorting type (must be unique).
     Please request a unique SortType value to Kiki, if you plan on creating a sorting plugin yourself.
  ]]
  SortType = 1;
  
  --[[
    Subscribers sorting function. [MANDATORY]
     Sorts all passed players in two lists.
     Params :
      - event : Event (STRUCT)
  ]]
  Sort = function(event)
    _GEM3_SORT_Sort(event);
  end;
  
  --[[
    Configure function. [OPTIONAL]
     Configures the plugin.
  ]]
  --Configure = function()
  --end;

  --[[
    Help parameter. [OPTIONAL]
     Help string displayed when you mouse over the sort type.
  ]]
  Help = GEM3_SORT_HELP;
  
  --[[
    Default parameter. [MUST NOT BE SET]
     Sets this plugin as the default one. Must only be set by the "Stamp" plugin.
  ]]
  Default = true;
};

-- Register Plugin
GEM3_SORT_RegisterPlugin(GEM3_SORT_Stamp);

