--[[
  Group Analyse by Kiki - European Cho'gall (Alliance) - Conseil des Ombres (Horde)
   
  Notes : 
    This addon should be used by all addons that hook the RAID_ROSTER_UPDATE event
   to retrieve raiders data using the GetRaidRosterInfo function.
   Calling this function in every addon lead to small freezes everytime a member joins/leaves/zones,
   while just reading data from a struct does not.
   All this because it takes much much less time to read a variable from a structure, than calling Blizzard functions.
  
  Usage :
   During the OnLoad of your AddOn, call the GroupAnalyse.RegisterForEvents(callback_func) function.
   You must pass a "function" parameter, which will be called everytime :
    - A new member joins your group/raid        (GroupAnalyse.EVENT_MEMBER_JOINED event)
    - A member leaves your group/raid           (GroupAnalyse.EVENT_MEMBER_LEFT event)
    - Group mode has changed (solo-group-raid)  (GroupAnalyse.EVENT_GROUP_MODE_CHANGED event)
    - Members datas changed                     (GroupAnalyse.EVENT_INFOS_CHANGED event)
   The function prototype must be "function Callback_Func(event,param)"
   Event being one of the following event, and param being :
    - GroupAnalyse.EVENT_MEMBER_JOINED : Name of the new member
    - GroupAnalyse.EVENT_MEMBER_LEFT : Name of the leaving member
    - GroupAnalyse.EVENT_GROUP_MODE_CHANGED : New mode - One of GroupAnalyse.MODE_SOLO, GroupAnalyse.MODE_GROUP or GroupAnalyse.MODE_RAID
    - GroupAnalyse.EVENT_INFOS_CHANGED : None

   You can then globally access the GroupAnalyse.Members[] struct (indexed by members name) to retrieve group informations such as :
    - STRING fullname  : Unit's full name (Name-Realm) in cross-server battlegrounds (equals to name out of BG, or from your realm) (BG RAID only)
    - STRING unitid    : UnitId, used by almost all unit functions like UnitHealth()
    - INT    rank      : Current group rank - 0=normal member, 1=raid assist, 2=group or raid leader
    - INT    subgrp    : Subgrp in raid mode. 1 if solo or in group
    - INT    level     : Level of the member
    - STRING class     : International class name of the member (like "PRIEST", "PALADIN", "DRUID"...)
    - STRING zone      : Current zone. Might be nil if not in raid
    - BOOL   online    : True if member is online, false otherwise
    - BOOL   isdead    : True if member is dead or in ghost form, false otherwise
    - BOOL   ischarmed : True if unit is charmed, false otherwise (not mind controlled)
    - STRING role      : Unit's role in the raid ("maintank", "mainassist", or "") (RAID only)
    - BOOL   isML      : True if unit is the master loot, false otherwise (RAID only)

  TODO :

 ChangeLog :
   - 2007/10/16 : (Version 2.4)
     - GA_ReleaseTable returns nil
   - 2007/08/31 : (Version 2.3)
     - Added new fields to the struct, to handle cross-realm bg, and new values in raid (masterLoot, maintank)
   - 2007/04/25 : (Version 2.2)
     - Great memory improvements (thanks to the new memory profiling tools)
     - Added table recycling functions
     - TOC update
   - 2007/01/31 : (Version 2.1)
     - Possibility to embed the addon
   - 2006/11/20 : (Version 2.0)
     - WoW 2.0 compatibility
     - Removed frame hooks
   - 2006/08/28 : (Version 1.4)
     - Updated TOC
     - Added a new global variable : GA_Myself
   - 2006/07/31 : (Version 1.3)
     - Added new frames to hook (thanks sarf)
   - 2006/07/19 :
     - Removed isfriend variable, and added ischarmed instead
   - 2006/05/29 : (Version 1.2)
     - Fixed possible lua error when zoning
     - Now hooking OnEvent function for known Frames that use the RAID_ROSTER_UPDATE event to call GetRaidRosterInfo (instead of relying on GroupAnalyse)
   - 2006/04/24 :
     - Now hooking GetRaidRosterInfo to reduce freezes. Addons calling this function should add an optionalDep to GroupAnalyse to insure accurate infos (if they don't want to use internal GA structs)
   - 2006/04/20 :
     - Fixed possible lua error
   - 2006/04/19 : 1.0
     - Module Created
]]


local GA_VERS = "2.4";

--------------- Version Check ---------------

local isBetterInstanceLoaded = ( GroupAnalyse and GroupAnalyse.version and GroupAnalyse.version >= GA_VERS );

if (not isBetterInstanceLoaded) then

if(not GroupAnalyse)
then
  GroupAnalyse = {};
end
GroupAnalyse.version = GA_VERS;


--------------- Shared Constantes ---------------

-- Events
GroupAnalyse.EVENT_GROUP_MODE_CHANGED = 1; -- Param = NewMode
GroupAnalyse.EVENT_MEMBER_JOINED = 2;              -- Param = Member Name;
GroupAnalyse.EVENT_MEMBER_LEFT = 3;                  -- Param = Member Name;
GroupAnalyse.EVENT_INFOS_CHANGED = 4;

-- Group Modes
GroupAnalyse.MODE_NONE = 0;
GroupAnalyse.MODE_SOLO = 1
GroupAnalyse.MODE_GROUP = 2
GroupAnalyse.MODE_RAID = 3;


--------------- Shared variables ---------------

GroupAnalyse.PlayerName = nil;
GroupAnalyse.Members = {};
GroupAnalyse.CurrentGroupMode = GroupAnalyse.MODE_NONE;
GroupAnalyse.Myself = nil;

--------------- Local Constantes ---------------

--------------- Local variables ---------------

local GA_Callbacks = {};
GroupAnalyse.Tables = {};

--------------- Internal functions ---------------

function GroupAnalyse.ChatPrint(str)
  if(DEFAULT_CHAT_FRAME)
  then
    DEFAULT_CHAT_FRAME:AddMessage("GroupAnalyse : "..str, 1.0, 0.7, 0.15);
  end
end

local function _GA_GetUnitInfos(unitid)
  local name = UnitName(unitid);
  local infos = GroupAnalyse.Members[name];
  if(infos == nil)
  then
    infos = GA_GetTable();
  end
  -- Fill infos
  infos.unitid = unitid;
  if(UnitIsPartyLeader(unitid))
  then
    infos.rank = 2;
  else
    infos.rank = 0;
  end
  infos.subgrp = 1;
  infos.level = UnitLevel(unitid);
  _,infos.class = UnitClass(unitid);
  infos.zone = nil;
  if(UnitIsConnected(unitid))
  then
    infos.online = true;
  else
    infos.online = false;
  end
  if(UnitIsDeadOrGhost(unitid) and UnitIsFeignDeath(unitid) == nil)
  then
    infos.isdead = true;
  else
    infos.isdead = false;
  end
  if(UnitIsCharmed(unitid))
  then
    infos.ischarmed = true;
  else
    infos.ischarmed = false;
  end
  infos.role = nil;
  infos.isML = false;
  return name,infos;
end

local function _GA_AnalyseGroupMembers()
  local newmode = GroupAnalyse.MODE_NONE;
  local new_members = GA_GetTable();
  local raid_count = GetNumRaidMembers();
  local party_count;
  local name,rank,subgrp,level,_,class,zone,online,isdead,role,isml;
  local infos;

  if(raid_count ~= 0) -- In a raid
  then
    if(GroupAnalyse.CurrentGroupMode ~= GroupAnalyse.MODE_RAID) -- But was not
    then
      GroupAnalyseFrame:UnregisterEvent("PARTY_MEMBERS_CHANGED");
    end
    newmode = GroupAnalyse.MODE_RAID;
    --GA_hooked_count = 0;
    --GA_Indexes = {}; -- Reset indexes
    for i = 1, raid_count
    do
      --name,rank,subgrp,level,localclass,class,zone,online,isdead = GA_Old_GetRaidRosterInfo(i);
      fullname,rank,subgrp,level,localclass,class,zone,online,isdead,role,isml = GetRaidRosterInfo(i);
      if(fullname and (fullname ~= UNKNOWNOBJECT) and (fullname ~= UKNOWNBEING))
      then
        local name = UnitName("raid"..i);
        infos = GroupAnalyse.Members[name];
        if(infos == nil)
        then
          infos = GA_GetTable();
        end
        new_members[name] = infos;
        -- Fill infos
        infos.name = name;
        infos.fullname = fullname;
        infos.unitid = "raid"..i;
        infos.rank = rank;
        infos.subgrp = subgrp;
        infos.level = level;
        infos.localclass = localclass;
        infos.class = class;
        infos.zone = zone;
        if(online) then
          infos.online = true;
        else
          infos.online = false;
        end
        if(UnitIsCharmed(infos.unitid)) then
          infos.ischarmed = true;
        else
          infos.ischarmed = false;
        end
        if(isdead) then
          infos.isdead = true;
        else
          infos.isdead = false;
        end
        infos.role = role;
        if(isml) then
          infos.isML = true;
        else
          infos.isML = false;
        end
        --[[GA_Indexes[i] = infos;
      else
        GA_Indexes[i] = nil;]]
      end
    end
  else -- Not in a RAID (in a group or solo)
    if(GroupAnalyse.CurrentGroupMode == GroupAnalyse.MODE_RAID) -- Was in a RAID
    then
      GroupAnalyseFrame:RegisterEvent("PARTY_MEMBERS_CHANGED");
    end
    name,infos = _GA_GetUnitInfos("player");
    new_members[name] = infos;
    party_count = GetNumPartyMembers();
    if(party_count ~= 0) -- In a group
    then
      newmode = GroupAnalyse.MODE_GROUP;
      for i = 1,party_count
      do
        name,infos = _GA_GetUnitInfos("party"..i);
        new_members[name] = infos;
      end
    else
      newmode = GroupAnalyse.MODE_SOLO;
    end
  end

  GroupAnalyse.Myself = new_members[GroupAnalyse.PlayerName];

  -- Update list of Raiders
  for n,tab in pairs(GroupAnalyse.Members) -- Remove old raiders
  do
    if(new_members[n] == nil) -- No longer in the raid
    then
      GA_ReleaseTable(GroupAnalyse.Members[n]);
      GroupAnalyse.Members[n] = nil;
      for _,func in pairs(GA_Callbacks)
      do
        func(GroupAnalyse.EVENT_MEMBER_LEFT,n);
      end
    end
  end

  for n,tab in pairs(new_members) -- Add new raiders
  do
    local raider = GroupAnalyse.Members[n];
    if(raider == nil) -- New member
    then
      GroupAnalyse.Members[n] = tab;
      for _,func in pairs(GA_Callbacks)
      do
        func(GroupAnalyse.EVENT_MEMBER_JOINED,n);
      end
    end
  end

  if(GroupAnalyse.CurrentGroupMode ~= newmode)
  then
    for _,func in pairs(GA_Callbacks)
    do
      func(GroupAnalyse.EVENT_GROUP_MODE_CHANGED,newmode);
    end
  end
  GroupAnalyse.CurrentGroupMode = newmode;

  for _,func in pairs(GA_Callbacks)
  do
    func(GroupAnalyse.EVENT_INFOS_CHANGED);
  end

  GA_ReleaseTable(new_members);
end

--------------- Shared functions ---------------

function GroupAnalyse.RegisterForEvents(callback_func)
  tinsert(GA_Callbacks,callback_func);
  if(GroupAnalyse.CurrentGroupMode ~= GroupAnalyse.MODE_NONE) -- Late initialization
  then
    callback_func(GroupAnalyse.EVENT_GROUP_MODE_CHANGED,newmode);
    for n in pairs(GroupAnalyse.Members)
    do
      callback_func(GroupAnalyse.EVENT_MEMBER_JOINED,n);
    end
    callback_func(GroupAnalyse.EVENT_INFOS_CHANGED);
  end
end

function GroupAnalyse.UnRegisterForEvents(callback_func)
  for i,v in ipairs(GA_Callbacks)
  do
    if(v == callback_func)
    then
      tremove(GA_Callbacks,i);
      return;
    end
  end
end


--------------- Event function ---------------

function GroupAnalyse.OnEvent(self,event,...)
  -- Group events
  if(event == "PARTY_MEMBERS_CHANGED" or event == "RAID_ROSTER_UPDATE")
  then
    _GA_AnalyseGroupMembers();
    -- Calling hooked functions
    --[[for i,tab in ipairs(GA_HookedFunctions)
    do
      this = tab.frame;
      tab.func();
    end]]
  elseif(event == "VARIABLES_LOADED")
  then
    GroupAnalyse.PlayerName = UnitName("player");
    GroupAnalyseFrame:Show();
  elseif(event == "PLAYER_ENTERING_WORLD")
  then
    _GA_AnalyseGroupMembers();
    --_GA_SetupHooks();
  end
end


--------------- GroupAnalyse Main Frame ---------------

  --Event Driver
  if(not GroupAnalyseFrame)
  then
    CreateFrame("Frame", "GroupAnalyseFrame");
  end
  GroupAnalyseFrame:Hide();
  --Event Registration
  GroupAnalyseFrame:RegisterEvent("VARIABLES_LOADED");
  GroupAnalyseFrame:RegisterEvent("PLAYER_ENTERING_WORLD");
  GroupAnalyseFrame:RegisterEvent("PARTY_MEMBERS_CHANGED");
  GroupAnalyseFrame:RegisterEvent("RAID_ROSTER_UPDATE");
  --Frame Scripts
  GroupAnalyseFrame:SetScript("OnEvent", GroupAnalyse.OnEvent);
  -- Print init message
  GroupAnalyse.ChatPrint("Version "..GA_VERS.." Loaded !");

end -- not isBetterInstanceLoaded

--------------- Table recycling functions ---------------

function GA_WipeTable(t1,recurseCount)
  if(type(t1) ~= "table")
  then
    if(t1 == nil)
    then
      return GA_GetTable();
    end
    return t1;
  end

  for k,v in pairs(t1)
  do
    if(recurseCount and (recurseCount > 0) and (type(v) == "table")) -- Recurse release table
    then
      GA_ReleaseTable(v,recurseCount-1);
    end
    t1[k] = nil;
  end

  return t1;
end

function GA_GetTable()
  local recTable;
  if(#GroupAnalyse.Tables >= 1)
  then
    recTable = tremove(GroupAnalyse.Tables);
  else
    recTable = {};
  end
  return recTable;
end

function GA_ReleaseTable(t1,recurseCount)
  if(type(t1) ~= "table")
  then
    return;
  end
  
  for k,v in pairs(t1)
  do
    if(recurseCount and (recurseCount > 0) and (type(v) == "table")) -- Recurse release table
    then
      GA_ReleaseTable(v,recurseCount-1);
    end
    t1[k] = nil;
  end
  
  tinsert(GroupAnalyse.Tables,t1);
  return nil;
end

