--[[
  Custom Chat Comm, by Kiki - Conseil des Ombres-EU
  

  TODO:
   - Detecter un mauvais mot de passe
   - Tester qd on relog a griphon

 ChangeLog:
   - 2008/09/05 : 0.6
     - Fixed incorrect ChatThrottleLib queuing for last section of a multiple parts message
   - 2008/09/04 : 0.5
     - Fixed UTF-8 double/triple bytes chars splitting
   - 2008/09/03 : 0.4
     - Fixed UnregisterComm function not defined in metatable
   - 2008/08/25 : 0.3
     - Fixed password initialization if alone in the channel
     - Fixed channel initialization detection
   - 2008/06/23 : 0.2
     - Startup don't init channel if unit is on taxi
   - 2008/06/18 : 0.1
     - Library Created
  
  Credits:
   - Message splitting code grabbed from AceComm
]]

local CCC_VERS = 0.6;
local CCC_DEBUG = false;

--------------- Quick access ---------------

local type = type
local strsub = string.sub
local strfind = string.find
local tinsert = table.insert
local tconcat = table.concat
local strlower = string.lower;

--------------- Version Check ---------------

local isBetterInstanceLoaded = ( CustomChatComm and CustomChatComm.version and CustomChatComm.version >= CCC_VERS );

if (not isBetterInstanceLoaded) then

if(not CustomChatComm)
then
  CustomChatComm = {};
  CustomChatComm.comms = {};
end
CustomChatComm.version = CCC_VERS;

local CallbackHandler = LibStub:GetLibrary("CallbackHandler-1.0")

--------------- AceComm-3.0,4 grabbed vars ---------------

-- for my sanity and yours, let's give the message type bytes names
local MSG_MULTI_FIRST = "\001"
local MSG_MULTI_NEXT  = "\002"
local MSG_MULTI_LAST  = "\003"

CustomChatComm.multipart_origprefixes = CustomChatComm.multipart_origprefixes or {} -- e.g. "Prefix\001"="Prefix", "Prefix\002"="Prefix"
CustomChatComm.multipart_reassemblers = CustomChatComm.multipart_reassemblers or {} -- e.g. "Prefix\001"="OnReceiveMultipartFirst"

-- the multipart message spool: indexed by a combination of sender+distribution+
CustomChatComm.multipart_spool = CustomChatComm.multipart_spool or {} 


--------------- Shared Constantes ---------------
--------------- Shared variables ---------------
--------------- Local Constantes ---------------

local CCC_CB_YOU_JOINED = "YOU_JOINED";
local CCC_CB_YOU_LEFT = "YOU_LEFT";

--------------- Local variables ---------------

local _CCC_init_timer = nil;
local _CCC_init_ready = false;
local _CCC_LogMainTable = nil;

--------------- Internal functions ---------------

local function _CCC_Log(buffer,msg)
  if(_CCC_LogMainTable == nil) then return end
  if(_CCC_LogMainTable[buffer] == nil) then _CCC_LogMainTable[buffer] = {} end
  tinsert(_CCC_LogMainTable[buffer],msg);
end

local function _CCC_WoWAPI_FireCallback(event,...)
  local channel = strlower(select(9,...));

  for i,com in ipairs(CustomChatComm.comms)
  do
    if(com.channel == channel)
    then
      local cb = com.callbacks[event];
      if(cb)
      then
        cb(...);
      end
    end
  end
end

local function _CCC_Channel_FireCallback(event,channel,...)
  for i,com in ipairs(CustomChatComm.comms)
  do
    if(com.channel == channel)
    then
      local cb = com.callbacks[event];
      if(cb)
      then
        cb(channel,...);
      end
    end
  end
end

local function _CCC_InitComm(com)
  if(GetChannelName(com.channel) ~= 0)
  then
    com.isInit = true;
    ListChannelByName(com.channel);
    _CCC_Channel_FireCallback(CCC_CB_YOU_JOINED,com.channel);
    return;
  end

  JoinChannelByName(com.channel,com.password);
end

local function _CCC_EnforceRejoin(com)
  if(com)
  then
    JoinChannelByName(com.channel,com.password);
  end
end

local function _CCC_EnforcePassword(channel)
  for i,com in ipairs(CustomChatComm.comms)
  do
    if(channel == com.channel)
    then
      if(com.password and com.password ~= "")
      then
        SetChannelPassword(channel,com.password);
      end
    end
  end
end

local function _CCC_AlreadyRegisteredChannel(channel)
  for i,com in ipairs(CustomChatComm.comms)
  do
    if(com.channel == channel)
    then
      return true;
    end
  end
  return false;
end

local function _CCC_ProcessChannelListing(event,listing, _, _, _, _, _, _, _, channelName)
  local channel = strlower(channelName);
  local cLeader = strbyte("*");
  local cModerator = strbyte("@");
  local cMute = strbyte("#");
  local char1;
  local playerName = UnitName("player");
  
  for value in gmatch(listing,"[^, ]+")
  do
    local isLeader = false;
    char1 = strbyte(value);
    if(char1 == cLeader)
    then
      value = strsub(value,2);
      if(value == playerName)
      then
        isLeader = true;
      end
    elseif(char1 == cModerator or char1 == cMute)
    then
      value = strsub(value,2);
    end
    if(isLeader)
    then
      _CCC_EnforcePassword(channel);
    end
  end
end

local function _CCC_ProcessChannelNotice(event,kind,player1,_,_,_,_,_,_,channelName)
  local channel = strlower(channelName);

  if(kind == "YOU_JOINED")
  then
    if(_CCC_AlreadyRegisteredChannel(channel))
    then
      ListChannelByName(channel);
    end
    _CCC_Channel_FireCallback(CCC_CB_YOU_JOINED,channel);
  elseif(kind == "YOU_LEFT")
  then
    _CCC_Channel_FireCallback(CCC_CB_YOU_LEFT,channel);
    -- Enforce rejoin
    local done = false;
    for _,com in ipairs(CustomChatComm.comms)
    do
      if(channel == com.channel)
      then
        com.isInit = false;
        if(done == false)
        then
          Chronos.schedule(0.5,_CCC_EnforceRejoin,com);
          done = true;
        end
      end
    end
  elseif(kind == "OWNER_CHANGED")
  then
    if(player1 == UnitName("player"))
    then
      _CCC_EnforcePassword(channel);
    end
  elseif(kind == "WRONG_PASSWORD")
  then
  end
end

local function _CCC_InitComplete()
  for i,com in ipairs(CustomChatComm.comms)
  do
    if(com.isInit == false)
    then
      _CCC_InitComm(com);
    end
  end
end

local function _CCC_CheckPlayerGuild()
  if(IsInGuild()) -- Have a guild
  then
    if(GetGuildInfo("player") == nil) -- Guild name still not init
    then
      return false;
    end
  end
  return true;
end

local function _CCC_ProcessChannelMessage(channel,sender,text)
  if(_CCC_AlreadyRegisteredChannel(channel) == false) -- Not for me
  then
    return;
  end
  if(CCC_DEBUG) then _CCC_Log("recv",text) end
  -- Try to get prefix and message
  local _,_,prefix,message = strfind(text,"([^\007]+)\007(.+)");
  -- Following code grabbed from AceComm-3.0,4
  local reassemblername = CustomChatComm.multipart_reassemblers[prefix]
  if reassemblername then
    -- multipart: reassemble
    local CustomChatCommReassemblerFunc = CustomChatComm[reassemblername]
    local origprefix = CustomChatComm.multipart_origprefixes[prefix]
    CustomChatCommReassemblerFunc(CustomChatComm, origprefix, message, channel, sender)
  else
    -- single part: fire it off immediately and let CallbackHandler decide if it's registered or not
    CustomChatComm.callbacks:Fire(prefix, message, channel, sender)
  end
end


----------------------------------------
-- Message receiving -- Grabbed from AceComm-3.0,4
----------------------------------------

do
	local compost = setmetatable({}, {__mode=="k"})
	local function new()
		local t = next(compost)
		if t then 
			compost[t]=nil
			for i=#t,3,-1 do	-- faster than pairs loop. don't even nil out 1/2 since they'll be overwritten
				t[i]=nil
			end
			return t
		end
		
		return {}
	end
	
	local function lostdatawarning(prefix,sender,where)
		DEFAULT_CHAT_FRAME:AddMessage(MAJOR..": Warning: lost network data regarding '"..tostring(prefix).."' from '"..tostring(sender).."' (in "..where..")")
	end

	function CustomChatComm:OnReceiveMultipartFirst(prefix, message, distribution, sender)
		local key = prefix.."\t"..distribution.."\t"..sender	-- a unique stream is defined by the prefix + distribution + sender
		local spool = CustomChatComm.multipart_spool
		
		--[[
		if spool[key] then 
			lostdatawarning(prefix,sender,"First")
			-- continue and overwrite
		end
		--]]
		
		spool[key] = message	-- plain string for now
	end

	function CustomChatComm:OnReceiveMultipartNext(prefix, message, distribution, sender)
		local key = prefix.."\t"..distribution.."\t"..sender	-- a unique stream is defined by the prefix + distribution + sender
		local spool = CustomChatComm.multipart_spool
		local olddata = spool[key]
		
		if not olddata then
			--lostdatawarning(prefix,sender,"Next")
			return
		end

		if type(olddata)~="table" then
			-- ... but what we have is not a table. So make it one. (Pull a composted one if available)
			local t = new()
			t[1] = olddata		-- add old data as first string
			t[2] = message      -- and new message as second string
			spool[key] = t		-- and put the table in the spool instead of the old string
		else
			tinsert(olddata, message)
		end
	end

	function CustomChatComm:OnReceiveMultipartLast(prefix, message, distribution, sender)
		local key = prefix.."\t"..distribution.."\t"..sender	-- a unique stream is defined by the prefix + distribution + sender
		local spool = CustomChatComm.multipart_spool
		local olddata = spool[key]
		
		if not olddata then
			--lostdatawarning(prefix,sender,"End")
			return
		end

		spool[key] = nil
		
		if type(olddata)=="table" then
			-- if we've received a "next", the spooled data will be a table for rapid & garbage-free tconcat
			tinsert(olddata, message)
			CustomChatComm.callbacks:Fire(prefix, tconcat(olddata, ""), distribution, sender)
			compost[olddata]=true
		else
			-- if we've only received a "first", the spooled data will still only be a string
			CustomChatComm.callbacks:Fire(prefix, olddata..message, distribution, sender)
		end
	end
end

--------------- Shared functions ---------------

function CustomChatComm.Todo(str,...)
  if(DEFAULT_CHAT_FRAME)
  then
    DEFAULT_CHAT_FRAME:AddMessage("CustomChatComm TODO: "..string.format(str,...), 1.0, 0.7, 0.15);
  end
end

function CustomChatComm.ChatPrint(str)
  if(DEFAULT_CHAT_FRAME)
  then
    DEFAULT_CHAT_FRAME:AddMessage("CustomChatComm: "..str, 1.0, 0.7, 0.15);
  end
end

function CustomChatComm.RegisterComm(prefix,channelName,password,callback_message,callback_you_joined,callback_you_left)
  local com = {};

  com.prefix = prefix;
  com.channel = strlower(channelName);
  com.password = password;
  com.callbacks = {};
  com.callbacks[CCC_CB_YOU_JOINED] = callback_you_joined;
  com.callbacks[CCC_CB_YOU_LEFT] = callback_you_left;
  com.isInit = false;
  CustomChatComm._RegisterComm(CustomChatComm,prefix,callback_message) -- created by CallbackHandler

  -- Add new com to list
  tinsert(CustomChatComm.comms,com); -- Add before checking for Init, or cb won't fire if channel already joined

  -- If global channels init is complete, join now
  if(_CCC_init_ready)
  then
    _CCC_InitComm(com);
  end
  
  return com;
end

function CustomChatComm.UnregisterComm(prefix,channelName)
  local channel = strlower(channelName);

  for i,com in ipairs(CustomChatComm.comms)
  do
    if(prefix == com.prefix and channel == com.channel)
    then
      tremove(CustomChatComm.comms,i);
      if(_CCC_AlreadyRegisteredChannel(channel) == false) -- Last one
      then
        LeaveChannelByName(channel);
      end
    end
  end
end

function CustomChatComm.RegisterCallback(com,callback_name,callback)
  com.callbacks[callback_name] = callback;
end

function CustomChatComm.UnregisterCallback(com,callback_name)
  com.callbacks[callback_name] = nil;
end

local UTF2CHAR = {
  [0xc2] = true,
  [0xc3] = true,
  [0xce] = true,
}

local UTF3CHAR = {
  [0xe2] = true,
}

local function _CCC_strsub(text,pos1,pos2) -- Special function to avoid UTF double/triple chars splitting (causes a 'Chat text must be in UTF-8 format' error message)
  local len = pos2 - pos1;
  local val = text:byte(pos2);
  if(UTF2CHAR[val]) -- UTF-8 2-bytes character
  then
    len = len - 1;
  end
  if(UTF3CHAR[val]) -- UTF-8 3-bytes character
  then
    len = len - 2;
  end
  return strsub(text,pos1,pos1+len),len+1;
end

function CustomChatComm.SendChatMessage(prefix,text,channelName)
  local index = GetChannelName(channelName);
  local channel = strlower(channelName);
  if(index and index > 0)
  then
    -- Following code grabbed from AceComm-3.0,4 -- Modified by Kiki to avoid UTF-8 chars splitting
    if strfind(prefix, "[\001-\003]") then
      error("CustomChatComm.SendChatMessage: Characters \\001--\\003) in prefix are reserved", 2);
    end

    local textlen = #text;
    local maxtextlen = 252 - #prefix; -- 252 is the max length of prefix + text that can be sent in one message <- Let's use 252 for more safety
    local queueName = prefix..channel;

    if textlen <= maxtextlen then
      -- fits all in one message
      if(CCC_DEBUG) then _CCC_Log("send",prefix.."\007"..text) end
      ChatThrottleLib:SendChatMessage("NORMAL",prefix,prefix.."\007"..text,"CHANNEL",nil,index,queueName);
    else
      maxtextlen = maxtextlen - 1	-- 1 extra byte for part indicator in suffix
      -- first part
      local chunk,chunklen = _CCC_strsub(text, 1, maxtextlen)
      if(CCC_DEBUG) then _CCC_Log("send",prefix..MSG_MULTI_FIRST.."\007"..chunk) end
      ChatThrottleLib:SendChatMessage("NORMAL",prefix,prefix..MSG_MULTI_FIRST.."\007"..chunk,"CHANNEL",nil,index,queueName);
      -- continuation
      local pos = chunklen + 1
      local prefix2 = prefix..MSG_MULTI_NEXT

      while pos+maxtextlen <= textlen do
        chunk,chunklen = _CCC_strsub(text, pos, pos+maxtextlen-1)
        if(CCC_DEBUG) then _CCC_Log("send",prefix2.."\007"..chunk) end
        ChatThrottleLib:SendChatMessage("NORMAL",prefix,prefix2.."\007"..chunk,"CHANNEL",nil,index,queueName);
        pos = pos + chunklen
      end

      -- final part
      chunk = strsub(text, pos)
      if(CCC_DEBUG) then _CCC_Log("send",prefix..MSG_MULTI_LAST.."\007"..chunk) end
      ChatThrottleLib:SendChatMessage("NORMAL",prefix,prefix..MSG_MULTI_LAST.."\007"..chunk,"CHANNEL",nil,index,queueName);
    end
    
    return true;
  end

  return false;
end

function CustomChatComm.SetDebugMode(mode)
  CCC_DEBUG = mode;
end

function CustomChatComm.SetLogTable(myTable)
  _CCC_LogMainTable = myTable;
end

function CustomChatComm.ResetLogTable()
  if(_CCC_LogMainTable)
  then
    _CCC_LogMainTable.send = nil;
    _CCC_LogMainTable.recv = nil;
  end
end

function CustomChatComm.TestSend(channel)
  local index = GetChannelName(channel);
  if(_CCC_LogMainTable and _CCC_LogMainTable.send and index)
  then
    for i,str in ipairs(_CCC_LogMainTable.send)
    do
      DEFAULT_CHAT_FRAME:AddMessage("Sending line "..i.." of "..#_CCC_LogMainTable.send, 0.9, 0.8, 0.10);
      ChatThrottleLib:SendChatMessage("NORMAL","CCCDebug",str,"CHANNEL",nil,index,"CCCDebug");
    end
  end
end


--------------- Event function ---------------

function CustomChatComm.OnUpdate(self,dt)
  if(_CCC_init_timer == nil) -- Variables not loaded yet
  then
    return;
  end
  _CCC_init_timer = _CCC_init_timer + dt;
  
  if(_CCC_init_timer > 4.0) -- Time to check for full init (after 4 sec)
  then
    local playerName = UnitName("player");
    if(playerName == nil or playerName == UNKNOWNBEING or playerName == UKNOWNBEING or playerName == UNKNOWNOBJECT or _CCC_CheckPlayerGuild() == false or UnitOnTaxi("player") == 1) -- Not ready yet
    then
      return;
    end
    _CCC_init_ready = true;
  end
  if(_CCC_init_ready)
  then
    _CCC_InitComplete();
    self:SetScript("OnUpdate",nil);
  end
end

function CustomChatComm.OnEvent(self,event,...)
  if(event == "CHAT_MSG_CHANNEL")
  then
    local text, sender, _, _, _, _, _, _, channel = select(1,...);
    _CCC_ProcessChannelMessage(strlower(channel),sender,text);
  elseif(event == "CHAT_MSG_CHANNEL_JOIN")
  then
    _CCC_WoWAPI_FireCallback(event,...);
  elseif(event == "CHAT_MSG_CHANNEL_LEAVE")
  then
    _CCC_WoWAPI_FireCallback(event,...);
  elseif(event == "CHAT_MSG_CHANNEL_LIST")
  then
    _CCC_ProcessChannelListing(event,...);
    _CCC_WoWAPI_FireCallback(event,...);
  elseif(event == "CHAT_MSG_CHANNEL_NOTICE")
  then
    _CCC_ProcessChannelNotice(event,...);
    _CCC_WoWAPI_FireCallback(event,...);
  elseif(event == "CHAT_MSG_CHANNEL_NOTICE_USER")
  then
    _CCC_ProcessChannelNotice(event,...);
    _CCC_WoWAPI_FireCallback(event,...);
  elseif(event == "VARIABLES_LOADED")
  then
    _CCC_init_timer = 0; -- Start timer now
  end
end

----------------------------------------
-- Embed CallbackHandler -- Grabbed from AceComm-3.0,4
----------------------------------------

if not CustomChatComm.callbacks then
	-- ensure that 'prefix to watch' table is consistent with registered
	-- callbacks
	CustomChatComm.__prefixes = {}

	CustomChatComm.callbacks = CallbackHandler:New(CustomChatComm,
						"_RegisterComm",
						"_UnregisterComm",
						"UnregisterAllComm")
end

function CustomChatComm.callbacks:OnUsed(target, prefix)
	CustomChatComm.multipart_origprefixes[prefix..MSG_MULTI_FIRST] = prefix
	CustomChatComm.multipart_reassemblers[prefix..MSG_MULTI_FIRST] = "OnReceiveMultipartFirst"
	
	CustomChatComm.multipart_origprefixes[prefix..MSG_MULTI_NEXT] = prefix
	CustomChatComm.multipart_reassemblers[prefix..MSG_MULTI_NEXT] = "OnReceiveMultipartNext"
	
	CustomChatComm.multipart_origprefixes[prefix..MSG_MULTI_LAST] = prefix
	CustomChatComm.multipart_reassemblers[prefix..MSG_MULTI_LAST] = "OnReceiveMultipartLast"
end

function CustomChatComm.callbacks:OnUnused(target, prefix)
	CustomChatComm.multipart_origprefixes[prefix..MSG_MULTI_FIRST] = nil
	CustomChatComm.multipart_reassemblers[prefix..MSG_MULTI_FIRST] = nil
	
	CustomChatComm.multipart_origprefixes[prefix..MSG_MULTI_NEXT] = nil
	CustomChatComm.multipart_reassemblers[prefix..MSG_MULTI_NEXT] = nil
	
	CustomChatComm.multipart_origprefixes[prefix..MSG_MULTI_LAST] = nil
	CustomChatComm.multipart_reassemblers[prefix..MSG_MULTI_LAST] = nil
end


--------------- CustomChatComm Main Frame ---------------

  --Event Driver
  if(not CustomChatCommFrame)
  then
    CreateFrame("Frame", "CustomChatCommFrame");
  end
  --Event Registration
  CustomChatCommFrame:RegisterEvent("VARIABLES_LOADED");
  CustomChatCommFrame:RegisterEvent("CHAT_MSG_CHANNEL");
  CustomChatCommFrame:RegisterEvent("CHAT_MSG_CHANNEL_JOIN");
  CustomChatCommFrame:RegisterEvent("CHAT_MSG_CHANNEL_LEAVE");
  CustomChatCommFrame:RegisterEvent("CHAT_MSG_CHANNEL_LIST");
  CustomChatCommFrame:RegisterEvent("CHAT_MSG_CHANNEL_NOTICE");
  CustomChatCommFrame:RegisterEvent("CHAT_MSG_CHANNEL_NOTICE_USER");
  
  --Frame Scripts
  CustomChatCommFrame:SetScript("OnEvent", CustomChatComm.OnEvent);
  CustomChatCommFrame:SetScript("OnUpdate",CustomChatComm.OnUpdate);
  -- Print init message
  CustomChatComm.ChatPrint("Version "..CCC_VERS.." Loaded !");

end -- not isBetterInstanceLoaded

--[[
--------------------------------- FILTERING CHAT MSG
local myChatFilter(msg)
  if strfind(msg, "buy gold") then
    return true
  end
  if arg2=="Knownspammer" then
    return true
  end
  if strfind(msg, "lol") then
    return false, gsub(msg, "lol", "")
  end
end
ChatFrame_AddMessageEventFilter("CHAT_MSG_CHANNEL", myChatFilter)

--------------------------------- STICKY, SLASH, ...
http://www.wowace.com/wiki/Prat
]]

-- CHAT_MSG_CHANNEL_JOIN(_, user, _, _, _, _, _, _, channel)
-- CHAT_MSG_CHANNEL_LEAVE(_, user, _, _, _, _, _, _, channel)
-- CHAT_MSG_CHANNEL_LIST(text, _, _, _, _, _, _, _, channel)
-- CHAT_MSG_CHANNEL_NOTICE(kind, _, _, deadName, _, _, _, num, channel)

