--[[
  Guild Event Manager (GEM) - Version 3
  by Kiki from European "Conseil des Ombres" (Horde)
  ----------------------------------------
  Out of game Sync'ing
]]

--[[
  Exported API Functions:

   GEM3_SYNC_CheckForExternalCommands()
    Purpose:
     Checks GEM3_ExternalCommands table for external commands

]]

--------------- Local variables ---------------

--------------- Internal functions ---------------

local function _GEM3_SYNC_SanityCheck(command,value,expected)
  if(command[value] == nil)
  then
    GEM3_ChatDebug(GEM3_DEBUG_WARNING,"Command importer error: Missing parameter '"..value.."' for command '"..tostring(command.cmd).."' (EventID="..tostring(command.ev_id)..")");
    return true;
  end
  if(type(command[value]) ~= expected)
  then
    GEM3_ChatDebug(GEM3_DEBUG_WARNING,"Command importer error: Unknown command '"..tostring(command.cmd).."'");
    return true;
  end
  
  return false; -- Everything OK
end

local function _GEM3_SYNC_ProcessCommand_Subscribe(command)
  -- Sanity check
  if(_GEM3_SYNC_SanityCheck(command,"pl_name","string") or _GEM3_SYNC_SanityCheck(command,"level","number") or _GEM3_SYNC_SanityCheck(command,"guild","string")
    or _GEM3_SYNC_SanityCheck(command,"class","string") or _GEM3_SYNC_SanityCheck(command,"role","number") or _GEM3_SYNC_SanityCheck(command,"comment","string") 
    or _GEM3_SYNC_SanityCheck(command,"queue","number") or _GEM3_SYNC_SanityCheck(command,"main_time","number") or _GEM3_SYNC_SanityCheck(command,"update_time","number")
    or _GEM3_SYNC_SanityCheck(command,"alt_role","number") or _GEM3_SYNC_SanityCheck(command,"source","string"))
  then
    return false;
  end

  if(command.queue == GEM3_SUB_FORCE_QUEUE_NONE) -- None means Titular, if available
  then
    command.queue = GEM3_SUB_FORCE_QUEUE_TITULAR;
  end
  
  local params = {};
 -- Common Cmd params
  tinsert(params,command.leader);
  tinsert(params,command.ev_date);
  tinsert(params,0); -- Not acked
  tinsert(params,command.leader);
  tinsert(params,command.pl_name);
  tinsert(params,command.update_time);
 -- Specific Cmd params
  tinsert(params,command.class);
  tinsert(params,command.role);
  tinsert(params,command.level);
  tinsert(params,command.guild);
  tinsert(params,command.comment);
  tinsert(params,command.queue);
  tinsert(params,command.alt_role);
  tinsert(params,command.main_time);
  tinsert(params,command.update_time);
  tinsert(params,command.source);
  
  _GEM3_COM_ParseChannelMessage(command.channel,"ImportedCommands",true,command.protocol,GEM3_CMD_CMD_SUBSCRIBE,command.ev_id,GEM3_STAMP_GetLocalStamp(),params);
  return true;
end


--------------- Exported functions ---------------

function GEM3_SYNC_ProcessCommand(command)
  -- Check for common mandatory params
  if(_GEM3_SYNC_SanityCheck(command,"channel","string") or _GEM3_SYNC_SanityCheck(command,"ev_id","string") or _GEM3_SYNC_SanityCheck(command,"leader","string")
    or _GEM3_SYNC_SanityCheck(command,"ev_date","number") or _GEM3_SYNC_SanityCheck(command,"cmd","string") or _GEM3_SYNC_SanityCheck(command,"protocol","number"))
  then
    return false;
  end
  -- Check protocol
  if(command.protocol ~= GEM3_COM_PROTOCOL_REVISION)
  then
    GEM3_ChatDebug(GEM3_DEBUG_SYNC,"GEM3_SYNC_ProcessCommand: Ignoring command '"..command.cmd.."' for EventID '"..command.ev_id.."' because invalid protocol version");
    return false;
  end
  -- Check for Configured channel
  if(GEM3_CHAN_GetChannelInfos(command.channel) == nil)
  then
    GEM3_ChatDebug(GEM3_DEBUG_SYNC,"GEM3_SYNC_ProcessCommand: Ignoring command '"..command.cmd.."' for EventID '"..command.ev_id.."' for a channel I don't have ("..command.channel..")");
    return false;
  end
  -- Check for Assistant
  if(not GEM3_IsMyReroll(command.leader) and not GEM3_EVT_AmIAssistant(command.ev_id))
  then
    GEM3_ChatDebug(GEM3_DEBUG_SYNC,"GEM3_SYNC_ProcessCommand: Ignoring command '"..command.cmd.."' for EventID '"..command.ev_id.."' for channel '"..command.channel.."' because I'm not assistant of this event");
    return false;
  end
  -- Process command
  if(command.cmd == "Subscribe")
  then
    GEM3_ChatDebug(GEM3_DEBUG_SYNC,"GEM3_SYNC_ProcessCommand: Processing command '"..command.cmd.."' for EventID '"..command.ev_id.."' for channel '"..command.channel.."'");
    return _GEM3_SYNC_ProcessCommand_Subscribe(command);
  else
    GEM3_ChatDebug(GEM3_DEBUG_WARNING,"Command importer error: Unknown command '"..tostring(command.cmd).."'");
  end
  return false;
end

function GEM3_SYNC_CheckForExternalCommands()
  for _,command in ipairs(GEM3_ExternalCommands)
  do
    GEM3_SYNC_ProcessCommand(command);
  end
  GEM3_ExternalCommands = {}; -- Wipe all commands, processed or not
end
