--[[
  Guild Event Manager (GEM) - Version 3
  by Kiki from European "Conseil des Ombres" (Horde)
  ----------------------------------------
  Subscribers module
]]

--[[
  Exported API Functions:

   GEM3_SUB_GetSubscribedInfos(ev_id,pl_name)
    Input:
     ev_id       : String -- The event I have subscribed to
     pl_name     : String -- Name of my toon
    Output:
     infos      : GEM3_Subscription -- My subscription info for this event, or nil if none found
    Purpose:
     Gets my subscription infos for specified event and reroll name. If "pl_name" is nil, get any subscribed infos for this event

   GEM3_SUB_IsPlayerBanned(ev_id,pl_name)
    Input:
     ev_id       : String -- The event 
     pl_name     : String -- Name of the player to check (or nil to search all, and only, my toons)
    Output:
     Banned      : Boolean -- Returns true if "pl_name" is banned from "ev_id", false otherwise
     Reason      : String -- Returns banned reason if Banned is true, nil otherwise
    Purpose:
     Checks if a player is banned from an event. If "pl_name" is nil, search for any banned reroll from "ev_id"

   GEM3_SUB_IsPlayerKicked(ev_id,pl_name)
    Input:
     ev_id       : String -- The event
     pl_name     : String -- Name of my toon
    Output:
     Kicked      : Boolean -- Returns true if "pl_name" has been kicked from "ev_id", false otherwise
     Reason      : String -- Returns kicked reason if Kicked is true, nil otherwise
    Purpose:
     Checks if I've been kicked from an event. If "pl_name" is nil, search for any kicked reroll from "ev_id"

   GEM3_SUB_Subscribe(ev_id,pl_name,guild,level,class,role,comment,force_queue,alt_role)
    Input:
     ev_id       : String -- The event I have subscribed to
     pl_name     : String -- Name of my toon
     guild       : String -- Guild of my toon (or nil)
     level       : Int    -- My current level
     class       : String -- My class
     role        : Int    -- Role of my class (one of GEM3_ROLE_xx)
     comment     : String -- My subscription comment
     force_queue : Int    -- Queue I want to be forced into (one of GEM3_SUB_FORCE_QUEUE_xx)
     alt_role    : Int    -- Alternative role of my class (one of GEM3_ROLE_xx)
    Output:
     Result      : Boolean -- True if success, false otherwise
     Reason      : String  -- Failure message, Nil if Result is true
    Events:
     "OnSubscribed"(ev_id,pl_name,infos.state,GEM3_SUB_SOURCE_SELF) is called if I'm the leader, and I've successfully subscribed to my event
    Purpose:
     Subscribe to an event

   GEM3_SUB_SubscribeUpdate(ev_id,pl_name,guild,level,class,role,comment,force_queue,alt_role)
    Input:
     ev_id       : String -- The event I have subscribed to
     pl_name     : String -- Name of my toon
     guild       : String -- Guild of my toon (or nil)
     level       : Int    -- My current level
     class       : String -- My class
     role        : Int    -- Role of my class (one of GEM3_ROLE_xx)
     comment     : String -- My subscription comment
     force_queue : Int    -- Queue I want to be forced into (one of GEM3_SUB_FORCE_QUEUE_xx)
     alt_role    : Int    -- Alternative role of my class (one of GEM3_ROLE_xx)
    Output:
     Result      : Boolean -- True if success, false otherwise
     Reason      : String  -- Failure message, Nil if Result is true
    Purpose:
     Send the updated version of my subscription

   GEM3_SUB_SubscribeExternal(ev_id,pl_name,guild,level,class,role,comment,force_queue,alt_role)
    Input:
     ev_id       : String -- The event id
     pl_name     : String -- Name of the player
     guild       : String -- Guild of the player (or nil)
     level       : Int    -- Current level of the player
     class       : String -- Class of the player
     role        : Int    -- Role of the player (one of GEM3_ROLE_xx)
     comment     : String -- Player's subscription comment
     force_queue : Int    -- Queue to be forced into (one of GEM3_SUB_FORCE_QUEUE_xx)
     alt_role    : Int    -- Alternative role of my class (one of GEM3_ROLE_xx)
    Output:
     Result      : Boolean -- True if success, false otherwise
     Reason      : String  -- Failure message, Nil if Result is true
    Purpose:
     Subscribe another player to an event

   GEM3_SUB_ResendSubscription(ev_id,pl_name)
    Input:
     ev_id       : String -- The event I have subscribed to
     pl_name     : String -- Name of my toon
    Output:
     Result      : Boolean -- True if success, false otherwise
     Reason      : String  -- Failure message, Nil if Result is true
    Purpose:
     Resend subscription to the event (only when leader is connected), if the 1st one seems to have been lost


   GEM3_SUB_IsLatestSubscriptionReceived(ev_id,pl_name)
    Input:
     ev_id       : String -- The event I have subscribed to
     pl_name     : String -- Name of my toon
    Output:
     Result      : Boolean -- True if leader acked my latest subscription infos
    Purpose:
     Checks if my latest subscription informations has been received by the leader

   GEM3_SUB_ForceQueueForPlayer(ev_id,pl_name,force_queue)
    Input:
     ev_id       : String -- The event ID
     pl_name     : String -- Name of my toon
     force_queue : Int    -- Queue I want to be forced into (one of GEM3_SUB_FORCE_QUEUE_xx)
    Events:
     "OnUpdatedEvent"(event) is called if I'm the leader, and I've successfully altered the event
    Purpose:
     Force a queue for a subscriber - Leader only

]]

--------------- Local variables ---------------

--~ GEM3_SUB_Plugins = {};


--------------- Internal functions ---------------

-- Returns false if ok, true if deadline passed
local function _GEM3_SUB_CheckDeadline(event)
  if(event.deadline and GEM3_STAMP_GetLocalStamp() > event.deadline)
  then
    return true;
  end
  return false;
end

-- Returns a GEM3_Player
local function _GEM3_SUB_GetSubscriberInfos(event,pl_name)
  for name,infos in pairs(event.players)
  do
    if(name == pl_name)
    then
      return infos;
    end
  end
  return nil;
end

local function _GEM3_SUB_IncreaseCount(event,tab,queue)
  event.classes[tab.class][queue] = event.classes[tab.class][queue] + 1;
  -- Tank
  if(tab.role == GEM3_ROLE_TANK) then
    event.roles[GEM3_ROLE_TANK][queue] = event.roles[GEM3_ROLE_TANK][queue] + 1;
  end
  -- CDps
  if(tab.role == GEM3_ROLE_CDPS) then
    event.roles[GEM3_ROLE_CDPS][queue] = event.roles[GEM3_ROLE_CDPS][queue] + 1;
    event.roles[GEM3_ROLE_DPS][queue] = event.roles[GEM3_ROLE_DPS][queue] + 1;
  end
  -- RDps
  if(tab.role == GEM3_ROLE_RDPS) then
    event.roles[GEM3_ROLE_RDPS][queue] = event.roles[GEM3_ROLE_RDPS][queue] + 1;
    event.roles[GEM3_ROLE_DPS][queue] = event.roles[GEM3_ROLE_DPS][queue] + 1;
  end
  -- Heal
  if(tab.role == GEM3_ROLE_HEAL) then
    event.roles[GEM3_ROLE_HEAL][queue] = event.roles[GEM3_ROLE_HEAL][queue] + 1;
  end
  event[queue] = event[queue] + 1;
end


--------------- Exported functions ---------------

function GEM3_SUB_CheckMySubscription(event)
  --local my_subs = GA_GetTable();

  for name,player in pairs(event.players) -- player is a GEM3_Player
  do
    if(GEM3_IsMyReroll(name)) -- One of my reroll
    then
      local infos = GEM3_SUB_GetSubscribedInfos(event.id,name); -- returns a GEM3_Subscription
      if(infos) -- I subscribed to this event myself
      then
        if(infos.state ~= player.current_queue) -- My queue changed
        then
          infos.state = player.current_queue;
          -- GUI Callback
          GEM3_TriggerCallback("OnSubscriptionChanged",event.id,name,player.current_queue);
        end
      else -- I haven't subscribed myself, or lost the infos -> Rebuild the infos
        GEM3_SUB_SubscribeMyself(event.id,name,player.guild,player.level,player.class,player.role,player.comment,player.force_queue,player.current_queue,player.alt_role,player.main_time,player.update_time,player.source);
        -- GUI Callback
        GEM3_TriggerCallback("OnSubscribed",event.id,name,player.current_queue,player.source);
      end
      --my_subs[name] = true; -- This toon of mine is in subscribers list
    end
  end

  -- Check for my unsubscriptions & lost subscriptions
  for name,infos in pairs(GEM3_QA_Events.subscribed[event.id])
  do
    --[[if(infos.state == GEM3_SUB_STATE_UNSUBSCRIBING) -- Unsubscribing, check if it is done  <- Ce cas n'existe plus, maintenant on "NOT_COMING" au lieu de "UNSUBSCRIBE"
    then
      if(my_subs[name] == nil) -- It is done
      then
        -- GUI Callback
        GEM3_TriggerCallback("OnUnSubscribed",event.id,name,infos.unsub_comment);
        GEM3_SUB_UnsubscribeMyself(event.id,name);
      end
    else]]if(infos.state ~= GEM3_SUB_STATE_NOT_ACKED)
    then
      local found = false;
      -- Search myself in the list
      for n in pairs(event.players)
      do
        if(n == name)
        then
          found = true;
          break;
        end
      end
      if(not found)
      then
        -- GUI Callback
        GEM3_TriggerCallback("OnSubscriptionLost",event.id,name);
      end
    end
  end
  --GA_ReleaseTable(my_subs);
end

-- Returns a GEM3_Player struct
function GEM3_SUB_BuildSubscriberInfos(name,guild,level,class,role,comment,force_queue,update_time,alt_role,main_time,source)
  local pl_sub = GA_GetTable();
  pl_sub.name = name;
  pl_sub.class = class;
  pl_sub.role = role;
  pl_sub.level = level;
  pl_sub.guild = guild;
  pl_sub.comment = comment;
  pl_sub.force_queue = force_queue;
  pl_sub.update_time = update_time;
  pl_sub.lead_force_queue = GEM3_SUB_FORCE_QUEUE_NONE;
  if(alt_role == role) -- Same role noob
  then
    alt_role = GEM3_ROLE_UNKNOWN;
  end
  pl_sub.alt_role = alt_role;
  pl_sub.main_time = main_time;
  if(source == nil)
  then
    source = GEM3_SUB_SOURCE_SELF;
  end
  pl_sub.source = source;
  
  return pl_sub;
end

-- infos is GEM3_Subscription
function GEM3_SUB_TouchSubscription(event,infos,immediat)
  if(infos)
  then
    infos.update_time = GEM3_STAMP_GetLocalStamp(); -- Touch the subscription
    if(immediat)
    then
      GEM3_COM_Subscribe(event,infos); -- Force broadcast now
    else
      GEM3_Todo("GEM3_SUB_TouchSubscription: TODO: GEM3_CMD_AddCommandToQueue");
      --GEM3_CMD_AddCommandToQueue();
    end
  end
end

--[[
 function GEM3_SUB_SubscribeMyself:
  Sets one of my toons as having subscribed to an event. Locally.
   ev_id       : String -- The event I have subscribed to
   name        : String -- Name of my toon
   guild       : String -- Guild of my toon (or nil)
   level       : Int    -- My current level
   class       : String -- My class
   role        : Int -- Role of my class (one of GEM3_ROLE_xx)
   comment     : String -- My subscription comment
   force_queue : Int    -- Queue I want to be forced into (one of GEM3_SUB_FORCE_QUEUE_xx)
   state       : Int    -- Current subscription state (one of GEM3_SUB_STATE_xx)
   alt_role    : Int -- Alternative role of my class (one of GEM3_ROLE_xx)
   main_time   : Int -- Date of the main subsciption
   update_time : Int -- Subscription updated time
   source      : Str -- Subscription source (one of GEM3_SUB_SOURCE_xx)
  Returns:
   Infos       : Struct or Nil -- Local subscriber struct, or nil if error (GEM3_Subscription struct)
   Reason      : String  -- Failure message, Nil if error
]]
function GEM3_SUB_SubscribeMyself(ev_id,name,guild,level,class,role,comment,force_queue,state,alt_role,main_time,update_time,source)
  local infos = GEM3_SUB_GetSubscribedInfos(ev_id,name); -- returns a GEM3_Subscription

  if(infos ~= nil)
  then
    return nil,GEM3_LOC_SUB_SUBSCRIBE_ALREADY_SUBSCRIBED;
  end

  infos = GA_WipeTable(infos,10);
  infos.name = name;
  infos.class = class;
  infos.role = role;
  infos.guild = guild;
  infos.level = level;
  infos.comment = comment;
  infos.force_queue = force_queue;
  infos.state = state;
  infos.main_time = main_time;
  infos.update_time = update_time;
  infos.alt_role = alt_role;
  infos.source = source;
  GEM3_QA_Events.subscribed[ev_id][name] = infos;

  GEM3_QA_Events.kicked[ev_id][name] = nil; -- Unkick me if I was

  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_ChatDebug(GEM3_DEBUG_SUBSCRIBERS,"GEM3_SUB_SubscribeMyself: Setting myself "..name.." as having subscribed to EventID "..ev_id.." with State="..state);
  end

  return infos,"";
end

--[[
 function GEM3_SUB_UnsubscribeMyself:
  Sets one of my toons as having unsubscribed to an event. Locally.
   ev_id       : String -- The event I have subscribed to
   name        : String -- Name of my toon
  Returns:
   Result      : Boolean -- True if ok
   Reason      : String  -- Failure message, Nil if error
]]
function GEM3_SUB_UnsubscribeMyself(ev_id,name)
  local infos = GEM3_SUB_GetSubscribedInfos(ev_id,name);
  
  if(infos == nil)
  then
    return false,GEM3_LOC_SUB_UNSUBSCRIBE_NOT_SUBSCRIBED;
  end

  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_ChatDebug(GEM3_DEBUG_SUBSCRIBERS,"GEM3_SUB_UnsubscribeMyself: Setting myself "..infos.name.." as having unsubscribed from EventID "..ev_id);
  end

  GEM3_QA_Events.subscribed[ev_id][infos.name] = GA_ReleaseTable(infos);

  return true,"";
end

-- Returns subscribed state (one of GEM3_SUB_STATE_xxx), and error string if GEM3_SUB_STATE_NONE
-- pl_sub is a GEM3_Player struct
function GEM3_SUB_CreateSubscriber(ev_id,pl_sub)
  GEM3_ChatDebug(GEM3_DEBUG_SUBSCRIBERS,"GEM3_SUB_CreateSubscriber: Received Subscription for EventID="..ev_id.." From="..pl_sub.name.." Class="..pl_sub.class.." Role="..pl_sub.role.." Level="..pl_sub.level.." Guild="..tostring(pl_sub.guild).." Comment="..tostring(pl_sub.comment).." Queue="..pl_sub.force_queue.." UpdateTime="..pl_sub.update_time);

  local event = GEM3_QA_Events.events[ev_id];
  if(event == nil)
  then
    return GEM3_SUB_STATE_NONE,GEM3_LOC_SUB_CREATESUB_UNKNOWN_EVENT;
  end

  -- Check if player has already subscribed, then update his info if update_time is more recent
  local existing = _GEM3_SUB_GetSubscriberInfos(event,pl_sub.name); -- returns a GEM3_Player
  if(existing) -- Already subscribed, check update_time
  then
    if(pl_sub.update_time > existing.update_time) -- Newer, update it
    then
      -- Retrieve forced value, only if player has not choose NC
      --if(pl_sub.force_queue ~= GEM3_SUB_FORCE_QUEUE_NOT_COMING)
      --then
        pl_sub.lead_force_queue = existing.lead_force_queue;
      --end
      GA_ReleaseTable(existing); -- Remove previous infos
      GEM3_ChatDebug(GEM3_DEBUG_SUBSCRIBERS,"GEM3_SUB_CreateSubscriber: Player already subscribed, updating his infos");
    else
      GA_ReleaseTable(pl_sub);
      return GEM3_SUB_STATE_DISCARD;
    end
  end

  -- Add subscriber
  event.players[pl_sub.name] = pl_sub;

  -- Sort subscribers
  GEM3_SORT_SortPlayers(event);

  -- Schedule an update
  GEM3_EVT_TouchEvent(event,false);

  -- Update counts
  GEM3_SUB_UpdateCounts(event);
  
  -- Check my subscription, if someone else subscribed me
  GEM3_SUB_CheckMySubscription(event);

  -- GUI Callback
  GEM3_TriggerCallback("OnUpdatedEvent",event,false);
  
  return pl_sub.current_queue,"";
end

function GEM3_SUB_RemoveSubscriber(event,pl_name,comment)
  GEM3_ChatDebug(GEM3_DEBUG_SUBSCRIBERS,"GEM3_SUB_RemoveSubscriber: EventID="..event.id.." From="..tostring(pl_name).." Comment="..tostring(comment));

  local existing = _GEM3_SUB_GetSubscriberInfos(event,pl_name);
  if(existing) -- Existing
  then
    -- GUI Callback
    GEM3_TriggerCallback("OnUnSubscribed",event.id,existing.name,comment);

    -- Remove subscriber
    event.players[existing.name] = GA_ReleaseTable(existing,10);

    -- Sort subscribers
    GEM3_SORT_SortPlayers(event);

    -- Schedule an update
    GEM3_EVT_TouchEvent(event,false);

    -- Update counts
    GEM3_SUB_UpdateCounts(event);

    return true,nil;
  end
  return false,GEM3_LOC_SUB_UNSUBSCRIBE_NOT_SUBSCRIBED;
end

function GEM3_SUB_CheckPlayersLevel(event)
  local to_remove = GA_GetTable();

  for name,tab in pairs(event.players)
  do
    if(tab.level < event.min_lvl or tab.level > event.max_lvl) -- Not in level range anymore
    then
      tinsert(to_remove,name);
      GEM3_ChatDebug(GEM3_DEBUG_SUBSCRIBERS,"GEM3_SUB_CheckPlayersLevel: Player "..name.." no longer matches required level range, must kick him");
    end
  end
  
  for _,name in ipairs(to_remove)
  do
    if(GEM3_IsMyReroll(name))
    then
      GEM3_SUB_UnsubscribeMyself(event.id,name);
    else
      GEM3_COM_SubscribeError(event,name,GEM3_SUB_ERROR_LEVEL_RANGE);
    end
    GEM3_SUB_RemoveSubscriber(event,name,GEM3_LOC_SUB_REMOVE_SUBSCRIBER_LEVEL_RANGE);
  end
  GA_ReleaseTable(to_remove);
end

function GEM3_SUB_UpdateCounts(event)
  -- Reset counts
  for n,tab in pairs(event.classes)
  do
    tab.tit_count = 0;
    tab.sub_count = 0;
    tab.repl_count = 0;
    tab.rej_count = 0;
  end
  for id,tab in pairs(event.roles)
  do
    tab.tit_count = 0;
    tab.sub_count = 0;
    tab.repl_count = 0;
    tab.rej_count = 0;
  end
  event.tit_count = 0;
  event.sub_count = 0;
  event.repl_count = 0;
  event.rej_count = 0;

  -- Fill counts
  for n,tab in pairs(event.players)
  do
    if(tab.current_queue == GEM3_SUB_STATE_TITULAR)
    then
      _GEM3_SUB_IncreaseCount(event,tab,"tit_count");
    elseif(tab.current_queue == GEM3_SUB_STATE_SUBSTITUTE)
    then
      _GEM3_SUB_IncreaseCount(event,tab,"sub_count");
    elseif(tab.current_queue == GEM3_SUB_STATE_REPLACEMENT)
    then
      _GEM3_SUB_IncreaseCount(event,tab,"repl_count");
    elseif(tab.current_queue == GEM3_SUB_STATE_NOT_COMING)
    then
      _GEM3_SUB_IncreaseCount(event,tab,"rej_count");
    end
  end
end

function GEM3_SUB_SubscriptionFailed(ev_id,pl_name,code)
  local result = GEM3_SUB_UnsubscribeMyself(ev_id,pl_name);
  
  if(result)
  then
    -- GUI Callback
    GEM3_TriggerCallback("OnSubscriptionFailed",ev_id,pl_name,code);
  end
end

function GEM3_SUB_SetKicked(ev_id,pl_name,reason)
  GEM3_SUB_UnsubscribeMyself(ev_id,pl_name);
  if(GEM3_QA_Events.kicked[ev_id] ~= nil)
  then
    GEM3_QA_Events.kicked[ev_id][pl_name] = reason;
  end
end

function GEM3_SUB_SetBanned(ev_id,pl_name,reason)
  GEM3_SUB_UnsubscribeMyself(ev_id,pl_name);
  if(GEM3_QA_Events.banned[ev_id] ~= nil)
  then
    GEM3_QA_Events.banned[ev_id][pl_name] = reason;
  end
end

function GEM3_SUB_SetUnBanned(ev_id,pl_name)
  if(GEM3_QA_Events.banned[ev_id] ~= nil)
  then
    GEM3_QA_Events.banned[ev_id][pl_name] = nil;
  end
end


--------------- API Exported functions ---------------

-- Returns a GEM3_Subscription struct, or nil. If "pl_name" is nil, get any subscribed infos for this event
function GEM3_SUB_GetSubscribedInfos(ev_id,pl_name)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_DBG_CheckTypes("GEM3_SUB_GetSubscribedInfos",ev_id,"ev_id","string",pl_name,"pl_name","string-nil");
  end
  local any_inf = nil;
  if(GEM3_QA_Events.subscribed[ev_id])
  then
    for name,infos in pairs(GEM3_QA_Events.subscribed[ev_id])
    do
      if(pl_name == nil)
      then
        any_inf = infos;
      elseif(name == pl_name)
      then
        return infos;
      end
    end
  end
  return any_inf;
end

-- Returns true if "pl_name" is banned from "ev_id". If "pl_name" is nil, search for any banned reroll from "ev_id"
function GEM3_SUB_IsPlayerBanned(ev_id,pl_name)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_DBG_CheckTypes("GEM3_SUB_IsPlayerBanned",ev_id,"ev_id","string",pl_name,"pl_name","string-nil");
  end
  if(GEM3_QA_Events.banned[ev_id] == nil)
  then
    return false,nil;
  end

  for name,reason in pairs(GEM3_QA_Events.banned[ev_id])
  do
    if(pl_name == nil and GEM3_IsMyReroll(name))
    then
      return true,reason;
    elseif(name == pl_name)
    then
      return true,reason;
    end
  end
  return false,nil;
end

-- Returns true if "pl_name" (being one of my rerolls) is kicked from "ev_id". If "pl_name" is nil, search for any kicked reroll from "ev_id"
function GEM3_SUB_IsPlayerKicked(ev_id,pl_name)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_DBG_CheckTypes("GEM3_SUB_IsPlayerKicked",ev_id,"ev_id","string",pl_name,"pl_name","string-nil");
  end
  if(GEM3_QA_Events.kicked[ev_id] == nil)
  then
    return false,nil;
  end

  for name,reason in pairs(GEM3_QA_Events.kicked[ev_id])
  do
    if(pl_name == nil)
    then
      return true,reason;
    elseif(name == pl_name)
    then
      return true,reason;
    end
  end
  return false,nil;
end

--[[
 function GEM3_SUB_Subscribe:
  Try to subscribe to an event
   ev_id       : String -- The event I have subscribed to
   pl_name     : String -- Name of my toon
   guild       : String -- Guild of my toon (or nil)
   level       : Int    -- My current level
   class       : String -- My class
   role        : Int    -- Role of my class (one of GEM3_ROLE_xx)
   comment     : String -- My subscription comment
   force_queue : Int    -- Queue I want to be forced into (one of GEM3_SUB_FORCE_QUEUE_xx)
   alt_role    : Int    -- Alternative role of my class (one of GEM3_ROLE_xx)
  Returns:
   Result      : Boolean -- True if success, false otherwise
   Reason      : String  -- Failure message, Nil if Result is true
]]
function GEM3_SUB_Subscribe(ev_id,pl_name,guild,level,class,role,comment,force_queue,alt_role)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_DBG_CheckTypes("GEM3_SUB_Subscribe",ev_id,"ev_id","string",pl_name,"pl_name","string",guild,"guild","string",level,"level","number",class,"class","string",role,"role","number",comment,"comment","string",force_queue,"force_queue","number",alt_role,"alt_role","number");
  end
  local event = GEM3_QA_Events.events[ev_id];
  if(event == nil)
  then
    return false,GEM3_LOC_SUB_SUBSCRIBE_UNKNOWN_EVENT;
  end
  local thisReroll,anyReroll = GEM3_CHAN_IsChannelConfiguredForAnyReroll(event.channel);
  if(anyReroll == false)
  then
    return false,GEM3_LOC_SUB_SUBSCRIBE_NOT_IN_CHANNEL;
  end
  if(_GEM3_SUB_CheckDeadline(event))
  then
    return false,GEM3_LOC_SUB_SUBSCRIBE_DEADLINE;
  end
  if(level < event.min_lvl or level > event.max_lvl)
  then
    return false,GEM3_LOC_SUB_SUBSCRIBE_LEVEL;
  end
  if(not GEM3_IsMyReroll(pl_name))
  then
    return false,GEM3_LOC_SUB_SUBSCRIBE_NOT_MY_REROLL;
  end

  if(force_queue == GEM3_SUB_FORCE_QUEUE_NONE) -- None means Titular, if available
  then
    force_queue = GEM3_SUB_FORCE_QUEUE_TITULAR;
  end
  if(GEM3_IsMyReroll(event.leader)) -- I'm the leader, don't need to ask permission to subscribe ;)
  then
    local stamp = GEM3_STAMP_GetLocalStamp();
    local infos,reason = GEM3_SUB_SubscribeMyself(ev_id,pl_name,guild,level,class,role,comment,force_queue,GEM3_SUB_STATE_NOT_ACKED,alt_role,stamp,stamp,GEM3_SUB_SOURCE_SELF); -- returns a GEM3_Subscription
    if(infos)
    then
      local pl_sub = GEM3_SUB_BuildSubscriberInfos(pl_name,guild,level,class,role,comment,force_queue,infos.update_time,infos.alt_role,infos.main_time,GEM3_SUB_SOURCE_SELF); -- Create subscriber struct (GEM3_Player)
      local state,reason = GEM3_SUB_CreateSubscriber(ev_id,pl_sub);
      if(state == GEM3_SUB_STATE_NONE)
      then
        GEM3_SUB_UnsubscribeMyself(ev_id,pl_name);
        return false,reason;
      elseif(state == GEM3_SUB_STATE_DISCARD)
      then
        return false,GEM3_INTERNAL_ERROR_1;
      end
      infos.state = state; -- Update my state
      -- GUI Callback
      GEM3_TriggerCallback("OnSubscribed",ev_id,pl_name,infos.state,GEM3_SUB_SOURCE_SELF);
      return true,nil;
    end
    return false,reason;
  else -- Not the leader, some checks are required before sending request
    if(GEM3_SUB_IsPlayerBanned(ev_id,pl_name)) -- Check for banned status
    then
      return false,GEM3_LOC_SUB_SUBSCRIBE_BANNED;
    end
    local infos = GEM3_SUB_GetSubscribedInfos(ev_id,pl_name);
    if(infos) -- Check for already subscribed status
    then
      if(infos.state == GEM3_SUB_STATE_NOT_ACKED)
      then
        return false,GEM3_LOC_SUB_SUBSCRIBE_PENDING;
      else
        return false,GEM3_LOC_SUB_SUBSCRIBE_ALREADY_SUBSCRIBED;
      end
    end
    local stamp = GEM3_STAMP_GetLocalStamp();
    local infos,reason = GEM3_SUB_SubscribeMyself(ev_id,pl_name,guild,level,class,role,comment,force_queue,GEM3_SUB_STATE_NOT_ACKED,alt_role,stamp,stamp,GEM3_SUB_SOURCE_SELF); -- returns a GEM3_Subscription
    if(infos)
    then
      GEM3_SUB_TouchSubscription(event,infos,true);
      return true,nil;
    end
    return false,reason;
  end
  return false,"GEM3_SUB_Subscribe: Inform Kiki!!"; -- Should never happen :)
end

function GEM3_SUB_SubscribeUpdate(ev_id,pl_name,guild,level,class,role,comment,force_queue,alt_role)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_DBG_CheckTypes("GEM3_SUB_SubscribeUpdate",ev_id,"ev_id","string",pl_name,"pl_name","string",guild,"guild","string",level,"level","number",class,"class","string",role,"role","number",comment,"comment","string",force_queue,"force_queue","number",alt_role,"alt_role","number");
  end
  local event = GEM3_QA_Events.events[ev_id];
  if(event == nil)
  then
    return false,GEM3_LOC_SUB_SUBSCRIBE_UPDATE_UNKNOWN_EVENT;
  end
  if(not GEM3_IsMyReroll(pl_name))
  then
    return false,GEM3_LOC_SUB_SUBSCRIBE_UPDATE_NOT_MY_REROLL;
  end
  local thisReroll,anyReroll = GEM3_CHAN_IsChannelConfiguredForAnyReroll(event.channel);
  if(anyReroll == false)
  then
    return false,GEM3_LOC_SUB_RESUBSCRIBE_NOT_IN_CHANNEL;
  end

  local infos = GEM3_SUB_GetSubscribedInfos(ev_id,pl_name);
  if(infos == nil)
  then
    return false,GEM3_LOC_SUB_SUBSCRIBE_UPDATE_NOT_SUBSCRIBED;
  end
  if(_GEM3_SUB_CheckDeadline(event)) -- Deadline passed
  then
    if(force_queue ~= GEM3_SUB_FORCE_QUEUE_NOT_COMING) -- Cannot change subscription after deadline
    then
      return false,GEM3_LOC_SUB_SUBSCRIBE_UPDATE_DEADLINE;
    end
  end
  if(level < event.min_lvl or level > event.max_lvl)
  then
    return false,GEM3_LOC_SUB_SUBSCRIBE_UPDATE_LEVEL;
  end

  if(force_queue == GEM3_SUB_FORCE_QUEUE_NONE) -- None means Titular, if available
  then
    force_queue = GEM3_SUB_FORCE_QUEUE_TITULAR;
  end
  
  -- Check same infos
  if(infos.guild == guild and infos.level == level and infos.role == role and infos.comment == comment and infos.force_queue == force_queue and infos.alt_role == alt_role) -- No need to check class ^^
  then
    return false,GEM3_LOC_SUB_SUBSCRIBE_UPDATE_SAME;
  end

  if(infos.role ~= role or infos.force_queue ~= force_queue) -- Major change, update main_time
  then
    infos.main_time = GEM3_STAMP_GetLocalStamp();
  end
  infos.guild = guild;
  infos.level = level;
  infos.role = role;
  infos.comment = comment;
  infos.force_queue = force_queue;
  infos.alt_role = alt_role;
  
  if(GEM3_IsMyReroll(event.leader)) -- I'm the leader, don't send infos
  then
    local pl_sub = GEM3_SUB_BuildSubscriberInfos(pl_name,guild,level,class,role,comment,force_queue,GEM3_STAMP_GetLocalStamp(),alt_role,infos.main_time,GEM3_SUB_SOURCE_SELF); -- Create subscriber struct (GEM3_Player)
    local state,reason = GEM3_SUB_CreateSubscriber(ev_id,pl_sub);
    if(state == GEM3_SUB_STATE_NONE)
    then
      GEM3_SUB_UnsubscribeMyself(ev_id,pl_name);
      return false,reason;
    elseif(state == GEM3_SUB_STATE_DISCARD)
    then
      return false,GEM3_LOC_SUB_SUBSCRIBE_UPDATE_TOO_SOON;
    end
    GEM3_SUB_CheckMySubscription(event);
  else
    infos.state = GEM3_SUB_STATE_NOT_ACKED;
    GEM3_SUB_TouchSubscription(event,infos,true);
  end
  return true,nil;
end

--[[
 function GEM3_SUB_SubscribeExternal:
  Submit an external subscription to an event
   ev_id       : String -- The event id
   pl_name     : String -- Name of the player
   guild       : String -- Guild of the player (or nil)
   level       : Int    -- Current level of the player
   class       : String -- Class of the player
   role        : Int    -- Role of the player (one of GEM3_ROLE_xx)
   comment     : String -- Player's subscription comment
   force_queue : Int    -- Queue to be forced into (one of GEM3_SUB_FORCE_QUEUE_xx)
   alt_role    : Int    -- Alternative role of my class (one of GEM3_ROLE_xx)
  Returns:
   Result      : Boolean -- True if success, false otherwise
   Reason      : String  -- Failure message, Nil if Result is true
]]
function GEM3_SUB_SubscribeExternal(ev_id,pl_name,guild,level,class,role,comment,force_queue,alt_role)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_DBG_CheckTypes("GEM3_SUB_SubscribeExternal",ev_id,"ev_id","string",pl_name,"pl_name","string",guild,"guild","string",level,"level","number",class,"class","string",role,"role","number",comment,"comment","string",force_queue,"force_queue","number",alt_role,"alt_role","number");
  end
  local event = GEM3_QA_Events.events[ev_id];
  if(event == nil)
  then
    return false,GEM3_LOC_SUB_SUBSCRIBE_UNKNOWN_EVENT;
  end
  local thisReroll,anyReroll = GEM3_CHAN_IsChannelConfiguredForAnyReroll(event.channel);
  if(anyReroll == false)
  then
    return false,GEM3_LOC_SUB_SUBSCRIBE_NOT_IN_CHANNEL;
  end
  if(_GEM3_SUB_CheckDeadline(event))
  then
    return false,GEM3_LOC_SUB_SUBSCRIBE_DEADLINE;
  end
  if(level < event.min_lvl or level > event.max_lvl)
  then
    return false,GEM3_LOC_SUB_SUBSCRIBE_LEVEL;
  end

  if(force_queue == GEM3_SUB_FORCE_QUEUE_NONE) -- None means Titular, if available
  then
    force_queue = GEM3_SUB_FORCE_QUEUE_TITULAR;
  end
  if(GEM3_IsMyReroll(event.leader)) -- I'm the leader, don't need to ask permission to subscribe ;)
  then
    local stamp = GEM3_STAMP_GetLocalStamp();
    local pl_sub = GEM3_SUB_BuildSubscriberInfos(pl_name,guild,level,class,role,comment,force_queue,stamp,alt_role,stamp,GEM3_SUB_SOURCE_EXTERNAL..GEM3_PlayerName); -- Create subscriber struct (GEM3_Player)
    local state,reason = GEM3_SUB_CreateSubscriber(ev_id,pl_sub);
    if(state == GEM3_SUB_STATE_NONE)
    then
      return false,reason;
    elseif(state == GEM3_SUB_STATE_DISCARD)
    then
      return false,GEM3_INTERNAL_ERROR_1;
    end
    return true,nil;
  else -- Not the leader, use sync module to create subscriber
    if(not GEM3_EVT_AmIAssistant(ev_id))
    then
      return false,GEM3_LOC_SUB_SUBSCRIBE_NOT_ASSISTANT;
    end
    local command = GA_GetTable();
    local stamp = GEM3_STAMP_GetLocalStamp();
    command["protocol"] = GEM3_COM_PROTOCOL_REVISION;
    command["cmd"] = "Subscribe";
    command["channel"] = event.channel;
    command["ev_id"] = ev_id;
    command["leader"] = event.leader;
    command["ev_date"] = event.ev_date;
    command["pl_name"] = pl_name;
    command["guild"] = guild;
    command["level"] = level;
    command["class"] = class;
    command["role"] = role;
    command["comment"] = comment;
    command["queue"] = force_queue;
    command["alt_role"] = alt_role;
    command["main_time"] = stamp;
    command["update_time"] = stamp;
    command["source"] = GEM3_SUB_SOURCE_EXTERNAL..GEM3_PlayerName;
    local result = GEM3_SYNC_ProcessCommand(command);
    GA_ReleaseTable(command);
    if(result == false)
    then
      return false,GEM3_INTERNAL_ERROR_2;
    end
    local pl_cmd = GEM3_CMD_GetNonAckedSubscription(ev_id,pl_name);
    if(pl_cmd) -- Not an error if pl_cmd is nil, if my command was ignored by the core (older command, for example)
    then
      GEM3_QUE_ScheduleCommand(pl_cmd,true);
    end
    return true,nil;
  end
  return false,"GEM3_SUB_SubscribeExternal: Inform Kiki!!"; -- Should never happen :)
end

--[[
 function GEM3_SUB_ResendSubscription:
  Resend subscription to the event (only when leader is connected)
   ev_id       : String -- The event I have subscribed to
   pl_name     : String -- Name of my toon
  Returns:
   Result      : Boolean -- True if success, false otherwise
   Reason      : String  -- Failure message, Nil if Result is true
]]
function GEM3_SUB_ResendSubscription(ev_id,pl_name)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_DBG_CheckTypes("GEM3_SUB_ResendSubscription",ev_id,"ev_id","string",pl_name,"pl_name","string");
  end
  -- Check if status is pending, then touchSubscription
  local event = GEM3_QA_Events.events[ev_id];

  if(event)
  then
    if(GEM3_PLAY_IsPlayerInChannel(event.channel,event.leader))
    then
      local infos = GEM3_SUB_GetSubscribedInfos(ev_id,pl_name);
      if(infos)
      then
        if(infos.state == GEM3_SUB_STATE_NOT_ACKED)
        then
          GEM3_SUB_TouchSubscription(event,infos,true);
          return true,nil;
        else
          return false,GEM3_LOC_SUB_RESUBSCRIBE_NOT_PENDING;
        end
      else
        return false,GEM3_LOC_SUB_RESUBSCRIBE_NO_SUB_INFOS;
      end
    else
      return false,GEM3_LOC_SUB_RESUBSCRIBE_LEADER_NOT_CONNECTED;
    end
  else
    return false,GEM3_LOC_SUB_RESUBSCRIBE_UNKNOWN_EVENT;
  end
end

-- Leader can force a player into a queue
function GEM3_SUB_ForceQueueForPlayer(ev_id,pl_name,forced_queue)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_DBG_CheckTypes("GEM3_SUB_ForceQueueForPlayer",ev_id,"ev_id","string",pl_name,"pl_name","string",forced_queue,"forced_queue","number");
  end
  local event = GEM3_QA_Events.events[ev_id];

  if(event and GEM3_IsMyReroll(event.leader) and event.players[pl_name])
  then
    local current_queue = event.players[pl_name].current_queue;

    if(event.players[pl_name].lead_force_queue ~= forced_queue) -- Not forcing same value
    then
      event.players[pl_name].lead_force_queue = forced_queue;

      -- Sort subscribers
      GEM3_SORT_SortPlayers(event);

      GEM3_SUB_CheckMySubscription(event); -- Check if my subscription changed, if I forced myself (or ejected myself by forcing someone else into titular)

      -- Update counts
      GEM3_SUB_UpdateCounts(event);

      -- GUI Callback
      GEM3_TriggerCallback("OnUpdatedEvent",event,false);

      -- Schedule an update in all cases, we want to share our "forced" value with everybody
      GEM3_EVT_TouchEvent(event,false); -- Don't update immediatly, maybe I want to force someone else
    end
  end
end

--[[
 function GEM3_SUB_IsLatestSubscriptionReceived:
  Checks if my latest subscription informations has been received by the leader
   ev_id       : String -- The event I have subscribed to
   pl_name     : String -- Name of my toon
  Returns:
   Result      : Boolean -- True if leader acked my latest subscription infos
]]
function GEM3_SUB_IsLatestSubscriptionReceived(ev_id,pl_name)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_DBG_CheckTypes("GEM3_SUB_IsLatestSubscriptionReceived",ev_id,"ev_id","string",pl_name,"pl_name","string");
  end

  if(GEM3_IsMyReroll(pl_name))
  then
    -- Get the CMD struct for this Event
    local cmd_infos = GEM3_CMD_GetCommandInfosForEvent(ev_id);
    if(cmd_infos == nil)
    then
      return false;
    end
    return GEM3_CMD_IsLatestCommandAcked(cmd_infos,pl_name,GEM3_CMD_CMD_SUBSCRIBE);
  end
  return false;
end

