--[[
  Guild Event Manager (GEM) - Version 3
  by Kiki from European "Conseil des Ombres" (Horde)
  ----------------------------------------
  Queues handling
]]

--[[
  Exported API Functions:

]]

--------------- Local variables ---------------
local _GEM3_QUE_queue = {};
local _GEM3_QUE_MAX_CMD_SEND_PER_SCHEDULE = 2;

--------------- Internal functions ---------------

local function _GEM3_QUE_BuildCommandsQueue(channel)
  local immediat = GEM3_CHAN_AmILeader(channel);

  for ev_id,cmdtab in pairs(GEM3_QA_Events.commands)
  do
    if(cmdtab.channel == channel)
    then
      if(GEM3_EVT_IsEventIgnored(ev_id)) -- Check for ignored event
      then
        GEM3_ChatDebug(GEM3_DEBUG_QUEUES,"_GEM3_QUE_BuildCommandsQueue: EventID "..ev_id.." is ignored");
      else
        for player,plcmd_infos in pairs(cmdtab.cmds)
        do
          for _,pl_cmd in ipairs(plcmd_infos)
          do
            if(not GEM3_CMD_IsCommandAcked(pl_cmd))
            then
              if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
                GEM3_ChatDebug(GEM3_DEBUG_QUEUES,"_GEM3_QUE_BuildCommandsQueue: "..channel.." ("..tostring(immediat)..") scheduling Cmd="..pl_cmd.cmd.." from player "..player);
              end
              GEM3_QUE_ScheduleCommand(pl_cmd,immediat);
            end
          end
        end
      end
    end
  end
end

local function _GEM3_QUE_BroadcastEvent_Schedule(ev_id) -- Scheduled CB function
  local event = GEM3_QA_Events.events[ev_id];

  if(event) -- Still valid
  then
    GEM3_ChatDebug(GEM3_DEBUG_QUEUES,"_GEM3_QUE_BroadcastEvent_Schedule: Broadcasting EventID "..ev_id);
    GEM3_COM_BroadcastEvent(event);
  end
end

local function _GEM3_QUE_BroadcastCommand_Schedule() -- Scheduled CB function
  local i = 0;

  GEM3_ChatDebug(GEM3_DEBUG_QUEUES,"_GEM3_QUE_BroadcastCommand_Schedule: Start bcast queue");
  while(_GEM3_QUE_queue[1])
  do
    if(i >= _GEM3_QUE_MAX_CMD_SEND_PER_SCHEDULE)
    then
      break;
    end
    GEM3_COM_SendCommand(_GEM3_QUE_queue[1]);
    tremove(_GEM3_QUE_queue,1);
    i = i + 1;
  end

  if(#_GEM3_QUE_queue ~= 0)
  then
    Chronos.scheduleByName("GEM3CommandsQueue",2,_GEM3_QUE_BroadcastCommand_Schedule);
    GEM3_ChatDebug(GEM3_DEBUG_QUEUES,"_GEM3_QUE_BroadcastCommand_Schedule: Too many commands to send, rescheduling in 2sec");
  end
  GEM3_ChatDebug(GEM3_DEBUG_QUEUES,"_GEM3_QUE_BroadcastCommand_Schedule: Done");
end


--------------- Exported functions ---------------

function GEM3_QUE_GetDebugInfos()
  local _evt_count = 0;
  
  for n,v in Chronos.getScheduledByName()
  do
    if(string.find(n,"^GEM3Queue_.+"))
    then
      _evt_count = _evt_count + 1;
    end
  end

  return #_GEM3_QUE_queue,_evt_count;
end

-- Adds an event to send queue. Send it (almost) now if channel leader, queue it otherwise
function GEM3_QUE_AddEventToQueue(channel,ev_id,adddelay)
  local delay = adddelay;

  if(not GEM3_CHAN_AmILeader(channel))
  then
    delay = delay + GEM3_QA_Config.my_bcast_offset;
  end

  GEM3_ChatDebug(GEM3_DEBUG_QUEUES,"GEM3_QUE_AddEventToQueue: Scheduling EventID "..ev_id.." in "..delay.." sec");
  Chronos.scheduleByName("GEM3Queue_"..ev_id,delay,_GEM3_QUE_BroadcastEvent_Schedule,ev_id); -- Schedule (or reschedule, if existing) event broadcast
end

function GEM3_QUE_RemoveEventFromQueue(channel,ev_id)
  GEM3_ChatDebug(GEM3_DEBUG_QUEUES,"GEM3_QUE_RemoveEventFromQueue: Removing EventID "..ev_id.." from broadcast queues");
  Chronos.unscheduleByName("GEM3Notif_"..ev_id); -- Remove from leader notify update queue
  Chronos.unscheduleByName("GEM3Queue_"..ev_id); -- Remove from non chan leader queue
end

-- pl_cmd : GEM3_PlayerCommand Struct
function GEM3_QUE_ScheduleCommand(pl_cmd,immediat)
  if(pl_cmd.cmd == nil or pl_cmd.channel == nil or pl_cmd.stamp == nil)
  then
    GEM3_ChatWarning("GEM3_QUE_ScheduleCommand: Incorrect or corrupted command!");
    return;
  end
  
  local delay = 1;
  if(not immediat)
  then
    delay = GEM3_QA_Config.my_bcast_offset + 5;
  end

  for i,cmd in ipairs(_GEM3_QUE_queue)
  do
    if(cmd == pl_cmd) -- Already scheduled?
    then
      if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
        GEM3_ChatDebug(GEM3_DEBUG_QUEUES,"GEM3_QUE_ScheduleCommand: Command ("..tostring(cmd)..") Cmd="..cmd.cmd.." src="..cmd.params[5].." dest="..cmd.params[4].." is already scheduled");
      end
      return;
    end
  end
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_ChatDebug(GEM3_DEBUG_QUEUES,"GEM3_QUE_ScheduleCommand: Scheduling command ("..tostring(pl_cmd)..") Cmd="..pl_cmd.cmd.." src="..pl_cmd.params[5].." dest="..pl_cmd.params[4].." in "..delay.." sec");
  end

  tinsert(_GEM3_QUE_queue,pl_cmd);
  if(not Chronos.isScheduledByName("GEM3CommandsQueue"))
  then
    if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
      GEM3_ChatDebug(GEM3_DEBUG_QUEUES,"GEM3_QUE_ScheduleCommand: First command. Scheduling broadcast in "..delay .." sec");
    end
    Chronos.scheduleByName("GEM3CommandsQueue",delay,_GEM3_QUE_BroadcastCommand_Schedule);
  end
end

-- pl_cmd : GEM3_PlayerCommand Struct
function GEM3_QUE_UnscheduleCommand(pl_cmd)
  for i,cmd in ipairs(_GEM3_QUE_queue)
  do
    if(cmd == pl_cmd)
    then
      if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
        GEM3_ChatDebug(GEM3_DEBUG_QUEUES,"GEM3_QUE_UnscheduleCommand: Removed command ("..tostring(cmd)..") Cmd="..cmd.cmd.." src="..cmd.params[5].." dest="..cmd.params[4]);
      end
      tremove(_GEM3_QUE_queue,i);
      if(#_GEM3_QUE_queue == 0)
      then
        if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
          GEM3_ChatDebug(GEM3_DEBUG_QUEUES,"GEM3_QUE_UnscheduleCommand: No more commands in queue");
        end
        Chronos.unscheduleByName("GEM3CommandsQueue");
      end
      return;
    end
  end
end

--[[
 function GEM3_QUE_RemoveCommands
  Remove all commands for this event.
   ev_id : String -- EventID for commands to remove from ACK list
]]
function GEM3_QUE_RemoveCommands(ev_id)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_ChatDebug(GEM3_DEBUG_QUEUES,"GEM3_QUE_RemoveCommands: Entering function for "..ev_id);
  end

  for id,cmdtab in pairs(GEM3_QA_Events.commands)
  do
    if(id == ev_id)
    then
      for player,plcmd_infos in pairs(cmdtab.cmds)
      do
        for _,pl_cmd in ipairs(plcmd_infos)
        do
          GEM3_QUE_UnscheduleCommand(pl_cmd);
        end
      end
    end
  end

  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_ChatDebug(GEM3_DEBUG_QUEUES,"GEM3_QUE_RemoveCommands: Leaving function");
  end
end

--[[
  Builds broadcast queues.
]]
function GEM3_QUE_BuildBroadcastQueues(channel,player)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_ChatDebug(GEM3_DEBUG_QUEUES,"GEM3_QUE_BuildBroadcastQueues: Entering function for "..channel.." and player "..tostring(player));
  end

  if(player ~= nil) -- Broadcast due to a player join
  then
    -- Check for player's last leave
    local last_leave = GEM3_PLAY_GetLastLeave(channel,player);
    local tim = time();

    if((last_leave + GEM3_QUE_MIN_DELAY_LOGOUT_LOGIN) > tim)
    then
      if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
        GEM3_ChatDebug(GEM3_DEBUG_QUEUES,"GEM3_QUE_BuildBroadcastQueues: "..channel..": Player "..tostring(player).." rejoining too soon the channel. Don't broadcast");
      end
      return;
    end
  end
  -- Check for expired events
  GEM3_EVT_CheckExpiredEvents();

  -- Queue events from specified channel
  local i = 0;
  for ev_id,event in pairs(GEM3_QA_Events.events)
  do
    if(GEM3_EVT_IsEventIgnored(ev_id)) -- Check for ignored event
    then
      GEM3_ChatDebug(GEM3_DEBUG_QUEUES,"GEM3_QUE_BuildBroadcastQueues: EventID "..ev_id.." is ignored");
    else
      if(GEM3_PLAY_IsPlayerIgnored(event.leader))
      then
        GEM3_ChatDebug(GEM3_DEBUG_QUEUES,"GEM3_QUE_BuildBroadcastQueues: Leader '"..event.leader.."' of EventID "..ev_id.." is ignored");
      else
        if(event.channel == channel)
        then
          GEM3_QUE_AddEventToQueue(channel,ev_id,i);
          i = i + 1;
        end
      end
    end
  end

  -- Check for expired commands
  GEM3_CMD_CheckExpiredCommands();
  
  -- Build commands
  _GEM3_QUE_BuildCommandsQueue(channel);

  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_ChatDebug(GEM3_DEBUG_QUEUES,"GEM3_QUE_BuildBroadcastQueues: Leaving function");
  end
end
