--[[
  Guild Event Manager (GEM) - Version 3
  by Kiki from European "Conseil des Ombres" (Horde)
  ----------------------------------------
  Players handling
]]

--[[
  Exported API Functions:

   GEM3_PLAY_IsPlayerInChannel(channel,player)
    Input:
     channel: string -- Name of the channel to check (real name, not GUILD constant if checking special guild channel)
     player:  string -- Name of the player to check
    Output:
     IsIn: bool -- True if player is in the joined GEM channel
    Purpose:
     Checks if a player is currently is a GEM channel

   GEM3_PLAY_GetPlayerInfos(player,channel)
    Input:
     player:  string -- Name of the player to get infos from
     channel: string or nil -- Name of the channel to get infos (nil will get infos for any channel)
    Output:
     Infos: Table -- Infos for the player, or Nil if nothing found (GEM3_ChannelPlayer struct)

   GEM3_PLAY_GetMyPlayerInfos(pl_name)
    Input:
     player:  string -- Name of my reroll to get infos from
    Output:
     Infos: Table -- Infos for the selected reroll, or Nil if nothing found (GEM3_ChannelPlayer struct)

   GEM3_PLAY_GetMyRerolls(channel)
    Input:
     channel:  string or nil -- Channel to get rerolls from, or all rerolls if nil
    Output:
     rerolls: Table of GEM3_ChannelPlayer struct -- Returned table must be freed using GA_ReleaseTable(rerolls,0) when no longer needed -- Indexed by INT (use ipairs() to process)

   GEM3_PLAY_IgnorePlayer(pl_name)
    Input:
     pl_name: string -- Name of the player
    Purpose:
     Ignores a player

   GEM3_PLAY_UnIgnorePlayer(pl_name)
    Input:
     pl_name: string -- Name of the player
    Purpose:
     Unignores a player

   GEM3_PLAY_IsPlayerIgnored(pl_name)
    Input:
     pl_name: string -- Name of the player
    Output:
     Ignored: bool -- True if player is ignored, false otherwise
    Purpose:
     Checks if a player is currently ignored

   GEM3_PLAY_GetAllowedRoles(class)
    Input:
     class: string -- Player's class
    Output:
     allowed_classes: bitfield -- Allowed roles for this class (one or many of GEM3_ROLE_xx)

   GEM3_PLAY_IsOfficer()
    Output:
     isOfficer: bool -- True if player is an officer, false otherwise
    Purpose:
     Checks if a player is an officer

]]



--------------- Shared variables ---------------

--------------- Local variables ---------------
local _GEM3_PLAY_TalentsToSpec = 
{
  ["DRUID"] = { GEM3_ROLE_RDPS,GEM3_ROLE_CDPS,GEM3_ROLE_HEAL },
  ["HUNTER"] = { GEM3_ROLE_RDPS,GEM3_ROLE_RDPS,GEM3_ROLE_RDPS },
  ["MAGE"] = { GEM3_ROLE_RDPS,GEM3_ROLE_RDPS,GEM3_ROLE_RDPS },
  ["PALADIN"] = { GEM3_ROLE_HEAL,GEM3_ROLE_TANK,GEM3_ROLE_CDPS, },
  ["PRIEST"] = { GEM3_ROLE_HEAL,GEM3_ROLE_HEAL,GEM3_ROLE_RDPS },
  ["ROGUE"] = { GEM3_ROLE_CDPS,GEM3_ROLE_CDPS,GEM3_ROLE_CDPS },
  ["SHAMAN"] = { GEM3_ROLE_RDPS,GEM3_ROLE_CDPS,GEM3_ROLE_HEAL },
  ["WARLOCK"] = { GEM3_ROLE_RDPS,GEM3_ROLE_RDPS,GEM3_ROLE_RDPS },
  ["WARRIOR"] = { GEM3_ROLE_CDPS,GEM3_ROLE_CDPS,GEM3_ROLE_TANK },
  ["DEATHKNIGHT"] = { GEM3_ROLE_CDPS,GEM3_ROLE_TANK,GEM3_ROLE_CDPS },
};

local _GEM3_PLAY_AllowedRoles =
{
  ["DRUID"] = bit.bor(GEM3_ROLE_RDPS,GEM3_ROLE_CDPS,GEM3_ROLE_TANK,GEM3_ROLE_HEAL),
  ["HUNTER"] = bit.bor(GEM3_ROLE_TANK,GEM3_ROLE_RDPS),
  ["MAGE"] = bit.bor(GEM3_ROLE_TANK,GEM3_ROLE_RDPS),
  ["PALADIN"] = bit.bor(GEM3_ROLE_HEAL,GEM3_ROLE_TANK,GEM3_ROLE_CDPS),
  ["PRIEST"] = bit.bor(GEM3_ROLE_TANK,GEM3_ROLE_HEAL,GEM3_ROLE_RDPS),
  ["ROGUE"] = bit.bor(GEM3_ROLE_TANK,GEM3_ROLE_CDPS),
  ["SHAMAN"] = bit.bor(GEM3_ROLE_TANK,GEM3_ROLE_RDPS,GEM3_ROLE_CDPS,GEM3_ROLE_HEAL),
  ["WARLOCK"] = bit.bor(GEM3_ROLE_TANK,GEM3_ROLE_RDPS),
  ["WARRIOR"] = bit.bor(GEM3_ROLE_TANK,GEM3_ROLE_CDPS),
  ["DEATHKNIGHT"] = bit.bor(GEM3_ROLE_TANK,GEM3_ROLE_CDPS),
};

--------------- Internal functions ---------------

local function _GEM3_PLAY_GetPlayerInfosInChannel(pl_name,chan_players)
  if(chan_players == nil)
  then
    return nil;
  end
  for name,infos in pairs(chan_players)
  do
    if(name == pl_name)
    then
      return infos;
    end
  end
  return nil;
end

local function _GEM3_PLAY_GetBiggestTree()
  local biggest = 0;
  local count_max = 0;
  
  for i=1,GetNumTalentTabs()
  do
    local name,_,count = GetTalentTabInfo(i);
    if(count > count_max)
    then
      biggest = i;
      count_max = count;
    end
  end
  
  return biggest;
end

function _GEM3_PLAY_HasTalent(tree,texture)
  local numTalents = GetNumTalents(tree);
  for i=1, numTalents
  do
    local name,icon,x,y,rank,max = GetTalentInfo(tree,i);
    if(icon == texture and rank == max)
    then
      return true;
    end
  end
  return false;
end

local function _GEM3_PLAY_GuessFeralDruidRole()
  local role = GEM3_ROLE_CDPS;
  
  if(_GEM3_PLAY_HasTalent(2,"Interface\\Icons\\INV_Misc_Pelt_Bear_03")) -- Thick Hide
  then
    if(_GEM3_PLAY_HasTalent(2,"Interface\\Icons\\Ability_Ambush")) -- Feral Instinct
    then
      if(_GEM3_PLAY_HasTalent(2,"Interface\\Icons\\Ability_Druid_Mangle2")) -- Mangle
      then
        role = GEM3_ROLE_TANK;
      end
    end
  end

  return role;
end

local function _GEM_PLAY_GetTags(tags)
  local name,id,remaining;
  local stamp = GEM3_STAMP_GetLocalStamp();
  local tag;
  local gem_id;

  tags = GA_WipeTable(tags,10); -- Wipe previous tags table, or create new one
  
  for i=1, GetNumSavedInstances()
  do
    name,id,remaining = GetSavedInstanceInfo(i);
    gem_id = GEM3_LOC_INSTANCE_NAMES[name];
    if(gem_id == nil) -- Instance name not found, must be an heroic tag
    then
      -- Print tempo debug message, until we have all instance names
      if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
        if(_GEM_PLAY_Tags == nil) then _GEM_PLAY_Tags = {} end
        if(_GEM_PLAY_Tags[name] == nil)
        then
          _GEM_PLAY_Tags[name] = true; -- To show message once
          GEM3_ChatPrint("Instance name '"..name.."' (locale: "..GetLocale()..") not found in GEM instances list.");
          GEM3_ChatPrint("If this is not an heroic instance, please report this exact message to GEM3 web site: http://www.wowgem.fr");
        end
      end
    else
      tag = GA_GetTable();
      tinsert(tag,id); -- GEM3_ACCESS_INST_TAG_ID
      tinsert(tag,stamp+remaining); -- GEM3_ACCESS_INST_TAG_RESET
      tags[gem_id] = tag;
    end
  end
  return tags;
end


--------------- Exported functions ---------------

function GEM3_PLAY_GuessPlayerRole()
  local biggest_tree = _GEM3_PLAY_GetBiggestTree();
  local _,clas = UnitClass("player");
  local role = GEM3_ROLE_UNKNOWN;
  
  if(biggest_tree ~= 0 and clas)
  then
    if(clas == "DRUID" and biggest_tree == 2) -- Special druid case
    then
      role = _GEM3_PLAY_GuessFeralDruidRole();
    else
      role = _GEM3_PLAY_TalentsToSpec[clas][biggest_tree];
    end
  end
  return role;
end

function GEM3_PLAY_FillMyPlayerInfos(channel)
  local name = GEM3_PlayerName;
  local guild,grank_name,grank_idx = GetGuildInfo("player");
  local officer = 0;
  local location = GetRealZoneText();
  local comment = GEM3_QA_Config.comment;
  local role = GEM3_QA_Config.role;
  local isLeader = false;
  if(location == nil)
  then
    location = GEM3_NA_FORMAT;
  end
  inInstance, instanceType = IsInInstance();
  if(inInstance and instanceType == "party") -- Maybe heroic instance
  then
    difficulty = GetInstanceDifficulty();
    if(difficulty == 2)
    then
      location = location .. " - "..DUNGEON_DIFFICULTY2;
    elseif(difficulty == 3)
    then
      location = location .. " - "..DUNGEON_DIFFICULTY3;
    end
  end
  local level = UnitLevel("player");
  local _,class = UnitClass("player");
  if(IsGuildLeader() == 1)
  then
    officer = 2;
  elseif(GEM3_PLAY_IsOfficer())
  then
    officer = 1;
  end
  
  if(guild == nil) -- No guild, set rank idx at -1
  then
    grank_idx = -1;
  end

  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_ChatDebug(GEM3_DEBUG_GLOBAL,"GEM3_PLAY_FillMyPlayerInfos: Channel "..tostring(channel).." for Player "..GEM3_PlayerName);
  end
  local infos = GEM3_PLAY_GetPlayerInfos(GEM3_PlayerName,channel);
  if(infos == nil)
  then
    infos = GA_GetTable();
    GEM3_QA_Players[channel][GEM3_PlayerName] = infos;
    infos.name = GEM3_PlayerName;
    infos.connected = true;
  end
  infos.guild = guild or "";
  infos.location = location;
  infos.level = level;
  infos.class = class;
  infos.role = role;
  infos.officer = officer;
  infos.grank_name = grank_name or "";
  infos.grank_idx = grank_idx;
  infos.version = GEM3_VERSION;
  infos.comment = comment;
  infos.lastlog = time();
  infos.tags = _GEM_PLAY_GetTags(infos.tags);

  isLeader = GEM3_CHAN_AmILeader(channel);
  infos.isLeader = isLeader;
  infos.isTimeRef = GEM3_QA_Config.time_ref;

  -- GUI Callback
  GEM3_TriggerCallback("OnPlayerInfosUpdate",channel,GEM3_QA_Players[channel][GEM3_PlayerName]);

  return name,guild,location,level,class,role,officer,grank_name,grank_idx,GEM3_VERSION,comment,isLeader,GEM3_QA_Config.time_ref,infos.tags;
end

function GEM3_PLAY_FillPlayerInfos(channel,name,guild,location,level,class,role,officer,grank_name,grank_idx,version,comment,isLeader,isTimeRef,tags)
  if(channel == nil)
  then
    return nil;
  end
  
  if(name == nil) -- Me
  then
    GEM3_ChatWarning("GEM3_PLAY_FillPlayerInfos: name = nil, ME ?");
    GEM3_PLAY_FillMyPlayerInfos(channel);
    return nil;
  end

  local infos = GEM3_PLAY_GetPlayerInfos(name,channel);
  if(infos == nil)
  then
    infos = GA_GetTable();
    GEM3_QA_Players[channel][name] = infos;
    infos.name = name;
    infos.connected = false;
  end
  infos.guild = guild or "";
  infos.location = location;
  infos.level = level;
  infos.class = class;
  infos.role = role;
  infos.officer = officer;
  infos.grank_name = grank_name or "";
  infos.grank_idx = grank_idx;
  infos.version = version;
  infos.comment = comment;
  infos.lastlog = time();
  infos.isLeader = isLeader;
  infos.isTimeRef = isTimeRef;
  infos.tags = tags;
  
  if(isLeader and channel == GEM3_GuildChannelName) -- This player is the GUILD channel leader
  then
    GEM3_CHAN_FoundGuildChannelLeader(name);
  end
  
  GEM3_PLAY_PlayerJoined(channel,name); -- Check for connected
  
  return infos;
end

function GEM3_PLAY_GetLastLeave(channel,pl_name)
  local infos = GEM3_PLAY_GetPlayerInfos(pl_name,channel);
  
  if(infos)
  then
    if(infos.lastleave)
    then
      return infos.lastleave;
    end
  end
  return 0;
end

function GEM3_PLAY_CheckExpiredPlayers(channel)
  if(GEM3_QA_Players[channel] == nil)
  then
    return;
  end

  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_ChatDebug(GEM3_DEBUG_GLOBAL,"GEMPlayers_CheckExpiredPlayers : Checking expired players in channel "..channel);
  end
  local tim = time();
  
  for name,playertab in pairs(GEM3_QA_Players[channel])
  do
    if(playertab.lastlog and (playertab.lastlog + GEM3_PLAY_LASTLOG_MAX) < tim)
    then
      if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
        GEM3_ChatDebug(GEM3_DEBUG_GLOBAL,"GEMPlayers_CheckExpiredPlayers : Too long since last log from player "..name.." in channel "..channel..", Removing !");
      end
      GEM3_QA_Players[channel][name] = nil;
    end
  end
end

function GEM3_PLAY_PlayerJoined(channel,player)
  local infos = GEM3_PLAY_GetPlayerInfos(player,channel);
  
  if(infos and not infos.connected)
  then
    infos.connected = true;
    -- GUI Callback
    GEM3_TriggerCallback("OnPlayerJoinedChannel",channel,player);
    if(player ~= GEM3_PlayerName)
    then
      -- Check for Broadcast
      GEM3_QUE_BuildBroadcastQueues(channel,player);
    end
    return true;
  end
  return false;
end

function GEM3_PLAY_PlayerLeft(channel,player)
  local infos = GEM3_PLAY_GetPlayerInfos(player,channel);
  
  if(infos and infos.connected)
  then
    -- DEBUG
    if(channel == GEM3_GuildChannelName and player == GEM3_PlayerName)
    then
      error("GEM3_PLAY_PlayerLeft: I JUST LEFT guild channel!!!");
    end
    -- END DEBUG
    infos.connected = false; -- Not connected
    infos.lastleave = time(); -- Update lastleave time
    infos.isLeader = false; -- No longer channel leader
    infos.isTimeRef = false; -- No longer a time reference
    -- GUI Callback
    GEM3_TriggerCallback("OnPlayerInfosUpdate",channel,infos);
    -- GUI Callback
    GEM3_TriggerCallback("OnPlayerLeftChannel",channel,player);
    return true;
  end
  return false;
end

function GEM3_PLAY_ResetJoinedPlayers(channel)
  local players = GEM3_QA_Players[channel];
  
  if(players)
  then
    for name,infos in pairs(players)
    do
      infos.connected = false;
      infos.lastleave = nil;
    end
  end
end


--------------- API Exported functions ---------------

function GEM3_PLAY_IsPlayerIgnored(pl_name)
  return GEM3_QA_Events.ignored_players[pl_name] ~= nil;
end

function GEM3_PLAY_IgnorePlayer(pl_name)
  GEM3_QA_Events.ignored_players[pl_name] = true;
  -- Check for events to clear because of ignoring this player
  for ev_id,event in pairs(GEM3_QA_Events.events)
  do
    if(event.leader == pl_name)
    then
      --GEM3_EVT_IgnoreEvent(ev_id); -- PurgeEvent is just enough :)
      GEM3_EVT_PurgeEvent(ev_id,nil); -- Pass nil, so the OnDeletedEvent CB won't be called
    end
  end
end

function GEM3_PLAY_UnIgnorePlayer(pl_name)
  GEM3_QA_Events.ignored_players[pl_name] = nil;
end

function GEM3_PLAY_GetPlayerInfos(pl_name,channel)
  if(channel)
  then
    return _GEM3_PLAY_GetPlayerInfosInChannel(pl_name,GEM3_QA_Players[channel]);
  end

  local infos;
  for _,players in pairs(GEM3_QA_Players)
  do
    infos = _GEM3_PLAY_GetPlayerInfosInChannel(pl_name,players);
    if(infos)
    then
      return infos;
    end
  end
  return nil;
end

function GEM3_PLAY_IsPlayerInChannel(channel,player)
  local infos = GEM3_PLAY_GetPlayerInfos(player,channel);
  
  if(infos)
  then
    return infos.connected;
  end
  return false;
end

function GEM3_PLAY_GetMyPlayerInfos(pl_name)
  local infos;
  local config = GEM3_Config[GEM3_Realm][pl_name];
  
  if(config)
  then
    for _,chaninfos in ipairs(config.channels)
    do
      infos = _GEM3_PLAY_GetPlayerInfosInChannel(pl_name,GEM3_QA_Players[chaninfos.name]);
      if(infos)
      then
        return infos;
      end
    end
  end
  return nil;
end

function GEM3_PLAY_GetMyRerolls(channel)
  local list = GA_GetTable();

  for name,config in pairs(GEM3_Config[GEM3_Realm])
  do
    if(channel == nil or GEM3_CHAN_IsChannelInList(channel,config.channels))
    then
      local infos = GEM3_PLAY_GetPlayerInfos(name,channel);
      if(infos)
      then
        tinsert(list,GEM3_PLAY_GetMyPlayerInfos(name));
      end
    end
  end
  
  return list;
end

function GEM3_PLAY_GetAllowedRoles(class)
  return _GEM3_PLAY_AllowedRoles[class];
end

function GEM3_PLAY_IsOfficer()
  return CanEditOfficerNote() == 1;
end
