--[[
  Guild Event Manager (GEM) - Version 3
  by Kiki from European "Conseil des Ombres" (Horde)
  ----------------------------------------
  Events part
]]

--[[
  Exported API Functions:

   GEM3_EVT_CreateNewEvent(channel,ev_leader,ev_date,deadline,ev_place,ev_comment,max_count,min_lvl,max_lvl,classes,roles,sortname,ev_type,ev_subtype,ev_duration)
    Input:
     channel:  string -- Channel tied to the event
     ev_leader:   string -- Leader of the event
     ev_date: timestamp -- UTC stamp for the event's date
     deadline: timestamp -- UTC stamp for the deadline signup
     ev_place: string -- Location of the event
     ev_comment: string -- Comment for the event
     max_count: number -- Max titulars count
     min_lvl: number -- Min level required
     max_lvl: number -- Max level required
     classes: array of GEM3_Limit -- Classes limits - indexed by CLASS_NAME (english uppercase names)
     roles: array of GEM3_Limit -- Roles limits - indexed by GEM3_ROLE_xx
     sortname: string -- Sort plugin name to use
     ev_type: number -- Event type (one of GEM3_EVT_TYPE_xx)
     ev_subtype: number -- Event sub-type (any "number")
     ev_duration: number -- Event duration (in seconds)
    Output:
     ev_id: string -- ev_id for the created event, or nil on error
     reason: string -- Failure message, Nil if ev_id is not nil
    Events:
     "OnEventCreated"(event) is called on successful event creation
    Purpose:
     Creates a new event

   GEM3_EVT_ModifyEvent(ev_id,ev_date,deadline,ev_place,ev_comment,max_count,min_lvl,max_lvl,classes,roles,sortname,ev_type,ev_subtype,ev_duration)
    Input:
     ev_id:  string -- Event id
     ev_date: timestamp -- UTC stamp for the event's date
     deadline: timestamp -- UTC stamp for the deadline signup
     ev_place: string -- Location of the event
     ev_comment: string -- Comment for the event
     max_count: number -- Max titulars count
     min_lvl: number -- Min level required
     max_lvl: number -- Max level required
     classes: array of GEM3_Limit -- Classes limits - indexed by CLASS_NAME (english uppercase names)
     roles: array of GEM3_Limit -- Roles limits - indexed by GEM3_ROLE_xx
     sortname: string -- Sort plugin name to use
     ev_type: number -- Event type (one of GEM3_EVT_TYPE_xx)
     ev_subtype: number -- Event sub-type (any "number")
     ev_duration: number -- Event duration (in seconds)
    Output:
     Result      : Boolean -- True if success, false otherwise
     Reason      : String  -- Failure message, Nil if Result is true
    Events:
     "OnUpdatedEvent"(event,major_update) is called on successful event edition
    Purpose:
     Edits an existing event (leader only) and broadcast it

   GEM3_EVT_CloseEvent(ev_id,reason)
    Input:
     ev_id:  string -- Event id
     reason:  string -- Reason
    Events:
     "OnClosedEvent"(event) is called if event successfully closed
    Purpose:
     Closes an opened event - Leader only

   GEM3_EVT_UnCloseEvent(ev_id)
    Input:
     ev_id:  string -- Event id
    Events:
     "OnUnClosedEvent"(event) is called if event successfully reopened
    Purpose:
     Reopens a closed event - Leader only
  
   GEM3_EVT_DeleteEvent(ev_id)
    Input:
     ev_id:  string -- Event id
    Events:
     "OnDeletedEvent"(event) is called if event successfully deleted
    Purpose:
     Fully deletes an event, with no possible reopening - Leader only

   GEM3_EVT_IgnoreEvent(ev_id)
    Input:
     ev_id:  string -- Event id
    Purpose:
     Ignores an event and all associated commmands

   GEM3_EVT_UnIgnoreEvent(ev_id)
    Input:
     ev_id:  string -- Event id
    Purpose:
     Unignores an event - Might takes some time before you fully recover it

   GEM3_EVT_IsEventIgnored(ev_id)
    Input:
     ev_id:  string -- Event id
    Output:
     ignored: boolean -- True if event is ignored, false otherwise
    Purpose:
     Checks if an event is currently ignored

   GEM3_EVT_SetAssistant(ev_id,pl_name)
    Input:
     ev_id:  string -- Event id
     pl_name: string -- Player to set as event assistant
    Purpose:
     Sets a player as event assistant. He'll have some rights over the event - Leader only

   GEM3_EVT_IsAssistant(ev_id,pl_name)
    Input:
     ev_id:  string -- Event id
     pl_name: string -- Player to check
    Output:
     isAssistant: boolean -- True if player is an event assistant, false otherwise
    Purpose:
     Checks if player is an event assistant

   GEM3_EVT_AmIAssistant(ev_id)
    Input:
     ev_id:  string -- Event id
    Output:
     isAssistant: boolean -- True if any of my reroll is an event assistant, false otherwise
    Purpose:
     Checks if any of my rerolls is an event assistant

   GEM3_EVT_ArchiveEvent(ev_id|event)
    Input:
     ev_id|event:  string|GEM3_Event -- Event id (or event struct) to archive
    Output:
     archived: boolean -- True if the event has successfully been archived
    Purpose:
     Archives an event (fully duplicate the event)

   GEM3_EVT_DeleteArchivedEvent(ev_id)
    Input:
     ev_id:  string -- Event id to remove from archive
    Output:
     deleted: boolean -- True if the event has successfully been deleted from archive
    Purpose:
     Deletes an archived event

   GEM3_EVT_KickFromEvent(ev_id,pl_name,reason)
    Input:
     ev_id:  string -- Event id
     pl_name: string -- Player to kick from event
     reason: string -- Reason to kick the player from the event
    Output:
     Result      : Boolean -- True if success, false otherwise
     Reason      : String  -- Failure message, Nil if Result is true
    Purpose:
     Kicks a subscriber from an event. Leader or assistant only

   GEM3_EVT_BanFromEvent(ev_id,pl_name,reason)
    Input:
     ev_id:  string -- Event id
     pl_name: string -- Player to ban from event
     reason: string -- Reason to ban the player from the event
    Output:
     Result      : Boolean -- True if success, false otherwise
     Reason      : String  -- Failure message, Nil if Result is true
    Purpose:
     Bans a player from an event. Leader or assistant only

   GEM3_EVT_UnBanFromEvent(ev_id,pl_name)
    Input:
     ev_id:  string -- Event id
     pl_name: string -- Player to unban from event
    Output:
     Result      : Boolean -- True if success, false otherwise
     Reason      : String  -- Failure message, Nil if Result is true
    Purpose:
     Un bans a player from an event. Leader or assistant only

   GEM3_EVT_GetBannedPlayers(ev_id)
    Input:
     ev_id:  string -- Event id
    Output:
     bans: Table of GEM3_Banned struct -- Returned table must be freed using GA_ReleaseTable(bans,2) when no longer needed -- Indexed by INT (use ipairs() to process)
    Purpose:
     Retrieves the list of banned players from an event


]]


--------------- Shared variables ---------------


--------------- Local variables ---------------

local _GEM3_EVT_RecoveringEvents = {};
local _GEM3_EVT_RecoveringEventsIgnored = {};
local _GEM3_EVT_UpdatingRecoveredEvents = {};
local _GEM3_EVT_UpdatingRecoveredEventsIgnored = {};


--------------- Internal functions ---------------

local function _GEM3_EVT_InitPlayersSpecific(list)
  for n,tab in pairs(list)
  do
    if(tab.lead_force_queue == nil)
    then
      tab.lead_force_queue = GEM3_SUB_FORCE_QUEUE_NONE;
    end
  end
end

local function _GEM3_EVT_SortSubscribersFunc(a,b)
  return a.queue_pos < b.queue_pos;
end


--------------- Exported functions ---------------

--[[
 function GEM3_EVT_TouchEvent:
  Sets an event as updated and broadcast it. Leader only!
   event    : Array -- The event
   immediat : Bool  -- Immediatly broadcast the event, bypassing send queue and expired checks
]]
function GEM3_EVT_TouchEvent(event,immediat)
  event.update_time = GEM3_STAMP_GetLocalStamp(); -- Touch the event
  if(immediat)
  then
    GEM3_COM_BroadcastEvent(event); -- Force broadcast now
  else
    GEM3_COM_NotifyEventUpdate(event.id); -- Schedule a broadcast in few secs
  end
end

local function _GEM3_EVT_SetEventDeleted(ev_id,deleted_time)
  local old_time = GEM3_QA_Events.others_deleted_events[ev_id];

  if(old_time == nil or old_time < deleted_time)
  then
    GEM3_QA_Events.others_deleted_events[ev_id] = deleted_time;
    if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
      GEM3_ChatDebug(GEM3_DEBUG_EVENTS,"_GEM3_EVT_SetEventDeleted: Setting EventId "..ev_id.." as deleted by the leader at "..date("%c",deleted_time).." ("..deleted_time..")");
    end
  end
end

local function _GEM3_EVT_IsEventDeleted(ev_id,update_time)
  local deleted_time = GEM3_QA_Events.others_deleted_events[ev_id];

  if(deleted_time == nil)
  then
    return false;
  end
  
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_ChatDebug(GEM3_DEBUG_EVENTS,"_GEM3_EVT_IsEventDeleted: Got a deleted_time for EventId "..ev_id..": Deleted -> "..tostring(deleted_time >= update_time));
  end
  return deleted_time >= update_time;
end

local function _GEM3_EVT_GetEventDeletedTime(ev_id)
  return GEM3_QA_Events.others_deleted_events[ev_id];
end

local function _GEM3_EVT_CheckClasses(classes)
  for n in pairs(GEM3_ClassToInt)
  do
    if(classes[n] == nil)
    then
      classes[n] = GA_GetTable();
    end
  end

  return classes;
end

local function _GEM3_EVT_CheckRoles(roles)
  if(roles[GEM3_ROLE_TANK] == nil) then
    roles[GEM3_ROLE_TANK] = GA_GetTable();
  end
  if(roles[GEM3_ROLE_CDPS] == nil) then -- Close DPS
    roles[GEM3_ROLE_CDPS] = GA_GetTable();
  end
  if(roles[GEM3_ROLE_RDPS] == nil) then -- Range DPS
    roles[GEM3_ROLE_RDPS] = GA_GetTable();
  end
  if(roles[GEM3_ROLE_DPS] == nil) then -- Any DPS
    roles[GEM3_ROLE_DPS] = GA_GetTable();
  end
  if(roles[GEM3_ROLE_HEAL] == nil) then
    roles[GEM3_ROLE_HEAL] = GA_GetTable();
  end

  return roles;
end

local function _GEM3_EVT_FillEvent(event,channel,ev_id,creat_time,leader,ev_date,ev_place,ev_comment,max_count,min_lvl,max_lvl,classes,roles,players,sorttype,closed_comment,deadline,recover_time,ev_type,ev_subtype,revision,ev_duration)
  event.channel = channel;
  event.id = ev_id;
  event.update_time = creat_time;
  event.recover_time = recover_time;
  event.leader = leader;
  event.ev_date = ev_date;
  event.ev_place = ev_place;
  event.ev_comment = ev_comment;
  event.max_count = max_count;
  event.min_lvl = min_lvl;
  event.max_lvl = max_lvl;
  event.ev_type = ev_type;
  event.ev_subtype = ev_subtype;
  event.revision = revision;
  event.ev_duration = ev_duration;
  if(classes) -- Classes provided? Remove previous table and update with new one
  then
    GA_ReleaseTable(event.classes,10);
    event.classes = _GEM3_EVT_CheckClasses(classes);
  end
  if(roles) -- Roles provided? Remove previous table and update with new one
  then
    GA_ReleaseTable(event.roles,10);
    event.roles = _GEM3_EVT_CheckRoles(roles);
  end
  if(players) -- Players provided? Remove previous table and update with new one
  then
    GA_ReleaseTable(event.players,10);
    event.players = players;
  end
  event.sorttype = sorttype;
  event.closed_comment = closed_comment;
  event.deadline = deadline;
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_ChatDebug(GEM3_DEBUG_EVENTS,"_GEM3_EVT_FillEvent: EventId "..ev_id.." Leader="..leader.." Date="..date("%c",ev_date).." ("..ev_date..") Deadline="..date("%c",deadline).." Place="..ev_place.." Duration="..ev_duration.." Revision="..revision);
  end
end

local function _GEM3_EVT_AllocEventTiedVariables(event)
  GEM3_QA_Events.kicked[event.id] = GA_GetTable();
  GEM3_QA_Events.banned[event.id] = GA_GetTable();
  GEM3_QA_Events.subscribed[event.id] = GA_GetTable();
  GEM3_QA_Events.assistants[event.id] = GA_GetTable();
  -- GEM3_QA_Events.commands[event.id] is allocated by GEM3_CMD_GetOrCreateCommandInfosForEvent
  -- GEM3_QA_Events.forward[event.id] is the name of the forwarder, does not need to be allocated
  -- GEM3_QA_Events.ignored[event.id] is a boolean, does not need to be allocated
  -- GEM3_QA_Events.archived[event.id] is a GEM3_Event, does not need to be allocated
  -- GEM3_QA_Events.others_deleted_events[event.id] is a number, does not need to be allocated
end

function GEM3_EVT_AddEvent(channel,ev_id,creat_time,leader,ev_date,ev_place,ev_comment,max_count,min_lvl,max_lvl,classes,roles,players,sorttype,closed_comment,deadline,recover_time,ev_type,ev_subtype,revision,ev_duration)
  local event = GA_GetTable();

  _GEM3_EVT_FillEvent(event,channel,ev_id,creat_time,leader,ev_date,ev_place,ev_comment,max_count,min_lvl,max_lvl,classes,roles,players,sorttype,closed_comment,deadline,recover_time,ev_type,ev_subtype,revision,ev_duration);
  GEM3_QA_Events.events[ev_id] = event;
  _GEM3_EVT_AllocEventTiedVariables(event);
  
  if(GEM3_IsMyReroll(leader)) -- Leader? -> Event was lost
  then
    _GEM3_EVT_InitPlayersSpecific(event.players);
    GEM3_SUB_CheckPlayersLevel(event);
    GEM3_SORT_SortPlayers(event);
    GEM3_EVT_TouchEvent(event,true); -- Update event time and broadcast it
  end
  GEM3_SUB_UpdateCounts(event);

  GEM3_SUB_CheckMySubscription(event);

  -- GUI Callback
  GEM3_TriggerCallback("OnNewEvent",event);
end

function GEM3_EVT_NeedsRebuildCrashedEvent(ev_id,update_time)
  local event = _GEM3_EVT_RecoveringEvents[ev_id];
  if(event and event.update_time >= update_time)
  then
    GEM3_ChatDebug(GEM3_DEBUG_EVENTS,"GEM3_EVT_NeedsRebuildCrashedEvent: Already processing a better version of this EventId "..ev_id.." -> "..event.update_time.." - "..update_time);
    return false;
  end
  if(_GEM3_EVT_RecoveringEventsIgnored[ev_id] and _GEM3_EVT_RecoveringEventsIgnored[ev_id] <= update_time)
  then
    GEM3_ChatDebug(GEM3_DEBUG_EVENTS,"GEM3_EVT_NeedsRebuildCrashedEvent: You choose to reject all events <= to ".._GEM3_EVT_RecoveringEventsIgnored[ev_id]..". This one is "..update_time);
    return false;
  end
  return true;
end

function GEM3_EVT_RebuildCrashedEvent(channel,ev_id,update_time,leader,ev_date,ev_place,ev_comment,max_count,min_lvl,max_lvl,classes,roles,players,sorttype,closed_comment,deadline,recover_time,ev_type,ev_subtype,revision,ev_duration)
  local event = GA_GetTable();

  GEM3_ChatDebug(GEM3_DEBUG_EVENTS,"GEM3_EVT_CheckCrashedEvent: Rebuilding lost event of EventId "..ev_id.." Leader="..leader.." Date="..date("%c",ev_date).." ("..ev_date..") Place="..ev_place);
  _GEM3_EVT_FillEvent(event,channel,ev_id,update_time,leader,ev_date,ev_place,ev_comment,max_count,min_lvl,max_lvl,classes,roles,players,sorttype,closed_comment,deadline,recover_time,ev_type,ev_subtype,revision,ev_duration);
  _GEM3_EVT_InitPlayersSpecific(event.players);
  GEM3_SUB_CheckPlayersLevel(event);
  GEM3_SORT_SortPlayers(event);
  GEM3_SUB_UpdateCounts(event);
  
  local previous = _GEM3_EVT_RecoveringEvents[event.id]; -- One already in recover
  _GEM3_EVT_RecoveringEvents[event.id] = event; -- Store it

  --GEM3_ChatPrint("WARNING: I'm supposed to be the leader of '"..ev_id.."' but I don't know it.");
  --GEM3_ChatPrint("This happen if you crashed the game after creating the event, or used another computer with the same account.");
  -- GUI Callback
  if(not GEM3_TriggerCallback("OnCrashedEvent",event))
  then
    GEM3_EVT_RejectLostEvent(event);
  end
  if(previous)
  then
    GA_ReleaseTable(previous,10); -- 10 recursion count to ensure full table release
  end
end

function GEM3_EVT_NeedsRebuildRecoverUpdateEvent(ev_id,update_time)
  local event = _GEM3_EVT_UpdatingRecoveredEvents[ev_id];
  if(event and event.update_time >= update_time)
  then
    GEM3_ChatDebug(GEM3_DEBUG_EVENTS,"GEM3_EVT_NeedsRebuildRecoverUpdateEvent: Already processing a better version of this EventId "..ev_id.." -> "..event.update_time.." - "..update_time);
    return false;
  end
  if(_GEM3_EVT_UpdatingRecoveredEventsIgnored[ev_id] and _GEM3_EVT_UpdatingRecoveredEventsIgnored[ev_id] <= update_time)
  then
    GEM3_ChatDebug(GEM3_DEBUG_EVENTS,"GEM3_EVT_NeedsRebuildRecoverUpdateEvent: You choose to reject all events <= to ".._GEM3_EVT_UpdatingRecoveredEventsIgnored[ev_id]..". This one is "..update_time);
    return false;
  end
  return true;
end

function GEM3_EVT_RebuildRecoverUpdateEvent(channel,ev_id,update_time,leader,ev_date,ev_place,ev_comment,max_count,min_lvl,max_lvl,classes,roles,players,sorttype,closed_comment,deadline,recover_time,ev_type,ev_subtype,revision,ev_duration)
  local event = GA_GetTable();

  GEM3_ChatDebug(GEM3_DEBUG_EVENTS,"GEM3_EVT_RebuildRecoverUpdateEvent: Rebuilding updated event of recovered EventId "..ev_id.." Leader="..leader.." Date="..date("%c",ev_date).." ("..ev_date..") Place="..ev_place);
  _GEM3_EVT_FillEvent(event,channel,ev_id,update_time,leader,ev_date,ev_place,ev_comment,max_count,min_lvl,max_lvl,classes,roles,players,sorttype,closed_comment,deadline,recover_time,ev_type,ev_subtype,revision,ev_duration);
  _GEM3_EVT_InitPlayersSpecific(event.players);
  GEM3_SUB_CheckPlayersLevel(event);
  GEM3_SORT_SortPlayers(event);
  GEM3_SUB_UpdateCounts(event);

  local previous = _GEM3_EVT_UpdatingRecoveredEvents[event.id]; -- One already in recover
  _GEM3_EVT_UpdatingRecoveredEvents[event.id] = event; -- Store it

  --GEM3_ChatPrint("WARNING: I recovered event '"..ev_id.."' but it seems there were a more recent version of it.");
  -- GUI Callback
  if(not GEM3_TriggerCallback("OnUpdatedRecoveredEvent",event,GEM3_QA_Events.events[ev_id]))
  then
    GEM3_EVT_RejectUpdatedRecoveredEvent(event);
  end
  if(previous)
  then
    GA_ReleaseTable(previous,10); -- 10 recursion count to ensure full table release
  end
end

-- For non-leader = Updated Event  ---  For leader = Updated version of my event, maybe I crashed after an update, or changed computer
function GEM3_EVT_UpdateEvent(channel,ev_id,update_time,leader,ev_date,ev_place,ev_comment,max_count,min_lvl,max_lvl,classes,roles,players,sorttype,closed_comment,deadline,recover_time,ev_type,ev_subtype,revision,ev_duration)
  local event = GEM3_QA_Events.events[ev_id];
  if(event == nil)
  then
    GEM3_ChatWarning("GEM3_EVT_UpdateEvent: EventID '"..ev_id.."' is unknown!!");
    return;
  end
  local isModified = (event.revision ~= revision);
  local isDateChanged = (event.ev_date ~= ev_date);
  _GEM3_EVT_FillEvent(event,channel,ev_id,update_time,leader,ev_date,ev_place,ev_comment,max_count,min_lvl,max_lvl,classes,roles,players,sorttype,closed_comment,deadline,recover_time,ev_type,ev_subtype,revision,ev_duration);
  
  if(GEM3_IsMyReroll(leader)) -- Leader? -> Event was lost
  then
    _GEM3_EVT_InitPlayersSpecific(event.players);
    GEM3_SUB_CheckPlayersLevel(event);
    GEM3_SORT_SortPlayers(event);
    GEM3_EVT_TouchEvent(event,true); -- Update event time and broadcast it
  end
  GEM3_SUB_UpdateCounts(event);
  
  GEM3_SUB_CheckMySubscription(event); -- Check if my subscription changed in this new event

  if(GEM3_IsMyReroll(leader)) -- Leader?
  then
    -- GUI Callback
    GEM3_TriggerCallback("OnUpdatedEventLost",event);
  else
    -- GUI Callback
    GEM3_TriggerCallback("OnUpdatedEvent",event,isModified,isDateChanged);
  end
end

--[[
 function GEM3_EVT_ActionForIncomingEvent
  Returns action to do for this event.
   GEM3_EVT_ACTION_DISCARD: Discard event
   GEM3_EVT_ACTION_NEW: New event
   GEM3_EVT_ACTION_UPDATED: Updated event
   GEM3_EVT_ACTION_CRASHED: Event was lost
   GEM3_EVT_ACTION_CRASHED_UPDATE: Event was lost and recovered, and this is an updated version
]]
function GEM3_EVT_ActionForIncomingEvent(channel,from,ev_id,leader,update_time,ev_date,ev_place,recover_time)
  GEM3_ChatDebug(GEM3_DEBUG_EVENTS,"GEM3_EVT_ActionForIncomingEvent: Received EventUpdate on channel '"..channel.."' from '"..from.."' for EventID "..ev_id.." of Leader '"..leader.."' UpdateTime="..date("%c",update_time).." RecoverTime="..date("%c",recover_time));

  local event = GEM3_QA_Events.events[ev_id];
  local tim = GEM3_STAMP_GetLocalStamp();

  if(GEM3_IsMyReroll(leader)) -- If I'm the leader of this event
  then
    if(event) -- I know this event
    then
      if(update_time == event.update_time) -- Same event I have: Don't send mine if it was in queue and discard this one
      then
        if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
          GEM3_ChatDebug(GEM3_DEBUG_EVENTS,"GEM3_EVT_ActionForIncomingEvent: Same I have, discard");
        end
        GEM3_QUE_RemoveEventFromQueue(channel,ev_id);
        return GEM3_EVT_ACTION_DISCARD;
      elseif(recover_time < event.recover_time and update_time > event.recover_time) -- Check for an updated version of a recovered event (if I didn't recover the latest version)
      then -- "recover_time < event.recover_time" means this event is from a time before I recovered it
           -- "update_time > event.recover_time" means this event is a newest version than the one I recovered
        if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
          GEM3_ChatDebug(GEM3_DEBUG_WARNING,"GEM3_EVT_ActionForIncomingEvent: I previously recovered '"..ev_id.."', but this version was more recent.");
        end
        GEM3_QUE_RemoveEventFromQueue(channel,ev_id);
        return GEM3_EVT_ACTION_RECOVER_UPDATE;
      elseif(update_time < event.update_time) -- This is an old version of my event: Broadcast mine and discard this one
      then
        if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
          GEM3_ChatDebug(GEM3_DEBUG_EVENTS,"GEM3_EVT_ActionForIncomingEvent: Older version, bcast mine and discard this one");
        end
        GEM3_COM_BroadcastEvent(event);
        return GEM3_EVT_ACTION_DISCARD;
      else -- This is a new version of my event, maybe I crashed or changed computer: Don't send mine if it was in queue and accept this one
        GEM3_QUE_RemoveEventFromQueue(channel,ev_id);
        return GEM3_EVT_ACTION_UPDATED;
      end
    else -- I don't know this event
      event = GEM3_EVT_GetDeletedEvent(ev_id);
      if(event == nil) -- Never heard of this event. Either I crashed or switched computer -> Ask user what to do
      then
        if((ev_date + GEM3_QA_Config.expiration_time) > tim) -- Not expired yet: Set it as crashed and accept the event
        then
          return GEM3_EVT_ACTION_CRASHED;
        else -- Expired event: Discard the event
          if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
            GEM3_ChatDebug(GEM3_DEBUG_WARNING,"GEM3_EVT_ActionForIncomingEvent: I'm supposed to be the leader of '"..ev_id.."', it's expired, ignoring.");
          end
          return GEM3_EVT_ACTION_DISCARD;
        end
      else -- I deleted this event: Broadcast deleted event and discard this one
        if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
          GEM3_ChatDebug(GEM3_DEBUG_EVENTS,"GEM3_EVT_ActionForIncomingEvent: Event "..ev_id.." was deleted ! Resending deleted event notification.");
        end
        GEM3_COM_BroadcastEvent(event); -- Immediatly broadcast my saved copy of deleted event
        return GEM3_EVT_ACTION_DISCARD;
      end
    end

  else -- I'm not the leader
    if(event == nil) -- Unknown, check expiration
    then
      if((ev_date + GEM3_QA_Config.expiration_time) < tim) -- Expired: Discard the event
      then
        if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
          GEM3_ChatDebug(GEM3_DEBUG_EVENTS,"GEM3_EVT_ActionForIncomingEvent: Unknown EventId "..ev_id.." expired! ev_date="..date("%c",ev_date).." ("..ev_date..")");
        end
        return GEM3_EVT_ACTION_DISCARD;
      end
      if(_GEM3_EVT_IsEventDeleted(ev_id,update_time)) -- Event was deleted!
      then
        GEM3_COM_DeletedEvent(channel,ev_id,_GEM3_EVT_GetEventDeletedTime(ev_id));
        return GEM3_EVT_ACTION_DISCARD;
      end
      return GEM3_EVT_ACTION_NEW; -- New event
    else -- Known event
      if(update_time == event.update_time) -- Same event I have: Don't send mine if it was in queue and discard this one
      then
        if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
          GEM3_ChatDebug(GEM3_DEBUG_EVENTS,"GEM3_EVT_ActionForIncomingEvent: Same I have, discard");
        end
        GEM3_QUE_RemoveEventFromQueue(channel,ev_id);
        return GEM3_EVT_ACTION_DISCARD;
      elseif(recover_time > event.recover_time) -- Check for an incoming recovered version
      then
        if(recover_time < event.update_time) -- I have a more recent version than the recovered one!!
        then
          if(ev_date == 0) -- Deleted by leader?
          then
            if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
              GEM3_ChatDebug(GEM3_DEBUG_WARNING,"GEM3_EVT_ActionForIncomingEvent: I have a more recent version of '"..ev_id.."' than the recovered one but leader deleted the event!");
            end
          else
            if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
              GEM3_ChatDebug(GEM3_DEBUG_WARNING,"GEM3_EVT_ActionForIncomingEvent: I have a more recent version of '"..ev_id.."' than the recovered one, keep it!");
            end
            if(GEM3_PLAY_IsPlayerInChannel(event.channel,event.leader)) -- Broadcast event if leader is in channel (other players will simply ignore my event because it's update_time is < to this one)
            then
              if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
                GEM3_ChatDebug(GEM3_DEBUG_WARNING,"GEM3_EVT_ActionForIncomingEvent: Event's leader '"..event.leader.."' is online, schedule a broadcast of my version of the event");
              end
              GEM3_QUE_AddEventToQueue(channel,ev_id,0);
            end
            return GEM3_EVT_ACTION_DISCARD;
          end
        end
        if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
          GEM3_ChatDebug(GEM3_DEBUG_WARNING,"GEM3_EVT_ActionForIncomingEvent: Recovered version of '"..ev_id.."' is better than mine, accept update");
        end
        -- Accept update, if not expired
      elseif(update_time <= event.update_time) -- Older than mine: Broadcast mine and discard this one
      then
        if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
          GEM3_ChatDebug(GEM3_DEBUG_EVENTS,"GEM3_EVT_ActionForIncomingEvent: Older version, bcast mine and discard this one");
        end
        GEM3_QUE_AddEventToQueue(channel,ev_id,0);
        return GEM3_EVT_ACTION_DISCARD;
      end
      -- Remove from queue and check expiration
      GEM3_QUE_RemoveEventFromQueue(channel,ev_id);
      -- Check expiration
      if((ev_date + GEM3_QA_Config.expiration_time) < tim) -- Expired: Purge event and discard this event
      then
        if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
          GEM3_ChatDebug(GEM3_DEBUG_EVENTS,"GEM3_EVT_ActionForIncomingEvent: EventId "..ev_id.." expired! ev_date="..date("%c",ev_date).." ("..ev_date..")");
        end
        if(ev_date == 0) -- Deleted
        then
          _GEM3_EVT_SetEventDeleted(ev_id,event.update_time);
          GEM3_EVT_ClearEvent(ev_id,GEM3_LOC_EVENT_CLOSED_DELETED,true);
        else
          GEM3_EVT_ClearEvent(ev_id,GEM3_LOC_EVENT_CLOSED_EXPIRED,true);
        end
        return GEM3_EVT_ACTION_DISCARD;
      end
      return GEM3_EVT_ACTION_UPDATED; -- Updated
    end
  end

  GEM3_ChatDebug(GEM3_DEBUG_EVENTS,"GEM3_EVT_ActionForIncomingEvent: End of function, discard");
  return GEM3_EVT_ACTION_DISCARD;
end

function GEM3_EVT_PurgeEvent(ev_id,event)
  -- Remove the event
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_ChatDebug(GEM3_DEBUG_EVENTS,"GEM3_EVT_PurgeEvent: Purging Event "..ev_id);
  end
  local event_to_delete = GEM3_QA_Events.events[ev_id];
  GEM3_QA_Events.events[ev_id] = nil; -- Un assign the table, so it won't appear in events list during OnDeleteEvent callback
  -- Clear all related infos
  GEM3_QA_Events.commands[ev_id] = GA_ReleaseTable(GEM3_QA_Events.commands[ev_id],10); -- Remove commands
  GEM3_QA_Events.subscribed[ev_id] = GA_ReleaseTable(GEM3_QA_Events.subscribed[ev_id],10); -- Remove my subscriptions
  GEM3_QA_Events.kicked[ev_id] = GA_ReleaseTable(GEM3_QA_Events.kicked[ev_id],10); -- Remove kicks
  GEM3_QA_Events.banned[ev_id] = GA_ReleaseTable(GEM3_QA_Events.banned[ev_id],10); -- Remove bans
  GEM3_QA_Events.assistants[ev_id] = GA_ReleaseTable(GEM3_QA_Events.assistants[ev_id],10); -- Remove assistants
  -- DON'T REMOVE FROM IGNORE LIST !!!

  if(event)
  then
    -- GUI Callback
    GEM3_TriggerCallback("OnDeletedEvent",event);
  end
  GA_ReleaseTable(event_to_delete,10); -- Don't use 'event' because it might be the backup table if leader called this function
end

--[[
 function GEM3_EVT_ClearEvent
  Remove all infos about this event.
   ev_id   : String  -- EventID to be removed
   comment : String  -- Comment for close event
   purge   : Boolean -- If event must be completly purged
]]
function GEM3_EVT_ClearEvent(ev_id,comment,purge)
  local event = GEM3_QA_Events.events[ev_id];

  -- Set close comment
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_ChatDebug(GEM3_DEBUG_EVENTS,"GEM3_EVT_ClearEvent: Clearing Event "..ev_id..": "..comment);
  end
  if(event)
  then
    event.closed_comment = comment;
    -- GUI Callback
    GEM3_TriggerCallback("OnClosedEvent",event,comment);
  end

  if(purge)
  then
    GEM3_EVT_PurgeEvent(ev_id,event);
  end
  -- Clear queues
  GEM3_QUE_RemoveCommands(ev_id);
end

--[[
 function GEM3_EVT_CheckExpiredEvent
  Checks if the Event has expired yet (well 10h after)
   ev_id    : String  -- EventID
  --
   Returns True if the event has expired. False otherwise
]]
function GEM3_EVT_CheckExpiredEvent(ev_id)
  local event = GEM3_QA_Events.events[ev_id];

  if(event)
  then
    local tim = GEM3_STAMP_GetLocalStamp();
    if(event.ev_date == nil)
    then
      return true;
    end
    local expire_time = GEM3_QA_Config.expiration_time;
    if(GEM3_IsMyReroll(event.leader))
    then
      expire_time = GEM3_QA_Config.expiration_time_self;
    end
    if((event.ev_date + expire_time) < tim)
    then
      if(event.ev_date == 0)
      then
        _GEM3_EVT_SetEventDeleted(ev_id,event.update_time);
        GEM3_EVT_ClearEvent(ev_id,GEM3_LOC_EVENT_CLOSED_DELETED,true);
      else
        GEM3_EVT_ClearEvent(ev_id,GEM3_LOC_EVENT_CLOSED_EXPIRED,true);
      end
    end
  end
  return false;
end

--[[
 function GEM3_EVT_CheckExpiredEvents
  Checks all events, for expired ones
]]
function GEM3_EVT_CheckExpiredEvents()
  for ev_id,event in pairs(GEM3_QA_Events.events)
  do
    GEM3_EVT_CheckExpiredEvent(ev_id);
  end
end

function GEM3_EVT_GetDeletedEvent(ev_id)
  return GEM3_QA_Events.my_deleted_events[ev_id];
end

function GEM3_EVT_SetEventAssistant(ev_id,pl_name) -- Internal version of GEM3_EVT_SetAssistant
  if(GEM3_QA_Events.assistants[ev_id] ~= nil)
  then
    GEM3_QA_Events.assistants[ev_id][pl_name] = true;
    GEM3_ChatDebug(GEM3_DEBUG_EVENTS,"GEM3_EVT_SetEventAssistant: I'm ("..pl_name..") now an assistant of EventID "..ev_id);
    -- GUI Callback
    GEM3_TriggerCallback("OnEventAssistant",ev_id,pl_name);
  end
end

function GEM3_EVT_SetMyselfNewLeader(event,new_leader)
  event.leader = new_leader;
  _GEM3_EVT_InitPlayersSpecific(event.players);
  GEM3_SUB_CheckPlayersLevel(event);
  GEM3_SORT_SortPlayers(event);
  GEM3_SUB_UpdateCounts(event);
  GEM3_EVT_TouchEvent(event,true); -- Broadcast it
  -- GUI Callback
  GEM3_TriggerCallback("OnNewLeader",event,new_leader);
end

function GEM3_EVT_NotifyDeletedEvent(ev_id,deleted_time)
  local event = GEM3_QA_Events.events[ev_id];
  
  if(event ~= nil) -- I still know this event
  then
    if(not GEM3_IsMyReroll(event.leader)) -- I'm not the leader
    then
      if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
        GEM3_ChatDebug(GEM3_DEBUG_EVENTS,"GEM3_EVT_NotifyDeletedEvent: I know EventID "..event.id.." but it was deleted at "..date("%c",deleted_time).." ("..deleted_time..")");
      end
      _GEM3_EVT_SetEventDeleted(ev_id,deleted_time);
      GEM3_EVT_ClearEvent(ev_id,GEM3_LOC_EVENT_CLOSED_DELETED,true);
    end
  else
    if(GEM3_QA_Events.my_deleted_events[ev_id] == nil) -- I'm not the leader
    then
      _GEM3_EVT_SetEventDeleted(ev_id,deleted_time); -- Set or update deleted time
    end
  end
end

--------------- API Exported functions ---------------

function GEM3_EVT_CreateNewEvent(channel,ev_leader,ev_date,deadline,ev_place,ev_comment,max_count,min_lvl,max_lvl,classes,roles,sortname,ev_type,ev_subtype,ev_duration)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_DBG_CheckTypes("GEM3_EVT_CreateNewEvent",channel,"channel","string",ev_leader,"ev_leader","string",ev_date,"ev_date","number",deadline,"deadline","number",ev_place,"ev_place","string",ev_comment,"ev_comment","string",max_count,"max_count","number",min_lvl,"min_lvl","number",max_lvl,"max_lvl","number",classes,"classes","table",roles,"roles","table",sortname,"sortname","string",ev_type,"ev_type","number",ev_subtype,"ev_subtype","number",ev_duration,"ev_duration","number");
  end
  local infos = GEM3_Config[GEM3_Realm][ev_leader];
  local sorttype = GEM3_SORT_GetSortType(sortname);
  
  if(infos == nil)
  then
    return nil,string.format(GEM3_LOC_EVENT_CREATE_ERROR_UNKNOWN_LEADER,ev_leader);
  end
  if(sorttype == nil)
  then
    return nil,string.format(GEM3_LOC_EVENT_CREATE_ERROR_UNKNOWN_SORTTYPE,sortname);
  end
  for clas,tab in pairs(classes)
  do
    if(GEM3_ClassToInt[clas] == nil) -- Unknown class
    then
      return nil,string.format(GEM3_LOC_EVENT_CREATE_ERROR_SANITY_FAILED,"classes");
    end
  end
  if(deadline == nil or deadline == 0)
  then
    deadline = ev_date;
  end
  if(ev_type == nil)
  then
    ev_type = GEM3_EVT_TYPE_NONE;
  end
  if(ev_subtype == nil)
  then
    ev_subtype = GEM3_EVT_TYPE_SUB_NONE;
  end

  infos.next_event_id = infos.next_event_id + 1;
  local ev_id = ev_leader..infos.next_event_id.."."..math.fmod(time(),1000);
  local event = GA_GetTable();
  local stamp = GEM3_STAMP_GetLocalStamp();
  local dup_classes = GEM3_DuplicateTable(classes);
  local dup_roles = GEM3_DuplicateTable(roles);
  -- recover_time is set to stamp-1, because it's needed to detect a recover update of the first version of the event ("recover_time < event.recover_time" test in GEM3_EVT_ActionForIncomingEvent)
  _GEM3_EVT_FillEvent(event,channel,ev_id,stamp,ev_leader,ev_date,ev_place,ev_comment,max_count,min_lvl,max_lvl,dup_classes,dup_roles,GA_GetTable(),sorttype,nil,deadline,stamp-1,ev_type,ev_subtype,1,ev_duration);
  GEM3_QA_Events.events[ev_id] = event;
  _GEM3_EVT_AllocEventTiedVariables(event);
  -- GUI Callback
  GEM3_TriggerCallback("OnEventCreated",event);
  -- Init, sort, update counts after GUI callback, since GUI might have added new subscribers
  _GEM3_EVT_InitPlayersSpecific(event.players);
  GEM3_SORT_SortPlayers(event);
  GEM3_SUB_UpdateCounts(event);
  GEM3_EVT_TouchEvent(event,true);
  GEM3_SUB_CheckMySubscription(event);
  return event.id,nil;
end

function GEM3_EVT_ModifyEvent(ev_id,ev_date,deadline,ev_place,ev_comment,max_count,min_lvl,max_lvl,classes,roles,sortname,ev_type,ev_subtype,ev_duration)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_DBG_CheckTypes("GEM3_EVT_ModifyEvent",ev_id,"ev_id","string",ev_date,"ev_date","number",deadline,"deadline","number",ev_place,"ev_place","string",ev_comment,"ev_comment","string",max_count,"max_count","number",min_lvl,"min_lvl","number",max_lvl,"max_lvl","number",classes,"classes","table",roles,"roles","table",sortname,"sortname","string",ev_type,"ev_type","number",ev_subtype,"ev_subtype","number");
  end
  local event = GEM3_QA_Events.events[ev_id];
  
  if(event == nil)
  then
    return false,GEM3_LOC_EVENT_MODIFY_ERROR_UNKNOWN_EVENT;
  end
  
  local sorttype = GEM3_SORT_GetSortType(sortname);
  
  if(sorttype == nil)
  then
    return false,GEM3_LOC_EVENT_MODIFY_ERROR_UNKNOWN_SORTTYPE;
  end
  if(deadline == nil or deadline == 0)
  then
    deadline = ev_date;
  end
  for clas,tab in pairs(classes)
  do
    if(GEM3_ClassToInt[clas] == nil) -- Unknown class
    then
      return nil,string.format(GEM3_LOC_EVENT_MODIFY_ERROR_SANITY_FAILED,"classes");
    end
  end

  local isDateChanged = (event.ev_date ~= ev_date);
  local dup_classes = GEM3_DuplicateTable(classes);
  local dup_roles = GEM3_DuplicateTable(roles);
  _GEM3_EVT_FillEvent(event,event.channel,event.id,event.update_time,event.leader,ev_date,ev_place,ev_comment,max_count,min_lvl,max_lvl,dup_classes,dup_roles,nil,sorttype,event.closed_comment,deadline,event.recover_time,ev_type,ev_subtype,event.revision+1,ev_duration);

  -- Check players's level
  GEM3_SUB_CheckPlayersLevel(event);

  -- Re order players and update counts
  GEM3_SORT_SortPlayers(event);
  GEM3_SUB_UpdateCounts(event);

  GEM3_EVT_TouchEvent(event,true); -- Update event time and broadcast it
  GEM3_SUB_CheckMySubscription(event);
  
  -- GUI Callback
  GEM3_TriggerCallback("OnUpdatedEvent",event,true,isDateChanged);

  return true,nil;
end

--[[
 function GEM3_EVT_CloseEvent
  Close an event (called by leader only).
   ev_id   : String -- EventID to be closed
   reason  : String -- Reason of the close
]]
function GEM3_EVT_CloseEvent(ev_id,reason)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_DBG_CheckTypes("GEM3_EVT_CloseEvent",ev_id,"ev_id","string",reason,"reason","string");
  end
  local event = GEM3_QA_Events.events[ev_id];
  if(event and GEM3_IsMyReroll(event.leader))
  then
    if(reason == nil or reason == "")
    then
      reason = GEM3_LOC_TEXT_NO_REASON;
    end
    GEM3_EVT_ClearEvent(ev_id,reason,false);
    GEM3_EVT_TouchEvent(event,true);
  end
end

--[[
 function GEM3_EVT_IgnoreEvent
  Ignore an event
   ev_id   : String -- EventID to be closed
]]
function GEM3_EVT_IgnoreEvent(ev_id)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_DBG_CheckTypes("GEM3_EVT_IgnoreEvent",ev_id,"ev_id","string");
  end
  local event = GEM3_QA_Events.events[ev_id];
  if(event and not GEM3_IsMyReroll(event.leader))
  then
    GEM3_QA_Events.ignored[ev_id] = true;
    if(event.update_time ~= 0)
    then
      event.update_time = 1; -- Decrease last update_time, so that if I unignore the event, it won't be accepted by other players
    end
    GEM3_EVT_ClearEvent(ev_id,GEM3_LOC_EVENT_CLOSED_IGNORED,false);
  end
end

function GEM3_EVT_UnIgnoreEvent(ev_id)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_DBG_CheckTypes("GEM3_EVT_UnIgnoreEvent",ev_id,"ev_id","string");
  end
  GEM3_QA_Events.ignored[ev_id] = nil;
end

--[[
 function GEM3_EVT_UnCloseEvent
  Unclose a closed event (called by leader only).
   ev_id   : String  -- EventID to be unclosed
]]
function GEM3_EVT_UnCloseEvent(ev_id)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_DBG_CheckTypes("GEM3_EVT_UnCloseEvent",ev_id,"ev_id","string");
  end
  local event = GEM3_QA_Events.events[ev_id];
  if(event and GEM3_IsMyReroll(event.leader))
  then
    event.closed_comment = nil;
    -- GUI Callback
    GEM3_TriggerCallback("OnUnClosedEvent",event);
    GEM3_EVT_TouchEvent(event,true);
  end
end

--[[
 function GEM3_EVT_DeleteEvent
  Forces all players to fully delete the event (called by leader only)
   ev_id   : String  -- EventID to be deleted
]]
function GEM3_EVT_DeleteEvent(ev_id)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_DBG_CheckTypes("GEM3_EVT_DeleteEvent",ev_id,"ev_id","string");
  end
  local event = GEM3_QA_Events.events[ev_id];
  if(event and GEM3_IsMyReroll(event.leader))
  then
    event.ev_date = 0; -- Force oldest date
    GEM3_QA_Events.my_deleted_events[ev_id] = event; -- Bakcup the event
    GEM3_EVT_TouchEvent(event,true);
    GEM3_QA_Events.events[ev_id] = nil; -- Prevent destruction of back'd up table in 'GEM3_EVT_PurgeEvent' call
    GEM3_EVT_PurgeEvent(ev_id,event);
  end
end

function GEM3_EVT_IsEventIgnored(ev_id)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_DBG_CheckTypes("GEM3_EVT_IsEventIgnored",ev_id,"ev_id","string");
  end
  if(GEM3_QA_Events.ignored[ev_id])
  then
    return true;
  end
  return false;
end

function GEM3_EVT_RecoverLostEvent(event)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_DBG_CheckTypes("GEM3_EVT_RecoverLostEvent",event,"event","table");
  end
  if(_GEM3_EVT_RecoveringEvents[event.id] == nil)
  then
    GEM3_ChatPrint("GEM3_EVT_RecoverLostEvent Error: There is no such event to recover");
    return;
  end
  --GEM3_ChatPrint("You choose to recover the event.");

  GEM3_QA_Events.events[event.id] = event;
  event.recover_time = event.update_time; -- Store recover time
  _GEM3_EVT_AllocEventTiedVariables(event);
  GEM3_EVT_TouchEvent(event,true); -- Update event time and broadcast it
  GEM3_SUB_CheckMySubscription(event);

  _GEM3_EVT_RecoveringEvents[event.id] = nil; -- Remove from table
  
  GEM3_ChatDebug(GEM3_DEBUG_EVENTS,"GEM3_EVT_RecoverLostEvent: Recovering EventID "..event.id.." on channel '"..event.channel.."' UpdateTime="..date("%c",event.update_time).." ("..event.update_time..") RecoverTime="..date("%c",event.recover_time).." ("..event.recover_time..")");

  -- GUI Callback
  GEM3_TriggerCallback("OnNewEventLost",event);
end

function GEM3_EVT_RejectLostEvent(event)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_DBG_CheckTypes("GEM3_EVT_RejectLostEvent",event,"event","table");
  end
  if(_GEM3_EVT_RecoveringEvents[event.id] == nil)
  then
    GEM3_ChatPrint("GEM3_EVT_RejectLostEvent Error: There is no such event to recover");
    return;
  end
  _GEM3_EVT_RecoveringEventsIgnored[event.id] = event.update_time;
  --GEM3_ChatPrint("You choose to ignore the event, discarding it.");
  _GEM3_EVT_RecoveringEvents[event.id] = nil; -- Remove from table
  GA_ReleaseTable(event,10); -- 10 recursion count to ensure full table release
end

function GEM3_EVT_DeleteLostEvent(event,reason)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_DBG_CheckTypes("GEM3_EVT_DeleteLostEvent",event,"event","table",reason,"reason","string");
  end
  if(_GEM3_EVT_RecoveringEvents[event.id] == nil)
  then
    GEM3_ChatPrint("GEM3_EVT_DeleteLostEvent Error: There is no such event to recover");
    return;
  end
  --GEM3_ChatPrint("You choose to delete the lost event.");
  _GEM3_EVT_RecoveringEvents[event.id] = nil; -- Remove from table

  GEM3_QA_Events.events[event.id] = event;
  event.recover_time = GEM3_STAMP_GetLocalStamp(); -- Force recover time now() so everybody will delete event
  GEM3_EVT_ClearEvent(event.id,reason,false);
  GEM3_EVT_DeleteEvent(event.id);
end

function GEM3_EVT_AcceptUpdatedRecoveredEvent(event)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_DBG_CheckTypes("GEM3_EVT_AcceptUpdatedRecoveredEvent",event,"event","table");
  end
  if(_GEM3_EVT_UpdatingRecoveredEvents[event.id] == nil)
  then
    GEM3_ChatPrint("GEM3_EVT_AcceptUpdatedRecoveredEvent Error: There is no such event to update");
    return;
  end
  --GEM3_ChatPrint("You choose to update recovered event.");

  -- Delete previous version
  GEM3_QUE_RemoveEventFromQueue(event.channel,event.id);
  GA_ReleaseTable(GEM3_QA_Events.events[event.id],10); -- 10 recursion count to ensure full table release
  -- Setup new version of the event
  GEM3_QA_Events.events[event.id] = event;
  event.recover_time = event.update_time; -- Store recover time
  GEM3_EVT_TouchEvent(event,true); -- Update event time and broadcast it
  GEM3_SUB_CheckMySubscription(event);

  _GEM3_EVT_UpdatingRecoveredEvents[event.id] = nil; -- Remove from table

  -- GUI Callback
  GEM3_TriggerCallback("OnUpdatedEvent",event,false);
end

function GEM3_EVT_RejectUpdatedRecoveredEvent(event)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_DBG_CheckTypes("GEM3_EVT_RejectUpdatedRecoveredEvent",event,"event","table");
  end
  if(_GEM3_EVT_UpdatingRecoveredEvents[event.id] == nil)
  then
    GEM3_ChatPrint("GEM3_EVT_RejectUpdatedRecoveredEvent Error: There is no such event to update");
    return;
  end
  _GEM3_EVT_UpdatingRecoveredEventsIgnored[event.id] = event.update_time;
  --GEM3_ChatPrint("You choose to ignore the updated event, discarding it.");
  local my_event = GEM3_QA_Events.events[event.id];
  _GEM3_EVT_UpdatingRecoveredEvents[event.id] = nil; -- Remove from table
  my_event.recover_time = event.update_time; -- Store recover time, so that I'll ignore previous events
  GEM3_EVT_TouchEvent(my_event,true); -- Update event time and broadcast it
  -- Remove rebuilded event
  GA_ReleaseTable(event,10); -- 10 recursion count to ensure full table release
end

function GEM3_EVT_SetAssistant(ev_id,pl_name)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_DBG_CheckTypes("GEM3_EVT_SetAssistant",ev_id,"ev_id","string",pl_name,"pl_name","string");
  end
  local event = GEM3_QA_Events.events[ev_id];
  
  if(event and GEM3_IsMyReroll(event.leader))
  then
    GEM3_QA_Events.assistants[ev_id][pl_name] = true;
    GEM3_COM_AddAssistant(event,pl_name);
  end
end

function GEM3_EVT_IsAssistant(ev_id,pl_name)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_DBG_CheckTypes("GEM3_EVT_IsAssistant",ev_id,"ev_id","string",pl_name,"pl_name","string");
  end
  if(GEM3_QA_Events.events[ev_id] == nil or GEM3_QA_Events.assistants[ev_id] == nil)
  then
    return false;
  end
  return GEM3_QA_Events.assistants[ev_id][pl_name] ~= nil;
end

function GEM3_EVT_AmIAssistant(ev_id)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_DBG_CheckTypes("GEM3_EVT_AmIAssistant",ev_id,"ev_id","string");
  end
  if(GEM3_QA_Events.events[ev_id] == nil or GEM3_QA_Events.assistants[ev_id] == nil)
  then
    return false;
  end
  for name in pairs(GEM3_Config[GEM3_Realm])
  do
    if(GEM3_QA_Events.assistants[ev_id][name] ~= nil)
    then
      return true;
    end
  end
  return false;
end

function GEM3_EVT_SetExpirationTimes(expiration_time,expiration_time_self)
  GEM3_QA_Config.expiration_time = expiration_time;
  GEM3_QA_Config.expiration_time_self = expiration_time_self;
end

function GEM3_EVT_GetExpirationTimes()
  return GEM3_QA_Config.expiration_time,GEM3_QA_Config.expiration_time_self;
end

function GEM3_EVT_ArchiveEvent(ev_id)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_DBG_CheckTypes("GEM3_EVT_ArchiveEvent",ev_id,"ev_id","string");
  end
  local event;
  if(type(ev_id) == "string")
  then
    event = GEM3_QA_Events.events[ev_id];
  elseif(type(ev_id) == "table")
  then
    event = ev_id;
  else
    return false;
  end
  
  if(event)
  then
    if(GEM3_QA_Events.archived[event.id] ~= nil) -- Already an archived event
    then
      GA_ReleaseTable(GEM3_QA_Events.archived[event.id],10);
    end
    GEM3_QA_Events.archived[event.id] = GEM3_DuplicateTable(event);
    return true;
  end
  return false;
end

function GEM3_EVT_DeleteArchivedEvent(ev_id)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_DBG_CheckTypes("GEM3_EVT_DeleteArchivedEvent",ev_id,"ev_id","string");
  end
  if(GEM3_QA_Events.archived[ev_id] ~= nil) -- Archived
  then
    GEM3_QA_Events.archived[ev_id] = GA_ReleaseTable(GEM3_QA_Events.archived[ev_id],10);
    return true;
  end
  return false;
end

function GEM3_EVT_KickFromEvent(ev_id,pl_name,reason)
  local event = GEM3_QA_Events.events[ev_id];
  if(event)
  then
    if(GEM3_IsMyReroll(event.leader) == false and GEM3_EVT_AmIAssistant(ev_id) == false)
    then
      return false,GEM3_LOC_EVENT_KICK_ERROR_NOT_LEAD_OR_ASSISTANT;
    end
    GEM3_QA_Events.kicked[ev_id][pl_name] = reason;
    GEM3_SUB_RemoveSubscriber(event,pl_name,string.format(GEM3_LOC_EVENT_KICK_MESSAGE_KICKED,reason));
    GEM3_COM_KickPlayer(event,pl_name,reason);
    return true,nil;
  end
  return false,GEM3_LOC_EVENT_KICK_ERROR_UNKNOWN_EVENT;
end

function GEM3_EVT_BanFromEvent(ev_id,pl_name,reason)
  local event = GEM3_QA_Events.events[ev_id];
  if(event)
  then
    if(GEM3_IsMyReroll(event.leader) == false and GEM3_EVT_AmIAssistant(ev_id) == false)
    then
      return false,GEM3_LOC_EVENT_BAN_ERROR_NOT_LEAD_OR_ASSISTANT;
    end
    GEM3_QA_Events.banned[ev_id][pl_name] = reason;
    GEM3_SUB_RemoveSubscriber(event,pl_name,string.format(GEM3_LOC_EVENT_BAN_MESSAGE_BANNED,reason));
    GEM3_COM_BanPlayer(event,pl_name,reason);
    return true,nil;
  end
  return false,GEM3_LOC_EVENT_BAN_ERROR_UNKNOWN_EVENT;
end

function GEM3_EVT_UnBanFromEvent(ev_id,pl_name)
  local event = GEM3_QA_Events.events[ev_id];
  if(event)
  then
    if(GEM3_IsMyReroll(event.leader) == false and GEM3_EVT_AmIAssistant(ev_id) == false)
    then
      return false,GEM3_LOC_EVENT_UNBAN_ERROR_NOT_LEAD_OR_ASSISTANT;
    end
    GEM3_QA_Events.banned[ev_id][pl_name] = nil;
    GEM3_COM_UnBanPlayer(event,pl_name);
    return true,nil;
  end
  return false,GEM3_LOC_EVENT_UNBAN_ERROR_UNKNOWN_EVENT;
end

function GEM3_EVT_GetBannedPlayers(ev_id)
  local list = GA_GetTable();
  local ban;

  if(GEM3_QA_Events.banned[ev_id])
  then
    for name,reason in pairs(GEM3_QA_Events.banned[ev_id])
    do
      ban = GA_GetTable();
      ban.name = name;
      ban.reason = reason;
      tinsert(list,ban);
    end
  end
  return list;
end
