--[[
  Guild Event Manager (GEM) - Version 3
  by Kiki from European "Conseil des Ombres" (Horde)
  ----------------------------------------
  Core part
]]

--[[
  Exported API Functions:

   GEM3_IsChannelInRerollList(pl_name,channel)
    Input:
     pl_name:  string -- Reroll name to check for channel
     channel:  string -- Real name of the channel to check
    Output:
     result: bool  -- True if reroll 'pl_name' has 'channel' in his configuration
    Purpose:
     Checks if one of my reroll has configured specified channel

   GEM3_IsMyReroll(name)
    Input:
     name:  string -- Reroll name to check
    Output:
     result: bool  -- True if 'name' is one of my rerolls
    Purpose:
     Checks if player is one of my rerolls

]]

--------------- Saved variables ---------------

GEM3_Config = {}; -- Configuration
GEM3_Events = {}; -- Events
GEM3_Players = {}; -- Players infos
GEM3_ExternalCommands = {}; -- Out of game sync'ing variable

--------------- Quick access variables ---------------

GEM3_QA_Events = nil; -- Quick access to events for this realm
GEM3_QA_Config = nil; -- Quick access to config for this character
GEM3_QA_Players = nil; -- Quick access to players for this realm


--------------- Shared variables ---------------

GEM3_NewMinorVersion = false;
GEM3_PlayerName = nil;
GEM3_Realm = nil;
GEM3_PlayerGuild = nil;
GEM3_GuildChannelName = nil;
GEM3_UI_Callbacks = {};

GEM3_Core_Addon = LibStub("AceAddon-3.0"):NewAddon("GEM3_Core_Addon","AceComm-3.0","AceSerializer-3.0");


--------------- Local variables ---------------

-- Optims vars (get a local copy)
local gmatch = string.gmatch;
local tonumber = tonumber;
local strfind = string.find;
local strformat = string.format;

--------------- Internal functions ---------------

local function _GEM3_Commands(command)
  local i,j, cmd, param = strfind(command, "^([^ ]+) (.+)$");
  if(not cmd) then cmd = command; end
  if(not cmd) then cmd = ""; end
  if(not param) then param = ""; end

  if((cmd == "") or (cmd == "help"))
  then
    GEM3_ChatPrint(GEM3_TEXT_USAGE);
    GEM3_CMD_Help();
  else
    if(not GEM3_CMD_Command(cmd,param))
    then
    GEM3_ChatPrint(GEM3_CHAT_CMD_UNKNOWN);
    end
  end
end

local function _GEM3_LoadDefaults()
  -- Default banned
  if(GEM3_Defaults["banned"])
  then
    local banned = GEM3_Defaults["banned"][GEM3_Realm];
    if(banned)
    then
      for name in pairs(banned)
      do
        GEM3_QA_Events.ignored_players[name] = true;
      end
    end
  end
end

local function _GEM3_CheckPlayerGuild()
  if(IsInGuild()) -- Have a guild
  then
    if(GEM3_PlayerGuild == nil) -- Not init yet or joined a guild
    then
      GEM3_PlayerGuild = GetGuildInfo("player");
      if(GEM3_PlayerGuild == nil) -- Guild name still not init
      then
        return false;
      end
      GEM3_GuildChannelName = "guild-"..strlower(GEM3_PlayerGuild);
    end
  else -- Not in guild
    if(GEM3_PlayerGuild ~= nil) -- Left my guild
    then
      GEM3_ChatDebug(GEM3_DEBUG_GLOBAL,"Player left his guild, removing GUILD channel if joined");
      GEM3_CHAN_DelChannel(GUILD); -- Automatically remove the GUILD channel, if was joined
      GEM3_COM_PlayerInfos(); -- Send an update of my infos for remaining channels
    end
    GEM3_PlayerGuild = nil;
    GEM3_GuildChannelName = nil;
  end
  return true;
end

local function _GEM3_InitRealmVars()
  -- Init events
  local newrealm = GA_WipeTable(GEM3_Events[GEM3_Realm],50); -- 50 recursion count to ensure full table release
  newrealm.events = {};
  newrealm.commands = {};
  newrealm.my_deleted_events = {};
  newrealm.ignored = {};
  newrealm.subscribed = {};
  newrealm.kicked = {};
  newrealm.banned = {};
  newrealm.ignored_players = {};
  newrealm.forward = {};
  newrealm.archived = {};
  newrealm.assistants = {};
  newrealm.others_deleted_events = {};
  GEM3_Events[GEM3_Realm] = newrealm;

  -- Init config
  local newrealmconf = GA_WipeTable(GEM3_Config[GEM3_Realm],50); -- 50 recursion count to ensure full table release
  GEM3_Config[GEM3_Realm] = newrealmconf;
end

local function _GEM3_InitToonVars()
  local newconfig = GA_WipeTable(GEM3_Config[GEM3_Realm][GEM3_PlayerName],50); -- 50 recursion count to ensure full table release
  newconfig.next_event_id = 0;
  newconfig.debug = 0;
  newconfig.debug_flags = GEM3_DEBUG_ALL_MASK;
  newconfig.time_ref = false;
  newconfig.role = GEM3_PLAY_GuessPlayerRole();
  newconfig.expiration_time = 60*60*1; -- 1 hour
  newconfig.expiration_time_self = 60*60*10; -- 10 hours
  GEM3_Config[GEM3_Realm][GEM3_PlayerName] = newconfig;
end

local function _GEM3_StartupInitVars()
  local playerName = UnitName("player");

  local reset_by_upgrade = false;
  local first_time_gem = false;
  local first_time_realm = false;
  local first_time_char = false;
  GEM3_PlayerName = playerName;
  GEM3_Realm = GetCVar("realmName");
  
  -- Initialize structs
  if(GEM3_Config.version == nil or GEM3_Config.min_config_version < GEM3_MIN_CONFIG_VERSION_RESET or GEM3_Config.locale == nil) -- First time module init (ever)
  then
    reset_by_upgrade = (GEM3_Config.min_config_version == nil or GEM3_Config.min_config_version < GEM3_MIN_CONFIG_VERSION_RESET);
    first_time_gem = true;
    GEM3_Events = GA_WipeTable(GEM3_Events,50); -- Reset every realms
    GEM3_Config = GA_WipeTable(GEM3_Config,50); -- Reset every realms
    GEM3_Config.min_config_version = GEM3_MIN_CONFIG_VERSION_RESET;
  end
  GEM3_Config.version = GEM3_VERSION; -- Update version
  GEM3_Config.locale = GetLocale(); -- Update locale
  if(GEM3_Events[GEM3_Realm] == nil) -- First time in this realm
  then
    first_time_realm = true;
    _GEM3_InitRealmVars();
  end
  GEM3_QA_Events = GEM3_Events[GEM3_Realm]; -- Quick access to events

  if(GEM3_Config[GEM3_Realm][GEM3_PlayerName] == nil) -- First time with this toon in this realm
  then
    first_time_char = true;
    _GEM3_InitToonVars();
  end
  GEM3_QA_Config = GEM3_Config[GEM3_Realm][GEM3_PlayerName];
  GEM3_DBG_SetDebugMode(GEM3_QA_Config.debug,false);
  GEM3_DebugLog = {}; -- Wipe debug log

  -- Load all defaults values
  _GEM3_LoadDefaults();

  -- Init run time values
  GEM3_QA_Config.my_bcast_offset = GEM3_QUE_SCHEDULE_DELAY_MIN + math.fmod(time(),GEM3_QUE_SCHEDULE_DELAY_MAX-GEM3_QUE_SCHEDULE_DELAY_MIN); -- Change value each login

  GEM3_CHAN_LoadChannelsConfig(GEM3_PlayerName); -- Load channels config
  GEM3_COM_InitRoutines();
  -- GUI Callback
  GEM3_TriggerCallback("OnVariablesLoaded",first_time_gem,first_time_realm,first_time_char,reset_by_upgrade);
end

local _GEM3_core_init_timer = nil;
local _GEM3_core_init_ready = false;
local function _GEM3_OnUpdate(self,dt)
  if(_GEM3_core_init_timer == nil) -- Variables not loaded yet
  then
    return;
  end
  _GEM3_core_init_timer = _GEM3_core_init_timer + dt;
  
  if(_GEM3_core_init_timer > 2.0) -- Time to check for full init (after 2 sec)
  then
    local playerName = UnitName("player");
    if(playerName == nil or playerName == UNKNOWNBEING or playerName == UKNOWNBEING or playerName == UNKNOWNOBJECT or _GEM3_CheckPlayerGuild() == false) -- Not ready yet
    then
      return;
    end
    _GEM3_core_init_ready = true;
  end
  if(_GEM3_core_init_ready)
  then
    _GEM3_StartupInitVars();
    self:SetScript("OnUpdate",nil);
  end
end

local function _GEM3_OnEvent(self,event,...)
  if(event == "VARIABLES_LOADED")
  then
    _GEM3_core_init_timer = 0; -- Start timer now
  end

  if(event == "PLAYER_GUILD_UPDATE")
  then
    _GEM3_CheckPlayerGuild();
  elseif(event == "ZONE_CHANGED_NEW_AREA")
  then
    GEM3_COM_PlayerInfos(); -- Send an update of my infos
  elseif(event == "CHARACTER_POINTS_CHANGED")
  then
    GEM3_ChatDebug(GEM3_DEBUG_GLOBAL,"Player's talent points changed, guessing new role");
    if(GEM3_QA_Config)
    then
      GEM3_QA_Config.role = GEM3_PLAY_GuessPlayerRole();
    end
  end
end


--------------- Initialization functions ---------------

local evFrame = CreateFrame("Frame");
-- Register events
evFrame:RegisterEvent("VARIABLES_LOADED");
evFrame:RegisterEvent("PLAYER_GUILD_UPDATE");
evFrame:RegisterEvent("ZONE_CHANGED_NEW_AREA");
evFrame:RegisterEvent("CHARACTER_POINTS_CHANGED");
evFrame:SetScript("OnEvent",_GEM3_OnEvent);
evFrame:SetScript("OnUpdate",_GEM3_OnUpdate);
--[[
-- Initialize Slash commands
SLASH_GEM1 = "/gem";
SlashCmdList["GEM"] = function(msg)
  _GEM3_Commands(msg);
end
]]
GEM3_ChatPrint("Version "..GEM3_VERSION.." "..GEM3_CHAT_LOADED);


--------------- Exported functions ---------------

function GEM3_IsChannelInRerollList(pl_name,channel)
  local config = GEM3_Config[GEM3_Realm][pl_name];
  
  if(config)
  then
    for i,chantab in ipairs(config.channels)
    do
      if(chantab.name and strlower(chantab.name) == channel)
      then
        return true;
      end
    end
  end
  return false;
end

function GEM3_IsMyReroll(name)
  return GEM3_Config[GEM3_Realm][name] ~= nil;
end

function GEM3_ForceConfigReset()  
  GEM3_Config.min_config_version = 0; -- Force config to be invalidated at next reload ui
end

function GEM3_DuplicateTable(tab)
  if(type(tab) ~= "table")
  then
    return nil;
  end
  local newtab = GA_GetTable();
  for n,v in pairs(tab)
  do
    if(type(v) == "table") -- recurs dup
    then
      newtab[n] = GEM3_DuplicateTable(v);
    else
      newtab[n] = v;
    end
  end
  return newtab;
end

function GEM3_TriggerCallback(cb_name,...)
  if(GEM3_UI_Callbacks[cb_name])
  then
    for _,func in ipairs(GEM3_UI_Callbacks[cb_name])
    do
      func(...);
    end
    return true;
  end
  return false;
end

function GEM3_RegisterCallback(name,func)
  if(GEM3_UI_Callbacks[name] == nil)
  then
    GEM3_UI_Callbacks[name] = {};
  end
  tinsert(GEM3_UI_Callbacks[name],func);
end

function GEM3_UnregisterCallback(name,func)
  if(GEM3_UI_Callbacks[name])
  then
    for i,f in ipairs(GEM3_UI_Callbacks[name])
    do
      if(f == func)
      then
        tremove(GEM3_UI_Callbacks[name],i);
      end
    end
    if(#GEM3_UI_Callbacks[name] == 0)
    then
      GEM3_UI_Callbacks[name] = nil;
    end
  end
end

function GEM3_GetVersion()
  return GEM3_VERSION;
end
