--[[
  Guild Event Manager (GEM) - Version 3
  by Kiki from European "Conseil des Ombres" (Horde)
  ----------------------------------------
  Commands module
]]

--[[
  Exported API Functions:

   GEM3_CMD_GetNonAckedSubscriptions(ev_id)
    Input:
     ev_id:  string -- EventID you want non acked subscriptions
    Output:
     subs: Table of GEM3_NonAckedSubscription struct (or nil) -- Returned table must be freed using GA_ReleaseTable(subs,10) when no longer needed -- Indexed by INT (use ipairs() to process)


]]


--------------- Shared variables ---------------


--------------- Local variables ---------------
local _GEM3_CMD_CloseEventSends = {};
local _GEM3_CMD_NON_AUTO_ACK_COMMANDS = {
  [GEM3_CMD_CMD_ASSISTANT] = true,
  [GEM3_CMD_CMD_NEW_LEADER] = true,
  [GEM3_CMD_CMD_CUSTOM] = true,
};

-- Optims vars (get a local copy)
local tonumber = tonumber;
local tsort = table.sort;

--------------- Internal functions ---------------

--[[
 function _GEM3_CMD_CheckExpiredCommand
  Checks if the Event has expired yet
   ev_id    : String  -- EventID
   ev_date  : Integer -- Event's date
   leader   : String  -- Event leader
  --
   Returns True if the event has expired. False otherwise
]]
local function _GEM3_CMD_CheckExpiredCommand(ev_id,ev_date,leader)
  if(ev_id == nil or ev_date == nil)
  then
    if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
      GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"_GEM3_CMD_CheckExpiredCommand : ev_id or ev_date is nil");
    end
    return true;
  end
  local tim = GEM3_STAMP_GetLocalStamp();
  local expire_time = GEM3_QA_Config.expiration_time;
  if(GEM3_IsMyReroll(leader))
  then
    expire_time = GEM3_QA_Config.expiration_time_self;
  end
  if((ev_date + expire_time) < tim)
  then
    if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
      GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"_GEM3_CMD_CheckExpiredCommand : Command expired for Event "..ev_id.." ! ("..date("%c",ev_date).."--"..date("%c",tim)..")");
    end
    GEM3_EVT_ClearEvent(ev_id,"Expired command",true);
    return true;
  end
  return false;
end

local function _GEM3_CMD_AckCommand(pl_cmd)
  pl_cmd.params[3] = 1;
end

--[[
  _GEM3_CMD_CheckCommandForward :
   Params[1] = leader (STRING) -- Leader of the event
   Params[2] = ev_date (INT) -- Date of the event
   Params[3] = ack (INT) -- 1 or 0, if the packet is an ack to the command or not
   Params[4] = pl_dest (STRING) -- Name of player the command is destinated to
   Params[5] = pl_src (STRING) -- Name of player the command is originating from
   Params[6] = stamp (INT) -- Date the command was first issued
  Returns true if command has been forwarded, false otherwise
]]
local function _GEM3_CMD_CheckCommandForward(channel,cmd,from,ev_id,params)
  local forwarder = GEM3_QA_Events.forward[ev_id];
  
  if(forwarder ~= nil) -- If I forwarded this event to a new leader
  then
    -- Send back the command's header, with ack'ed value as 2 and new leader
    local new_params = GA_GetTable();
    tinsert(new_params,forwarder); -- New leader
    tinsert(new_params,params[2]);
    tinsert(new_params,2); -- Special ack'ed value
    tinsert(new_params,params[4]);
    tinsert(new_params,params[5]);
    tinsert(new_params,params[6]);
    GEM3_Todo("_GEM3_CMD_CheckCommandForward: TODO: Check it's ok");
    GEM3_COM_SendForwardCommand(channel,cmd,ev_id,params)
    return true;
  end
  return false;
end

local function _GEM3_CMD_GetCommandInfosForPlayer(cmd_infos,pl_src)
  local plcmd_infos = cmd_infos.cmds[pl_src];
  if(plcmd_infos == nil) -- First command from this player
  then
    plcmd_infos = GA_GetTable();
    cmd_infos.cmds[pl_src] = plcmd_infos;
  end
  return plcmd_infos;
end

local function _GEM3_CMD_SortPlayerCommands(a,b)
  if(a.stamp > b.stamp)
  then
    return true;
  else
    return false;
  end
end

local function _GEM3_CMD_AckOldCommands(plcmd_infos)
  local count = #plcmd_infos;
  local i = 1;

  -- Search first command that supports auto-acking
  while(i <= count)
  do
    if(_GEM3_CMD_NON_AUTO_ACK_COMMANDS[plcmd_infos[i].cmd] == nil)
    then
      break;
    end
    i = i + 1;
  end

  local cmd = plcmd_infos[i];
  i = i + 1; -- Increment counter, we start acking after this command
  
  -- Auto-ack remaining commands
  while(i <= count)
  do
    if(_GEM3_CMD_NON_AUTO_ACK_COMMANDS[plcmd_infos[i].cmd] == nil and not GEM3_CMD_IsCommandAcked(plcmd_infos[i]))
    then
      _GEM3_CMD_AckCommand(plcmd_infos[i]); -- Locally ack the command
      if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
        GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"_GEM3_CMD_AckOldCommands: Auto acking Cmd "..plcmd_infos[i].cmd.." ("..plcmd_infos[i].stamp..") because of "..cmd.cmd.." ("..cmd.stamp..")");
      end
    end
    i = i + 1;
  end
end

local function _GEM3_CMD_GetPlayerCommand(plcmd_infos,cmd)
  for i,pl_cmd in ipairs(plcmd_infos)
  do
    if(pl_cmd.cmd == cmd)
    then
      return pl_cmd,i;
    end
  end
  return nil;
end

local function _GEM3_CMD_GetObsoletingCommand(plcmd_infos,cmd,stamp)
  if(_GEM3_CMD_NON_AUTO_ACK_COMMANDS[cmd]) -- Command is a non-auto-acking command, nothing can set it obsolete
  then
    return nil;
  end

  local count = #plcmd_infos;
  local i = 1;
  
  -- Search first command that supports auto-acking
  while(i <= count)
  do
    if(_GEM3_CMD_NON_AUTO_ACK_COMMANDS[plcmd_infos[i].cmd] == nil)
    then
      break;
    end
    i = i + 1;
  end

  if(plcmd_infos[i] == nil) -- No more command
  then
    return nil;
  end

  if(plcmd_infos[i].stamp > stamp) -- 'cmd@stamp' is old
  then
    return plcmd_infos[i]; -- This command sets 'cmd@stamp' obsolete, return it
  end
  return nil;
end

local function _GEM3_CMD_UpdateForwardedCommand(cmd_infos,plcmd_infos,cmd,new_leader)
  local pl_cmd = _GEM3_CMD_GetPlayerCommand(plcmd_infos,cmd);

  if(pl_cmd and pl_cmd.params[1] ~= new_leader and pl_cmd.params[4] ~= new_leader and not GEM3_CMD_IsCommandAcked(pl_cmd)) -- This updated command is for me
  then
    if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
      GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"_GEM3_CMD_UpdateForwardedCommand: Updating my command "..cmd.." from "..pl_cmd.params[5].." to "..pl_cmd.params[1].." to new leader "..new_leader);
    end
    -- Update the command
    pl_cmd.params[1] = new_leader;
    pl_cmd.params[4] = new_leader;
    -- Update the leader in global commands struct
    cmd_infos.leader = new_leader;
    -- Schedule the command
    GEM3_QUE_ScheduleCommand(pl_cmd,true);
  end
end

local function _GEM3_CMD_IsNewCommand(plcmd_infos,cmd,stamp)
  local pl_cmd = _GEM3_CMD_GetPlayerCommand(plcmd_infos,cmd);
  if(pl_cmd == nil)
  then
    return nil;
  end
  if(pl_cmd.stamp < stamp) -- New if incoming stamp is > to stored one
  then
    return nil;
  end
  return pl_cmd;
end


--------------- Exported functions ---------------

function GEM3_CMD_IsCommandAcked(pl_cmd)
  return pl_cmd.params[3] == 1;
end

function GEM3_CMD_GetOrCreateCommandInfosForEvent(channel,ev_id,leader,ev_date)
  local s_cmd = GEM3_QA_Events.commands[ev_id];
  if(s_cmd == nil)
  then
    s_cmd = GA_GetTable();
    s_cmd.id = ev_id;
    s_cmd.channel = channel;
    s_cmd.leader = leader;
    s_cmd.ev_date = ev_date;
    s_cmd.cmds = GA_GetTable();
    GEM3_QA_Events.commands[ev_id] = s_cmd;
    if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
      GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"GEM3_CMD_GetOrCreateCommandInfosForEvent: Creating cmds for eventID "..ev_id);
    end
  end
  return s_cmd;
end

function GEM3_CMD_GetCommandInfosForEvent(ev_id)
  return GEM3_QA_Events.commands[ev_id];
end

function GEM3_CMD_CreateCommandForPlayer(channel,ev_id,cmd_infos,cmd,params)
  local pl_src = params[5];
  local stamp = params[6];
  local plcmd_infos = _GEM3_CMD_GetCommandInfosForPlayer(cmd_infos,pl_src);
  local pl_cmd,idx = _GEM3_CMD_GetPlayerCommand(plcmd_infos,cmd);

  if(pl_cmd ~= nil) -- Already have a similar command?
  then
    GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"GEM3_CMD_CreateCommandForPlayer: Already having a command "..cmd.." for this player. Unscheduling and removing old one ("..idx.."), for recreation");
    -- Remove from queue
    GEM3_QUE_UnscheduleCommand(pl_cmd);
    -- Wipe old data
    GA_ReleaseTable(pl_cmd,10); -- 10 recursion count to ensure full table release
    -- Remove from table
    tremove(plcmd_infos,idx);
  end
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"GEM3_CMD_CreateCommandForPlayer: Creating player cmds for eventID "..ev_id.." Cmd="..cmd.." Stamp="..stamp);
  end
  pl_cmd = GA_GetTable();
  pl_cmd.cmd = cmd;
  pl_cmd.id = ev_id;
  pl_cmd.channel = channel;
  pl_cmd.stamp = stamp;
  pl_cmd.params = params;
  tinsert(plcmd_infos,pl_cmd);
  table.sort(plcmd_infos,_GEM3_CMD_SortPlayerCommands); -- Sort by stamp, from more recent to older
  _GEM3_CMD_AckOldCommands(plcmd_infos);
  return pl_cmd;
end

--[[
  GEM3_CMD_ReceivedCommand :
   Params[1] = leader (STRING) -- Leader of the event
   Params[2] = ev_date (INT) -- Date of the event
   Params[3] = ack (INT) -- 1 or 0, if the packet is an ack to the command or not
   Params[4] = pl_dest (STRING) -- Name of player the command is destinated to
   Params[5] = pl_src (STRING) -- Name of player the command is originating from
   Params[6] = stamp (INT) -- Date the command was first issued
]]
function GEM3_CMD_ReceivedCommand(channel,cmd,from,ev_id,params)
  local must_process = false;

  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"GEM3_CMD_ReceivedCommand: Received Command: Leader="..params[1].." EvDate="..params[2].." Ack="..params[3].." Dest="..params[4].." Src="..params[5].." Stamp="..params[6]);
  end
  
  if(GEM3_PLAY_IsPlayerIgnored(params[1])) -- Check if leader is in ignore list
  then
    if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
      GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"Player "..params[1].." is in ignore list (leader), discarding message");
    end
    return must_process;
  end
  if(GEM3_PLAY_IsPlayerIgnored(params[4])) -- Check if pl_dest is in ignore list
  then
    if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
      GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"Player "..params[1].." is in ignore list (dest), discarding message");
    end
    return must_process;
  end
  if(GEM3_PLAY_IsPlayerIgnored(params[5])) -- Check if pl_src is in ignore list
  then
    if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
      GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"Player "..params[1].." is in ignore list (src), discarding message");
    end
    return must_process;
  end

  -- Check if command is expired
  if(_GEM3_CMD_CheckExpiredCommand(ev_id,params[2],params[1]))
  then
    if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
      GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"GEM3_CMD_ReceivedCommand: EventID "..ev_id..", has expired so cmd="..cmd.." is not being processed");
    end
    return must_process;
  end
  
  local cmd_infos = GEM3_CMD_GetOrCreateCommandInfosForEvent(channel,ev_id,params[1],params[2]);
  local plcmd_infos = _GEM3_CMD_GetCommandInfosForPlayer(cmd_infos,params[5]);

  -- Check for command forwarding, if this command is for me, both as leader and dest, and not ack'ed (meaning a subscribe/unsubscribe/reject command)
  if(GEM3_IsMyReroll(params[1]) and GEM3_IsMyReroll(params[4]) and params[3] == 0)
  then
    if(_GEM3_CMD_CheckCommandForward(channel,cmd,from,ev_id,params))
    then
      if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
        GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"GEM3_CMD_ReceivedCommand: Not dispatching Cmd "..cmd.." because I forwarded it to the new leader");
      end
      return false; -- Immediatly return - We should receive soon the new updated command (don't store the command)
    end
  end

  -- Check for forwarded status command
  if(params[3] == 2) -- Special code for forwarded command update
  then
    _GEM3_CMD_UpdateForwardedCommand(cmd_infos,plcmd_infos,cmd,params[1]);
    return false; -- Immediatly return with no further processing
  end

  local obsoleting = _GEM3_CMD_GetObsoletingCommand(plcmd_infos,cmd,params[6]); -- Get any command I might have stored with a stamp more recent than this one
  if(obsoleting) -- I have another command that makes this one obsolete
  then
    if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
      GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"GEM3_CMD_ReceivedCommand: Found obsoleting Cmd "..obsoleting.cmd.." ("..obsoleting.stamp..") for Cmd "..cmd.." ("..params[6]..") from "..params[5]);
    end
    GEM3_QUE_ScheduleCommand(obsoleting,false); -- Schedule the obsoleting command
  else -- Not obsolete command
    local pl_cmd = _GEM3_CMD_IsNewCommand(plcmd_infos,cmd,params[6]);
    if(pl_cmd == nil) -- New command for this player (new 'cmd' or more recent)
    then
      pl_cmd = GEM3_CMD_CreateCommandForPlayer(channel,ev_id,cmd_infos,cmd,params); -- Store command
      if(GEM3_IsMyReroll(params[4]) and params[3] == 0) -- Command for me and not acked
      then
        if(GEM3_IsMyReroll(params[1]) and GEM3_QA_Events.events[ev_id] == nil) -- Command is for me as leader, but I don't know the event -> Don't process the command, don't ack it (will be resent later)
        then
          -- Check if event has been deleted
          local del_evt = GEM3_EVT_GetDeletedEvent(ev_id);
          if(del_evt) -- Yes, deleted by me
          then
            if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
              GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"GEM3_CMD_ReceivedCommand: Cmd "..pl_cmd.cmd.." ("..pl_cmd.stamp.." from "..params[5]..") for me as leader but I deleted the event. Broadcast the deleted event and ACK the command");
            end
            _GEM3_CMD_AckCommand(pl_cmd); -- Locally ack the command
            GEM3_COM_SendCommand(pl_cmd); -- Send it immediatly
            GEM3_COM_BroadcastEvent(del_evt); -- Immediatly broadcast my saved copy of deleted event
          else
            if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
              GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"GEM3_CMD_ReceivedCommand: Cmd "..pl_cmd.cmd.." ("..pl_cmd.stamp.." from "..params[5]..") for me as leader but I don't know the event. DO NOT ACK");
            end
          end
          must_process = false; -- Don't process the command
        else
          if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
            GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"GEM3_CMD_ReceivedCommand: Cmd "..pl_cmd.cmd.." ("..pl_cmd.stamp.." from "..params[5]..") for me, ACKing it");
          end
          _GEM3_CMD_AckCommand(pl_cmd); -- Locally ack the command
          GEM3_COM_SendCommand(pl_cmd); -- Send it immediatly
          must_process = true; -- And allow processing of the command
        end
      else
        if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
          GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"GEM3_CMD_ReceivedCommand: Received a new command, but not for me");
        end
      end
    else -- Same command - Check ACK status
      if(params[6] ~= pl_cmd.stamp) -- Diffent time, should not happen (older trapped by _GEM3_CMD_GetObsoletingCommand and newer by _GEM3_CMD_IsNewCommand)
      then
        GEM3_ChatWarning("GEM3_CMD_ReceivedCommand: Time differs: "..params[6].." - "..pl_cmd.stamp);
      end
      if(params[3] == 0 and pl_cmd.params[3] == 1) -- NACK coming, but I have an ACK
      then
        GEM3_QUE_ScheduleCommand(pl_cmd,false); -- Schedule a send of my ACK'ed command
        if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
          GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"GEM3_CMD_ReceivedCommand: NACK coming, but I have an ACK");
        end
      elseif(params[3] == 1 and pl_cmd.params[3] == 0) -- ACK coming, but I have a NACK
      then
        GEM3_QUE_UnscheduleCommand(pl_cmd); -- Unschedule a possible queue send
        _GEM3_CMD_AckCommand(pl_cmd); -- Locally ack the command
        if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
          GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"GEM3_CMD_ReceivedCommand: ACK coming, but I have a NACK");
        end
      else -- Same ACK status
        if(GEM3_IsMyReroll(params[4]) and params[3] == 0) -- Command for me and not acked, I must have lost the event, check again
        then
          if(GEM3_IsMyReroll(params[1]) and GEM3_QA_Events.events[ev_id] == nil) -- Command is for me as leader, but I don't know the event -> Don't process the command, don't ack it (will be resent later)
          then
            if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
              GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"GEM3_CMD_ReceivedCommand: Cmd "..pl_cmd.cmd.." ("..pl_cmd.stamp.." from "..params[5]..") for me as leader but I still don't know the event. DO NOT ACK");
            end
            must_process = false; -- Don't process the command
          else
            if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
              GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"GEM3_CMD_ReceivedCommand: Cmd "..pl_cmd.cmd.." ("..pl_cmd.stamp.." from "..params[5]..") for me and I now know the event, ACKing it");
            end
            _GEM3_CMD_AckCommand(pl_cmd); -- Locally ack the command
            GEM3_COM_SendCommand(pl_cmd); -- Send it immediatly
            must_process = true; -- And allow processing of the command
          end
        else
          GEM3_QUE_UnscheduleCommand(pl_cmd); -- Unschedule a possible queue send
          if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
            GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"GEM3_CMD_ReceivedCommand: Same command I have: Discarding");
          end
        end
      end
    end
  end

  return must_process;
end


--[[
 function GEM3_CMD_CheckExpiredCommands
  Checks all commands, for expired ones
]]
function GEM3_CMD_CheckExpiredCommands()
  for ev_id,cmdtab in pairs(GEM3_QA_Events.commands)
  do
    _GEM3_CMD_CheckExpiredCommand(ev_id,cmdtab.ev_date,cmdtab.leader);
  end
end

function GEM3_CMD_IsLatestCommandAcked(cmd_infos,pl_name,cmd)
  local plcmd_infos = _GEM3_CMD_GetCommandInfosForPlayer(cmd_infos,pl_name);
  if(plcmd_infos)
  then
    local pl_cmd = _GEM3_CMD_GetPlayerCommand(plcmd_infos,cmd);
    if(pl_cmd)
    then
      return GEM3_CMD_IsCommandAcked(pl_cmd);
    end
  end
  return false;
end

local function _GEM3_CMD_GetNonAckedSubscriptions_sort(a,b)
  return a.main_time < b.main_time;
end

function GEM3_CMD_GetNonAckedSubscriptions(ev_id)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_DBG_CheckTypes("GEM3_CMD_GetNonAckedSubscriptions",ev_id,"ev_id","string");
  end
  local event = GEM3_QA_Events.events[ev_id];

  if(event)
  then
    -- Get the CMD struct for this Event
    local cmd_infos = GEM3_CMD_GetCommandInfosForEvent(ev_id);
    if(cmd_infos == nil)
    then
      return nil;
    end

    local list = GA_GetTable();
    for pl_name,plcmd_infos in pairs(cmd_infos.cmds)
    do
      local pl_cmd = _GEM3_CMD_GetPlayerCommand(plcmd_infos,GEM3_CMD_CMD_SUBSCRIBE);
      if(pl_cmd)
      then
        if(not GEM3_CMD_IsCommandAcked(pl_cmd))
        then
          local infos = GA_GetTable();
          infos.name = pl_name;
          infos.class = pl_cmd.params[7];
          infos.role = pl_cmd.params[8];
          infos.level = pl_cmd.params[9];
          infos.guild = pl_cmd.params[10];
          infos.comment = pl_cmd.params[11];
          infos.force_queue = pl_cmd.params[12];
          infos.alt_role = pl_cmd.params[13];
          infos.main_time = pl_cmd.params[14];
          infos.update_time = pl_cmd.params[15];
          infos.source = pl_cmd.params[16];
          tinsert(list,infos);
        end
      end
    end
    tsort(list,_GEM3_CMD_GetNonAckedSubscriptions_sort);
    return list;
  end
  return nil;
end

function GEM3_CMD_GetNonAckedSubscription(ev_id,pl_name)
  local event = GEM3_QA_Events.events[ev_id];

  if(event)
  then
    -- Get the CMD struct for this Event
    local cmd_infos = GEM3_CMD_GetCommandInfosForEvent(ev_id);
    if(cmd_infos == nil)
    then
      return nil;
    end

    for name,plcmd_infos in pairs(cmd_infos.cmds)
    do
      if(name == pl_name)
      then
        local pl_cmd = _GEM3_CMD_GetPlayerCommand(plcmd_infos,GEM3_CMD_CMD_SUBSCRIBE);
        if(pl_cmd)
        then
          return pl_cmd;
        end
      end
    end
  end
  return nil;
end


-- ################ TEMPO DEBUG TO REMOVE #####################
GEM3_CMD_GetCommandInfosForPlayer = _GEM3_CMD_GetCommandInfosForPlayer;
GEM3_CMD_GetPlayerCommand = _GEM3_CMD_GetPlayerCommand;
