--[[
  Guild Event Manager (GEM) - Version 3
  by Kiki from European "Conseil des Ombres" (Horde)
  ----------------------------------------
  Low level communication module
]]


--[[
  Exported API Functions:

   GEM3_COM_RegisterCustomMessage(identifier,func)
    Input:
     identifier: String   -- Unique identifier for your custom message
     func:       Function -- Function to be called when you receive a message with this identifier: function func(channel,from,stamp,...)
    Output:
     result: bool -- True if the custom dispatch has been registered. False if 'identifier' is already registered
    Purpose:
     Adds a command dispatch callback function

   GEM3_COM_RegisterCustomCommand(identifier,func)
    Input:
     identifier: String   -- Unique identifier for your custom message
     func:       Function -- Function to be called when you receive a command with this identifier: function func(channel,from,stamp,ev_id,...)
    Output:
     result: bool -- True if the custom dispatch has been registered. False if 'identifier' is already registered
    Purpose:
     Adds a command dispatch callback function

   GEM3_COM_SendCustomMessage(channel,identifier,...)
    Input:
     channel:    String -- Channel to send message to
     identifier: String -- Unique identifier for your custom message
     ...:        Any    -- Parameters to send
    Output:
     result: bool -- True if the custom dispatch for 'identifier' exists, false otherwise
    Purpose:
     Sends a custom message to all connected GEM users (message is volatile and not saved by GEM3_Core)

   GEM3_COM_SendCustomCommand(channel,identifier,ev_id,pl_dest,...)
    Input:
     channel:    String -- Channel to send message to
     identifier: String -- Unique identifier for your custom message
     ev_id     : String -- Event the command is tied to
     pl_dest   : String -- Player the command is destinated to
     ...:        Any    -- Parameters to send
    Output:
     result: bool -- True if the custom dispatch for 'identifier' and the event exists, false otherwise
    Purpose:
     Sends a custom command to all connected GEM users (command is stored by all users and broadcasted until pl_dest receives it)

   GEM3_COM_IsDisabled()
    Output:
     disabled: bool -- True if GEM communication have been disabled, false otherwise
     reason: string or nil -- The reason for GEM to have been disabled, or nil if GEM is fully functionnal
    Purpose:
     Checks if GEM communication have been disabled, and why.

   GEM3_COM_Disable(disabled,reason)
    Input:
     disabled: bool -- Set to true to disable GEM3 communications, false to re-enable it
     reason: string or nil -- The reason to disable GEM (ignored if disabled=true)
    Purpose:
     Enables or disables GEM communications.


]]

--------------- Local Constantes ---------------


--------------- Shared variables ---------------

GEM3_GLOBAL_CMD = "GblEvtId1"; -- special ev_id tag for global commands
-- Volatile messages
GEM3_CMD_EVENT_UPDATE = 1; -- Event updated (or created)
GEM3_CMD_PLAYER_INFOS = 3; -- New infos about myself
GEM3_CMD_INCORRECT_STAMP = 4; -- Player has an incorrect time stamp
GEM3_CMD_CUSTOM_MSG = 5; -- Custom message - not saved
GEM3_CMD_DELETED_EVENT = 6; -- Event has been deleted
GEM3_CMD_CHAT_MESSAGE = 7; -- Chat message

-- Stored messages
GEM3_FIRST_EVENT_COMMAND = 21; -- Commands I must store (non volatile commands)
GEM3_CMD_CMD_SUBSCRIBE = 21; -- Subscribe: <Class> <Role> <Level> <Guild> <Comment> <Queue> <AltRole> <MainTime> <UpdateTime> <Source>
GEM3_CMD_CMD_UNSUBSCRIBE = 22; -- Unsubscribe: <Comment>
GEM3_CMD_CMD_KICK = 25; -- Kick: <Reason>
GEM3_CMD_CMD_BAN = 26; -- Ban: <Reason>
GEM3_CMD_CMD_UNBAN = 27; -- UnBan
GEM3_CMD_CMD_ASSISTANT = 28; -- Add assistant
GEM3_CMD_CMD_NEW_LEADER = 29; -- New event leader

GEM3_CMD_CMD_CUSTOM = 41; -- Custom command - saved

GEM3_CMD_ERR_SUBSCRIBE = 51; -- Error subscribing: <ErrorCode>

-- Other variables
GEM3_COM_LastJoinerTime = 0;
GEM3_IntToClass = { [1] = "WARRIOR"; [2] = "PALADIN"; [3] = "SHAMAN"; [4] = "ROGUE"; [5] = "MAGE"; [6] = "WARLOCK"; [7] = "HUNTER"; [8] = "DRUID"; [9] = "PRIEST"; [0] = "DEATHKNIGHT" };
--GEM3_ClassToInt = { ["WARRIOR"] = 1; ["PALADIN"] = 2; ["SHAMAN"] = 3; ["ROGUE"] = 4; ["MAGE"] = 5; ["WARLOCK"] = 6; ["HUNTER"] = 7; ["DRUID"] = 8; ["PRIEST"] = 9; ["DEATHKNIGHT"] = 0 };
GEM3_ClassToInt = { ["WARRIOR"] = 1; ["PALADIN"] = 2; ["SHAMAN"] = 3; ["ROGUE"] = 4; ["MAGE"] = 5; ["WARLOCK"] = 6; ["HUNTER"] = 7; ["DRUID"] = 8; ["PRIEST"] = 9; }; -- Don't set DK right now


--------------- Local variables ---------------

local _GEMComm_Dispatch = {};
local _GEMComm_CustomMessageDispatch = {};
local _GEMComm_CustomCommandDispatch = {};
local _GEM3_COM_RoutineScheduled = false;
local _GEM3_COM_LastPlayerInfosSend = 0;
local GEM3_COMM_NIL_VALUE = "nil value";
local GEM3_COM_OldVersion = false;
local _GEM3_COM_DisableMyself = false;
local _GEM3_COM_DisableMyselfReason = nil;

local _GEM3_COM_CmdToString = { [GEM3_CMD_EVENT_UPDATE] = "EvtUpdt"; [GEM3_CMD_PLAYER_INFOS] = "PlayerInfos";[GEM3_CMD_INCORRECT_STAMP] = "IncorrectStamp";[GEM3_CMD_CUSTOM_MSG] = "CustomMsg";[GEM3_CMD_DELETED_EVENT] = "DeletedEvent";[GEM3_CMD_CHAT_MESSAGE] = "ChatMessage";
                               [GEM3_CMD_CMD_SUBSCRIBE] = "Subscribe"; [GEM3_CMD_CMD_UNSUBSCRIBE] = "Unsubscribe"; [GEM3_CMD_CMD_KICK] = "Kick"; [GEM3_CMD_CMD_BAN] = "Ban"; [GEM3_CMD_CMD_UNBAN] = "UnBan"; [GEM3_CMD_CMD_ASSISTANT] = "Assistant"; [GEM3_CMD_CMD_NEW_LEADER] = "NewLeader"; 
                               [GEM3_CMD_CMD_CUSTOM] = "CustomCmd";
                               [GEM3_CMD_ERR_SUBSCRIBE] = "Sub_Error";
};

-- Optims vars (get a local copy)
local tinsert = table.insert;
local gmatch = string.gmatch;
local tonumber = tonumber;


--------------- Internal functions ---------------

local function _GEM3_COM_ScheduledPlayerInfos()
  if((_GEM3_COM_LastPlayerInfosSend + GEM3_PLAY_UPDATE_INFOS_DELAY < time()))
  then
    if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
      GEM3_ChatDebug(GEM3_DEBUG_GLOBAL,"_GEM3_COM_ScheduledPlayerInfos : Too long since last player infos send.");
    end
    GEM3_COM_PlayerInfos();
  end

  -- Reschedule routine
  Chronos.schedule(10,_GEM3_COM_ScheduledPlayerInfos);
end

local function _GEM3_COM_UpdateNilValues(params)
  for i,val in ipairs(params)
  do
    if(val == GEM3_COMM_NIL_VALUE)
    then
      params[i] = nil;
    end
  end
end

--[[
 function _GEM3_COM_ParseChannelMessage
   Channel parsing function.
    channel: real channel name
    from: Message sender
    protocol: GEM protocol version
    cmd: GEM message command
    ev_id: EventID of the message
    stamp: Stamp of the sender
    params: Params of the message
]]
function _GEM3_COM_ParseChannelMessage(channel,from,result,protocol,cmd,ev_id,stamp,params)
  if(from == GEM3_PlayerName) -- Don't process my messages
  then
    -- Except for ChatMessage
    if(cmd ~= GEM3_CMD_CHAT_MESSAGE)
    then
      return;
    end
  end
  if(result == false)
  then
    GEM3_ChatDebug(GEM3_DEBUG_WARNING,"GEM3_COM_Deserialize error from "..from..": "..protocol); -- Parameter after 'result' returned by AceSerializer:Deserialize is the error string (result of pcall() )
    return;
  end
  if(_GEM3_COM_DisableMyself) -- I disabled myself because of incorrect time settings?
  then
    return;
  end
  --[[if(not GEM3_CHAN_IsChannelActive(channel)) -- Don't check this, we need to receive from any channel from any reroll
  then
    GEM3_ChatWarning("_GEM3_COM_ParseChannelMessage: Channel '"..channel.."' is not active, but received a message in this channel");
    return;
  end]]
  
  -- Some Debug
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1)
  then
    GEM3_ChatDebug(GEM3_DEBUG_PROTOCOL,"_GEM3_COM_ParseChannelMessage: Got a message from "..tostring(from)..": Vers="..tostring(protocol).." - Cmd="..tostring(cmd).." - EventID="..tostring(ev_id).." - Stamp="..tostring(stamp));
    if(params and type(params) == "table")
    then
      for i,param in ipairs(params)
      do
        if(param == GEM3_COMM_NIL_VALUE)
        then
          GEM3_ChatDebug(GEM3_DEBUG_PROTOCOL,"Param"..i..": Type=nil");
        else
          GEM3_ChatDebug(GEM3_DEBUG_PROTOCOL,"Param"..i..": Type="..type(param).." - Val="..tostring(param));
          if(type(param) == "table")
          then
            for j,k in pairs(param)
            do
              GEM3_ChatDebug(GEM3_DEBUG_PROTOCOL," Sub param "..j..": Type="..type(k).." - Val="..tostring(k));
              --[[if(type(k) == "table")
              then
                for l,m in pairs(k)
                do
                  GEM3_ChatDebug(GEM3_DEBUG_PROTOCOL,"  Sub sub param "..l..": Type="..type(m).." - Val="..tostring(m));
                  if(type(m) == "table")
                  then
                    for n,o in pairs(m)
                    do
                      GEM3_ChatDebug(GEM3_DEBUG_PROTOCOL,"   Sub sub sub param "..n..": Type="..type(o).." - Val="..tostring(o));
                    end
                  end
                end
              end]]
            end
          end
        end
      end
    end
  end

  -- Check for ignored player
  if(GEM3_PLAY_IsPlayerIgnored(from))
  then
    if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
      GEM3_ChatDebug(GEM3_DEBUG_PROTOCOL,"Player "..from.." is in ignore list, discarding message");
    end
    return;
  end

  -- Check for invalid EventID
  if(ev_id == nil)
  then
    if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
      GEM3_ChatWarning("_GEM3_COM_ParseChannelMessage: Received a NIL EventID message");
    end
    return;
  end

  -- Check for ignored event
  if(GEM3_EVT_IsEventIgnored(ev_id))
  then
    if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
      GEM3_ChatDebug(GEM3_DEBUG_PROTOCOL,"Event "..ev_id.." is in ignore list, discarding message");
    end
    return;
  end

  -- Check GEM version
  if(protocol ~= GEM3_COM_PROTOCOL_REVISION)
  then
    if(protocol > GEM3_COM_PROTOCOL_REVISION)
    then
      if(GEM3_COM_OldVersion == false)
      then
        -- GUI Callback
        GEM3_TriggerCallback("OnUpgradeNeeded",protocol);
        GEM3_COM_OldVersion = true;
      end
    else
      -- GUI Callback
      GEM3_TriggerCallback("OnIncompatibleVersion",channel,from,protocol);
      if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
        GEM3_ChatDebug(GEM3_DEBUG_WARNING,from.." is using an old GEM3 protocol ("..protocol..") version and should upgrade");
      end
    end
    return;
  end

  -- Check sender timestamp
  if(GEM3_STAMP_CheckStamp(channel,from,stamp,cmd,params) == false) -- Invalid stamp, ignore message
  then
    return;
  end

  -- Check expired event
  if(ev_id ~= GEM3_GLOBAL_CMD and GEM3_EVT_CheckExpiredEvent(ev_id)) -- Message for an expired event, just return
  then
    GEM3_ChatDebug(GEM3_DEBUG_PROTOCOL,"GEM3_COM_ParseChannelMessage: Received message is for expired EventID "..ev_id.."! Discarding message");
    return;
  end

  -- Check unknown command
  if(_GEMComm_Dispatch[cmd] == nil)
  then
    if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
      GEM3_ChatDebug(GEM3_DEBUG_PROTOCOL,"Unknown command : "..cmd.." from "..from);
    end
    return;
  end

  -- Change all GEM3_COMM_NIL_VALUE back to nil
  _GEM3_COM_UpdateNilValues(params);

  -- Check for a 'command' message
  if(cmd >= GEM3_FIRST_EVENT_COMMAND)
  then
    if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
      GEM3_ChatDebug(GEM3_DEBUG_PROTOCOL,"GEM3_COM_ParseChannelMessage : Received command "..tostring(_GEM3_COM_CmdToString[cmd]).." from "..from.." for EventID "..ev_id);
    end
    if(GEM3_CMD_ReceivedCommand(channel,cmd,from,ev_id,params) == false) -- This command does not need more processing, don't dispatch, and return now
    then
      GEM3_ChatDebug(GEM3_DEBUG_PROTOCOL,"No need to dispatch command "..tostring(_GEM3_COM_CmdToString[cmd]));
      return;
    end
  end

  -- Dispatch packet
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_ChatDebug(GEM3_DEBUG_PROTOCOL,"GEM3_COM_ParseChannelMessage : Dispatching command "..tostring(_GEM3_COM_CmdToString[cmd]).." from "..from.." for EventID "..ev_id);
  end
  _GEMComm_Dispatch[cmd](channel,from,stamp,ev_id,params);
end

function _GEM3_COMM_OnMessageGUILD(prefix,Msgs,distribution,target)
  if(distribution ~= "GUILD")
  then
    GEM3_ChatWarning("_GEM3_COMM_OnMessageGUILD: Not from GUILD: "..distribution);
    return;
  end
  if(GEM3_GuildChannelName ~= nil)
  then
    _GEM3_COM_ParseChannelMessage(GEM3_GuildChannelName,target,GEM3_Core_Addon:Deserialize(Msgs));
  else
    GEM3_ChatDebug(GEM3_DEBUG_WARNING,"GEM3_COM_OnReceiveMessageGUILD: GEM3_GuildChannelName is currently nil, dropping message");
  end
end

function GEM3_COM_OnReceiveMessage(prefix,message,channel,sender)
  _GEM3_COM_ParseChannelMessage(channel,sender,GEM3_Core_Addon:Deserialize(message));
end

local function _GEM3_COM_NilToValue(v)
  if(v == nil) then return GEM3_COMM_NIL_VALUE; end
  return v;
end

local function _GEM3_COM_ValueToNil(v)
  if(v == GEM3_COMM_NIL_VALUE) then return nil; end
  return v;
end

local function _GEM3_COM_BuildLimitToInt(limits)
  local lim = GA_GetTable();
  
  tinsert(lim,_GEM3_COM_NilToValue(limits["min"]));
  tinsert(lim,_GEM3_COM_NilToValue(limits["max"]));
  return lim;
end

local function _GEM3_COM_UnbuildLimitFromInt(limits)
  local lim = GA_GetTable();
  
  lim["min"] = _GEM3_COM_ValueToNil(limits[1]);
  lim["max"] = _GEM3_COM_ValueToNil(limits[2]);
  return lim;
end

local function _GEM3_COM_BuildClassesToInt(classes)
  local clas = GA_GetTable();
  if(classes ~= nil)
  then
    for name,tab in pairs(classes)
    do
      if(GEM3_ClassToInt[name])
      then
        clas[GEM3_ClassToInt[name]] = _GEM3_COM_BuildLimitToInt(tab);
      end
    end
  end
  return clas;
end

local function _GEM3_COM_UnbuildClassesFromInt(classes)
  local clas = GA_GetTable();
  if(classes ~= nil)
  then
    for idx,tab in pairs(classes)
    do
      clas[GEM3_IntToClass[idx]] = _GEM3_COM_UnbuildLimitFromInt(tab);
    end
  end
  return clas;
end

local function _GEM3_COM_BuildRolesToInt(roles)
  local rol = GA_GetTable();
  if(roles ~= nil)
  then
    for id,tab in pairs(roles)
    do
      rol[id] = _GEM3_COM_BuildLimitToInt(tab);
    end
  end
  return rol;
end

local function _GEM3_COM_UnbuildRolesFromInt(roles)
  local rol = GA_GetTable();
  if(roles ~= nil)
  then
    for id,tab in pairs(roles)
    do
      rol[id] = _GEM3_COM_UnbuildLimitFromInt(tab);
    end
  end
  return rol;
end

local function _GEM3_COM_BuildPlayerToInt(player)
  local pl = GA_GetTable();

  tinsert(pl,_GEM3_COM_NilToValue(player.name));
  tinsert(pl,_GEM3_COM_NilToValue(player.class));
  tinsert(pl,_GEM3_COM_NilToValue(player.role));
  tinsert(pl,_GEM3_COM_NilToValue(player.level));
  tinsert(pl,_GEM3_COM_NilToValue(player.guild));
  tinsert(pl,_GEM3_COM_NilToValue(player.update_time));
  tinsert(pl,_GEM3_COM_NilToValue(player.comment));
  tinsert(pl,_GEM3_COM_NilToValue(player.force_queue));
  tinsert(pl,_GEM3_COM_NilToValue(player.alt_role));
  tinsert(pl,_GEM3_COM_NilToValue(player.main_time));
  tinsert(pl,_GEM3_COM_NilToValue(player.current_queue));
  tinsert(pl,_GEM3_COM_NilToValue(player.queue_pos));
  tinsert(pl,_GEM3_COM_NilToValue(player.lead_force_queue));
  tinsert(pl,_GEM3_COM_NilToValue(player.source));

  return pl;
end

local function _GEM3_COM_UnbuildPlayerFromInt(player)
  local pl = GA_GetTable();

  pl.name = _GEM3_COM_ValueToNil(player[1]);
  pl.class = _GEM3_COM_ValueToNil(player[2]);
  pl.role = _GEM3_COM_ValueToNil(player[3]);
  pl.level = _GEM3_COM_ValueToNil(player[4]);
  pl.guild = _GEM3_COM_ValueToNil(player[5]);
  pl.update_time = _GEM3_COM_ValueToNil(player[6]);
  pl.comment = _GEM3_COM_ValueToNil(player[7]);
  pl.force_queue = _GEM3_COM_ValueToNil(player[8]);
  pl.alt_role = _GEM3_COM_ValueToNil(player[9]);
  pl.main_time = _GEM3_COM_ValueToNil(player[10]);
  pl.current_queue = _GEM3_COM_ValueToNil(player[11]);
  pl.queue_pos = _GEM3_COM_ValueToNil(player[12]);
  pl.lead_force_queue = _GEM3_COM_ValueToNil(player[13]);
  pl.source = _GEM3_COM_ValueToNil(player[14]);
  if(pl.source == nil)
  then
    pl.source = GEM3_SUB_SOURCE_SELF;
  end

  return pl;
end

local function _GEM3_COM_BuildSubscribersToInt(event)
  local pls = GA_GetTable();

  for _,tab in pairs(event.players)
  do
    tinsert(pls,_GEM3_COM_BuildPlayerToInt(tab));
  end

  return pls;
end

local function _GEM3_COM_UnbuildSubscribersFromInt(players)
  local pls = GA_GetTable();

  for _,tab in ipairs(players)
  do
    local pl = _GEM3_COM_UnbuildPlayerFromInt(tab);
    pls[pl.name] = pl;
  end

  return pls;
end

local function _GEM3_COM_RealSendMessage(channel,cmd,ev_id,params)
  if(_GEM3_COM_DisableMyself) -- I disabled myself because of incorrect time settings?
  then
    return;
  end
  local rt = GEM3_RT_Channels[channel];
  if(rt)
  then
    local tim = GEM3_STAMP_GetLocalStamp();
    if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
      GEM3_ChatDebug(GEM3_DEBUG_PROTOCOL,"Sending message to channel '"..tostring(channel).."': Stamp="..tostring(tim).." Cmd="..tostring(_GEM3_COM_CmdToString[cmd]).." EventId="..tostring(ev_id).." ParamsCount="..#params);
    end
    local str = GEM3_Core_Addon:Serialize(GEM3_COM_PROTOCOL_REVISION,cmd,ev_id,tim,params);
    if(rt.saved.guild)
    then
      GEM3_Core_Addon:SendCommMessage(GEM3_COMM_PREFIX,str,"GUILD");
    else
      CustomChatComm.SendChatMessage(GEM3_COMM_PREFIX,str,channel)
    end
  else
    GEM3_ChatDebug(GEM3_DEBUG_WARNING,"_GEM3_COM_RealSendMessage: Channel '"..tostring(channel).."' is unknown to GEM RTChannels. Dropping message!");
  end
end

--[[
 function _GEM3_COM_SendNormalMessage
  Low level sending function
   channel : String  -- Channel to send message to
   cmd     : String  -- Command to send
   ev_id   : String  -- EventId command is refering to
   ...     : Any     -- Params associated with command
]]
local function _GEM3_COM_SendNormalMessage(channel,cmd,ev_id,...)
  if(_GEM3_COM_DisableMyself) -- I disabled myself because of incorrect time settings?
  then
    return;
  end

  local params = GA_GetTable();
  for i=1, select("#",...)
  do
    local val = select(i,...);
    if(val == nil)
    then
      val = GEM3_COMM_NIL_VALUE;
    end
    tinsert(params,val);
  end

  _GEM3_COM_RealSendMessage(channel,cmd,ev_id,params);
  GA_ReleaseTable(params);
end

--[[
 function _GEM3_COM_PrepareAndSendCommand
  Low level command storing and sending function
   event   : Array      -- The event the command is refering to
   cmd     : String     -- Command to store/send
   pl_dest : String     -- Name of player the command is destinated to
   pl_src  : String     -- Name of player the command is originating from
   ...     : Any        -- Params associated with command
]]
local function _GEM3_COM_PrepareAndSendCommand(event,cmd,pl_dest,pl_src,...)
  local stamp = GEM3_STAMP_GetLocalStamp();
  local params = GA_GetTable();
  local v;

  tinsert(params,event.leader); -- Cmd[1] = Leader
  tinsert(params,event.ev_date); -- Cmd[2] = Event Date
  tinsert(params,0); -- Cmd[3] = Ack status (0=Not acked)
  tinsert(params,pl_dest); -- Cmd[4] = Dest
  tinsert(params,pl_src); -- Cmd[5] = Src
  tinsert(params,stamp); -- Cmd[6] = Stamp command is issued

  for i=1, select("#",...)
  do -- Build params array to store command
    v = select(i,...);
    if(v == nil)
    then
      tinsert(params,GEM3_COMM_NIL_VALUE);
    else
      tinsert(params,v);
    end
  end

  -- Create the CMD struct for this Event, if it does not exist
  local cmd_infos = GEM3_CMD_GetOrCreateCommandInfosForEvent(event.channel,event.id,event.leader,event.ev_date);
  
  -- Save the command
  GEM3_CMD_CreateCommandForPlayer(event.channel,event.id,cmd_infos,cmd,params);
  
  -- Send it
  _GEM3_COM_RealSendMessage(event.channel,cmd,event.id,params);
end


--------------- Process functions ---------------
--[[
  Process EventUpdate :
   Params[1] = update_time (INT)
   Params[2] = leader (STRING)
   Params[3] = ev_date (INT)
   Params[4] = ev_place (STRING)
   Params[5] = ev_comment (STRING)
   Params[6] = max_count (INT)
   Params[7] = min_lvl (INT)
   Params[8] = max_lvl (INT)
   Params[9] = classes (ARRAY)
   Params[10] = roles (ARRAY)
   Params[11] = Players (ARRAY)
   Params[12] = Sorttype (INT)
   Params[13] = Closed comment (STRING)
   Params[14] = Signup Deadline (INT)
   Params[15] = recover_time (INT)
   Params[16] = ev_type (INT)
   Params[17] = ev_subtype (INT)
   Params[18] = revision (INT)
   Params[19] = ev_duration (INT)
]]
local function _GEM3_COM_Process_EventUpdate(channel,from,stamp,ev_id,params)
  if(GEM3_PLAY_IsPlayerIgnored(params[2]))
  then
    if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
      GEM3_ChatDebug(GEM3_DEBUG_PROTOCOL,"Leader "..params[2].." of EventID "..ev_id.." is in ignore list, discarding event update");
    end
    return;
  end
  local ret = GEM3_EVT_ActionForIncomingEvent(channel,from,ev_id,params[2],params[1],params[3],params[4],params[15]);
  if(ret == GEM3_EVT_ACTION_DISCARD) -- Discard
  then
    GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"_GEM3_COM_Process_EventUpdate: DISCARDED - Channel="..channel.." From="..from.." at "..date("%c",stamp).." for "..ev_id);
    return;
  end
  local classes = _GEM3_COM_UnbuildClassesFromInt(params[9]);
  local roles = _GEM3_COM_UnbuildRolesFromInt(params[10]);
  local players = _GEM3_COM_UnbuildSubscribersFromInt(params[11]);
  if(ret == GEM3_EVT_ACTION_NEW) -- New event
  then
    GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"_GEM3_COM_Process_EventUpdate: NEW - Channel="..channel.." From="..from.." at "..date("%c",stamp).." for "..ev_id);
    GEM3_EVT_AddEvent(channel,ev_id,params[1],params[2],params[3],params[4],params[5],params[6],params[7],params[8],classes,roles,players,params[12],params[13],params[14],params[15],params[16],params[17],params[18],params[19]);
  elseif(ret == GEM3_EVT_ACTION_UPDATED) -- Update for event
  then
    GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"_GEM3_COM_Process_EventUpdate: UPDATED - Channel="..channel.." From="..from.." at "..date("%c",stamp).." for "..ev_id);
    GEM3_EVT_UpdateEvent(channel,ev_id,params[1],params[2],params[3],params[4],params[5],params[6],params[7],params[8],classes,roles,players,params[12],params[13],params[14],params[15],params[16],params[17],params[18],params[19]);
  elseif(ret == GEM3_EVT_ACTION_CRASHED) -- Re create lost event -- Leader only
  then
    GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"_GEM3_COM_Process_EventUpdate: CRASHED - Channel="..channel.." From="..from.." at "..date("%c",stamp).." for "..ev_id);
    if(GEM3_EVT_NeedsRebuildCrashedEvent(ev_id,params[1]))
    then
      GEM3_EVT_RebuildCrashedEvent(channel,ev_id,params[1],params[2],params[3],params[4],params[5],params[6],params[7],params[8],classes,roles,players,params[12],params[13],params[14],params[15],params[16],params[17],params[18],params[19]);
    end
  elseif(ret == GEM3_EVT_ACTION_RECOVER_UPDATE)
  then
    GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"_GEM3_COM_Process_EventUpdate: RECOVER UPDATE - Channel="..channel.." From="..from.." at "..date("%c",stamp).." for "..ev_id);
    if(GEM3_EVT_NeedsRebuildRecoverUpdateEvent(ev_id,params[1]))
    then
      GEM3_EVT_RebuildRecoverUpdateEvent(channel,ev_id,params[1],params[2],params[3],params[4],params[5],params[6],params[7],params[8],classes,roles,players,params[12],params[13],params[14],params[15],params[16],params[17],params[18],params[19]);
    end
  end
end

--[[
  Process CmdNewLeader :
   -- Common Cmd params
   Params[1] = leader (STRING) -- Leader of the event
   Params[2] = ev_date (INT) -- Date of the event
   Params[3] = ack (INT) -- 1 or 0, if the packet is an ack to the command or not
   Params[4] = pl_dest (STRING) -- Name of player the command is destinated to
   Params[5] = pl_src (STRING) -- Name of player the command is originating from
   Params[6] = stamp (INT) -- Date the command was first issued
]]
local function _GEM3_COM_Process_CmdNewLeader(channel,from,stamp,ev_id,params)
  local event = GEM3_QA_Events.events[ev_id];

  if(event == nil)
  then
    GEM3_ChatWarning("_GEM3_COM_Process_CmdNewLeader: Event is nil!!!");
    return;
  end
  -- Set myself as leader
  GEM3_EVT_SetMyselfNewLeader(event,params[4]);
end

--[[
  Process CmdSubscribe :
   -- Common Cmd params
   Params[1] = leader (STRING) -- Leader of the event
   Params[2] = ev_date (INT) -- Date of the event
   Params[3] = ack (INT) -- 1 or 0, if the packet is an ack to the command or not
   Params[4] = pl_dest (STRING) -- Name of player the command is destinated to
   Params[5] = pl_src (STRING) -- Name of player the command is originating from
   Params[6] = stamp (INT) -- Date the command was first issued
   -- Specific Cmd params
   Params[7] = Class (STRING) -- Class of the subscribing player
   Params[8] = Role (INT) -- Role of the subscribing player
   Params[9] = Level (INT)    -- Level of the subscribing player
   Params[10] = Guild (STRING) -- Guild of the subscribing player
   Params[11] = Comment (STRING) -- Subscribe comment string
   Params[12] = ForceQueue (INT) -- Queue to be forced into
   Params[13] = AlternativeRole (INT) -- Alternative role
   Params[14] = main_time (INT) -- Date of the main subsciption
   Params[15] = update_time (INT) -- Date of the updated subsciption
   Params[16] = source (STR) -- Subscription source
   -- If adding params, don't forget to update "GEM3_CMD_GetNonAckedSubscriptions" and "WebSync" functions
]]
local function _GEM3_COM_Process_CmdSubscribe(channel,from,stamp,ev_id,params)
  local pl_level = params[9];
  local event = GEM3_QA_Events.events[ev_id];

  if(event == nil)
  then
    GEM3_ChatWarning("_GEM3_COM_Process_CmdSubscribe: Event is nil!!!");
    return;
  end

  -- Check player's level
  if(pl_level < event.min_lvl or pl_level > event.max_lvl)
  then
    GEM3_ChatDebug(GEM3_DEBUG_PROTOCOL,"_GEM3_COM_Process_CmdSubscribe: Player "..params[5].." level range incorrect");
    GEM3_COM_SubscribeError(event,params[5],GEM3_SUB_ERROR_LEVEL_RANGE);
    return;
  end

  local result,reason = GEM3_SUB_IsPlayerBanned(ev_id,params[5]);
  if(result)
  then
    GEM3_ChatDebug(GEM3_DEBUG_PROTOCOL,"_GEM3_COM_Process_CmdSubscribe: Player "..params[5].." is banned");
    GEM3_COM_SubscribeError(event,params[5],GEM3_SUB_ERROR_BANNED);
    return;
  end

  if(params[14] > event.deadline) -- Deadline passed when he subscribed
  then
    GEM3_ChatDebug(GEM3_DEBUG_PROTOCOL,"_GEM3_COM_Process_CmdSubscribe: Player "..params[5].." subscribed after deadline");
    GEM3_COM_SubscribeError(event,params[5],GEM3_SUB_ERROR_DEADLINE);
    return;
  end

  local pl_sub = GEM3_SUB_BuildSubscriberInfos(params[5],params[10],pl_level,params[7],params[8],params[11],params[12],params[15],params[13],params[14],params[16]);
  local state,reason = GEM3_SUB_CreateSubscriber(ev_id,pl_sub);
  if(state == GEM3_SUB_STATE_NONE)
  then
    GEM3_COM_SubscribeError(event,params[5],GEM3_SUB_ERROR_UNKNOWN_EVENT); -- Leader lost the event
    return;
  elseif(state == GEM3_SUB_STATE_DISCARD)
  then
    GEM3_ChatDebug(GEM3_DEBUG_PROTOCOL,"_GEM3_COM_Process_CmdSubscribe: Discarding old subscribe infos for player "..params[5]);
    return;
  end
end

--[[
  Process CmdUnsubscribe :
   -- Common Cmd params
   Params[1] = leader (STRING) -- Leader of the event
   Params[2] = ev_date (INT) -- Date of the event
   Params[3] = ack (INT) -- 1 or 0, if the packet is an ack to the command or not
   Params[4] = pl_dest (STRING) -- Name of player the command is destinated to
   Params[5] = pl_src (STRING) -- Name of player the command is originating from
   Params[6] = stamp (INT) -- Date the command was first issued
   -- Specific Cmd params
   Params[7] = Comment (STRING) -- Unsubscribe comment string
]]
local function _GEM3_COM_Process_CmdUnsubscribe(channel,from,stamp,ev_id,params)
  local event = GEM3_QA_Events.events[ev_id];

  if(event == nil)
  then
    GEM3_ChatWarning("_GEM3_COM_Process_CmdUnsubscribe: Event is nil!!!");
    return;
  end

  GEM3_SUB_RemoveSubscriber(event,params[5],params[7]); -- We don't care for the result, player already removed himself from subscription
end

--[[
  Process CmdKick :
   -- Common Cmd params
   Params[1] = leader (STRING) -- Leader of the event
   Params[2] = ev_date (INT) -- Date of the event
   Params[3] = ack (INT) -- 1 or 0, if the packet is an ack to the command or not
   Params[4] = pl_dest (STRING) -- Name of player the command is destinated to
   Params[5] = pl_src (STRING) -- Name of player the command is originating from
   Params[6] = stamp (INT) -- Date the command was first issued
   -- Specific Cmd params
   Params[7] = Reason (STRING) -- Reason of the kick
]]
local function _GEM3_COM_Process_CmdKick(channel,from,stamp,ev_id,params)
  local event = GEM3_QA_Events.events[ev_id];

  if(event == nil)
  then
    --GEM3_ChatWarning("_GEM3_COM_Process_CmdKick: Event is nil!!!");
    return;
  end

  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"_GEM3_COM_Process_CmdKick: From="..params[5].." at "..date("%c",params[6]).." for "..ev_id..": Kicked: "..params[7]);
  end

  GEM3_SUB_SetKicked(ev_id,params[4],params[7]);
  -- GUI Callback
  GEM3_TriggerCallback("OnKickedFromEvent",ev_id,params[4],params[5],params[7]);
end


--[[
  Process CmdBan :
   -- Common Cmd params
   Params[1] = leader (STRING) -- Leader of the event
   Params[2] = ev_date (INT) -- Date of the event
   Params[3] = ack (INT) -- 1 or 0, if the packet is an ack to the command or not
   Params[4] = pl_dest (STRING) -- Name of player the command is destinated to
   Params[5] = pl_src (STRING) -- Name of player the command is originating from
   Params[6] = stamp (INT) -- Date the command was first issued
   -- Specific Cmd params
   Params[7] = Reason (STRING) -- Reason of the ban
]]
local function _GEM3_COM_Process_CmdBan(channel,from,stamp,ev_id,params)
  local event = GEM3_QA_Events.events[ev_id];

  if(event == nil)
  then
    --GEM3_ChatWarning("_GEM3_COM_Process_CmdBan: Event is nil!!!");
    return;
  end

  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"_GEM3_COM_Process_CmdBan: From="..params[5].." at "..date("%c",params[6]).." for "..ev_id..": Banned: "..params[7]);
  end

  GEM3_SUB_SetBanned(ev_id,params[4],params[7]);
  -- GUI Callback
  GEM3_TriggerCallback("OnBannedFromEvent",ev_id,params[4],params[5],params[7]);
end

--[[
  Process CmdUnBan :
   -- Common Cmd params
   Params[1] = leader (STRING) -- Leader of the event
   Params[2] = ev_date (INT) -- Date of the event
   Params[3] = ack (INT) -- 1 or 0, if the packet is an ack to the command or not
   Params[4] = pl_dest (STRING) -- Name of player the command is destinated to
   Params[5] = pl_src (STRING) -- Name of player the command is originating from
   Params[6] = stamp (INT) -- Date the command was first issued
   -- Specific Cmd params
]]
local function _GEM3_COM_Process_CmdUnBan(channel,from,stamp,ev_id,params)
  local event = GEM3_QA_Events.events[ev_id];

  if(event == nil)
  then
    --GEM3_ChatWarning("_GEM3_COM_Process_CmdUnBan: Event is nil!!!");
    return;
  end

  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"_GEM3_COM_Process_CmdUnBan: From="..params[5].." at "..date("%c",params[6]).." for "..ev_id..": UnBanned");
  end

  GEM3_SUB_SetUnBanned(ev_id,params[4]);
  -- GUI Callback
  GEM3_TriggerCallback("OnUnBannedFromEvent",ev_id,params[4],params[5]);
end

--[[
  Process ErrSubscribe :
   -- Common Cmd params
   Params[1] = leader (STRING) -- Leader of the event
   Params[2] = ev_date (INT) -- Date of the event
   Params[3] = ack (INT) -- 1 or 0, if the packet is an ack to the command or not
   Params[4] = pl_dest (STRING) -- Name of player the command is destinated to
   Params[5] = pl_src (STRING) -- Name of player the command is originating from
   Params[6] = stamp (INT) -- Date the command was first issued
   -- Specific Cmd params
   Params[7] = Code (INT) -- Error code
]]
local function _GEM3_COM_Process_ErrSubscribe(channel,from,stamp,ev_id,params)
  local code = params[7];

  GEM3_SUB_SubscriptionFailed(ev_id,params[4],code);
end

--[[
  Process PlayerInfos :
   Params[1] = name (STRING)
   Params[2] = guild (STRING)
   Params[3] = location (STRING)
   Params[4] = level (INT)
   Params[5] = class (STRING)
   Params[6] = role (INT)
   Params[7] = officer (INT)
   Params[8] = rank name (STRING)
   Params[9] = rank idx (INT)
   Params[10] = GEM-Version (STRING)
   Params[11] = Comment (STRING)
   Params[12] = IsLeader (BOOL)
   Params[13] = IsTimeRef (BOOL)
   Params[14] = InstanceTags (TABLE)
]]
local function _GEM3_COM_Process_Global_PlayerInfos(channel,from,stamp,ev_id,params)
  if(params[4] == nil)
  then
    GEM3_ChatWarning("_GEM3_COM_Process_Global_PlayerInfos : pl_level sent by '"..tostring(from).."' is nil !");
    return;
  end
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"_GEM3_COM_Process_Global_PlayerInfos: From="..from.." at "..date("%c",stamp).." Fill infos: Guild="..tostring(params[2]).." Location="..params[3].." Level="..params[4].." Class="..params[5].." isChanLeader="..tostring(params[12]).." isTimeRef="..tostring(params[13]));
  end
  local infos = GEM3_PLAY_FillPlayerInfos(channel,params[1],params[2],params[3],params[4],params[5],params[6],params[7],params[8],params[9],params[10],params[11],params[12],params[13],params[14]);
  if(infos)
  then
    -- GUI Callback
    GEM3_TriggerCallback("OnPlayerInfosUpdate",channel,infos);
  end
end

--[[
  Process IncorrectStamp :
   Params[1] = name (STRING)
   Params[2] = offset (INT)
]]
local function _GEM3_COM_Process_Global_IncorrectStamp(channel,from,stamp,ev_id,params)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"_GEM3_COM_Process_Global_IncorrectStamp: From="..from.." at "..date("%c",stamp).." Concerning: "..tostring(params[1]).." Offset="..tostring(params[2]));
  end
  if(GEM3_IsMyReroll(params[1]))
  then
    if(GEM3_STAMP_IsTimeRef()) -- But!? what? I'm a timeref too!!
    then
      GEM3_STAMP_SetTimeRef(false); -- Disable myself as a timeref
      -- GUI Callback
      GEM3_TriggerCallback("OnTimeRefConflict",channel,from,params[2]);
      GEM3_COM_DisableMyself(GEM3_LOC_DISABLED_TIMEREF_CONFLICT);
    else
      -- GUI Callback
      GEM3_TriggerCallback("OnIncorrectTimeSettings",channel,from,params[2]);
      GEM3_COM_DisableMyself(GEM3_LOC_DISABLED_INCORRECT_TIMESTAMP);
    end
    if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
      GEM3_ChatDebug(GEM3_DEBUG_WARNING,"_GEM3_COM_Process_Global_IncorrectStamp: Damned, my time settings seems incorrect, disabling myself!");
    end
  end
end

--[[
  Process CustomMessage :
   Params[1] = identifier (STRING)
   -- Custom params
]]
local function _GEM3_COM_Process_Global_CustomMessage(channel,from,stamp,ev_id,params)
  local identifier = params[1];
  local func = _GEMComm_CustomMessageDispatch[identifier];

  if(func)
  then
    tremove(params,1);
    func(channel,from,stamp,unpack(params));
  else
    if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
      GEM3_ChatDebug(GEM3_DEBUG_PROTOCOL,"_GEM3_COM_Process_Global_CustomMessage: Got an unknown custom message ("..identifier..") from "..from.." on channel "..channel);
    end
  end
end

--[[
  Process DeletedEvent :
   Params[1] = ev_id (STRING)
   Params[2] = deleted_time (INT)
]]
local function _GEM3_COM_Process_Global_DeletedEvent(channel,from,stamp,ev_id,params)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"_GEM3_COM_Process_Global_DeletedEvent: From="..from.." at "..date("%c",stamp).." Concerning: "..tostring(params[1]).." DeletedTime="..tostring(params[2]));
  end
  GEM3_EVT_NotifyDeletedEvent(params[1],params[2]);
  Chronos.unscheduleByName("GEM3DelEvt_"..params[1]); -- Unschedule mine, if it was
end

--[[
  Process ChatMessage :
   Params[1] = message (STRING)
]]
local function _GEM3_COM_Process_Global_ChatMessage(channel,from,stamp,ev_id,params)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"_GEM3_COM_Process_Global_ChatMessage: From="..from.." at "..date("%c",stamp).." Message: "..tostring(params[1]));
  end
  GEM3_CHAN_DisplayChatMessage(channel,from,stamp,params[1]);
end

--[[
  Process CmdAssistant :
   -- Common Cmd params
   Params[1] = leader (STRING) -- Leader of the event
   Params[2] = ev_date (INT) -- Date of the event
   Params[3] = ack (INT) -- 1 or 0, if the packet is an ack to the command or not
   Params[4] = pl_dest (STRING) -- Name of player the command is destinated to
   Params[5] = pl_src (STRING) -- Name of player the command is originating from
   Params[6] = stamp (INT) -- Date the command was first issued
   -- Specific Cmd params
]]
local function _GEM3_COM_Process_CmdAssistant(channel,from,stamp,ev_id,params)
  local event = GEM3_QA_Events.events[ev_id];

  if(event == nil)
  then
    --GEM3_ChatWarning("_GEM3_COM_Process_CmdAssistant: Event is nil!!!");
    return;
  end

  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_ChatDebug(GEM3_DEBUG_COMMANDS,"_GEM3_COM_Process_CmdAssistant: From="..from.." at "..date("%c",stamp).." for "..ev_id.." : From "..params[5].." to "..params[4]);
  end

  GEM3_EVT_SetEventAssistant(ev_id,params[4]); -- Set myself assistant
end

--[[
  Process CmdCustom :
   -- Common Cmd params
   Params[1] = leader (STRING) -- Leader of the event
   Params[2] = ev_date (INT) -- Date of the event
   Params[3] = ack (INT) -- 1 or 0, if the packet is an ack to the command or not
   Params[4] = pl_dest (STRING) -- Name of player the command is destinated to
   Params[5] = pl_src (STRING) -- Name of player the command is originating from
   Params[6] = stamp (INT) -- Date the command was first issued
   -- Specific Cmd params
   Params[7] = identifier (STRING)
   ... -- Custom params
]]
local function _GEM3_COM_Process_CmdCustom(channel,from,stamp,ev_id,params)
  local event = GEM3_QA_Events.events[ev_id];

  if(event == nil)
  then
    return;
  end

  local identifier = params[7];
  local func = _GEMComm_CustomCommandDispatch[identifier];

  if(func)
  then
    local pl_src = params[5];
    local pl_stamp = params[6];
    params = GEM3_DuplicateTable(params);
    for i=1,7 -- Remove all non-custom params
    do
      tremove(params,1);
    end
    func(channel,pl_src,pl_stamp,ev_id,unpack(params));
  else
    if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
      GEM3_ChatDebug(GEM3_DEBUG_PROTOCOL,"_GEM3_COM_Process_CmdCustom: Got an unknown custom message ("..identifier..") from "..from.." on channel "..channel);
    end
  end
end

--------------- Exported functions ---------------


function GEM3_COM_InitRoutines()
  -- Schedule routine
  if(_GEM3_COM_RoutineScheduled == false)
  then
    Chronos.schedule(10,_GEM3_COM_ScheduledPlayerInfos);
    _GEM3_COM_RoutineScheduled = true;
  end

end

--[[
 function GEM3_COM_BroadcastEvent :
  Broadcast an event
   event : Array -- The event struct to broadcast
]]
function GEM3_COM_BroadcastEvent(event)
  if(event == nil) then return; end;

  local classes = _GEM3_COM_BuildClassesToInt(event.classes);
  local roles = _GEM3_COM_BuildRolesToInt(event.roles);
  local players = _GEM3_COM_BuildSubscribersToInt(event);
  _GEM3_COM_SendNormalMessage(event.channel,GEM3_CMD_EVENT_UPDATE,event.id,event.update_time,event.leader,event.ev_date,event.ev_place,event.ev_comment,event.max_count,event.min_lvl,event.max_lvl,classes,roles,players,event.sorttype,event.closed_comment,event.deadline,event.recover_time,event.ev_type,event.ev_subtype,event.revision,event.ev_duration);
  GEM3_QUE_RemoveEventFromQueue(event.channel,event.id); -- Remove event from the queue
end

--[[
 function GEM3_COM_SendCommand :
  Send a command
   pl_cmd : GEM3_PlayerCommand struct
]]
function GEM3_COM_SendCommand(pl_cmd)
  _GEM3_COM_RealSendMessage(pl_cmd.channel,pl_cmd.cmd,pl_cmd.id,pl_cmd.params);
end

--[[
 function GEM3_COM_SendForwardCommand :
  Forward the command to the new leader
]]
function GEM3_COM_SendForwardCommand(channel,cmd,ev_id,params)
  _GEM3_COM_RealSendMessage(channel,cmd,ev_id,params);
end

--[[
 function GEM3_COM_BroadcastCommand
  Broadcast an ACK for a cmd
   ev_id  : String -- Event ID
   cmd_id : String -- Cmd ID to send
]]
function GEM3_COM_BroadcastCommand(ev_id,cmd_id)
--~   local cmds = GEM3_QA_Events.commands[ev_id];
--~   local cmdtab = cmds.cmds[cmd_id];

--~   if(cmdtab == nil)
--~   then
--~     GEM3_ChatWarning("GEM3_COM_BroadcastCommand : Failed to find cmd_id '"..cmd_id.."'");
--~     return;
--~   end
--~   if(not GEM3_CMD_IsCommandAcked(ev_id,cmd_id))
--~   then
--~     _GEM3_COM_SendNormalMessage(cmds.channel,cmdtab.cmd,ev_id,unpack(cmdtab.params));
--~   end
end


--------------- Exported functions ---------------

local function _GEM3_COM_NotifyEventUpdate_Schedule(ev_id) -- Scheduled CB function
  local event = GEM3_QA_Events.events[ev_id];

  if(event) -- Still valid
  then
    GEM3_COM_BroadcastEvent(event);
  end
end

local function _GEM3_COM_NotifyDeletedEvent_Schedule(channel,ev_id,deleted_time) -- Scheduled CB function
  _GEM3_COM_SendNormalMessage(channel,GEM3_CMD_DELETED_EVENT,GEM3_GLOBAL_CMD,ev_id,deleted_time);
end

function GEM3_COM_SendChatMessage(channel,message)
  _GEM3_COM_SendNormalMessage(channel,GEM3_CMD_CHAT_MESSAGE,GEM3_GLOBAL_CMD,message);
end

--[[
 function GEM3_NotifyEventUpdate :
  Notify all players of an update in the event (called only by leader of ev_id).
   ev_id : String -- Event ID to notify an update from
]]
function GEM3_COM_NotifyEventUpdate(ev_id)
  local event = GEM3_QA_Events.events[ev_id];
  
  if(event and GEM3_IsMyReroll(event.leader))
  then
    Chronos.scheduleByName("GEM3Notif_"..ev_id,5,_GEM3_COM_NotifyEventUpdate_Schedule,ev_id); -- Schedule (or reschedule, if existing) in 5sec
  end
end

--[[
 function GEM3_COM_Subscribe :
  Subscribes to an event and broadcast it (Sent by player).
   Event : Array (GEM3_Event struct) -- The Event
   infos : Array (GEM3_Subscription struct) -- Infos to broadcast
]]
function GEM3_COM_Subscribe(event,infos)
  if(event)
  then
    _GEM3_COM_PrepareAndSendCommand(event,GEM3_CMD_CMD_SUBSCRIBE,event.leader,infos.name, -- Header (event,cmd,pl_dest,pl_src)
      infos.class,infos.role,infos.level,infos.guild,infos.comment,infos.force_queue,infos.alt_role,infos.main_time,infos.update_time,infos.source); -- Specifics
  end
end

--[[
 function GEM3_COM_Unsubscribe :
  Unsubscribes from an event and broadcast it (Sent by player, once).
   Event   : Array -- The Event
   pl_name : String -- Name of the toon to unsubscribe
   comment : String -- Comment to send with unsubscription
]]
function GEM3_COM_Unsubscribe(event,pl_name,comment)
  if(event)
  then
    _GEM3_COM_PrepareAndSendCommand(event,GEM3_CMD_CMD_UNSUBSCRIBE,event.leader,pl_name, -- Header (event,cmd,pl_dest,pl_src)
      comment); -- Specifics
  end
end

--[[
 function GEM3_COM_KickPlayer :
  Kicks a player from an event I'm the leader.
   Event : Array -- The Event
   pl_name : String -- Player to promote assistant
   reason  : String -- Why you kick him
]]
function GEM3_COM_KickPlayer(event,pl_name,reason)
  _GEM3_COM_PrepareAndSendCommand(event,GEM3_CMD_CMD_KICK,pl_name,event.leader, -- Header (event,cmd,pl_dest,pl_src)
    reason); -- Specifics
end

--[[
 function GEM3_COM_BanPlayer :
  Bans a player from an event I'm the leader.
   Event : Array -- The Event
   pl_name : String -- Player to ban from event
   reason  : String -- Why you ban him
]]
function GEM3_COM_BanPlayer(event,pl_name,reason)
  _GEM3_COM_PrepareAndSendCommand(event,GEM3_CMD_CMD_BAN,pl_name,event.leader, -- Header (event,cmd,pl_dest,pl_src)
    reason); -- Specifics
end

--[[
 function GEM3_COM_UnBanPlayer :
  UnBans a player from an event I'm the leader.
   Event : Array -- The Event
   pl_name : String -- Player to unban from event
]]
function GEM3_COM_UnBanPlayer(event,pl_name)
  _GEM3_COM_PrepareAndSendCommand(event,GEM3_CMD_CMD_UNBAN,pl_name,event.leader); -- Header (event,cmd,pl_dest,pl_src)
end

--[[
 function GEM3_COM_SubscribeError :
  Error subscribing player (Sent by leader).
   Event : Array -- The Event
   pl_name: String -- Player I cannot subscribe
   code: Int -- Error code (one of GEM3_SUB_ERROR_xx)
]]
function GEM3_COM_SubscribeError(event,pl_name,code)
  if(event)
  then
    GEM3_ChatDebug(GEM3_DEBUG_PROTOCOL,"GEM3_COM_SubscribeError: Sending Subscription error code for player "..pl_name..": "..code);
    _GEM3_COM_PrepareAndSendCommand(event,GEM3_CMD_ERR_SUBSCRIBE,pl_name,event.leader, -- Header (event,cmd,pl_dest,pl_src)
      code); -- Specifics
  end
end

--[[
 function GEM3_COM_PlayerInfos :
  Sends my player informations to all channels.
]]
function GEM3_COM_PlayerInfos()
  for channel,chantab in pairs(GEM3_RT_Channels)
  do
    if(chantab.id ~= 0)
    then
      _GEM3_COM_LastPlayerInfosSend = time();
      _GEM3_COM_SendNormalMessage(channel,GEM3_CMD_PLAYER_INFOS,GEM3_GLOBAL_CMD,GEM3_PLAY_FillMyPlayerInfos(channel));
    end
  end
end

--[[
 function GEM3_COM_PlayerInfosSingle :
  Sends my player informations to a single channel.
]]
function GEM3_COM_PlayerInfosSingle(channel)
  _GEM3_COM_LastPlayerInfosSend = time();
  _GEM3_COM_SendNormalMessage(channel,GEM3_CMD_PLAYER_INFOS,GEM3_GLOBAL_CMD,GEM3_PLAY_FillMyPlayerInfos(channel));
end

--[[
 function GEM3_COM_AddAssistant :
  Promotes a player assistant of an event I'm the leader.
   Event : Array -- The Event
   pl_name : String -- Player to promote assistant
]]
function GEM3_COM_AddAssistant(event,pl_name)
  _GEM3_COM_PrepareAndSendCommand(event,GEM3_CMD_CMD_ASSISTANT,pl_name,event.leader); -- Header (event,cmd,pl_dest,pl_src)
end

--[[
 function GEM3_COM_IncorrectTimeStamp:
  Inform a user he has an incorrect time stamp
   channel : String -- Channel
   pl_name : String -- Player with incorrect stamp
   offset: Int -- Error Offset
]]
function GEM3_COM_IncorrectTimeStamp(channel,pl_name,offset)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_ChatDebug(GEM3_DEBUG_PROTOCOL,"GEM3_COM_IncorrectTimeStamp: Notifying player "..pl_name.." for his incorrect stamp");
  end
  _GEM3_COM_SendNormalMessage(channel,GEM3_CMD_INCORRECT_STAMP,GEM3_GLOBAL_CMD,pl_name,offset);
end

--[[
 function GEM3_COM_DeletedEvent:
  Inform a user he has an incorrect time stamp
   channel : String -- Channel
   ev_id : String -- EventID
   deleted_time: Int -- Time the event was deleted
]]
function GEM3_COM_DeletedEvent(channel,ev_id,deleted_time)
  if(GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_ChatDebug(GEM3_DEBUG_PROTOCOL,"GEM3_COM_DeletedEvent: Scheduling deleted event notification for EventID "..ev_id.." at "..deleted_time);
  end
  Chronos.scheduleByName("GEM3DelEvt_"..ev_id,GEM3_QA_Config.my_bcast_offset,_GEM3_COM_NotifyDeletedEvent_Schedule,channel,ev_id,deleted_time); -- Schedule (or reschedule, if existing)
end

--[[
 function GEM3_COM_ChangeLeader :
  Subscribes to an event and broadcast it (Sent by player).
   ev_id: String -- Event ID
   new_leader: String -- The new leader
  Return:
   - Bool: Result
   - String: Error message
]]
function GEM3_COM_ChangeLeader(ev_id,new_leader)
  local event = GEM3_QA_Events.events[ev_id];

  if(event == nil)
  then
    return false,"Unknown event";
  end

  if(GEM3_IsMyReroll(new_leader))
  then
    return false,"Cannot set one of my reroll as new event leader";
  end

  if(not GEM3_IsMyReroll(event.leader))
  then
    return false,"I'm not the leader of this event";
  end

  if(not GEM3_PLAY_IsPlayerInChannel(event.channel,new_leader))
  then
    return false,"New leader is not online";
  end
  
  GEM3_ChatDebug(GEM3_DEBUG_PROTOCOL,"GEM3_COM_ChangeLeader: Player "..new_leader.." is the new leader of EventID "..ev_id);
  _GEM3_COM_PrepareAndSendCommand(event,GEM3_CMD_CMD_NEW_LEADER,new_leader,event.leader); -- Header (event,cmd,pl_dest,pl_src)
  GEM3_QA_Events.forward[ev_id] = new_leader;
  event.leader = new_leader; -- Change leader's name to block API functions
  return true;
end

function GEM3_COM_DisableMyself(reason)
  _GEM3_COM_DisableMyself = true; -- Fully disable myself
  _GEM3_COM_DisableMyselfReason = reason;
end


--------------- Exported API  ---------------

function GEM3_COM_RegisterCustomMessage(identifier,func)
  if(_GEMComm_CustomMessageDispatch[identifier] ~= nil)
  then
    return false;
  end
  _GEMComm_CustomMessageDispatch[identifier] = func;
  return true;
end

function GEM3_COM_RegisterCustomCommand(identifier,func)
  if(_GEMComm_CustomCommandDispatch[identifier] ~= nil)
  then
    return false;
  end
  _GEMComm_CustomCommandDispatch[identifier] = func;
  return true;
end

function GEM3_COM_SendCustomMessage(channel,identifier,...)
  if(_GEMComm_CustomMessageDispatch[identifier] == nil)
  then
    return false;
  end
  _GEM3_COM_SendNormalMessage(channel,GEM3_CMD_CUSTOM_MSG,GEM3_GLOBAL_CMD,identifier,...);
  return true;
end

function GEM3_COM_SendCustomCommand(channel,identifier,ev_id,pl_dest,...)
  local event = GEM3_QA_Events.events[ev_id];

  if(event == nil)
  then
    return false;
  end
  if(_GEMComm_CustomCommandDispatch[identifier] == nil)
  then
    return false;
  end
  _GEM3_COM_PrepareAndSendCommand(event,GEM3_CMD_CMD_CUSTOM,pl_dest,GEM3_PlayerName, -- Header (event,cmd,pl_dest,pl_src)
    identifier,...); -- Specifics
  return true;
end

function GEM3_COM_IsDisabled()
  return _GEM3_COM_DisableMyself,_GEM3_COM_DisableMyselfReason;
end

function GEM3_COM_Disable(disabled,reason)
  if(disabled)
  then
    _GEM3_COM_DisableMyself = true;
    _GEM3_COM_DisableMyselfReason = reason;
  else
    _GEM3_COM_DisableMyself = false;
    _GEM3_COM_DisableMyselfReason = nil;
  end
end


--------------- Init dispatch table ---------------
_GEMComm_Dispatch[GEM3_CMD_EVENT_UPDATE] = _GEM3_COM_Process_EventUpdate;
_GEMComm_Dispatch[GEM3_CMD_PLAYER_INFOS] = _GEM3_COM_Process_Global_PlayerInfos;
_GEMComm_Dispatch[GEM3_CMD_INCORRECT_STAMP] = _GEM3_COM_Process_Global_IncorrectStamp;
_GEMComm_Dispatch[GEM3_CMD_CUSTOM_MSG] = _GEM3_COM_Process_Global_CustomMessage;
_GEMComm_Dispatch[GEM3_CMD_DELETED_EVENT] = _GEM3_COM_Process_Global_DeletedEvent;
_GEMComm_Dispatch[GEM3_CMD_CHAT_MESSAGE] = _GEM3_COM_Process_Global_ChatMessage;
_GEMComm_Dispatch[GEM3_CMD_CMD_SUBSCRIBE] = _GEM3_COM_Process_CmdSubscribe;
_GEMComm_Dispatch[GEM3_CMD_CMD_UNSUBSCRIBE] = _GEM3_COM_Process_CmdUnsubscribe;
_GEMComm_Dispatch[GEM3_CMD_CMD_KICK] = _GEM3_COM_Process_CmdKick;
_GEMComm_Dispatch[GEM3_CMD_CMD_BAN] = _GEM3_COM_Process_CmdBan;
_GEMComm_Dispatch[GEM3_CMD_CMD_UNBAN] = _GEM3_COM_Process_CmdUnBan;
_GEMComm_Dispatch[GEM3_CMD_ERR_SUBSCRIBE] = _GEM3_COM_Process_ErrSubscribe;
_GEMComm_Dispatch[GEM3_CMD_CMD_ASSISTANT] = _GEM3_COM_Process_CmdAssistant;
_GEMComm_Dispatch[GEM3_CMD_CMD_NEW_LEADER] = _GEM3_COM_Process_CmdNewLeader;
_GEMComm_Dispatch[GEM3_CMD_CMD_CUSTOM] = _GEM3_COM_Process_CmdCustom;
