--[[
  Guild Event Manager (GEM) - Version 3
  by Kiki from European "Conseil des Ombres" (Horde)
  ----------------------------------------
  Channels handling
]]

--[[
  Exported API Functions:

   GEM3_CHAN_AddChannel(name,pwd,alias,slash)
    Input:
     name:  string -- Name of the channel to add (automatically lowercased). If name is the GUILD constant, it will add the special guild channel
     pwd:   string -- Password to access the channel (not used for GUILD channel)
     alias: string -- Alias to be displayed (not used for GUILD channel)
     slash: string -- Slash command to talk on the channel (not used for GUILD channel)
    Output:
     result: bool  -- True if ok, false if error
    Events:
     "OnAddChannel"(chanstruct) is called on successful AddChannel (check chanstruct.saved.name for lowercased added name)
     "OnAddChannelFailed"(name,message) is called on error
    Purpose:
     Adds a new GEM channel

   GEM3_CHAN_DelChannel(name)
    Input:
     name: string -- Name of the channel to remove (lowercased). If name is the GUILD constant, it will remove the special guild channel
    Output:
     None
    Events:
     "OnDelChannel"(name) is called on successful DelChannel
    Purpose:
     Removes an existing GEM channel

   GEM3_CHAN_UpdateChannel(name,pwd,alias,slash)
    Input:
     name:  string -- Name of the channel to update (automatically lowercased). If name is the GUILD constant it won't do anything, because there is nothing to edit to this channel
     pwd:   string -- Password to access the channel
     alias: string -- Alias to be displayed
     slash: string -- Slash command to talk on the channel
    Output:
     None
    Events:
     "OnUpdateChannel"(chanstruct) is called on successful UpdateChannel
     "OnUpdateChannelFailed"(name,message) is called on error
    Purpose:
     Updates an existing channel (password and/or alias-slash)

   GEM3_CHAN_IsChannelActive(name)
    Input:
     name:  string -- Name of the channel to check (real name, not GUILD constant if checking special guild channel)
    Output:
     active: bool -- True if channel is a joined GEM channel for this character
    Purpose:
     Checks if a channel is a GEM channel, and initialized

   GEM3_CHAN_IsChannelConfiguredForAnyReroll(channel)
    Input:
     channel:  string -- Name of the channel to check (real name, not GUILD constant if checking special guild channel)
    Output:
     thisReroll: bool -- True if current player has specified channel configured (anyReroll is true if thisReroll is true)
     anyReroll: bool -- True if any of my rerolls has specified channel configured
    Purpose:
     Checks if any of my rerolls has specified channel in its config

   GEM3_CHAN_GetChannelID(name)
    Input:
     name:  string -- Name of the channel to check (real name, not GUILD constant if checking special guild channel)
    Output:
     chanid: int -- Internal ID used by WoW for channels (0 if not joined, -1 if special guild channel, >0 otherwise)
    Purpose:
     Gets a channel's internal ID (0 if not joined, -1 if special guild channel)

   GEM3_CHAN_GetChannelInfos(name)
    Input:
     name:  string -- Name of the channel to get infos (real name, not GUILD constant if checking special guild channel)
    Output:
     infos: array -- Saved channel struct (don't modify!!)
       infos.name     : string -- Real channel name
       infos.password : string -- Channel password (or "" if none)
       infos.alias    : string -- Alias for channel (or "" if none)
       infos.slash    : string -- Slash cmd to speak in channel (or "" if none)
       infos.guild    : bool   -- If channel is special guild channel
    Purpose:
     Gets channel's infos


]]

--------------- Quick access variables ---------------

GEM3_QA_Channels = nil; -- Quick access: channels for this character


--------------- Shared variables ---------------

GEM3_RT_Channels = {}; -- Run time channels
GEM3_CHAN_Frame = nil;


--------------- Local variables ---------------

local _GEM3_chan_auto_set_myself_leader_timer = 0;
-- Optims vars (get a local copy)
local gmatch = string.gmatch;
local tonumber = tonumber;
local strbyte = string.byte;
local strsub = string.sub;
local strfind = string.find;
local strupper = string.upper;
local strgsub = string.gsub;

--------------- Internal functions ---------------

local function _GEM3_CHAN_YouAreLeader(channel)
  if(not GEM3_CHAN_AmILeader(channel)) -- Not already channel leader
  then
    local chanstruct = GEM3_RT_Channels[channel];
    chanstruct.leader = true; -- You are the leader
    GEM3_ChatDebug(GEM3_DEBUG_CHANNEL,"I'm the new leader on channel '"..channel.."'.");
    GEM3_COM_PlayerInfosSingle(channel);
    -- Check for broadcast on this channel
    GEM3_QUE_BuildBroadcastQueues(channel,nil);
  end
end

local function _GEM3_CHAN_ScheduleGuildLeaderSearch()
  _GEM3_chan_auto_set_myself_leader_timer = time();
  GEM3_ChatDebug(GEM3_DEBUG_CHANNEL,"_GEM3_CHAN_ScheduleGuildLeaderSearch: Guild leader still not found, setting myself as leader");
  _GEM3_CHAN_YouAreLeader(GEM3_GuildChannelName);
  GEM3_COM_PlayerInfosSingle(GEM3_GuildChannelName);
end

local function _GEM3_CHAN_ScheduleGuildRoster()
  GEM3_ChatDebug(GEM3_DEBUG_CHANNEL,"_GEM3_CHAN_ScheduleGuildRoster: Request GuildRoster update for Guild channel: "..tostring(GEM3_GuildChannelName));
  GuildRoster(); -- Request a Guild members update
end

local function _GEM3_CHAN_CheckGuildMembers()
  if(GEM3_GuildChannelName == nil)
  then
    return;
  end

  GEM3_ChatDebug(GEM3_DEBUG_CHANNEL,"_GEM3_CHAN_CheckGuildMembers: Checking for Join/Leave in the Guild channel: "..tostring(GEM3_GuildChannelName));

  local count = GetNumGuildMembers(true);
  local new_connected = {};
  local connected_count = 0;

  for i=1,count
  do
    local name, _, _, _, _, _, _, _, online = GetGuildRosterInfo(i);
    if(name and online)
    then
      connected_count = connected_count + 1;
      new_connected[name] = true;
      --GEM3_PLAY_PlayerJoined(GEM3_GuildChannelName,name); <- Player will send his info, and then I'll consider him as connected
    end
  end

  if(connected_count == 1 and GEM3_CHAN_AmILeader(GEM3_GuildChannelName) == false) -- Alone in the channel
  then
    GEM3_ChatDebug(GEM3_DEBUG_CHANNEL,"Seems like I'm alone in the Guild, setting myself as channel leader");
    _GEM3_CHAN_YouAreLeader(GEM3_GuildChannelName);
    Chronos.unscheduleByName("GEM3GuildLeaderSearch"); -- Unschedule search
  end

  -- Check for disconnected players
  for n,infos in pairs(GEM3_QA_Players[GEM3_GuildChannelName])
  do
    if(infos.connected and new_connected[n] == nil) -- No longer connected
    then
      if(infos.isLeader and connected_count ~= 1) -- Leader left and I'm not alone
      then
        GEM3_ChatDebug(GEM3_DEBUG_CHANNEL,"GUILD leader ("..n..") left the channel, searching a new leader");
        Chronos.scheduleByName("GEM3GuildLeaderSearch",random(2,GEM3_PLAY_UPDATE_INFOS_DELAY+15),_GEM3_CHAN_ScheduleGuildLeaderSearch); -- Schedule randomly btw 2 and GEM3_PLAY_UPDATE_INFOS_DELAY+15 sec
      end
      GEM3_PLAY_PlayerLeft(GEM3_GuildChannelName,n);
    end
  end

  Chronos.scheduleByName("GEM3GuildRoster",30,_GEM3_CHAN_ScheduleGuildRoster); -- Reschedule in 30sec
end

local function _GEM3_CHAN_OnEvent(self,event,...)
  if(event == "GUILD_ROSTER_UPDATE")
  then
    _GEM3_CHAN_CheckGuildMembers();
  end
end

local function _GEM3_CHAN_ChatFilter(msg)
  if(strfind(msg,"^GEM3"))
  then
    return true;
  end
end
ChatFrame_AddMessageEventFilter("CHAT_MSG_CHANNEL",_GEM3_CHAN_ChatFilter);

local function _GEM3_CHAN_AddRTChannel(chanstruct)
  local newchan = {};
  newchan.id = 0;
  newchan.retries = 0;
  newchan.leader = false;
  newchan.saved = chanstruct;
  GEM3_RT_Channels[chanstruct.name] = newchan;
  
  GEM3_PLAY_ResetJoinedPlayers(chanstruct.name);
  GEM3_PLAY_CheckExpiredPlayers(chanstruct.name);

  if(chanstruct.guild)
  then
    newchan.id = 1; -- Set to 'joined' right now, since it will be right now (and no notice)
    GEM3_CHAN_Frame:RegisterEvent("GUILD_ROSTER_UPDATE");
    GEM3_Core_Addon:RegisterComm(GEM3_COMM_PREFIX,_GEM3_COMM_OnMessageGUILD);
    Chronos.scheduleByName("GEM3GuildRoster",11,_GEM3_CHAN_ScheduleGuildRoster); -- Schedule in 15sec
    Chronos.scheduleByName("GEM3GuildLeaderSearch",GEM3_PLAY_UPDATE_INFOS_DELAY+15,_GEM3_CHAN_ScheduleGuildLeaderSearch); -- Schedule in GEM3_PLAY_UPDATE_INFOS_DELAY+15 sec
    -- DEBUG
    local myinfos = GEM3_PLAY_GetPlayerInfos(GEM3_PlayerName,chanstruct.name);
    if(myinfos == nil)
    then
      error("GUILD CHAN INIT, Failed to find my struct for channel "..tostring(chanstruct.name));
    else
      if(myinfos.connected ~= false)
      then
        error("GUILD CHAN INIT, my connected status is not FALSE : "..tostring(myinfos.connected));
      end
    end
    -- END DEBUG
    GEM3_PLAY_PlayerJoined(chanstruct.name,GEM3_PlayerName);
    GEM3_COM_PlayerInfosSingle(chanstruct.name);
    GEM3_ChatDebug(GEM3_DEBUG_CHANNEL,"Channel "..chanstruct.name.." joined, schedule a broadcast of my data");
    GEM3_QUE_BuildBroadcastQueues(chanstruct.name,nil); -- Schedule a broadcast for my data
    GEM3_TriggerCallback("OnChannelJoined",chanstruct.name);
  else
    local com = CustomChatComm.RegisterComm(GEM3_COMM_PREFIX,chanstruct.name,chanstruct.password,GEM3_COM_OnReceiveMessage,GEM3_CHAN_Process_YouJoined,GEM3_CHAN_Process_YouLeft);
    CustomChatComm.RegisterCallback(com,"CHAT_MSG_CHANNEL_NOTICE_USER",GEM3_CHAN_ProcessChannelNoticeUser);
    CustomChatComm.RegisterCallback(com,"CHAT_MSG_CHANNEL_LIST",GEM3_CHAN_ProcessChannelListing);
    CustomChatComm.RegisterCallback(com,"CHAT_MSG_CHANNEL_LEAVE",GEM3_CHAN_ProcessChannelLeave);
  end
  return newchan;
end

local function _GEM3_CHAN_DelRTChannel(name)
  local rt = GEM3_RT_Channels[name];
  
  if(rt)
  then
    GEM3_PLAY_ResetJoinedPlayers(name);
    if(rt.saved.guild)
    then
      GEM3_CHAN_Frame:UnregisterEvent("GUILD_ROSTER_UPDATE");
      Chronos.unscheduleByName("GEM3GuildRoster"); -- Unschedule
      Chronos.unscheduleByName("GEM3GuildLeaderSearch");
      GEM3_Core_Addon:UnregisterComm(GEM3_COMM_PREFIX); -- Not sure about function prototype, but seems to work
      rt.id = 0; -- Set to 'leaved' right now, since it will be right now (and no notice)
    else
      -- TODO: unalias/unslash channel
      CustomChatComm.UnregisterComm(GEM3_COMM_PREFIX,name);
    end
    GEM3_RT_Channels[name] = nil;
  end
end

local function _GEM3_CHAN_AddConfigChannel(name,password,alias,slash,guild)
  local newchan = {};
  newchan.name = name;
  newchan.password = password;
  newchan.alias = alias;
  newchan.slash = slash;
  newchan.guild = guild;

  tinsert(GEM3_QA_Channels,newchan);
  
  -- Create Players struct
  if(GEM3_QA_Players[name] == nil) -- First time in this channel
  then
    GEM3_QA_Players[name] = GA_GetTable();
    GEM3_ChatDebug(GEM3_DEBUG_CHANNEL,"GEM3_CHAN_AddChannel : First time in channel "..name.." for Realm "..GEM3_Realm..", creating GEM3_Players struct for this channel");
  end
  GEM3_ChatDebug(GEM3_DEBUG_CHANNEL,"GEM3_CHAN_AddChannel : Added new channel : "..name);
  
  return newchan;
end

local function _GEM3_CHAN_DelConfigChannel(name)
  for i,chanstruct in ipairs(GEM3_QA_Channels)
  do
    if(chanstruct.name == name)
    then
      table.remove(GEM3_QA_Channels,i);
      return;
    end
  end
  GEM3_QA_Players[name] = GA_ReleaseTable(GEM3_QA_Players[name],10); -- 10 recursion count to ensure full table release
end

local function _GEM3_CHAN_LoadDefaultChannels()
  GEM3_QA_Channels = {};
  
  for realm,channels in pairs(GEM3_Defaults["channels"])
  do
    if(realm == "*" or realm == GEM3_Realm) -- All realms, or this specific realm
    then
      for name,channel in pairs(channels)
      do
        if(channel.guild)
        then
          name = GEM3_GuildChannelName; -- GEM3_GuildChannelName not yet defined
        end
        if(name) -- Trap the case where GUILD channel requested, but player has no guild
        then
          _GEM3_CHAN_AddConfigChannel(strlower(name),channel.password,channel.alias,channel.slash,channel.guild);
          GEM3_PLAY_FillMyPlayerInfos(strlower(name)); -- Init player's infos
        end
      end
    end
  end
end

--[[
   GEM3_CHAN_LoadChannels(pl_name)
    Input:
     pl_name: string -- Name of the character to load channels
    Output:
     None
    Events:
     "OnChannelsConfigLoaded"() is called when all channels are loaded from config
    Purpose:
     Loads channels configuration
]]
function GEM3_CHAN_LoadChannelsConfig(pl_name)
  local newchan;

  GEM3_RT_Channels = {};

  if(GEM3_Players[GEM3_Realm] == nil) -- First time in this realm
  then
    GEM3_Players[GEM3_Realm] = {};
    GEM3_ChatDebug(GEM3_DEBUG_CHANNEL,"GEM3_CHAN_LoadChannelsConfig : First time in Realm "..GEM3_Realm..", creating GEM3_Players struct");
  end
  GEM3_QA_Players = GEM3_Players[GEM3_Realm]; -- Setup quick access

  GEM3_QA_Channels = GEM3_Config[GEM3_Realm][pl_name].channels;
  if(GEM3_QA_Channels == nil) -- Never init channels
  then
    GEM3_ChatDebug(GEM3_DEBUG_CHANNEL,"GEM3_CHAN_LoadChannelsConfig : First time in Realm "..GEM3_Realm..", loading channel defaults");
    _GEM3_CHAN_LoadDefaultChannels();
    GEM3_Config[GEM3_Realm][pl_name].channels = GEM3_QA_Channels;
  end

  for i,chantab in ipairs(GEM3_QA_Channels)
  do
    _GEM3_CHAN_AddRTChannel(chantab);
  end
  
  -- GUI Callback
  GEM3_TriggerCallback("OnChannelsConfigLoaded");
end

local function _GEM3_SlashCommand(index,message)
  local _,channel = GetChannelName(index);
  if(channel and message and message ~= "")
  then
    GEM3_COM_SendChatMessage(channel,strgsub(message,"|","\127p"));
  end
end

local function _GEM3_SlashCommand1(message) _GEM3_SlashCommand(1,message) end
local function _GEM3_SlashCommand2(message) _GEM3_SlashCommand(2,message) end
local function _GEM3_SlashCommand3(message) _GEM3_SlashCommand(3,message) end
local function _GEM3_SlashCommand4(message) _GEM3_SlashCommand(4,message) end
local function _GEM3_SlashCommand5(message) _GEM3_SlashCommand(5,message) end
local function _GEM3_SlashCommand6(message) _GEM3_SlashCommand(6,message) end
local function _GEM3_SlashCommand7(message) _GEM3_SlashCommand(7,message) end
local function _GEM3_SlashCommand8(message) _GEM3_SlashCommand(8,message) end
local function _GEM3_SlashCommand9(message) _GEM3_SlashCommand(9,message) end
local _GEM3_SlashCommandArray = { _GEM3_SlashCommand1, _GEM3_SlashCommand2, _GEM3_SlashCommand3, _GEM3_SlashCommand4, _GEM3_SlashCommand5, _GEM3_SlashCommand6, _GEM3_SlashCommand7, _GEM3_SlashCommand8, _GEM3_SlashCommand9 };

local function _GEM3_CHAN_InitSlash(rt)
  local slash = rt.saved.slash;
  local upper = strupper(slash);

  SlashCmdList[upper] = _GEM3_SlashCommandArray[rt.id];
  setglobal("SLASH_"..upper.."1", "/"..slash);
  ChatTypeInfo["CHANNEL"..rt.id].sticky = 1;
end

local function _GEM3_CHAN_UninitSlash(rt)
  local slash = rt.saved.slash;
  local upper = strupper(slash);

  SlashCmdList[upper] = nil;
  setglobal("SLASH_"..upper.."1", nil);
end


--------------- Exported functions ---------------

function GEM3_CHAN_FoundGuildChannelLeader(name)
  local infos = GEM3_RT_Channels[GEM3_GuildChannelName];

  if(infos == nil)
  then
    GEM3_ChatWarning("GEM3_CHAN_FoundGuildChannelLeader: Not in GUILD channel!!");
    return;
  end
  if((_GEM3_chan_auto_set_myself_leader_timer+1.0) > time()) -- Too soon, maybe at the same time than me
  then
    GEM3_ChatDebug(GEM3_DEBUG_CHANNEL,"Found Guild channel leader: "..name.." ("..tostring(infos.leader).."), but too soon, I just set myself leader "..time()-_GEM3_chan_auto_set_myself_leader_timer.." sec before");
  else
    GEM3_ChatDebug(GEM3_DEBUG_CHANNEL,"Found Guild channel leader: "..name.." ("..tostring(infos.leader)..")");
    infos.leader = false; -- Someone else is the leader
  end
  Chronos.unscheduleByName("GEM3GuildLeaderSearch"); -- Unschedule search timeout
end

function GEM3_CHAN_AmILeader(name)
  if(GEM3_RT_Channels[name] and GEM3_RT_Channels[name].id ~= 0 and GEM3_RT_Channels[name].leader)
  then
    return true;
  end
  return false;
end

function GEM3_CHAN_AddChannel(channel,password,alias,slash)
  if(GEM3_QA_Config and GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_DBG_CheckTypes("GEM3_CHAN_AddChannel",channel,"channel","string",password,"password","string-nil",alias,"alias","string-nil",slash,"slash","string-nil");
  end
  local newchan;
  local guild = false;

  channel = strlower(channel);
  if(channel == strlower(GUILD))
  then
    guild = true;
    channel = GEM3_GuildChannelName;
    if(channel == nil)
    then
      -- GUI Callback
      GEM3_TriggerCallback("OnAddChannelFailed",GUILD,GEM3_ERR_ADD_CHANNEL_FAILED_NOT_IN_GUILD);
      return false;
    end
  else
    -- Check for "Custom channel" mistake
    if(channel == GEM3_GuildChannelName)
    then
      -- GUI Callback
      GEM3_TriggerCallback("OnAddChannelFailed",channel,GEM3_ERR_ADD_CHANNEL_FAILED_MISTAKE);
      return false;
    end
    --[[
    -- Not supported YET
    -- GUI Callback
    GEM3_TriggerCallback("OnAddChannelFailed",channel,GEM3_ERR_ADD_CHANNEL_FAILED_NOT_SUPPORTED);
    return false;]]
  end

  if(GEM3_RT_Channels[channel] ~= nil) -- Already exists
  then
    -- GUI Callback
    GEM3_TriggerCallback("OnAddChannelFailed",channel,GEM3_ERR_ADD_CHANNEL_FAILED_ALREADY_EXISTS);
    return false;
  end

  if(guild)
  then
    password = "";
    alias = "";
    slash = "";
  else
    if(password == nil) then password = ""; end
    if(alias == nil) then alias = ""; end
    if(slash == nil) then slash = ""; end
  end

  -- Check for alias
  if((alias == "" and slash ~= "") or (alias ~= "" and slash == ""))
  then
    -- GUI Callback
    GEM3_TriggerCallback("OnAddChannelFailed",channel,GEM3_ERR_ADD_CHANNEL_FAILED_ALIAS_SLASH);
    return false;
  end

  -- Add to saved config
  newchan = _GEM3_CHAN_AddConfigChannel(channel,password,alias,slash,guild);

  -- Add to running config
  local chanstruct = _GEM3_CHAN_AddRTChannel(newchan);

  -- GUI Callback
  GEM3_TriggerCallback("OnAddChannel",chanstruct);
  return true;
end

function GEM3_CHAN_DelChannel(channel)
  if(GEM3_QA_Config and GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_DBG_CheckTypes("GEM3_CHAN_DelChannel",channel,"channel","string");
  end
  local guild = false;

  channel = strlower(channel);
  if(channel == strlower(GUILD))
  then
    guild = true;
    channel = GEM3_GuildChannelName;
  end
  local infos = GEM3_RT_Channels[channel];
  if(infos) -- If channel is in my list, remove it
  then
    -- Remove from running config
    _GEM3_CHAN_DelRTChannel(channel);

    -- Remove from saved config
    _GEM3_CHAN_DelConfigChannel(channel);

    local thisReroll, anyReroll = GEM3_CHAN_IsChannelConfiguredForAnyReroll(channel);
    if(anyReroll == false)
    then
      GEM3_ChatDebug(GEM3_DEBUG_CHANNEL,"GEM3_CHAN_DelChannel: Removing all events I'm not leader of, for channel "..channel);
      -- Clear all events from that channel if this is my last reroll on it, and if not leader
      for ev_id,event in pairs(GEM3_QA_Events.events)
      do
        if(event.channel == channel)
        then
          if(not GEM3_IsMyReroll(event.leader))
          then
            GEM3_EVT_ClearEvent(ev_id,GEM3_LOC_EVENT_CLOSED_LEFT_CHANNEL,true);
          end
        end
      end
    else
      GEM3_ChatDebug(GEM3_DEBUG_CHANNEL,"GEM3_CHAN_DelChannel: I still have a reroll on channel "..channel..", not removing events");
    end
    
    GEM3_ChatDebug(GEM3_DEBUG_CHANNEL,"GEM3_CHAN_DelChannel: Removed channel : "..channel);

    -- GUI Callback
    GEM3_TriggerCallback("OnDelChannel",channel);
  end
end

function GEM3_CHAN_UpdateChannel(channel,password,alias,slash)
  if(GEM3_QA_Config and GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_DBG_CheckTypes("GEM3_CHAN_UpdateChannel",channel,"channel","string",password,"password","string",alias,"alias","string",slash,"slash","string");
  end
  local guild = false;

  channel = strlower(channel);
  if(channel == strlower(GUILD))
  then
    guild = true;
    channel = GEM3_GuildChannelName;
  end
  local infos = GEM3_RT_Channels[channel];
  if(infos == nil) -- Channel doesn't exist
  then
    -- GUI Callback
    GEM3_TriggerCallback("OnUpdateChannelFailed",channel,GEM3_ERR_UPDATE_CHANNEL_FAILED_NOT_FOUND);
    return;
  end

  if(guild)
  then
    -- GUI Callback
    GEM3_TriggerCallback("OnUpdateChannelFailed",channel,GEM3_ERR_UPDATE_CHANNEL_FAILED_GUILD_CHAN);
    return;
  end

  if(password == nil) then password = ""; end
  if(alias == nil) then alias = ""; end
  if(slash == nil) then slash = ""; end

  -- Check for password change
  if(infos.saved.password ~= password)
  then
    infos.saved.password = password;
    -- Try to update password
    SetChannelPassword(channel,password);
  end
  
  -- Sanity check
  if((alias == "" and slash ~= "") or (alias ~= "" and slash == ""))
  then
    -- GUI Callback
    GEM3_TriggerCallback("OnUpdateChannelFailed",channel,GEM3_ERR_UPDATE_CHANNEL_FAILED_ALIAS_SLASH);
    return;
  end
  infos.saved.alias = alias;
  -- Check for slash change
  if(infos.saved.slash ~= slash)
  then
    if(slash == "") -- No slash anymore
    then
      _GEM3_CHAN_UninitSlash(infos);
      infos.saved.slash = "";
    else
      _GEM3_CHAN_UninitSlash(infos); -- Remove old slash
      infos.saved.slash = slash;
      _GEM3_CHAN_InitSlash(infos); -- Init new slash
    end
  end

  -- GUI Callback
  GEM3_TriggerCallback("OnUpdateChannel",infos);
end

function GEM3_CHAN_IsChannelActive(name)
  if(GEM3_QA_Config and GEM3_QA_Config.debug and GEM3_QA_Config.debug >= 1) then
    GEM3_DBG_CheckTypes("GEM3_CHAN_IsChannelActive",name,"name","string");
  end
  if(GEM3_RT_Channels[name] and GEM3_RT_Channels[name].id ~= 0)
  then
    return true;
  end
  return false;
end

function GEM3_CHAN_IsChannelInList(channel,chan_list)
  for _,chan_infos in ipairs(chan_list)
  do
    if(chan_infos.name == channel)
    then
      return true;
    end
  end
  return false;
end

function GEM3_CHAN_IsChannelConfiguredForAnyReroll(channel)
  if(GEM3_CHAN_IsChannelActive(channel))
  then
    return true,true;
  end

  -- Check rerolls
  for name,config in pairs(GEM3_Config[GEM3_Realm])
  do
    if(GEM3_CHAN_IsChannelInList(channel,config.channels))
    then
      return false,true;
    end
  end
  
  return false,false;
end

function GEM3_CHAN_GetFirstAvailableChannel()
  -- Select first available channel
  for name in pairs(GEM3_RT_Channels)
  do
    if(GEM3_CHAN_IsChannelActive(name))
    then
      return name;
    end
  end
end

function GEM3_CHAN_GetChannelID(name)
  if(name == GEM3_GuildChannelName)
  then
    return -1;
  end
  return GetChannelName(name);
end

function GEM3_CHAN_ProcessChannelNoticeUser(kind,player1,_,_,player2,_,_,_,channel)
  if(channel and GEM3_CHAN_IsChannelActive(channel))
  then
    if(kind == "OWNER_CHANGED")
    then
      GEM3_ChatDebug(GEM3_DEBUG_CHANNEL,"Owner changed for channel "..channel.." to "..player1);
      GEM3_RT_Channels[channel].leader = false; -- Reset leader state
      if(player1 == GEM3_PlayerName)
      then
        _GEM3_CHAN_YouAreLeader(channel);
      end
    end
  end
end

function GEM3_CHAN_ProcessChannelListing(listing, _, _, _, _, _, _, _, channel)
  if(channel and listing and GEM3_CHAN_IsChannelActive(channel))
  then
    local cLeader = strbyte("*");
    local cModerator = strbyte("@");
    local cMute = strbyte("#");
    local char1;
    
    GEM3_RT_Channels[channel].leader = false; -- Reset leader state
    for value in gmatch(listing,"[^, ]+")
    do
      local isLeader = false;
      char1 = strbyte(value);
      if(char1 == cLeader)
      then
        value = strsub(value,2);
        if(value == GEM3_PlayerName)
        then
          isLeader = true;
        end
      elseif(char1 == cModerator or char1 == cMute)
      then
        value = strsub(value,2);
      end
      --GEM3_PLAY_PlayerJoined(channel,value); <- Player will send his info, and then I'll consider him as connected
      if(isLeader)
      then
        _GEM3_CHAN_YouAreLeader(channel);
      end
    end
  end
end

function GEM3_CHAN_DisplayChatMessage(channel,from,stamp,message)
  local index = GetChannelName(channel);
  
  if(index == nil or index == 0)
  then
    GEM3_ChatWarning("GEM3_CHAN_DisplayChatMessage: Not in channel "..channel);
    return;
  end

  local infos = ChatTypeInfo["CHANNEL"..index];
  if(infos == nil)
  then
    GEM3_ChatWarning("GEM3_CHAN_DisplayChatMessage: Failed to get channel "..index.." infos");
    return;
  end
  
  local rt = GEM3_RT_Channels[channel];
  if(rt == nil)
  then
    GEM3_ChatWarning("GEM3_CHAN_DisplayChatMessage: Failed to get RT infos for channel "..channel);
    return;
  end

  local alias = channel;
  if(rt.saved.alias and rt.saved.alias ~= "")
  then
    alias = rt.saved.alias;
  end
  -- Scan ChatFrames for active channel in each of them
  DEFAULT_CHAT_FRAME:AddMessage("["..alias.."] ["..from.."]: "..strgsub(message,"\127p","|"), infos.r, infos.g, infos.b); -- Tempo
end

function GEM3_CHAN_Process_YouJoined(channel)
  local rt = GEM3_RT_Channels[channel];
  
  if(rt == nil)
  then
    GEM3_ChatWarning("GEM3_CHAN_Process_YouJoined: rt is nil for channel "..channel);
    return;
  end
  rt.id = GEM3_CHAN_GetChannelID(channel);

  --ListChannelByName(channel); -- Done by CustomChatComm
  GEM3_PLAY_PlayerJoined(channel,GEM3_PlayerName);
  GEM3_COM_PlayerInfosSingle(channel);
  GEM3_ChatDebug(GEM3_DEBUG_CHANNEL,"Channel "..channel.." joined, schedule a broadcast of my data");
  GEM3_QUE_BuildBroadcastQueues(channel,nil); -- Schedule a broadcast for my data
  GEM3_TriggerCallback("OnChannelJoined",channel);
  -- Setup Slash
  if(rt.id > 0 and rt.saved.slash and rt.saved.slash ~= "") -- Not special guild channel
  then
    _GEM3_CHAN_InitSlash(rt);
  end
end

function GEM3_CHAN_Process_YouLeft(channel)
  local rt = GEM3_RT_Channels[channel];
  
  if(rt == nil)
  then
    GEM3_ChatWarning("GEM3_CHAN_Process_YouLeft: rt is nil for channel "..channel);
    return;
  end
  _GEM3_CHAN_UninitSlash(rt);
  rt.id = 0;
  GEM3_ChatDebug(GEM3_DEBUG_CHANNEL,"Channel "..channel.." left");
  GEM3_TriggerCallback("OnChannelLeft",channel);
end

function GEM3_CHAN_ProcessChannelLeave(_, player, _, _, _, _, _, _, channel)
  if(channel and GEM3_CHAN_IsChannelActive(channel))
  then
    GEM3_PLAY_PlayerLeft(channel,player);
  end
end

function GEM3_CHAN_GetChannelInfos(channel)
  for _,infos in ipairs(GEM3_QA_Channels)
  do
    if(infos.name == channel)
    then
      return infos;
    end
  end
  return nil;
end

--------------- API Exported functions ---------------

GEM3_CHAN_Frame = CreateFrame("Frame");
GEM3_CHAN_Frame:SetScript("OnEvent",_GEM3_CHAN_OnEvent);
