local L = Rock("LibRockLocale-1.0"):GetTranslationNamespace("GuildCheck")
GuildCheck = Rock:NewAddon("GuildCheck", "LibRockEvent-1.0", "LibRockDB-1.0", "LibRockConfig-1.0", "LibRockTimer-1.0")
local RC = Rock("LibRollCall-2.0")
local CT = LibStub("LibBabble-Class-3.0"):GetLookupTable()
local RCT = LibStub("LibBabble-Class-3.0"):GetReverseLookupTable()

GuildCheck.version = "2.02"
GuildCheck.debug = false

StaticPopupDialogs["GUILDCHECK_DELETEGUILD"] = {
  text = L["It seems that your last character has left this guild.\nDo you want to delete the data that GuildCheck has saved for this guild?"],
  button1 = L["Yes"],
  button2 = L["No"],
  OnAccept = function()
      GuildCheck:DeleteGuild()
  end,
  timeout = 0,
  whileDead = 1,
  hideOnEscape = 0
};

local options = {
	name = "GuildCheck",
	desc = L["Shows changes in your guild since your last time online."],
	handler = GuildCheck,
	type = "group",
	args = {
		gui = {
			type = "toggle",
			name = L["GUI"],
			desc = L["Defines whether the changes are diplayed in the chat or in the GUI on login."],
			get = "IsShowGui",
			set = "ToggleShowGui"
		},
		delay = {
			type = "string",
			name = L["Delay"],
			desc = L["Defines the seconds after login until the changes are reported."],
			usage = L["<Seconds>"],
			validate = function(v)
				if not tonumber(v) then
					return false, "Must be a number."
				else
					return true, nil
				end
			end,
			get = "GetDelay",
			set = "SetDelay"
		},
		show = {
			type = "execute",
			name = L["Show GUI"],
			buttonText = L["GUI"],
			desc = L["Shows a frame where you can display and copy changes."],
			func = "ShowGui"
		},
		offline = {
			type = "execute",
			name = L["Show Offline Changes"],
			buttonText = L["Offline"],
			desc = L["Shows changes since last session."],
			func = function()
				GuildCheck:ShowChanges(false)
			end
		},
		online = {
			type = "execute",
			name = L["Show Online Changes"],
			buttonText = L["Online"],
			desc = L["Shows changes during this session."],
			func = function()
				GuildCheck:ShowChanges(true)
			end
		}
	}
}

GuildCheck:SetDatabase("GCDB")
GuildCheck:SetDatabaseDefaults("profile", {
	gui = false,
	delay = 1
})
GuildCheck:SetDatabaseDefaults("realm", {
	guildChars = {},
	guildData = {}
})

local changes = {
	offline = {
		count = 0,
		chars = {}
	},
	online = {
		count = 0,
		chars = {}
	}
}

function GuildCheck:OnInitialize()
	self:Debug("OnInitialize()")
	self:SetConfigSlashCommand("/egc")
	self:SetConfigTable(options)
	self.faction, _ = UnitFactionGroup("player")
end

function GuildCheck:OnEnable()
	self:Debug("OnEnable()")
	--self:AddEventListener("LibRollCall-2.0", "MemberAdded", "AddMember")
	--self:AddEventListener("LibRollCall-2.0", "MemberRemoved", "RemoveMember")
	self:AddEventListener("LibRollCall-2.0", "Updated", "ScanGuild")
	self:AddEventListener("LibRollCall-2.0", "Joined", "JoinedGuild")
	self:AddEventListener("LibRollCall-2.0", "Left", "LeftGuild")
end

function GuildCheck:OnDisable()
	self:Debug("OnDisable()")
end

function GuildCheck:IsShowGui()
	self:Debug("IsShowGui()")
	return self.db.profile.gui
end

function GuildCheck:ToggleShowGui(value)
	self:Debug("ToggleShowGui()")
	self.db.profile.gui = not self.db.profile.gui
	return self.db.profile.gui
end

function GuildCheck:GetDelay()
	self:Debug("GetDelay()")
	return self.db.profile.delay
end

function GuildCheck:SetDelay(newValue)
	self:Debug("SetDelay()")
	self.db.profile.delay = newValue
end

function GuildCheck:ShowGui(online)
	self:Debug("ShowGui()")
	if not self.gui then
		self:CreateGui()
	end
	self:SetGuiText(online)
	self.gui:Show()
end

function GuildCheck:ShowChanges(online)
	self:Debug("ShowChanges(" .. tostring(online) .. ")")
	if self:IsShowGui() then
		self:ShowGui(online)
	else
		local target
		if online then
			target = changes.online
		else
			target = changes.offline
		end
		
		self:Out(string.format("GuildCheck (v%s):", self.version))
		for name, data in pairs(target.chars) do
			self:Out(string.format("  %s:", name))
			for i, change in pairs(data) do
				self:Out(change)
			end
		end
		
		if target.count > 1 then
			self:Out(string.format(L["Total: %d changes"], target.count))
		elseif target.count == 1 then
			self:Out(L["Total: 1 change"])
		else
			self:Out(L["no changes found"])
		end
	end
end

function GuildCheck:ScanGuild()
	self:Debug("ScanGuild()")
	if IsInGuild() and not self.scanning then
		self.scanning = true
		self.updatetime = time()
		
		if not self.guild then self.guild = RC:GetGuildName() end
		if not self.guildstring then
			self.guildstring = self.faction .. ":" .. self.guild
		end
		if not self.db.realm.guildChars[self.guildstring] then
			self.db.realm.guildChars[self.guildstring] = {}
		end
		self:AddPlayerToGuild()
		
		if not self.db.realm.guildData[self.guildstring] then
			self.db.realm.guildData[self.guildstring] = {}
			for name in RC:GetIterator("NAME", true) do
				self.db.realm.guildData[self.guildstring][name] = {
					rank		= RC:GetRank(name),
					level		= RC:GetLevel(name),
					class		= RCT[RC:GetClass(name)],
					note		= RC:GetNote(name),
					onote		= RC:GetOfficerNote(name),
					update	= self.updatetime
				}
			end
			self:Out("GuildCheck: " .. L["Initial scan done."])
			
		else
			for name in RC:GetIterator("NAME", true) do
				if not self.db.realm.guildData[self.guildstring][name] then
					self:AddMember(name)
				else
					local member = self.db.realm.guildData[self.guildstring][name]
					local rank = RC:GetRank(name)
					local level = RC:GetLevel(name)
					local class = RCT[RC:GetClass(name)]
					local note = RC:GetNote(name)
					local onote = RC:GetOfficerNote(name)
					if member.rank ~= rank then
						self:AddChange("Rank", name, rank)
						member.rank = rank
					end
					if member.level ~= level then
						self:AddChange("Level", name, level)
						member.level = level
					end
					if member.class ~= class then
						self:AddChange("Class", name, class)
						member.class = class
					end
					if member.note ~= note then
						self:AddChange("Note", name, note)
						member.note = note
					end
					if member.onote ~= onote then
						self:AddChange("Officernote", name, onote)
						member.onote = onote
					end
					
					member.update = self.updatetime
				end
			end
			
			for name, data in pairs(self.db.realm.guildData[self.guildstring]) do
				if data.update ~= self.updatetime then
					self:RemoveMember(name)
				end
			end
			
		end
		
		self.scanning = false
		
		if not self.firstscan then
			self:Debug("Delay: " .. self:GetDelay())
			self:AddTimer("GuildCheck", tonumber(self:GetDelay()), "ShowChanges")
			self.firstscan = true
		end
	end
end

function GuildCheck:AddPlayerToGuild()
	self:Debug("AddPlayerToGuild()")
	local name = UnitName("player")
	for i, player in pairs(self.db.realm.guildChars[self.guildstring]) do
		if player == name then
			return true
		end
	end
	table.insert(self.db.realm.guildChars[self.guildstring], name)
	return true
end

function GuildCheck:AddMember(name)
	self:Debug("AddMember(" .. name .. ")")
	if self.guildstring then
		
		self:AddChange("add", name)
		self.db.realm.guildData[self.guildstring][name] = {
			rank		= RC:GetRank(name),
			level		= RC:GetLevel(name),
			class		= RCT[RC:GetClass(name)],
			note		= RC:GetNote(name),
			onote		= RC:GetOfficerNote(name),
			update	= self.updatetime
		}
		
	end
end

function GuildCheck:RemoveMember(name)
	self:Debug("RemoveMember(" .. name .. ")")
	if self.guildstring then
		
		self:AddChange("remove", name)
		self.db.realm.guildData[self.guildstring][name] = nil
		
	end
end

function GuildCheck:AddChange(change, name, newValue)
	self:Debug("AddChange(" .. change .. ", " .. name .. ", " .. tostring(newValue) .. ")")
	local target
	if not self.firstscan then
		target = changes.offline
	else
		target = changes.online
	end
	
	if change == "add" then
		target.chars[name] = {
			string.format("    %s", L["has joined the guild"]),
			string.format("    %s: %s", L["Rank"], RC:GetRank(name)),
			string.format("    %s: %s", L["Level"], RC:GetLevel(name)),
			string.format("    %s: %s", L["Class"], RC:GetClass(name))
		}
		local note = RC:GetNote(name)
		local onote = RC:GetOfficerNote(name)
		if note then
			table.insert(target.chars[name], string.format("    %s: %s", L["Note"], note))
		end
		if onote then
			table.insert(target.chars[name], string.format("    %s: %s", L["Officernote"], onote))
		end
		
	elseif change == "remove" then
		local member = self.db.realm.guildData[self.guildstring][name]
		target.chars[name] = {
			string.format("    %s", L["has left the guild"]),
			string.format("    %s: %s", L["Rank"], member.rank),
			string.format("    %s: %s", L["Level"], member.level),
			string.format("    %s: %s", L["Class"], CT[member.class])
		}
		if member.note then
			table.insert(target.chars[name], string.format("    %s: %s", L["Note"], member.note))
		end
		if member.onote then
			table.insert(target.chars[name], string.format("    %s: %s", L["Officernote"], member.onote))
		end
		
	elseif change == "Officernote" then
		local member = self.db.realm.guildData[self.guildstring][name]
		if not target.chars[name] then
			target.chars[name] = {}
		end
		table.insert(target.chars[name], string.format("    %s: %s => %s", L["Officernote"], member.onote or "", newValue or ""))
		
	else
		local member = self.db.realm.guildData[self.guildstring][name]
		if not target.chars[name] then
			target.chars[name] = {}
		end
		if change == "Class" then
			newValue = CT[newValue]
		end
		table.insert(target.chars[name], string.format("    %s: %s => %s", L[change], member[string.lower(change)] or "", newValue or ""))
		
	end
	
	target.count = target.count + 1
end

function GuildCheck:Out(msg)
	self:Debug("Out()")
	DEFAULT_CHAT_FRAME:AddMessage(msg, 0, 1, 0)
end

function GuildCheck:JoinedGuild()
	self:Debug("JoinedGuild()")
	self.scanning = true
	
	self.updatetime = time()
	self.guild = RC:GetGuildName()
	self.guildstring = self.faction .. ":" .. self.guild
	
	if not self.db.realm.guildChars[self.guildstring] then
		self.db.realm.guildChars[self.guildstring] = {}
	end
	self:AddPlayerToGuild()
		
	if not self.db.realm.guildData[self.guildstring] then
		self.db.realm.guildData[self.guildstring] = {}
		for name in RC:GetIterator("NAME", true) do
			self.db.realm.guildData[self.guildstring][name] = {
				rank		= RC:GetRank(name),
				level		= RC:GetLevel(name),
				class		= RCT[RC:GetClass(name)],
				note		= RC:GetNote(name),
				onote		= RC:GetOfficerNote(name),
				update	= self.updatetime
			}
		end
		self:Out("GuildCheck: " .. L["Initial scan done."])
	end
	
	self.scanning = false
end

function GuildCheck:LeftGuild()
	self:Debug("LeftGuild()")
	if self.db.realm.guildChars[self.guildstring] then
		for i, name in pairs(self.db.realm.guildChars[self.guildstring]) do
			if name == UnitName("player") then
				table.remove(self.db.realm.guildChars[self.guildstring], i)
				break
			end
		end
		
		if #self.db.realm.guildChars[self.guildstring] == 0 then
			StaticPopup_Show("GUILDCHECK_DELETEGUILD");
		end
	end
end

function GuildCheck:DeleteGuild()
	self:Debug("DeleteGuild()")
	self.db.realm.guildChars[self.guildstring] = nil
	self.db.realm.guildData[self.guildstring] = nil
	self:Out("GuildCheck: " .. string.format(L["The guild '%s' has been deleted."], self.guild))
	
	self.guild = nil
	self.guildstring = nil
end

function GuildCheck:Debug(msg)
	if self.debug then
		DEFAULT_CHAT_FRAME:AddMessage("GC: " .. msg, 1, 0, 0)
	end
end

function GuildCheck:CreateGui()
	self.gui = CreateFrame("Frame", "GuildCheckGui", UIParent)
	self.gui:SetPoint("CENTER", "UIParent", "CENTER")
	self.gui:SetClampedToScreen(true)
	self.gui:SetWidth(400)
	self.gui:SetHeight(400)
	self.gui:EnableMouse(true)
	self.gui:SetMovable(true)
	self.gui:SetFrameStrata("HIGH")
	self.gui:RegisterForDrag("LeftButton")
	self.gui:SetScript("OnDragStart", function()
		this:StartMoving()
	end)
	self.gui:SetScript("OnDragStop", function()
		this:StopMovingOrSizing()
	end)
	
	self.gui:SetBackdrop({
		bgFile = [[Interface\TutorialFrame\TutorialFrameBackground]],
		edgeFile = [[Interface\DialogFrame\UI-DialogBox-Border]],
		tile = true,
		tileSize = 32,
		edgeSize = 16,
		insets = {left = 5, top = 5, right = 5, bottom = 5}
	})
	self.gui:SetBackdropColor(0,0,0,1)
	
	self.gui.bg = CreateFrame("Frame", "GuildCheckGuiBg", self.gui)
	self.gui.bg:SetWidth(370)
	self.gui.bg:SetHeight(340)
	self.gui.bg:SetPoint("TOPLEFT", self.gui, "TOPLEFT", 5, -25)
	self.gui.bg:SetBackdrop({
		edgeFile = [[Interface\Tooltips\UI-Tooltip-Border]],
		tile = true,
		edgeSize = 16,
		tileSize = 16,
		insets = {left = 5, top = 5, right = 5, bottom = 5}
	})
	
	self.gui.title = self.gui:CreateFontString(nil, "BACKGROUND", "GameFontNormal")
	self.gui.title:SetWidth(200)
	self.gui.title:SetHeight(12)
	self.gui.title:SetPoint("TOP", self.gui, "TOP", 0, -10)
	self.gui.title:SetText("GuildCheck (v" .. self.version .. ")")
	
	self.gui.scroll = CreateFrame("ScrollFrame", "GuildCheckGuiScrollBar", self.gui, "UIPanelScrollFrameTemplate")
	self.gui.scroll:SetWidth(355)
	self.gui.scroll:SetHeight(320)
	self.gui.scroll:SetPoint("TOPLEFT", self.gui, "TOPLEFT", 15, -35)
	
	self.gui.box = CreateFrame("EditBox", "GuildCheckGuiBox", self.gui.scroll)
	self.gui.box:SetMultiLine(true)
	self.gui.box:SetWidth(350)
	self.gui.box:SetHeight(310)
	self.gui.box:SetAutoFocus(false)
	--self.gui.box:SetNonSpaceWrap(true)
	self.gui.box:SetFontObject(GameFontHighlightSmall)
	self.gui.box:SetScript("OnTextChanged", function()
		local scrollBar = getglobal(this:GetParent():GetName().."ScrollBar") 
		this:GetParent():UpdateScrollChildRect()
		local min, max = scrollBar:GetMinMaxValues()
		if ( max > 0 and (this.max ~= max) ) then 
			this.max = max; scrollBar:SetValue(max)
		end
	end)
	self.gui.box:SetScript("OnEscapePressed", function()
		this:ClearFocus()
	end)
	
	self.gui.scroll:SetScrollChild(self.gui.box)
	
	self.gui.offline = CreateFrame("Button", "GuildCheckGuiOnline", self.gui, "OptionsButtonTemplate")
	self.gui.offline:SetText(L["Offline"])
	self.gui.offline:SetWidth(80)
	self.gui.offline:SetHeight(25)
	self.gui.offline:SetPoint("TOPLEFT", self.gui, "TOPLEFT", 150, -365)
	self.gui.offline:SetScript("OnClick", function()
		GuildCheck:SetGuiText(false)
	end)
	
	self.gui.online = CreateFrame("Button", "GuildCheckGuiOffline", self.gui, "OptionsButtonTemplate")
	self.gui.online:SetText(L["Online"])
	self.gui.online:SetWidth(80)
	self.gui.online:SetHeight(25)
	self.gui.online:SetPoint("TOPLEFT", self.gui, "TOPLEFT", 230, -365)
	self.gui.online:SetScript("OnClick", function()
		GuildCheck:SetGuiText(true)
	end)
	
	self.gui.exit = CreateFrame("Button", "GuildCheckGuiExit", self.gui, "OptionsButtonTemplate")
	self.gui.exit:SetText(L["Exit"])
	self.gui.exit:SetWidth(80)
	self.gui.exit:SetHeight(25)
	self.gui.exit:SetPoint("TOPLEFT", self.gui, "TOPLEFT", 310, -365)
	self.gui.exit:SetScript("OnClick", function()
		this:GetParent():Hide()
	end)
end

function GuildCheck:SetGuiText(online)
	local target
	if online then
		target = changes.online
	else
		target = changes.offline
	end
	
	local text = "GuildCheck (v" .. self.version .. "):\n"
	
	for name, data in pairs(target.chars) do
		text = text .. "  " .. name .. ":\n"
		for i, change in pairs(data) do
			text = text .. "    " .. change .. "\n"
		end
	end
	
	if target.count > 1 then
		text = text .. string.format(L["Total: %d changes"], target.count)
	elseif target.count == 1 then
		text = text .. L["Total: 1 change"]
	else
		text = text .. L["no changes found"]
	end
	
	self.gui.box:SetFocus()
	self.gui.box:SetText(text)
	self.gui.box:HighlightText()
	if online then
		self.gui.online:Disable()
		self.gui.offline:Enable()
	else
		self.gui.offline:Disable()
		self.gui.online:Enable()
	end
end