﻿--[[
Name: Grim Reaper
Revision: $Rev: 75238 $
Author: Zek (zeksie@googlemail.com)
Description: Combat log goodness review with health status.
]]

if (GetBuildInfo() >= "1.0.0" and GetBuildInfo() < "2.4.0") then
	-- Grim Reaper 2 requires WoW 2.4 or later
	return
end

local L = LibStub("AceLocale-2.2"):new("GrimReaper")
local dewdrop = LibStub("Dewdrop-2.0")
local roster = LibStub("Roster-2.1")
local SM = LibStub("LibSharedMedia-3.0")
local tablet = LibStub("Tablet-2.0")

BINDING_HEADER_GRIMREAPER		= L["TITLE"]
BINDING_NAME_GRIMREAPER_REPORT	= L["BINDING_REPORT"]
BINDING_NAME_GRIMREAPER_LOCK	= L["BINDING_LOCK"]
BINDING_NAME_GRIMREAPER_HOLD	= L["BINDING_HOLD"]

GrimReaper = LibStub("AceAddon-2.0"):new("AceConsole-2.0", "AceDB-2.0", "AceEvent-2.0", "AceHook-2.1", "FuBarPlugin-2.0")
local gr = GrimReaper

DisableAddOn("XPerl_GrimReaper")		-- Same author, same functionality, superceded by seperated addon

gr.version = "2.0 $Revision: 75238 $"
gr.title = L["TITLE"]
gr.hasIcon = "Interface\\Addons\\GrimReaper\\Images\\Death"
gr.defaultMinimapPosition = 300
gr.cannotDetachTooltip = true
gr.clickableTooltip = false
gr.overrideMenu = true

local band = bit.band

local line

local new, del, copy, deepDel
do
	local list = setmetatable({},{__mode='k'})
	function new(...)
		local t = next(list)
		if t then
			list[t] = nil
			for i = 1, select('#', ...) do
				t[i] = select(i, ...)
			end
			return t
		else
			return {...}
		end
	end
	function del(t)
		if (t) then
			for k in pairs(t) do
				t[k] = nil
			end
			t[''] = true
			t[''] = nil
			list[t] = true
		end
	end
	function deepDel(t)
		if (t) then
			for k,v in pairs(t) do
				if type(v) == "table" then
					deepDel(v)
				end
				t[k] = nil
			end
			t[''] = true
			t[''] = nil
			list[t] = true
		end
	end
	function copy(old)
		if (not old) then
			return
		end
		local n = new()
		for k,v in pairs(old) do
			if (type(v) == "table") then
				n[k] = copy(v)
			else
				n[k] = v
			end
		end
		return n
	end
end
gr.new, gr.del, gr.deepDel, gr.copy = new, del, deepDel, copy

local classColour = XPerlColourTable or setmetatable({},{
	__index = function(self, class)
		local c = RAID_CLASS_COLORS[strupper(class or "")]
		if (c) then
			c = format("|c00%02X%02X%02X", 255 * c.r, 255 * c.g, 255 * c.b)
		else
			c = "|c00808080"
		end
		self[class] = c
		return c
	end
})

local lastTime = 0
local millisecs = 0

gr.offset = 0
gr.needHealth = {}
gr.timeWidth = 90

local basicColour = {1, 1, 1}
local healColour = {1, 1, 1}
local specialColour = {0.5, 1, 1}
local buffColour = {0, 1, 0}
local debuffColour = {1, 0, 0}
local buffColourCleanse = {1, 1, 0.5}
local debuffColourCleanse = {1, 0.5, 1}

local recent = setmetatable({}, {__mode = "k"})
local schools = {[0] = "None", [1] = SPELL_SCHOOL0_CAP, [2] = DAMAGE_SCHOOL2, [4] = DAMAGE_SCHOOL3, [8] = DAMAGE_SCHOOL4, [16] = DAMAGE_SCHOOL5, [32] = DAMAGE_SCHOOL6, [64] = DAMAGE_SCHOOL7}
local schoolColour = {}
local schoolColourHex

local function getOption(v)
	return gr.db.profile[v]
end
local function setOption(v, n)
	gr.db.profile[v] = n
	gr:Tip()
end

gr.options = {
	handler = gr,
	type = 'group',
	args = {
		[L["CMD_LOCK"]] = {
			type = 'toggle',
			name = L["Lock"],
			desc = L["Lock to the current unit"],
			get = function() return gr.locked end,
			set = "Lock",
			disabled = function() return not gr.lastUnit end,
			order = 1,
		},
		[L["CMD_HIDE"]] = {
			type = 'toggle',
			name = L["Hide"],
			desc = L["Hide the grim reaper, but keep it active. You can also do this by clicking on the Grim Reaper FuBar/Minimap icon"],
			get = getOption,
			set = function(v,n) setOption(v,n) gr:DoIcon() gr.cycleState = nil end,
			passValue = "hide",
			order = 2,
		},
		[L["CMD_HOVER"]] = {
			type = 'group',
			name = L["Hover"],
			desc = L["Hover options for expanding information on the reaper lines"],
			order = 20,
			args = {
				[L["CMD_BLIZZLINE"]] = {
					type = 'toggle',
					name = L["Blizzard Lines"],
					desc = L["Use Blizzard's own combat log line generation on mouseover"],
					get = getOption,
					set = setOption,
					passValue = "blizzLines",
					order = 5,
				},
				[L["CMD_TOKENS"]] = {
					type = 'toggle',
					name = L["Unit Tokens"],
					desc = L["Use unit tokens to uniquely identify units of the same name"],
					get = getOption,
					set = setOption,
					passValue = "unitTokens",
					disabled = function() return gr.db.profile.blizzLines end,
					order = 10,
				},
				[L["CMD_BLIZZTOKENS"]] = {
					type = 'toggle',
					name = L["Blizzard Tokens"],
					desc = L["Use Blizzard's own unit tokens"].." (Not working with PTR build 8031)",
					get = getOption,
					set = setOption,
					passValue = "blizzardTokens",
					disabled = function() return gr.db.profile.blizzLines end,
					order = 15,
				},
				[L["CMD_BUFFTIPS"]] = {
					type = 'toggle',
					name = L["Buff Tips"],
					desc = L["Display buff tooltips detailing any buffs used on this line"],
					get = getOption,
					set = setOption,
					passValue = "buffTips",
					order = 20,
				},
				[L["CMD_DEBUFFTIPS"]] = {
					type = 'toggle',
					name = L["Debuff Tips"],
					desc = L["Display debuff tooltips detailing any debuffs used on this line"],
					get = getOption,
					set = setOption,
					passValue = "debuffTips",
					order = 25,
				},
			},
		},
		[L["CMD_DISPLAY"]] = {
			type = 'group',
			name = L["Display"],
			desc = L["Display options"],
			order = 50,
			args = {
				[L["CMD_INCLUDE"]] = {
					type = 'group',
					name = L["Include"],
					desc = L["What to show in the list"],
					order = 1,
					args = {
						[L["CMD_INCLUDE_CASTS"]] = {
							type = 'toggle',
							name = L["Casts on player"],
							desc = L["Display spells cast on player"],
							get = getOption,
							set = function(k,v) gr.db.profile[k] = v gr:MakeEventList() gr:Tip() end,
							passValue = "casts",
							order = 1,
						},
						[L["CMD_INCLUDE_CURES"]] = {
							type = 'toggle',
							name = L["Cures & Steals"],
							desc = L["Show when player is cured or has buff stolen"],
							get = getOption,
							set = function(k,v) gr.db.profile[k] = v gr:MakeEventList() gr:Tip() end,
							passValue = "curesAndSteals",
							order = 5,
						},
						[L["CMD_INCLUDE_BUFFS"]] = {
							type = 'toggle',
							name = L["Buff Gains/Losses"],
							desc = L["Display buff gains and losses"],
							get = getOption,
							set = setOption,
							passValue = "buffs",
							order = 10,
						},
						[L["CMD_INCLUDE_DEBUFFS"]] = {
							type = 'toggle',
							name = L["Debuff Gains/Losses"],
							desc = L["Display debuff gains and losses"],
							get = getOption,
							set = setOption,
							passValue = "debuffs",
							order = 11,
						},
					},
				},
				[L["CMD_TIME"]] = {
					type = 'group',
					name = L["Time"],
					desc = L["Timestamp formatting"],
					order = 1,
					args = {
						[L["CMD_NONE"]] = {
							type = 'toggle',
							name = L["None"],
							desc = L["Don't show timestamps"],
							get = function() return not gr.db.profile.showtime end,
							set = function(s) gr.db.profile.showtime = false gr:Tip() end,
							order = 1,
							isRadio = true,
						},
						[L["CMD_FULL"]] = {
							type = 'toggle',
							name = L["Full Time"],
							desc = L["Displays full time stamps"],
							get = function() return gr.db.profile.showtime and not gr.db.profile.deltaTime end,
							set = function(s) gr.db.profile.showtime = true gr.db.profile.deltaTime = nil gr.timeWidth = 90 gr:Tip() end,
							order = 2,
							isRadio = true,
						},
						[L["CMD_DELTA"]] = {
							type = 'toggle',
							name = L["Delta Time"],
							desc = L["Display time stamps as an offset from the most recent line"],
							get = function() return gr.db.profile.showtime and gr.db.profile.deltaTime == 1 end,
							set = function(s) gr.db.profile.showtime = true gr.db.profile.deltaTime = s and 1 or nil gr.timeWidth = 90 gr:Tip() end,
							order = 5,
							isRadio = true,
						},
						[L["CMD_NEXT"]] = {
							type = 'toggle',
							name = L["Delta Time Next"],
							desc = L["Display time stamps as an offset from the next line"],
							get = function() return gr.db.profile.showtime and gr.db.profile.deltaTime == 2 end,
							set = function(s) gr.db.profile.showtime = true gr.db.profile.deltaTime = s and 2 or nil gr:Tip() gr.timeWidth = 90 end,
							order = 6,
							isRadio = true,
						},
						space = {
							type = 'header',
							name = " ",
							order = 10,
						},
						[L["CMD_FORMAT"]] = {
							type = 'text',
							name = L["Time Format"],
							desc = L["Adjust the time format (Default: %X)"],
							usage = L["<format>\r%X = HH:MM:SS\r%H = HH, %M = MM, %S = SS\reg: %M:%S"],
							get = function() return gr.db.profile.timeformat end,
							set = function(f)
								if (f and f ~= "") then
									local ok, err = pcall(loadstring("date('"..f.."',time())"))
									if (ok) then
										gr.db.profile.timeformat = f
									else
										err = strmatch(err, ":1: (.*)")
										if (err) then
											gr:Print(L["Error: %s"], err)
											return
										end
									end
								else
									f = "%X"
								end
								gr.timeWidth = 90
								gr:Tip()
							end,
							order = 20,
						},
					},
				},
				spacer = {
					type = 'header',
					name = " ",
					order = 10
				},
				[L["CMD_COLOURS"]] = {
					type = 'toggle',
					name = L["Blizzard Colours"],
					desc = L["Use Blizzard magic school colours"],
					get = getOption,
					set = function(v,n) setOption(v,n) gr:SetColours() gr:Tip() end,
					passValue = "blizzardColours",
					order = 15,
				},
				[L["CMD_LINES"]] = {
					type = 'range',
					name = L["Lines"],
					desc = L["How many lines to show"],
					get = getOption,
					set = setOption,
					passValue = "lines",
					order = 20,
					min = 5,
					max = 30,
					step = 1,
				},
				[L["CMD_BARS"]] = {
					type = 'toggle',
					name = L["Health Bars"],
					desc = L["Show health bars"],
					get = getOption,
					set = setOption,
					passValue = "bars",
					order = 22,
				},
				[L["CMD_BARSINSIDE"]] = {
					type = 'toggle',
					name = L["Bars Inside"],
					desc = L["Show health bars inside the frame"],
					get = getOption,
					set = setOption,
					passValue = "barsInside",
					order = 23,
				},
				[L["CMD_BARSLEFT"]] = {
					type = 'toggle',
					name = L["Bars Left"],
					desc = L["Show health bars on left of frame"],
					get = getOption,
					set = setOption,
					disabled = function() return gr.db.profile.barsInside end,
					passValue = "barsLeft",
					order = 24,
				},
				[L["CMD_BARTEXTURE"]] = {
					type = 'text',
					name = L["Bar Texture"],
					desc = L["Set the texture for the buff timer bars"],
					validate = SM:List("statusbar"),
					order = 26,
					get = getOption,
					set = function(v,n) gr.bartexture = SM:Fetch("statusbar", n) setOption(v,n) end,
					passValue = "bartexture",
				},
				[L["CMD_DOCK"]] = {
					type = 'group',
					name = L["Docking"],
					desc = L["Docking options"],
					order = 50,
					args = {
						[L["CMD_ENABLE"]] = {
							type = 'toggle',
							name = L["Enable"],
							desc = L["Enable docking to the game's default tooltip"],
							get = getOption,
							set = function(k,v) gr.cycleState = nil setOption(k,v) end,
							passValue = "dockToTooltip",
							order = 1,
						},
						[L["CMD_DOCKP"]] = {
							type = 'text',
							name = L["Dock Point"],
							desc = L["Enable docking to the game's default tooltip"],
							validate = {TOPLEFT = L["TOPLEFT"], TOPRIGHT = L["TOPRIGHT"], BOTTOMLEFT = L["BOTTOMLEFT"], BOTTOMRIGHT = L["BOTTOMRIGHT"]},
							get = getOption,
							set = setOption,
							disabled = function() return not gr.db.profile.dockToTooltip end,
							passValue = "dockPoint",
							order = 2,
						},
					},
				},
				[L["CMD_WIDTH"]] = {
					type = 'range',
					name = L["Width"],
					desc = L["Adjust the width of the Grim Reaper"],
					get = getOption,
					set = function(v,n) gr:SetWidth(n) end,
					passValue = "frameWidth",
					min = 150,
					max = 300,
					step = 1,
					bigStep = 10,
					order = 60,
				},
				[L["CMD_SCALE"]] = {
					type = 'range',
					name = L["Scale"],
					desc = L["Adjust the scale of the Grim Reaper"],
					get = getOption,
					set = function(v,n) setOption(v,n) if (gr.attachment) then gr.attachment:SetScale(n) gr.timeWidth = 90 gr:Tip() end end,
					passValue = "scale",
					min = 0.3,
					max = 2,
					step = 0.01,
					bigStep = 0.1,
					order = 61,
				},
			},
		},
		[L["CMD_REPORT"]] = {
			type = 'group',
			name = L["Report"],
			desc = L["Report options"],
			order = 99,
			args = {
				[L["CMD_SHOWN"]] = {
					type = 'execute',
					name = L["Report"],
					desc = L["Report the currently displayed combat lines to chat"],
					order = 1,
					func = function() gr:ReportShown() dewdrop:Close() end,
				},
				spacer = {
					type = 'header',
					name = " ",
					order = 20,
				},
				channel = {
					type = 'group',
					name = L["Channel"],
					desc = L["Channel output options"],
					order = 50,
					args = {},
				}
			},
		},
		[L["CMD_LOG"]] = {
			type = 'group',
			name = L["Log"],
			desc = L["Show this player's combat log"],
			order = 120,
			disabled = function() return not gr.lastUnit end,
			args = {
				[L["CMD_LOGBOTH"]] = {
					type = 'execute',
					name = L["Both"],
					desc = L["Show this player's combat log for all events"],
					func = function() gr:ShowLog("BOTH") end,
					order = 1,
				},
				[L["CMD_LOGIN"]] = {
					type = 'execute',
					name = L["Incoming"],
					desc = L["Show this player's combat log for incoming events"],
					func = function() gr:ShowLog("INCOMING") end,
					order = 2,
				},
				[L["CMD_LOGOUT"]] = {
					type = 'execute',
					name = L["Outgoing"],
					desc = L["Show this player's combat log for outgoing events"],
					func = function() gr:ShowLog("OUTGOING") end,
					order = 3,
				},
			},
		},
	},
}

-- GetChatColour
local function GetChatColour(name)
	local info = ChatTypeInfo[name]
	local clr = {r = 0.5, g = 0.5, b = 0.5}
	if (info) then
		clr.r = (info.r or 0.5)
		clr.g = (info.g or 0.5)
		clr.b = (info.b or 0.5)
	end
	return clr
end

-- GetChannelList
function gr:GetChannelList()
	local cList = {}
	local l = {"RAID", "OFFICER", "GUILD", "PARTY", "SAY"}
	for k,v in pairs(l) do
		tinsert(cList, {display = getglobal("CHAT_MSG_"..v), channel = v, colour = GetChatColour(v)})
	end

	for i = 1,10 do
		local c, name = GetChannelName(i)
		if (name and c ~= 0) then
			tinsert(cList, {display = name, channel = "CHANNEL", index = c, colour = GetChatColour("CHANNEL"..c)})
		end
	end

	return cList
end

-- OnMenuRequest
function gr:OnMenuRequest(level, value)
	if (gr:FeedChannelOptions(level, value)) then
		return
	end

	dewdrop:FeedAceOptionsTable(gr.options)
end

-- FeedChannelOptions
function gr:FeedChannelOptions(level, value)
	if (type(level) == "string") then
		if (level == "channel") then
			level, value = 3, "channel"
		elseif (level == "whisper") then
			level, value = 4, "whisper"
		end
	end

	if (level == 3) then
		if (value == "channel") then
			local cList = self:GetChannelList()

			for i,entry in pairs(cList) do
				dewdrop:AddLine('text', entry.display,
						'textR', entry.colour.r,
						'textG', entry.colour.g,
						'textB', entry.colour.b,
						'func', function(c,n) gr:SetChannelRaw(c,n) end,
						'arg1', entry.channel,
						'arg2', entry.index,
						'checked', self.db.profile.channel == entry.channel and self.db.profile.channelIndex == entry.index
						)
			end

			dewdrop:AddLine('text', L["Self"],
					'textR', 1,
					'textG', 1,
					'textB', 1,
					'func', function(c,n) gr:SetChannelRaw(c,n) end,
					'arg1', "SELF",
					'arg2', "",
					'checked', self.db.profile.channel == "SELF"
					)

			dewdrop:AddLine('text', WHISPER,
					'value', "whisper",
					'textR', ChatTypeInfo.WHISPER.r,
					'textG', ChatTypeInfo.WHISPER.g,
					'textB', ChatTypeInfo.WHISPER.b,
					'checked', self.db.profile.channel == "WHISPER",
					'hasArrow', true
					)

			return true
		end
	elseif (level == 4) then
		if (value == "whisper") then
			local list = new()
			for unit in roster:IterateRoster() do
				tinsert(list, {name = unit.name, class = unit.class})
			end
			sort(list, function(a, b) return a.name < b.name end)

			for i,entry in pairs(list) do
				local colour = RAID_CLASS_COLORS[entry.class]

				dewdrop:AddLine('text', entry.name,
						'textR', colour and colour.r or 1,
						'textG', colour and colour.g or 1,
						'textB', colour and colour.b or 1,
						'func', function(c,n) gr:SetChannelRaw(c,n) end,
						'arg1', "WHISPER",
						'arg2', entry.name,
						'checked', self.db.profile.channel == "WHISPER" and self.db.profile.channelIndex == entry.name
						)
			end

			deepDel(list)
			return true
		end
	end
end

-- UnitNameFull
local function UnitNameFull(unit)
	local n, s = UnitName(unit)
	return s and s ~= "" and format("%s-%s", n, s) or n
end

-- WasDead
function gr:WasDeadOrOld(name)
	if (self.db.profile.healthList) then
		local hl = self.db.profile.healthList[name]
		if (hl) then
			hl = hl[#hl]
			if (not hl or hl.health == "DEAD" or hl.time < time() - 10) then
				return true
			end
		end
	end
end

local playerMask = 0x100		-- Is set properly later once Blizzard_CombatLog loads
local groupMask = 0x007			-- ditto	- 0x107

local tokens = {}
local function gr_CombatLog_String_GetToken(unitGUID, unitName, unitFlags)
	if (not unitName or band(unitFlags, playerMask) ~= 0) then
		return unitName
	end

	if (tokens[unitGUID]) then
		if (tokens[unitGUID] == 0) then
			return unitName
		end
	else
		if (not tokens[unitName]) then
			tokens[unitName] = 1
			tokens[unitGUID] = 0
			return unitName
		else
      			tokens[unitName] = tokens[unitName] + 1
			tokens[unitGUID] = tokens[unitName]
		end
	end

	local token = tokens[unitGUID]
	if (token == 0) then
		return unitName
	end
	return format("%s(%d)", unitName, token)
end

-- COMBAT_LOG_EVENT_UNFILTERED
function gr:COMBAT_LOG_EVENT_UNFILTERED(timestamp, event, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags, ...)
	local ev = gr.eventList[event]
	if (ev ~= nil) then
		if (band(dstFlags, groupMask) ~= 0) then
			if (ev or self:WasDeadOrOld(dstName)) then
				self.needHealth[dstName] = timestamp

				if (not de and self.lastUnit) then
					-- was dead or old or nothing
					self:UNIT_HEALTH(self.lastUnit)
					self.needHealth[dstName] = timestamp
				end
			end

			if (self.lastUnit and UnitGUID(self.lastUnit) == dstGUID) then
				if (self:OptionalFilters(timestamp, event, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags, ...)) then
					if (self.offset > 0 or (self.shownLines and self.shownLines < self.db.profile.lines)) then
						self.offset = 0
						self:Tip()
					end

					self:Tip(timestamp, event, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags, ...)
				end
			end
		end
	end
end

-- gr:UNIT_HEALTH()
function gr:UNIT_HEALTH(unit)
	local want
	-- Since we can get UNIT_HEALTH events multiple times for the same unit (party1 & raid1 for example), we filter out the surplus
	if (GetNumRaidMembers() > 0) then
		if (strfind(unit, "^raid%d+")) then
			want = true
		end
	elseif (GetNumPartyMembers() > 0) then
		if (unit == "player" or strfind(unit, "^party%d")) then
			want = true
		end
	else
		if (unit == "player") then
			want = true
		end
	end

	if (want) then		--UnitInParty(unit) or UnitInRaid(unit) or UnitIsUnit("player", unit) or UnitIsUnit("pet", unit)) then
		local name = UnitName(unit)
		local ts = self.needHealth[name]
		if (ts) then
			local n, o
			local list = self.db.profile.healthList[name]
			if (not list) then
				list = new()
				self.db.profile.healthList[name] = list
			end

			self.needHealth[name] = nil
			o = list[#list]
			if (o and o.time == ts) then
				n = o
			else
				n = new()
				o = nil
			end

			n.dead = UnitIsDeadOrGhost(unit)
			n.health = n.dead and "DEAD" or UnitHealth(unit)
			n.healthmax = UnitHealthMax(unit)
			n.realtime = time()
			n.time = ts
--ChatFrame1:AddMessage("got health for "..date("%X", ts)..", health = "..format("%d%%", n.health / n.healthmax * 100))

			if (not o) then
				tinsert(list, n)
			end

			local cutoff = time() - 60 * 60	-- 300
			for i,a in pairs(list) do
				if (not a.time or a.time < cutoff) then
					del(tremove(list, i))
				end
			end

			if (unit == self.lastUnit) then
				self:Tip()		-- Refresh if shown
			end
		end
	end
end

-- RAID_ROSTER_UPDATE
function gr:RAID_ROSTER_UPDATE()
	self:Validate()
end

-- PARTY_MEMBERS_CHANGED
function gr:PARTY_MEMBERS_CHANGED()
	if (GetNumRaidMembers() == 0) then
		-- Don't do this if in a raid, no need
		self:Validate()
	end
end

-- Validate
function gr:Validate()

	-- Clear out members of list that have left raid
	local current = new()
	if (GetNumRaidMembers() > 0) then
		for i = 1,GetNumRaidMembers() do
			local name = GetRaidRosterInfo(i)
			if (name) then
				current[name] = true
			end
		end
	else
		current[UnitName("player")] = true
		for i = 1,GetNumPartyMembers() do
			local name = UnitName("party"..i)
			if (name) then
				current[name] = true
			end
		end
	end

	for name,list in pairs(self.db.profile.healthList) do
		if (not current[name]) then
			deepDel(list)
			self.db.profile.healthList[name] = nil
		end
	end

	del(current)
end

-- HouseCleaning
function gr:HouseCleaning(cutoff)
	-- Remove any very old (probably only ever our own)
	if (not cutoff) then
		cutoff = time() - 6 * 60 * 60
	end
	for name,list in pairs(self.db.profile.healthList) do
		while (true) do
			local a = list[1]
			if (a and (not a.time or a.time < cutoff)) then
				del(tremove(list, 1))
			else
				break
			end
		end
	end
end

-- SetColours
function gr:SetColours()

	local bliz = Blizzard_CombatLog_CurrentSettings and Blizzard_CombatLog_CurrentSettings.colors
	if (bliz and bliz.schoolColoring and self.db and self.db.profile.blizzardColours) then
		for i,c in pairs(bliz.schoolColoring) do
			schoolColour[i] = {c.r, c.g, c.b}
		end
	else
		schoolColour = {[0] = {0.5, 0.5, 0.5}, [1] = {1, 0.5, 0.5}, [2] = {1, 1, 0.5}, [4] = {1, 0, 0}, [8] = {0, 1, 0}, [16] = {0.3, 0.3, 1}, [32] = {0.5, 0, 0.5}, [64] = {0.5, 0.5, 0.5}}
	end
	schoolColour[1] = {1, 0.5, 0.5}

	schoolColourHex = {}
	for k,v in pairs(schoolColour) do
		schoolColourHex[k] = format("|cFF%02X%02X%02X", v[1] * 255, v[2] * 255, v[3] * 255)
	end
end

-- gr:FixLastUnit()
function gr:FixUnit(unit)
	if (GetNumRaidMembers() > 0) then
		if (not strfind(unit, "^raid")) then
			for i = 1,GetNumRaidMembers() do
				if (UnitIsUnit("raid"..i, unit)) then
					return "raid"..i
				end
			end
		end
	else
		if (UnitIsUnit("player", unit)) then
			return "player"
		end

		if (GetNumPartyMembers() > 0) then
			if (not strfind(unit, "^party")) then
				for i = 1,GetNumRaidMembers() do
					if (UnitIsUnit("party"..i, unit)) then
						return "party"..i
					end
				end
			end
		end
	end

	return unit
end

-- grOnMouseWheel
local function grOnMouseWheel(self, amount)
	if (IsAltKeyDown()) then
		gr.db.profile.scale = max(0.2, min(2, gr.db.profile.scale + (0.025 * amount)))
		gr.attachment:SetScale(gr.db.profile.scale)
		gr.timeWidth = 90
		gr:Tip()
	else
		local oldOffset = gr.offset
		gr.offset = max(0, min(1000, gr.offset + amount))
		if (gr.offset ~= oldOffset) then
			gr:Tip()
		end
		gr:TriggerEvent("GrimReaper_Scrolled")
	end
end

-- grDragStart
local function grDragStart(self)
	if (not gr.db.profile.dockToTooltip) then
		gr.attachment.anchor:StartMoving()
	end
end

-- grDragStop
local function grDragStop(self)
	gr.attachment.anchor:StopMovingOrSizing()
	gr:SavePosition(self.anchor)
end

-- SavePosition
function gr:SavePosition(frame)
	if (frame) then
		if (not gr.db.profile.savedPositions) then
			gr.db.profile.savedPositions = {}
		end
		gr.db.profile.savedPositions[frame.name] = {top = frame:GetTop(), left = frame:GetLeft()}
	end
end

-- RestorePosition
function gr:RestorePosition(frame)
	if (gr.db.profile.savedPositions) then
		local pos = gr.db.profile.savedPositions[frame.name]
		if (pos) then
			if (pos.left or pos.right) then
				frame:ClearAllPoints()
				frame:SetPoint("TOPLEFT", UIParent, "BOTTOMLEFT", pos.left, pos.top)
			end
		end
	end
end

-- CreateAttachment
function gr:CreateAttachment()
	local att = self:CreateMainFrame("GrimReaperFrame")
	att.frames = {}
	self.attachment = att

	att:SetScale(self.db.profile.scale)
	att:SetWidth(self.db.profile.frameWidth)

	att:SetScript("OnMouseUp",
		function(self, button)
			if (button == "RightButton") then
				gr:ShowMenu(self)
			elseif (button == "LeftButton") then
				if (IsModifiedClick("CHATLINK")) then
					gr:ChatInsert(self, "line")
				elseif (IsModifiedClick("DRESSUP")) then
					gr:ChatInsert(self, "spell")
				end
			end
		end)
	att:SetScript("OnMouseWheel", grOnMouseWheel)
	att:SetScript("OnDragStart", grDragStart)
	att:SetScript("OnDragStop", grDragStop)

	GameTooltip:HookScript("OnHide",
		function(self)
			if (gr.db.profile.dockToTooltip) then
				gr.attachment:Hide()
				gr.lastUnit = nil
			end
		end)

	gr.CreateAttachment = nil
	return att
end

-- ChatInsert
function gr:ChatInsert(row, which)
	local a = row.a
	if (not a) then
		return
	end

	local str
	if (which == "line") then
		str = self:MakeCombatLogLine(a, true)
		--str = str:gsub("|", "||")

	elseif (which == "spell") then
		if (a.spellId) then
			str = GetSpellLink(a.spellId)
		end
	end

	if (str) then
		ChatEdit_InsertLink(str)
	end
end

-- CreateMainFrame
function gr:CreateMainFrame(name)
	local anchor = CreateFrame("Frame", name.."Anchor", UIParent)
	anchor.name = name
	anchor:SetPoint("CENTER")
	anchor:SetWidth(1)
	anchor:SetHeight(1)
	anchor:SetMovable(true)

	gr:RestorePosition(anchor)			-- Yes, GR not SELF

	local att = CreateFrame("Frame", name, anchor)
	att.anchor = anchor
	att:SetWidth(100)
	att:SetHeight(100)
	att:SetPoint("TOPLEFT")
	att:SetClampedToScreen(true)
	att:EnableMouse(true)
	att:RegisterForDrag("LeftButton")

	local bgDef = {bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
			edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
			tile = true, tileSize = 32, edgeSize = 24,
			insets = { left = 6, right = 6, top = 6, bottom = 6 }
		}
	att:SetBackdrop(bgDef)
	att:SetBackdropColor(0, 0, 0, 0.8)
	att:SetBackdropBorderColor(1, 1, 1, 1)

	att.title = att:CreateFontString(nil, "BORDER", "GameFontNormal")
	att.title:SetText(L["TITLE"])
	att.title:SetPoint("BOTTOM", att, "TOP")
	att.title:SetHeight(15)
	att.title:SetTextColor(1, 1, 1)

	return att
end

-- ShowMenu
function gr:ShowMenu(frame)
	dewdrop:Open(frame, "children", gr.OnMenuRequest, 'cursorX', true, 'cursorY', true)
end

-- SetAttachPoint
function gr:SetAttachPoint()
	local att = self.attachment
	if (not att) then
		att = self:CreateAttachment()
	end

	self.lastTip = self.lastTip or GameTooltip

	att:ClearAllPoints()
	if (self.db.profile.dockToTooltip) then
		att:SetParent(self.lastTip)
		att.anchor:SetMovable(false)
		att:EnableMouseWheel(false)
		att:EnableMouse(false)

		if (self.db.profile.dockPoint == "BOTTOMRIGHT") then
			att:SetPoint("BOTTOMLEFT", self.lastTip, "BOTTOMRIGHT")
		elseif (self.db.profile.dockPoint == "TOPLEFT") then
			att:SetPoint("TOPRIGHT", self.lastTip, "TOPLEFT")
		elseif (self.db.profile.dockPoint == "TOPRIGHT") then
			att:SetPoint("TOPLEFT", self.lastTip, "TOPRIGHT")
		else
			att:SetPoint("BOTTOMRIGHT", self.lastTip, "BOTTOMLEFT")
		end
	else
		att:SetParent(att.anchor)
		att:SetPoint("TOPLEFT", att.anchor, "TOPLEFT")
		att.anchor:SetMovable(true)
		att:EnableMouse(true)
		att:EnableMouseWheel(true)
	end
end

-- gr:Title
function gr:Title(unit)
	if (not unit) then
		unit = self.lastUnit
	end
	if (unit) then
		local unitName = UnitName(unit)
		local str
		if (unitName) then
			str = format("%s - %s%s%s", L["TITLE"], classColour[select(2, UnitClass(unit))], unitName, (self.locked and " |c00FF8080(L)") or "")
		else
			str = L["TITLE"]
		end
		self.attachment.title:SetText(str)
	end
end

-- ColourUnit
function gr:ColourUnit(unit)
	local c = select(2, UnitClass(unit))
	if (c) then
		c = classColour[c]
		return format("%s%s|r", c, UnitNameFull(unit))
	end
	return tostring(unit)
end

-- ColourUnitByName
function gr:ColourUnitByName(name)
	if (name) then
		if (UnitIsPlayer(name)) then
			local c = select(2, UnitClass(name))
			c = classColour[c]
			if (not c) then
				return name
			end
			return format("%s%s|r", c, name)
		else
			if (UnitIsFriend("player", name)) then
				return format("|cFF00FF00%s|r", name)
			else
				return format("|cFFFF0000%s|r", name)
			end
		end
	end
	return "?"
end

-- MakeBlizzardCombatCall
function gr:MakeBlizzardCombatCall(a)
	local critical, glancing, crushing
	if (a.modifier == "CRITICAL") then
		critical = 1
	elseif (a.modifier == "CRUSHING") then
		crushing = 1
	elseif (a.modifier == "GLANCING") then
		glancing = 1
	end
	local event = a.event
	local settings = Blizzard_CombatLog_CurrentSettings
	if (event == "SWING_DAMAGE" or event == "RANGE_DAMAGE") then
		return CombatLog_OnEvent(settings, a.timestamp, event, a.srcGUID, a.srcName, a.srcFlags, a.dstGUID, a.dstName, a.dstFlags, a.amount, a.school, a.resisted, a.blocked, a.absorbed, critical, glancing, crushing)
	elseif (event == "SWING_MISSED" or event == "RANGE_MISSED") then
		return CombatLog_OnEvent(settings, a.timestamp, event, a.srcGUID, a.srcName, a.srcFlags, a.dstGUID, a.dstName, a.dstFlags, a.missType)
	elseif (event == "SPELL_DAMAGE" or event == "SPELL_PERIODIC_DAMAGE" or event == "DAMAGE_SHIELD" or event == "SPELL_CAST_SUCCESS" or event == "SPELL_INSTAKILL") then
		return CombatLog_OnEvent(settings, a.timestamp, event, a.srcGUID, a.srcName, a.srcFlags, a.dstGUID, a.dstName, a.dstFlags, a.spellId, a.spellName, a.spellSchool, a.amount, a.school, a.resisted, a.blocked, a.absorbed, critical, glancing, crushing)
	elseif (event == "SPELL_MISSED" or event == "SPELL_PERIODIC_MISSED") then
		return CombatLog_OnEvent(settings, a.timestamp, event, a.srcGUID, a.srcName, a.srcFlags, a.dstGUID, a.dstName, a.dstFlags, a.spellId, a.spellName, a.spellSchool, a.missType)
	elseif (event == "SPELL_HEAL" or event == "SPELL_PERIODIC_HEAL") then
		return CombatLog_OnEvent(settings, a.timestamp, event, a.srcGUID, a.srcName, a.srcFlags, a.dstGUID, a.dstName, a.dstFlags, a.spellId, a.spellName, a.spellSchool, a.amount, critical)
	elseif (event == "SPELL_AURA_DISPELLED" or event == "SPELL_AURA_STOLEN") then
		return CombatLog_OnEvent(settings, a.timestamp, event, a.srcGUID, a.srcName, a.srcFlags, a.dstGUID, a.dstName, a.dstFlags, a.spellId, a.spellName, a.spellSchool, a.extraSpellId, a.extraSpellName, a.extraSpellSchool, a.auraType)
	elseif (event == "SPELL_AURA_APPLIED" or event == "SPELL_AURA_REMOVED") then
		return CombatLog_OnEvent(settings, a.timestamp, event, a.srcGUID, a.srcName, a.srcFlags, a.dstGUID, a.dstName, a.dstFlags, a.spellId, a.spellName, a.spellSchool, a.auraType)
	elseif (event == "ENVIRONMENTAL_DAMAGE") then
		return CombatLog_OnEvent(settings, a.timestamp, event, a.srcGUID, a.srcName, a.srcFlags, a.dstGUID, a.dstName, a.dstFlags, a.environmentalType, a.amount, a.school, a.resisted, a.blocked, a.absorbed, critical, glancing, crushing)
	else
		return CombatLog_OnEvent(settings, a.timestamp, event, a.srcGUID, a.srcName, a.srcFlags, a.dstGUID, a.dstName, a.dstFlags)
	end
end

-- MakeCombatLogLine
function gr:MakeCombatLogLine(a, report)
	if (not report and self.db.profile.blizzLines and CombatLog_OnEvent and Blizzard_CombatLog_CurrentSettings) then
		return self:MakeBlizzardCombatCall(a, a.timestamp, a.event, a.srcGUID, a.srcName, a.srcFlags, a.dstGUID, a.dstName, a.dstFlags)
	end

	local eventText
	if (a.auraType) then
		eventText = getglobal(format("ACTION_%s_%s", a.event, a.auraType))
	end
	if (not eventText) then
		eventText = getglobal("ACTION_"..a.event) or ""
	end
	if (report) then
		eventText = eventText:gsub("|c[0-9A-Fa-f][0-9A-Fa-f][0-9A-Fa-f][0-9A-Fa-f][0-9A-Fa-f][0-9A-Fa-f][0-9A-Fa-f][0-9A-Fa-f]", ""):gsub("|r", "")
	end

	if (a.dstName) then
		local school = a.school and a.school > 1 and schools[a.school]
		local dmgColour = schoolColourHex[a.school]
		local endColour = not report and "|r" or ""

		local tail
		if (a.missType) then
			tail = a.missType ~= "MISS" and getglobal(format("ACTION_%s_%s", a.event, a.missType))
			tail = tail and format("(%s)", tail) or ""
		else
			local info2Colour = a.extraSpellSchool and schoolColourHex[a.extraSpellSchool] or ""
			tail = format("%s%s%s%s%s%s", not report and dmgColour or "", a.amount or "", (school and " "..school) or "", not report and info2Colour or "", (report and a.extraSpellId and a.extraSpellName and GetSpellLink(a.extraSpellId)) or a.extraSpellName or "", endColour)
		end

		if (a.modifier) then
			local Mod = getglobal(format("TEXT_MODE_A_STRING_RESULT_%s%s", a.modifier, a.spellName and "_SPELL" or ""))
			if (Mod) then
				tail = format("%s (%s)", tail, Mod)
			end
		end

		if (a.resisted) then
			tail = format("%s %s(%d %s)%s", tail, not report and "|cFFFFFF80" or "", a.resisted, TEXT_MODE_A_STRING_RESULT_RESISTED, endColour)
		end
		if (a.blocked) then
			tail = format("%s %s(%d %s)%s", tail, not report and "|cFFC0C0FF" or "", a.blocked, TEXT_MODE_A_STRING_RESULT_BLOCKED, endColour)
		end
		if (a.absorbed) then
			tail = format("%s %s(%d %s)%s", tail, not report and "|cFFC0FFC0" or "", a.absorbed, TEXT_MODE_A_STRING_RESULT_ABSORBED, endColour)
		end

		local spellColour = (a.spellSchool and schoolColourHex[a.spellSchool])

		local sourceName, targetName
		if (self.db.profile.unitTokens) then
			if (self.db.profile.blizzardTokens and CombatLog_String_GetToken) then
				sourceName = CombatLog_String_GetToken(a.srcGUID, a.srcName, a.srcFlags)
				targetName = CombatLog_String_GetToken(a.dstGUID, a.dstName, a.dstFlags)
			else
				sourceName = gr_CombatLog_String_GetToken(a.srcGUID, a.srcName, a.srcFlags)
				targetName = gr_CombatLog_String_GetToken(a.dstGUID, a.dstName, a.dstFlags)
			end
		else
			sourceName = a.srcName
			targetName = a.dstName
		end

		local source, middle, target, eventPrefix
		source = sourceName and format("%s[%s] ", gr:UnitIcon(a.srcFlags, report), report and sourceName or gr:ColourUnitByName(sourceName)) or ""
		target = targetName and format(" %s[%s] ", gr:UnitIcon(a.dstFlags, report), report and targetName or gr:ColourUnitByName(targetName)) or ""

		if (strfind(a.event, "SPELL_AURA_APPLIED")) then
			eventPrefix = report and "++" or (a.auraType == "BUFF" and "|cFF80FF80++|r" or "|cFFFF8080++|r")
			eventText = ""
			source = target
			target = ""
			
		elseif (strfind(a.event, "SPELL_AURA_REMOVED")) then
			eventPrefix = report and "--" or (a.auraType == "BUFF" and "|cFF80FF80--|r" or "|cFFFF8080--|r")
			eventText = ""
			source = target
			target = ""
		else
			eventPrefix = ""
		end

		if (report and a.spellId) then
			middle = format("%s %s %s", eventPrefix, GetSpellLink(a.spellId), eventText)
		else
			middle = format("%s %s%s%s %s", eventPrefix, not report and spellColour or "", a.spellName or "", endColour, eventText)
		end

		local msg = format("%s%s%s%s", source, middle, target, tail)

		msg = msg:gsub("  ", " ")

		return msg
	end
end

-- UnitIcon
function gr:UnitIcon(flags, report)
	if (flags) then
		flags = band(flags, self.iconFlagsMask)
		local icon = self.iconFlags[flags]
		if (icon) then
			if (report) then
				return icon[2]
			else
				return icon[1]:gsub("$size", "0")
			end
		end
	end
	return ""
end

local icon_list = {"{star}", "{circle}", "{diamond}", "{triangle}", "{moon}", "{square}", "{cross}", "{skull}"}

if GetLocale() == "koKR" then
	icon_list = {"{별}", "{동그라미}", "{다이아몬드}", "{세모}", "{달}", "{네모}", "{가위표}", "{해골}"}
end

-- MakeIconFlags
function gr:MakeIconFlags()
	self.iconFlags = {}
	self.iconFlagsMask = 0
	for i = 1,8 do
		local mask = getglobal("COMBATLOG_OBJECT_RAIDTARGET"..i)
		if (mask) then
			self.iconFlagsMask = self.iconFlagsMask + mask
			self.iconFlags[mask] = {getglobal("COMBATLOG_ICON_RAIDTARGET"..i), icon_list[i]}
		end
	end
end

-- CreateTooltipFrame
function gr:CreateTooltipFrame(number)
	if (number == 1) then
		self.spellTip = {}
	end

	local att = self.attachment
	local tip = CreateFrame("GameTooltip", "GrimReaperInfoTip"..number, gr.extraInfoTip, "GameTooltipTemplate")
	self.spellTip[number] = tip

	if (self.spellTip[1] and self.spellTip[2]) then
		self.CreateTooltipFrame = nil
	end

	return gr.extraInfoTip
end

-- fadeInSpellInfo
local function fadeInSpellInfo(self, elapsed)
	if (gr.spellTip) then
		for i = 1,2 do
			if (gr.spellTip[i] and gr.spellTip[i]:IsShown()) then
				local alpha = gr.spellTip[i]:GetAlpha()
				gr.spellTip[i]:SetAlpha(min(1, alpha + elapsed * 2))
			end
		end
	end
end

-- CreateExtraFrame
function gr:CreateExtraFrame()
	local att = self.attachment
	gr.extraInfoTip = CreateFrame("GameTooltip", "GrimReaperInfoTip", att, "GameTooltipTemplate")

	gr.extraInfoTip:SetScript("OnUpdate", fadeInSpellInfo)
	gr.extraInfoTip:SetScript("OnShow", function(self)
		if (gr.spellTip) then
			if (gr.spellTip[1]) then
				gr.spellTip[1]:Hide()
			end
			if (gr.spellTip[2]) then
				gr.spellTip[2]:Hide()
			end
		end
	end)

	gr.extraInfoTip:SetScript("OnHide", function(self) self.a = nil end)

	self.CreateExtraFrame = nil
	return gr.extraInfoTip
end

-- ShowLineTip
function gr:ShowLineTip(msg, r, g, b)
	local tip = self.extraInfoTip
	if (not tip) then
		tip = self:CreateExtraFrame()
	end

	tip:SetOwner(self.attachment, "ANCHOR_TOP")
	tip:SetText(msg, r, g, b)
	tip:Show()

	local childPoint11, childPoint12, childPoint21, childPoint22
	tip:ClearAllPoints()
	if (gr.attachment:GetBottom() * self.attachment:GetScale() > 130) then
		if (gr.attachment:GetLeft() * self.attachment:GetScale() > 500) then
			childPoint11 = "TOPRIGHT"
			childPoint12 = "BOTTOMRIGHT"
			childPoint21 = "TOPRIGHT"
			childPoint22 = "TOPLEFT"
		else
			childPoint11 = "TOPLEFT"
			childPoint12 = "BOTTOMLEFT"
			childPoint21 = "TOPLEFT"
			childPoint22 = "TOPRIGHT"
		end
	else
		if (gr.attachment:GetLeft() * self.attachment:GetScale() > 500) then
			childPoint11 = "BOTTOMRIGHT"
			childPoint12 = "TOPRIGHT"
			childPoint21 = "TOPLEFT"
			childPoint22 = "TOPRIGHT"
		else
			childPoint11 = "BOTTOMLEFT"
			childPoint12 = "TOPLEFT"
			childPoint21 = "TOPRIGHT"
			childPoint22 = "TOPLEFT"
		end
	end

	tip:SetPoint(childPoint11, self.attachment, childPoint12)
	
	return childPoint11, childPoint12, childPoint21, childPoint22
end

-- grShowExtraInfo
local function grShowExtraInfo(self)
	local msg = gr:MakeCombatLogLine(self.a)
	if (msg) then
		local childPoint11, childPoint12, childPoint21, childPoint22 = gr:ShowLineTip(msg, 1, 1, 1)

		for i = 1,2 do
			local spellId = i == 1 and self.a.spellId or self.a.extraSpellId
			local harm = i == 1 and self.a.spellHarm or self.a.spell2Harm
			if (spellId and ((harm and gr.db.profile.debuffTips) or (not harm and gr.db.profile.buffTips))) then
				if (not gr.spellTip or not gr.spellTip[i]) then
					gr:CreateTooltipFrame(i)
				end

				gr.spellTip[i]:SetOwner(self, "ANCHOR_TOP")
				gr.spellTip[i]:SetHyperlink(format("|Hspell:%d|h |h", spellId))

				local line1 = getglobal(gr.spellTip[i]:GetName().."TextLeft1")
				if (line1) then
					local school = i == 1 and self.a.spellSchool or self.a.extraSpellSchool
					if (school) then
						school = schoolColour[school]
						if (school) then
							line1:SetTextColor(unpack(school))
						end
					end
				end

				gr.spellTip[i]:Show()
				gr.spellTip[i]:ClearAllPoints()
				if (i == 1) then
					gr.spellTip[i]:SetPoint(childPoint11, gr.extraInfoTip, childPoint12)
				else
					gr.spellTip[i]:SetPoint(childPoint21, gr.spellTip[1], childPoint22)
				end
				gr.spellTip[i]:SetAlpha(0)
			else
				if (gr.spellTip and gr.spellTip[i]) then
					gr.spellTip[i]:Hide()
				end
			end
		end
	end
end

-- grHideExtraInfo
local function grHideExtraInfo(self)
	if (gr.extraInfoTip) then
		gr.extraInfoTip:Hide()
	end
end

-- SetWidth
function gr:SetWidth(w)
	self.db.profile.frameWidth = w

	local att = self.attachment
	att:SetWidth(w)

	att.title:SetWidth(w)

	self:Tip()
end

-- CreateAttachmentLine
function gr:CreateAttachmentLine(i)
	local att = self.attachment
	local frame = CreateFrame("Frame", "GrimReaper_Attachment"..i, att)
	att.frames[i] = frame
	frame.a = new()

	if (i == 1) then
		frame:SetPoint("BOTTOMRIGHT", att, "BOTTOMRIGHT", -6, 10)
		frame:SetPoint("TOPLEFT", att, "BOTTOMLEFT", 6, 25)
	else
		frame:SetPoint("BOTTOMRIGHT", att.frames[i - 1], "TOPRIGHT")
		frame:SetPoint("TOPLEFT", att.frames[i - 1], "TOPLEFT", 0, 15)
	end

	frame:SetWidth(self.db.profile.frameWidth - 10)

	frame.text = frame:CreateFontString(nil, "ARTWORK", "GameFontNormal")
	frame.text:SetJustifyH("RIGHT")
	frame.text:SetPoint("RIGHT")		-- SetAllPoints(frame)
	frame.text:SetHeight(15)

	frame.time = frame:CreateFontString(nil, "ARTWORK", "GameFontNormal")
	frame.time:SetJustifyH("RIGHT")		--"LEFT")
	frame.time:SetPoint("LEFT")
	frame.time:SetHeight(15)
	frame.time:SetTextColor(1, 1, 0.5)

	frame.highlight = frame:CreateTexture(nil, "HIGHLIGHT")
	frame.highlight:SetTexture("Interface\\Buttons\\UI-Common-MouseHilight")
	frame.highlight:SetBlendMode("ADD")
	frame.highlight:SetTexCoord(0.25, 0.75, 0.25, 0.75)
	frame.highlight:SetAlpha(0.3)
	frame.highlight:SetAllPoints(true)

	frame:EnableMouse(true)
	frame:SetScript("OnMouseWheel", function(self,amount) grOnMouseWheel(gr.attachment,amount) end)
	frame:SetScript("OnEnter", grShowExtraInfo)
	frame:SetScript("OnLeave", grHideExtraInfo)
	frame:SetScript("OnDragStart", function(self) grDragStart(gr) end)
	frame:SetScript("OnDragStop", function(self) grDragStop(gr) end)
	frame:SetScript("OnMouseUp", att:GetScript("OnMouseUp"))

	frame:RegisterForDrag("LeftButton")

	--frame.UpdateTooltip = grShowExtraInfo

	return frame
end

-- HealthBar
function gr:HealthBar(line, health, healthmax, guess)
	local att = self.attachment
	local frame = att.frames[line]
	if (self.db.profile.bars) then
		if (not frame.texture) then
			frame.texture = frame:CreateTexture(nil, "BORDER")
			frame.texture:SetWidth(1)
			frame.texture:SetHeight(15)

			frame.healthText = frame:CreateFontString(nil, "ARTWORK", "GameFontNormalSmall")
			frame.healthText:SetWidth(90)
			frame.healthText:SetTextColor(1, 1, 1)
		end

		local barTex = self.bartexture

		frame.texture:SetTexture(barTex)

		frame.texture:ClearAllPoints()
		frame.healthText:ClearAllPoints()
		if (self.db.profile.barsInside) then
			frame.texture:SetPoint("RIGHT", frame, "RIGHT")
			frame.healthText:Hide()
		else
			if (self.db.profile.barsLeft) then
				frame.texture:SetPoint("RIGHT", frame, "LEFT", -4, 0)
				frame.healthText:SetJustifyH("RIGHT")
				frame.healthText:SetPoint("RIGHT", frame, "LEFT", -7, 1)
			else
				frame.texture:SetPoint("LEFT", frame, "RIGHT", 4, 0)
				frame.healthText:SetJustifyH("LEFT")
				frame.healthText:SetPoint("LEFT", frame, "RIGHT", 7, 1)
			end
			frame.healthText:Show()
		end
		frame.texture:Show()

		if (health == "DEAD") then
			frame.healthText:SetText(DEAD)
			frame.texture:SetWidth(1)
			frame.texture:SetVertexColor(1, 0, 0)
			frame.health = "DEAD"

		elseif (health) then
			local percentage = health / healthmax
			local perc = percentage * 100
			if (guess) then
				if (self.db.profile.barsLeft) then
					frame.healthText:SetFormattedText("* %d%%", perc)
				else
					frame.healthText:SetFormattedText("%d%% *", perc)
				end
			else
				frame.healthText:SetFormattedText("%d%%", perc)
			end
			if (self.db.profile.barsInside) then
				frame.texture:SetWidth((self.db.profile.frameWidth - 12) * percentage)
			else
				frame.texture:SetWidth(min(100, max(1, perc)))
			end
			frame.health = perc

			local r, g, b
			if (percentage < 0.5) then
				r = 1
				g = 2*percentage
				b = 0
			else
				g = 1
				r = 2*(1 - percentage)
				b = 0
			end

			if (r >= 0 and g >= 0 and b >= 0 and r <= 1 and g <= 1 and b <= 1) then
				frame.texture:SetVertexColor(r, g, b)
			end
		end
	else
		if (frame and frame.texture) then
			frame.texture:Hide()
			frame.healthText:Hide()
			frame.health = nil
		end
	end
end

-- Save/Load filters
local noSaveFilter
local function grCombatLogAddFilter(...)
	if (not noSaveFilter) then
		if (not gr.savedState) then
			gr.savedState = new()
		end
		tinsert(gr.savedState, new(...))
	end
end

local function grCombatLogResetFilter()
	if (not noSaveFilter) then
		gr.savedState = deepDel(gr.savedState)
	end
end

hooksecurefunc("CombatLogResetFilter", grCombatLogResetFilter)
hooksecurefunc("CombatLogAddFilter", grCombatLogAddFilter)

function gr:CombatLogSaveFilter()
	local a = self.savedState
	self.savedState = nil
	return a
end

function gr:CombatLogLoadFilter(restore)
	if (restore) then
		noSaveFilter = true
		CombatLogResetFilter()
		for k,v in ipairs(restore) do
			CombatLogAddFilter(v[1], v[2], v[3])
		end
		deepDel(self.savedState)
		self.savedState = restore
		noSaveFilter = nil
	end
end

-- ShuffleListUpward
function gr:ShuffleListUpward(timestamp)
	local shown = 0
	local att = self.attachment

	if (#att.frames < self.db.profile.lines) then
		self:CreateAttachmentLine(#att.frames + 1)
	end

	if (#att.frames > 1) then
		local bottom = att.frames[1]
		local lines = #att.frames
		local removed = tremove(att.frames, lines)
		tinsert(att.frames, 1, removed)

		removed:ClearAllPoints()
		bottom:ClearAllPoints()

		removed:SetPoint("BOTTOMRIGHT", att, "BOTTOMRIGHT", -6, 10)
		removed:SetPoint("TOPLEFT", att, "BOTTOMLEFT", 6, 25)

		bottom:SetPoint("BOTTOMRIGHT", removed, "TOPRIGHT")
		bottom:SetPoint("TOPLEFT", removed, "TOPLEFT", 0, 15)

		for i,frame in ipairs(att.frames) do
			if (frame:IsShown()) then
				shown = i - 1
			end
		end
	else
		shown = 0
	end

	if (shown > 0 and self.db.profile.showtime) then
		if (self.db.profile.deltaTime == 1) then
			for i = 2,#att.frames do
				local frame = att.frames[i]
				frame.time:SetText(self:MakeDeltaTime(timestamp - frame.a.timestamp, true))
				frame.time:SetJustifyH("RIGHT")
				frame.time:SetWidth(self.timeWidth)
			end
		elseif (self.db.profile.deltaTime == 2) then
			local frame = att.frames[2]
			frame.time:SetText(self:MakeDeltaTime(timestamp - frame.a.timestamp, true))
			frame.time:SetJustifyH("RIGHT")
			frame.time:SetWidth(self.timeWidth)
		end
	end

	if (att.frames[1]) then
		att.frames[1].text:SetText("")
		att.frames[1].time:SetText("")
	end

	return shown
end

-- GetModifier
local function GetModifier(frame, critical, glancing, crushing)
	if (critical) then
		frame.a.modifier = "CRITICAL"
	elseif (glancing) then
		frame.a.modifier = "GLANCING"
	elseif (crushing) then
		frame.a.modifier = "CRUSHING"
	end
end

-- AddAmount
local function AddAmount(ftext, amount, resisted, blocked, absorbed, crushing)
	if (resisted or blocked or absorbed) then
		local prefix = format("%s%s%s", resisted and "r" or "", blocked and "b" or "", absorbed and "a" or "")
		local r = (resisted or 0) + (blocked or 0) + (absorbed or 0)
		local percentResist = 100 - (amount / (amount + r) * 100)
		if (crushing) then
			ftext:SetFormattedText("|cFF80FFFF(%s%d%%)|r %s -%d", prefix, percentResist, L["CRUSHING"], amount)
		else
			ftext:SetFormattedText("|cFF80FFFF(%s%d%%)|r -%d", prefix, percentResist, amount)
		end
	else
		if (crushing) then
			ftext:SetFormattedText("%s -%d", L["CRUSHING"], amount)
		else
			ftext:SetFormattedText("-%d", amount)
		end
	end
end

function gr:AddLine(timestamp, event, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags, ...)
	line = line + 1
	local att = self.attachment
	local frame = att.frames[line]
	if (not frame) then
		frame = self:CreateAttachmentLine(line)
	end

	local color, missType
	local spellId, spellName, spellSchool, amount, school, resisted, blocked, absorbed, critical, glancing, crushing

	local ftext = frame.text
	frame.a = del(frame.a)
	frame.a = new()
	frame.a.timestamp = timestamp
	frame.a.event = event
	frame.a.srcGUID = srcGUID
	frame.a.dstGUID = dstGUID
	frame.a.srcFlags = srcFlags
	frame.a.dstFlags = dstFlags
	frame.a.srcName = srcName
	frame.a.dstName = dstName

	if (event == "SWING_DAMAGE") then
		amount, school, resisted, blocked, absorbed, critical, glancing, crushing = ...
		AddAmount(ftext, amount, resisted, blocked, absorbed, crushing)
		frame.a.amount = amount
		frame.a.school = school
		frame.a.spellName = ACTION_SWING
		frame.a.resisted = resisted
		frame.a.blocked = blocked
		frame.a.absorbed = absorbed
		GetModifier(frame, critical, glancing, crushing)

	elseif (event == "SWING_MISSED") then
		missType = ...
		ftext:SetText(getglobal(missType) or tostring(missType))
		color = specialColour
		frame.a.spellName = ACTION_SWING
		frame.a.missType = missType

	elseif (event == "RANGE_DAMAGE") then
		amount, school, resisted, blocked, absorbed, critical, glancing, crushing = select(4, ...)
		AddAmount(ftext, amount, resisted, blocked, absorbed, crushing)
		frame.a.amount = amount
		frame.a.school = school
		frame.a.spellName = ACTION_RANGED
		frame.a.resisted = resisted
		frame.a.blocked = blocked
		frame.a.absorbed = absorbed
		GetModifier(frame, critical, glancing, crushing)

	elseif (event == "RANGE_MISSED") then
		local missType = select(4, ...)
		ftext:SetText(getglobal(missType) or tostring(missType))
		color = specialColour
		frame.a.spellName = ACTION_RANGED
		frame.a.missType = missType

	elseif (strmatch(event, "^SPELL_")) then
		spellId, spellName, spellSchool = ...
		frame.a.spellId = spellId
		frame.a.spellName = spellName
		frame.a.spellSchool = spellSchool

		if (event == "SPELL_DAMAGE" or event == "SPELL_PERIODIC_DAMAGE") then
			amount, school, resisted, blocked, absorbed, critical, glancing, crushing = select(4, ...)
			AddAmount(ftext, amount, resisted, blocked, absorbed, crushing)
			frame.a.spellHarm = true
			frame.a.amount = amount
			frame.a.school = school
			frame.a.resisted = resisted
			frame.a.blocked = blocked
			frame.a.absorbed = absorbed
			GetModifier(frame, critical, glancing, crushing)

		elseif (event == "SPELL_MISSED" or event == "SPELL_PERIODIC_MISSED") then
			local missType = select(4, ...)
			ftext:SetText(getglobal(missType) or tostring(missType))
			frame.a.spellHarm = true
			frame.a.missType = missType

		elseif (event == "SPELL_HEAL" or event == "SPELL_PERIODIC_HEAL") then
			amount, critical = select(4, ...)
			ftext:SetFormattedText("+%d", amount)
			color = healColour
			frame.a.amount = amount
			GetModifier(frame, critical, glancing, crushing)

		elseif (event == "SPELL_AURA_DISPELLED" or event == "SPELL_AURA_STOLEN") then
			local extraSpellId, extraSpellName, extraSpellSchool, auraType = select(4, ...)
			frame.a.extraSpellName = extraSpellName
			frame.a.extraSpellSchool = extraSpellSchool
			frame.a.auraType = auraType
			frame.a.extraSpellId = extraSpellId

			local c
			if (band(srcFlags, COMBATLOG_OBJECT_CONTROL_PLAYER) and srcName and UnitGUID(srcName) == srcGUID) then
				c = select(2, UnitClass(srcName))
				if (c) then
					ftext:SetFormattedText("%s%s|r %s", classColour[c], srcName:sub(1, 8), spellName)
				end
			end
			if (not c) then
				ftext:SetFormattedText("%s", spellName)
			end
			if (auraType == "DEBUFF") then
				frame.a.spell2Harm = true
			end

		elseif (event == "SPELL_AURA_APPLIED" or event == "SPELL_AURA_REMOVED") then
			local auraType = select(4, ...)
			ftext:SetFormattedText("%s%s", event == "SPELL_AURA_APPLIED" and "++" or "--", spellName)
			frame.a.auraType = auraType
			if (auraType == "BUFF") then
				color = buffColour
			else
				color = debuffColour
				frame.a.spellHarm = true
			end

		elseif (event == "SPELL_INSTAKILL") then
			ftext:SetText("Kill")
			frame.a.spellHarm = true

		elseif (event == "SPELL_CAST_SUCCESS") then
			ftext:SetFormattedText("*%s*", spellName)
		else
			-- Ignoring things like energise, which are not health related
			frame.a.spellId, frame.a.spellName, frame.a.spellSchool = nil, nil, nil
			ignore = true
		end

	elseif (event == "DAMAGE_SHIELD") then
		spellId, spellName, spellSchool, amount, school, resisted, blocked, absorbed, critical, glancing, crushing = ...

		ftext:SetFormattedText("DS -%d", amount)
		frame.a.spellName = spellName
		frame.a.spellSchool = spellSchool
		frame.a.amount = amount
		frame.a.school = school
		frame.a.resisted = resisted
		frame.a.blocked = blocked
		frame.a.absorbed = absorbed
		GetModifier(frame, critical, glancing, crushing)

	elseif (event == "ENVIRONMENTAL_DAMAGE") then
		local environmentalType
		environmentalType, amount, school, resisted, blocked, absorbed, critical, glancing, crushing = ...
		spellName = getglobal(format("ACTION_%s_%s", event, environmentalType))

		ftext:SetFormattedText("%s -%d", spellName or "Env", amount)
		frame.a.environmentalType = environmentalType
		frame.a.spellName = spellName
		frame.a.spellSchool = school
		frame.a.amount = amount
		frame.a.school = school
		frame.a.resisted = resisted
		frame.a.blocked = blocked
		frame.a.absorbed = absorbed
		GetModifier(frame, critical, glancing, crushing)

		frame.a.amount = amount
	end

	if (not color) then
		if (school or spellSchool) then
			color = schoolColour[school or spellSchool or 0] or basicColour
		else
			color = basicColour
		end
	end

	if (critical or crushing) then
		frame.bold = true
		frame.text:SetFontObject("GameFontNormalLarge")
	else
		frame.bold = nil
		frame.text:SetFontObject("GameFontNormal")
	end

	frame.text:SetTextColor(unpack(color))

	if (self.db.profile.showtime) then
		local time = format("%s|cFFC0FFC0.%03d", date(self.db.profile.timeformat, timestamp), timestamp * 1000 % 1000)

		frame.time:SetJustifyH("LEFT")
		frame.time:SetWidth(self.db.profile.frameWidth)
		frame.time:Show()

		if (line == 1 and self.offset == 0) then
			local test
			if (strlen(time) < 17) then	-- We need at least "-00.000" characters (exluding the |c colour code)
				test = "-00.000"
			else
				test = time
			end
			frame.time:SetText(test)
			local w = frame.time:GetStringWidth()
			if (not self.timeWidth or w > self.timeWidth) then
				self.timeWidth = w
			end
		end

		frame.time:SetText(time)
		frame.time:SetWidth(self.timeWidth)
		frame.time:SetJustifyH("RIGHT")

		if (self.db.profile.deltaTime and (self.offset > 0 or line > 1)) then
			frame.time:SetWidth(self.timeWidth)
			frame.time:SetJustifyH("RIGHT")
			local t
			if (self.db.profile.deltaTime == 1) then
				t = timestamp - self.startTimeDelta
			else
				local nextFrame = att.frames[line - 1]
				if (nextFrame) then
					local prevTime = nextFrame.a.timestamp
					t = timestamp - prevTime
				else
					t = timestamp - self.startTimeDelta
				end
			end
			frame.time:SetText(self:MakeDeltaTime(t, true))
		end

		frame.text:SetWidth(self.db.profile.frameWidth - 12 - self.timeWidth)
	else
		frame.time:Hide()
		frame.text:SetWidth(self.db.profile.frameWidth - 12)
	end

	frame:Show()

	if (gr.extraInfoTip and gr.extraInfoTip:IsOwned(frame)) then
		grShowExtraInfo(frame)
	end
end

-- MakeDeltaTime
function gr:MakeDeltaTime(t, coloured)
	t = abs(t)
	local colour = coloured and "|cFFC0FFC0" or ""
	if (t < 60) then
		return format("-%d%s.%03d", t, colour, t * 1000 % 1000)
	elseif (t < 3600) then
		return format("-%s%s.%03d", date("%M:%S", t):gsub("^0([0-9])", "%1"), colour, t * 1000 % 1000)
	else
		return format("-%s%s.%03d", date("%H:%M:%S", t):gsub("^0([0-9])", "%1"), colour, t * 1000 % 1000)
	end
end

-- FinalizeLines
function gr:FinalizeLines()
	local att = self.attachment
	for i = line + 1,#att.frames do
		local frame = att.frames[i]
		if (frame) then
			frame:Hide()
		end
	end
end

-- UpdateAmount
function gr:UpdateHealth(line, health, healthmax)
	local att = self.attachment
	local frame = att.frames[line]
	local a = frame.a
	local event = a.event
	local amount = a.amount

	if (a.amount and health ~= "DEAD") then
		if (strfind(event, "DAMAGE")) then
	    		health = max(0, min(healthmax, health - a.amount))
		else
	    		health = max(0, min(healthmax, health + a.amount))
		end
	end

	return health, healthmax
end

-- DoHealth
function gr:DoHealth()
--ChatFrame1:AddMessage("---")
	local att = self.attachment
	local healthList = self.db.profile.healthList[self.lastName]
	if (healthList) then
		local startTime, startLine
		for i = #att.frames,1,-1 do
			local frame = att.frames[i]
			if (frame:IsShown()) then
				startTime = frame.a.timestamp
				startLine = i
				break
			end
		end
		local endTime
		if (att.frames[1]) then
			endTime = att.frames[1].a.timestamp
		end

		if (startTime and endTime) then
			local entry, entryNo, guess

			-- Find health update for this time
			for i = 1,#healthList do
				local e = healthList[i]
				if (e and e.time == startTime) then
					entry = e
					entryNo = i
					break
				end
			end
			if (not entry) then
				-- See if we can find an entry that spans the top list item
				for i = #healthList,2,-1 do
					local e = healthList[i]
					if (e.time > startTime) then
						local nextEntry = healthList[i - 1]
						if (nextEntry and nextEntry.time < startTime) then
							entry = nextEntry
							entryNo = i - 1
							guess = true
							break
						end
					end
				end
				if (not entry) then
					local e = healthList[#healthList]
					if (e and e.time < startTime) then
						-- Just use the last one we have
						entry = e
						entryNo = #healthList
						guess = true
					end
				end
			end
			if (not entry) then
				-- Health list starts late, so find how far down the list we need to begin
				entryNo = 1
				entry = healthList[1]
				while (entry and startTime and startTime ~= entry.time) do
					local nextLine = att.frames[startLine - 1]
					if (nextLine and nextLine.a and nextLine.a.timestamp > entry.time and entry.time > startTime) then
						break
					end
					startLine = startLine - 1
					startTime = att.frames[startLine] and att.frames[startLine].a.timestamp
				end
				if (not startTime) then
					entryNo, entry = nil, nil
				end
			end

			if (entry and entryNo) then
				local health, healthmax = entry.health, entry.healthmax

				if (startTime <= endTime) then
					for j = #att.frames,startLine + 1,-1 do
						local frame = att.frames[j]
						if (frame.texture) then
							frame.health = nil
							frame.texture:Hide()
							frame.healthText:Hide()
						end
					end

					for j = startLine,1,-1 do
						local frame = att.frames[j]

						while (true) do
							local entry2 = healthList[entryNo + 1]
							if (entry2 and entry2.time <= frame.a.timestamp) then
								entry = entry2
								entryNo = entryNo + 1
								health, healthmax = entry.health, entry.healthmax
								guess = nil
							else
								break
							end
						end

						if (guess) then
							if (j < startLine) then
								health, healthmax = self:UpdateHealth(j, health, healthmax)
							end
						end

						self:HealthBar(j, health, healthmax, guess)

						guess = true
					end
					return
				end
			end
		end
	end

	for i,frame in pairs(att.frames) do
		if (frame.texture) then
			frame.texture:Hide()
			frame.healthText:Hide()
			frame.health = nil
		end
	end
end

-- OptionalFilters
function gr:OptionalFilters(timestamp, event, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags, ...)
	if (event == "SPELL_ENERGIZE") then
		return false			-- Never want this
	end

	if (event == "SPELL_AURA_REMOVED" or event == "SPELL_AURA_APPLIED") then
		local auraType = select(4, ...)
		if (auraType == "BUFF") then
			return self.db.profile.buffs
		elseif (auraType == "DEBUFF") then
			return self.db.profile.debuffs
		end
	end

	return true
end

-- DoLines
function gr:DoLines()
	local unitName = self.lastName
	line = 0
	self.startTimeDelta = nil
	self.prevLineStamp = nil
	self.timeSpanStart = nil
	self.timeSpanEnd = nil
	self.firstTimeStamp = nil

	local gotValid = 0
	while (true) do
		local color
		local timestamp, event, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags = CombatLogGetCurrentEntry()
		if (dstName == unitName) then
			if (self:OptionalFilters(CombatLogGetCurrentEntry())) then
				if (not self.firstTimeStamp) then
					self.firstTimeStamp = timestamp
				end
				if (not self.startTimeDelta) then
					self.startTimeDelta = timestamp
				end

				gotValid = gotValid + 1

				if (gotValid == self.offset) then
					self.prevLineStamp = timestamp
				end

				if (gotValid > self.offset) then
					if (not self.timeSpanEnd) then
						self.timeSpanEnd = timestamp
					end

					self:AddLine(CombatLogGetCurrentEntry())
					if (line >= self.db.profile.lines) then
						self.timeSpanStart = timestamp
						break
					end
				end
			end
		end

		if (not CombatLogAdvanceEntry(-1)) then
			self.oldestTimeStamp = timestamp
			-- Hit top
			if (self.offset > 0) then
				self.offset = self.offset - 1
				return true, gotValid
			end
			break
		end
	end
	return false, gotValid
end

-- CreateScroll
function gr:CreateScroll()
	local att = self.attachment
	att.scroll = att:CreateTexture(nil, "OVERLAY")
	att.scroll:SetTexture("Interface\\Tooltips\\UI-Tooltip-Background")
	att.scroll:Hide()
	att.scroll:SetWidth(3)
	att.scroll:SetVertexColor(0, 0.9, 0.9)
end

-- DoScroll
function gr:DoScroll()
	local att = self.attachment

	if (self.db.profile.dockToTooltip) then
		if (att.scroll) then
			att.scroll:Hide()
		end
		return
	end

	if (not att.scroll) then
		self:CreateScroll()
	end

	-- Get time span of log so we can position the scroll bar
	CombatLogSetCurrentEntry(-1)
	local timestampEnd = self.firstTimeStamp	-- CombatLogGetCurrentEntry()
	CombatLogSetCurrentEntry(1)
	local timestampStart = CombatLogGetCurrentEntry()
	local timeSpan = timestampEnd and (timestampEnd - timestampStart)

	self:HouseCleaning(timestampStart)

	if (timestampEnd and self.timeSpanStart and self.timeSpanEnd) then
		local windowTimeSpan = self.timeSpanEnd - self.timeSpanStart
		local targetWindowHeight = 15 * min(line, self.db.profile.lines)

		local t = timestampStart
		timestampStart = timestampStart - t
		timestampEnd = timestampEnd - t
		self.timeSpanStart = self.timeSpanStart - t
		self.timeSpanEnd = self.timeSpanEnd - t

		local barSize = windowTimeSpan / timeSpan * targetWindowHeight
		local offsetTop = self.timeSpanStart / timestampStart * targetWindowHeight
		local offsetBottom = self.timeSpanEnd / timestampEnd * targetWindowHeight

		att.scroll:ClearAllPoints()
		att.scroll:SetHeight(barSize)
		if (self.db.profile.barsLeft) then
			att.scroll:SetPoint("BOTTOMLEFT", att, "BOTTOMRIGHT", 0, targetWindowHeight - offsetBottom + 8)
		else
			att.scroll:SetPoint("BOTTOMRIGHT", att, "BOTTOMLEFT", -0, targetWindowHeight - offsetBottom + 8)
		end

		att.scroll:Show()
	else
		att.scroll:Hide()
	end

	if (self.offset > 0) then
		local se = att.scrollEnd
		if (not se) then
			se = self:CreateScrollEnd()
		end
		att.scrollEnd:Show()

	elseif (att.scrollEnd) then
		att.scrollEnd:Hide()
	end
end

-- CreateScrollEnd
function gr:CreateScrollEnd()
	local att = self.attachment
	local se = CreateFrame("Button", nil, att)

	se:SetNormalTexture("Interface\\ChatFrame\\UI-ChatIcon-ScrollEnd-Up")
	se:SetPushedTexture("Interface\\ChatFrame\\UI-ChatIcon-ScrollEnd-Down")
	se:SetDisabledTexture("Interface\\ChatFrame\\UI-ChatIcon-ScrollEnd-Disabled")
	se:SetHighlightTexture("Interface\\Buttons\\UI-Common-MouseHilight", "ADD")
	se:SetWidth(32)
	se:SetHeight(32)

	se:SetScript("OnClick",
		function(self)
			gr.offset = 0
			gr:Tip()
			gr:TriggerEvent("GrimReaper_Scrolled")
		end
	)

	se:SetPoint("TOPRIGHT", att, "BOTTOMRIGHT", 2, 4)

	att.scrollEnd = se
	return se
end

-- Tip
--function gr:Tip(unit, tooltip)
function gr:Tip(timestamp, event, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags, ...)
	local prevUnit = self.lastUnit
	local unit, tooltip
	if (type(timestamp) == "number") then
		-- 1 single new event
	else
		unit, tooltip = timestamp, event
		timestamp, event = nil, nil
	end

	local att = self.attachment
	if (not self.db or not self.db.profile.enabled or self.db.profile.hide) then
		if (att) then
			att:Hide()
		end
		return
	end

	if (not unit) then
		unit = self.lastUnit
		tooltip = self.lastTip
	else
		if (self.locked) then
			return
		end
	end
	self.lastTip = tooltip or GameTooltip

	if (not unit) then
		return
	end
	local mask = (UnitIsUnit(unit, "player") and 1) or (UnitInParty(unit) and 2) or (UnitInRaid(unit) and 4)
	if (not mask) then
		return
	end

	if (not att) then
		att = self:CreateAttachment()
	end
	self:SetAttachPoint()

	if (self.emptyText) then
		self.emptyText:Hide()
	end

	local oldShownLines = self.shownLines
	line = 0

	if (timestamp and self.offset == 0) then
		local shown = self:ShuffleListUpward(timestamp)
		self:AddLine(timestamp, event, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags, ...)

		line = shown + 1
		self:DoHealth()
	else
		if (CombatLogUpdateFrame and CombatLogUpdateFrame.refiltering and CombatLogQuickButtonFrame_CustomProgressBar) then
			if (not self:IsHooked(CombatLogQuickButtonFrame_CustomProgressBar, "OnHide")) then
				self:HookScript(CombatLogQuickButtonFrame_CustomProgressBar, "OnHide",
					function(self)
						gr:Tip()
						gr:Unhook(CombatLogQuickButtonFrame_CustomProgressBar, "OnHide")
					end)
			end
			return
		end

		local oldFilter = self:CombatLogSaveFilter()

		unit = self:FixUnit(unit)
		local unitName = UnitName(unit)
		self.lastName = unitName
		local changed
		if (unit ~= self.lastUnit) then
			self.lastUnit = unit
			changed = true
		end

		local preCount = CombatLogGetNumEntries()
		noSaveFilter = true
		CombatLogResetFilter()
		CombatLogAddFilter(self.eventString, nil, mask + 0x10 + 0x100 + 0x400)
		noSaveFilter = nil

		local valid = CombatLogSetCurrentEntry(-1)
		local count = CombatLogGetNumEntries()

		if (valid and count > 0) then
			local hitTop, gotLines = self:DoLines()
			if (hitTop) then
				if (self.oldestTimeStamp) then
					self:HouseCleaning(self.oldestTimeStamp)
				end
				CombatLogSetCurrentEntry(-1)

				-- Need to redo because we hit the top
				hitTop, gotLines = self:DoLines()
				if (gotLines == 0) then
					self:NothingToShow()
				end
			end

			if (gotLines > 0) then
				self:FinalizeLines()
				self:DoHealth()
				self:DoScroll()
			else
				self:NothingToShow()
			end
		else
			if (self.db.profile.dockToTooltip) then
				att:Hide()
				self:CombatLogLoadFilter(oldFilter)
				return

			elseif (not self.shownLines) then
				self:NothingToShow()
				return

			else
				self:CombatLogLoadFilter(oldFilter)
				self.lastUnit = prevUnit
				return
			end
		end

		self:Title()
		self:CombatLogLoadFilter(oldFilter)

		if (changed) then
			self:TriggerEvent("GrimReaper_UnitChanged", unit)
		end
	end

	self.shownLines = min(line,self.db.profile.lines)
	att:SetHeight(15 * max(1, self.shownLines) + 17)
	att.anchor:SetHeight(15 * max(1, self.shownLines) + 17)
	att:Show()

	self:TriggerEvent("GrimReaper_Update")
end

-- NothingToShow
function gr:NothingToShow()
	local att = self.attachment
	if (att.scroll) then
		att.scroll:Hide()
	end

	if (self.db.profile.dockToTooltip) then
		att:Hide()
	else
		-- Show blank reaper
		self:Title()

		for k,v in pairs(att.frames) do
			v:Hide()
		end

		if (not self.emptyText) then
			self.emptyText = att:CreateFontString(nil, "ARTWORK", "GameFontNormal")
			self.emptyText:SetAllPoints(att)
			self.emptyText:SetTextColor(0.5, 0.5, 0.5)
			self.emptyText:SetText(L["NOINFO"])
		end
		self.emptyText:Show()

		att:SetHeight(42)
		att.anchor:SetHeight(42)
		att:Show()
		line = 1
	end

	self.shownLines = 0
	att:SetHeight(15 + 17)
	att.anchor:SetHeight(15 + 17)
end

-- Output
function gr:Output(...)
	local channel = self.db.profile.channel
	if (channel == "RAID" and GetNumRaidMembers() == 0) then
		channel = "PARTY"
	end
	if (channel == "PARTY" and GetNumPartyMembers() == 0) then
		return
	end
	if (channel == "CHANNEL" and GetChannelName(self.db.profile.channelIndex) == 0) then
		return
	end
	if (channel == "SELF") then
		DEFAULT_CHAT_FRAME:AddMessage(format(...))
	else
		SendChatMessage(format(...), channel, nil, self.db.profile.channelIndex)
	end
end

-- SetChannel
function gr:SetChannelRaw(chan, ind)
	self.db.profile.channel = chan
	self.db.profile.channelIndex = ind
end

-- SetChannel
function gr:SetChannel(chan)
	local oldChannel = self.db.profile.channel
	if (chan) then
		self.db.profile.channel = strupper(chan)

		if (tonumber(self.db.profile.channel) and tonumber(self.db.profile.channel) > 0) then
			if (GetChannelName(self.db.profile.channelIndex) > 0) then
				self.db.profile.channelIndex = tonumber(self.db.profile.channel)
				self.db.profile.channel = "CHANNEL"
			else
				self.db.profile.channel = oldChannel
				self:Print(REAPER_CHANNEL_INVALID)
				return
			end

		elseif (not (self.db.profile.channel == "RAID" or self.db.profile.channel == "PARTY" or self.db.profile.channel == "GUILD" or self.db.profile.channel == "SAY" or self.db.profile.channel == "BATTLEGROUND" or self.db.profile.channel == "YELL" or self.db.profile.channel == "OFFICER")) then
			local c = self.db.profile.channel
			self.db.profile.channelIndex = self.db.profile.channel
			self.db.profile.channel = "WHISPER"
			for i = 1,10 do
				local used, name = GetChannelName(i)
				if (name) then
					if (strlower(name) == strlower(c)) then
						self.db.profile.channelIndex = name
						self.db.profile.channel = "CHANNEL"
						break
					end
				end
			end
		end
	end

	if (not self.db.profile.channel) then
		self.db.profile.channel = "RAID"
	end

	self:Print(REAPER_CHANNEL, self:GetChannelDisplay())
end

-- GetChannelDisplay
function gr:GetChannelDisplay()
	local c, ret
	if (self.db.profile.channel == "WHISPER") then
		local color
		for unit in roster:IterateRoster() do
			if (strlower(unit.name) == strlower(self.db.profile.channelIndex)) then
				color = classColour[unit.class]
				break
			end
		end

		ret = (getglobal(self.db.profile.channel) or self.db.profile.channel).." "..(color or "")..self.db.profile.channelIndex.."|r"

	elseif (self.db.profile.channel == "CHANNEL") then
		local id, name = GetChannelName(self.db.profile.channelIndex)
		if (name) then
			local t = ChatTypeInfo["CHANNEL"..self.db.profile.channelIndex]
			if (t) then
				c = {r = t.r, g = t.g, b = t.b}
			end
			ret = name
		end
	else
		ret = getglobal("CHAT_MSG_"..(self.db.profile.channel or "RAID")) or self.db.profile.channel or "RAID"
	end

	if (not ret) then
		ret = "RAID"
	end

	if (not c) then
		if (ChatTypeInfo[self.db.profile.channel]) then
			local t = ChatTypeInfo[self.db.profile.channel]
			if (t) then
				c = {r = t.r, g = t.g, b = t.b}
			end
		end
	end

	if (c) then
		ret = format("|c00%02X%02X%02X%s|r", c.r * 255, c.g * 255, c.b * 255, ret)
	end

	return ret
end

-- Report
function gr:ReportShown()
	local att = self.attachment
	if (not att) then
		return
	end

	local unitName = UnitName(self.lastUnit)
	self:Output(L["REPORTTITLE"], unitName)

	local lastTime = att.frames and att.frames[1] and att.frames[1].a and att.frames[1].a.timestamp

	for i = #att.frames,1,-1 do
		local frame = att.frames[i]
		if (frame and frame:IsShown()) then
			local a = frame.a
			local time, msg, health
			local suffix = ""

			if (i == 1) then
				time = format("%s.%03d>", date(self.db.profile.timeformat, a.timestamp), a.timestamp * 1000 % 1000)
			else
				local t
				if (self.db.profile.deltaTime == 2) then
					local f = att.frames[i - 1]
					local nextTimeStamp = f and f.a and f.a.timestamp
					if (nextTimeStamp) then
						t = a.timestamp - nextTimeStamp
					else
						t = a.timestamp - lastTime
					end
				else
					t = a.timestamp - lastTime
				end
				local temp = self:MakeDeltaTime(t)..">"
		        time = strrep(" ", 21 - (strlen(temp) * 3 / 2))..temp
			end

			msg = frame.text:GetText()
			msg = msg:gsub("|c[0-9A-F][0-9A-F][0-9A-F][0-9A-F][0-9A-F][0-9A-F][0-9A-F][0-9A-F]", "")
			msg = msg:gsub("|r", "")

			if (a.event == "SPELL_HEAL" or a.event == "SPELL_PERIODIC_HEAL") then
				suffix = L["Heal"]
			else
				if (a.school and a.school > 1) then
					suffix = schools[a.school]
				end
			end
			if (a.modifier) then
				local Mod = getglobal(format("TEXT_MODE_A_STRING_RESULT_%s%s", a.modifier, a.spellName and "_SPELL" or ""))
				if (Mod) then
					suffix = format("%s (%s)", suffix, Mod)
				end
			end

			health = frame.health

			if (health) then
				if (health == "DEAD") then
					local dead = format("_____%s_____", DEAD)
					health = strsub(dead, strlen(dead) / 2 - 4, strlen(dead) / 2 + 5)
				else
					local c = health / 10
					health = strrep("#", c)..strrep("_", 10 - c)
				end
			else
				health = strrep("_", 10)
			end

			local icon = self:UnitIcon(a.srcFlags, true)
			self:Output("%s %s %s %s%s", health, time, msg, suffix, icon)
		end
	end

	return
end

-- Lock
function gr:Lock()
	self.cycleState = nil
	if (self.lastUnit or self.locked) then
		self.locked = not self.locked
		if (self.locked) then
			self:Print(L["LOCKED"], self:ColourUnit(self.lastUnit))
		end
		self:Tip()
	end
end

-- DisableXPerlGrimReaper
function gr:DisableXPerlGrimReaper()
	if (GetAddOnInfo("XPerl_GrimReaper")) then
		DisableAddOn("XPerl_GrimReaper")
		if (XPerl_GrimReaper_Config) then
			XPerl_GrimReaper_Config.enabled = false
		end
		if (XPerl_GrimReaper) then
			XPerl_GrimReaper.Tip = function() end
			XPerl_GrimReaper.lastUnit = nil
			XPerl_GrimReaper:Hide()
			XPerl_GrimReaper:UnregisterAllEvents()
			XPerl_GrimReaper:SetScript("OnEvent", nil)
			if (XPerl_GrimReaper.attachment) then
				XPerl_GrimReaper.attachment:Hide()
			end
		end
		SlashCmdList["XPERL_GRIMREAPER"] = nil
	end
	self.DisableXPerlGrimReaper = nil
end

-- ADDON_LOADED
function gr:ADDON_LOADED(addon)
	if (addon == "Blizzard_CombatLog") then
		playerMask = COMBATLOG_OBJECT_CONTROL_PLAYER
		groupMask = COMBATLOG_OBJECT_AFFILIATION_MINE + COMBATLOG_OBJECT_AFFILIATION_PARTY + COMBATLOG_OBJECT_AFFILIATION_RAID -- + COMBATLOG_OBJECT_CONTROL_PLAYER
		self:SetColours()
		self:MakeIconFlags()

	elseif (addon == "XPerl_GrimReaper") then
		if (self.DisableXPerlGrimReaper) then
			self:DisableXPerlGrimReaper()
		end
	end
end

-- PLAYER_ENTERING_WORLD
function gr:PLAYER_ENTERING_WORLD()
	tokens = {}
end

-- ShowLog
function gr:ShowLog(which)
	if (self.lastUnit and UnitExists(self.lastUnit)) then
		Blizzard_CombatLog_UnitMenuClick(which, UnitName(self.lastUnit), UnitGUID(self.lastUnit))
	end
end

-- OnTooltipUpdate
function gr:OnTooltipUpdate()
	tablet:SetTitle(L["TITLECOLOUR"])
	tablet:SetHint(L["HINT"])
end

-- OnClick
function gr:OnClick()
	self.db.profile.hide = not self.db.profile.hide
	self.cycleState = nil
	self:Tip()
	self:DoIcon()
end

-- DoIcon
function gr:DoIcon()
	if (self.db.profile.hide) then
		gr.hasIcon = "Interface\\Addons\\GrimReaper\\Images\\Death"
	else
		gr.hasIcon = "Interface\\Addons\\GrimReaper\\Images\\DeathEyes"
	end
	self:SetIcon(gr.hasIcon)
end

-- MakeEventList
function gr:MakeEventList()
	-- Event list: true signifies damage/heal events which we'll want to check UNIT_HEALTH for on next update
	self.eventList = {
		SWING_DAMAGE = true, SWING_MISSED = false,
		RANGE_DAMAGE = true, RANGE_MISSED = false,
		SPELL_PERIODIC_DAMAGE = true, SPELL_PERIODIC_MISSED = false,
		SPELL_HEAL = true, SPELL_PERIODIC_HEAL = true,
		SPELL_DAMAGE = true, SPELL_MISSED = false,
		DAMAGE_SHIELD = true,
		ENVIRONMENTAL_DAMAGE = true,
		SPELL_AURA_REMOVED = false, SPELL_AURA_APPLIED = false,
		SPELL_INSTAKILL = false
	}

	if (self.db.profile.curesAndSteals) then
		gr.eventList.SPELL_AURA_DISPELLED = false
		gr.eventList.SPELL_AURA_STOLEN = false
	end

	if (self.db.profile.casts) then
		gr.eventList.SPELL_CAST_SUCCESS = false
	end

	local temp = new()
	for e in pairs(self.eventList) do
		tinsert(temp, e)
	end
	self.eventString = table.concat(temp, ",")
	del(temp)
end

-- ShowCombatLog
function gr:ShowCombatLog()
	local unit = _G[UIDROPDOWNMENU_INIT_MENU].name
	if (unit) then
		if (not UnitInRaid(unit) and not UnitInParty(unit)) then
			if (UnitName("target") == unit) then
				unit = "target"
			elseif (UnitName("focus") == unit) then
				unit = "focus"
			end
		end

		local guid = UnitGUID(unit)
		if (guid) then
			Blizzard_CombatLog_UnitMenuClick("BOTH", UnitName(unit), guid)
		end
	end
end

-- UnitPopup_ShowMenu()
local function UnitPopup_ShowMenu(dropdownMenu, which, unit, name, userData, ...)
	if (UIDROPDOWNMENU_MENU_LEVEL > 1) then
		return
	end

	local addCLOption
	if (which == "RAID_TARGET_ICON") then
		for i = 2,DropDownList1.numButtons do
			getglobal("DropDownList1Button"..i):Hide()
		end
		DropDownList1.numButtons = 1

		if (GetNumRaidMembers() > 0 or GetNumPartyMembers() > 0) then
			local info = UIDropDownMenu_CreateInfo()
			info.text = RAID_TARGET_ICON
			info.dist = 0
			info.value = "RAID_TARGET_ICON"
			info.hasArrow = 1
			info.notCheckable = 1
			UIDropDownMenu_AddButton(info, UIDROPDOWNMENU_MENU_LEVEL)
		end
		
		addCLOption = true

	elseif (which == "RAID" or which == "PLAYER" or which == "PARTY" or which == "SELF" or which == "RAID_PLAYER") then
		DropDownList1.numButtons = max(0, DropDownList1.numButtons - 1)
		addCLOption = true
	end
	
	if (addCLOption) then
		local info = UIDropDownMenu_CreateInfo()
		info.text = "|cFF80FF80"..COMBAT_LOG
		info.dist = 0
		info.func = gr.ShowCombatLog
		info.notCheckable = 1
		UIDropDownMenu_AddButton(info, UIDROPDOWNMENU_MENU_LEVEL)

		info = UIDropDownMenu_CreateInfo()
		info.text = CANCEL
		info.owner = which
		info.dist = 0
		info.func = UnitPopup_OnClick
		info.notCheckable = 1
		UIDropDownMenu_AddButton(info, UIDROPDOWNMENU_MENU_LEVEL)
	end
end

-- CombatLogMenus
function gr:CombatLogMenus()
	hooksecurefunc("UnitPopup_ShowMenu", UnitPopup_ShowMenu)
	self.CombatLogMenus  = nil
end

-- Hold (Keybinding function)
function gr:Hold(keystate)
	if (not self.db.profile.dockToTooltip) then
		if (self.db.profile.hide or self.cycleState) then
			local tip
			if (self.db.profile.hide and keystate == "down") then
				self.db.profile.hide = nil
				self.cycleState = 1
				if (UnitInParty("mouseover") or UnitInRaid("mouseover")) then
					self.lastUnit = "mouseover"
				elseif (not self.lastUnit) then
					self.lastUnit = "player"
				end
				gr:Tip()
				tip = L["Select the player to view, then release the hold key"]
			
			elseif (self.cycleState == 1 and keystate == "up") then
				self.locked = true
				self:Tip()
				self.cycleState = 2
				tip = L["Press the Hold key once more to hide Grim Reaper"]

			elseif (self.cycleState == 2) then
				self.locked = nil
				self.db.profile.hide = true
				if (self.attachment) then
					self.attachment:Hide()
				end
				if (gr.extraInfoTip) then
					gr.extraInfoTip:Hide()
				end
			end

			if (tip and self.attachment) then
				gr:ShowLineTip(tip)
			end
		end
	end
end

-- OnInitialize
function gr:OnInitialize()
	if (self.DisableXPerlGrimReaper) then
		self:DisableXPerlGrimReaper()
	end

	self:RegisterDB("GrimReaperDB")
	self:RegisterDefaults("profile", {
		dockToTooltip	= false,
		scale			= 0.6,
		lines			= 15,
		bars			= true,
		barsInside		= false,
		barsLeft		= true,
		enabled			= true,
		dockPoint		= "BOTTOMLEFT",
		healthList		= {},
		frameWidth		= 200,
		blizzardColours	= true,
		buffs			= true,
		debuffs			= true,
		channel			= "RAID",
		bartexture		= "BantoBar",
		buffTips		= false,
		debuffTips		= true,
		blizzLines		= false,
		casts			= false,
		curesAndSteals	= true,
		showtime		= true,
		timeformat		= "%X",
	} )
	self:RegisterChatCommand("/grim", self.options)

	self:SetColours()

	hooksecurefunc(GameTooltip, "SetUnit",
		function(self, unit)
			if (gr:IsActive()) then
				gr:Tip(unit, self)
			end
		end)
		
	self:CombatLogMenus()

	self.OnInitialize = nil
end

-- OnEnable
function gr:OnEnable()
	self.bartexture = SM:Fetch("statusbar", self.db.profile.bartexture)

	local events = {"RAID_ROSTER_UPDATE", "PARTY_MEMBERS_CHANGED", "UNIT_HEALTH", "COMBAT_LOG_EVENT_UNFILTERED", "ADDON_LOADED", "PLAYER_ENTERING_WORLD"}
	for i,e in pairs(events) do
		self:RegisterEvent(e)
	end

	self:MakeEventList()
	self:SetColours()
	self:MakeIconFlags()
	self:Tip()
	self:DoIcon()
end

-- OnDisable
function gr:OnDisable()
	self.eventString = nil
	self.iconFlags = nil
	self.bartexture = nil
	schoolColour = nil
	schoolColourHex = nil
end
