local MAJOR_VERSION = "LibShieldLeft-1.0"
local MINOR_VERSION = tonumber(("$Revision: 5 $"):match("%d+"))

local lib = LibStub:NewLibrary(MAJOR_VERSION, MINOR_VERSION)
if not lib then return end

local playerName = UnitName('player')
local playerClass = select(2, UnitClass('player'))
local isShielder = (playerClass == "PRIEST") or (playerClass == "MAGE")

if not isShielder then return end

-----------------
-- Event Frame --
-----------------

lib.EventFrame = lib.EventFrame or CreateFrame("Frame")
lib.EventFrame:SetScript("OnEvent", function (this, event, ...) lib[event](lib, ...) end)
lib.EventFrame:UnregisterAllEvents()

-- Register Events
lib.EventFrame:RegisterEvent("LEARNED_SPELL_IN_TAB")
lib.EventFrame:RegisterEvent("COMBAT_LOG_EVENT_UNFILTERED")

-- Prune data at zone change
lib.EventFrame:RegisterEvent("PLAYER_ENTERING_WORLD")


----------------------
-- Scanning Tooltip --
----------------------

if (not lib.Tooltip) then
    lib.Tooltip = CreateFrame("GameTooltip")
    lib.Tooltip:SetOwner(UIParent, "ANCHOR_NONE")
    for i = 1, 4 do
        lib["TooltipTextLeft" .. i] = lib.Tooltip:CreateFontString()
        lib["TooltipTextRight" .. i] = lib.Tooltip:CreateFontString()
        lib.Tooltip:AddFontStrings(lib["TooltipTextLeft" .. i], lib["TooltipTextRight" .. i])
    end
end


-------------------------------
-- Embed CallbackHandler-1.0 --
-------------------------------

lib.Callbacks = LibStub("CallbackHandler-1.0"):New(lib)


-----------------
-- Static Data --
-----------------

-- Cache of spells and shield sizes
local SpellCache = {}

local Shields = {
    --Power World: Shield
    [GetSpellInfo(17)] = {
        name = GetSpellInfo(17),
        duration = 30,
        order = 30,
    },
    --Divine Aegis
    [GetSpellInfo(47753)] = {
        name = GetSpellInfo(47753),
        duration = 12,
        order = 12,
    },
    --Ice Barrier
    [GetSpellInfo(13032)] = {
        name = GetSpellInfo(13032),
        duration = 60,
        order = 100,
    },
    --Mana Shield
    [GetSpellInfo(10193)] = {
        name = GetSpellInfo(10193),
        duration = 60,
        order = 150,
    },
    --Fire Ward
    [GetSpellInfo(43010)] = {
        name = GetSpellInfo(43010),
        duration = 30,
        order = 0,
        schools = 0x04, --fire
    },
    --Frost Ward
    [GetSpellInfo(6143)] = {
        name = GetSpellInfo(6143),
        duration = 30,
        order = 0,
        schools = 0x10, --frost
    },
    --Hand of protection
    [GetSpellInfo(5599)] = {
        name = GetSpellInfo(5599),
        duration = 8,
        order = 0,
    },
    --Divine shield
    [GetSpellInfo(642)] = {
        name = GetSpellInfo(642),
        duration = 12,
        order = 0,
    },
    --Sacred Shield
    [GetSpellInfo(53601)] = {
        name = GetSpellInfo(53601),
        duration = 6,
        order = 6,
    },
    --Shadow ward 6229
    [GetSpellInfo(6229)] = {
        name = GetSpellInfo(6229),
        duration = 30,
        order = 0,
        shool = 0x20,
    },
}

local EventParse =
{
	
}

-- active Shields
local unitData = {}

local currentTimestamp = 0

---------------------------------
-- Frequently Accessed Globals --
---------------------------------
local bit_band = _G.bit.band


---------------
-- Utilities --
---------------

local function unitFullName(unit)
    local name, realm = UnitName(unit)
    if (realm and realm ~= "") then
        return name .. "-" .. realm
    else
        return name
    end
end

-- Spellbook Scanner --
local function getBaseShieldSize(name)

    -- Check if info is already cached
    if (SpellCache[name]) then
        return SpellCache[name]
    end

    SpellCache[name] = {}

    -- Gather info (only done if not in cache)
    local i = 1

    while true do

        local spellName, spellRank = GetSpellName(i, BOOKTYPE_SPELL)
        
        if (not spellName) then 
            break 
        end

        if (spellName == name) then
            -- This is the spell we're looking for, gather info

            -- Determine rank
            spellRank = tonumber(spellRank:match("(%d+)"))
            lib.Tooltip:SetSpell(i, BOOKTYPE_SPELL)
    
            -- Determine healing
            local shieldSize = select(3, string.find(lib.TooltipTextLeft4:GetText() or lib.TooltipTextLeft3:GetText() or "", "[(absorb)]+.-(%d+).-[(damage)(Schaden)]+"))
            
            SpellCache[spellName][spellRank] = shieldSize
        end
        i = i + 1
    end

    return SpellCache[name]
end


-- Detects if a buff is present on the unit and returns the application number
local function detectBuff(unit, buffName)
    for i = 1, 40 do
        local name, _, _, count = UnitBuff(unit, i)
        if (not name) then
            return false;
        end
        if (name == buffName) then
            return count
        end
    end
end

local function getSpellBonusHealingPenalty(spellLevel, playerLevel)
    if (not spellLevel or ((spellLevel + 6) > playerLevel)) then
        return 1.0
    else
        return (spellLevel + 6) / playerLevel;
    end
end

local function isInPartyOrRaidOrMe(unitFlags)
    return  bit_band(unitFlags, COMBATLOG_OBJECT_AFFILIATION_RAID) == COMBATLOG_OBJECT_AFFILIATION_RAID or 
            bit_band(unitFlags, COMBATLOG_OBJECT_AFFILIATION_PARTY) == COMBATLOG_OBJECT_AFFILIATION_PARTY or 
            bit_band(unitFlags, COMBATLOG_OBJECT_AFFILIATION_MINE) == COMBATLOG_OBJECT_AFFILIATION_MINE 
end

-----------------------------
-- Unit Data Management --
-----------------------------

local function getUnitData(unitName, dontCreate)
    local data = unitData[unitName]
    if not data and not dontCreate then
        data = {}
        unitData[unitName] = data
    end
    
    return data
end

local function getUnitActiveShields(unitName, dontCreate)
    local data = getUnitData(unitName, dontCreate)
    
    if not data then return end
    
    local activeShields = data.activeShields
        
    if not activeShields and not dontCreate then
        activeShields = {amount = 0, count = 0}
        data.activeShields = activeShields
    end
    
    return activeShields
end

----------------------
-- Public Functions --
----------------------


--------------------
-- Class Specific --
--------------------

local ClassSpecific = {}

-- Mage --

if (playerClass == "MAGE") then

    ClassSpecific.GetAmountForShield = function(shieldData, target)        
        return shieldData.amount
    end
    
    ClassSpecific.GetDurationForShield = function(shieldData, target)        
        return shieldData.duration
    end
end


-- Priest --

if (playerClass == "PRIEST") then
    local PWShield = GetSpellInfo(17)
    local DivineAegis = GetSpellInfo(47753)

    ClassSpecific.Heal = function (lib, timestamp, eventtype, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags, spellId, spellName, spellSchool, amount, overhealing, critical)
        if not srcName then
            print("Got a heal withouth source on "..dstName)
            return
        end
        if critical and srcName == playerName then
            --print("Crit heal with"..amount.." and "..(overhealing or 0).." overheal")
            local data = getUnitData(dstName)
            
            data.lastCritAmount = amount
            data.lastCritTime = currentTimestamp
        end
    end

    EventParse["SPELL_HEAL"] = ClassSpecific.Heal
    EventParse["SPELL_PERIODIC_HEAL"] = ClassSpecific.Heal
    EventParse["SPELL_BUILDING_HEAL"] = ClassSpecific.Heal
    EventParse["RANGE_HEAL"] = ClassSpecific.Heal
    
    ClassSpecific.GetDurationForShield = function(shieldData, target)        
        return shieldData.duration
    end

    ClassSpecific.GetAmountForShield = function(shieldData, target)   
        if shieldData == Shields[PWShield] then
            local name, rank, icon, count, debuffType, duration, expirationTime, isMine, isStealable = UnitBuff(target, shieldData.name)
            if not name then
                print("Error: Could not find "..shieldData.name.." as aura on unit!")
                return
            end
            
            if not isMine then
                return
            end
            
            rank = tonumber(rank:match("(%d+)"))
            
            local base = getBaseShieldSize(shieldData.name)[rank]
            if not base then 
                print("Error: Could not find base shield size for "..shieldData.name)
                return
            end
            
            local bonus = GetSpellBonusHealing();

            -- Borrowed Time - increases absorbed value by 8/16/24/32/40% of spell power
            local _, _, _, _, talentBrwdTime = GetTalentInfo(1,27);   
            talentBrwdTime = bonus * talentBrwdTime * 0.08
            
            -- Improved Power word: Shield - increases absorbed value by 5/10/15%
            local _, _, _, _, talentImprShield = GetTalentInfo(1,7);     
            talentImprShield = 1 + talentImprShield * 0.05
            
            return (base + talentBrwdTime + bonus * 0.75) * talentImprShield
            
        elseif shieldData == Shields[DivineAegis] then
            local data = getUnitData(target)
            
            local name, rank, icon, count, debuffType, duration, expirationTime, isMine, isStealable = UnitBuff(target, shieldData.name)
            if not name then
                print("Error: Could not find "..shieldData.name.." as aura on unit!")
                return
            end
            
            if not isMine then
                return
            end
            
            if not data.lastCritTime or currentTimestamp - data.lastCritTime > 0.8 then
                return
            end
            
            -- Divine Aegis - adds an aegis which absorbes 10/20/30% of last crit heal
            local _, _, _, _, talentDivAegis = GetTalentInfo(1,24);  
            if not talentDivAegis or talentDivAegis == 0 then return end
            
            return data.lastCritAmount * talentDivAegis * 0.1
        else
            return shieldData.amount
        end
    end

end

function lib:NewShield(shieldData, unitName)
    --get active shields of the unit
    local activeShields = getUnitActiveShields(unitName)
    
    --get head of shield list
    local shield = activeShields.first
    
    if not shield  or shield.data.order > shieldData.order then --shield list is empty or our new entry is smallest, add new shield as first
        shield = {data = shieldData} --create new entry
        shield.next = activeShields.first --set current first as next of new entry
        activeShields.first = shield --new entry becomes new current first
    else --otherwise find position to insert new shieldData entry
        while shield.data ~= shieldData and shield.next and shield.data.order <= shieldData.order do
            shield = shield.next
        end
        
        if shield.data ~= shieldData then
            local buff = shield.next --buffer successor
            shield.next = {data = shieldData} --create new entry as successor
            shield = shield.next --go to new entry
            shield.next = buff --set buffer as next of new entry
        else
            print("Error: New shield already in list")
        end
    end
    --in any case "shield" now points to the new entry for the shield    
    
    local newAmount = ClassSpecific.GetAmountForShield(shieldData, unitName)
    if newAmount then
        shield.amountLeft = newAmount
        activeShields.amount = activeShields.amount + shield.amountLeft
    end
    shield.expTime = ClassSpecific.GetDurationForShield(shieldData)
    activeShields.count = activeShields.count + 1
    self.Callbacks:Fire("ShieldLeft_NewShield", unitName, shieldData.name, shield.amountLeft, activeShields.amount, activeShields.count)
end

function lib:RefreshShield(shieldData, unitName, expTime)
    --get active shields of the unit
    local activeShields = getUnitActiveShields(unitName)
    
    --get head of shield list
    local shield = activeShields.first
    
    if not shield  or shield.data.order > shieldData.order then --shield list is empty or our new entry is smallest, add new shield as first
        shield = {data = shieldData} --create new entry
        shield.next = activeShields.first --set current first as next of new entry
        activeShields.first = shield --new entry becomes new current first
    else --otherwise find position to insert new shieldData entry
        while shield.data ~= shieldData and shield.next and shield.data.order <= shieldData.order do
            shield = shield.next
        end
        
        if shield.data ~= shieldData then
            local buff = shield.next --buffer successor
            shield.next = {data = shieldData} --create new entry as successor
            shield = shield.next --go to new entry
            shield.next = buff --set buffer as next of new entry
        end
    end
    --in any case "shield" now points to the new entry for the shield
    
    local newAmount = ClassSpecific.GetAmountForShield(shieldData, unitName)
    if newAmount then
        activeShields.amount = activeShields.amount + (newAmount - (shield.amountLeft or 0))
        shield.amountLeft = newAmount
    end
    shield.expTime = ClassSpecific.GetDurationForShield(shieldData)
    self.Callbacks:Fire("ShieldLeft_RefreshShield", unitName, shieldData.name, shield.amountLeft, activeShields.amount, activeShields.count)
end

function lib:RemoveShield(shieldData, unitName)
    --get active shields of the unit
    local activeShields = getUnitActiveShields(unitName, true)
    if not activeShields then
        return
    end
    
    --get head of shield list
    local shield = activeShields.first
    
    if not shield or shield.data.order > shieldData.order then  --shield list is empty or our new entry is smallest
        return                                                  --shield is not in list
    end
    
    if shield and shield.data == shieldData then --shield to be removed is first
        activeShields.first = shield.next
        
    else
        --find predecessor of the shield to be removed
        while shield.next and shield.next.data ~= shieldData and shield.next.data.order <= shieldData.order do
            shield = shield.next
        end
        
        if not shield.next or shield.next.data ~= shieldData then --shield not in list
            return
        end
        
        --remove shield from list
        local buff = shield.next
        shield.next = buff.next
        shield = buff
    end
    
    --true if shield is removed before its original time out
    local preTimeout = not shield.expTime or (currentTimestamp - shield.expTime) > 0.3 
    
    
    local lastAbsorb = getUnitData(unitName).lastAbsorb     
    --assume it broken (means absorbed everything it could) if it is pre timeout and has absorbed something in the last moment
    local broken = lastAbsorb and (currentTimestamp - lastAbsorb) < 0.3 and preTimeout
    
    
    if broken then
        if shield.forwardAbsorb then
            self:UnitAbsorbed(unitName, shield.forwardAbsorb, shield.data.school or 0x01, false)
        end
    else
        --the remove message of some shields like divine aegis comes before the damage message which removed it
        --if this is the case and there are other shields left we forward the amount left to them 
        --(because they will get the damage message this one already absorbed)
        --this is not the nicest way but works
        if shield.amountLeft and shield.amountLeft > 0 then
            if shield.next and preTimeout then
                getUnitData(unitName).lastRemove = currentTimestamp
                shield.next.forwardCap = shield.amountLeft
            end
            activeShields.amount = activeShields.amount - shield.amountLeft
        end
    end
    
    activeShields.count = activeShields.count - 1
    if activeShields.count == 0 then
        activeShields.amount = 0
    end
    self.Callbacks:Fire("ShieldLeft_RemoveShield", unitName, shieldData.name, shield.amountLeft, activeShields.amount, activeShields.count)
end

function lib:UnitAbsorbed(unitName, amount, school, partly)
    --get active shields of the unit
    --print("absorb:"..amount)
    local activeShields = getUnitActiveShields(unitName, true)
    if not activeShields then
        --print("no active shield")
        return
    end
    
    --get head of shield list
    local shield = activeShields.first
    
    while shield and shield.data.school and shield.data.school ~= school do
        shield = shield.next
    end
    
    if not shield then  --shield list is empty or our new entry is smallest
        --print("no shield or no appliable")
        return          --shield is not in list
    end
    
    local time = currentTimestamp
    if shield.forwardCap then
        local lastRemove = getUnitData(unitName).lastRemove
        if lastRemove and (time - lastRemove) < 0.3 then
            shield.amountLeft = shield.amountLeft + shield.forwardCap
            activeShields.amount = activeShields.amount + shield.forwardCap
        end
        shield.forwardCap = nil
    end
    getUnitData(unitName).lastAbsorb = time
    
    if shield.amountLeft then        
        shield.amountLeft = shield.amountLeft - amount
        shield.forwardAbsorb = min(shield.amountLeft, 0)
        activeShields.amount = activeShields.amount - amount
        --print("fire update "..activeShields.amount)
        self.Callbacks:Fire("ShieldLeft_UpdateShield", unitName, shield.data.name, shield.amountLeft, activeShields.amount, activeShields.count)
    else
        shield.forwardAbsorb = amount
    end
end

--------------------
-- Event Handlers --
--------------------

function lib:AuraApplied(timestamp, eventtype, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags, spellId, spellName, spellSchool, auraType)    
    local shieldData = Shields[spellName]
    
    if not shieldData then return end
    
    local fullName = unitFullName(dstName)
    if not fullName then return end
    
    self:NewShield(shieldData, fullName)
end

function lib:AuraRefresh(timestamp, eventtype, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags, spellId, spellName, spellSchool, auraType)    
    local shieldData = Shields[spellName]
    
    if not shieldData then return end
    
    local fullName = unitFullName(dstName)
    if not fullName then return end
    
    self:RefreshShield(shieldData, fullName)
end

function lib:AuraRemoved(timestamp, eventtype, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags, spellId, spellName, spellSchool, auraType)    
    local shieldData = Shields[spellName]
    
    if not shieldData then return end
    
    local fullName = unitFullName(dstName)
    if not fullName then return end
    
    self:RemoveShield(shieldData, fullName) 
end

function lib:AuraBroken(timestamp, eventtype, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags, spellId, spellName, spellSchool, auraType)    
    local shieldData = Shields[spellName]
    
    if not shieldData then return end
    
    local fullName = unitFullName(dstName)
    if not fullName then return end
    
    self:RemoveShield(shieldData, fullName)  
end

function lib:AuraBrokenSpell(timestamp, eventtype, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags, spellId, spellName, spellSchool, auraType)    
    local shieldData = Shields[spellName]
    
    if not shieldData then return end
    
    local fullName = unitFullName(dstName)
    if not fullName then return end
    
    self:RemoveShield(shieldData, fullName) 
end

function lib:SwingDamage(timestamp, eventtype, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags, amount, overkill, school, resisted, blocked, absorbed)  
    if absorbed and absorbed > 0 then
        local fullName = unitFullName(dstName)
        if not fullName then return end
    
        self:UnitAbsorbed(fullName, absorbed, 0x01, true)  
    end
end

function lib:SwingMissed(timestamp, eventtype, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags, missType, amount)    
    if missType == "ABSORB" then
        local fullName = unitFullName(dstName)
        if not fullName then return end
        
        self:UnitAbsorbed(fullName, amount, 0x01, false)  
    end
end

function lib:SpellDamage(timestamp, eventtype, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags, spellId, spellName, spellSchool, amount, overkill, school, resisted, blocked, absorbed)    
    if absorbed and absorbed > 0 then
        local fullName = unitFullName(dstName)
        if not fullName then return end
        
        self:UnitAbsorbed(fullName, absorbed, spellSchool, true)  
    end
end

function lib:SpellMissed(timestamp, eventtype, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags, spellId, spellName, spellSchool, missType, amount)    
    if missType == "ABSORB" then
        local fullName = unitFullName(dstName)
        if not fullName then return end
        
        self:UnitAbsorbed(fullName, amount, spellSchool, false)  
    end
end

function lib:LEARNED_SPELL_IN_TAB()
    -- Invalidate cached spell data when learning new spells
    SpellCache = {};       
end

EventParse["SPELL_AURA_APPLIED"] = lib.AuraApplied
EventParse["SPELL_AURA_REFRESH"] = lib.AuraRefresh
EventParse["SPELL_AURA_REMOVED"] = lib.AuraRemoved 
EventParse["SPELL_AURA_BROKEN"] = lib.AuraBroken
EventParse["SPELL_AURA_BROKEN_SPELL"] = lib.AuraBrokenSpell

EventParse["SWING_DAMAGE"] = lib.SwingDamage
EventParse["SWING_MISSED"] = lib.SwingMissed
--["ENVIRONMENTAL_DAMAGE"] = GridStatusShield.EnvironmentalDamage,
--["ENVIRONMENTAL_MISSED"] = GridStatusShield.EnvironmentalMissed,
EventParse["SPELL_DAMAGE"] = lib.SpellDamage
EventParse["SPELL_MISSED"] = lib.SpellMissed
EventParse["SPELL_PERIODIC_DAMAGE"] = lib.SpellDamage
EventParse["SPELL_PERIODIC_MISSED"] = lib.SpellMissed
EventParse["SPELL_BUILDING_DAMAGE"] = lib.SpellDamage
EventParse["SPELL_BUILDING_MISSED"] = lib.SpellMissed
EventParse["RANGE_DAMAGE"] = lib.SpellDamage
EventParse["RANGE_MISSED"] = lib.SpellMissed

function lib:COMBAT_LOG_EVENT_UNFILTERED(timestamp, eventtype, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags, ...)    
    if not isInPartyOrRaidOrMe(dstFlags) then
        return
    end
    
	local parsefunc = EventParse[eventtype]
	
	if parsefunc then  
        currentTimestamp = timestamp
		parsefunc(self, timestamp, eventtype, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags, ...)
	end
end

function lib:PLAYER_ENTERING_WORLD()
    --clear cache
    unitData = {}
end
