--[[
GridStatusParty.lua
Add grid indicators for party leader, raid leader, raid assistant, and master looter

This file is covered under the BSD-2.0 license.  See LICENSE.TXT for details
Authors: Byron Shelden (Dashkal), Pastamancer
(Annoy Dashkal for support)
]]--

-- Libraries
local L = AceLibrary("AceLocale-2.2"):new("GridStatusParty")
local RL = AceLibrary("Roster-2.1")

GridStatusParty = GridStatus:NewModule("GridStatusParty")
GridStatusParty.menuName = L["Party Flags"]

-- Locals for execution speed
local GridStatusParty = GridStatusParty

-- Defaults
GridStatusParty.defaultDB = {
	alert_partylead = {
		text = L["PL"],
		enable = true,
		color = { r = 0, g = 0.5, b = 1, a = 1 },
		priority = 25,
		range = false,
		icon = "Interface\\GroupFrame\\UI-Group-LeaderIcon",
	},
	alert_raidlead = {
		text = L["RL"],
		enable = true,
		color = { r = 1, g = 0.5, b = 0, a = 1 },
		priority = 25,
		range = false,
		icon = "Interface\\GroupFrame\\UI-Group-LeaderIcon",
	},
	alert_raidassist = {
		text = L["RA"],
		enable = true,
		color = { r = 1, g = 1, b = 0, a = 1 },
		priority = 23,
		range = false,
		icon = "Interface\\GroupFrame\\UI-Group-AssistantIcon",
	},
	alert_masterloot = {
		text = L["ML"],
		enable = true,
		color = { r = 1, g = 1, b = 1, a = 1 },
		priority = 24,
		range = false,
		icon = "Interface\\GroupFrame\\UI-Group-MasterLooter",
	},
}

-- Mod init
function GridStatusParty:OnInitialize()
	self.super.OnInitialize(self)
	self:Debug("OnInitialize")
	
	self:RegisterStatus("alert_partylead", L["Party: Party Leader"])
	self:RegisterStatus("alert_raidlead", L["Party: Raid Leader"])
	self:RegisterStatus("alert_raidassist", L["Party: Raid Assistant"])
	self:RegisterStatus("alert_masterloot", L["Party: Master Looter"])
end

-- Mod enable
function GridStatusParty:OnEnable()
	self:Debug("OnEnable")

	self:RegisterEvent("RosterLib_RosterUpdated", "Update")

	self:RegisterEvent("PARTY_LEADER_CHANGED", "UpdateParty")
	self:RegisterEvent("PARTY_LOOT_METHOD_CHANGED", "UpdateParty")

	self:Update()
end

function GridStatusParty:OnDisable()
	self:Debug("OnDisable")

	self:UnregisterEvent("RosterLib_RosterUpdated")
	self:UnregisterEvent("PARTY_LEADER_CHANGED")
	self:UnregisterEvent("PARTY_LOOT_METHOD_CHANGED")

	self:ClearRaidStatuses()
	self:ClearPartyStatuses()
end

-- Reset
function GridStatusParty:Reset()
	self.super.Reset(self)
	self:OnDisable()
	self:OnEnable()
end

function GridStatusParty:Update()
	self:Debug("Update")

	local inParty = GetNumPartyMembers() > 0
	local inRaid = GetNumRaidMembers() > 0

	if inRaid then
		self:ClearPartyStatuses()
		self:UpdateRaid()
	elseif inParty then
		self:ClearRaidStatuses()
		self:UpdateParty()
	else
		-- so lonely
		self:ClearRaidStatuses()
		self:ClearPartyStatuses()
	end
end

function GridStatusParty:ClearRaidStatuses()
	self:ClearAll("alert_raidlead")
	self:ClearAll("alert_raidassist")
	self:ClearAll("alert_masterloot")
end

function GridStatusParty:ClearPartyStatuses()
	self:ClearAll("alert_partylead")
	self:ClearAll("alert_masterloot")
end

function GridStatusParty:ClearAll(status)
	for name in self.core:CachedStatusIterator(status) do
		self.core:SendStatusLost(name, status)
	end
end

-- lookup for UpdateParty
local party_units = { [0] = "player" }
for i = 1, 4 do
	party_units[i] = "party" .. i
end

function GridStatusParty:UpdateParty()
	self:Debug("UpdateParty")

	local lootmethod, ml_partyID, ml_raidID = GetLootMethod()

	local settings_partylead = self.db.profile.alert_partylead
	local settings_masterloot = self.db.profile.alert_masterloot

	for i = 0, GetNumPartyMembers() do
		local unitid = party_units[i]
		local unitname = UnitName(unitid)

		if UnitIsPartyLeader(unitid) then
			self.core:SendStatusGained(unitname, "alert_partylead",
									   settings_partylead.priority,
									   settings_partylead.range,
									   settings_partylead.color,
									   settings_partylead.text,
									   nil, nil,
									   settings_partylead.icon)
		else
			self.core:SendStatusLost(unitname, "alert_partylead")
		end

		if lootmethod == "master" and i == ml_partyID then
			self.core:SendStatusGained(unitname, "alert_masterloot",
									   settings_masterloot.priority,
									   settings_masterloot.range,
									   settings_masterloot.color,
									   settings_masterloot.text,
									   nil, nil,
									   settings_masterloot.icon)
		else
			self.core:SendStatusLost(unitname, "alert_masterloot")
		end
	end
end

function GridStatusParty:UpdateRaid()
	self:Debug("UpdateRaid")

	local settings_raidlead = self.db.profile.alert_raidlead
	local settings_raidassist = self.db.profile.alert_raidassist
	local settings_masterloot = self.db.profile.alert_masterloot

	for u in RL:IterateRoster() do
		local unitname = u.name

		if u.rank == 2 then
			self.core:SendStatusLost(unitname, "alert_raidassist")
			self.core:SendStatusGained(unitname, "alert_raidlead",
									   settings_raidlead.priority,
									   settings_raidlead.range,
									   settings_raidlead.color,
									   settings_raidlead.text,
									   nil, nil,
									   settings_raidlead.icon)
		elseif u.rank == 1 then
			self.core:SendStatusLost(unitname, "alert_raidlead")
			self.core:SendStatusGained(unitname, "alert_raidassist",
									   settings_raidassist.priority,
									   settings_raidassist.range,
									   settings_raidassist.color,
									   settings_raidassist.text,
									   nil, nil,
									   settings_raidassist.icon)
		else
			self.core:SendStatusLost(unitname, "alert_raidlead")
			self.core:SendStatusLost(unitname, "alert_raidassist")
		end
		
		-- rosterlib only checks ML in raids
		if u.ML then
			self.core:SendStatusGained(unitname, "alert_masterloot",
									   settings_masterloot.priority,
									   settings_masterloot.range,
									   settings_masterloot.color,
									   settings_masterloot.text,
									   nil, nil,
									   settings_masterloot.icon)
		else
			self.core:SendStatusLost(unitname, "alert_masterloot")
		end
	end
end
