﻿if (not Grid) then return end

-- Libraries

local L = AceLibrary("AceLocale-2.2"):new("GridStatusHealPriority")
local BabbleSpell = LibStub:GetLibrary("LibBabble-Spell-3.0")
local BS = nil
if BabbleSpell then
	BS = BabbleSpell:GetLookupTable()
end
local Aura = AceLibrary("SpecialEvents-Aura-2.0")
local Banzai = nil
if AceLibrary:HasInstance("LibBanzai-2.0") then
	Banzai = AceLibrary("LibBanzai-2.0")
end
local RL = AceLibrary("Roster-2.1")
local Threat = LibStub("Threat-2.0")
local HealComm3 = LibStub:GetLibrary("LibHealComm-3.0", true)


-- New Grid status module

GridStatusHealPriority = GridStatus:NewModule("GridStatusHealPriority")
GridStatusHealPriority.menuName = L["HealPriority"]


-- AceDB defaults

GridStatusHealPriority.defaultDB = {
	alert_firstPriority = {
		text = L["1st"],
		enable = true,
		color = { r = 1, g = .2, b = 0, a = 1 },
		priority = 99,
		range = false,
	},
	alert_secondPriority = {
		text = L["2nd"],
		enable = true,
		color = { r = .6, g = .5, b = 0, a = .7 },
		priority = 99,
		range = false,
	},
}


-- Locals

local firstPriorityUnit = nil
local firstPriorityValue = 998
local secondPriorityUnit = nil
local secondPriorityValue = 999

local playerClass = select(2, UnitClass("player"))

local classModifiers = {
	["PRIEST"]   = { ["WARRIOR"] =  0, ["MAGE"] =  5, ["PRIEST"] =  5, ["DRUID"] =  5, ["PALADIN"] = 10, ["SHAMAN"] = 10, ["WARLOCK"] = 10, ["HUNTER"] = 10, ["ROGUE"] = 10 },
	["DRUID"]    = { ["WARRIOR"] =  5, ["MAGE"] = 10, ["PRIEST"] = 10, ["DRUID"] =  5, ["PALADIN"] =  0, ["SHAMAN"] =  5, ["WARLOCK"] =  5, ["HUNTER"] =  5, ["ROGUE"] = 10 },
	["PALADIN"]  = { ["WARRIOR"] =  5, ["MAGE"] = 10, ["PRIEST"] =  0, ["DRUID"] = 10, ["PALADIN"] =  5, ["SHAMAN"] =  5, ["WARLOCK"] = 10, ["HUNTER"] = 10, ["ROGUE"] =  5 },
	["SHAMAN"]   = { ["WARRIOR"] =  5, ["MAGE"] = 10, ["PRIEST"] = 10, ["DRUID"] = 10, ["PALADIN"] =  5, ["SHAMAN"] =  5, ["WARLOCK"] = 8, ["HUNTER"] = 10, ["ROGUE"] =  0 },
}

local antiHealing = {
	[BS["Mortal Strike"]] = true,          -- -50%, warriors
	[BS["Aimed Shot"]] = true,             -- -50%, hunters
	[BS["Blood Fury"]] = true,             -- -50%, orc racial
	[L["Brood Affliction: Green"]] = true, -- -50%, Chromaggus, BWL
	[L["Mortal Cleave"]] = true,           -- -50%, High Priest Thekal, ZG
	[L["Gehennas\' Curse"]] = true,        -- -75%, Gehennas, MC
	[L["Veil of Shadow"]] = true,          -- -75%, Nefarian, BWL
	[L["Necrotic Poison"]] = true,         -- -90%, Maexxna, Naxx
	-- Burning Crusade
	[L["Soul Strike"]] = true,             -- -50%, Ethereal Crypt Raider, Mana-Tombs
	[L["Solar Strike"]] = true,            -- -50%, Mechanaar
	[L["Magma-Thrower\'s Curse"]] = true,  -- -50%, Harbinger Skyriss, Arcatraz
	[L["Carrion Swarm"]] = true,           -- -75%, Anetheron, Hyjal
}

local antiHealingStackable = {
	[BS["Wound Poison"]] = true,           -- -10%, stacks up to 5x, rogues
	[L["Mortal Wound"]] = true,            -- -10%, stacks up to 7x, Kurinnaxx, Fankriss, Gluth, Gargolmar, Temporus
}

local healingPrevention = {
	[BS["Cyclone"]] = true,                -- druids
	[BS["Banish"]] = true,                 -- warlocks
	[L["Enfeeble"]] = true,                -- Malchezaar, Kara
	[L["Aura of Suffering"]] = true,       -- Essence of Souls, BT
}

local invulnerable = {
	[BS["Divine Protection"]] = true,
	[BS["Divine Shield"]] = true,
	[BS["Ice Block"]] = true,
	[L["Petrification"]] = true,           -- Flask of Petrification
}

local hots = {
	[BS["Renew"]] = true,
	[BS["Rejuvenation"]] = true,
	[BS["Regrowth"]] = true,
	[BS["Lifebloom"]] = true,
}

local battlegroundFlags = {
	[BS["Warsong Flag"]] = true,
	[BS["Silverwing Flag"]] = true,
	[BS["Netherstorm Flag"]] = true,
}

local subgroupModifier = {
	[1] = 0,
	[2] = 0,
	[3] = 0,
	[4] = 0,
	[5] = 0,
	[6] = 0,
	[7] = 0,
	[8] = 0,
}


-- Init

function GridStatusHealPriority:OnInitialize()
	self.super.OnInitialize(self)
	self:RegisterStatus("alert_firstPriority", L["Heal Priority: 1st"], nil, true)
	self:RegisterStatus("alert_secondPriority", L["Heal Priority: 2nd"], nil, true)
end

function GridStatusHealPriority:OnEnable()
	self:RegisterEvent("UNIT_HEALTH", "UpdateUnitPriority")
	self:RegisterEvent("Banzai_UnitGainedAggro", "UpdateUnitPriority")
	self:RegisterEvent("Banzai_UnitLostAggro", "UpdateUnitPriority")
	self:RegisterEvent("SpecialEvents_UnitBuffGained", "UnitAuraChanged")
	self:RegisterEvent("SpecialEvents_UnitBuffLost", "UnitAuraChanged")
	self:RegisterEvent("SpecialEvents_UnitDebuffGained", "UnitAuraChanged")
	self:RegisterEvent("SpecialEvents_UnitDebuffLost", "UnitAuraChanged")
	self:RegisterEvent("Grid_StatusGained", "UnitStatusChanged")
	self:RegisterEvent("Grid_StatusLost", "UnitStatusChanged")
	self:RegisterBucketEvent({"RosterLib_RosterUpdated","PLAYER_ENTERING_WORLD"}, 5, "UpdateSubgroupModifiers")
	if HealComm3 then
		HealComm3.RegisterCallback(self, "HealComm_DirectHealStart")
		HealComm3.RegisterCallback(self, "HealComm_DirectHealStop")
		HealComm3.RegisterCallback(self, "HealComm_DirectHealDelayed")
	end
end


-- Main functions

function GridStatusHealPriority:UnitDoesntNeedPriority(unitid)
	if UnitIsDeadOrGhost(unitid) or (not UnitIsConnected(unitid)) or (not UnitInRange(unitid))
			or (not UnitIsFriend(unitid, "player")) or (not UnitAffectingCombat(unitid)) or (((UnitHealth(unitid) / UnitHealthMax(unitid)) > .9) and (not Banzai:GetUnitAggroByUnitId(unitid))) then
		return true
	else
		return nil
	end
end

function GridStatusHealPriority:UnitIsMainTank(unitid)
	local mainTankTable = nil
	if oRA and oRA.maintanktable then
		mainTankTable = oRA.maintanktable
	elseif CT_RA_MainTanks then
		mainTankTable = CT_RA_MainTanks
	end
	if mainTankTable then
		for key, val in pairs(mainTankTable) do
			if (UnitName(unitid) == val) then
				return true
			end
		end
	end
	return nil
end

function GridStatusHealPriority:GetHighThreatTarget()
	local maxThreat = 0
	local highThreatTarget = nil
	local threat = 0
	local mainTankTable = nil
	if UnitExists("target") and UnitIsEnemy("player", "target") and (not UnitIsPlayer("target")) then
		threat = Threat:GetMaxThreatOnTarget(UnitGUID("target"))
		if threat > maxThreat then
			highThreatTarget = "target"
			maxThreat = threat
		end
	elseif UnitExists("target") and UnitExists("targettarget") and UnitIsEnemy("player", "targettarget") and (not UnitIsPlayer("targettarget")) then
		threat = Threat:GetMaxThreatOnTarget(UnitGUID("targettarget"))
		if threat > maxThreat then
			highThreatTarget = "targettarget"
			maxThreat = threat
		end
	end
	if oRA and oRA.maintanktable then
		mainTankTable = oRA.maintanktable
	elseif CT_RATarget and CT_RATarget.MainTanks then
		mainTankTable = CT_RATarget.MainTanks
	end
	if mainTankTable then
		for key, val in pairs(mainTankTable) do
			local threat = 0
			local target = nil
			if UnitExists(RL:GetUnitIDFromName(val)) and UnitExists(RL:GetUnitIDFromName(val).."target") then
				target = RL:GetUnitIDFromName(val).."target"
			end
			local targettarget = nil
			if target and UnitExists(RL:GetUnitIDFromName(val).."targettarget") then
				target = RL:GetUnitIDFromName(val).."targettarget"
			end
			if target and UnitIsEnemy("player", target) and (not UnitIsPlayer(target)) then
				threat = Threat:GetMaxThreatOnTarget(UnitGUID(target))
				if threat > maxThreat then
					highThreatTarget = target
					maxThreat = threat
				end
			elseif target and targettarget and UnitIsEnemy("player", targettarget) and (not UnitIsPlayer(targettarget)) then
				threat = Threat:GetMaxThreatOnTarget(UnitGUID(targettarget))
				if threat > maxThreat then
					highThreatTarget = targettarget
					maxThreat = threat
				end
			end
		end
	end
	return highThreatTarget
end

function GridStatusHealPriority:GetModifier(unitid)
	local m = 0

	-- ABORT
	-- Can't heal
	for debuff in pairs(healingPrevention) do
		if Aura:UnitHasDebuff(unitid, debuff) then
			return 999
		end
	end
	-- Imp pet's phase shift
	if Aura:UnitHasBuff(unitid, BS["Phase Shift"]) then
		return 999
	end

	-- RAISE PRIORITY
	-- Has aggro
	if Banzai:GetUnitAggroByUnitId(unitid) then
		m = m - 5
	end
	-- Is main tank
	if UnitInRaid("player") and self:UnitIsMainTank(unitid) then
		m = m - 10
	end
	-- Has high threat
	if Threat then
		target = self:GetHighThreatTarget()
		if target then
			threatMax = Threat:GetMaxThreatOnTarget(UnitGUID(target))
			threat = Threat:GetThreat(UnitGUID(unitid), UnitGUID(target))
			if threat >= (threatMax * .95) then
				m = m - 4
			elseif threat >= (threatMax * .9) then
				m = m - 3
			elseif threat >= (threatMax * .8) then
				m = m - 2
			end
		end
	end
	-- Is focus unit
	if UnitIsUnit(unitid, "focus") then
		m = m - 5
	end
	-- Has battleground flag
	for buff in pairs(battlegroundFlags) do
		local buffIndex = Aura:UnitHasBuff(unitid, buff)
		if buffIndex then
			m = m - 5
		end
	end
	-- Has anti-healing debuff
	for debuff in pairs(antiHealing) do
		if Banzai:GetUnitAggroByUnitId(unitid) and Aura:UnitHasDebuff(unitid, debuff) then
			m = m - 5
		elseif Aura:UnitHasDebuff(unitid, debuff) then
			m = m + 10
		end
	end
	-- Has anti-healing stackable debuff
	for debuff in pairs(antiHealingStackable) do
		local debuffIndex, applications = Aura:UnitHasDebuff(unitid, debuff)
		if Banzai:GetUnitAggroByUnitId(unitid) and debuffIndex then
			m = m - (1 * applications)
		elseif debuffIndex then
			m = m + (2 * applications)
		end
	end

	-- LOWER PRIORITY
	-- Has incoming heal
	local healamount = 0
	if HealComm3 then
		healamount = HealComm3.UnitIncomingHealGet(unitid, GetTime() + 3) or 0
	end
	if (healamount > 0) then
		m = m + (healamount / UnitHealthMax(unitid) * 60)
	end
	-- Has heal over time
	for buff in pairs(hots) do
		local buffIndex, applications = Aura:UnitHasBuff(unitid, buff)
		if buffIndex then
			m = m + (2 * applications)
		end
	end
	-- Has prayer of mending
	if Aura:UnitHasBuff(unitid, BS["Prayer of Mending"]) and ((UnitHealth(unitid) / UnitHealthMax(unitid)) > .3) then
		m = m + 5
	end
	-- Has shield buff
	if Aura:UnitHasBuff(unitid, BS["Power Word: Shield"]) then
		m = m + 5
	end
	-- Is invulnerable
	for buff in pairs(invulnerable) do
		local buffIndex = Aura:UnitHasBuff(unitid, buff)
		if buffIndex then
			m = m + 10
		end
	end
	-- Is out of mana
	if (UnitPowerType(unitid) == 0) then
		local mana = UnitMana(unitid) / UnitManaMax(unitid)
		if (mana < 0.10) then
			m = m + 10
		elseif (mana < 0.20) then
			m = m + 5
		end
	end
	-- Is AFK
	if UnitIsAFK(unitid) then
		m = m + 20
	end
	-- Is pet
	if (UnitPlayerOrPetInRaid(unitid) or UnitPlayerOrPetInParty(unitid)) and (not UnitInRaid(unitid)) and (not UnitInParty(unitid)) then
		m = m + 10
	end
	-- Has burning adrenaline (from Vaelastrasz, 10 seconds until dead)
	if Aura:UnitHasDebuff(unitid, BS["Burning Adrenaline"]) then
		m = m + 15
	end
	-- If PvE raid, de-prioritize healing certain classes based on what healing class you are
	if (select(2, IsInInstance()) == "raid") and (GetNumRaidMembers() > 10) and ((playerClass == "PRIEST") or (playerClass == "PALADIN") or (playerClass == "DRUID") or (playerClass == "SHAMAN")) then
		local unitClass = select(2, UnitClass(unitid))
		m = m + classModifiers[playerClass][unitClass]
	end
	-- If PvE raid, modify priority depending on how many healers, main tanks, & melee classes are in each subgroup
	if (select(2, IsInInstance()) == "raid") and (GetNumRaidMembers() > 5) then
		local u = RL:GetUnitObjectFromUnit(unitid)
		if u then
			m = m + subgroupModifier[u.subgroup]
		end
	end
	return m
end

function GridStatusHealPriority:UpdateUnitPriority(unitid)
	-- If you are not grouped with the unit then abort
	if (not UnitPlayerOrPetInRaid(unitid)) and (not UnitPlayerOrPetInParty(unitid)) then
		return
	end
	-- Check to see if current priority units should stay priority units
	if secondPriorityUnit and self:UnitDoesntNeedPriority(secondPriorityUnit) then
		self.core:SendStatusLost(UnitName(secondPriorityUnit), "alert_secondPriority")
		secondPriorityUnit = nil
		secondPriorityValue = 999
	end
	local firstPrioritySettings = self.db.profile.alert_firstPriority
	if firstPriorityUnit and self:UnitDoesntNeedPriority(firstPriorityUnit) then
		self.core:SendStatusLost(UnitName(firstPriorityUnit), "alert_firstPriority")
		if secondPriorityUnit then
			self.core:SendStatusLost(UnitName(secondPriorityUnit), "alert_secondPriority")
			self.core:SendStatusGained(UnitName(secondPriorityUnit), "alert_firstPriority",
				firstPrioritySettings.priority,
				(firstPrioritySettings.range and 40),
				firstPrioritySettings.color,
				firstPrioritySettings.text,
				nil,
				nil,
				firstPrioritySettings.icon)
			firstPriorityUnit = secondPriorityUnit
			firstPriorityValue = secondPriorityValue
			secondPriorityUnit = nil
			secondPriorityValue = 999
		else
			firstPriorityUnit = nil
			firstPriorityValue = 998
		end
	end
	-- If it isn't appropriate to prioritize this unit then abort
	if self:UnitDoesntNeedPriority(unitid) then
		return
	end
	-- Now check to see if this unit should be a priority unit
	local unitValue = (UnitHealth(unitid)/UnitHealthMax(unitid)*100) + self:GetModifier(unitid)
	local secondPrioritySettings = self.db.profile.alert_secondPriority
	if (unitValue < firstPriorityValue) and ((not firstPriorityUnit) or (not UnitIsUnit(unitid, firstPriorityUnit))) then
		if firstPriorityUnit then
			self.core:SendStatusLost(UnitName(firstPriorityUnit), "alert_firstPriority")
			if secondPriorityUnit then
				self.core:SendStatusLost(UnitName(secondPriorityUnit), "alert_secondPriority")
			end
			self.core:SendStatusGained(UnitName(firstPriorityUnit), "alert_secondPriority",
				secondPrioritySettings.priority,
				(secondPrioritySettings.range and 40),
				secondPrioritySettings.color,
				secondPrioritySettings.text,
				nil,
				nil,
				secondPrioritySettings.icon)
			secondPriorityUnit = firstPriorityUnit
			secondPriorityValue = firstPriorityValue
		end
		self.core:SendStatusGained(UnitName(unitid), "alert_firstPriority",
			firstPrioritySettings.priority,
			(firstPrioritySettings.range and 40),
			firstPrioritySettings.color,
			firstPrioritySettings.text,
			nil,
			nil,
			firstPrioritySettings.icon)
		firstPriorityUnit = unitid
		firstPriorityValue = unitValue
	elseif firstPriorityUnit and UnitIsUnit(unitid, firstPriorityUnit) and (unitValue ~= firstPriorityValue) then
		firstPriorityValue = unitValue
	elseif (unitValue < secondPriorityValue) and ((not secondPriorityUnit) or (not UnitIsUnit(unitid, secondPriorityUnit))) then
		if secondPriorityUnit then
			self.core:SendStatusLost(UnitName(secondPriorityUnit), "alert_secondPriority")
		end
		self.core:SendStatusGained(UnitName(unitid), "alert_secondPriority",
			secondPrioritySettings.priority,
			(secondPrioritySettings.range and 40),
			secondPrioritySettings.color,
			secondPrioritySettings.text,
			nil,
			nil,
			secondPrioritySettings.icon)
		secondPriorityUnit = unitid
		secondPriorityValue = unitValue
	elseif secondPriorityUnit and UnitIsUnit(unitid, secondPriorityUnit) and (unitValue ~= secondPriorityValue) then
		secondPriorityValue = unitValue
	end
end


-- Event handlers

function GridStatusHealPriority:UnitAuraChanged(unitid, aura)
	if (UnitPlayerOrPetInRaid(unitid) or UnitPlayerOrPetInParty(unitid)) then
		local needsUpdate = nil
		if (aura == BS["Power Word: Shield"]) or (aura == BS["Prayer of Mending"]) or (aura == BS["Burning Adrenaline"]) then
			needsUpdate = true
		end
		if (not needsUpdate) then
			for buff in pairs(battlegroundFlags) do
				if aura == buff then
					needsUpdate = true
					break
				end
			end
		end
		if (not needsUpdate) then
			for debuff in pairs(antiHealing) do
				if aura == debuff then
					needsUpdate = true
					break
				end
			end
		end
		if (not needsUpdate) then
			for debuff in pairs(antiHealingStackable) do
				if aura == debuff then
					needsUpdate = true
					break
				end
			end
		end
		if (not needsUpdate) then
			for debuff in pairs(healingPrevention) do
				if aura == debuff then
					needsUpdate = true
					break
				end
			end
		end
		if (not needsUpdate) then
			for buff in pairs(invulnerable) do
				if aura == buff then
					needsUpdate = true
					break
				end
			end
		end
		if needsUpdate then
			self:UpdateUnitPriority(unitid)
		end
	end
end

function GridStatusHealPriority:UnitStatusChanged(name, status)
	if (status == "alert_range_oor") or (status == "alert_range_40") then
		self:UpdateUnitPriority(RL:GetUnitIDFromName(name))
	end
end

function GridStatusHealPriority:HealComm_DirectHealStart(e, healer, target)
	if (UnitPlayerOrPetInRaid(unitid)) or (UnitPlayerOrPetInParty(unitid)) then
		self:UpdateUnitPriority(RL:GetUnitIDFromName(target))
	end
end

function GridStatusHealPriority:HealComm_DirectHealStop(e, healer, target)
	if (UnitPlayerOrPetInRaid(unitid)) or (UnitPlayerOrPetInParty(unitid)) then
		self:UpdateUnitPriority(RL:GetUnitIDFromName(target))
	end
end

function GridStatusHealPriority:HealComm_DirectHealDelayed(e, healer, target)
	if (UnitPlayerOrPetInRaid(unitid)) or (UnitPlayerOrPetInParty(unitid)) then
		self:UpdateUnitPriority(RL:GetUnitIDFromName(target))
	end
end

function GridStatusHealPriority:UpdateSubgroupModifiers()
	local i
	for i = 1, 8 do
		subgroupModifier[i] = 0
	end
	if (select(2, IsInInstance()) == "raid") and (GetNumRaidMembers() > 5) then
		local u
		for u in RL:IterateRoster(nil) do
			local unitClass = select(2, UnitClass(u.unitid))
			if self:UnitIsMainTank(u.unitid) then
				subgroupModifier[u.subgroup] = subgroupModifier[u.subgroup] - 3
			elseif (unitClass == "WARRIOR") or (unitClass == "ROGUE") then
				subgroupModifier[u.subgroup] = subgroupModifier[u.subgroup] - 1
			elseif ((unitClass == "PRIEST") or (unitClass == "DRUID") or (unitClass == "PALADIN") or (unitClass == "SHAMAN")) and (not UnitIsUnit(u.unitid, "player")) then
				subgroupModifier[u.subgroup] = subgroupModifier[u.subgroup] + 6
			end
		end
	end
end
