if (not Grid) then return end


-- Libraries

local L = AceLibrary("AceLocale-2.2"):new("GridLayoutPlus")
local RL = AceLibrary("Roster-2.1")
local TalentQuery = LibStub:GetLibrary("LibTalentQuery-1.0")


-- New Grid module

GridLayoutPlus = Grid:NewModule("GridLayoutPlus", "AceConsole-2.0")

GridLayoutPlus.defaultDB = {
	enableDynamicLayout = true,
	layoutSorting = "GROUP",
	showPets = true,
	showTanks = nil,
	showPetSpacer = nil,
	showTankSpacer = nil,
	unitsPerColumn = 5,
	classOrder = "WARRIOR,ROGUE,DRUID,PALADIN,SHAMAN,PRIEST,MAGE,WARLOCK,HUNTER",
	petOrder = "HUNTER,WARLOCK",
	roleOrder = "MAINTANK,MAINASSIST",
	columnsBeforeHide = 20,
	warnPetsHidden = nil,
	warnTanksHidden = nil,
}


-- Locals

local playerColumns = 0
local petColumns = 0
local tankColumns = 0
local tankType = nil
local raTankList = ""

local roleList = ""
local updateNeeded = nil


-- Default layout

GridLayoutPlus.DynamicLayout = {
	defaults = {
		maxColumns = 8,
	},
}


-- Init

function GridLayoutPlus:OnEnable()
	local layouts = GridLayout.options.args.layout.validate
	GridLayout.options.args = {
		["frameheader"] = {
			type = "header",
			name = L["Frame"],
			order = 5,
		},
		["display"] = {
			type = "text",
			name = L["Show Frame"],
			desc = L["Sets when the layout frame is visible."],
			order = 10,
			get =	function()
						return GridLayout.db.profile.FrameDisplay
					end,
			set =	function(v)
						GridLayout.db.profile.FrameDisplay = v
						GridLayout:CheckVisibility()
					end,
			validate = {
				["Always"] = L["Always"],
				["Grouped"] = L["Grouped"],
				["Raid"] = L["Raid"]
			},
		},
		["lock"] = {
			type = "toggle",
			name = L["Lock Frame"],
			desc = L["Locks/unlocks the layout frame for movement."],
			order = 20,
			get =	function()
						return GridLayout.db.profile.FrameLock
					end,
			set =	function(v)
						GridLayout.db.profile.FrameLock = v
					end,
		},
		["horizontal"] = {
			type = "toggle",
			name = L["Horizontal Groups"],
			desc = L["Switches between horizontal/vertical groups."],
			order = 50,
			get =	function()
						return GridLayout.db.profile.horizontal
					end,
			set =	function(v)
						GridLayout.db.profile.horizontal = v
						GridLayout:ReloadLayout()
					end,
		},
		["framesettings"] = {
			type = "group",
			name = L["Frame Settings"],
			desc = L["Layout frame settings."],
			order = 30,
			args = {
				["layoutanchor"] = {
					type = "text",
					name = L["Frame Anchor"],
					desc = L["Sets where the layout frame is anchored relative to the screen."],
					order = 10,
					get = 	function()
								return GridLayout.db.profile.anchor
							end,
					set =	function(v)
								GridLayout.db.profile.anchor = v
								GridLayout:SavePosition()
								GridLayout:RestorePosition()
							end,
					validate = {
						["CENTER"] = L["CENTER"],
						["TOP"] = L["TOP"],
						["BOTTOM"] = L["BOTTOM"],
						["LEFT"] = L["LEFT"],
						["RIGHT"] = L["RIGHT"],
						["TOPLEFT"] = L["TOPLEFT"],
						["TOPRIGHT"] = L["TOPRIGHT"],
						["BOTTOMLEFT"] = L["BOTTOMLEFT"],
						["BOTTOMRIGHT"] = L["BOTTOMRIGHT"],
					},
				},
				["groupanchor"] = {
					type = "text",
					name = L["Group Anchor"],
					desc = L["Sets where groups are anchored relative to the layout frame."],
					order = 20,
					get =	function()
								return GridLayout.db.profile.groupAnchor
							end,
					set =	function(v)
								GridLayout.db.profile.groupAnchor = v
								GridLayout:ReloadLayout()
							end,
					validate = {
						["TOPLEFT"] = L["TOPLEFT"],
						["TOPRIGHT"] = L["TOPRIGHT"],
						["BOTTOMLEFT"] = L["BOTTOMLEFT"],
						["BOTTOMRIGHT"] = L["BOTTOMRIGHT"],
					},
				},
				["displayheader"] = {
					type = "header",
					order = 45,
				},
				["scale"] = {
					type = "range",
					name = L["Scale"],
					desc = L["Adjust layout frame scale."],
					order = 50,
					min = 0.5,
					max = 2.0,
					step = 0.05,
					isPercent = true,
					get =	function()
								return GridLayout.db.profile.ScaleSize
							end,
					set =	function(v)
								GridLayout.db.profile.ScaleSize = v
								GridLayout:Scale()
							end,
				},
				["padding"] = {
					type = "range",
					name = L["Padding"],
					desc = L["Adjust layout frame padding."],
					order = 60,
					max = 20,
					min = 0,
					step = 1,
					get =	function()
								return GridLayout.db.profile.Padding
							end,
					set =	function(v)
								GridLayout.db.profile.Padding = v
								GridLayout:ReloadLayout()
							end,
				},
				["spacing"] = {
					type = "range",
					name = L["Spacing"],
					desc = L["Adjust layout frame spacing."],
					order = 70,
					max = 25,
					min = 0,
					step = 1,
					get =	function()
								return GridLayout.db.profile.Spacing
							end,
					set =	function(v)
								GridLayout.db.profile.Spacing = v
								GridLayout:ReloadLayout()
							end,
				},
				["background"] = {
					type = "color",
					name = L["Background"],
					desc = L["Adjust background color and alpha."],
					order = 80,
					get =	function()
								local settings = GridLayout.db.profile
								return settings.BackgroundR, settings.BackgroundG, settings.BackgroundB, settings.BackgroundA
							end,
					set =	function(r, g, b, a)
								local settings = GridLayout.db.profile
								settings.BackgroundR, settings.BackgroundG, settings.BackgroundB, settings.BackgroundA = r, g, b, a
								GridLayout:UpdateColor()
							end,
					hasAlpha = true,
				},
				["border"] = {
					type = "color",
					name = L["Border"],
					desc = L["Adjust border color and alpha."],
					order = 90,
					get =	function()
								local settings = GridLayout.db.profile
								return settings.BorderR, settings.BorderG, settings.BorderB, settings.BorderA
							end,
					set =	function(r, g, b, a)
								local settings = GridLayout.db.profile
								settings.BorderR, settings.BorderG, settings.BorderB, settings.BorderA = r, g, b, a
								GridLayout:UpdateColor()
							end,
					hasAlpha = true,
				},
				["advanced"] = {
					type = "group",
					name = L["Advanced"],
					desc = L["Advanced options for layout frame settings."],
					order = -1,
					args = {
						["clickthrough"] = {
							type = "toggle",
							name = L["Click-through Frame"],
							desc = L["Allow mouse-click-through on the layout frame."],
							order = 10,
							get =	function()
										return GridLayout.db.profile.ClickThrough
									end,
							set =	function(v)
										GridLayout.db.profile.ClickThrough = v
										GridLayout.frame:EnableMouse(not v)
									end,
							disabled =	function()
											return not GridLayout.db.profile.FrameLock
										end,
						},
						["clamp"] = {
							type = "toggle",
							name = L["Clamp to Screen"],
							desc = L["Toggle whether to permit movement of frame off screen."],
							order = 20,
							get =	function()
										return GridLayout.db.profile.clamp
									end,
							set =	function(v)
										GridLayout.db.profile.clamp = v
										GridLayout:SetClamp()
									end,
						},
						["reset"] = {
							type = "execute",
							name = L["Reset Frame Position"],
							desc = L["Resets the layout frame's position and anchor."],
							order = -1,
							func =	function()
										GridLayout:ResetPosition()
									end,
						},
					},
				},
			},
		},
		["raidheadergap"] = {
			type = "header",
			order = 105,
		},
		["raidheader"] = {
			type = "header",
			name = L["Raid"],
			order = 106,
		},
		["enableDynamicLayout"] = {
			type = "toggle",
			name = L["Enable Dynamic Layout"],
			desc = L["Enable dynamic layout auto-adjustments."],
			order = 107,
			get =	function()
						return GridLayoutPlus.db.profile.enableDynamicLayout
					end,
			set =	function(v)
						GridLayoutPlus.db.profile.enableDynamicLayout = v
						if (v) then
							updateNeeded = true
							GridLayoutPlus:FullUpdate()
						end
					end,
		},
		["layoutSorting"] = {
			type = "text",
			name = L["Layout Sorting"],
			desc = L["Choose between Group, Class, or Role sorting."],
			order = 110,
			disabled =	function()
							return (not GridLayoutPlus.db.profile.enableDynamicLayout)
						end,
			get =	function()
						return GridLayoutPlus.db.profile.layoutSorting
					end,
			set =	function(v)
						GridLayoutPlus.db.profile.layoutSorting = v
						updateNeeded = true
						GridLayoutPlus:RosterUpdate()
					end,
			validate = {
				["GROUP"] = L["By Group"],
				["CLASS"] = L["By Class"],
				["ROLE"] = L["By Role"],
			},
		},
		["showPets"] = {
			type = "toggle",
			name = L["Show Pets"],
			desc = L["Show pets in raids."],
			order = 120,
			disabled =	function()
							return (not GridLayoutPlus.db.profile.enableDynamicLayout)
						end,
			get =	function()
						return GridLayoutPlus.db.profile.showPets
					end,
			set =	function(v)
						GridLayoutPlus.db.profile.showPets = v
						updateNeeded = true
						GridLayoutPlus:RosterUpdate()
					end,
		},
		["showPetSpacer"] = {
			type = "toggle",
			name = L["Show Pet Spacer"],
			desc = L["Show a spacer column between players and pets."],
			order = 130,
			disabled =	function()
							return (not GridLayoutPlus.db.profile.enableDynamicLayout)
						end,
			get =	function()
						return GridLayoutPlus.db.profile.showPetSpacer
					end,
			set =	function(v)
						GridLayoutPlus.db.profile.petSpacer = v
						updateNeeded = true
						GridLayoutPlus:LayoutUpdate()
					end,
		},
		["showTanks"] = {
			type = "toggle",
			name = L["Show Tanks"],
			desc = L["Show tanks in raids."],
			order = 140,
			disabled =	function()
							return (not GridLayoutPlus.db.profile.enableDynamicLayout)
						end,
			get =	function()
						return GridLayoutPlus.db.profile.showTanks
					end,
			set =	function(v)
						GridLayoutPlus.db.profile.showTanks = v
						GridLayoutPlus:FullUpdate()
					end,
		},
		["showTankSpacer"] = {
			type = "toggle",
			name = L["Show Tank Spacer"],
			desc = L["Show a spacer column between tanks and players."],
			order = 150,
			disabled =	function()
							return (not GridLayoutPlus.db.profile.enableDynamicLayout)
						end,
			get =	function()
						return GridLayoutPlus.db.profile.showTankSpacer
					end,
			set =	function(v)
						GridLayoutPlus.db.profile.showTankSpacer = v
						updateNeeded = true
						GridLayoutPlus:LayoutUpdate()
					end,
		},
		["advanced"] = {
			type = "group",
			name = L["Advanced"],
			desc = L["Advanced options for dynamic layouts."],
			order = 160,
			disabled =	function()
							return (not GridLayoutPlus.db.profile.enableDynamicLayout)
						end,
			args = {
				["unitsPerColumn"] = {
					type = "range",
					name = L["Units Per Column"],
					desc = L["How many units to show in a single column."],
					order = 10,
					min = 5,
					max = 40,
					step = 5,
					get =	function()
								return GridLayoutPlus.db.profile.unitsPerColumn
							end,
					set =	function(v)
								GridLayoutPlus.db.profile.unitsPerColumn = v
								updateNeeded = true
								GridLayoutPlus:FullUpdate()
							end,
				},
				["classOrder"] = {
					type = "text",
					name = L["Class Order"],
					desc = L["Order in which player units appear."],
					usage = L["<CLASS1>,<CLASS2>,..."],
					order = 20,
					get =	function()
								return GridLayoutPlus.db.profile.classOrder
							end,
					set =	function(v)
								GridLayoutPlus.db.profile.classOrder = v
								updateNeeded = true
								GridLayoutPlus:RosterUpdate()
							end,
				},
				["petOrder"] = {
					type = "text",
					name = L["Pet Order/Filter"],
					desc = L["Order in which pet units appear (also filters)."],
					usage = L["<CLASS1>,<CLASS2>,..."],
					order = 30,
					get =	function()
								return GridLayoutPlus.db.profile.petOrder
							end,
					set =	function(v)
								GridLayoutPlus.db.profile.petOrder = v
								updateNeeded = true
								GridLayoutPlus:RosterUpdate()
							end,
				},
				["roleOrder"] = {
					type = "text",
					name = L["Blizz-tank Order/Filter"],
					desc = L["Order in which Blizzard tanks appear (also filters)."],
					usage = L["<ROLE1>,<ROLE2>"],
					order = 40,
					get =	function()
								return GridLayoutPlus.db.profile.roleOrder
							end,
					set =	function(v)
								GridLayoutPlus.db.profile.roleOrder = v
								updateNeeded = true
								GridLayoutPlus:RosterUpdate()
							end,
				},
				["columnsBeforeHide"] = {
					type = "range",
					name = L["Columns Before Hiding"],
					desc = L["Max Grid columns before hiding tanks/pets."],
					order = 50,
					min = 5,
					max = 20,
					step = 1,
					get =	function()
								return GridLayoutPlus.db.profile.columnsBeforeHide
							end,
					set =	function(v)
								GridLayoutPlus.db.profile.columnsBeforeHide = v
								updateNeeded = true
								GridLayoutPlus:LayoutUpdate()
							end,
				},
				["warnPetsHidden"] = {
					type = "toggle",
					name = L["Warn if Pets are Hidden"],
					desc = L["Show warning when pet units are forced to be hidden."],
					order = 60,
					get =	function()
								return GridLayoutPlus.db.profile.warnPetsHidden
							end,
					set =	function(v)
								GridLayoutPlus.db.profile.warnPetsHidden = v
								updateNeeded = true
								GridLayoutPlus:LayoutUpdate()
							end,
				},
				["warnTanksHidden"] = {
					type = "toggle",
					name = L["Warn if Tanks are Hidden"],
					desc = L["Show warning when tank units are forced to be hidden."],
					order = 70,
					get =	function()
								return GridLayoutPlus.db.profile.warnTanksHidden
							end,
					set =	function(v)
								GridLayoutPlus.db.profile.warnTanksHidden = v
								updateNeeded = true
								GridLayoutPlus:LayoutUpdate()
							end,
				},
			},
		},
		["staticheader"] = {
			type = "header",
			order = 165,
		},
		["layout"] = {
			type = "text",
			name = L["Static Layout"],
			desc = L["Select a static layout to use instead of the dynamic layout."],
			order = 180,
			disabled =	function()
							return GridLayoutPlus.db.profile.enableDynamicLayout
						end,
			get =	function()
						return GridLayout.db.profile.layout
					end,
			set =	function(v)
						GridLayout.db.profile.layout = v
						GridLayout:LoadLayout(v)
					end,
			validate = layouts,
		},
		["party"] = {
			type = "toggle",
			name = L["Show My Party First"],
			desc = L["Show my party as an extra column in front of all other columns."],
			order = 190,
			get =	function()
						return GridLayout.db.profile.showParty
					end,
			set =	function(v)
						GridLayout.db.profile.showParty = v
						GridLayout:ReloadLayout()
					end,
		},
		["partyheadergap"] = {
			type = "header",
			order = 205,
		},
		["partyheader"] = {
			type = "header",
			name = L["Party"],
			order = 206,
		},
		["partypets"] = {
			type = "toggle",
			name = L["Show Pets"],
			desc = L["Show pets below the party."],
			order = 210,
			get =	function()
						return GridLayout.db.profile.showPartyPets
					end,
			set =	function(v)
						GridLayout.db.profile.showPartyPets = v
						GridLayout:ReloadLayout()
					end,
		},
	}
	self:RegisterEvent("Grid_JoinedRaid", "FullUpdate")
	self:RegisterEvent("Grid_JoinedBattleground", "FullUpdate")
	self:RegisterEvent("Grid_LeftParty", "ResetLocals")
	self:RegisterBucketEvent({"RosterLib_RosterUpdated", "ZONE_CHANGED"}, 2, "RosterUpdate")
	self:RegisterEvent("GridLayoutPlus_LayoutUpdate", "LayoutUpdate")
	self:ScheduleRepeatingEvent("GridLayoutPlus_LayoutUpdate", 2)
	if oRA then
		self:RegisterEvent("oRA_MainTankUpdate", "RATanksUpdate")
	elseif CT_RA_MainTanks then
		hooksecurefunc("CT_RAOptions_UpdateMTs", function() self:RATanksUpdate() end)
	end
	TalentQuery.RegisterCallback(self, "TalentQuery_Ready")
	GridLayout:AddLayout(L["DynamicLayout"], self.DynamicLayout)
	self:FullUpdate()
end


-- Event handlers

function GridLayoutPlus:TalentQuery_Ready(_, name)
	local u = RL:GetUnitObjectFromName(name)
	if u then
		local isnotplayer = not UnitIsUnit(u.unitid, "player")
		u.talentrole = "NONSPECIALIZED"
		for tab = 1, GetNumTalentTabs(isnotplayer) do
			local treename, _, pointsspent = GetTalentTabInfo(tab, isnotplayer)
			if ((treename == "Protection") or (treename == "Feral Combat")) and (pointsspent > 30) then
				u.talentrole = "TANK"
			elseif ((treename == "Retribution") or (treename == "Enhancement")) and (pointsspent >=30) then
				u.talentrole = "MELEE"
			elseif ((treename == "Holy") or (treename == "Restoration")) and (pointsspent > 30) then
				u.talentrole = "HEALER"
			elseif ((treename == "Shadow") or (treename == "Balance") or (treename == "Elemental")) and (pointsspent > 30) then
				u.talentrole = "RANGED"
			end
		end
		self:RosterUpdate()
	end
end

function GridLayoutPlus:ResetLocals()
	playerColumns = 0
	petColumns = 0
	tankColumns = 0
	tankType = nil
	raTankList = ""
	roleList = ""
	updateNeeded = nil
end

function GridLayoutPlus:FullUpdate()
	if (GetNumRaidMembers() < 1) then return end
	self:RATanksUpdate()
	self:RosterUpdate()
	updateNeeded = true
	self:LayoutUpdate()
end

function GridLayoutPlus:RATanksUpdate()
	local ratanktable = nil
	local numberoftanks = 0
	raTankList = ""
	if GridLayoutPlus.db.profile.showTanks and oRA and oRA.maintanktable then
		ratanktable = oRA.maintanktable
	elseif GridLayoutPlus.db.profile.showTanks and CT_RA_MainTanks then
		ratanktable = CT_RA_MainTanks
	end
	if ratanktable then
		for key, val in pairs(ratanktable) do
			local unitid = RL:GetUnitIDFromName(val)
			if unitid and UnitExists(unitid) and UnitPlayerOrPetInRaid(unitid) then
				numberoftanks = numberoftanks + 1
				raTankList = raTankList..val..","
			end
		end
	end
	tankColumns = floor((numberoftanks / GridLayoutPlus.db.profile.unitsPerColumn) + ((GridLayoutPlus.db.profile.unitsPerColumn - 1) / GridLayoutPlus.db.profile.unitsPerColumn))
	if (tankColumns > 0) then
		tankType = "RA"
	elseif (tankType == "RA") then
		tankType = nil
		raTankList = ""
	end
	updateNeeded = true
end

function GridLayoutPlus:RosterUpdate()
	if (GetNumRaidMembers() < 1) then return end
	local maxsubgroup = 0
	local numberofpotentialpets = 0
	local numberoftanks = 0
	local roletable = {}
	local roleorder = 9
	local oldrolelist = roleList
	roleList = ""
	for u in RL:IterateRoster(nil) do
		if (GridLayoutPlus.db.profile.layoutSorting == "GROUP") and (u.subgroup > maxsubgroup) then
			maxsubgroup = u.subgroup
		end
		if GridLayoutPlus.db.profile.showPets and string.find(GridLayoutPlus.db.profile.petOrder, u.class) then
			numberofpotentialpets = numberofpotentialpets + 1
		end
		if GridLayoutPlus.db.profile.showTanks and (tankType ~= "RA") and u.role and string.find(GridLayoutPlus.db.profile.roleOrder, u.role) then
			numberoftanks = numberoftanks + 1
		end
		if (GridLayoutPlus.db.profile.layoutSorting == "ROLE") then
			if (not u.talentrole) and ((u.class == "WARRIOR") or (u.class == "PALADIN") or (u.class == "DRUID") or (u.class == "SHAMAN") or (u.class == "PRIEST")) then
				if UnitIsUnit(u.unitid, "player") then
					self:TalentQuery_Ready(_, u.name)
				else
					TalentQuery:Query(u.unitid)
				end
			end
			if u.talentrole and (u.talentrole == "TANK") then
				roleorder = 1
			elseif (u.talentrole and (u.talentrole == "MELEE"))
					or (u.class == "ROGUE")
					or ((u.class == "WARRIOR") and ((not u.talentrole) or (u.talentrole == "NONSPECIALIZED"))) then
				roleorder = 2
			elseif u.talentrole and (u.talentrole == "HEALER") then
				roleorder = 3
			elseif (u.talentrole and (u.talentrole == "RANGED"))
					or ((u.class == "MAGE") or (u.class == "WARLOCK") or (u.class == "HUNTER")) then
				roleorder = 5
			else
				roleorder = 4
			end
			table.insert(roletable, roleorder .. "," .. string.format("%02d", string.find(GridLayoutPlus.db.profile.classOrder, u.class)) .. "," .. string.format("%02d", (81 - UnitLevel(u.unitid))) .. "," .. u.unitid)
		end
	end
	if (GridLayoutPlus.db.profile.layoutSorting == "GROUP") and (playerColumns ~= floor((maxsubgroup * 5 / GridLayoutPlus.db.profile.unitsPerColumn) + ((GridLayoutPlus.db.profile.unitsPerColumn - 1) / GridLayoutPlus.db.profile.unitsPerColumn))) then
		playerColumns = floor((maxsubgroup * 5 / GridLayoutPlus.db.profile.unitsPerColumn) + ((GridLayoutPlus.db.profile.unitsPerColumn - 1) / GridLayoutPlus.db.profile.unitsPerColumn))
		updateNeeded = true
	elseif (playerColumns ~= floor((GetNumRaidMembers() / GridLayoutPlus.db.profile.unitsPerColumn) + ((GridLayoutPlus.db.profile.unitsPerColumn - 1) / GridLayoutPlus.db.profile.unitsPerColumn))) then
		playerColumns = floor((GetNumRaidMembers() / GridLayoutPlus.db.profile.unitsPerColumn) + ((GridLayoutPlus.db.profile.unitsPerColumn - 1) / GridLayoutPlus.db.profile.unitsPerColumn))
		updateNeeded = true
	end
	if (petColumns ~= floor((numberofpotentialpets / GridLayoutPlus.db.profile.unitsPerColumn) + ((GridLayoutPlus.db.profile.unitsPerColumn - 1) / GridLayoutPlus.db.profile.unitsPerColumn))) then
		petColumns = floor((numberofpotentialpets / GridLayoutPlus.db.profile.unitsPerColumn) + ((GridLayoutPlus.db.profile.unitsPerColumn - 1) / GridLayoutPlus.db.profile.unitsPerColumn))
		updateNeeded = true
	end
	if (tankColumns ~= floor((numberoftanks / GridLayoutPlus.db.profile.unitsPerColumn) + ((GridLayoutPlus.db.profile.unitsPerColumn - 1) / GridLayoutPlus.db.profile.unitsPerColumn))) then
		tankColumns = floor((numberoftanks / GridLayoutPlus.db.profile.unitsPerColumn) + ((GridLayoutPlus.db.profile.unitsPerColumn - 1) / GridLayoutPlus.db.profile.unitsPerColumn))
		if (tankColumns > 0) then
			tankType = "BLIZZ"
		elseif (tankType == "BLIZZ") then
			tankType = nil
		end
		updateNeeded = true
	end
	if (GridLayoutPlus.db.profile.layoutSorting == "ROLE") then
		table.sort(roletable)
		local unit, name, realm, namerealm = nil
		for key, val in ipairs(roletable) do
			unit = select(4, strsplit(",", val))
			name, realm = UnitName(unit)
			namerealm = (realm and (realm ~= "") and (name .. "-" .. realm)) or name
			if namerealm then
				roleList = roleList .. namerealm .. ","
			end
		end
		if (oldrolelist ~= roleList) then
			updateNeeded = true
		end
	end
end


-- Update layout

function GridLayoutPlus:LayoutUpdate()
	if (updateNeeded) then
		if GridLayoutPlus.db.profile.enableDynamicLayout and (GetNumRaidMembers() > 0) and (not InCombatLockdown()) then
			local canfitpets = nil
			local canfittanks = nil
			local groupindex = 1
			local petSpacer = 0
			local tankSpacer = 0
			if GridLayoutPlus.db.profile.showPets and (petColumns > 0) then
				petSpacer = (GridLayoutPlus.db.profile.showPetSpacer and 1 or 0)
			end
			if GridLayoutPlus.db.profile.showTanks and (tankColumns > 0) then
				tankSpacer = (GridLayoutPlus.db.profile.showTankSpacer and 1 or 0)
			end
			-- Can fit pets / tanks
			if (GridLayoutPlus.db.profile.columnsBeforeHide >= (playerColumns + petColumns + petSpacer + tankColumns + tankSpacer)) then
				canfitpets = true
				canfittanks = true
			elseif (GridLayoutPlus.db.profile.columnsBeforeHide >= (playerColumns + petColumns + petSpacer)) then
				canfitpets = true
			end
			-- Alter layout
			if GridLayoutPlus.db.profile.showTanks and canfittanks and (tankType == "RA") then
				GridLayoutPlus.DynamicLayout[groupindex] = {
					nameList = raTankList,
					sortMethod = "INDEX",
					unitsPerColumn = GridLayoutPlus.db.profile.unitsPerColumn,
				}
				groupindex = groupindex + 1
			elseif GridLayoutPlus.db.profile.showTanks and canfittanks and (tankType == "BLIZZ") then
				GridLayoutPlus.DynamicLayout[groupindex] = {
					groupFilter = GridLayoutPlus.db.profile.roleOrder,
					groupingOrder = GridLayoutPlus.db.profile.roleOrder,
					groupBy = "ROLE",
					unitsPerColumn = GridLayoutPlus.db.profile.unitsPerColumn,
				}
				groupindex = groupindex + 1
			end
			if GridLayoutPlus.db.profile.showTanks and canfittanks and GridLayoutPlus.db.profile.showTankSpacer then
				GridLayoutPlus.DynamicLayout[groupindex] = {
					groupFilter = ""
				}
				groupindex = groupindex + 1
			end
			if (GridLayoutPlus.db.profile.layoutSorting == "GROUP") then
				local groupnumber = 1
				local strgroupfilter
				for i = 1, playerColumns do
					strgroupfilter = ""
					for j = groupnumber, (groupnumber + (GridLayoutPlus.db.profile.unitsPerColumn / 5) - 1) do
						strgroupfilter = strgroupfilter .. j .. ","
						groupnumber = j
					end
					GridLayoutPlus.DynamicLayout[groupindex] = {
						groupFilter = strgroupfilter,
						groupingOrder = GridLayoutPlus.db.profile.classOrder,
						groupBy = "CLASS",
						unitsPerColumn = GridLayoutPlus.db.profile.unitsPerColumn,
					}
					groupnumber = groupnumber + 1
					groupindex = groupindex + 1
				end
			elseif (GridLayoutPlus.db.profile.layoutSorting == "ROLE") then
				if (string.len(roleList) > 0) then
					GridLayoutPlus.DynamicLayout[groupindex] = {
						nameList = roleList,
						sortMethod = "INDEX",
						unitsPerColumn = GridLayoutPlus.db.profile.unitsPerColumn,
					}
					groupindex = groupindex + 1
				end
			else
				GridLayoutPlus.DynamicLayout[groupindex] = {
					groupFilter = GridLayoutPlus.db.profile.classOrder,
					groupingOrder = GridLayoutPlus.db.profile.classOrder,
					groupBy = "CLASS",
					unitsPerColumn = GridLayoutPlus.db.profile.unitsPerColumn,
				}
				groupindex = groupindex + 1
			end
			if GridLayoutPlus.db.profile.showPets and canfitpets and GridLayoutPlus.db.profile.showPetSpacer then
				GridLayoutPlus.DynamicLayout[groupindex] = {
					groupFilter = ""
				}
				groupindex = groupindex + 1
			end
			if GridLayoutPlus.db.profile.showPets and canfitpets and (petColumns > 0) then
				GridLayoutPlus.DynamicLayout[groupindex] = {
					isPetGroup = true,
					groupFilter = GridLayoutPlus.db.profile.petOrder,
					groupingOrder = GridLayoutPlus.db.profile.petOrder,
					groupBy = "CLASS",
					filterOnPet = true,
					unitsPerColumn = GridLayoutPlus.db.profile.unitsPerColumn,
				}
				groupindex = groupindex + 1
			end
			for i = groupindex, 10 do
				GridLayoutPlus.DynamicLayout[i] = nil
			end
			-- Refresh layout
			GridLayout:LoadLayout("DynamicLayout")
			-- Warn if hidden units
			if GridLayoutPlus.db.profile.warnPetsHidden and GridLayoutPlus.db.profile.showPets and (petColumns > 0) and (not canfitpets) then
				self:Print(L["Your maxColumnsBeforeHide setting is causing pet units to be hidden"])
			end
			if GridLayoutPlus.db.profile.warnTanksHidden and GridLayoutPlus.db.profile.showTanks and (tankColumns > 0) and (not canfittanks) then
				self:Print(L["Your maxColumnsBeforeHide setting is causing tank units to be hidden"])
			end
			updateNeeded = nil
		elseif GridLayoutPlus.db.profile.enableDynamicLayout and (GetNumRaidMembers() > 0) and InCombatLockdown() then
			self:ScheduleLeaveCombatAction(self, "LayoutUpdate")
		end
	end
end
