------------ Grid and ACE stuff ------------


local GridDynamicLayout = AceLibrary("AceAddon-2.0"):new("AceConsole-2.0", "AceEvent-2.0")
local Aura = AceLibrary("SpecialEvents-Aura-2.0")
--local BS = AceLibrary("Babble-Spell-2.2")
local L = AceLibrary("AceLocale-2.2"):new("GridDynamicLayout")

L:RegisterTranslations("enUS", function()
	return {
		["Dynamic Layout"] = true,
		["Options for GridDynamicLayout."] = true,

		["ForcedHeader_title"] = "Force players into group",
		["ForcedTanks_desc"] = "Player names list separated by commas, those will always appear in the tank group",
		["ForcedHeals_desc"] = "Player names list separated by commas, those will always appear in the heal group",
		["ForcedRDPS_desc"] = "Player names list separated by commas, those will always appear in the ranged DPS group",
		["ForcedMDPS_desc"] = "Player names list separated by commas, those will always appear in the melee DPS group",
		["ForcedTanks_name"] = "Forced Tanks",
		["ForcedHeals_name"]="Forced Heals",
		["ForcedMDPS_name"]="Forced Melee DPS",
		["ForcedRDPS_name"]="Forced Ranged DPS",

		["TrashHeader_title"] = "Inactive raidmembers",
		["ActivateAFKToTrash_name"] = "AFK",
		["ActivateAFKToTrash_desc"] = "What to do with AFK players",
		["ActivateOfflineToTrash_name"] = "Offline",
		["ActivateOfflineToTrash_desc"] = "What to do with Offline players",
		["ActivateDifferentZoneToTrash_name"] = "Players in a different zone",
		["ActivateDifferentZoneToTrash_desc"] = "What to do with players in different zones",
		["ActivateNotVisibleToTrash_name"] = "Players too far away",
		["ActivateNotVisibleToTrash_desc"] = "What to do with players too far away",
		["TrashOption_noAction"] = "Leave in group",
		["TrashOption_hide"] = "Hide",
		["TrashOption_toTrashGroup"] = "To trash group",

		["PetHeader_title"] = "Optional pet group",
		["ActivateWarlockPet_name"] = "Warlock pets",
		["ActivateWarlockPet_desc"] = "If activated, warlocks pets will be added to the pet group",
		["ActivateHunterPet_name"] = "Hunter pets",
		["ActivateHunterPet_desc"] = "If activated, hunters pets will be added to the pet group",

		["MiscHeader_title"] = "Miscellaneous",
		["DefaultDPSWars_name"] = "Defaulted DPS warriors / deathknights",
		["DefaultDPSWars_desc"] = "Player names list separated by commas, those will be defaulted as DPS warriors or deathknights",
		["ActivateStats_name"] = "Activate statistical analysis",
		["ActivateStats_desc"] = "Mod will attempt to assign a default behaviour to user based on active buff / form to improve performances",

		["(.+) gains (.+)."] = true,
	}
end)

L:RegisterTranslations("koKR", function()
	return {
		["Dynamic Layout"] = "기능적 배치",
		["Options for GridDynamicLayout."] = "GridDynamicLayout을 위한 옵션을 설정합니다.",

		["ForcedHeader_title"] = "강요된 플레이어 그룹",
		["ForcedTanks_desc"] = "플레이어 이름 목록은 콤마에 의하여 분리되며, 언제나 탱커 그룹에 표시합니다.",
		["ForcedHeals_desc"] = "플레이어 이름 목록은 콤마에 의하여 분리되며, 언제나 힐러 그룹에 표시합니다.",
		["ForcedRDPS_desc"] = "플레이어 이름 목록은 콤마에 의하여 분리되며, 언제나 원거리 DPS 그룹에 표시합니다.",
		["ForcedMDPS_desc"] = "플레이어 이름 목록은 콤마에 의하여 분리되며, 언제나 근거리 DPS 그룹에 표시합니다.",
		["ForcedTanks_name"] = "강요된 탱커",
		["ForcedHeals_name"] = "강요된 힐러",
		["ForcedMDPS_name"] = "강요된 근거리 DPS",
		["ForcedRDPS_name"] = "강요된 원거리 DPS",

		["TrashHeader_title"] = "일반 그룹",
		["ActivateAFKToTrash_name"] = "자리비움",
		["ActivateAFKToTrash_desc"] = "만약 사용한다면, 모든 자리비움 사용자는 다음 배치 업데이트시 일반 그룹에 저장됩니다.",
		["ActivateOfflineToTrash_name"] = "오프라인",
		["ActivateOfflineToTrash_desc"] = "만약 사용한다면, 모든 오프라인 사용자는 다음 배치 업데이트시 일반 그룹에 저장됩니다.",
		["ActivateDifferentZoneToTrash_name"] = "다른 지역의 플레이어",
		["ActivateDifferentZoneToTrash_desc"] = "만약 사용한다면, 당신과 같은 지역에 있지 않는 모든 사용자는 다음 배치 업데이트시 일반 그룹에 저장됩니다.",
		["ActivateNotVisibleToTrash_name"] = "멀리 떨어진 플레이어",
		["ActivateNotVisibleToTrash_desc"] = "만약 사용한다면, 당신과 너무 멀리 떨어진 모든 사용자는 다음 배치 업데이트시 일반 그룹에 저장됩니다.",
		["TrashOption_noAction"] = "그룹 나가기",
		["TrashOption_hide"] = "숨김",
		["TrashOption_toTrashGroup"] = "일반 그룹",

		["PetHeader_title"] = "소환수 그룹 선택",
		["ActivateWarlockPet_name"] = "흑마법사 소환수",
		["ActivateWarlockPet_desc"] = "만약 사용한다면, 흑마법사의 소환수를 소환수 그룹에 추가합니다.",
		["ActivateHunterPet_name"] = "사냥꾼 소환수",
		["ActivateHunterPet_desc"] = "만약 사용한다면, 사냥꾼의 소환수를 소환수 그룹에 추가합니다.",

		["MiscHeader_title"] = "기타",
		["DefaultDPSWars_name"] = "DPS 전사",
		["DefaultDPSWars_desc"] = "플레이어 이름 목록은 콤마에 의하여 분리되며, DPS 전사로 탱커 그룹에 표시되지 않습니다.",
		["ActivateStats_name"] = "통계적인 분석 사용",
		["ActivateStats_desc"] = "모드는 효율을 향상시키기 위해 기본 행동의 버프/변신에 기초를 둔 배치를 지정합니다.",

		["(.+) gains (.+)."] = "(.+)|1이;가; (.+) 효과를 얻었습니다.",
	}
end)

GridDynamicLayout = Grid:NewModule("GridDynamicLayout")
GridDynamicLayout.menuName = L["Dynamic Layout"]


------------ Locals ------------

local GROUPINDEX_TANK = 1
local GROUPINDEX_HEAL = 2
local GROUPINDEX_RDPS = 3
local GROUPINDEX_MDPS = 4
local GROUPINDEX_PET = 5
local GROUPINDEX_TRASH = 6
local GROUPINDEX_TRASH_ORIGINAL = 6

local UNITS_PER_COLUMN = 5
local MAX_COLUMNS = 8

-- Definition of spell info
local BS = {}
BS["Berserker Stance"]=GetSpellInfo(2458)
BS["Battle Stance"]=GetSpellInfo(2457)
BS["Defensive Stance"]=GetSpellInfo(71)
BS["Righteous Fury"]=GetSpellInfo(25780)
BS["Improved Righteous Fury"]=GetSpellInfo(20470)
BS["Shadowform"]=GetSpellInfo(15473)
BS["Cat Form"]=GetSpellInfo(768)
BS["Dire Bear Form"]=GetSpellInfo(9634)
BS["Bear Form"]=GetSpellInfo(5487)
BS["Moonkin Form"]=GetSpellInfo(24858)
BS["Frost Presence"]=GetSpellInfo(48263)
BS["Blood Presence"]=GetSpellInfo(48266)
BS["Unholy Presence"]=GetSpellInfo(48265)

local dynamicLayout = {
	defaults = {
		unitsPerColumn = UNITS_PER_COLUMN,
		maxColumns = MAX_COLUMNS,
		columnAnchorPoint = "LEFT",
		sortMethod="INDEX",
	},
	[GROUPINDEX_TANK] = {					 -- tanks
		nameList = "",
	},
	[GROUPINDEX_HEAL] = {					 -- healers
		nameList = "",
	},
	[GROUPINDEX_RDPS] = {					 -- ranged DPS
		nameList = "",
	},
	[GROUPINDEX_MDPS] = {					 -- melee DPS
		nameList = "",
	},
}

local dpsWarList = {
}

local MAX_SPECSCORE=6
local MIN_SIGNIFICANT_SPECSCORE=2
local UPDATE_DELAY = 1

------------ GridDynamicLayoutACE Options -------------

GridDynamicLayout.options = {
	ForcedHeader = {
		type = "header",
		name = L["ForcedHeader_title"],
		order = 10,
	},
	ForcedTanks = {
		type = "text",
		name = L["ForcedTanks_name"],
		desc = L["ForcedTanks_desc"],
		get = function() return GridDynamicLayout:generateOptionValue(GridDynamicLayout.db.profile.tankList) end,
		usage = L["ForcedTanks_desc"],
		set = function(v)
			GridDynamicLayout.db.profile.tankList = GridDynamicLayout:parseOptionValue(v)
			GridDynamicLayout:setUpdateNeeded("Forced Tank update")
		end,
		order = 11,
	},
	ForcedHeals = {
		type = "text",
		name = L["ForcedHeals_name"],
		desc = L["ForcedHeals_desc"],
		get = function() return GridDynamicLayout:generateOptionValue(GridDynamicLayout.db.profile.healList) end,
		usage = L["ForcedHeals_desc"],
		set = function(v)
			GridDynamicLayout.db.profile.healList = GridDynamicLayout:parseOptionValue(v)
			GridDynamicLayout:setUpdateNeeded("Forced Heal update")
		end,
		order = 12,
	},
	ForcedMDPS = {
		type = "text",
		name = L["ForcedMDPS_name"],
		desc = L["ForcedMDPS_desc"],
		get = function() return GridDynamicLayout:generateOptionValue(GridDynamicLayout.db.profile.MDPSList) end,
		usage = L["ForcedMDPS_desc"],
		set = function(v)
			GridDynamicLayout.db.profile.MDPSList = GridDynamicLayout:parseOptionValue(v)
			GridDynamicLayout:setUpdateNeeded("Forced melee DPS update")
		end,
		order = 13,
	},
	ForcedRDPS = {
		type = "text",
		name = L["ForcedRDPS_name"],
		desc = L["ForcedRDPS_desc"],
		get = function() return GridDynamicLayout:generateOptionValue(GridDynamicLayout.db.profile.RDPSList) end,
		usage = L["ForcedRDPS_name"],
		set = function(v)
			GridDynamicLayout.db.profile.RDPSList = GridDynamicLayout:parseOptionValue(v)
			GridDynamicLayout:setUpdateNeeded("Forced range DPS update")
		end,
		order = 14,
	},

	TrashHeader = {
		type = "header",
		name = L["TrashHeader_title"],
		order = 20,
	},
	ActivateAFKToTrash = {
		type = "text",
		name = L["ActivateAFKToTrash_name"],
		desc = L["ActivateAFKToTrash_desc"],
		get = function() return GridDynamicLayout.db.profile.modeAFK end,
		set = function(v)
			GridDynamicLayout.db.profile.modeAFK = v
			GridDynamicLayout:setUpdateNeeded("AFK mode changed")
		end,
		validate = {["noAction"]=L["TrashOption_noAction"],["hide"]=L["TrashOption_hide"], ["toTrashGroup"]=L["TrashOption_toTrashGroup"]},
		order = 21,
	},
	ActivateOfflineToTrash = {
		type = "text",
		name = L["ActivateOfflineToTrash_name"],
		desc = L["ActivateOfflineToTrash_desc"],
		get = function() return GridDynamicLayout.db.profile.modeOffline end,
		set = function(v)
			GridDynamicLayout.db.profile.modeOffline = v
			GridDynamicLayout:setUpdateNeeded("Offline mode changed")
		end,
		validate = {["noAction"]=L["TrashOption_noAction"],["hide"]=L["TrashOption_hide"], ["toTrashGroup"]=L["TrashOption_toTrashGroup"]},
		order = 22,
	},
	ActivateDifferentZoneToTrash = {
		type = "text",
		name = L["ActivateDifferentZoneToTrash_name"],
		desc = L["ActivateDifferentZoneToTrash_desc"],
		get = function() return GridDynamicLayout.db.profile.modeDifferentZone end,
		set = function(v)
			GridDynamicLayout.db.profile.modeDifferentZone = v
			GridDynamicLayout:setUpdateNeeded("Different zone mode changed")
		end,
		validate = {["noAction"]=L["TrashOption_noAction"],["hide"]=L["TrashOption_hide"], ["toTrashGroup"]=L["TrashOption_toTrashGroup"]},
		order = 23,
	},
	ActivateNotVisibleToTrash = {
		type = "text",
		name = L["ActivateNotVisibleToTrash_name"],
		desc = L["ActivateNotVisibleToTrash_desc"],
		get = function() return GridDynamicLayout.db.profile.modeNotVisible end,
		set = function(v)
			GridDynamicLayout.db.profile.modeNotVisible = v
			GridDynamicLayout:setUpdateNeeded("Not visible mode changed")
		end,
		validate = {["noAction"]=L["TrashOption_noAction"],["hide"]=L["TrashOption_hide"], ["toTrashGroup"]=L["TrashOption_toTrashGroup"]},
		order = 24,
	},

	PetHeader = {
		type = "header",
		name = L["PetHeader_title"],
		order = 30,
	},
	ActivateWarlockPet = {
		type = "toggle",
		name = L["ActivateWarlockPet_name"],
		desc = L["ActivateWarlockPet_desc"],
		get = function() return GridDynamicLayout.db.profile.activateWarlockPet end,
		set = function(v)
			GridDynamicLayout.db.profile.activateWarlockPet = v
			GridDynamicLayout:setUpdateNeeded("Toggle Warlocks Pets")
		end,
		order = 31,
	},
	ActivateHunterPet = {
		type = "toggle",
		name = L["ActivateHunterPet_name"],
		desc = L["ActivateHunterPet_desc"],
		get = function() return GridDynamicLayout.db.profile.activateHunterPet end,
		set = function(v)
			GridDynamicLayout.db.profile.activateHunterPet = v
			GridDynamicLayout:setUpdateNeeded("Toggle Hunters Pets")
		end,
		order = 32,
	},

	MiscHeader = {
		type = "header",
		name = L["MiscHeader_title"],
		order = 40,
	},
	DefaultDPSWar = {
		type = "text",
		name = L["DefaultDPSWars_name"],
		desc = L["DefaultDPSWars_desc"],
		get = function() return GridDynamicLayout:generateOptionValue(GridDynamicLayout.db.profile.defaultDPSWarList) end,
		usage = L["DefaultDPSWars_desc"],
		set = function(v)
			GridDynamicLayout.db.profile.defaultDPSWarList = GridDynamicLayout:parseOptionValue(v)
		end,
		order = 41,
	},
	ActivateStats = {
		type = "toggle",
		name = L["ActivateStats_name"],
		desc = L["ActivateStats_desc"],
		get = function() return GridDynamicLayout.db.profile.activateStats end,
		usage = L["ActivateStats_desc"],
		set = function(v)
			GridDynamicLayout.db.profile.activateStats = v
		end,
		order = 42,
	},
	--[[DumpStats = {
		type = "execute",
		name = "Dump Stats DB",
		desc = "Dump Stats DB",
		handler = GridDynamicLayout,
		func = "dumpStatsDB",
		order = 43,
	},]]--
}

function GridDynamicLayout:parseOptionValue(s)
	local result = {}
	if string.find(s, ",") then
		for a in gmatch(s..",", "(.-),") do
			if a ~= "" then
				result[a] = true
			end
		end
	elseif s ~= "" then
		result[s] = true
	end
	return result
end

function GridDynamicLayout:generateOptionValue(list)
	local result = ""
	for key,value in pairs(list) do
		if value then
			result = result .. key .. ","
		end
	end
	return result
end

-- Hook into GridLayout options
GridLayout.options.args["DynamicLayout"] = {
	type = "group",
	name = L["Dynamic Layout"],
	desc = L["Options for GridDynamicLayout."],
	disabled = InCombatLockdown,
	args = GridDynamicLayout.options
}

------------ GridDynamicLayoutDefaultDB -------------

GridDynamicLayout.defaultDB = {
	dbVersion = 1.2,
	debug=false,
	activateStats=false,
	modeAFK="noAction",
	modeOffline="noAction",
	modeDifferentZone="noAction",
	modeNotVisible="noAction",
	activateWarlockPet=true,
	activateHunterPet=true,
	tankList = {},
	healList = {},
	MDPSList = {},
	RDPSList = {},
	defaultDPSWarList = {},
}

------------ GridDynamicLayout ------------

local function copyTable(t)
	result = {}
	for k,v in pairs(t) do
		if type(v) == "table" then
			result[k] = copyTable(v)
		else
			result[k] = v
		end
	end
	return result
end

function GridDynamicLayout:OnInitialize()
	self.super.OnInitialize(self)
	-- Initializes session-specific data
	dpsWarList = copyTable(self.db.profile.defaultDPSWarList)
	self.ExitingCombatScheduled = false
	self.pertinentEventsRegistered = false
	self.updateNeeded = false
	self.stats = {}
end

function GridDynamicLayout:OnEnable()
	self:RegisterEvent("Grid_JoinedBattleground", "initializeLayoutJoinedBG")
	self:RegisterEvent("Grid_JoinedRaid", "initializeLayoutJoinedRaid")
	self:RegisterEvent("Grid_JoinedParty", "initializeLayoutJoinedParty")
	self:RegisterEvent("Grid_LeftParty", "unregisterPertinentEvents")
	self:ScheduleRepeatingEvent("GridDynamicLayout_CheckUpdate", UPDATE_DELAY)
	self:RegisterEvent("GridDynamicLayout_CheckUpdate", "initializeLayout")
end

function GridDynamicLayout:initializeLayoutJoinedBG()
	self:initializeLayoutJoined("Joined BG")
end

function GridDynamicLayout:initializeLayoutJoinedParty()
	self:initializeLayoutJoined("Joined Party")
end

function GridDynamicLayout:initializeLayoutJoinedRaid()
	self:initializeLayoutJoined("Joined Raid")
end

function GridDynamicLayout:initializeLayoutJoined(reason)
	self:setUpdateNeeded(reason)
	self:registerPertinentEvents()
end


function GridDynamicLayout:unregisterPertinentEvents()
	if self.pertinentEventsRegistered then
		self:UnregisterEvent("Grid_UnitLeft")
		self:UnregisterEvent("Grid_UnitJoined")
		self:UnregisterEvent("SpecialEvents_UnitBuffLost")
		self:UnregisterEvent("SpecialEvents_UnitBuffGained")
		self:UnregisterEvent("COMBAT_LOG_EVENT_UNFILTERED")
		self:UnregisterEvent("PLAYER_REGEN_ENABLED")
	end
	self.pertinentEventsRegistered = false
end

function GridDynamicLayout:registerPertinentEvents()
	if not self.pertinentEventsRegistered then
		self:RegisterEvent("Grid_UnitLeft", "initializeLayoutLeft")
		self:RegisterEvent("Grid_UnitJoined", "initializeLayoutJoin")
		self:RegisterEvent("SpecialEvents_UnitBuffLost", "checkBuff")
		self:RegisterEvent("SpecialEvents_UnitBuffGained", "checkBuff")
		self:RegisterEvent("COMBAT_LOG_EVENT_UNFILTERED", "handleCombatLogEvent")
		self:RegisterEvent("PLAYER_REGEN_ENABLED", "UpdateStatisticalAnalysis")
	end
	self.pertinentEventsRegistered = true
end

function GridDynamicLayout:initializeLayoutLeft(name)
	self:setUpdateNeeded("Unit Left - "..name)
end

function GridDynamicLayout:initializeLayoutJoin(name, unitid)
	self:setUpdateNeeded("Unit Join - "..name.." ("..unitid..")")
end

-- ========================================================================== --
-- Combat log constants
-- ========================================================================== --
-- Affiliation
local COMBATLOG_OBJECT_AFFILIATION_MINE		= COMBATLOG_OBJECT_AFFILIATION_MINE		or 0x00000001
local COMBATLOG_OBJECT_AFFILIATION_PARTY	= COMBATLOG_OBJECT_AFFILIATION_PARTY	or 0x00000002
local COMBATLOG_OBJECT_AFFILIATION_RAID		= COMBATLOG_OBJECT_AFFILIATION_RAID		or 0x00000004

function GridDynamicLayout:handleCombatLogEvent(timestamp, event, sourceGUID, sourceName, sourceFlags, destGUID, destName, destFlags, ...)
    local raidMemberMask = COMBATLOG_OBJECT_AFFILIATION_MINE+COMBATLOG_OBJECT_AFFILIATION_PARTY+COMBATLOG_OBJECT_AFFILIATION_RAID
    if bit.band(sourceFlags, raidMemberMask)~=0 and event == "SPELL_AURA_APPLIED" then
        local _, spellName = ...
        if spellName == BS["Frost Presence"] or spellName == BS["Defensive Stance"] then
            dpsWarList[sourceName]=false
			self:setUpdateNeeded("War/DK stance change - "..sourceName.." - to tank")
		elseif spellName == BS["Blood Presence"] or spellName == BS["Unholy Presence"] or spellName == BS["Berserker Stance"] or spellName == BS["Battle Stance"] then
		    dpsWarList[sourceName]=true
			self:setUpdateNeeded("War/DK stance change - "..sourceName.." - to DPS")
        end
    end
end

function GridDynamicLayout:getRealmedName(unit)
	local name, realm = UnitName(unit)

	if realm ~= nil and realm ~= "" then
		name = name .. "-" .. realm
	end

	return name
end

function GridDynamicLayout:addToGroup(grpNumber, unit)
	local name = self:getRealmedName(unit)
	if grpNumber==nil then
		self:Debug(name .. " (".. unit .. ") : is hidden" )
	elseif not dynamicLayout[grpNumber] then
		self:Debug("Attempted to add " .. name .. " (".. unit .. ") to non existing group number : " .. grpNumber )
	else
		dynamicLayout[grpNumber].nameList = dynamicLayout[grpNumber].nameList .. name .. ","
	end
end

function GridDynamicLayout:setUpdateNeeded(reason)
	self:Debug("Update needed : " .. reason)
	self.updateNeeded = true
end

local function getPetGroupFilter(hasHunter, hasWarlock)
	if hasHunter and hasWarlock then
		return "HUNTER,WARLOCK"
	elseif hasHunter then
		return "HUNTER"
	elseif hasWarlock then
		return "WARLOCK"
	else
		return ""
	end
end

function GridDynamicLayout:initializeLayout()
	if not self.updateNeeded then
		return
	end

	self:Debug("----- Initializing Layout ------")

	local i
    
    -- create or delete pet group as needed
	if self.db.profile.activateWarlockPet or
		self.db.profile.activateHunterPet
	then
		if dynamicLayout[GROUPINDEX_PET] == nil then
			dynamicLayout[GROUPINDEX_PET] = {
			}
		end
		dynamicLayout[GROUPINDEX_PET].isPetGroup = true
		dynamicLayout[GROUPINDEX_PET].filterOnPet = true
		dynamicLayout[GROUPINDEX_PET].groupFilter = getPetGroupFilter(self.db.profile.activateHunterPet, self.db.profile.activateWarlockPet)
		-- If pet group activated then trash group index must be original
		GROUPINDEX_TRASH = GROUPINDEX_TRASH_ORIGINAL
	else
	    if dynamicLayout[GROUPINDEX_PET] ~= nil then
			dynamicLayout[GROUPINDEX_PET] = nil
		end
		-- If pet group not activated then trash group index must be pet group index
		GROUPINDEX_TRASH = GROUPINDEX_PET
		if dynamicLayout[GROUPINDEX_TRASH_ORIGINAL] ~= nil then
			dynamicLayout[GROUPINDEX_TRASH_ORIGINAL] = nil
		end
	end
    
	-- create or delete trash group as needed
	if self.db.profile.modeAFK == "toTrashGroup" or
		self.db.profile.modeOffline == "toTrashGroup" or
		self.db.profile.modeDifferentZone == "toTrashGroup" or
		self.db.profile.modeNotVisible == "toTrashGroup"
	then
		if dynamicLayout[GROUPINDEX_TRASH] == nil then
			dynamicLayout[GROUPINDEX_TRASH] = {
				nameList = "",
			}
			dynamicLayout[GROUPINDEX_TRASH].isPetGroup = nil
		    dynamicLayout[GROUPINDEX_TRASH].filterOnPet = nil
		end
	else
		if dynamicLayout[GROUPINDEX_TRASH] ~= nil then
			dynamicLayout[GROUPINDEX_TRASH] = nil
		end
	end

	for _,v in pairs(dynamicLayout) do
		if v.nameList then
			v.nameList = ""
		end
	end

	-- Sorts raid by class / name
	sortedRaid = {}
	for i=1,40 do
		if UnitExists("raid"..i) then
			table.insert(sortedRaid, i)
		end
	end
	table.sort(sortedRaid, function(a, b)
		local _, aClass = UnitClass("raid"..a)
		local _, bClass = UnitClass("raid"..b)
		if aClass~=nil and bClass~=nil then
			if aClass < bClass then
				return true
			elseif aClass == bClass and UnitName("raid"..a) < UnitName("raid"..b) then
				return true
			else
				return false
			end
		end
		return false
	end)

	-- Scans the raid and place each unit in appropriate nameList
	self:Debug("--------------")
	for i=1,#sortedRaid do
		--self:Debug(sortedRaid[i]..">"..UnitName("raid"..sortedRaid[i]).."-"..UnitClass("raid"..sortedRaid[i]))
		if UnitExists("raid"..sortedRaid[i]) then
			self:addToGroup(self:getLayoutGroupIndex(sortedRaid[i]), "raid"..sortedRaid[i])
		end
	end
	self:Debug("--------------")


	if GridLayout.db.profile.layout == "Dynamic Layout" then
		if not InCombatLockdown() then
			GridLayout:ReloadLayout()
		elseif not self.ExitingCombatScheduled then
			self:ScheduleLeaveCombatAction(GridDynamicLayout, "ExitingCombat")
			self.ExitingCombatScheduled = true
		end
	end
	self.updateNeeded = false
end

function GridDynamicLayout:ExitingCombat()
	self:ScheduleLeaveCombatAction(GridLayout, "ReloadLayout")
	self.ExitingCombatScheduled = false
end

function GridDynamicLayout:getLayoutGroupIndex(i)
	local unit = "raid"..i
	-- Add to trash group if not in right subzone / is offline
	local _, _, _, _, _, _, zone, online = GetRaidRosterInfo(i)
	if
		(self.db.profile.modeAFK == "toTrashGroup" and UnitIsAFK(unit)) or
		(self.db.profile.modeOffline == "toTrashGroup" and not online) or
		(self.db.profile.modeDifferentZone == "toTrashGroup" and zone~=GetRealZoneText()) or
		(self.db.profile.modeNotVisible == "toTrashGroup" and not UnitIsVisible(unit))
	then
		return GROUPINDEX_TRASH
	elseif
		(self.db.profile.modeAFK == "hide" and UnitIsAFK(unit)) or
		(self.db.profile.modeOffline == "hide" and not online) or
		(self.db.profile.modeDifferentZone == "hide" and zone~=GetRealZoneText()) or
		(self.db.profile.modeNotVisible == "hide" and not UnitIsVisible(unit))
	then
		return nil -- unit is not in any group (unit is hidden)
	-- Add to tanks if "by name" setting used
	elseif self.db.profile.tankList[UnitName(unit)] then
		return GROUPINDEX_TANK
	-- Add to healers if "by name" setting used
	elseif self.db.profile.healList[UnitName(unit)] then
		return GROUPINDEX_HEAL
	-- Add to ranged DPS if "by name" setting used
	elseif self.db.profile.RDPSList[UnitName(unit)] then
		return GROUPINDEX_RDPS
	-- Add to melee DPS if "by name" setting used
	elseif self.db.profile.MDPSList[UnitName(unit)] then
		return GROUPINDEX_MDPS
	-- Use class / buff setting to assign role
	else
		local class
		_, class = UnitClass(unit)

		-- Add Shamans and Paladins to heal group
		if class=="SHAMAN" then
			return GROUPINDEX_HEAL
		-- Add Hunters, Mages and Warlocks to ranged DPS group
		elseif class=="HUNTER" or class=="MAGE" or class=="WARLOCK" then
			return GROUPINDEX_RDPS
		-- Add Rogues to melee DPS group
		elseif class=="ROGUE" then
			return GROUPINDEX_MDPS
		elseif class=="PALADIN" then
			return self:getLayoutGroupIndex_Paladin(unit)
		elseif class=="WARRIOR" or class=="DEATHKNIGHT" then
			-- Handling of warriors / deathknights (cannot detect stance by normal means) --
			if dpsWarList[UnitName(unit)] then
				return GROUPINDEX_MDPS
			else
				return GROUPINDEX_TANK
			end
		elseif class=="PRIEST" then
			return self:getLayoutGroupIndex_Priest(unit)
		elseif class=="DRUID" then
			return self:getLayoutGroupIndex_Druid(unit)
		end
	end
end

function GridDynamicLayout:getLayoutGroupIndex_Paladin(unit)
	local assumedSpec = nil
	local useStatData = false
	if self.db.profile.activateStats and self.stats[self:getRealmedName(unit)] then
		assumedSpec, useStatData = self:getAssumedSpec(unit)
	end

	if useStatData then
		if assumedSpec == "prot" then
			return GROUPINDEX_TANK
		else
			return GROUPINDEX_HEAL
		end
	else
		if Aura:UnitHasBuff(unit, BS["Righteous Fury"]) or Aura:UnitHasBuff(unit, BS["Improved Righteous Fury"]) then
			return GROUPINDEX_TANK
		else
			return GROUPINDEX_HEAL
		end
	end
end

function GridDynamicLayout:getLayoutGroupIndex_Priest(unit)
	local assumedSpec = nil
	local useStatData = false
	if self.db.profile.activateStats and self.stats[self:getRealmedName(unit)] then
		assumedSpec, useStatData = self:getAssumedSpec(unit)
	end

	if useStatData then
		if assumedSpec == "shadow" then
			return GROUPINDEX_RDPS
		else
			return GROUPINDEX_HEAL
		end
	else
		if Aura:UnitHasBuff(unit, BS["Shadowform"]) then
			return GROUPINDEX_RDPS
		else
			return GROUPINDEX_HEAL
		end
	end
end

function GridDynamicLayout:getLayoutGroupIndex_Druid(unit)
	local assumedSpec = nil
	local useStatData = false
	if self.db.profile.activateStats and self.stats[self:getRealmedName(unit)] then
		assumedSpec, useStatData = self:getAssumedSpec(unit)
	end

	if useStatData then
		if assumedSpec == "feral" then
			-- If detected as feral choices are cat / bear
			-- if cat put in melee dps, otherwise (even if in caster form) in tank group
			if Aura:UnitHasBuff(unit, BS["Cat Form"]) then
				return GROUPINDEX_MDPS
			else
				return GROUPINDEX_TANK
			end
		elseif assumedSpec == "balance" then
			return GROUPINDEX_RDPS
		else
			return GROUPINDEX_HEAL
		end
	else
		if Aura:UnitHasBuff(unit, BS["Dire Bear Form"]) or Aura:UnitHasBuff(unit, BS["Bear Form"]) then
			return GROUPINDEX_TANK
		elseif Aura:UnitHasBuff(unit, BS["Cat Form"]) then
			return GROUPINDEX_MDPS
		elseif Aura:UnitHasBuff(unit, BS["Moonkin Form"]) then
			return GROUPINDEX_RDPS
		else
			return GROUPINDEX_HEAL
		end
	end
end

function GridDynamicLayout:checkBuff(unit, name, count, icon, rank, duration)
	local class
	local proceed

	_, class = UnitClass(unit)

	if self.db.profile.activateStats then
		local _, isSignificant = self:getAssumedSpec(unit)
		if isSignificant then
			-- If Stat Analysis mode is activated and stats are significant for unit then
			-- the only buffs that are likely to have an effect are the druids cat/bear ones
			if class=="DRUID" and (
				name==BS["Dire Bear Form"] or name==BS["Cat Form"] or name==BS["Bear Form"]
			) then
				proceed = true
			else
				proceed=false
			end
		else
			proceed = true
		end
	else
		proceed = true
	end

	if proceed and (
		(
			class=="PRIEST" and name==BS["Shadowform"]
		) or (
			class=="DRUID" and (
				name==BS["Dire Bear Form"] or name==BS["Cat Form"] or name==BS["Moonkin Form"] or name==BS["Bear Form"]
			)
		) or (
			class=="PALADIN" and (
				name==BS["Righteous Fury"] or name==BS["Improved Righteous Fury"]
			)
		)
	) then
		self:setUpdateNeeded("Pertinent buff change found - "..UnitName(unit).." ("..name..")")
	end
end

-- Scans the whole raid (at beginning and end of a fight) and add statistics on
-- the supposed role of each unit (currently supports only PAL / DRU / PRI)
-- Will call setUpdateNeeded if it detects an assumed role change
function GridDynamicLayout:UpdateStatisticalAnalysis()
	-- Is only done if statistical analysis is activated for profile
	if not self.db.profile.activateStats then return end

	-- Initializes the list of members whose roles have changed according to statistical analysis (debug purposes)
	local membersRoleUpdated = ""
	-- initializes the updateNeeded boolean (by default no layout update is needed)
	local updateNeeded = false

	for i=1,40 do
		local unit = "raid"..i
		if UnitExists(unit) and not UnitIsDeadOrGhost(unit) then
			local name = self:getRealmedName(unit)
			local _, class = UnitClass(unit)
			if class == "DRUID" then
				-- Initializes stats database for unit if necessary
				if not self.stats[name] then
					self.stats[name] = {
						currentSpec = "resto",
						breakdown = {
							feral=0,
							balance=0,
							resto=0,
						},
					}
				end
				if Aura:UnitHasBuff(unit, BS["Dire Bear Form"]) or Aura:UnitHasBuff(unit, BS["Bear Form"]) or Aura:UnitHasBuff(unit, BS["Cat Form"]) then
					self:incrementSpecStat(name, "feral")
				elseif Aura:UnitHasBuff(unit, BS["Moonkin Form"]) then
					self:incrementSpecStat(name, "balance")
				else
					self:incrementSpecStat(name, "resto")
				end
			elseif class == "PRIEST" then
				-- Initializes stats database for unit if necessary
				if not self.stats[name] then
					self.stats[name] = {
						currentSpec = "holy",
						breakdown = {
							shadow=0,
							holy=0,
						},
					}
				end
				if Aura:UnitHasBuff(unit, BS["Shadowform"]) then
					self:incrementSpecStat(name, "shadow")
				else
					self:incrementSpecStat(name, "holy")
				end
			elseif class == "PALADIN" then
				-- Initializes stats database for unit if necessary
				if not self.stats[name] then
					self.stats[name] = {
						currentSpec = "holy",
						breakdown = {
							prot=0,
							holy=0,
						},
					}
				end
				if Aura:UnitHasBuff(unit, BS["Righteous Fury"]) or Aura:UnitHasBuff(unit, BS["Improved Righteous Fury"]) then
					self:incrementSpecStat(name, "prot")
				else
					self:incrementSpecStat(name, "holy")
				end
			end
			-- Checks if an update is needed, that is if someone's assumed spec has changed since the last statistical analysis
			if self.stats[name] then
				--[[if name == "Mokhtar" then
			 		self:Debug("  "..self.stats[name].currentSpec.."~="..self:getAssumedSpec(unit))
			 	end]]--
				if self.stats[name].currentSpec ~= self:getAssumedSpec(unit) then
					updateNeeded = true
					membersRoleUpdated = membersRoleUpdated .. name .. ", "
					self.stats[name].currentSpec = self:getAssumedSpec(unit)
				end
			end
		end
	end

	if updateNeeded then self:setUpdateNeeded("Statistical analysis - member role changed ("..membersRoleUpdated..")") end
end

function GridDynamicLayout:incrementSpecStat(name, spec)
	for k,v in pairs(self.stats[name].breakdown) do
		if k==spec then
			self.stats[name].breakdown[k] = self.stats[name].breakdown[k]+2
		else
			self.stats[name].breakdown[k] = self.stats[name].breakdown[k]-1
		end
		-- spec score has to be between 0 and MAX_SPECSCORE
		if self.stats[name].breakdown[k] > MAX_SPECSCORE then
			self.stats[name].breakdown[k] = MAX_SPECSCORE
		elseif self.stats[name].breakdown[k] < 0 then
			self.stats[name].breakdown[k] = 0
		end
	end
end

function GridDynamicLayout:getAssumedSpec(unit)
	local name = self:getRealmedName(unit)
	if self.stats[name] then
		local spec = ""
		local specScore = -1
		local isSignificant = false
		for k,v in pairs(self.stats[name].breakdown) do
			if v > specScore then
				spec = k
				specScore = v
			end
		end
		if specScore >= MIN_SIGNIFICANT_SPECSCORE then
			isSignificant = true
		end
		return spec, isSignificant
	else
		return nil, false
	end
end

function GridDynamicLayout:dumpStatsDB()
	self:Debug("----------- DB DUMP -------------")
	for n,s in pairs(self.stats) do
		local output = n .. " ("..s.currentSpec..") : "
		for spec,specScore in pairs(s.breakdown) do
			output = output..spec..">"..specScore..", "
		end
		self:Debug(output)
	end
end

-- Adds dynamic layout to Grid's layouts
GridLayout:AddLayout("Dynamic Layout", dynamicLayout)
