﻿-- ----------------------------------------------------------------------------
-- GridAutoFrameSize by smuggles
-- ----------------------------------------------------------------------------
local L = AceLibrary("AceLocale-2.2"):new("GridAutoFrameSize")
local GridL = AceLibrary("AceLocale-2.2"):new("Grid")
local AceOO = AceLibrary("AceOO-2.0")
local RL = AceLibrary("Roster-2.1")

local math_ceil = math.ceil
local math_max = math.max
local GetNumPartyMembers = GetNumPartyMembers
local GetNumRaidMembers = GetNumRaidMembers
local GetRaidRosterInfo = GetRaidRosterInfo

GridAutoFrameSize = Grid:NewModule("GridAutoFrameSize")
GridAutoFrameSize.defaultDB = {
	autoSizeFrames = 1,
	autoZoneSize = 0,
	zoneLock = 0, -- set to 1 when the raid frame is set by zone and not to be changed by raid size
	zonePreset = {		
	["Alterac Valley"] = 40,		
	["Sunwell Plateau"] = 25,			
	["Black Temple"] = 25,		
	["Hyjal Summit"] = 25,		
	["Grull's Lair"] = 25,		
	["Eye of the Storm"] = 15,		
	["Zul'Aman"] = 10,		
	},
}

function GridAutoFrameSize:OnEnable()
	local opt = GridLayout.options.args
	opt["autoSizeFrames"] = {
		type = "toggle",
		name = L["Auto Size Raid Frame"],
		desc = L["Automatically adjusts frame layout to raid size"],
		order = 28,
		get = function ()
			return self.db.profile.autoSizeFrames
		end,
		set = function (v)
			self.db.profile.autoSizeFrames = v
			self:CheckRoster()
		end,	
	}
	opt["autoZoneSize"] = {
		type = "toggle",
		name = L["Use predefined zone size"],
		desc = L["Automatically adjust size of the layout to the zone you enter and locks it so it doesnt get resized when groups are changed. e.g. Black Temple / Sunwell by Group 25"],
		order = 29,
		get = function ()
			return self.db.profile.autoZoneSize
		end,
		set = function (v)
			self.db.profile.autoZoneSize = v
			self.db.profile.zoneLock = 0;
			self:CheckRoster()
		end,	
	}
	
	self.db.profile.zoneLock = 0;
	self:RegisterBucketEvent({"RAID_ROSTER_UPDATE", "PARTY_MEMBERS_CHANGED"}, 1, "CheckRoster")
	--self:RegisterEvent("ZONE_CHANGED", "CheckZone")	
	self:RegisterEvent("ZONE_CHANGED_NEW_AREA", "CheckZone")		
end

function GridAutoFrameSize:CheckZone()
	local zone = GetRealZoneText();
	if (self.db.profile.zonePreset[zone]) then
		self.db.profile.zoneLock = 1;
		self:Debug("frame size zone locked to : "..zone.." - "..self.db.profile.zonePreset[zone]);
		
		local newLayout = "By Group "..self.db.profile.zonePreset[zone];
		
		newLayout = newLayout and GridL[newLayout]
		
		if not newLayout or newLayout == GridLayout.db.profile.layout then
			self:Debug("no layout change")		
			return
		end				
		self:Debug("new layout: "..newLayout)		
		GridLayout.db.profile.layout = newLayout;
		GridLayout:LoadLayout(newLayout)		
	else
		self.db.profile.zoneLock = 0;
		self:Debug("zone unlocked");
	end
	self:Debug("zoned into : "..zone);
end

function GridAutoFrameSize:CheckRoster()
	if ( (not self.db.profile.autoSizeFrames) or (self.db.profile.zoneLock == 1) ) then
		if (self.db.profile.zoneLock == 1) then
			self:Debug("zone lock, cant change !");
		end
		return
	end
	
	local newLayout
	local raidSize = GetNumRaidMembers()
	if raidSize > 0 then
		
		local maxGroup = math_ceil(raidSize / 5)
		for i = 1, raidSize do
			local group = select(3, GetRaidRosterInfo(i))
			maxGroup = math_max(maxGroup, group)
		end
		if (maxGroup > 5) or (raidSize > 25) then
			newLayout = "By Group 40"
		else
			newLayout = "By Group " .. 5*maxGroup
		end
		self:Debug(("raid settings: size=%d, maxgroup=%d => layout=%s"):format(raidSize, maxGroup, newLayout))

	elseif GetNumPartyMembers() > 0 then
		newLayout = "By Group 5"
		self:Debug(("in party or not grouped => layout=%s"):format(raidSize, maxGroup, newLayout))
	end
	
	newLayout = newLayout and GridL[newLayout]
	
	if not newLayout or newLayout == GridLayout.db.profile.layout then
		self:Debug("no layout change")		
		return
	end
	
	self:Debug("new layout: "..newLayout)		
	GridLayout.db.profile.layout = newLayout;
	GridLayout:LoadLayout(newLayout)	
end
