﻿----------------------------
--      Declaration       --
----------------------------

local AceOO = AceLibrary("AceOO-2.0")
local BS = AceLibrary("Babble-Spell-2.2")
local L = AceLibrary("AceLocale-2.2"):new("GotWoodCircle")
local _G = getfenv(0)

GotWood.circle = AceOO.Class(GotWood.abstractview, "AceDB-2.0", "AceEvent-2.0")

local defaults = {
	skin = "Serenity",
	timertextcolor = {r=1,g=0,b=0,a=1},
	alphavalue = 0.5,
}

-- Skin and code provided by Sairen, thx goes to him for this nice view!
local skins = {
	["Serenity"] = {
		["icon"] 	= { w=27, h=27, x=6,  y=-6, },
		["overlay"] = { w=43, h=43, x=-8, y=8,  tex="Interface\\AddOns\\GotWood\\images\\serenity0" },
		["flash"]	= { w=30, h=30, x=3,  y=-3, tex="Interface\\AddOns\\GotWood\\images\\overlayred" },
		["cooldown"]= { w=30, h=30, x=0,  y=0, },
	},
}

GotWood:RegisterDefaults("circle", "profile", defaults)

----------------------------
--      Translations      --
----------------------------

L:RegisterTranslations("enUS", function()
	return {
	["Timer Text color"] = true,
	["Color of the timer text"] = true,
	["Alpha Value"] = true,
	["Alpha value of the circle after time runs out"] = true,
	}
end)

L:RegisterTranslations("deDE", function()
	return {
	["Timer Text color"] = "Zeitfarbe",
	["Color of the timer text"] = "Farbe der Zeit ändern",
	["Alpha Value"] = "Alpha Wert",
	["Alpha value of the circle after time runs out"] = "Alpha Wert der Kreise nachdem die Zeit abgelaufen ist",
	}
end)

----------------------------
--      Main Functions    --
----------------------------

function GotWood.circle.prototype:init(totemOrder)
	-- Call Superclass
	GotWood.circle.super.prototype.init(self)

	-- Save the totem order
	self.totemOrder = totemOrder
	self.totemLife = {}
	self.circles = {}

	-- Acquire db namespace
	self.db = GotWood:AcquireDBNamespace("circle")
	
	self.api24 = GotWood.api24
	self.slotmaprev = GotWood.slotmaprev
	
	self.options = {
		timertextcolor = {
			name = L["Timer Text color"], type = "color",
			desc = L["Color of the timer text"],
			get = function()
				local color = self.db.profile.timertextcolor
				return color.r, color.g, color.b, color.a
			end,
			set = function(r, g, b, a)
				self:UpdateTimerTextColor(r,g,b,a)
			end,
			hasAlpha = true,
			order = 2,
		},
		alphavalue = {
			name = L["Alpha Value"], type = "range",
			desc = L["Alpha value of the circle after time runs out"],
			get = function() return self.db.profile.alphavalue end,
			set = function(v)
				self:SetAlphaValue(v)
			end,
			min = 0,
			max = 1,
			step = 0.05,
			order = 3
		},
	}

	self:CreateCircles()
end

function GotWood.circle.prototype:CircleOnClick(mousebutton)
	if (mousebutton == "RightButton" and this.slot ) then
		DestroyTotem( this.slot )
	end
end

function GotWood.circle.prototype:CircleOnEnter()
	if ( this.slot ) then
		-- setting the tooltip
		GameTooltip_SetDefaultAnchor(GameTooltip, this);
		GameTooltip:SetTotem(this.slot);
	end
end

function GotWood.circle.prototype:CircleOnLeave()
	-- hiding the tooltip
	GameTooltip:Hide();
end

function GotWood.circle.prototype:CreateCircles()
	local icon = skins[self.db.profile.skin].icon
	for k,v in ipairs(self.totemOrder) do
		self.circles[v] = {}
		self.circles[v].frame = CreateFrame("Button", "GotWoodCircle"..v, UIParent)
		self.circles[v].frame:SetWidth(icon.w)
		self.circles[v].frame:SetHeight(icon.h)
		self:SkinOverlay(self.circles[v].frame)
		self:AddCooldownDisplay(self.circles[v].frame)
		
		-- past-2.4-only features
		if (GotWood.api24 == true) then
			-- setting this property so the scripts belong can find out what totem slot they're being executed upon
			self.circles[v].frame.slot = GotWood.slotmaprev[v]
			-- click handling for destroying single totems
			self.circles[v].frame:RegisterForClicks("RightButtonUp")
			self.circles[v].frame:SetScript("OnClick", self.CircleOnClick)
			-- tooltip handling
			self.circles[v].frame:SetScript("OnEnter", self.CircleOnEnter)
			self.circles[v].frame:SetScript("OnLeave", self.CircleOnLeave)
		end
		
		self.circles[v].time = self.circles[v].frame:CreateFontString("$parentTime","ARTWORK","GameFontNormalSmall")
		self.circles[v].time:SetWidth(36)
		self.circles[v].time:SetHeight(10)
		local ttc = self.db.profile.timertextcolor
		self.circles[v].time:SetTextColor(ttc.r,ttc.g,ttc.b,ttc.a)
		self.circles[v].time:SetPoint("TOP", self.circles[v].frame, "TOP", -1, 10)
		self.circles[v].time:SetDrawLayer("OVERLAY")
		local font = self.circles[v].time:GetFont()
		self.circles[v].time:SetFont(font, 15, "OUTLINE")
		self.circles[v].time:Show()

		self.circles[v].frame:Hide()
	end
end

function GotWood.circle.prototype:AddCooldownDisplay(circle)
	local skin = skins[self.db.profile.skin].cooldown
	local cooldown = CreateFrame("Cooldown", "$parentCooldown", circle, "CooldownFrameTemplate")
	cooldown:SetScale(skins[self.db.profile.skin].icon.w/37)
	cooldown:SetFrameLevel(1)
	cooldown:SetAllPoints(circle)
end

function GotWood.circle.prototype:SkinOverlay(circle)
	local skin = skins[self.db.profile.skin].overlay
	local overlay
	overlay = circle:CreateTexture("$parentOverlay", "OVERLAY")
	overlay:SetTexture(skin.tex)
	overlay:SetWidth(skin.w)
	overlay:SetHeight(skin.w)
	overlay:SetPoint("TOPLEFT", circle, "TOPLEFT", skin.x, skin.y)
end

function GotWood.circle.prototype:SkinIcon(name)
	local skin = skins[self.db.profile.skin].icon
	_G[name]:SetWidth(skin.w)
	_G[name]:SetHeight(skin.h)
end

function GotWood.circle.prototype:SetTotem(tName, tRank)
	-- Get the ele type of the totem
	local ele = GotWoodData[tName].ele

	-- Set the texture
	local frameName = self.circles[ele].frame:GetName()
	self.circles[ele].frame:SetNormalTexture(BS:GetSpellIcon(tName) or GotWoodData[tName].icon)
	self:SkinIcon(frameName)
	
	-- Set the time
	local time
	if tRank and GotWoodData[tName][tRank] and GotWoodData[tName][tRank].duration then
		time = GotWoodData[tName][tRank].duration
	else
		time = GotWoodData[tName].duration
	end
	CooldownFrame_SetTimer(_G[frameName.."Cooldown"], GetTime(), time, 1)
	self.circles[ele].frame:SetAlpha(1)
	self.circles[ele].time:SetText(time)
	self.circles[ele].timeleft = time-1

	-- Store the life in our local table
	local life
	if tRank and GotWoodData[tName][tRank] and GotWoodData[tName][tRank].life then
		life = GotWoodData[tName][tRank].life
	else
		life = GotWoodData[tName].life
	end
	self.totemLife[ele] = {["name"] = tName, ["life"] = life}

	-- Start the timer
	self.circles[ele].frame:Show()
	if self.circles[ele].timer and self:IsEventScheduled(self.circles[ele].timer) then
		self:CancelScheduledEvent(self.circles[ele].timer)
	end
	self.circles[ele].timer = "GotWood-CircleView-" .. math.random()
	self:ScheduleRepeatingEvent(self.circles[ele].timer, self.UpdateTime, 1, self, ele)
end

function GotWood.circle.prototype:SetTotemFromEvent(name, startTime, duration, icon, ele)
	-- Set the texture
	local frameName = self.circles[ele].frame:GetName()
	self.circles[ele].frame:SetNormalTexture(icon)
	self:SkinIcon(frameName)
	
	-- Set the time
	CooldownFrame_SetTimer(_G[frameName.."Cooldown"], startTime, duration, 1)
	self.circles[ele].frame:SetAlpha(1)
	self.circles[ele].time:SetText(duration)
	self.circles[ele].timeleft = duration

	-- life code removed. no need when totem death triggers events now.
	
	-- Start the timer
	self.circles[ele].frame:Show()
	if self.circles[ele].timer and self:IsEventScheduled(self.circles[ele].timer) then
		self:CancelScheduledEvent(self.circles[ele].timer)
	end
	self.circles[ele].timer = "GotWood-CircleView-" .. math.random()
	self:ScheduleRepeatingEvent(self.circles[ele].timer, self.UpdateTime, 1, self, ele)
end

function GotWood.circle.prototype:UpdateTime(ele)
	local time = 0
	
	if ( self.api24 ) then
		time = GetTotemTimeLeft(self.slotmaprev[ele])
	else
		time = self.circles[ele].timeleft
	end
	
	if time > 0 then
		self.circles[ele].timeleft = time - 1
		self.circles[ele].time:SetText(time)
	else
		self:CancelScheduledEvent(self.circles[ele].timer)
		self.circles[ele].frame:SetAlpha(self.db.profile.alphavalue)
		self.circles[ele].time:SetText("")
	end
end

function GotWood.circle.prototype:RemoveTotem(ele)
	if self.circles[ele].timer and self:IsEventScheduled(self.circles[ele].timer) then
		self:CancelScheduledEvent(self.circles[ele].timer)
	end
	self.circles[ele].frame:SetAlpha(self.db.profile.alphavalue)
	self.circles[ele].time:SetText("")
	_G["GotWoodCircle"..ele.."Cooldown"]:Hide()
end

function GotWood.circle.prototype:SetTotemDamage(tName, dmg)
	-- Search for the correct button
	for ele,v in pairs(self.totemLife) do
		if string.find(tName, v.name) then
			v.life = v.life-dmg
			if v.life<0 then
				-- Stop timer
				if self.circles[ele].timer and self:IsEventScheduled(self.circles[ele].timer) then
					self:CancelScheduledEvent(self.circles[ele].timer)
				end
				self.circles[ele].frame:SetAlpha(self.db.profile.alphavalue)
				self.circles[ele].time:SetText("")
				_G["GotWoodCircle"..ele.."Cooldown"]:Hide()
				self.totemLife[ele] = nil
			end
			break
		end
	end
end

function GotWood.circle.prototype:UpdateOrder()
	-- Set the positions
	for k,v in ipairs(self.totemOrder) do
		if k==1 then
			self.circles[v].frame:SetPoint("TOPLEFT", GotWood.frame, "TOPLEFT", 10, -22)
		else
			-- Attach the button to the previous one
			self.circles[v].frame:SetPoint("TOPLEFT", self.circles[self.totemOrder[k-1]].frame, "TOPRIGHT", 12, 0)
		end
	end
end

function GotWood.circle.prototype:GetOptions()
	return self.options
end

function GotWood.circle.prototype:GetWidth()
	return 164
end

function GotWood.circle.prototype:GetHeight()
	return 65
end

function GotWood.circle.prototype:SetScale(scale)
	for k,v in ipairs(self.totemOrder) do
		self.circles[v].frame:SetScale(scale)
	end
end

function GotWood.circle.prototype:UpdatePosition()
	self:UpdateOrder()
end

function GotWood.circle.prototype:Disable()
	for k,v in ipairs(self.totemOrder) do
		local name = self.circles[v].frame:GetName()
		if self.circles[v].timer and self:IsEventScheduled(self.circles[v].timer) then
			self:CancelScheduledEvent(self.circles[v].timer)
		end
		self.circles[v].frame:Hide()
		_G[name.."Overlay"] = nil
		_G[name.."Cooldown"] = nil
		self.circles[v].frame = nil
		self.circles[v].time = nil
	end
end

--------------------------
--   Options Handling   --
--------------------------

function GotWood.circle.prototype:UpdateTimerTextColor(r,g,b,a)
	self.db.profile.timertextcolor = {r=r,g=g,b=b,a=a}
	for k,v in ipairs(self.totemOrder) do
		self.circles[v].time:SetTextColor(r,g,b,a)
	end
end

function GotWood.circle.prototype:SetAlphaValue(alpha)
	self.db.profile.alphavalue = alpha
	for k,v in ipairs(self.totemOrder) do
		if self.circles[v].timeleft and self.circles[v].timeleft <= 0 then
			self.circles[v].frame:SetAlpha(alpha)
		end
	end
end

-- Important: GotWood needs to know that this view exists, so add it to its table
-- The second abbreviation has to be the same name as the class name
GotWood:AddView("Circles", "circle")