﻿----------------------------
--      Declaration       --
----------------------------

local AceOO = AceLibrary("AceOO-2.0")
local BS = AceLibrary("Babble-Spell-2.2")
local paint = AceLibrary("PaintChips-2.0")
local L = AceLibrary("AceLocale-2.2"):new("GotWoodCandyBar")

-- Initialize textures
local media = AceLibrary("SharedMedia-1.0")

GotWood.candybar = AceOO.Class(GotWood.abstractview, "CandyBar-2.0", "AceDB-2.0")

local defaults = {
	texture = "BantoBar",
	fade = 2,
	width = 200,
	height = 16,
	fontsize = 11,
	textcolor = "ffffff",
	timertextcolor = "ffffff",
	usecandygroup = true,
}
GotWood:RegisterDefaults("candybar", "profile", defaults)

----------------------------
--      Translations      --
----------------------------

L:RegisterTranslations("enUS", function()
	return {
	["Textures"] = true,
	["Change the texture of the candy bars"] = true,
	["Fade Time"] = true,
	["Set the Fade Time"] = true,
	["Width"] = true,
	["Set the width of the bars"] = true,
	["Height"] = true,
	["Set the height of the bars"] = true,
	["Font Size"] = true,
	["Set the size of the font"] = true,
	["Set colors"] = true,
	["Set some colors to the bars"] = true,
	["Text color"] = true,
	["Color of the totem text"] = true,
	["Timer Text color"] = true,
	["Color of the timer text"] = true,
	["Candy Group"] = true,
	["Group the candy bars"] = true,
	}
end)

L:RegisterTranslations("koKR", function()
	return {
	["Textures"] = "텍스쳐",
	["Change the texture of the candy bars"] = "바의 텍스쳐를 변경합니다",
	["Fade Time"] = "사라짐 시간",
	["Set the Fade Time"] = "사라짐 효과의 시간을 설정합니다",
	["Width"] = "너비",
	["Set the width of the bars"] = "바의 너비를 설정합니다",
	["Height"] = "높이",
	["Set the height of the bars"] = "바의 높이를 설정합니다",
	["Font Size"] = "폰트",
	["Set the size of the font"] = "폰트의 크기를 설정합니다",
	["Set colors"] = "색상",
	["Set some colors to the bars"] = "바의 구성요소의 색상을 변경합니다",
	["Text color"] = "텍스트 색상",
	["Color of the totem text"] = "토템의 텍스트 색상을 변경합니다",
	["Timer Text color"] = "타미어 색상",
	["Color of the timer text"] = "타이머 텍스트의 색상을 변경합니다",
	}
end)

L:RegisterTranslations("deDE", function()
	return {
	["Textures"] = "Texturen",
	["Change the texture of the candy bars"] = "Die Texturen der Candy Bars ändern",
	["Fade Time"] = "Ausblendungs-Zeit",
	["Set the Fade Time"] = "Die Ausblendungs-Zeit setzen",
	["Width"] = "Breite",
	["Set the width of the bars"] = "Die Breite der Bars setzen",
	["Height"] = "Höhe",
	["Set the height of the bars"] = "Die Höhe der Bars setzen",
	["Font Size"] = "Schriftgröße",
	["Set the size of the font"] = "Die Größe der Schrift setzen",
	["Set colors"] = "Farben ändern",
	["Set some colors to the bars"] = "Einige Farben der Bars ändern",
	["Text color"] = "Textfarbe",
	["Color of the totem text"] = "Farbe des Totem Textes ändern",
	["Timer Text color"] = "Zeitfarbe",
	["Color of the timer text"] = "Farbe der Zeit ändern",
	["Candy Group"] = "Bar Gruppierung",
	["Group the candy bars"] = "Candy bars gruppieren",
	}
end)

----------------------------
--      Main Functions    --
----------------------------

function GotWood.candybar.prototype:init(totemOrder)
	-- Call Superclass
	GotWood.candybar.super.prototype.init(self)
	
	-- Save the totem order
	self.totemOrder = totemOrder
	self.totemLife = {}
	
	-- Acquire db namespace
	self.db = GotWood:AcquireDBNamespace("candybar")
	
	self.api24 = GotWood.api24
	self.slotmaprev = GotWood.slotmaprev
	
	self.options = {
		texture = {
			name = L["Textures"], type = "text",
			desc = L["Change the texture of the candy bars"],
			get = function()
				return self.db.profile.texture
			end,
			set = function(name)
				self:SetTexture(name)
			end,
			validate = media:List("statusbar"),
			order = 1,
		},
		fade = {
			name = L["Fade Time"], type = "range",
			desc = L["Set the Fade Time"],
			get = function() return self.db.profile.fade end,
			set = function(v)
				self:SetFadeTime(v)
			end,
			min = 0,
			max = 5,
			step = 0.1,
			order = 2
		},
		width = {
			name = L["Width"], type = "range",
			desc = L["Set the width of the bars"],
			get = function() return self.db.profile.width end,
			set = function(v)
				self:SetWidth(v)
			end,
			min = 50,
			max = 300,
			step = 5,
			order = 3
		},
		height = {
			name = L["Height"], type = "range",
			desc = L["Set the height of the bars"],
			get = function() return self.db.profile.height end,
			set = function(v)
				self:SetHeight(v)
			end,
			min = 5,
			max = 40,
			step = 1,
			order = 4
		},
		fontsize = {
			name = L["Font Size"], type = "range",
			desc = L["Set the size of the font"],
			get = function() return self.db.profile.fontsize end,
			set = function(v)
				self:SetFontSize(v)
			end,
			min = 5,
			max = 30,
			step = 1,
			order = 5
		},
		colors = {
			name = L["Set colors"], type = "group",
			desc = L["Set some colors to the bars"],
			args = {
				textcolor = {
					name = L["Text color"], type = "color",
					desc = L["Color of the totem text"],
					get = function()
						local tColor, bla = paint:GetRGBPercent(self.db.profile.textcolor)
						return tColor[1], tColor[2], tColor[3]
					end,
					set = function(r, g, b, a)
						self:UpdateTextColor(r,g,b)
					end,
					hasAlpha = false,
					order = 1,
				},
				timertextcolor = {
					name = L["Timer Text color"], type = "color",
					desc = L["Color of the timer text"],
					get = function()
						local tColor, bla = paint:GetRGBPercent(self.db.profile.timertextcolor)
						return tColor[1], tColor[2], tColor[3]
					end,
					set = function(r, g, b, a)
						self:UpdateTimerTextColor(r,g,b)
					end,
					hasAlpha = false,
					order = 2,
				}
			},
			order = 6,
		},
		candygroup = {
			name = L["Candy Group"], type = "toggle",
			desc = L["Group the candy bars"],
			get = function() return self.db.profile.usecandygroup end,
			set = function(v)
				self:UseCandyGroup(v)
			end,
			order = 7
		},
	}
	
	media:Register("statusbar", "BantoBar", "Interface\\Addons\\GotWood\\images\\BantoBar")
	
	-- Create our cute candy bars
	self:CreateCandyBars()
end

function GotWood.candybar.prototype:CandybarOnClick(mousebutton, givenSlot)
	if (mousebutton == "RightButton" and givenSlot ) then
		DestroyTotem( givenSlot )
	end
end

function GotWood.candybar.prototype:CreateCandyBars()
	-- Paint it!
	local textcolor = self.db.profile.textcolor
	local timertextcolor = self.db.profile.timertextcolor
	paint:RegisterHex(textcolor)
	paint:RegisterHex(timertextcolor)
	
	for k,v in ipairs(self.totemOrder) do
		-- Register the candy bars
		self:RegisterCandyBar("gotwood"..v, 1, "N/A", nil, "Green", "Yellow", "Orange", "Red")
		self:SetCandyBarFade("gotwood"..v, self.db.profile.fade)
		self:SetCandyBarTexture("gotwood"..v, media:Fetch("statusbar", self.db.profile.texture))
		self:SetCandyBarWidth("gotwood"..v, self.db.profile.width)
		self:SetCandyBarHeight("gotwood"..v, self.db.profile.height)
		self:SetCandyBarFontSize("gotwood"..v, self.db.profile.fontsize)
		self:SetCandyBarTextColor("gotwood"..v, textcolor)
		self:SetCandyBarTimerTextColor("gotwood"..v, timertextcolor)
		-- past-2.4-only features
		if (GotWood.api24 == true) then
			self:SetCandyBarOnClick("gotwood"..v, self.CandybarOnClick, GotWood.slotmaprev[v])
		end
	end
end

function GotWood.candybar.prototype:SetTotem(tName, tRank)
	-- Get the ele type of the totem
	local ele = GotWoodData[tName].ele
	
	-- Set the name on the bar
	if tRank then
		self:SetCandyBarText("gotwood"..ele, tName.." ("..tRank..")")
	else
		self:SetCandyBarText("gotwood"..ele, tName)
	end
	
	-- Set the texture
	self:SetCandyBarIcon("gotwood"..ele, BS:GetSpellIcon(tName) or GotWoodData[tName].icon)
	
	-- Set the time
	local time
	if tRank and GotWoodData[tName][tRank] and GotWoodData[tName][tRank].duration then
		time = GotWoodData[tName][tRank].duration
	else
		time = GotWoodData[tName].duration
	end
	self:SetCandyBarTime("gotwood"..ele, time)
	
	-- Store the life in our local table
	local life
	if tRank and GotWoodData[tName][tRank] and GotWoodData[tName][tRank].life then
		life = GotWoodData[tName][tRank].life
	else
		life = GotWoodData[tName].life
	end
	self.totemLife[ele] = {["name"] = tName, ["life"] = life}
	
	-- Start the candy bar
	self:StartCandyBar("gotwood"..ele)
end

function GotWood.candybar.prototype:SetTotemFromEvent(name, startTime, duration, icon, ele)
	-- Set the name
	self:SetCandyBarText("gotwood"..ele, name)
	
	-- Set the texture
	self:SetCandyBarIcon("gotwood"..ele, icon)
	
	-- Set the time
	self:SetCandyBarTime("gotwood"..ele, duration)

	-- life code removed. no need when totem death triggers events now.
	
	-- Start the candy bar
	self:StartCandyBar("gotwood"..ele)
end

function GotWood.candybar.prototype:RemoveTotem(ele)
	self:StopCandyBar("gotwood"..ele)
end

function GotWood.candybar.prototype:SetTotemDamage(tName, dmg)
	-- Search for the correct candy bar
	for ele,v in pairs(self.totemLife) do
		if string.find(tName, v.name) then
			v.life = v.life-dmg
			if v.life<0 then
				-- Stop candy bar
				self:StopCandyBar("gotwood"..ele)
				self.totemLife[ele] = nil
			end
			break
		end
	end
end

function GotWood.candybar.prototype:UpdateOrder()
	if self.db.profile.usecandygroup then
		self:RegisterCandyBarGroup("gotwoodgroup")
		self:SetCandyBarGroupPoint("gotwoodgroup", "TOPLEFT", GotWood.frame, "TOPLEFT", 10, -15)
		for k,v in ipairs(self.totemOrder) do
			self:RegisterCandyBarWithGroup("gotwood"..v, "gotwoodgroup")
		end
	else
		if self:IsCandyBarGroupRegistered("gotwoodgroup") then
			for k,v in ipairs(self.totemOrder) do
				self:UnregisterCandyBarWithGroup("gotwood"..v, "gotwoodgroup")
			end
			self:UnregisterCandyBarGroup("gotwoodgroup")
		end
		-- Set the positions
		for k,v in ipairs(self.totemOrder) do
			if k==1 then
				self:SetCandyBarPoint("gotwood"..v, "TOPLEFT", GotWood.frame, "TOPLEFT", 10, -15)
			else
				-- Attach the bars to the previous ones
				local previousBarFrame = AceLibrary("CandyBar-2.0").handlers["gotwood"..self.totemOrder[k-1]].frame
				self:SetCandyBarPoint("gotwood"..v, "TOPLEFT", previousBarFrame, "BOTTOMLEFT", 0, -2)
			end
		end
	end
end

function GotWood.candybar.prototype:GetOptions()
	return self.options
end

function GotWood.candybar.prototype:GetWidth()
	return self.db.profile.width + self.db.profile.height + 20
end

function GotWood.candybar.prototype:GetHeight()
	return 4 * self.db.profile.height + 30
end

function GotWood.candybar.prototype:SetScale(scale)
	for _,v in ipairs(self.totemOrder) do
		self:SetCandyBarScale("gotwood"..v, scale)
	end
end

function GotWood.candybar.prototype:UpdatePosition()
	self:UpdateOrder()
end

function GotWood.candybar.prototype:Disable()
	for k,v in ipairs(self.totemOrder) do
		self:StopCandyBar("gotwood"..v)
	end
end

--------------------------
--   Options Handling   --
--------------------------

function GotWood.candybar.prototype:SetTexture(name)
	self.db.profile.texture = name
	local texture = media:Fetch("statusbar", name)
	for k,v in ipairs(self.totemOrder) do
		self:SetCandyBarTexture("gotwood"..v, texture)
	end
end

function GotWood.candybar.prototype:SetFadeTime(time)
	self.db.profile.fade = time
	for k,v in ipairs(self.totemOrder) do
		self:SetCandyBarFade("gotwood"..v, time)
	end
end

function GotWood.candybar.prototype:SetWidth(width)
	self.db.profile.width = width
	for k,v in ipairs(self.totemOrder) do
		self:SetCandyBarWidth("gotwood"..v, width)
	end
	GotWood:RefreshFrame()
end

function GotWood.candybar.prototype:SetHeight(height)
	self.db.profile.height = height
	for k,v in ipairs(self.totemOrder) do
		self:SetCandyBarHeight("gotwood"..v, height)
	end
	GotWood:RefreshFrame()
end

function GotWood.candybar.prototype:UpdateTextColor(r,g,b)
	local hex = string.format("%02x%02x%02x", r*255, g*255, b*255)
	paint:RegisterHex(hex)
	self.db.profile.textcolor = hex
	
	for k,v in ipairs(self.totemOrder) do
		self:SetCandyBarTextColor("gotwood"..v, hex)
	end
end

function GotWood.candybar.prototype:UpdateTimerTextColor(r,g,b)
	local hex = string.format("%02x%02x%02x", r*255, g*255, b*255)
	paint:RegisterHex(hex)
	self.db.profile.timertextcolor = hex
	
	for k,v in ipairs(self.totemOrder) do
		self:SetCandyBarTimerTextColor("gotwood"..v, hex)
	end
end

function GotWood.candybar.prototype:SetFontSize(size)
	self.db.profile.fontsize = size
	
	for k,v in ipairs(self.totemOrder) do
		self:SetCandyBarFontSize("gotwood"..v, size)
	end
end

function GotWood.candybar.prototype:UseCandyGroup(value)
	self.db.profile.usecandygroup = value
	self:UpdateOrder()
end

-- Important: GotWood needs to know that this view exists, so add it to its table
-- The second abbreviation has to be the same name as the class name
GotWood:AddView("Candy Bars", "candybar")