﻿----------------------------
--      Declaration       --
----------------------------

GotWood = AceLibrary("AceAddon-2.0"):new("AceEvent-2.0", "AceDB-2.0", "AceConsole-2.0")
local BS = AceLibrary("Babble-Spell-2.2")
local dewdrop = AceLibrary("Dewdrop-2.0")
local deformat = AceLibrary("Deformat-2.0")

local L = AceLibrary("AceLocale-2.2"):new("GotWood")
L:RegisterTranslations("enUS", function() 
	return {
	["air"] = true,
	["water"] = true,
	["earth"] = true,
	["fire"] = true,
	["totem"] = true,
	
	["Lock"] = true,
	["Lock the frame"] = true,
	["Scale"] = true,
	["Scale the frame"] = true,
	["Order"] = true,
	["The order of the totems"] = true,
	["Remove Timer on Death"] = true,
	["Remove all timers if the player dies"] = true,
	["Select View"] = true,
	["Select your preferred view"] = true,
	["View Options"] = true,
	["Specific options for the current selected view"] = true,

	["Change order of type %s."] = true,
	}
end);

L:RegisterTranslations("koKR", function() 
	return {
	["air"] = "바람",
	["water"] = "물",
	["earth"] = "대지",
	["fire"] = "불",
    ["totem"] = "토템",
	
	["Lock"] = "고정",
	["Lock the frame"] = "창의 위치를 고정합니다",
	["Scale"] = "크기",
	["Scale the frame"] = "창의 크기를 변경합니다",
	["Order"] = "순서",
	["The order of the totems"] = "토템의 순서를 변경합니다",
	["Remove Timer on Death"] = "사망시 타이머 제거",
	["Remove all timers if the player dies"] = "플레이어가 사망할 때에 모든 타이머를 제거합니다",
	["Select View"] = "표시방식 선택",
	["Select your preferred view"] = "표시 방식을 선택합니다",
	["View Options"] = "보기 설정",
	["Specific options for the current selected view"] = "현재 선택된 표시 방식에 특정한 설정을 더합니다",

	["Change order of type %s."] = "%s 종류의 순서를 변경합니다",
	}
end);

L:RegisterTranslations("deDE", function() 
	return {
	["air"] = "Luft",
	["water"] = "Wasser",
	["earth"] = "Erde",
	["fire"] = "Feuer",
	["totem"] = "totem",
	
	["Lock"] = "Sperren",
	["Lock the frame"] = "Fenster sperren",
	["Scale"] = "Skalierung",
	["Scale the frame"] = "Fenster skalieren",
	["Order"] = "Reihenfolge",
	["The order of the totems"] = "Die Reihenfolge der Totems",
	["Remove Timer on Death"] = "Bei Tod Totemanzeige ausschalten",
	["Remove all timers if the player dies"] = "Die Totemanzeige ausschalten wenn der Spieler stirbt",
	["Select View"] = "Ansicht wählen",
	["Select your preferred view"] = "Die bevorzugte Ansicht auswählen",
	["View Options"] = "Ansichts-Optionen",
	["Specific options for the current selected view"] = "Spezifische Optionen der Ansicht ändern",

	["Change order of type %s."] = "Position ändern für Typ %s.",
	}
end)
local LData = AceLibrary("AceLocale-2.2"):new("GotWoodData")

-- Store the views in this table
local views = {}

local defaults = {
	Locked = false,
	View = "candybar",
	Scale = 1.0,
	RemoveOnDeath = true,

	totemorder = {"air","water","earth","fire"}
}
GotWood:RegisterDB("GotWoodDB")
GotWood:RegisterDefaults('profile', defaults)

----------------------------
--      Main Functions    --
----------------------------

function GotWood:AddView(fullName, name)
	views[name] = fullName
end

function GotWood:OnInitialize()
	-- Command structure
	self.options = {
		type="group",
		args = {
			addonheader = {
				name = "GotWood?",
				type = "header",
				order = 1,
			},
			lock = {
				name = L["Lock"], type = "toggle",
				desc = L["Lock the frame"],
				get = function() return self.db.profile.Locked end,
				set = function(v)
					self:SetLocked(v)
				end,
				order = 2
			},
			scale = {
				name = L["Scale"], type = "range",
				desc = L["Scale the frame"],
				get = function() return self.db.profile.Scale end,
				set = function(v)
					self.db.profile.Scale = v
					self:UpdateScale()
				end,
				min = 0.5,
				max = 1.5,
				step = 0.05,
				order = 3
			},
			order = {
				name = L["Order"], type = "group",
				desc = L["The order of the totems"],
				args = {
				},
				order = 4,
			},
			removeondeath = {
				name = L["Remove Timer on Death"], type = "toggle",
				desc = L["Remove all timers if the player dies"],
				get = function() return self.db.profile.RemoveOnDeath end,
				set = function(v)
					self.db.profile.RemoveOnDeath = v
				end,
				order = 5
			},
			blankone = {
				type = "header",
				order = 6
			},
			view = {
				name = L["Select View"], type = 'text',
				desc = L["Select your preferred view"],
				get = function()
					return self.db.profile.View
				end,
				set = function(name)
					self:SetView(name)
				end,
				validate = views,
				order = 7,
			},
			viewoptions = {
				name = L["View Options"], type = "group",
				desc = L["Specific options for the current selected view"],
				args = {},
				order = 8,
			},
		}
	}

	self.formatStrings = {
		SPELLLOGCRITOTHEROTHER,
		SPELLLOGCRITSCHOOLOTHEROTHER,
		SPELLLOGOTHEROTHER,
		SPELLLOGSCHOOLOTHEROTHER,
		COMBATHITCRITOTHEROTHER,
		COMBATHITOTHEROTHER,
	}

	-- slot mapping for the blizzard slot number to a string used by GotWood
	self.slotmap = {
		[1] = "fire",
		[2] = "earth",
		[3] = "water",
		[4] = "air"
	}
	self.slotmaprev = {
		["fire"] = 1,
		["earth"] = 2,
		["water"] = 3,
		["air"] = 4
	}
	-- temporary check what realm we're on. this can be removed once 2.4 is live everywhere 
	-- along with a lot of other code in here.
	local _, wowbuild, _ = GetBuildInfo()
	if ( tonumber(wowbuild) > 7897 ) then
		self.api24 = true
	else
		self.api24 = false
	end
	
	
	-- Register everything
	self:FillOrder()
	self:RegisterChatCommand({"/gotwood","/gw"}, self.options )
end

function GotWood:OnEnable()
	-- Initialize the view
	if not self[self.db.profile.View] then
		self:Print("Old GotWood Version detected. Upgraded to the new one!")
		self:ResetDB("profile")
	end
	self.view = self[self.db.profile.View]:new(self.db.profile.totemorder)
	self.options.args.viewoptions.args = self.view:GetOptions()

	-- Initialize the main GotWood Frame
	self:InitFrame()

	-- Update the position of the view
	self.view:UpdatePosition()

	-- Updates the scale of the frame and the view
	self:UpdateScale()

	dewdrop:Register(self.frame,
					'children', function()
						dewdrop:FeedAceOptionsTable(self.options)
					end
					)

	if ( self.api24 == true ) then
		-- use new 2.4 api
		--self:Print("GotWood: Using new 2.4 API.")
		-- new in 2.4, event fired when something in the totemsetup changes (this includes all kinds of totem removals aswell)
		self:RegisterEvent("PLAYER_TOTEM_UPDATE")
	else
		--self:Print("GotWood: Using old combatlog parsing.")
		-- Watch for damage done to our totems
		self:RegisterEvent("CHAT_MSG_SPELL_CREATURE_VS_SELF_DAMAGE", "TotemDamage")
		self:RegisterEvent("CHAT_MSG_COMBAT_CREATURE_VS_SELF_HITS", "TotemDamage")
		self:RegisterEvent("CHAT_MSG_SPELL_HOSTILEPLAYER_DAMAGE", "TotemDamage")
		self:RegisterEvent("CHAT_MSG_COMBAT_HOSTILEPLAYER_HITS", "TotemDamage")
		
		-- The new spell event in tbc to watch for
		self:RegisterEvent("UNIT_SPELLCAST_SUCCEEDED")
	end
	
	-- Event when the player dies
	self:RegisterEvent("PLAYER_DEAD")
end

function GotWood:OnDisable()
	self.view:Disable()

	self.frame:Hide()
	self.gwfs = nil
	self.frame = nil
end

function GotWood:InitFrame()
	self.frame = CreateFrame("Frame", "GotWoodFrame", UIParent)
	self.frame:Hide()
	self.frame:EnableMouse(true)
	self.frame:SetFrameStrata("MEDIUM")
	self.frame:SetMovable(true)
	self.frame:SetWidth(self.view:GetWidth())
	self.frame:SetHeight(self.view:GetHeight())

	-- Create Font String
	self.gwfs = self.frame:CreateFontString("$parentTitle","ARTWORK","GameFontNormal")
	self.gwfs:SetText("GotWood?")
	self.gwfs:SetFont("Fonts\\skurri.ttf", 20, "THICKOUTLINE")
	self.gwfs:ClearAllPoints()
	self.gwfs:SetTextColor(0.6,0.2,0.1,1)
	self.gwfs:SetWidth(200)
	self.gwfs:SetHeight(25)
	self.gwfs:SetJustifyH("CENTER")
	self.gwfs:SetJustifyV("MIDDLE")
	self.gwfs:SetPoint("TOP",0,10)
	self.gwfs:Show()

	-- Backdrop options
	self.texture = self.frame:CreateTexture("$parentWood","BACKGROUND")
	self.texture:SetAllPoints(self.frame)
	self.texture:SetTexture("Interface\\Addons\\GotWood\\images\\wood.tga")
	self.texture:Show()

	self.frame:ClearAllPoints()
	self.frame:SetPoint("CENTER", WorldFrame, "CENTER", 0, 0)
	self.frame:SetScript("OnMouseDown",function()
		if ( arg1 == "LeftButton" ) then
			this:StartMoving()
		end
	end)
	self.frame:SetScript("OnMouseUp",function()
		if ( arg1 == "LeftButton" ) then
			this:StopMovingOrSizing()
			self:SavePosition()
		end
	end)
	self.frame:SetScript("OnHide",function() this:StopMovingOrSizing() end)

	-- Frame cannot be dragged off the screen
	self.frame:SetClampedToScreen(true)

	-- Loads the position of the frame
	self:LoadPosition()

	if not self.db.profile.Locked then
		self.frame:Show()
	end
end

function GotWood:UNIT_SPELLCAST_SUCCEEDED(player, sName, sRank)
	if GotWoodData[sName] then
		-- Get the rank
		local rank
		if sRank and sRank ~= "" then
			rank = tonumber(string.sub(sRank, string.find(sRank, "%d")))
		end
		-- check for trinket
		if (sName == BS["Mana Spring Totem"] and not rank) then
			sName = LData["Enamored Water Spirit"]
		end
		
		-- The view handles now everything
		self.view:SetTotem(sName, rank)
	elseif sName == BS["Totemic Call"] then
		for _,v in ipairs(self.db.profile.totemorder) do
			self.view:RemoveTotem(v)
		end
	end
end

function GotWood:PLAYER_DEAD()
	if not self.db.profile.RemoveOnDeath then return end
	for _,v in ipairs(self.db.profile.totemorder) do
		self.view:RemoveTotem(v)
	end
end

function GotWood:PLAYER_TOTEM_UPDATE(slot)
	local haveTotem, name, startTime, duration, icon = GetTotemInfo(slot)
	
	--if ( haveTotem ) then
	if ( haveTotem and name ~= "") then
		-- New totem created
		-- The view handles now everything
		local timeleft = GetTotemTimeLeft(slot) -- more consistent than duration for some reason
		self.view:SetTotemFromEvent(name, startTime, timeleft, icon, self.slotmap[slot])
	--else
	elseif ( haveTotem ) then 
		--self:Print("Totem death")
		-- Totem just got removed or killed.
		self.view:RemoveTotem(self.slotmap[slot])
	end
end

-- useless after 2.4
function GotWood:TotemDamage(msg)
	if not string.find(string.lower(msg), string.lower(L["totem"])) then return end
	local totem, dmg
	for _,v in ipairs(self.formatStrings) do
		local arg1, arg2, arg3, arg4 = deformat(msg, v)
		if arg4 and type(arg4)=='number' then
			totem, dmg = arg3, arg4
			break
		elseif arg2 and type(arg3)=='number' then
			totem, dmg = arg2, arg3
			break
		end
	end

	if totem and dmg then
		self.view:SetTotemDamage(totem, dmg)
	end
end

function GotWood:SavePosition()
	local scale = self.frame:GetEffectiveScale()
	local worldscale = UIParent:GetEffectiveScale()

	local x,y = self.frame:GetLeft()*scale,self.frame:GetTop()*scale - (UIParent:GetTop())*worldscale

	if not self.db.profile.Position then 
		self.db.profile.Position = {}
	end

	self.db.profile.Position.x = x
	self.db.profile.Position.y = y
end

function GotWood:LoadPosition()
	if(self.db.profile.Position) then
		local x = self.db.profile.Position.x
		local y = self.db.profile.Position.y
		local scale = self.frame:GetEffectiveScale()

		self.frame:SetPoint("TOPLEFT", UIParent,"TOPLEFT", x/scale, y/scale)
	else
		self.frame:SetPoint("CENTER", UIParent, "CENTER")
	end
end

function GotWood:SetLocked(lock)
	self.db.profile.Locked = lock
	if lock then
		self.frame:Hide()
	else
		self.frame:Show()
	end
end

function GotWood:SetView(view)
	self.db.profile.View = view

	-- Disable the current view
	self.view:Disable()

	-- Create the new one
	self.view = self[view]:new(self.db.profile.totemorder)
	self:UpdateTotemPosition()
	self:RefreshFrame()
	self.options.args.viewoptions.args = self.view:GetOptions()
end

function GotWood:FillOrder()
	local order = self.options.args.order.args
	for k,v in ipairs(self.db.profile.totemorder) do
		local num = k
		local string_num = tostring(k)
		local name = v
		
		order[string_num] = {}
		order[string_num].name = L[name]
		order[string_num].type = 'text'
		order[string_num].desc = string.format(L["Change order of type %s."], L[name])
		order[string_num].get = function() return string_num end
		order[string_num].set = function(num_new) self:ChangeOrder(num, num_new)end
		order[string_num].validate = {"1","2","3","4"}
		order[string_num].order = num
	end
end

function GotWood:ChangeOrder(old, new)
	local ele = self.db.profile.totemorder[old]

	table.remove(self.db.profile.totemorder, old)
	table.insert(self.db.profile.totemorder, new, ele)

	self:FillOrder()
	self:UpdateTotemPosition()
	dewdrop:Close(3)
	dewdrop:Refresh(2)
end

function GotWood:UpdateTotemPosition()
	self.view:UpdatePosition()
	self:UpdateScale()
end

function GotWood:UpdateScale()
	local scale = self.db.profile.Scale
	if self.frame then
		self.frame:SetScale(scale)
		self.view:SetScale(scale)
		self:LoadPosition()
	end
end

function GotWood:RefreshFrame()
	self.frame:SetWidth(self.view:GetWidth())
	self.frame:SetHeight(self.view:GetHeight())
end