local tonumber = tonumber
local time = time
local floor = math.floor

local TimerFu = AceLibrary("AceAddon-2.0"):new("FuBarPlugin-2.0", "AceDB-2.0", "AceConsole-2.0", "AceEvent-2.0")
TimerFu:RegisterDB("TimerFuDB")
TimerFu:RegisterDefaults('profile', {
	counting = false,
	message = true,
	sound = true,
	loop = true,
	alarm = true,
})

TimerFu.hasIcon = "Interface\\Icons\\INV_Misc_PocketWatch_01"
TimerFu.defaultPosition = "CENTER"

local L = AceLibrary("AceLocale-2.2"):new("TimerFu")

local tablet = AceLibrary("Tablet-2.0")
local dewdrop = AceLibrary("Dewdrop-2.0")
local crayon = AceLibrary("Crayon-2.0")

local function getTimerString(value)
	local s, e, hours, minutes, seconds
	if not value then return end
	minutes = value:match("^(%d+)$")
	if minutes then
		return tonumber(minutes)*60
	end
	minutes, seconds = value:match("^(%d+):(%d+)$")
	if minutes then
		return tonumber(minutes)*60+tonumber(seconds)
	end
	hours, minutes, seconds = value:match("^(%d+):(%d+):(%d+)$")
	if hours then
		return tonumber(hours)*3600+tonumber(minutes)*60+tonumber(seconds)
	end
end

do
	local options = {
		desc = L["An addon to track time"],
		type = "group",
		args = {
			start = {
				name = L["start"],
				desc = L["Start counting"],
				type = "execute",
				func = function ()
					local p = TimerFu.db.profile
					if not p.counting then
						p.counting = true
						p.decounting = false
						p.target = time()
					end
				end,
			},
			toggle = {
				name = L["toggle"],
				desc = L["Toggle pause"],
				type = "execute",
				func = function ()
					if TimerFu.db.profile.counting then
						TimerFu:TogglePause()
					end
				end,
			},
			pause = {
				name = L["pause"],
				desc = L["Pause counting"],
				type = "execute",
				func = function ()
					local p = TimerFu.db.profile
					if p.counting and not p.paused then
						TimerFu:TogglePause()
					end
				end,
			},
			unpause = {
				name = L["unpause"],
				desc = L["Restart counting if paused"],
				type = "execute",
				func = function ()
					local p = TimerFu.db.profile
					if p.counting and p.paused then
						TimerFu:TogglePause()
					end
				end,
			},
			reset = {
				name = L["reset"],
				desc = L["Reset the timer"],
				type = "execute",
				func = function ()
					TimerFu:OnDoubleClick()
				end,
			},
			alarm = {
				name = L["alarm"],
				desc = L["Set alarm mode"],
				type = "toggle",
				get = function () return TimerFu.db.profile.alarm end,
				set = function (v) TimerFu.db.profile.alarm = v end,
				disabled = function () return TimerFu.db.profile.loop end,
			},
			set = {
				name = L["set"],
				desc = L["Set the counter for a given time"],
				usage= L["<time>"],
				type = "text",
				get = false,
				set = function (v) TimerFu:StartTimerString(v) end,
				validate = getTimerString,
			},
			message = {
				name = L["message"],
				desc = L["Show a message in ChatFrame when the timer reaches 0"],
				type = "toggle",
				get = function () return TimerFu.db.profile.message end,
				set = function (v) TimerFu.db.profile.message = v end,
			},
			sound = {
				name = L["sound"],
				desc = L["Play a sound when the timer reaches 0"],
				type = "toggle",
				get = function () return TimerFu.db.profile.sound end,
				set = function (v) TimerFu.db.profile.sound = v end,
			},
			loop = {
				name = L["loop"],
				desc = L["Loop the counter when the timer reaches 0"],
				type = "toggle",
				get = function () return TimerFu.db.profile.loop end,
				set = function (v)
					local p = TimerFu.db.profile
					p.loop = v 
					if p.loop then
						p.alarm = false
					end
				end,
			},
		}
	}

	function TimerFu:OnInitialize()
		self:RegisterChatCommand(L["CHAT_COMMANDS"], options)
	end
end

function TimerFu:OnEnable()
	self:ScheduleRepeatingEvent(self.UpdateText, 1, self)
end

function TimerFu:FormatTime(s)
	local t
	if s < 0 then
		t = "-"
		s = -s
	else
		t = ""
	end
	local h, m
	h = floor(s / 3600); s = s - (h * 3600)
	m = floor(s / 60); s = s - (m * 60)
	return ("%s%2.2d:%2.2d:%2.2d"):format(t, h, m, s)
end

function TimerFu:OnTextUpdate()
	local p = self.db.profile
	local t, color
	if not p.counting then
		t = 0
		color = crayon.White
	else
		if p.paused then
			color = crayon.Orange
			t = - p.remaining
		else
			color = crayon.Green
			t = time() - p.target
			if p.decounting and t >= 0 then
				if p.message and t < 1 then
					self:Print(L["Timer has finished !"])
				end
				if p.sound then
					PlaySound("JewelcraftingFinalize")
				end
				if p.loop then
					p.target = time() + p.last_value
				elseif p.alarm then
					color = crayon.Red
				else
					p.counting = false
				end
			end
		end
	end
	self:SetText(color(crayon, self:FormatTime(t)))
end

function TimerFu:OnTooltipUpdate()
	tablet:SetHint(L["Tip Hint"])
end

function TimerFu:StartTimer(count)
	local p = self.db.profile
	p.counting = true
	p.decounting = true
	if p.paused then
		p.remaining = count
	else
		p.target = time() + count
	end
	self:UpdateText()
end

function TimerFu:StartTimerString(value)
	value = getTimerString(value)
	if value then
		local p = self.db.profile
		p.counting = true
		p.decounting = true
		p.last_value = value
		if p.paused then
			p.remaining = value
		else
			p.target = time() + value
		end
		self:UpdateText()
	end
end

function TimerFu:OnMenuRequest(level, value)
	if level == 1 then
		local p = self.db.profile
		dewdrop:AddLine(
			'text', L["message"],
			'checked', p.message,
			'func', function () p.message = not p.message end,
			'tooltipTitle', L["message"],
			'tooltipText', L["Show a message in ChatFrame when the timer reaches 0"]
			)
		dewdrop:AddLine(
			'text', L["sound"],
			'checked', p.sound,
			'func', function () p.sound = not p.sound end,
			'tooltipTitle', L["sound"],
			'tooltipText', L["Play a sound when the timer reaches 0"]
			)
		dewdrop:AddLine(
			'text', L["loop"],
			'checked', p.loop,
			'func', function ()
				p.loop = not p.loop
				if p.loop then
					p.alarm = false
				end
			end,
			'tooltipTitle', L["loop"],
			'tooltipText', L["Loop the counter when the timer reaches 0"]
			)
		dewdrop:AddLine(
			'text', L["alarm"],
			'checked', p.alarm,
			'disabled', p.loop,
			'func', function () p.alarm = not p.alarm end,
			'tooltipTitle', L["alarm"],
			'tooltipText', L["Set alarm mode"]
			)
		dewdrop:AddLine(
			'closeWhenClicked', true,
			'text', L["Pause"],
			'func', function() self:TogglePause() end,
			'checked', p.paused
			)
		dewdrop:AddLine(
			'closeWhenClicked', true,
			'text', L["Reset"],
			'func', function() 
						p.counting = false 
						p.paused = false 
					end
			)
		dewdrop:AddLine(
			'closeWhenClicked', true,
			'text', L["Start counting"],
			'func', function() 
						p.counting = true 
						p.decounting = true 
						p.target = time()
						p.paused = false 
					end,
			'checked', p.paused
			)
		dewdrop:AddLine(
			'text', L["Start a Timer"],
			'hasArrow', true,
			'value', "timing"
		)
	elseif level == 2 then
		if value == "timing" then
			dewdrop:AddLine(
				'closeWhenClicked', true,
				'text', string.format(L["1 minute"], n),
				'func', self.StartTimer,
				'arg1', self,
				'arg2', 60
			)
			for _, n in ipairs({2, 5, 10, 15, 30, 60}) do
				dewdrop:AddLine(
					'closeWhenClicked', true,
					'text', string.format(L["%d minutes"], n),
					'func', self.StartTimer,
					'arg1', self,
					'arg2', n*60
				)
			end
			dewdrop:AddLine(
				'closeWhenClicked', true,
				'text', string.format(L["Other timing"], n),
				'hasArrow', true,
				'value', "timingOther",
				'hasEditBox', true,
				'editBoxText ', "00:00:00",
				'editBoxFunc',self.StartTimerString,
				'editBoxArg1',self,
				'editBoxValidateFunc ', getTimerString
			)
		end
	end
end

function TimerFu:TogglePause()
	local p = self.db.profile
	if p.paused then
		p.paused = false
		p.target = time() + p.remaining
	else
		p.paused = true
		p.remaining = p.target - time()
	end
end

function TimerFu:OnDoubleClick()
	local p = self.db.profile
	p.counting = false
	p.paused = false
	self:UpdateText()
end

function TimerFu:OnClick()
	local p = self.db.profile
	if p.paused or p.counting then
		self:TogglePause()
	else
		p.counting = true
		p.decounting = false
		p.target = time()
	end
	self:UpdateText()
end
