-----------------------------------------------------------------------
-- Locals
-----------------------------------------------------------------------

local ThreatLib
local ThreatKLHTMSupport
local Tablet

local updateBucketedEvents = {
	'Threat_Activate',
	'Threat_Deactivate',
	'Threat_PartyChanged',
	'FuBar_ThreatFu_UpdateRequest',
	'Threat_KLHTM_TargetChanged',
	'Threat_KLHTM_BroadcastEnabled',
	'Threat_KLHTM_BroadcastLoaded',
	'Threat_KLHTM_BroadcastDisabled',
}

local db
local threatMajor, threatMinor

local isGroupOfficer

local options = {
	type = "group"
}

local COLORS = {
	ThrCompat   = { r=0.2, g=1.0, b=0.2 },
	ThrIncompat = { r=1.0, g=0.5, b=0.2 },
	KTM         = { r=0.2, g=0.5, b=1.0 },
	nothing     = { r=1.0, g=0.2, b=0.2 },
	offline     = { r=0.4, g=0.4, b=0.4 },
}

-----------------------------------------------------------------------
-- Addon declaration
-----------------------------------------------------------------------

FuBar_ThreatFu = AceLibrary("AceAddon-2.0"):new(
	"AceEvent-2.0",
	"AceConsole-2.0",
	"AceDB-2.0",
	"FuBarPlugin-2.0"
)

local FuBar_ThreatFu = FuBar_ThreatFu

FuBar_ThreatFu:RegisterDB("FuBar_ThreatFuDB")

FuBar_ThreatFu.title = "ThreatFu"
FuBar_ThreatFu.OnMenuRequest = options
FuBar_ThreatFu.cannotHideText = true
FuBar_ThreatFu.hideWithoutStandby = true

local partyVersions = {}
local printedUpdateNotice

-----------------------------------------------------------------------
-- Helpers
-----------------------------------------------------------------------

function isGroupOfficer() 
	if GetNumRaidMembers() > 0 then
		return IsRaidLeader("player") or IsRaidOfficer("player")
	elseif GetNumPartyMembers() > 0 then
		return IsPartyLeader("player")
	else
		return false
	end
end 

-----------------------------------------------------------------------
-- Addon initialization
-----------------------------------------------------------------------

function FuBar_ThreatFu:OnInitialize()
	ThreatLib = AceLibrary('Threat-1.0')
	ThreatKLHTMSupport = ThreatLib:GetModule('KLHTMSupport')

	self.revision = tonumber(("$Revision: 48847 $"):match("%d+"))
	self.version = self.version .. "." .. self.revision
	
	self:RegisterDefaults('profile', {
		enableBroadcast = true,
		enableDebug = false,
	})

	db = self.db.profile
	
	options.args = {
		broadcast = {
			type = "toggle",
			name = "KLHTM broadcast",
			desc = "Enable ThreatLib's KLHTM broadcast",
			get = "IsBroadcastEnabled",
			set = "ToggleBroadcast",
			disabled = function() return _G.klhtm end, 
			order = 100,
		},
		KLHTM = {
			type = "header",
			name = " ",
			order = 119,
		},
		version_query = {
			type = "execute",
			name = "Query KLHTM versions",
			desc = "Issue a KLHTM 'versionquery'",
			handler = ThreatKLHTMSupport,
			func = "QueryKLHTMVersions",
			hidden = function() return not ThreatLib:IsKLHTMSupportLoaded() end,
			disabled = function() return not isGroupOfficer() end,
			order = 120,
		},
		target = {
			type = "execute",
			name = "Set master target",
			desc = "Set your current target as the KLTHM master target",
			handler = ThreatKLHTMSupport,
			func = "SetKLHTMMasterTarget",
			hidden = function() return not ThreatLib:IsKLHTMSupportLoaded() end,
			disabled = function() return not isGroupOfficer() end,
			order = 121,
		},
		clear_target = {
			type = "execute",
			name = "Clear master target",
			desc = "Clear the KLTHM master target",
			handler = ThreatKLHTMSupport,
			func = "ClearKLHTMMasterTarget",
			hidden = function() return not ThreatLib:IsKLHTMSupportLoaded() end,
			disabled = function() return not isGroupOfficer() end,
			order = 122,
		},
		show_target = {
			type = "toggle",
			name = "Show master target",
			desc = "Show master target name in FuBar",
			get = function() return db.showTarget end,
			set = function(v) db.showTarget = v self:UpdateDisplay() end,
			hidden = function() return not ThreatLib:IsKLHTMSupportLoaded() end,
			order = 123,
		},
		debugHeader = {
			type = "header",
			name = " ",
			order = 149,
		},
		debug = {
			type = "toggle",
			name = "Debugging",
			desc = "Enable ThreatLib debugging output",
			get = "IsDebugEnabled",
			set = "ToggleDebug",
			order = 150,
		},
		dump_rev = {
			type = "execute",
			name = "Dump ThreatLib revisions",
			desc = "Dump ThreatLib revisions of party and guild members.",
			handler = ThreatLib,
			func = "DumpQueriedRevisions",
			order = 151,
		},
	}
	
	self:RegisterChatCommand({"/ttk", "/threattoktm", "/threatfu"}, options) 		
end

function FuBar_ThreatFu:OnEnable(first)
	if first then
		self:RegisterFakeCommands()
	end

	playerName = UnitName("player")	
	threatMajor, threatMinor = ThreatLib:GetLibraryVersion()
	
	self:RegisterEvent('Threat_OutOfDateNotice', 'OutOfDateNotice')
	self:RegisterBucketEvent(updateBucketedEvents, 0.3, 'UpdateDisplay')
	
	self:RegisterEvent('Threat_PartyChanged', 'CleanUpParty')
	self:RegisterEvent('Threat_KLHTM_NewPeer', 'NewPeer')
	self:RegisterEvent('Threat_KLHTM_VersionResponse', 'VersionResponse')
	self:RegisterEvent('Threat_KLHTM_TargetChanged', 'TargetChanged')
		
	ThreatLib.DebugEnabled = db.enableDebug and true or false
	ThreatLib:EnableKLHTMBroadcast(db.enableBroadcast)
end

function FuBar_ThreatFu:OnDisable()
	ThreatLib:EnableKLHTMBroadcast(false)
	ThreatLib.DebugEnabled = nil
end

function FuBar_ThreatFu:OutOfDateNotice(version, latestSeenRevision)
	if not printedUpdateNotice then
		self:Print("Your version of ThreatLib is out of date. Please consider upgrading.")
		printedUpdateNotice = true
	end
end

-----------------------------------------------------------------------
-- KTM chat command fake
-----------------------------------------------------------------------

local function noop() end

local function fakeOption(name) 
	return {
		type = 'execute',
		name = name,
		desc = 'Fake command',
		func = noop,
	}
end

function FuBar_ThreatFu:RegisterFakeCommands()

	-- KTM compat commands
	self:RegisterChatCommand({"/ktm", "/klhtm", "/klhthreatmeter"}, {
		type = "group",
		args = {
			boss = fakeOption('boss'),
			test = fakeOption('test'),
			resetraid = fakeOption('resetraid'),
			gui = fakeOption('gui'),
			version = {
				name = "Version",
				desc = "Version",
				type = "group",
				args = {
					query = {
						type = "execute",
						name = "Query",
						desc = "Query party member versions.",
						handler = ThreatKLHTMSupport,
						func = "QueryKLHTMVersions",
					},
					notify = fakeOption('notify'),
				},
			},
			mastertarget = {
				type = "execute",
				name = "Set master target",
				desc = "Set (or clear) KLHTM master target.",
				func = function()
					if UnitExists('target') and UnitCanAttack('player', 'target') then
						ThreatKLHTMSupport:SetKLHTMMasterTarget()
					else
						ThreatKLHTMSupport:ClearKLHTMMasterTarget()
					end
				end,
			},
			enable = {
				type = "execute",
				name = "Enable",
				desc = "Enable KLHTM broadcast",
				func = function() self:ToggleBroadcast(true) end,
			},
			disable = {
				type = "execute",
				name = "Disable",
				desc = "Disable KLHTM broadcast",
				func = function() self:ToggleBroadcast(false) end,
			},
		},
	})

end

-----------------------------------------------------------------------
-- Settings
-----------------------------------------------------------------------

function FuBar_ThreatFu:IsBroadcastEnabled()
	return db.enableBroadcast
end

function FuBar_ThreatFu:ToggleBroadcast(value)
	db.enableBroadcast = value and true or false
	ThreatLib:EnableKLHTMBroadcast(db.enableBroadcast)
	self:Print("KLHTM broadcast %s", db.enableBroadcast and "enabled" or "disabled")
end

function FuBar_ThreatFu:IsDebugEnabled()
	return db.enableDebug
end

function FuBar_ThreatFu:ToggleDebug(value)
	db.enableDebug = value and true or false
	ThreatLib.DebugEnabled = db.enableDebug
	self:Print("ThreatLib debug %s", db.enableDebug and "enabled" or "disabled")
end

-----------------------------------------------------------------------
-- Event handling
-----------------------------------------------------------------------

function FuBar_ThreatFu:NewPeer(sender)
	partyVersions[sender] = '??.??'
	self:TriggerEvent('FuBar_ThreatFu_UpdateRequest')
end

function FuBar_ThreatFu:VersionResponse(sender, version)
	partyVersions[sender] = version
	self:TriggerEvent('FuBar_ThreatFu_UpdateRequest')
end

function FuBar_ThreatFu:CleanUpParty()
	local revs = ThreatLib:GetPartyRevisions()
	local dirty = false
	for name in pairs(partyVersions) do
		if not revs[name] then
			partyVersions[name] = nil
			dirty = true
		end
	end
	if dirty then
		self:TriggerEvent('FuBar_ThreatFu_UpdateRequest')
	end
end

function FuBar_ThreatFu:TargetChanged(target)
	if target then
		self:Print('Master target : %s', target);
	else
		self:Print('Master target reset');
	end
	self:TriggerEvent('FuBar_ThreatFu_UpdateRequest')
end

-----------------------------------------------------------------------
-- Fubar Plugin
-----------------------------------------------------------------------

function FuBar_ThreatFu:OnTextUpdate()
	if db.showTarget and ThreatKLHTMSupport.KLHTMtarget then
		self:SetText(ThreatKLHTMSupport.KLHTMtarget)
	elseif ThreatKLHTMSupport:IsKLHTMBroadcastActive() then
		self:SetText("|cffffffffThreatFu|r")
	else
		self:SetText("|cff777777ThreatFu|r")
	end
end

function FuBar_ThreatFu:OnClick()
	if IsShiftKeyDown() then
		self:ToggleBroadcast(not db.enableBroadcast)
	elseif ThreatKLHTMSupport.KLHTMtarget then
		ThreatKLHTMSupport:ClearKLHTMMasterTarget();
	else
		ThreatKLHTMSupport:SetKLHTMMasterTarget();
	end
end

function FuBar_ThreatFu:OnTooltipUpdate()
	if not Tablet then
		Tablet = AceLibrary('Tablet-2.0')
	end
	
	local cat
	
	cat = Tablet:AddCategory("text", "Status", "columns", 2)
	cat:AddLine("text", threatMajor, "text2", ThreatLib:IsActive() and "Active" or "Inactive")

	local status, active
	active = ThreatKLHTMSupport:IsKLHTMBroadcastActive()
	
	if not ThreatLib:IsKLHTMSupportLoaded() then
		status = "Not loaded"
	elseif not ThreatLib:IsKLHTMBroadcastEnabled() then
		status = "Disabled"
	elseif active then
		status = "Active"
	else
		status = "Inactive"
	end	
	cat:AddLine("text", "KLHTM broadcast", "text2", status)

	if db.enableBroadcast then
		if isGroupOfficer() and active then
			Tablet:SetHint("Click to (re)set master target, Shift-Click to disable KLHTM broadcast.")
		else
			Tablet:SetHint("Shift-Click to disable KLHTM broadcast.")
		end
	else
		Tablet:SetHint("Shift-Click to allow KLHTM broadcast.")
	end
	
	if ThreatLib:IsActive() then
		cat = Tablet:AddCategory("columns", 2)
		cat:AddLine("text", "Master target", "text2", ThreatKLHTMSupport.KLHTMtarget or "-")
	
		local revs = ThreatLib:GetPartyRevisions()
		cat = Tablet:AddCategory("columns", 3, "text", "Player", "text2", "KLHTM", "text3", "Threat")
		
		for name, rev in pairs(revs) do
			
			local revision = rev:match(" v(%S+)")
			
			local ktmVer, threatVer, color = "-", "-", COLORS.offline
			if UnitIsConnected(name) then
				color = COLORS.nothing
				if revision then
					if rev:find("compatible") then
						threatVer, color = revision..' (!)', COLORS.ThrIncompat
					else
						threatVer, color = revision, COLORS.ThrCompat
					end
				elseif partyVersions[name] then
					ktmVer, color = partyVersions[name], COLORS.KTM
				end
			end
			cat:AddLine(
				"text", name, "textR", color.r, "textG", color.g, "textB", color.b,
				"text2", ktmVer,
				"text3", threatVer
			)
			
		end
	end
	
	cat = Tablet:AddCategory("text", "Versions", "columns", 3)
	cat:AddLine("text", threatMajor, "text2", " ", "text3", threatMinor)
	
end
