local tablet = AceLibrary("Tablet-2.0")
local dewdrop = AceLibrary("Dewdrop-2.0")
local abacus = AceLibrary("Abacus-2.0")
local crayon = LibStub("LibCrayon-3.0")
local L = AceLibrary("AceLocale-2.2"):new("RestFu")

RestFu = AceLibrary("AceAddon-2.0"):new("FuBarPlugin-2.0", "AceDB-2.0", "AceEvent-2.0", "AceConsole-2.0")
RestFu.hasIcon = "Interface\\AddOns\\FuBar_RestFu\\icon.tga"
RestFu.defaultPosition = "RIGHT"
RestFu.hasNoText = true
RestFu.hideWithoutStandby = true
RestFu.clickableTooltip = true
RestFu:RegisterDB("RestFuDB")

local pairs = pairs
local ipairs = ipairs
local string_format = string.format
local table_insert = table.insert
local table_sort = table.sort

local maxLevel = MAX_PLAYER_LEVEL_TABLE[GetAccountExpansionLevel()]

local options = {
	handler = RestFu,
	type = 'group',
	args = {
		filters = {
			type = "group",
			name = L["Filter"],
			desc = L["Filter Tooltip Characters"],
		},
		purge = {
			type = "group",
			name = L["Purge"],
			desc = L["Purge Characters"],
		},
	},
}

function RestFu:OnEnable()
	self:RegisterEvent("PLAYER_UPDATE_RESTING", "Save")
	self:RegisterEvent("PLAYER_XP_UPDATE", "Save")
	self:RegisterEvent("PLAYER_REGEN_ENABLED", "Save")
	self:RegisterEvent("TIME_PLAYED_MSG")

	self:ScheduleRepeatingEvent("RestFu_TimePlayed", self.OnUpdate_TimePlayed, 1, self)

	self:Save()
end

function RestFu:OnMenuRequest()
	if not self.cached_options then 
		-- Purge/Filter settings
		options.args.filters.args = {}
		options.args.purge.args = {}
		local filter = options.args.filters.args
		local purge = options.args.purge.args

		for realm, realm_data in pairs(self.data) do
			local cur_realm = realm
			if not filter[cur_realm] then
				filter[cur_realm] = {
					type = 'group',
					name = cur_realm,
					desc = cur_realm,
					args = {},
				}
				purge[cur_realm] = {
					type = 'group',
					name = cur_realm,
					desc = cur_realm,
					args = {},
				}
			end
			for char, _ in pairs(realm_data) do
				local cur_char  = char
				local fullchar = ("%s %s %s"):format(char, L["of"], realm)
				filter[cur_realm].args[fullchar] = {
					type = 'toggle',
					name = fullchar,
					desc = L["Filter"] .. " " .. fullchar,
					set = function()
						RestFu:ToggleFiltered(cur_realm, cur_char)
					end,
					get = function()
						return not RestFu:IsFiltered(cur_realm, cur_char)
					end
				}
				
				purge[cur_realm].args[fullchar] = {
					type = 'execute',
					name = fullchar,
					desc = L["Purge"] .. " " .. fullchar,
					func = function()
						RestFu:Purge(cur_realm, cur_char)
					end
				}
			end
		end
		self.cached_options = options
	end
	dewdrop:FeedAceOptionsTable(self.cached_options)
end

function RestFu:IsFiltered(realm, person)
	return self.data[realm][person].hidden
end

function RestFu:ToggleFiltered(realm, person)
	self.data[realm][person].hidden = not self.data[realm][person].hidden
end

function RestFu:Purge(realm, person)
	self.data[realm][person] = nil
	if not next(self.data[realm]) then
		self.data[realm] = nil
	end
	self:ReIndex()
	self.cached_options = nil
end

function RestFu:ReIndex()
	if not self:IsEventScheduled("RestFu_OnUpdate") then 
		self:ScheduleRepeatingEvent("RestFu_OnUpdate", self.OnUpdate, 3, self) 
		self:ScheduleRepeatingEvent("RestFu_UpdateTooltip", self.UpdateTooltip, 60, self) 
	end

	if self.db.raw.full and self.db.raw.full.realms then
		for realm, realm_data in pairs(self.db.raw.full.realms) do
			for char, char_data in pairs(realm_data) do
				if not self.db.account[realm] then
					self.db.account[realm] = {}
				end
				if not self.db.account[realm][char] then 
					self.db.account[realm][char] = char_data
					self:Print(L["RestFu: Converted: %s of %s"], char, realm)
				end
			end
		end
		self.db.raw.full = nil
		self:Print(L["RestFu: Converted DB to AceDB-2.0"])
	end
	
	if not self.db.account[self.REALM or GetRealmName()] then
		self.db.account[self.REALM or GetRealmName()] = {}
	end
	if not self.db.account[self.REALM or GetRealmName()][self.NAME or UnitName("player")] then
	 	self.db.account[self.REALM or GetRealmName()][self.NAME or UnitName("player")] = {}
	end

	self.realmList = {}
	self.charList = {}

	self.data = self.db.account
	for realm, realm_data in pairs(self.db.account) do
		table_insert(self.realmList, realm)
		for char, char_data in pairs(realm_data) do
			if (not self.charList[realm]) then 
				self.charList[realm] = {}
			end
			table_insert(self.charList[realm], char)
		end
	end

	self.myData = self.db.account[self.REALM or GetRealmName()][self.NAME or UnitName("player")]

	self:Save()
	self:OnUpdate()
end

local percentPerSecond = 0.05 / 28800

function RestFu:OnTooltipUpdate()
	self:Save()
	self:OnUpdate()

	local oneRealm = table.getn(self.realmList) == 1
	
	local hasHorde, hasAlly
	for _,realm in ipairs(self.realmList) do
		for _,char in ipairs(self.charList[realm]) do
			if not self:IsFiltered(realm, char) then
				local data = self.data[realm][char]
				if data.faction == "Horde" then
					hasHorde = true
				elseif data.faction == "Alliance" then
					hasAlly = true
				end
				if hasHorde and hasAlly then
					break
				end
			end
		end
		if hasHorde and hasAlly then
			break
		end
	end
	
	local supercat = tablet:AddCategory(
		'columns', 6
	)
	for _,realm in ipairs(self.realmList) do
		local cat = supercat:AddCategory(
			'columns', 6,
			'text', oneRealm and L["Name"] or realm,
			'text2', L["Time played"],
			'text3', L["Time to rest"],
			'text4', L["Current XP"],
			'text5', L["Rest XP"],
			'text6', L["Zone"],
			'child_text1R', 1,
			'child_text1G', 1,
			'child_text1B', 0,
			'child_text2R', 1,
			'child_text2G', 1,
			'child_text2B', 0,
			'child_text3R', 1,
			'child_text3G', 1,
			'child_text3B', 0,
			'child_text4R', 1,
			'child_text4G', 1,
			'child_text4B', 1,
			'child_text6R', 1,
			'child_text6G', 1,
			'child_text6B', 1,
			'func', "RemoveRealm",
			'arg1', self,
			'arg2', realm
		)

		for _,char in ipairs(self.charList[realm]) do
			if not self:IsFiltered(realm, char) then
				local data = self.data[realm][char]
				if (data.localclass == nil) then data.localclass = "PRIEST" end
				local classColor = string_format("%02x%02x%02x", RAID_CLASS_COLORS[data.localclass].r * 255, RAID_CLASS_COLORS[data.localclass].g * 255, RAID_CLASS_COLORS[data.localclass].b * 255)
				if data.level ~= maxLevel then
					local r, g, b = crayon:GetThresholdColor(data.restXP / data.nextXP, 0, 0.5, 1, 1.25, 1.5)
					local timePassed = data.restXP / data.nextXP / percentPerSecond
					local timeToMax = 864000 - timePassed
					if not data.isResting then
						timeToMax = timeToMax * 4
					end
					local playedTime
					if realm == self.REALM and char == self.NAME and self.timePlayed then
						playedTime = self.timePlayed + time() - self.timePlayedMsgTime
					else
						playedTime = data.timePlayed or 0
					end
					local text = ("|cff%s%s|r [|cffffffff%d|r]"):format(classColor, char, data.level or 0)
					if hasHorde and hasAlly then
						if data.faction == "Horde" then
							text = text .. " |cffcf0000(H)|r"
						elseif data.faction == "Alliance" then
							text = text .. " |cffff3f3f(A)|r"
						end
					end
					cat:AddLine(
						'text', text,
						'text2', abacus:FormatDurationCondensed(playedTime, true, true),
						'text3', timeToMax > 0 and abacus:FormatDurationCondensed(timeToMax, true, true) or format("|cff00ff00%s|r", L["Fully rested"]),
						'text4', ("%.0f%%"):format(data.currXP / data.nextXP * 100),
						'text5', ("(%+.0f%%)"):format(data.restXP / data.nextXP * 100),
						'text6', data.zone or L["Unknown"],
						'text5R', r,
						'text5G', g,
						'text5B', b,
						'func', "RemoveChar",
						'arg1', self,
						'arg2', realm,
						'arg3', char
					)
				else
					local playedTime
					if realm == self.REALM and char == self.NAME and self.timePlayed then
						playedTime = self.timePlayed + time() - self.timePlayedMsgTime
					else
						playedTime = data.timePlayed or 0
					end
					cat:AddLine(
						'text', ("|cff%s%s|r [|cffffffff%d|r]"):format(classColor, char, data.level),
						'text2', abacus:FormatDurationCondensed(playedTime, true, true),
						'text6', ("|cffffffff%s|r"):format(data.zone or L["Unknown"])
					)
				end
			end
		end
	end
	local cat = tablet:AddCategory(
		'columns', 2
	)
	local total = 0
	for _,realm in ipairs(self.realmList) do
		for _,char in ipairs(self.charList[realm]) do
			if realm == self.REALM and char == self.NAME and self.timePlayed then
				total = total + self.timePlayed + time() - self.timePlayedMsgTime
			else
				total = total + (self.data[realm][char].timePlayed or 0)
			end
		end
	end
	cat:AddLine(
		'text', L["Total time played"],
		'text2', abacus:FormatDurationExtended(total, true, true),
		'textR', 1,
		'textG', 1,
		'textB', 1
	)
	
	tablet:SetHint(L["TOOLTIP_HINT"])
end

function RestFu:Save()
	local zone = GetRealZoneText()
	if zone == nil or zone == "" then
		self:ScheduleEvent(self.Save, 5, self)
	elseif UnitLevel("player") ~= 0 then
		if not self.myData then
			self:ReIndex()
		end
		
		local t = self.myData
		t.level = UnitLevel("player")
		t.class,t.localclass = UnitClass("player")
		t.currXP = UnitXP("player")
		t.nextXP = UnitXPMax("player")
		t.restXP = GetXPExhaustion() or 0
		t.isResting = IsResting() and true or false
		t.zone = zone
		if self.timePlayed then
			t.timePlayed = self.timePlayed + time() - self.timePlayedMsgTime
		elseif not t.timePlayed then
			t.timePlayed = 0
		end
		t.faction = UnitFactionGroup("player")
		t.time = time()
	end
end

local sortChars_realm
local function sortChars(alpha, bravo)
	alpha = RestFu.data[sortChars_realm][alpha]
	bravo = RestFu.data[sortChars_realm][bravo]
	if alpha.level == maxLevel then
		return false
	elseif bravo.level == maxLevel then
		return true
	end
	local timePassed = alpha.restXP / alpha.nextXP / percentPerSecond
	local timeToMaxAlpha = 864000 - timePassed
	if not alpha.isResting then
		timeToMaxAlpha = timeToMaxAlpha * 4
	end
	local timePassed = bravo.restXP / bravo.nextXP / percentPerSecond
	local timeToMaxBravo = 864000 - timePassed
	if not bravo.isResting then
		timeToMaxBravo = timeToMaxBravo * 4
	end
	if timeToMaxAlpha ~= timeToMaxBravo then
		return timeToMaxAlpha < timeToMaxBravo
	elseif alpha.level ~= bravo.level then
		return alpha.level < bravo.level
	else
		return alpha.currXP / alpha.nextXP > bravo.currXP / bravo.nextXP
	end
end
local function sortRealms(alpha, bravo)
	local alphaChar = RestFu.charList[alpha][1]
	local bravoChar = RestFu.charList[bravo][1]
	alpha = RestFu.data[alpha][alphaChar]
	bravo = RestFu.data[bravo][bravoChar]
	if not bravo then
		return true
	elseif not alpha then
		return false
	end
	local timePassed = alpha.restXP / alpha.nextXP / percentPerSecond
	local timeToMaxAlpha = 864000 - timePassed
	if not alpha.isResting then
		timeToMaxAlpha = timeToMaxAlpha * 4
	end
	local timePassed = bravo.restXP / bravo.nextXP / percentPerSecond
	local timeToMaxBravo = 864000 - timePassed
	if not bravo.isResting then
		timeToMaxBravo = timeToMaxBravo * 4
	end
	if timeToMaxAlpha ~= timeToMaxBravo then
		return timeToMaxAlpha < timeToMaxBravo
	else
		return alpha.currXP / alpha.nextXP > bravo.currXP / bravo.nextXP
	end
end

function RestFu:OnUpdate()
	if not self.data then
		return
	end

	local now = time()
	
	for realm, data in pairs(self.data) do
		for char, data in pairs(data) do
			if data.level ~= maxLevel and data.restXP < data.nextXP * 1.5 then
				local seconds = now - data.time
				local gained = data.nextXP * percentPerSecond * seconds
				if not data.isResting then
					gained = gained / 4
				end
				data.time = now
				data.restXP = data.restXP + gained
				if data.restXP > data.nextXP * 1.5 then
					data.restXP = data.nextXP * 1.5
				end
			end
		end
	end
	
	for realm, data in pairs(self.data) do
		sortChars_realm = realm
		if self.charList[realm] then
			table_sort(self.charList[realm], sortChars)
		end
	end
	table_sort(self.realmList, sortRealms)
end

function RestFu:OnUpdate_TimePlayed()
	if self:IsEventScheduled("RestFu_TimePlayed") then
		self:CancelScheduledEvent("RestFu_TimePlayed")
	end
	RequestTimePlayed()
end

function RestFu:TIME_PLAYED_MSG()
	if self:IsEventScheduled("RestFu_TimePlayed") then
		self:CancelScheduledEvent("RestFu_TimePlayed")
	end
	self.timePlayed = arg1
	self.timePlayedMsgTime = time()
	self:Save()
end

function RestFu:RemoveChar(realm, char)
	if not IsControlKeyDown() then return end
	self.data[realm][char] = nil
	self:Update()
end

function RestFu:RemoveRealm(realm)
	if not IsControlKeyDown() then return end
	self.data[realm] = nil
	self:Update()
end
