--[[

FuBar_PTPTReagentFu v3, a reagent tracker for World of Warcraft.
Copyright (C) 2007 Rabbit

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

]]

PTReagentFu = AceLibrary("AceAddon-2.0"):new("AceEvent-2.0", "AceDB-2.0", "FuBarPlugin-2.0")
local PTReagentFu = PTReagentFu

PTReagentFu.revision = tonumber(string.sub("$Revision: 30 $", 12, -3))

local PT = AceLibrary("LibPeriodicTable-3.1")
local L = AceLibrary("AceLocale-2.2"):new("PTReagentFu")
local crayon = AceLibrary("Crayon-2.0")

local track = {}
local trackAsOne = {}
local max = {}

local playerClass = nil
local reagentCount = {}
local reagentNames = {}

local additionalSets = {
	WARLOCK = {"Consumable.Warlock.Healthstone", "Consumable.Warlock.Spellstone", "Consumable.Warlock.Soulstone", "Consumable.Warlock.Firestone"},
	ROGUE = {"Consumable.Potion.Other.Energy", "Consumable.Weapon Buff.Poison.Anesthetic", "Consumable.Weapon Buff.Poison.Wound", "Consumable.Weapon Buff.Poison.Mind Numbing", "Consumable.Weapon Buff.Poison.Instant", "Consumable.Weapon Buff.Poison.Crippling", "Consumable.Weapon Buff.Poison.Deadly"},
	MAGE = {"Consumable.Mage.Mana Stone"},
	HUNTER = {"Misc.Reagent.Ammo.Bullet", "Misc.Reagent.Ammo.Arrow"},
	WARRIOR = {"Misc.Reagent.Ammo.Bullet", "Misc.Reagent.Ammo.Arrow"},
}

local options = {
	type = "group",
	args = {
		shortnames = {
			type = "toggle",
			name = L["Show names"],
			desc = L["Show reagent names in the FuBar text."],
			get = function() return PTReagentFu.db.profile.showShortNames end,
			set = function()
				PTReagentFu.db.profile.showShortNames = not PTReagentFu.db.profile.showShortNames
				PTReagentFu:Update()
			end,
			order = 101,
		},
		needmore = {
			type = "toggle",
			name = L["Only show what you need"],
			desc = L["Only show reagents that you do not have a full stack or more of in your inventory."],
			get = function() return PTReagentFu.db.profile.needMore end,
			set = function(v)
				PTReagentFu.db.profile.needMore = v
				PTReagentFu:Update()
			end,
			order = 102,
		},
		spacer = {
			type = "header",
			name = " ",
			order = 103,
		},
		reagents = {
			type = "group",
			name = L["Reagents"],
			desc = L["Toggle what reagents to track."],
			pass = true,
			get = function(key) return not PTReagentFu.db.char.hideReagent[key] end,
			set = function(key, value)
				PTReagentFu.db.char.hideReagent[key] = not value
				PTReagentFu:UpdateDisplay()
			end,
			args = {},
			order = 104,
		},
	}
}

function PTReagentFu:OnInitialize()
	self.version = (self.version or "3").."."..tostring(self.revision or "1")

	self.OnMenuRequest = options
	self.hasIcon = "Interface\\Icons\\INV_Misc_Book_09"
	self.defaultPosition = "LEFT"
	self.blizzardTooltip = true

	self:RegisterDB("PTReagentFuDB", "PTReagentFuCharDB")
	self:RegisterDefaults("profile", {
		showShortNames = false,
		needMore = false,
	})
	self:RegisterDefaults("char", {
		hideReagent = {},
	})

	local _, playerClass = UnitClass("player")
	local class = playerClass:lower():gsub("^%l", string.upper)

	local ptSet = PT:GetSetTable("Misc.Reagent.Class."..class)
	if type(ptSet) == "table" then
		for k, v in pairs(ptSet) do
			table.insert(track, k)
		end
	end

	if type(additionalSets[playerClass]) == "table" then
		for i, v in ipairs(additionalSets[playerClass]) do
			trackAsOne[v] = PT:GetSetTable(v)
		end
	end

	local opt = options.args.reagents.args
	for i, v in ipairs(track) do
		local name = GetItemInfo(v)
		if name then
			opt[name] = {
				type = "toggle",
				name = name,
				desc = L["Toggle showing %s."]:format(name),
				order = i,
			}
		end
	end

	for setName, setItems in pairs(trackAsOne) do
		if L:HasTranslation(setName) then
			setName = L[setName]
		end
		opt[setName] = {
			type = "toggle",
			name = setName,
			desc = L["Toggle showing %s."]:format(setName),
		}
	end
end

function PTReagentFu:OnEnable()
	local icon = nil
	if playerClass == "DRUID" then
		icon = "Misc_Branch_01"
	elseif playerClass == "MAGE" then
		icon = "Misc_Dust_01"
	elseif playerClass == "PALADIN" then
		icon = "Stone_WeightStone_05"
	elseif playerClass == "PRIEST" then
		icon = "Misc_Candle_03"
	elseif playerClass == "ROGUE" then
		icon = "Misc_Powder_Purple"
	elseif playerClass == "SHAMAN" then
		icon = "Jewelry_Talisman_06"
		elseif playerClass == "WARLOCK" then
		icon = "Misc_Gem_Amethyst_02"
	else
		icon = "Misc_Book_09"
	end
	self:SetIcon("Interface\\Icons\\INV_" .. icon)

	self:RegisterBucketEvent("BAG_UPDATE", 5, "Update")
end

function PTReagentFu:OnDataUpdate()
	for i in ipairs(reagentNames) do reagentNames[i] = nil end

	for i, v in ipairs(track) do
		local name, link = GetItemInfo(v)
		if name and link then
			table.insert(reagentNames, name)
			reagentCount[name] = GetItemCount(link)
			if type(max[name]) ~= "number" then
				local stackSize = select(8, GetItemInfo(v))
				max[name] = stackSize
			end
		end
	end

	for setName, setItems in pairs(trackAsOne) do
		local counter = 0
		local stackSize = 0
		for k, v in pairs(setItems) do
			local link = select(2, GetItemInfo(k))
			if link then
				local size = select(8, GetItemInfo(k))
				if size > stackSize then stackSize = size end
				counter = counter + GetItemCount(link)
			end
		end
		if L:HasTranslation(setName) then
			setName = L[setName]
		end
		table.insert(reagentNames, setName)
		reagentCount[setName] = counter
		max[setName] = stackSize
	end

	table.sort(reagentNames)
end

function PTReagentFu:OnTextUpdate()
	local text = ""
	local needMore = self.db.profile.needMore
	local shortNames = self.db.profile.showShortNames

	local db = self.db.char.hideReagent
	for i, v in ipairs(reagentNames) do
		local m = max[v]
		local num = reagentCount[v]
		if not db[v] and type(num) == "number" and (not needMore or (type(m) == "number" and num < m)) then
			if text ~= "" then
				text = text.."/"
			end
			if shortNames then
				text = text .. v ..": "
			end
			text = text .. string.format("|cff%s%d|r", crayon:GetThresholdHexColor(num, 0, m), num)
		end
	end
	self:SetText(text)
end

function PTReagentFu:OnTooltipUpdate()
	GameTooltip:AddLine("Reagents")
	local needMore = self.db.profile.needMore
	local db = self.db.char.hideReagent
	for i, v in ipairs(reagentNames) do
		local num = reagentCount[v]
		local m = max[v]
		if not db[v] and type(num) == "number" and (not needMore or (type(m) == "number" and num < m)) then
			local r, g, b = crayon:GetThresholdColor(num, 0, m)
			GameTooltip:AddDoubleLine(v, num, 1,1,0, r,g,b)
		end
	end
end

